package retypar.converter.helper;

import retypar.retypar;

import java.awt.image.BufferedImage;
import java.awt.*;
import java.util.LinkedList;

/**
 * Generates a <code>LinkedList</code> of {@link retypar.converter.helper.Letter}s and calculates their values.
 * @author Jens Heuser
 */
public class FontRenderer {

    /**
     * The image in which the current <code>Letter</code> is drawn to be calculated.
     */
    private BufferedImage sign;

    /**
     * The list of <code>Letter</code>s used for the outstanding conversion.<br>
     * This list is a concatenation of the {@link FontRenderer#bigL}, {@link FontRenderer#smallL}, {@link FontRenderer#numb}, {@link FontRenderer#spec}
     * lists as needed. These lists are only generated once, while <code>letters</code> gets assembled from scratch before every conversion.
     */
    static LinkedList letters   = new LinkedList();

    /**
     * Letters A-Z
     */
    private LinkedList bigL      = new LinkedList();
    /**
     * Letters a-z
     */
    private LinkedList smallL    = new LinkedList();
    /**
     * Numbers 0-9
     */
    private LinkedList numb      = new LinkedList();
    /**
     * Special characters. Currently only whitespace.
     */
    private LinkedList spec      = new LinkedList();

    private String lastFont = "";
    private int lastFontSize = 0;


    /**
     * This method is responsible for the creation of the lists.<br>
     * The boolean parameters tell which list has to be generated.
     * @param font The font to be used when rendering the ASCII signs.
     * @param uppercase "A-Z"?
     * @param lowercase "a-z"?
     * @param numbers "0-9"?
     * @param special " "? (currently only whitespace)
     */
    public void render(Font font, boolean uppercase, boolean lowercase, boolean numbers, boolean special){
        sign = new BufferedImage(retypar.fontSize, retypar.fontSize, BufferedImage.TYPE_INT_RGB);
        sign.getGraphics().setFont(font);
        Graphics2D g = sign.createGraphics();

        letters.clear();

        if ((!font.getFontName().equals(lastFont) || (retypar.fontSize != lastFontSize))){
            bigL = new LinkedList();
            smallL = new LinkedList();
            numb = new LinkedList();
            spec = new LinkedList();
        }

        lastFont = font.getFontName();
        lastFontSize = retypar.fontSize;

        if(uppercase) {
            if (bigL.isEmpty()) generate(65, 90, g, bigL);
            letters.addAll(bigL);
        }

        if(lowercase) {
            if (smallL.isEmpty()) generate(97, 122, g, smallL);
            letters.addAll(smallL);
        }

        if(numbers) {
            if (numb.isEmpty()) generate(48, 57, g, numb);
            letters.addAll(numb);
        }

        if(special) {
            if (spec.isEmpty()) generate(32, 32, g, spec);
            letters.addAll(spec);
        }
        g.dispose();
        g = null;
    }



    /**
     * Draws the signs into an image and runs calculation through {@link FontRenderer#calcValues}.<br>
     * The sign then gets added to the given list.
     * @param startX The ASCII code of the first sign.
     * @param startY The ASCII code of the last sign.
     * @param g The signs get drawed in here. Declared in {@link FontRenderer#render}, so only one instance created.
     * @param list The list of {@link retypar.converter.helper.Letter}s representing the given signs.
     */
    private void generate(int startX, int startY, Graphics2D g, LinkedList list){
        Letter letter;

        for (int l = startX; l < startY+1; l++){
            letter = new Letter((char)l);

            g.setColor(Color.WHITE);
            g.fillRect(0, 0, retypar.fontSize - 1, retypar.fontSize - 1);
            g.setColor(Color.BLACK);
            g.drawString(letter.toString(), 0, retypar.fontSize);

            calcValues(letter);
            list.add(letter);
        }
    }


    /**
     * The calculation of a sign's values happens here. The values are added to the given <code>Letter</code>'s {@link retypar.converter.helper.Letter#value}.
     * @param letter The <code>Letter</code> to be calculated.
     */
    private void calcValues(Letter letter){
        for(int c = 0; c < retypar.fontSize; c++){
            for(int d = 0; d < retypar.fontSize; d++)
                letter.addValue(sign.getRGB(c, d));
        }
    }
}
