/* SimpleFigApplet - jfig.gui.SimpleFigApplet.java 
 *
 * 01.03.02 - added enableJava2D applet param and check
 * 05.06.01 - add showMessages, add ItemListener stuff
 * 02.06.01 - add help and printing to menu, document parameters
 * 01.06.01 - changes in FigBasicCanvas to allow printing
 * 24.04.01 - added popup menu
 * 19.11.00 - first version, really minimal viewer, one file
 *
 * (C) F.N.Hendrich, hendrich@informatik.uni-hamburg.de
 */

package jfig.gui;

import jfig.canvas.*;
import jfig.objects.*;
import jfig.utils.BoundingBoxCalculator;

import java.applet.*;
import java.awt.*;
import java.awt.event.*;
import java.io.*;
import java.net.URL;
import java.util.Enumeration;

/**
 * minimal FIG viewer applet: one static file, no GUI
 */
public class SimpleFigApplet extends Applet 
      implements ActionListener, MouseListener, KeyListener, ItemListener,
                  FigDrawableEnumerator 
{

  Button          redrawButton, zoomInButton, zoomOutButton;
  // FigBasicCanvas  objectCanvas;
  PresentationViewerCanvas objectCanvas;

  PopupMenu       popupMenu;

  FigAttribs      attribs;
  FigParser       parser;
  FigObjectList   objectList;

  InputStream     inputStream;
  URL             url;

  // some applet parameter settable flags...
  boolean debug = false;
  boolean enablePrinting = false;

  // zoom region stuff
  boolean         zoomRegionMode = false;
  private Point   zoomRegionFirstCorner, zoomRegionSecondCorner;

  // help dialog stuff
  Frame    helpFrame;
  TextArea helpTextArea;
  Button   helpOKButton;

  // global menu items
  CheckboxMenuItem   popupConsoleMI, antiAliasMI, renderQualityMI;

  public String getAppletInfo() {
    return 
         "jfig simple FIG viewer applet (20020301.a) \n"
       + "See http://tech-www.informatik.uni-hamburg.de/applets/jfig/\n" 
       + "(C) 1996-2002 F.N.Hendrich, hendrich@informatik.uni-hamburg.de\n"
    ;
  }

  public String[][] getParameterInfo() {
    String info[][] = {
      { "figfile", "<filename>",  "the FIG file to show, e.g. gold.fig" },
      { "basedir", "<pathname>",  "optional directory path name"  },
      { "debug",   "boolean",  "verbose startup messages, e.g. false" },
      { "printing","boolean",  "allow printing, e.g. true" },
      { "enableJava2D", "boolean", "use Java2D when available" }, 
    };
    return info;
  }


  public void init() {
    if (debug) msg( "-#- init..." );
    jfig.utils.ExceptionTracer.setEnabled( false );
    enablePrinting = "true".equals( getParameter( "printing" ));
    debug          = "true".equals( getParameter( "debug" ));

    boolean j2d    = !"false".equals( getParameter( "enableJava2D" ));
    FigAttribs dummy = new FigAttribs(); // force class initialization
    FigAttribs.enableJava2D = j2d;
    

    if (debug) msg( "-#- Java2D enable is: " + j2d );
    
    String figfile = getParameter( "figfile" );
    String zoom    = getParameter( "zoom" );
    String basedir = getParameter( "basedir" );
    String server  = getParameter( "server" );

    if (debug) msg( "-#- parameters: " + figfile + " " + zoom );

    if (server != null) ImageHelper.setAppletServerName( server );
    if (basedir!= null) ImageHelper.setAppletBaseDir( basedir );

    if (debug) msg ( "-#- creating the attribs and objectList..." );
    attribs    = new FigAttribs();
    objectList = new FigObjectList();

    if (debug) msg( "-#- creating the canvas..." );
    // objectCanvas = new FigBasicCanvas();
    objectCanvas = new PresentationViewerCanvas();
    objectCanvas.showRulers( false );
    objectCanvas.getTrafo().setGridMode( FigTrafo2D.NO_GRID );
    objectCanvas.getTrafo().setSnapRelative( FigTrafo2D.NO_SNAP );
    objectCanvas.getTrafo().set_zoom( Double.valueOf(zoom).doubleValue() );
    objectCanvas.setObjectEnumerator( this );
    // objectCanvas.removeMouseListener( objectCanvas );
    objectCanvas.addMouseListener( this );
    objectCanvas.addKeyListener( this );

    if (debug) msg( "-#- creating the buttons..." );
    Panel buttonPanel = new Panel();
    redrawButton = new Button( "Redraw" );
    zoomInButton = new Button( "Zoom+" );
    zoomOutButton= new Button( "Zoom-" );
    buttonPanel.add( redrawButton );
    buttonPanel.add( zoomInButton );
    buttonPanel.add( zoomOutButton );
    redrawButton.addActionListener( this );
    zoomInButton.addActionListener( this );
    zoomOutButton.addActionListener( this );

    if (debug) msg( "-#- creating the popup menu..." );
    buildPopup();

    if (debug) msg( "-#- basic GUI layout..." );
    this.setLayout( new BorderLayout() );
    // this.add( "North", buttonPanel );
    this.add( "Center", objectCanvas );
    this.add( popupMenu );

    if (debug) msg( "-#- creating the parser..." );
    parser     = new FigParser();
    parser.setObjectPainter( objectCanvas );

    if (debug) msg( "-#- constructing the input URL..." );
    try {
      url         = new URL( getCodeBase(), figfile );
      inputStream = url.openStream();
    }
    catch( Exception e ) {
      msg( "-E- " + e );
    }
    
    if (debug) msg( "-#- parsing: " + url.toExternalForm() );
    parser.setFilenameAndType( url.toExternalForm(), "URL" );
    parser.parse_fig_file_not_threaded( 
                  inputStream, 
                  true, false, false, // toplevel, compound, merge
                  attribs, 
                  objectCanvas.getTrafo(),
                  objectList );

    if (debug) msg( "-#- parser ok, repainting..." );
    repaint();
  }


  public void start() {
  }

  public void stop() {
  }

  public void paint( Graphics g ) {
    if (debug) msg( "-#- paint..." );
    objectCanvas.doFullRedraw();
  }


  public void buildPopup() {
    Menu       navigateMenu, zoomMenu, sizeMenu, panMenu, optionsMenu;
    MenuItem   quitMI, helpMI, browseMI, gotoSlideMI, messagesMI;
    MenuItem   openMI, printMI;
    MenuItem   firstSlideMI, nextSlideMI, nextChapterMI;
    MenuItem   lastSlideMI, prevSlideMI, prevChapterMI;
    MenuItem   zoom11MI, zoomInMI, zoomOutMI, zoomFitMI, zoomWidthMI;
    MenuItem   zoomIn10MI, zoomOut10MI, zoomRegionMI;
    MenuItem   size400MI, size640MI, size800MI, size1024MI;
    MenuItem   sizeDefaultMI, sizeFullMI;
    MenuItem   panHomeMI, panLeftMI, panRightMI, panUpMI, panDownMI;

    helpMI        = new MenuItem( S_Help );
    messagesMI    = new MenuItem( S_Messages );
    quitMI        = new MenuItem( S_Quit );
    browseMI      = new MenuItem( S_Browse );
    openMI        = new MenuItem( S_OpenURL );
    openMI.setEnabled( false );

    printMI       = new MenuItem( S_Print );
    printMI.setEnabled( enablePrinting );

    //firstSlideMI  = new MenuItem( S_FirstSlide );
    //nextSlideMI   = new MenuItem( S_NextSlide );
    //nextChapterMI = new MenuItem( S_NextChapter );
    //prevChapterMI = new MenuItem( S_PrevChapter );
    //prevSlideMI   = new MenuItem( S_PrevSlide );
    //lastSlideMI   = new MenuItem( S_LastSlide );
    //gotoSlideMI   = new MenuItem( S_GotoSlide );

    zoom11MI      = new MenuItem( S_Zoom100 );
    zoomFitMI     = new MenuItem( S_ZoomFit );
    zoomInMI      = new MenuItem( S_ZoomIn );
    zoomIn10MI    = new MenuItem( S_ZoomIn10 );
    zoomOut10MI   = new MenuItem( S_ZoomOut10 );
    zoomOutMI     = new MenuItem( S_ZoomOut );
    zoomWidthMI   = new MenuItem( S_ZoomFitWidth );
    zoomRegionMI  = new MenuItem( S_ZoomRegion );

    //size400MI     = new MenuItem( S_400x300 );
    //size640MI     = new MenuItem( S_640x480 );
    //size800MI     = new MenuItem( S_800x600 );
    //size1024MI    = new MenuItem( S_1024x768 );
    //sizeDefaultMI = new MenuItem( S_DefaultSize );

    panHomeMI     = new MenuItem( S_PanHome );
    panLeftMI     = new MenuItem( S_PanLeft );
    panRightMI    = new MenuItem( S_PanRight );
    panDownMI     = new MenuItem( S_PanDown );
    panUpMI       = new MenuItem( S_PanUp );

    antiAliasMI     = new CheckboxMenuItem( S_AntiAlias );
    renderQualityMI = new CheckboxMenuItem( S_RenderQuality );
    popupConsoleMI  = new CheckboxMenuItem( S_PopupConsole,
                                 Console.getConsole().isPopupConsoleOnWEF() );

    antiAliasMI.addItemListener( this );
    renderQualityMI.addItemListener( this );
    popupConsoleMI.addItemListener( this );



    //
    // set menu callback target
    //
    zoomRegionMI  .addActionListener( this );
    zoom11MI      .addActionListener( this );
    zoomInMI      .addActionListener( this );
    zoomIn10MI    .addActionListener( this );
    zoomOut10MI   .addActionListener( this );
    zoomOutMI     .addActionListener( this );
    zoomFitMI     .addActionListener( this );

    panHomeMI     .addActionListener( this );
    panLeftMI     .addActionListener( this );
    panRightMI    .addActionListener( this );
    panDownMI     .addActionListener( this );
    panUpMI       .addActionListener( this );

    helpMI        .addActionListener( this );
    messagesMI    .addActionListener( this );
    quitMI        .addActionListener( this );
    browseMI      .addActionListener( this );
    printMI       .addActionListener( this );

    //gotoSlideMI   .addActionListener( this );
    //prevSlideMI   .addActionListener( this );
    //nextSlideMI   .addActionListener( this );
    //firstSlideMI  .addActionListener( this );
    //lastSlideMI   .addActionListener( this );
    //nextChapterMI .addActionListener( this );
    //prevChapterMI .addActionListener( this );

    //
    // build menu structure
    //
    //navigateMenu  = new Menu( S_NavigateMenu );
    //navigateMenu.add( nextSlideMI );
    //navigateMenu.add( prevSlideMI );
    //navigateMenu.addSeparator();
    //navigateMenu.add( browseMI );
    //navigateMenu.add( gotoSlideMI );
    //navigateMenu.addSeparator();
    //navigateMenu.add( firstSlideMI );
    //navigateMenu.add( prevChapterMI );
    //navigateMenu.add( nextChapterMI );
    //navigateMenu.add( lastSlideMI );

    zoomMenu = new Menu( S_ZoomMenu );
    zoomMenu.add( zoomFitMI );
    zoomMenu.add( zoom11MI );
    zoomMenu.addSeparator();
    zoomMenu.add( zoomInMI );
    zoomMenu.add( zoomIn10MI );
    zoomMenu.add( zoomOut10MI );
    zoomMenu.add( zoomOutMI );
    zoomMenu.add( zoomRegionMI );
    zoomMenu.addSeparator();
    zoomMenu.add( zoomRegionMI );
    //zoomMenu.add( zoomWidthMI );

    panMenu = new Menu( S_PanMenu );
    panMenu.add( panHomeMI );
    panMenu.add( panLeftMI );
    panMenu.add( panRightMI );
    panMenu.add( panDownMI );
    panMenu.add( panUpMI );

    sizeMenu = new Menu( S_SizeMenu );

    optionsMenu = new Menu( S_OptionsMenu );
    optionsMenu.add( antiAliasMI );
    optionsMenu.add( renderQualityMI );
    //optionsMenu.addSeparator();
    //optionsMenu.add( popupConsoleMI );

    antiAliasMI.setEnabled( FigAttribs.enableJava2D );
    renderQualityMI.setEnabled( FigAttribs.enableJava2D );

    popupMenu = new PopupMenu();
    popupMenu.addSeparator();
    // popupMenu.add( navigateMenu );
    popupMenu.add( zoomMenu );
    // popupMenu.add( sizeMenu );
    popupMenu.add( panMenu );
    popupMenu.add( optionsMenu );
    popupMenu.addSeparator();
    // popupMenu.add( openMI );
    popupMenu.add( helpMI );
    popupMenu.add( messagesMI );
    popupMenu.add( printMI );
    // popupMenu.addSeparator();
    // popupMenu.add( quitMI );
  }


  /**************************************************************************/
  /**************************************************************************/
  /* actionPerformed                                                        */
  /**************************************************************************/
  /**************************************************************************/

  public void actionPerformed( ActionEvent e ) {
    String s = (String) e.getActionCommand();

    // System.err.println( "-X- actionPerformed: " + e + " " + s );

    if      (s.equals( S_Quit      ))      doQuit();
    else if (s.equals( S_Help      ))      doShowHelp();
    else if (s.equals( S_Messages  ))      doShowMessages();
    else if (s.equals( S_Print     ))      doPrint();

    // else if (s.equals( S_Browse    ))      doOpenBrowser();
    // else if (s.equals( S_GotoSlide ))      doOpenGotoDialog();

    // else if (s.equals( S_NextSlide ))      doShowNextSlide();
    // else if (s.equals( S_PrevSlide ))      doShowPrevSlide();
    // else if (s.equals( S_FirstSlide ))     doShowFirstSlide();
    // else if (s.equals( S_LastSlide ))      doShowLastSlide();
    // else if (s.equals( S_NextChapter ))    doShowNextChapter();
    // else if (s.equals( S_PrevChapter ))    doShowPrevChapter();

    else if (s.equals( S_Zoom100 ))        doZoom11();
    else if (s.equals( S_ZoomIn ))         doZoomIn14();
    else if (s.equals( S_ZoomIn10 ))       doZoomIn11();
    else if (s.equals( S_ZoomOut10 ))      doZoomOut09();
    else if (s.equals( S_ZoomOut ))        doZoomOut07();
    else if (s.equals( S_ZoomFit ))        doZoomFit();
    else if (s.equals( S_ZoomFitWidth ))   doZoomFitWidth();
    else if (s.equals( S_ZoomRegion ))     doZoomRegion();

    else if (s.equals( S_PanHome  ))       doPanHome( e );
    else if (s.equals( S_PanLeft  ))       doPanLeft( e );
    else if (s.equals( S_PanRight ))       doPanRight( e );
    else if (s.equals( S_PanDown  ))       doPanDown( e );
    else if (s.equals( S_PanUp    ))       doPanUp( e );

    // else if (s.equals( S_400x300 ))        doResize( 400, 300 );
    // else if (s.equals( S_640x480 ))        doResize( 640, 480 );
    // else if (s.equals( S_800x600 ))        doResize( 800, 600 );
    // else if (s.equals( S_1024x768 ))       doResize( 1024, 768 );
    // else if (s.equals( S_DefaultSize ))    doResizeToDefaultSize();
    // else if (s.equals( S_FullScreen ))     doResizeToFullScreen();

    else if (e.getSource() == helpOKButton) {
      helpFrame.setVisible( false );
    }

    else {
      msg( "-W- unknown action command: " + e );
    }
  }

  public void itemStateChanged( ItemEvent e ) {
    if      (e.getSource() == antiAliasMI )    doUpdateAntiAlias();
    else if (e.getSource() == renderQualityMI) doUpdateRenderQuality();
    else {
      msg( "-W- unknown item event: " + e );
    }
  }

  /**************************************************************************/
  /**************************************************************************/
  /* command handlers                                                       */
  /**************************************************************************/
  /**************************************************************************/


  public void doCancel() {
    // empty
  }


  public void doQuit() {
    // empty
  }

  public void doShowMessages() {
    Console console = jfig.gui.Console.getConsole();
    if (console != null) console.show();
  }

  public void doShowHelp() {
    if (helpFrame == null) {
      helpFrame = new Frame( "jfig applet help" );
      helpTextArea = new TextArea( 20, 70 );
      helpTextArea.setText( 
         getAppletInfo() 
       + "\n"
       + "Use the popup menu for panning, zooming, options, printing.\n"
       + "Use the cursor keys, or <shift>+cursor keys for panning.\n" 
       + "Use 's' 'd' 'f' 'g' 'h' to zoom: OUT out fit in IN\n"
       + "Use 'r' to select a region to zoom into.\n" 
       + "\n"
       + "applet parameters:\n" 
       + "name=figfile    value=<filename> (e.g. gold.fig)\n"
       + "name=basedir    value=<pathname> (e.g. /pub/figs/flowers)\n"
       + "name=debug      value=true|false (verbose messages)\n"
       + "name=printing   value=true|false (allow to print the FIG file)\n"
       + "name=zoom       value=<double>   (initial zoom factor, e.g. 1.0)\n"
       + "name=enableJava2D value=true|false (enable Java2D rendering)\n"
       + "\n"
       + "Note: Java printing is very buggy and may not work on your system." 
      );
      helpOKButton = new Button( "OK" );
      helpOKButton.addActionListener( this );
      helpFrame.add( "Center", helpTextArea );
      helpFrame.add( "South",  helpOKButton );
    }
    helpFrame.pack();
    helpFrame.show();
  }

  public void doPrint() {
    msg( "-W- doPrint: not yet implemented." );
    msg( "Please print the embedding HTML page from your browser." );
  }

  public void doRedraw() {
    objectCanvas.doFullRedraw();
  }

  public void setZoomMessage() {
    // empty
  }

  public void doZoomIn14() {
    objectCanvas.doZoomIn14();
    setZoomMessage();
  }

  public void doZoomIn11() {
    doZoomFactor( 1.1 );
  }

  public void doZoomOut09() {
    doZoomFactor( 1.0/1.1 );  // 0.90909...
  }

  public void doZoomFactor( double factor ) {
    FigTrafo2D trafo = objectCanvas.getTrafo();
    Dimension size   = objectCanvas.getSize();

    int       dx     = trafo.screen_to_wc( size.width );
    int       dy     = trafo.screen_to_wc( size.height );
    double    zoom   = trafo.getZoom();
    Point     anchor = trafo.getAnchor();

    trafo.set_zoom( factor * zoom );
    trafo.setAnchor( new Point( anchor.x + (int) (0.5*(1.0-1.0/factor)*dx),
                                anchor.y + (int) (0.5*(1.0-1.0/factor)*dy)  ));

    doRedraw();
    setZoomMessage();
  }

  public void doZoomOut07() {
    objectCanvas.doZoomOut07();
    setZoomMessage();
  }

  public void doZoom11() {
    objectCanvas.doZoomFull();
    objectCanvas.doPanning( FigCanvas.panHOME, false );
    setZoomMessage();
  }

 public void setZoomFactor( double factor ) {
    objectCanvas.getTrafo().set_zoom( factor );
    doRedraw();
    setZoomMessage();
  }

  public void doZoomRegion() {
    msg("click on the first corner of the region to zoom into" );
    objectCanvas.changeRubberbandMode( FigCanvasRubberband.pointRubber );

    zoomRegionFirstCorner = null;
    zoomRegionSecondCorner = null;
    zoomRegionMode = true;
  }

  public void handleMousePressedZoomRegion( MouseEvent e ) {
    if (zoomRegionFirstCorner == null) { // this is the first click
      zoomRegionFirstCorner = new Point( e.getX(), e.getY() );
      objectCanvas.setRubberbandBasePoint( zoomRegionFirstCorner );
      objectCanvas.changeRubberbandMode( FigCanvasRubberband.rectRubber );
      msg( "click on the second corner of the region to zoom into" );
      return;
    }

    // if we arrive here, we have a second mouse click:
    // retrieve the corner world coordinates and do the zooming,
    // reset zoomRegionMode
    //
    zoomRegionSecondCorner = new Point( e.getX(), e.getY() );
    FigTrafo2D trafo = objectCanvas.getTrafo();
    Point W1         = trafo.getWorldCoords( zoomRegionFirstCorner );
    Point W2         = trafo.getWorldCoords( zoomRegionSecondCorner );

    objectCanvas.changeRubberbandMode( FigCanvasRubberband.noRubber );
    objectCanvas.doZoomRegion( W1.x, W1.y, W2.x, W2.y );
    doRedraw();
    zoomRegionMode = false;
  }

  public void doZoomFit() {
    FigBbox bbox = BoundingBoxCalculator.getBoundingBox(objectList.elements());

    objectCanvas.doZoomRegionFree( bbox );
    setZoomMessage();

  }

  public void doZoomFitWidth() {
    setZoomMessage();
  }



  public void doPanHome( Object arg ) {
    objectCanvas.doPanning( FigCanvas.panHOME, false );
    setZoomMessage();
    //doRedraw();
  }

  public void doPanLeft( Object arg ) {
    boolean shift = false;
    if (arg instanceof KeyEvent) shift = ((KeyEvent) arg).isShiftDown();
    objectCanvas.doPanning( FigCanvas.panLEFT, shift );
  }

  public void doPanRight( Object arg ) {
    boolean shift = false;
    if (arg instanceof KeyEvent) shift = ((KeyEvent) arg).isShiftDown();
    objectCanvas.doPanning( FigCanvas.panRIGHT, shift );
  }

  public void doPanUp( Object arg ) {
    boolean shift = false;
    if (arg instanceof KeyEvent) shift = ((KeyEvent) arg).isShiftDown();
    objectCanvas.doPanning( FigCanvas.panUP, shift );
  }

  public void doPanDown( Object arg ) {
    boolean shift = false;
    if (arg instanceof KeyEvent) shift = ((KeyEvent) arg).isShiftDown();
    objectCanvas.doPanning( FigCanvas.panDOWN, shift );
  }


  public void doSetGridNone( Object arg ) {
    objectCanvas.getTrafo().setGridMode( FigTrafo2D.NO_GRID );
    doRedraw();
  }

  public void doSetGridMedium( Object arg ) {
    objectCanvas.getTrafo().setGridMode( FigTrafo2D.MEDIUM_GRID );
    doRedraw();
  }

  public void doUpdateAntiAlias() {
    boolean b = antiAliasMI.getState();
    System.out.println( "-#- doUpdateAntiAlias: " + b );

    if (FigAttribs.enableJava2D) { 
      objectCanvas.getOptions2D().requestAntiAliasing( b );
      objectCanvas.doFullRedraw();
    }
  }

  public void doUpdateRenderQuality() {
    boolean b = renderQualityMI.getState();
    System.out.println( "-#- doUpdateRenderQuality: " + b );

    if (FigAttribs.enableJava2D) { 
      objectCanvas.getOptions2D().requestRenderQuality( b );
      objectCanvas.doFullRedraw();
    }
  }



  /**************************************************************************/
  /**************************************************************************/
  /* mouse and key listeners                                                */
  /**************************************************************************/
  /**************************************************************************/

  public void mousePressed( MouseEvent e ) {
    if (zoomRegionMode) handleMousePressedZoomRegion( e );
    else if (e.isPopupTrigger()) {
      //popupPosition = new Point( e.getX(), e.getY() );
      popupMenu.show( objectCanvas, e.getX(), e.getY() );
      //sleep( 50 );
    }
    else {
      objectCanvas.requestFocus();
    }
  }

  public void mouseReleased( MouseEvent e ) {
    if (e.isPopupTrigger()) {
      //popupPosition = new Point( e.getX(), e.getY() );
      popupMenu.show( objectCanvas, e.getX(), e.getY() );
      //sleep( 50 );
    }
  }

  public void mouseClicked( MouseEvent e ) {}
  public void mouseEntered( MouseEvent e ) {
  }
  public void mouseExited ( MouseEvent e ) {}


  public void keyReleased( KeyEvent evt ) {
    // empty!
  }

  public void keyTyped( KeyEvent evt ) {
    // empty!
  }

  public void keyPressed( KeyEvent evt ) {
    char key  = evt.getKeyChar();
    int  code = evt.getKeyCode();
    boolean debug = false;

    if (evt.isActionKey()) { // some keys for panning etc.
      boolean shift = evt.isShiftDown();
      switch (code) {
        case KeyEvent.VK_DOWN :
                           objectCanvas.doPanning( FigCanvas.panDOWN, shift );
                           break;
        case KeyEvent.VK_UP   :
                           objectCanvas.doPanning( FigCanvas.panUP, shift );
                           break;
        case KeyEvent.VK_LEFT :
                           objectCanvas.doPanning( FigCanvas.panLEFT, shift );
                           break;
        case KeyEvent.VK_RIGHT:
                           objectCanvas.doPanning( FigCanvas.panRIGHT, shift );
                           break;

        default:
          msg( "-I- Unhandled action key: " + key );
          break;
      }
    }
    else {
      switch( key ) {
        case 'r'   : doZoomRegion();
                     break;
        case 's'   : doZoomOut07();
                     break;
        case 'd'   : doZoomOut09();
                     break;
        case 'f'   : doZoomFit();
                     break;
        case 'g'   : doZoomIn11();
                     break;
        case 'h'   : doZoomIn14();
                     break;
        case 'w'   :
                     doZoomFitWidth();
                     break;
        case '?'   : doShowHelp();
                     break;
        default:
          if (debug) {
            msg( "-W- ignoring key: " + key + " " + ((int) key));
          }
      } // switch
    }
  }





  public void msg( String s ) {
    System.out.println( s );
  }




  /** ** ** ObjectEnumerator stuff ** ** ** **/

  public Enumeration getDrawableObjects() {
    return objectList.elements();
  }

  public FigDrawable getTmpObject() {
    return null;
  }



  private static final String   // GUI string constants
    S_OptionsMenu         = "options...",
    S_NavigateMenu        = "navigate...",
    S_SizeMenu            = "window size...",
    S_PanMenu             = "pan...",
    S_ZoomMenu            = "zoom...",

    S_Help                = "help...",
    S_Messages            = "show messages...",
    S_Quit                = "quit",
    S_Browse              = "browse...",
    S_OpenURL             = "open URL...",
    S_Print               = "print...",

    S_FirstSlide          = "first slide",
    S_NextSlide           = "next slide",
    S_NextChapter         = "next chapter",
    S_PrevChapter         = "previous chapter",
    S_PrevSlide           = "previous slide",
    S_LastSlide           = "last slide",
    S_GotoSlide           = "goto...",

    S_Zoom100             = "zoom 100%",
    S_ZoomFit             = "zoom fit",
    S_ZoomIn              = "zoom in (1.4)",
    S_ZoomIn10            = "zoom in (1.1)",
    S_ZoomOut             = "zoom out (0.7)",
    S_ZoomOut10           = "zoom out (0.9)",
    S_ZoomFitWidth        = "zoom fit width",
    S_ZoomRegion          = "zoom region",

    S_PanHome             = "pan home",
    S_PanLeft             = "pan left",
    S_PanRight            = "pan right",
    S_PanDown             = "pan down",
    S_PanUp               = "pan up",

    S_400x300             = "400x300",
    S_640x480             = "640x480",
    S_800x600             = "800x600",
    S_1024x768            = "1024x768",
    S_DefaultSize         = "default size",
    S_FullScreen          = "full screen",

    S_PageName            = "show page name",
    S_PageNumber          = "show page number",
    S_AntiAlias           = "Java2D antialias",
    S_RenderQuality       = "Java2D rendering quality",
    S_PopupConsole        = "console popup on errors"
  ;


}
