#include <ctype.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <unistd.h>
#ifdef HAVE_GETOPT_LONG
#include <getopt.h>
#else
#include "support/getopt.h"
#include "support/getopt.c"
#include "support/getopt1.c"
#endif

#include "jftpgw.h"


int init_logfiles(const char*);
void reset_confst(struct configstruct*);
void print_version(void);
void print_help(void);

struct configstruct config;
struct clientinfo clntinfo;
extern struct loginfo li;
int multithread;
int timeout;
int main_server_pid = 0;
int chrooted = 0;
extern int should_read_config;

static struct option const long_option_arr[] =
{
	{ "single", no_argument, NULL, 's' },
	{ "encrypt", no_argument, NULL, 'e' },
	{ "version", no_argument, NULL, 'v' },
	{ "help", no_argument, NULL, 'h' },
	{ NULL, 0, NULL, 0 }
};


int main(int argc, char** argv) {

	/* We bind to 0.0.0.0, but our real IP is the one returned by the
	 * PASV command */

	struct sigaction sa, cf;
	int ret;
	const char* option;
	const char* bindaddress;

	/* default: Multithread, i.e. fork for each connection */
	multithread = 1;
	/* parse the command line */
	while ((ret=getopt_long(argc, argv, "sevh", long_option_arr, NULL)) != EOF) {
		switch (ret) {
			case 0: break;
			case 's':
				multithread = 0;
				break;
			case 'e':
				encrypt_password();
				exit(0);
			case 'v':
				print_version();
				exit(0);
			case 'h':
				print_help();
				exit(0);
			default:
				break;
		}
	}

	/* Read the configuration */
	ret = read_config(DEFAULTCONFFILE);
	if (ret) {
		return 1;
	}

#ifndef __EMX__
	/* Initialise the log files  -  not yet chroot()ed */
	if (get_option("atstartchangerootdir")) {
		ret = init_logfiles(get_option("atstartchangerootdir"));
	} else if (get_option("changerootdir")) {
		ret = init_logfiles(get_option("changerootdir"));
	} else {
#endif
		ret = init_logfiles((char*) 0);
#ifndef __EMX__
	}

	option = get_option("atstartchangerootdir");
	if (option) {
		if (geteuid() == 0) {
			chdir(option);
			ret = chroot(option);
			if (!ret) {
				log(7, "Changed root directory at start up"
					"to %s", option);
				chrooted = 1;
			} else {
				log(7, "Error changeing root "
				"directory on start up to %s: %s",
				option, strerror(errno));
				chrooted = 0;
			}
			chdir("/");
		}
	}

	if (getuid() == 0) {
		/* set the effective UID to the one of "runuser". We can't
		 * drop the real user id since we never know if the user
		 * wants to bind to a priviledged port (he could send us a
		 * HUP signal and we have to reread the configuration file.
		 * So we can't say, if port > 1024 change real uid)
		 */
		log(8, "Changing ID to the runasuser-ID (initial)");
		changeid(get_option("runasuser"), EUID);
	}
#endif

	/* Install the signal handlers */

	/* for SIGCHLD install just the reap function
	 *
	 * the register/unregister thing is installed after we've bound
	 * successfully */

	sa.sa_handler = reap_chld_info;
	sigemptyset (&sa.sa_mask);
	sa.sa_flags = SA_RESTART;
	sigaction (SIGCHLD, &sa, 0);

	cf.sa_handler = read_default_conf;
	should_read_config = 0;
	sigemptyset(&cf.sa_mask);
	cf.sa_flags = SA_RESTART;
	sigaction (SIGHUP, &cf, 0);

	cf.sa_handler = terminate;
	sigemptyset(&cf.sa_mask);
#ifdef __EMX__
	sigaction (SIGINT, &cf, 0);
#endif
	sigaction (SIGTERM, &cf, 0);
	sigaction (SIGQUIT, &cf, 0);
	sigaction (SIGABRT, &cf, 0);

	clntinfo.user = clntinfo.pass = clntinfo.destination = (char*) 0;
	clntinfo.throughput = 0;
	clntinfo.boundsocket_list = (int*) 0;

	atexit(closedescriptors);

	bindaddress = getlisten();

	if (!ret) {
		ret = handlecmds(bindaddress, 
				 &clntinfo);
		log(8, "Exited from handlecmds");
	}

	log(8, "Exiting");
	if (ret) {
		return 1;
	} else {
		return 0;
	}
}


int daemonize()
{
	int childpid;

	if( (childpid = fork()) < 0) return(-1);
	else if(childpid > 0) exit(0);

	errno = 0;
#ifndef __EMX__
	chdir("/");
	setsid();
#endif
	return(0);
}


void read_default_conf(int signo) {
	should_read_config = 1;
}

int reread_config() {
	int ret;
	const char* tmp;
	sigset_t sigset, oldset;
	log(6, "SIGHUP received - Rereading config file. My pid: %d", getpid());
	/* block other SIGHUP signals */

	sigemptyset(&sigset);
	sigemptyset(&oldset);
	sigaddset(&sigset, SIGCHLD);
	while ((ret = sigprocmask(SIG_BLOCK, &sigset,
		&oldset)) < 0 && errno == EINTR)  {}
	if (ret < 0) {
		log(2, "Error blocking signals: %s", strerror(errno));
	}
	clean_allowedlist();
	conf_delete_config();
	ret = read_config(DEFAULTCONFFILE);
	if (ret) {
		/* the following line is a BADF if we had a SIGHUP */
		log(1, "Error rereading config file. Exiting");
		exit(2);
	}
#ifndef __EMX__	
	/* Are we chroot()ed ? */
	if (chrooted) {
		tmp = get_option("atstartchangerootdir");
		if (!tmp) {
			tmp = get_option("changerootdir");
		}
	} else {
#endif
		tmp = (char*) 0;
#ifndef __EMX__	
	}
#endif
	init_logfiles(tmp);
	should_read_config = 0;
	while ((ret = sigprocmask(SIG_UNBLOCK, &sigset,
		&oldset)) < 0 && errno == EINTR) {}
	if (ret < 0) {
		log(2, "Error unblocking signal mask: %s", strerror(errno));
		return -1;
	}
	return 0;
}

void terminate (int signo) {
	/* exit is not POSIX-reentrant but in SVR4 SVID */
	exit(0);
}

int init_logfiles(const char* dir_prefix) {
	int ret =0, i;
	char buf[1024];
	const char* option;
	const char* specs;
	char* cpy;
	const int MIDLENGTH = 64;
	struct cmdlogent_t* cmdlogfilescur = &config.cmdlogfiles;
	struct cmdlogent_t* cmdlogdirscur = &config.cmdlogdirs;

	if (!dir_prefix) {
		dir_prefix = "";
	}

	config.syslog = 0;
	config.debuglevel = 9;
	/* read the logfile name from the configuration file */
	option = get_option("logfile");
	if (!option) {
		config.logf_name = (char*) malloc(strlen(dir_prefix) + strlen(DEFAULTLOGFILE) + 1 + 1);
		enough_mem(config.logf_name);
		strcpy(config.logf_name, dir_prefix);
		if (strlen(dir_prefix)) {
			strcat(config.logf_name, "/");
		}
		strcat(config.logf_name, DEFAULTLOGFILE);
		char_squeeze(config.logf_name, '/');
		log(4, "No logfile specified in configuration file");
	} else {
		config.logf_name = (char*) malloc(strlen(dir_prefix)
			+ strlen(option) + 1 + 1);
		enough_mem(config.logf_name);
		strcpy(config.logf_name, dir_prefix);
		if (strlen(dir_prefix)) {
			strcat(config.logf_name, "/");
		}
		strcat(config.logf_name, option);
		char_squeeze(config.logf_name, '/');
	}
	if (!get_option("debuglevel")) {
		log(4, "No debug level specified. Using level 7");
		config.debuglevel = 7;
	} else {
		const char* tmp = get_option("debuglevel");
		errno = 0;
		config.debuglevel = strtol(tmp, (char**) 0, 10);
		if (errno || config.debuglevel < 0 || config.debuglevel > 9) {
			log(4, "Invalid debuglevel specified: \"%s\". Using level 7", tmp);
			config.debuglevel = 7;
		}
	}


	/* Open the log file if not using syslog */
	if (!config.syslog) {
		if (!(config.logf = open_logfile(config.logf_name))) {
			return -1;
		}
	}
	/* else {
	*
	*	from tinyproxy
	*
	*	if (godaemon == TRUE)
			openlog("tinyproxy", LOG_PID, LOG_DAEMON);
		else
			openlog("tinyproxy", LOG_PID, LOG_USER);
		}
	*/
	/* Open the cmdlogfilex logfile(s) */

	for (i =1; i < 9999; i++) {
		sprintf(buf, "cmdlogfile%d", i);
		option = get_option(buf);
		if (!option) {
			break;
		}
		else {
			sprintf(buf, "cmdlogfile%d-specs", i);
			specs = get_option(buf);
			cpy = (char*) malloc(strlen(option) + strlen(dir_prefix) + 2);
			enough_mem(cpy);
			strcpy(cpy, dir_prefix);
			if (strlen(dir_prefix)) {
				strcat(cpy, "/");
			}
			strcat(cpy, option);
			char_squeeze(cpy, '/');

			if (i != 1) {
				cmdlogfilescur->next = malloc(sizeof
					(struct cmdlogent_t));
				enough_mem(cmdlogfilescur->next);
				cmdlogfilescur = cmdlogfilescur->next;
			}
			cmdlogfilescur->logf_name = cpy;
			cpy = (char*) 0;
			cmdlogfilescur->specs = (char*) malloc(
					strlen(specs) + 2);
			enough_mem(cmdlogfilescur->specs);
			strcpy(cmdlogfilescur->specs, specs);
			strcat(cmdlogfilescur->specs, " ");
			cmdlogfilescur->next = 0;
			if ((cmdlogfilescur->logf = 
				open_logfile(cmdlogfilescur->logf_name))
					!= NULL) {
				ret = 0;
			} else {
				return -1;
			}
		}
	}

	/* Do the same with directories */

	for (i = 1; i < 9999; i++) {
		sprintf(buf, "connectionlogdir%d", i);
		option = get_option(buf);
		if (!option) {
			break;
		} else {
			const char* prefix =0, *suffix =0;
			int size;
			sprintf(buf, "connectionlogdir%d-specs", i);
			specs = get_option(buf);
			if (!specs) {
				/* will never log */
				log(4, "Warning: no specs specified (%s)", buf);
				specs = "";
			}
			sprintf(buf, "connectionlogdir%d-fileprefix", i);
			prefix = get_option(buf);
			if (!prefix) {
				prefix = "";
			}
			sprintf(buf, "connectionlogdir%d-filesuffix", i);
			suffix = get_option(buf);
			if (!suffix) {
				suffix = "";
			}
			size = strlen(option) + MIDLENGTH +
				strlen(prefix) + strlen(suffix) + 1;
			cpy = (char*) malloc(size);
			enough_mem(cpy);

#ifdef __EMX__			
			sprintf(cpy, "%s/%s%%Y-%%m-%%d--%%H.%%M.%%S-%%%%d%s",
				option, prefix, suffix);
#else				
			sprintf(cpy, "%s/%s%%Y-%%m-%%d--%%H:%%M:%%S-%%%%d%s",
				option, prefix, suffix);
#endif				

			log(9, "logging to (raw) file: %s", cpy);

			if (i != 1) {
				cmdlogdirscur->next = malloc(sizeof
					(struct cmdlogent_t));
				enough_mem(cmdlogdirscur->next);
				cmdlogdirscur = cmdlogdirscur->next;
			}
			cmdlogdirscur->logf_name = cpy;
			cpy = (char*) 0;
			cmdlogdirscur->logf_size = size;
			cmdlogdirscur->logf = NULL;
			cmdlogdirscur->specs = (char*) malloc(
					strlen(specs) + 2);
			enough_mem(cmdlogdirscur->specs);
			strcpy(cmdlogdirscur->specs, specs);
			strcat(cmdlogdirscur->specs, " ");
			cmdlogdirscur->next = 0;
		}
	}
	return ret;
}

int init_logdirs(const char* prefix) {
	struct cmdlogent_t* clf = &config.cmdlogdirs;
	time_t nowtime;
	char* tmp =0;
	int ret = 0;

	if (config.cmdlogdirs.logf_name == NULL) {
		return 0;
	}

	nowtime = time(NULL);

	if (!prefix) {
		prefix = "";
	}
	tmp = (char*) malloc(clf->logf_size);
	enough_mem(tmp);

	while (clf && !ret) {
		strftime(tmp, clf->logf_size, clf->logf_name,
				localtime(&nowtime));
		sprintf(clf->logf_name, tmp, getpid());
		free(tmp);
		if (prefix) {
			tmp = (char*) malloc(strlen(clf->logf_name)
					+ strlen(prefix)
					+ 1
					+ 1);
			enough_mem(tmp);
			strcpy(tmp, prefix);
#ifdef __EMX__
			if ( strlen(prefix) )  strcat(tmp, "/");
#else
			strcat(tmp, "/");
#endif
			strcat(tmp, clf->logf_name);
			char_squeeze(tmp, '/');
			free(clf->logf_name);
			clf->logf_name = tmp;
			tmp = (char*) 0;
		}
		if ((clf->logf = open_logfile(clf->logf_name))) {
			ret = 0;
		} else {
			ret = -1;
		}
		clf = clf->next;
	}
	return ret;
}

void print_version(void) {
	printf(PACKAGE" v"JFTPGW_VERSION);
#ifdef __EMX__
	printf("  -  OS/2 Warp");
#endif
#ifdef SFTP_SUPPORT
	printf("  -  sftp support enabled");
#else
	printf("  -  without sftp support");
#endif
	printf("\n");
}

void print_help(void) {
	print_version();
	printf("usage: jftpgw [OPTION]\n\n");
	printf("Valid options:\n");
	printf("  -h, --help        Display this help text\n");
	printf("  -e, --encrypt     Use jftpgw to obtain an encrypted password\n");
	printf("  -s, --single      Run jftpgw single threaded (do not fork)\n");
	printf("  -v, --version     Display the version\n");
	printf("\nReport bugs to Joachim Wieland <joe@mcknight.de>\n");
#ifdef __EMX__
	printf("\nOS/2 version compiled by Eugene Romanenko.\n");
	printf("Report bugs for OS/2 version to <themeditat0r@usa.net>\n");
#endif
}

void removepidfile(void) {
	const char* option;

	if (getpid() != main_server_pid) {
		/* the program has not become a daemon */
		return;
	}

	option = get_option("pidfile");
	if (option) {
		int i;
#ifndef __EMX__
		if (getuid() == 0 && geteuid() != 0) {
			log(8, "Changing ID to root (create pidfile)");
			changeid("root", UID);
		}
#endif
		i = unlink(option);
		if (i < 0) {
			log(3, "Could not unlink the pidfile %s", option);
		}
#ifndef __EMX__
		if (getuid() == 0) {
			log(8, "Changing id back (deleting pidfile)");
			changeid(get_option("runasuser"), EUID);
		}
#endif
	}
}

void sayterminating(void) {
	log(6, "jftpgw terminating");
}

void closedescriptors(void) {
	int i;
	log(9, "In closedescriptors()");

	/* free the log info structure. Free the members, the structure for
	 * itself is on the stack */
	/* li.cmd must not be freed, it's   li.cmd = buffer;  */
	if (li.host) {
		free(li.host);
	}
	if (li.user) {
		free(li.user);
	}

	/* close the logfiles and delete the structures */
	reset_confst(&config);

	/* maybe these variables are not yet allocated */
	if (clntinfo.user) {
		free(clntinfo.user);
	}
	if (clntinfo.pass) {
		free(clntinfo.pass);
	}
	if (clntinfo.destination) {
		free(clntinfo.destination);
	}

	clean_allowedlist();

	if (clntinfo.boundsocket_list) {
		for (i = 0; i < clntinfo.boundsocket_niface; i++) {
	        	close(clntinfo.boundsocket_list[i]);
		}
		free(clntinfo.boundsocket_list);
	}
	close(clntinfo.clientsocket);
	close(clntinfo.serversocket);
}

