/**********************************************************************/
/*                                                                    */
/* RexxMail installation program.                                     */
/*                                                                    */
/**********************************************************************/
/*                                                                    */
/* This program forms part of the RexxMail package, and may not be    */
/* distributed separately.                                            */
/*                                                                    */
/**********************************************************************/
/*                                                                    */
/* The latest version of RexxMail can be found at                     */
/*               www.degeus.com/rexx                                  */
/*                                                                    */
/**********************************************************************/
/*                                                                    */
/* This program is released under the terms of the GNU license, see   */
/*               www.gnu.org/copyleft/gpl.html                        */
/*                                                                    */
/**********************************************************************/
/*                                                                    */
/* (c) 2000-2003 Marcus de Geus                                       */
/*               marcus@degeus.com                                    */
/*               www.degeus.com                                       */
/*                                                                    */
/**********************************************************************/
/*                                                                    */
/* History:                                                           */
/*                                                                    */
/* 2001-04-03 : Corrected "chaining error" bug when using CMD.EXE.    */
/* 2001-05-11 : Complete rebuild; now includes update intelligence.   */
/* 2001-06-09 : Added session title in parameters field of            */
/*              <REXXMAIL_DISPATCH> object.                           */
/* 2001-06-14 : Bugfix: CMD.EXE's 'move' cannot overwrite files.      */
/*              Added RexxMail build numbers to log entries.          */
/* 2001-06-17 : No concurrent view for mail folder objects            */
/*              (CCVIEW=NO) to prevent multiple instances of folder   */
/*              objects on some systems when default folder object    */
/*              open behaviour is set to 'Create new window'.         */
/* 2001-06-20 : Added Luc van Bogaert's folder background bitmap.     */
/* 2001-07-25 : Changed Configuration folder settings; added shadow.  */
/* 2001-09-03 : Minor cosmetic changes.                               */
/* 2001-09-15 : A few minor changes in connection with the new        */
/*              command-line help mechanism.                          */
/* 2001-10-05 : Removed 'Update info files?' option.                  */
/* 2001-12-03 : Changed 'Info' program object into URL.               */
/* 2001-12-15 : Added OS and REXX DLL level checks.                   */
/* 2002-03-02 : Added CWMailFile class installation.                  */
/* 2002-03-13 : Bugfix: the use of the xcopy command to copy the      */
/*              program files caused the destinaton dir title to      */
/*              change, causing path problems. Back to simple copy.   */
/* 2002-12-23 : Changed "Dispatch" to "Collect & Send".               */
/*              Added "Send" object.                                  */
/*              Added "Collect" object.                               */
/*              Added "OK/not OK to send" object.                     */
/* 2003-09-11 : Added "Add Addresses to All Open Messages" object.    */
/*              Added "Attach to All Messages in Out Folder" object.  */
/*              Added "Attach to Open Message Folders" object.        */
/*              Added "Open Attachments Folder" object.               */
/*              Added "Close Attachments Folder" object.              */
/*              Added "Collect Mail Interactively" object.            */
/*              Added "RexxMail Reference Guide" object.              */
/*              Added "RexxMail Tutorial" object.                     */
/*              Removed "RexxMail Help" object.                       */
/*              Removed superfluous user queries.                     */
/*                                                                    */
/**********************************************************************/

/**********************************************************************/
/* Some general bits                                                  */
/**********************************************************************/

signal on Error  /* handles general error condition */
signal on Failure  /* handles external program errors */
signal on Halt  /* handles halt condition */
signal on NoValue  /* handles initialization errors */
signal on Syntax  /* handles syntax errors */

signal off NotReady  /* we handle I/O errors */

/**********************************************************************/
/* Define the build number                                            */
/**********************************************************************/

Global.Build = 20040119.094720  /* define the build number */

/**********************************************************************/
/* Define the program directory                                       */
/**********************************************************************/

parse source OS . ProgSpec  /* get the program name */
ProgDir = strip(filespec('D',ProgSpec)||filespec('P',ProgSpec),'T','\')  /* get the program directory */

/**********************************************************************/
/* Start a log file                                                   */
/**********************************************************************/

Global.LogFile = ProgDir||'\install.log'  /* define the log file name */
call WriteLog date('S')||' '||time('N')||' : Installation started.'  /* report */

/**********************************************************************/
/* Make sure this is OS/2                                             */
/**********************************************************************/

if (OS >< 'OS/2') then  /* if the OS is not OS/2 */
do
 call WriteLog '  ERROR: this program requires OS/2'  /* report */
 call Halt  /* and quit */
end

/**********************************************************************/
/* Check for the REXX utilities library                               */
/**********************************************************************/

if (\LoadRexxUtils()) then  /* if we cannot use or load the Rexx utility functions */
do
 call WriteLog '  ERROR: Cannot load Rexx utility functions.'  /* report */
 call Halt  /* and quit */
end

/**********************************************************************/
/* Make sure we have the utilities level we need                      */
/**********************************************************************/

if (rxfuncquery('sysutilversion')) then  /* if we cannot find the SysUtilVersion function */
do
 RexxVersionOK = 0  /* all is not well */
end
else  /* if we can find the SysUtilVersion function */
do
 RexxVersionOK = (sysutilversion() >= '2.00')  /* if the utilities level is at least 2.00, all is well */
end

if (\RexxVersionOK) then  /* if we do not have the right REXX utilities version */
do
 call WriteLog '  ERROR: REXXUTIL.DLL is not at the required level'  /* report */
 call Halt  /* and quit */
end

/**********************************************************************/
/* Prompt for action                                                  */
/**********************************************************************/

call StartExplain  /* explain to the user what's going to happen */

if (\UserQuery('Continue?')) then  /* if the user gives a negative reply */
do
 call WriteLog '         Installation aborted by user.'  /* report */
 call Halt  /* quit */
end

/**********************************************************************/
/* See what the current update status looks like                      */
/**********************************************************************/

LocFile = ProgDir||'\location.txt'  /* the user location file */
NewInstall = (stream(LocFile,'C','QUERY EXISTS') = '')  /* if we have no location file, it is a new installation */

if (NewInstall) then  /* if we have a new installation */
do

 call WriteLog '         Creating new installation for build '||Global.Build||'.'  /* report */
 call InstallExplain  /* explain what's going to happen */

 if (UserQuery('Continue?')) then  /* if the user reply is positive */
 do

  if (\DoInstall(NewInstall)) then  /* if we cannot install the program */
  do
   call Halt  /* quit */
  end

 end
 else  /* if the user reply was negative */
 do
  call WriteLog '         Program installation refused by user.'  /* report */
 end

end
else  /* if it is not a new installation */
do

 if (stream(ProgDir||'\update\program\location.txt','C','QUERY EXISTS') = '') then  /* if we have already installed or updated the program */
 do

  Message = 'There are no program components to install or update.'  /* the text to report */
  call WriteLog '         '||Message  /* report */

  call ClearScreen  /* clear the screen */

  call lineout 'StdOut:',Message  /* report */
  call lineout 'StdOut:',''  /* empty line */
  call lineout 'StdOut:',''  /* empty line */
  call charout 'StdOut:','Press any key to continue'  /* report */
  call sysgetkey 'NOECHO'  /* wait for a key */

 end
 else  /* if we still have an update to do */
 do

  call WriteLog '         Updating existing installation to build '||Global.Build||'.'  /* report */
  call InstallExplain  /* explain what's going to happen */
  call UpdateExplain  /* explain to the user */

  if (UserQuery('Continue?')) then  /* if the user reply is positive */
  do

   if (\DoInstall(NewInstall)) then  /* if we cannot install the program */
   do
    call Halt  /* quit */
   end

  end
  else  /* if the user reply was negative */
  do
   call WriteLog '         Program update refused by user.'  /* report */
  end

 end

end

if (stream(LocFile,'C','QUERY EXISTS') = '') then  /* if we have no location file */
do
 call WriteLog '  ERROR: Missing location file.'  /* report */
 call Halt  /* and quit */
end

parse arg UserName  /* look for a user name on the command line */

if (UserName = '') then  /* if there is none */
do

 UserName = value('USER',,'OS2ENVIRONMENT')  /* look for a user name in the OS/2 environment */

 if (UserName = '') then  /* if no user name is found */
 do
  UserName = 'DEFAULT'  /* use the default value */
 end

end

if (\GotUser(LocFile,UserName)) then  /* if we do not find this user in the location file */
do

 call WriteLog '         User "'||UserName||'" is not in "'||LocFile||'"; installing new user.'  /* report */
 call UserInstallExplain  /* explain to the user */

 if (UserQuery('Install mail folders for user "'||UserName||'"?')) then  /* if the user reply is positive */
 do

  if (\InstallUser()) then  /* if we cannot install a new user */
  do
   call Halt  /* quit */
  end

 end
 else  /* if the user reply is negative */
 do
  call WriteLog '         New user installation refused by user.'  /* report */
 end

end
else  /* if we found this user in the location file */
do
 call WriteLog '         User "'||UserName||'" is in "'||LocFile||'"; no new user installation required.'  /* report */
end

call WriteLog date('S')||' '||time('N')||' : Installation completed.'  /* report */

call ClearScreen  /* clear the screen */

call lineout 'StdOut:','Installation completed.'  /* report */
call lineout 'StdOut:',''  /* empty line */
call lineout 'StdOut:','For further instructions, please refer to the RexxMail Reference Guide.'  /* report */

call Quit  /* that's all, folks! */

/**********************************************************************/
DoInstall:  /* optionally (re)installs all the bits and pieces */
/**********************************************************************/

parse arg NewInstall  /* get the argument */

if (NewInstall) then  /* if this is a new installation */
do

 call ClearScreen  /* clear the screen */

 ScreenSize = systextscreensize()  /* get the screen dimensions */
 ScreenCols = word(ScreenSize,2)  /* the screen width */

 call charout 'StdOut:',copies('_',ScreenCols)  /* line */
 call lineout 'StdOut:','IMPORTANT:'  /* report */
 call charout 'StdOut:',copies('',ScreenCols)  /* line */
 call charout 'StdOut:',copies('_',ScreenCols)  /* line */
 call lineout 'StdOut:','Do not install RexxMail in a new directory if you already have a previous'  /* report */
 call lineout 'StdOut:','RexxMail installation in a different program directory on your system!'  /* report */
 call charout 'StdOut:',copies('',ScreenCols)  /* line */

 if (\UserQuery('Continue?')) then  /* if the user reply is negative */
 do
  return 0  /* quit with an error */
 end

end
else  /* if this is not a new installation */
do

 call sysfiledelete ProgDir||'\update\program\location.txt'  /* get rid of the location.txt template */

 if (sysdestroyobject(ProgDir||'\help')) then  /* if we can get rid of any remaining old help dir */
 do
  call WriteLog '         Removed obsolete Help folder from program directory.'  /* report */
 end

 if (sysdestroyobject('<REXXMAIL_HELP>')) then  /* if we can get rid of any remaining old help object */
 do
  call WriteLog '         Removed obsolete Help object.'  /* report */
 end

end

if (\InstallProgram()) then  /* if we have a negative result */
do
 return 0  /* quit with an error */
end

if (\InstallIcons()) then  /* if we cannot install the icons */
do
 return 0  /* quit with an error */
end

if (\InstallObjects()) then  /* if we cannot install the program objects */
do
 return 0  /* quit with an error */
end

call sysdestroyobject ProgDir||'\update\icons'  /* get rid of the icons update folder */
call sysdestroyobject ProgDir||'\update\program'  /* get rid of the program files update folder */

return 1  /* end of DoInstall */

/**********************************************************************/
InstallProgram:  /* (re)installs the program files */
/**********************************************************************/

if (\CheckDir(ProgDir||'\update\program')) then  /* if the program dir is not OK */
do
 return 0  /* return with an error */
end

address cmd 'copy '||ProgDir||'\update\program\* '||ProgDir||' > NUL'  /* copy the program files */

if (RC = 0) then  /* if no error occurred */
do
 call WriteLog '         Copied RexxMail program files to "'||ProgDir||'".'  /* report */
end
else  /* if we cannot copy the files */
do
 call WriteLog '         Cannot copy RexxMail program files to "'||ProgDir||'".'  /* report */
 return 0  /* return with no success */
end

GotCWMFC = 0  /* we have no CWMailFile class yet */

call sysqueryclasslist('ClassList.')  /* get the class list */

do Index = 1 to ClassList.0  /* look at each entry */

 if (word(ClassList.Index,1) = 'CWMailFile') then  /* if it is what we want */
 do
  call WriteLog '         (The CWMailFileClass is already registered.)'  /* report */
  GotCWMFC = 1  /* we have a CWMailFile class */
 end

end

if (\GotCWMFC) then  /* if the CWMailFile class is not registered */
do

 if (MakeFolder('CWMailFileClass',ProgDir)) then  /* if we can create a CWMailFileClass folder in the program dir */
 do

  address cmd 'copy '||ProgDir||'\update\program\cwmailfileclass\* '||ProgDir||'\cwmailfileclass > NUL'  /* copy the CWMailFile class files */

  if (RC = 0) then  /* if no error occurred */
  do
   call WriteLog '         Copied CWMailFile class files to "'||ProgDir||'\CWMailFileClass".'  /* report */
  end
  else  /* if we cannot copy the files */
  do
   call WriteLog '         Cannot copy CWMailFile class files to "'||ProgDir||'\CWMailFileClass".'  /* report */
   return 0  /* return with no success */
  end

  if (MakeFolder('Sources',ProgDir||'\CWMailFileClass')) then  /* if we can create a Sources folder in the CWMailFileClass dir */
  do

   address cmd 'copy '||ProgDir||'\update\program\cwmailfileclass\sources\* '||ProgDir||'\cwmailfileclass\sources > NUL'  /* copy the source files */

   if (RC = 0) then  /* if no error occurred */
   do
    call WriteLog '         Copied CWMailFile source files to "'||ProgDir||'\CWMailFileClass\Sources".'  /* report */
   end
   else  /* if we cannot copy the files */
   do
    call WriteLog '         Cannot copy CWMailFile source files to "'||ProgDir||'\CWMailFileClass\Sources".'  /* report */
    return 0  /* return with no success */
   end

  end

 end

 call CWMailFileExplain  /* explain what's going to happen */

 if (UserQuery('Register the CWMailFile class?')) then  /* if the user wants to register the CWMailFile class */
 do

  if (sysregisterobjectclass('CWMailFile',ProgDir||'\cwmailfileclass\rexxmail.dll')) then  /* if we can register the class */
  do
   call WriteLog '         The CWMailFile class has been registered.'  /* report */
  end
  else  /* if we cannot register the class */
  do
   call WriteLog '  ERROR: Cannot register the CWMailFile class.'  /* report */
   return 0  /* quit with an error */
  end

 end

end

return 1  /* end of InstallProgram */

/**********************************************************************/
InstallIcons:  /* installs the icons and background bitmap */
/**********************************************************************/

if (\CheckDir(ProgDir||'\update\icons')) then  /* if the icons dir does not check out OK */
do
 return 0  /* return with an error */
end

call sysfiletree ProgDir||'\update\icons\*','PaperDirs.','DO'  /* look for subdirectories of the new icons dir */

do Index = 1 to PaperDirs.0  /* for each one we find */

 if (\CheckDir(PaperDirs.Index)) then  /* if the dir does not check out OK */
 do
  return 0  /* return with an error */
 end

end

call sysdestroyobject ProgDir||'\icons'  /* get rid of any old icons dir */

if (\sysmoveobject(ProgDir||'\update\icons',ProgDir)) then  /* if we cannot move the icons folder to the prog dir */
do
 call WriteLog '  ERROR: Cannot move new icons folder to "'||Progdir||'".'  /* report */
 return 0  /* return with an error */
end

call WriteLog '         Moved new icons folder to "'||ProgDir||'".'  /* report */

return 1  /* end of InstallIcons */

/**********************************************************************/
InstallObjects:  /* installs the program objects */
/**********************************************************************/

/**********************************************************************/
/* Note order of calling MakeObject to make sure the program/file     */
/* associations are created in the correct order of preference.       */
/**********************************************************************/

Location = ProgDir||'\Objects'  /* the objects will be created in the Objects subdir of the program dir */

if (\MakeDir(Location)) then  /* if we cannot make the dir if necessary */
do
 return 0  /* return with an error */
end

ExeName = ProgDir||'\rexxmail.cmd'  /* the following objects use rexxmail.cmd in the program directory */

Settings = 'MINIMIZED=YES;'||,  /* start the following programs minimized */
           'NOAUTOCLOSE=NO;'||,  /* close the program session after completion */
           'PROGTYPE=WINDOWABLEVIO;'||,  /* use a VIO (OS/2) window */
           'STARTUPDIR='||ProgDir  /* start the program in the RexxMail program directory */

AssocType = 'Mail Message In,Mail Message Sent'  /* the associations to use for the next few objects */

Title = 'View Message'  /* the title to use */
ObjectID = '<REXXMAIL_VIEW>'  /* the object id to use */
Parameters = '/View'  /* the RexxMail switch to use */
IconFile = ProgDir||'\Icons\MailOpen.ICO'  /* the icon file to use */
call MakeObject Title,Location,ExeName,Parameters,AssocType,ObjectID,IconFile,Settings  /* make the object */

Title = 'View Raw Message'  /* the title to use */
ObjectID = '<REXXMAIL_RAW>'  /* the object id to use */
Parameters = '/Raw'  /* the RexxMail switch to use */
IconFile = ProgDir||'\Icons\MailOpen.ICO'  /* the icon file to use */
call MakeObject Title,Location,ExeName,Parameters,AssocType,ObjectID,IconFile,Settings  /* make the object */

AssocType = 'Mail Message In'  /* the association to use for the next few objects */

Title = 'Forward'  /* the title to use */
ObjectID = '<REXXMAIL_FORWARD>'  /* the object id to use */
Parameters = '/Forward'  /* the RexxMail switch to use */
IconFile = ProgDir||'\Icons\MailForw.ICO'  /* the icon file to use */
call MakeObject Title,Location,ExeName,Parameters,AssocType,ObjectID,IconFile,Settings  /* make the object */

Title = 'Reply'  /* the title to use */
ObjectID = '<REXXMAIL_REPLY>'  /* the object id to use */
Parameters = '/Reply'  /* the RexxMail switch to use */
IconFile = ProgDir||'\Icons\MailRepl.ICO'  /* the icon file to use */
call MakeObject Title,Location,ExeName,Parameters,AssocType,ObjectID,IconFile,Settings  /* make the object */

Title = 'Reply to All'  /* the title to use */
ObjectID = '<REXXMAIL_REPLYTOALL>'  /* the object id to use */
Parameters = '/ReplyToAll'  /* the RexxMail switch to use */
IconFile = ProgDir||'\Icons\MailRepl.ICO'  /* the icon file to use */
call MakeObject Title,Location,ExeName,Parameters,AssocType,ObjectID,IconFile,Settings  /* make the object */

Title = 'Reply to Sender'  /* the title to use */
ObjectID = '<REXXMAIL_REPLYTOSENDER>'  /* the object id to use */
Parameters = '/ReplyToSender'  /* the RexxMail switch to use */
IconFile = ProgDir||'\Icons\MailRepl.ICO'  /* the icon file to use */
call MakeObject Title,Location,ExeName,Parameters,AssocType,ObjectID,IconFile,Settings  /* make the object */

Title = 'Reply to Recipients'  /* the title to use */
ObjectID = '<REXXMAIL_REPLYTORECIPIENTS>'  /* the object id to use */
Parameters = '/ReplyToRecipients'  /* the RexxMail switch to use */
IconFile = ProgDir||'\Icons\MailRepl.ICO'  /* the icon file to use */
call MakeObject Title,Location,ExeName,Parameters,AssocType,ObjectID,IconFile,Settings  /* make the object */

AssocType = 'Mail Message Out'  /* the association to use for the next few objects */

Title = 'Edit Message'  /* the title to use */
ObjectID = '<REXXMAIL_EDIT>'  /* the object id to use */
Parameters = '/Edit'  /* the RexxMail switch to use */
IconFile = ProgDir||'\Icons\MailOpen.ICO'  /* the icon file to use */
call MakeObject Title,Location,ExeName,Parameters,AssocType,ObjectID,IconFile,Settings  /* make the object */

Title = 'ASCII/Quoted-Printable'  /* the title to use */
ObjectID = '<REXXMAIL_SETASCIIQP>'  /* the object id to use */
Parameters = '/SetASCIIQP'  /* the RexxMail switch to use */
IconFile = ProgDir||'\Icons\MailOpen.ICO'  /* the icon file to use */
call MakeObject Title,Location,ExeName,Parameters,AssocType,ObjectID,IconFile,Settings  /* make the object */

Title = 'OK/Not OK to Send'  /* the title to use */
ObjectID = '<REXXMAIL_SETOKNOTOKTOSEND>'  /* the object id to use */
Parameters = '/SetOKnotOKtoSend'  /* the RexxMail switch to use */
IconFile = ProgDir||'\Icons\MailOpen.ICO'  /* the icon file to use */
call MakeObject Title,Location,ExeName,Parameters,AssocType,ObjectID,IconFile,Settings  /* make the object */

AssocType = 'Mail Message In,Mail Message Out,Mail Message Sent'  /* the associations to use for the next few objects */

Title = 'Addresses'  /* the title to use */
ObjectID = '<REXXMAIL_ADDRESS>'  /* the object id to use */
Parameters = '/Addresses'  /* the RexxMail switch to use */
IconFile = ProgDir||'\Icons\MailAddr.ICO'  /* the icon file to use */
call MakeObject Title,Location,ExeName,Parameters,AssocType,ObjectID,IconFile,Settings  /* make the object */

Title = 'Add Addresses to All Open Messages'  /* the title to use */
ObjectID = '<REXXMAIL_ADDADDRESSES>'  /* the object id to use */
Parameters = '/AddAddresses'  /* the RexxMail switch to use */
IconFile = ProgDir||'\Icons\MailAddr.ICO'  /* the icon file to use */
call MakeObject Title,Location,ExeName,Parameters,AssocType,ObjectID,IconFile,Settings  /* make the object */

Title = 'Open Attachments Folder'  /* the title to use */
ObjectID = '<REXXMAIL_OPENATTACH>'  /* the object id to use */
Parameters = '/OpenAttach'  /* the RexxMail switch to use */
IconFile = ProgDir||'\Icons\MailOpen.ICO'  /* the icon file to use */
call MakeObject Title,Location,ExeName,Parameters,AssocType,ObjectID,IconFile,Settings  /* make the object */

Title = 'Close Attachments Folder'  /* the title to use */
ObjectID = '<REXXMAIL_CLOSEATTACH>'  /* the object id to use */
Parameters = '/CloseAttach'  /* the RexxMail switch to use */
IconFile = ProgDir||'\Icons\MailOpen.ICO'  /* the icon file to use */
call MakeObject Title,Location,ExeName,Parameters,AssocType,ObjectID,IconFile,Settings  /* make the object */

AssocType = ''  /* no associations for the next few objects */

Title = 'Export Addresses'  /* the title to use */
ObjectID = '<REXXMAIL_EXPORT>'  /* the object id to use */
Parameters = '/Export'  /* the RexxMail switch to use */
IconFile = ProgDir||'\Icons\MailAdEx.ICO'  /* the icon file to use */
call MakeObject Title,Location,ExeName,Parameters,AssocType,ObjectID,IconFile,Settings  /* make the object */

Title = 'Import Addresses'  /* the title to use */
ObjectID = '<REXXMAIL_IMPORT>'  /* the object id to use */
Parameters = '/Import'  /* the RexxMail switch to use */
IconFile = ProgDir||'\Icons\MailAdIm.ICO'  /* the icon file to use */
call MakeObject Title,Location,ExeName,Parameters,AssocType,ObjectID,IconFile,Settings  /* make the object */

Title = 'Open Mail Folders'  /* the title to use */
ObjectID = '<REXXMAIL_OPEN>'  /* the object id to use */
Parameters = '/Open %'  /* the RexxMail switch to use */
IconFile = ProgDir||'\Icons\MailOpen.ICO'  /* the icon file to use */
call MakeObject Title,Location,ExeName,Parameters,AssocType,ObjectID,IconFile,Settings  /* make the object */

Title = 'Attach to All Messages in Out Folder'  /* the title to use */
ObjectID = '<REXXMAIL_ATTACHTOALLINOUT>'  /* the object id to use */
Parameters = '/AttachToAllInOut'  /* the RexxMail switch to use */
IconFile = ProgDir||'\Icons\MailOpen.ICO'  /* the icon file to use */
call MakeObject Title,Location,ExeName,Parameters,AssocType,ObjectID,IconFile,Settings  /* make the object */

Title = 'Attach to Open Message Folders'  /* the title to use */
ObjectID = '<REXXMAIL_ATTACHTOOPEN>'  /* the object id to use */
Parameters = '/AttachToOpen'  /* the RexxMail switch to use */
IconFile = ProgDir||'\Icons\MailOpen.ICO'  /* the icon file to use */
call MakeObject Title,Location,ExeName,Parameters,AssocType,ObjectID,IconFile,Settings  /* make the object */

Settings = 'MINIMIZED=NO;'||,  /* start the program visible */
           'NOAUTOCLOSE=NO;'||,  /* close the program session after completion */
           'PROGTYPE=WINDOWABLEVIO;'||,  /* use a VIO (OS/2) window */
           'STARTUPDIR='||ProgDir  /* start the program in the RexxMail program directory */

ExeName = '*'  /* the following objects use the default, i.e. the command processor */

Title = 'Collect & Send Mail'  /* the title to use */
ObjectID = '<REXXMAIL_DISPATCH>'  /* the object id to use */
Parameters = '/c start /c "Collect and Send Mail" '||ProgDir||'\rexxmail.cmd /collect /sendready'  /* start a new session running RexxMail.CMD */
IconFile = ProgDir||'\Icons\MailDisp.ICO'  /* the icon file to use */
call MakeObject Title,Location,ExeName,Parameters,AssocType,ObjectID,IconFile,Settings  /* make the object */

Title = 'Collect Mail'  /* the title to use */
ObjectID = '<REXXMAIL_COLLECT>'  /* the object id to use */
Parameters = '/c start /c "Collect Mail" '||ProgDir||'\rexxmail.cmd /collect %'  /* start a new session running RexxMail.CMD */
IconFile = ProgDir||'\Icons\MailOpen.ICO'  /* the icon file to use */
call MakeObject Title,Location,ExeName,Parameters,AssocType,ObjectID,IconFile,Settings  /* make the object */

Title = 'Collect Mail Interactively'  /* the title to use */
ObjectID = '<REXXMAIL_COLLECTPOP3INTERACTIVE>'  /* the object id to use */
Parameters = '/c start /c "Collect Mail Interactively" '||ProgDir||'\rexxmail.cmd /pop3interactive /collect %'  /* start a new session running RexxMail.CMD */
IconFile = ProgDir||'\Icons\MailOpen.ICO'  /* the icon file to use */
call MakeObject Title,Location,ExeName,Parameters,AssocType,ObjectID,IconFile,Settings  /* make the object */

AssocType = 'Mail Message Out'  /* the association to use for the next object */

Title = 'Send'  /* the title to use */
ObjectID = '<REXXMAIL_SEND>'  /* the object id to use */
Parameters = '/c start /c "Send Mail" '||ProgDir||'\rexxmail.cmd /send %*'  /* start a new session running RexxMail.CMD */
IconFile = ProgDir||'\Icons\MailOpen.ICO'  /* the icon file to use */
call MakeObject Title,Location,ExeName,Parameters,AssocType,ObjectID,IconFile,Settings  /* make the object */

AssocType = ''  /* no associations for the next objects */

ExeName = ProgDir||'\foldutil.cmd'  /* the following object uses foldutil.cmd in the program directory */

Title = 'User Folders Utility'  /* the title to use */
ObjectID = '<REXXMAIL_LOCK>'  /* the object id to use */
Parameters = ''  /* no parameters */
IconFile = ProgDir||'\Icons\MailUtil.ICO'  /* the icon file to use */
call MakeObject Title,Location,ExeName,Parameters,AssocType,ObjectID,IconFile,Settings  /* make the object */

Settings = 'PROGTYPE=PM;'  /* use a PM session for the following objects */

ExeName = 'view.exe'  /* the following objects use the info viewer program */

Title = 'RexxMail Reference Guide'  /* the title to use */
ObjectID = '<REXXMAIL_REFERENCE>'  /* the object id to use */
Parameters = ProgDir||'\RexxMail_Reference_Guide.INF'  /* use this file */
IconFile = ProgDir||'\Icons\Mail_Ref.ICO'  /* the icon file to use */
call MakeObject Title,Location,ExeName,Parameters,AssocType,ObjectID,IconFile,Settings  /* make the object */

Title = 'RexxMail Tutorial'  /* the title to use */
ObjectID = '<REXXMAIL_TUTORIAL>'  /* the object id to use */
Parameters = ProgDir||'\RexxMail_Tutorial.INF'  /* use this file */
IconFile = ProgDir||'\Icons\Mail_Tut.ICO'  /* the icon file to use */
call MakeObject Title,Location,ExeName,Parameters,AssocType,ObjectID,IconFile,Settings  /* make the object */

/**********************************************************************/
/* Install a CWMailFile class folder                                  */
/**********************************************************************/

Settings = 'ALWAYSSORT=YES;'||,  /* keep the folder sorted */
           'DEFAULTVIEW=DETAILS;'||,  /* the default view of the main directory is details */
           'DETAILSCLASS=CWMailFile;'||,  /* use the CWMailFile details class if possible */
           'SORTCLASS=CWMailFile;'||,  /* use the CWMailFile sorting class if possible */
           'OBJECTID=<REXXMAIL_FOLDER>;'||,  /* the object ID */
           'TEMPLATE=YES;'||,  /* make it a template object */
           'BACKGROUND='||ProgDir||'\icons\folderbg.bmp,N,,I,255 255 255;'  /* use the single background bitmap */

call MakeFolder 'Mail Folder',Location,Settings  /* try to make the Mail Folder template */

return 1  /* end of InstallObjects */

/**********************************************************************/
InstallUser:  /* Install a new user */
/**********************************************************************/

MailLoc = GetLoc(ProgDir)  /* get the location for the main Rexxmail mail folder */

if (MailLoc = '*') then  /* if we have no location value */
do
 call WriteLog '         User refused creation of main folder'  /* report */
 return 1  /* quit with no error */
end

Settings = 'ALWAYSSORT=YES;'||,  /* keep the main directory sorted */
           'CCVIEW=NO;'||,  /* resurface existing folder view */
           'WORKAREA=YES;'||,  /* make the main directory a work area */
           'DEFAULTVIEW=ICON;'||,  /* the default view of the main directory is icons */
           'ICONVIEW=NORMAL,NONGRID;'||,  /* use standard view for these folders */
           'NORENAME=YES;'||,  /* the next folders cannot be renamed */
           'NOTVISIBLE=NO;'||,  /* the next folders are visible (in the main folder) */
           'DEFAULTSORT=-2;'||,  /* sort the next folder's contents by name */
           'BACKGROUND='||ProgDir||'\icons\folderbg.bmp,N,,I,255 255 255;'  /* use the single background bitmap */

if (\syssetobjectdata(MailLoc,Settings)) then  /* if we cannot change the main location settings */
do
 call WriteLog '  ERROR: Cannot attach settings to directory "'||Location||'".'  /* report */
end

if (\CheckDir(ProgDir||'\update\configuration')) then  /* if the original files dir is not OK */
do
 return 0  /* return with an error */
end

if (\CheckDir(ProgDir||'\update\configuration\lists')) then  /* if the original files dir is not OK */
do
 return 0  /* return with an error */
end

Settings = 'ALWAYSSORT=YES;'||,  /* always sort the next folders */
           'CCVIEW=NO;'||,  /* resurface existing folder view */
           'WORKAREA=NO;'||,  /* do not make the next folders work areas */
           'DEFAULTVIEW=ICON;'||,  /* use icon views for the next folders */
           'ICONVIEW=NORMAL,NONGRID;'||,  /* use standard view for these folders */
           'NORENAME=YES;'||,  /* the next folders cannot be renamed */
           'NOTVISIBLE=NO;'||,  /* the next folders are visible (in the main folder) */
           'DEFAULTSORT=-2;'||,  /* sort the next folders' contents by name */
           'BACKGROUND='||ProgDir||'\icons\folderbg.bmp,N,,I,255 255 255;'  /* use the single background bitmap */

AddSettings = 'ICONFILE='||ProgDir||'\icons\mail_acc.ico;'  /* additional settings for the Accessories folder */

if (\MakeFolder('Accessories',MailLoc,Settings||AddSettings)) then  /* if we cannot make the Accessories folder */
do
 return 0  /* return with an error */
end

Settings = 'ALWAYSSORT=YES;'||,  /* always sort the next folders */
           'CCVIEW=NO;'||,  /* resurface existing folder view */
           'WORKAREA=YES;'||,  /* make the next folders work areas */
           'DEFAULTVIEW=ICON;'||,  /* use icon views for the next folders */
           'ICONVIEW=NORMAL,FLOWED;'||,  /* use flowed (multi-column) view for the folders */
           'NORENAME=YES;'||,  /* the next folders cannot be renamed */
           'NOTVISIBLE=YES;'||,  /* the next folders are invisible (in the main folder) */
           'DEFAULTSORT=-2;'||,  /* sort the next folders' contents by name */
           'BACKGROUND='||ProgDir||'\icons\folderbg.bmp,N,,I,255 255 255;'  /* use the single background bitmap */

AddSettings = 'ICONFILE='||ProgDir||'\icons\mail_cfg.ico;'  /* additional settings for the Configuration folder */

if (\MakeFolder('Configuration',MailLoc,Settings||AddSettings)) then  /* if we cannot make the Configuration folder */
do
 return 0  /* return with an error */
end

AddSettings = 'ICONFILE='||ProgDir||'\icons\mailaddr.ico;'  /* additional settings for the Addresses folder */

if (\MakeFolder('Addresses',MailLoc,Settings||AddSettings)) then  /* if we cannot make the Addresses folder */
do
 return 0  /* return with an error */
end

if (\MakeFolder('Temp',MailLoc,Settings)) then  /* if we cannot make the Temp folder, no special icon */
do
 return 0  /* return with an error */
end

if (\MakeFolder('In',MailLoc,Settings)) then  /* if we cannot make the In folder, no special icon */
do
 return 0  /* return with an error */
end

if (\MakeFolder('In Archive',MailLoc,Settings)) then  /* if we cannot make the In Archive folder, no special icon */
do
 return 0  /* return with an error */
end

if (\MakeFolder('Out',MailLoc,Settings)) then  /* if we cannot make the Out folder, no special icon */
do
 return 0  /* return with an error */
end

if (\MakeFolder('Out Archive',MailLoc,Settings)) then  /* if we cannot make the Out Archive folder, no special icon */
do
 return 0  /* return with an error */
end

Settings = 'ALWAYSSORT=YES;'||,  /* always sort the next folders */
           'CCVIEW=NO;'||,  /* resurface existing folder view */
           'WORKAREA=YES;'||,  /* make the next folders work areas */
           'DEFAULTVIEW=ICON;'||,  /* use icon views for the next folders */
           'ICONVIEW=NORMAL,FLOWED;'||,  /* use flowed (multi-column) view for the folders */
           'NORENAME=NO;'||,  /* the next folders can be renamed */
           'NOTVISIBLE=NO;'||,  /* the next folders are visible */
           'DEFAULTSORT=-2;'||,  /* sort the next folders' contents by name */
           'BACKGROUND='||ProgDir||'\icons\folderbg.bmp,N,,I,255 255 255;'  /* use the single background bitmap */

if (\MakeFolder('Lists',MailLoc||'\Configuration',Settings)) then  /* if we cannot make the Lists folder */
do
 return 0  /* return with an error */
end

if (\MakeFolder('Read',MailLoc||'\In Archive',Settings)) then  /* if we cannot make the Read folder */
do
 return 0  /* return with an error */
end

if (\MakeFolder('Sent',MailLoc||'\Out Archive',Settings)) then  /* if we cannot make the Sent folder */
do
 return 0  /* return with an error */
end

if (\MakeFolder('RexxMail',MailLoc||'\In Archive',Settings)) then  /* if we cannot make the incoming RexxMail folder */
do
 return 0  /* return with an error */
end

if (\MakeFolder('RexxMail',MailLoc||'\Out Archive',Settings)) then  /* if we cannot make the outgoing RexxMail folder */
do
 return 0  /* return with an error */
end

UpdateFile = ProgDir||'\settupda.txt'  /* the name of the settings update template file */
SettingsFile = MailLoc||'\configuration\settings.txt'  /* the name of the new user settings file */

if (stream(UpdateFile,'C','OPEN READ') >< 'READY:') then  /* if we cannot open the settings update template file */
do
 call WriteLog '  ERROR: Cannot open settings template file "'||UpdateFile||'".'  /* report */
 return 0  /* return with an error */
end

if (stream(SettingsFile,'C','OPEN') >< 'READY:') then  /* if we cannot open the new settings file */
do
 call WriteLog '  ERROR: Cannot open user settings file "'||SettingsFile||'".'  /* report */
 return 0  /* return with an error */
end

WatchNext = 0  /* nothing doing yet */

do while (lines(UpdateFile))  /* as long as we have lines left in the update template file */

 InLine = linein(UpdateFile)  /* get the next line */

 if (WatchNext) then  /* if we have to watch this one */
 do

  if (pos(' = ',InLine) > 0) then  /* if it is a parameter line */
  do
   Inline = subword(InLine,2)  /* get rid of the first word (the comment "#") */
  end

 end

 WatchNext = (InLine = copies('#',72))  /* if the line is a separator, watch the next one */
 call lineout SettingsFile,InLine  /* write the result to the new settings file */

end

call stream UpdateFile,'C','CLOSE'  /* close the template file */
call stream SettingsFile,'C','CLOSE'  /* close the settings file */

call sysfiletree ProgDir||'\update\configuration\*','Files.','FO'  /* look for files in the original files source directory */

do Index = 1 to Files.0  /* take each of the files we found */

 if (syscopyobject(Files.Index,MailLoc||'\Configuration')) then  /* if we can copy it to the Configuration folder */
 do
  call WriteLog '         Copied file "'||Files.Index||'" to "'||MailLoc||'\Configuration".'  /* report */
 end
 else  /* if we cannot copy the file to the Configuration folder */
 do
  call WriteLog '         Cannot copy "'||Files.Index||'" to "'||MailLoc||'\Configuration".'  /* report */
 end

end

ListFile = ProgDir||'\update\configuration\lists\maillist'  /* the mailing list sample file */

if (syscopyobject(ListFile,MailLoc||'\Configuration\Lists')) then  /* if we can copy the maillist sample to the Lists folder */
do
 call WriteLog '         Copied file "'||ListFile||'" to "'||MailLoc||'\Configuration\Lists".'  /* report */
end
else  /* if we cannot copy the file to the Lists folder */
do
 call WriteLog '         Cannot copy "'||ListFile||'" to "'||MailLoc||'\Configuration\Lists".'  /* report */
end

if (stream(LocFile,'C','OPEN WRITE') >< 'READY:') then  /* if we cannot open the location file */
do
 call WriteLog '  ERROR: Cannot write main mail folder location "'||MailLoc||'" to "'||LocFile||'".'  /* report */
 return 0  /* return with an error */
end

call lineout LocFile,d2c(13)||d2c(10)||MailLoc||' = '||UserName||';'  /* write the location and user name to the location file, starting on a new line */
call stream LocFile,'C','CLOSE'  /* close the file */

call WriteLog '         Main mail folder location "'||MailLoc||'" written to "'||LocFile||'".'  /* report */

StartID = '<REXXMAIL_START_FOLDER>'  /* the object ID of the RexxMail Start folder */

if (sysdestroyobject(StartID)) then  /* if we destroyed a previous instance */
do
 call WriteLog '         Existing start folder found and destroyed'  /* report */
end

Settings = 'ALWAYSSORT=YES;'||,  /* always sort the RexxMail Start folder */
           'DEFAULTVIEW=ICON;'||,  /* use icon view in the RexxMail Start folder */
           'OBJECTID='||StartID  /* use this object ID for the RexxMail Start folder */

if (\MakeFolder('RexxMail - Start','<WP_DESKTOP>',Settings)) then  /* if we can make the RexxMail Start folder */
do
 return 0  /* return with an error */
end

if (\MakeShadow('<REXXMAIL_OPEN>',StartID,ProgDir||'\Icons\MailOpen.ICO')) then  /* if we cannot make a shadow of the Open program object */
do
 return 0  /* return with an error */
end

if (\MakeShadow(ProgDir,StartID,ProgDir||'\Icons\MailOpen.ICO')) then  /* if we cannot make a shadow of the program dir */
do
 return 0  /* return with an error */
end

if (\MakeShadow('<REXXMAIL_REFERENCE>',StartID,ProgDir||'\Icons\Mail_Ref.ICO')) then  /* if we cannot make a shadow of the Reference Guide program object */
do
 return 0  /* return with an error */
end

if (\MakeShadow('<REXXMAIL_TUTORIAL>',StartID,ProgDir||'\Icons\Mail_Tut.ICO')) then  /* if we cannot make a shadow of the Tutorial program object */
do
 return 0  /* return with an error */
end

if (\MakeShadow('<REXXMAIL_DISPATCH>',MailLoc,ProgDir||'\Icons\MailDisp.ICO')) then  /* if we cannot make a shadow of the Dispatch program object */
do
 return 0  /* return with an error */
end

if (\MakeShadow('<REXXMAIL_REPLY>',MailLoc,ProgDir||'\Icons\MailRepl.ICO')) then  /* if we cannot make a shadow of the Reply program object */
do
 return 0  /* return with an error */
end

if (\MakeShadow('<REXXMAIL_FORWARD>',MailLoc,ProgDir||'\Icons\MailForw.ICO')) then  /* if we cannot make a shadow of the Forward program object */
do
 return 0  /* return with an error */
end

if (\MakeShadow('<REXXMAIL_ADDRESS>',MailLoc,ProgDir||'\Icons\MailAddr.ICO')) then  /* if we cannot make a shadow of the Add to Addresses program object */
do
 return 0  /* return with an error */
end

if (\MakeShadow('<REXXMAIL_EXPORT>',MailLoc||'\Accessories',ProgDir||'\Icons\MailAdEx.ICO')) then  /* if we cannot make a shadow of the Export Addresses program object */
do
 return 0  /* return with an error */
end

if (\MakeShadow('<REXXMAIL_IMPORT>',MailLoc||'\Accessories',ProgDir||'\Icons\MailAdIm.ICO')) then  /* if we cannot make a shadow of the Import Addresses program object */
do
 return 0  /* return with an error */
end

if (\MakeShadow('<REXXMAIL_REFERENCE>',MailLoc||'\Accessories',ProgDir||'\Icons\Mail_Ref.ICO')) then  /* if we cannot make a shadow of the Reference Guide program object */
do
 return 0  /* return with an error */
end

if (\MakeShadow('<REXXMAIL_TUTORIAL>',MailLoc||'\Accessories',ProgDir||'\Icons\Mail_Tut.ICO')) then  /* if we cannot make a shadow of the Tutorial program object */
do
 return 0  /* return with an error */
end

if (\MakeShadow('<REXXMAIL_LOCK>',MailLoc||'\Accessories',ProgDir||'\Icons\MailUtil.ICO')) then  /* if we cannot make a shadow of the Lock & Unlock Mail Folders program object */
do
 return 0  /* return with an error */
end

if (\MakeShadow('<REXXMAIL_FOLDER>',MailLoc||'\Accessories',ProgDir||'\Icons\MailOpen.ICO')) then  /* if we cannot make a shadow of the Mail Folder object */
do
 return 0  /* return with an error */
end

if (\MakeShadow(MailLoc||'\Configuration',MailLoc||'\Accessories',ProgDir||'\Icons\Mail_Cfg.ICO')) then  /* if we cannot make a shadow of the Configuration folder object */
do
 return 0  /* return with an error */
end

address cmd 'call '||ProgDir||'\rexxmail.cmd /user '||UserName||' /import '||ProgDir||'\update\configuration\addresses.txt'  /* import the default addresses */

if (RC = 0) then  /* if all went well */
do
 call WriteLog '         Created sample message templates in "'||MailLoc||'\Addresses".'  /* report */
end
else  /* if an error occurred */
do
 call WriteLog '  ERROR: Cannot create message templates in "'||MailLoc||'\Addresses".'  /* report */
end

call ClearScreen  /* clear the screen */

call lineout 'StdOut:','The following combination has been added to "location.txt" :'  /* report */
call lineout 'StdOut:',''  /* empty line */
call lineout 'StdOut:','  User name   : "'||UserName||'"'  /* report the user name */
call lineout 'StdOut:','  Main folder : "'||MailLoc||'"'  /* report the main folder location */
call lineout 'StdOut:',''  /* empty line */
call lineout 'StdOut:',''  /* empty line */
call charout 'StdOut:','Press any key to continue'  /* report */
call sysgetkey 'NOECHO'  /* wait for a key */

return 1  /* end of InstallUser */

/**********************************************************************/
GetLoc: procedure expose Global.  /* get a location for the main folder */
/**********************************************************************/

parse arg ProgDir  /* get the argument */

do until (Location >< '')  /* go on until we can leave */

 call ClearScreen  /* clear the screen */

 call lineout 'StdOut:','Enter the full drive, path and name for the main directory:'  /* show a message */
 call lineout 'StdOut:',''  /* empty line */

 parse pull Location  /* get a location from the user */

 if (Location >< '') then  /* if we have something */
 do

  Location = strip(Location,'B',' ')  /* remove excess spaces */
  Location = strip(Location,'B','\')  /* remove any trailing backslash */

  DriveBit = filespec('D',Location)  /* get the drive bit */

  if (DriveBit = '') then  /* if we have no drive bit */
  do
   Location = ''  /* try again */
  end
  else  /* if we do have a drive bit */
  do

   PathBit = filespec('P',Location)||filespec('N',Location)  /* get the path bit (and any name bit) */

   if (PathBit = '') then  /* if we have no path bit */
   do
    Location = ''  /* try again */
   end

  end

  if (Location >< '') then  /* if we still have something */
  do

   call sysfiletree Location,'Dir.','DO'  /* see if the dir exists */

   if (Dir.0 = 1) then  /* if it does */
   do

    Location = Dir.1  /* use it */

    if (translate(Location) = ProgDir) then  /* if the location is the same as the program dir */
    do
     Location = ''  /* make sure we can try again */
    end
    else  /* if it is some other directory */
    do

     call sysfiletree Location||'\*.*','Stuff.','BO'  /* look for files and directories in the location */

     if (Stuff.0 > 0) then  /* if we found anything */
     do
      call lineout 'StdOut:',''  /* empty line */
      call lineout 'StdOut:','"'||Location||'" is not empty;'  /* report */
      call lineout 'StdOut:',''  /* empty line */

      if (\UserQuery('Use the directory anyway?')) then  /* if the reply is negative */
      do
       Location = ''  /* try again */
      end

     end

    end

   end
   else  /* if the location does not exist */
   do

    call lineout 'StdOut:',''  /* empty line */
    call lineout 'StdOut:','"'||Location||'" does not exist;'  /* report */
    call lineout 'StdOut:',''  /* empty line */

    if (\UserQuery('Create a new directory?')) then  /* if the reply is negative */
    do
     Location = ''  /* try again */
    end
    else  /* if we can go ahead */
    do

     NextPos = 3  /* start looking at position 3 (well, actually 4, since we add 1 later) */

     do until (NextPos = 0)  /* go on until we reach the end of the line */

      NextPos = pos('\',Location,(NextPos + 1))  /* get the next backslash position */

      if (NextPos = 0) then  /* if we have reached the end of the line */
      do
       NextDir = Location  /* we have the whole dir spec */
      end
      else  /* if not */
      do
       NextDir = left(Location,(NextPos - 1))  /* get the next dir level */
      end

      call sysfiletree NextDir,'NextDir.','DO'  /* see if the dir exists */

      if (NextDir.0 = 0) then  /* if it does not */
      do

       if (sysmkdir(NextDir) = 0) then  /* if we can create the dir */
       do
        call WriteLog '         Created directory "'||NextDir||'".'  /* report */
       end
       else  /* if not */
       do
        call WriteLog 'ERROR:   Cannot create directory "'||NextDir||'".'  /* report */
        return '*'  /* make sure we quit */
       end

      end

     end

    end

   end

  end

 end

 if (Location = '') then  /* if we have no location, we can try again */
 do

  if (\UserQuery('You have not given a valid location. Try again?')) then  /* if the reply is negative */
  do
   return '*'  /* make sure we quit */
  end

 end

end

return Location  /* end of GetLoc */

/**********************************************************************/
CheckDir: procedure expose Global.  /* checks the existence of a directory */
/**********************************************************************/

parse arg DirName  /* get the name of the directory to check */

OrgDir = directory()  /* store the current directory name */

if (directory(DirName) = '') then  /* if we cannot change to the directory */
do
 call WriteLog '  ERROR: Cannot find directory "'||DirName||'".'  /* report */
 return 0  /* return with an error */
end

call directory OrgDir  /* change back to the original directory */
call WriteLog '         Checked directory "'||DirName||'".'  /* report */

return 1  /* end of CheckDir */

/**********************************************************************/
MakeDir: procedure expose Global.  /* check for the existence of a dir, and if not found, create it */
/**********************************************************************/

parse arg NewDir  /* get the name of the dir to make */

call sysfiletree NewDir,'Dirs.','DO'  /* look for the dir */

if (Dirs.0 = 0) then  /* if it is not found */
do

 if (sysmkdir(NewDir) >< 0) then  /* if we cannot create the dir */
 do
  call WriteLog '  ERROR: Cannot create directory "'||NewDir||'".'  /* report */
  return 0  /* return with an error */
 end

end

return 1  /* end of MakeDir */

/**********************************************************************/
MakeFolder: procedure expose Global.  /* creates a folder */
/**********************************************************************/

parse arg Name,Location,Settings  /* get the arguments */

if (\syscreateobject('WPFolder',Name,Location,Settings,'FAIL')) then  /* if we cannot create the folder */
do
 call WriteLog '         Cannot create folder "'||Name||'" in "'||Location||'".'  /* report */
 return 0  /* return with an error */
end

call WriteLog '         Created folder "'||Name||'" in "'||Location||'".'  /* report */

return 1  /* end of MakeFolder */

/**********************************************************************/
MakeObject: procedure expose Global.  /* creates a program object */
/**********************************************************************/

parse arg Title,Location,ExeName,Parameters,AssocType,ObjectID,IconFile,AddSettings  /* get the arguments */

Settings = 'EXENAME='||ExeName||';'||,  /* include the executable path and name */
           'PARAMETERS='||Parameters||';'||,  /* include the executable parameters */
           'ASSOCTYPE='||AssocType||';'||,  /* include the association type(s) */
           'ICONFILE='||IconFile||';'||,  /* include the icon file path and name */
           'TITLE='||Title||';'||,  /* change the title if we have an old object */
           'OBJECTID='||ObjectID||';'||,  /* include the object ID */
           AddSettings  /* include any additional settings */

if (\syscreateobject('WPProgram',Title,Location,Settings,'UPDATE')) then  /* if we cannot create the program object */
do
 call WriteLog '         Cannot create or update object with object ID "'||ObjectID||'" and title "'||Title||'" in "'||Location||'".'  /* report */
 return 0  /* return with an error */
end

if (\syssaveobject(ObjectID,1)) then  /* if we cannot save the program object */
do
 call WriteLog '         Cannot save object with object ID "'||ObjectID||'"'  /* report */
 return 0  /* return with an error */
end

call WriteLog '         Created or updated object, object ID "'||ObjectID||'" and title "'||Title||'" in "'||Location||'".'  /* report */

return 1  /* end of MakeObject */

/**********************************************************************/
MakeShadow: procedure expose Global.  /* creates a shadow */
/**********************************************************************/

parse arg Name,Location,IconFile  /* get the arguments */

if (\syscreateshadow(Name,Location)) then  /* if we cannot create the shadow */
do
 call WriteLog '         Cannot create shadow of "'||Name||'" in "'||Location||'"; shadow may already exist.'  /* report */
 return 0  /* return with an error */
end

call syssetobjectdata Name,'ICONFILE='||IconFile  /* a kludge to make the icon stick again after shadowing */
call WriteLog '         Created shadow of "'||Name||'" in "'||Location||'".'  /* report */

return 1  /* end of MakeShadow */

/**********************************************************************/
WriteLog: procedure expose Global.  /* writes an entry in the log file */
/**********************************************************************/

parse arg Text  /* get the argument */

if (stream(Global.LogFile,'C','OPEN WRITE') >< 'READY:') then  /* if we cannot open the log file for writing */
do
 call lineout 'StdOut:','ERROR: Cannot write to log file.'  /* show a message */
 return 0  /* return with an error */
end

call lineout Global.LogFile,Text  /* write the text to the file */
call stream Global.LogFile,'C','CLOSE'  /* close the file */

return 1  /* end of WriteLog */

/**********************************************************************/
GotUser: procedure expose Global.  /* looks for a user entry in the location.txt file */
/**********************************************************************/

parse arg LocFile,UserName  /* get the arguments */

if (stream(LocFile,'C','OPEN READ') >< 'READY:') then  /* if we cannot open the location file */
do
 return 0  /* return with no result */
end

GotUserName = 0  /* we haven't found it yet */

do while (lines(LocFile))  /* as long as we have lines left */

 NextLine = linein(LocFile)  /* get the next one */

 if (NextLine >< '') then  /* if the line is not empty */
 do

  if (substr(NextLine,1,1) >< '#') then  /* if it is not a comment */
  do

   parse var NextLine Location ' = ' User ';'  /* get the bits we want */

   if (translate(UserName) = translate(User)) then  /* if it is what we're looking for */
   do
    GotUserName = 1  /* we have a user name */
   end

  end

 end

end

call stream LocFile,'C','CLOSE'  /* close the locations file */

return GotUserName  /* end of GotUser */

/**********************************************************************/
GetLocations: procedure expose Global.  /* collects user entries from the location.txt file */
/**********************************************************************/

parse arg LocFile  /* get the argument */

if (stream(LocFile,'C','OPEN READ') >< 'READY:') then  /* if we cannot open the location file */
do
 return 0  /* return with no result */
end

Locations = ''  /* start with nothing */

do while (lines(LocFile))  /* as long as we have lines left */

 NextLine = linein(LocFile)  /* get the next one */

 if (NextLine >< '') then  /* if the line is not empty */
 do

  if (substr(NextLine,1,1) >< '#') then  /* if it is not a comment */
  do
   parse upper var NextLine Location ' = ' User ';'  /* get the bits we want */
   Locations = Locations||' '||Location  /* add it to what we have */
  end

 end

end

call stream LocFile,'C','CLOSE'  /* close the locations file */

Locations = strip(Locations)  /* get rid of excess whitespace */

return Locations  /* end of GetLocations */

/**********************************************************************/
StartExplain: procedure expose Global.  /* explain to the user what's going to happen */
/**********************************************************************/

call ClearScreen  /* clear the screen */

call lineout 'StdOut:','This is the RexxMail installation program.'  /* screen info */
call lineout 'StdOut:',''  /* empty line */
call lineout 'StdOut:','This program will prompt you before making any changes to your setup.'  /* screen info */
call lineout 'StdOut:',''  /* empty line */
call lineout 'StdOut:','A record of all the changes made by this program will be written to the'  /* screen info */
call lineout 'StdOut:','"install.log" file in the RexxMail program directory (i.e. the one from'  /* screen info */
call lineout 'StdOut:','which you are running this installation program) for future reference.'  /* screen info */
call lineout 'StdOut:',''  /* empty line */
call lineout 'StdOut:','You can press [Ctrl]+[C] at any time to abort the installation program.'  /* screen info */

return 1  /* end of StartExplain */

/**********************************************************************/
UpdateExplain: procedure expose Global.  /* explain to the user what's going to happen when we update the program files */
/**********************************************************************/

call lineout 'StdOut:',''  /* empty line */
call lineout 'StdOut:','A previous version of RexxMail has been found; it will be updated.'  /* screen info */

return 1  /* end of UpdateExplain */

/**********************************************************************/
InstallExplain: procedure expose Global.  /* explain to the user what's going to happen when we install the program files */
/**********************************************************************/

call ClearScreen  /* clear the screen */

call lineout 'StdOut:','The RexxMail program files will be installed in the current directory.'  /* screen info */
call lineout 'StdOut:','The RexxMail icon files will be installed in an Icons subdirectory.'  /* screen info */
call lineout 'StdOut:','The RexxMail program objects will be created in an Objects subdirectory.'  /* screen info */
call lineout 'StdOut:','If you did not register the CWMailFile class before, its files will be'  /* screen info */
call lineout 'StdOut:','installed in a CWMailFileClass subdirectory, and you will be given the option'  /* screen info */
call lineout 'StdOut:','of registering the CWMailFile class. You can always (de)register the class'  /* screen info */
call lineout 'StdOut:','later using the utilities you will find in the CWMailFileClass subdirectory.'  /* screen info */
call lineout 'StdOut:',''  /* empty line */
call lineout 'StdOut:','Apart from defining the file-program associations needed to make the system'  /* screen info */
call lineout 'StdOut:','recognize mail message files, the installation program will not change any'  /* screen info */
call lineout 'StdOut:','system files (like CONFIG.SYS). If you decide to register the CWMailFile'  /* screen info */
call lineout 'StdOut:','class, the installation program will register the functions contained in the'  /* screen info */
call lineout 'StdOut:','rexxmail.dll file (which you will find in the CWMailFileClass subdirectory).'  /* screen info */

return 1  /* end of InstallExplain */

/**********************************************************************/
CWMailFileExplain: procedure expose Global.  /* Explain to the user what's going to happen */
/**********************************************************************/

call ClearScreen  /* clear the screen */

call lineout 'StdOut:','The CWMailFile class has not yet been registered. Registering the class'  /* screen info */
call lineout 'StdOut:','will enable you to sort RexxMail messages by sender, recipients, date,'  /* screen info */
call lineout 'StdOut:','subject, and attachment flag in any WPS folder to which the class is'  /* screen info */
call lineout 'StdOut:','applied. If you register the class now, the RexxMail installation program'  /* screen info */
call lineout 'StdOut:','will apply the CWMailFile sorting capability to the RexxMail user folders.'  /* screen info */
call lineout 'StdOut:','You can always (de)register the class, and apply its sorting capability to'  /* screen info */
call lineout 'StdOut:','selected folders, using the utilities in the CWMailFileClass subdirectory.'  /* screen info */

return 1  /* end of CWMailFileExplain */

/**********************************************************************/
UserInstallExplain: procedure expose Global.  /* Explain to the user what's going to happen */
/**********************************************************************/

call ClearScreen  /* clear the screen */

call lineout 'StdOut:','You will now be asked for a main user directory name. This can not be the same'  /* screen info */
call lineout 'StdOut:','directory as the program directory. The directory will be optionally created if'  /* screen info */
call lineout 'StdOut:','it does not exist. The main directory will contain the mail folders (In, Out,'  /* screen info */
call lineout 'StdOut:','In Archive, Out Archive) and the Addresses, Accessories, and Configuration'  /* screen info */
call lineout 'StdOut:','folders.'  /* empty line */
call lineout 'StdOut:','Some file objects and object shadows will be created in these folders (the'  /* screen info */
call lineout 'StdOut:','objects themselves are in the "Objects" subfolder of the RexxMail program'  /* screen info */
call lineout 'StdOut:','directory -- the directory from which you are running this program).'  /* screen info */
call lineout 'StdOut:','The installation program will also create a start folder on your desktop.'  /* screen info */

return 1  /* end of UserInstallExplain */

/**********************************************************************/
UserQuery: procedure expose Global.  /* get a reply from the user */
/**********************************************************************/

parse arg Prompt  /* get the argument */

call lineout 'StdOut:',''  /* start a new line */
call lineout 'StdOut:',''  /* start a new line */
call charout 'StdOut:',Prompt||' [Y/N] Y'||d2c(8)  /* user prompt */
Reply = (pos(sysgetkey('ECHO'),'Yy+'||d2c(13)) > 0)  /* affirmative = 1 */
call lineout 'StdOut:',''  /* start a new line */

return Reply  /* end of UserQuery */

/**********************************************************************/
ClearScreen: procedure expose Global.  /* clears the screen ("syscls" does not preserve screen colours) */
/**********************************************************************/

ScreenSize = systextscreensize()  /* get the screen dimensions */
ScreenRows = word(ScreenSize,1)  /* the screen height */
ScreenCols = word(ScreenSize,2)  /* the screen width */

do Index = 1 to ScreenRows  /* for each row on the screen */
 call syscurpos Index,0  /* move to the row start */
 call charout 'StdOut:',copies(' ',ScreenCols)  /* clear the line */
end

call syscurpos 0,0  /* move to the top left-hand corner of the screen */

call charout 'StdOut:',copies('_',ScreenCols)  /* line */
call lineout 'StdOut:','RexxMail Installation Program for build '||Global.Build  /* screen info */
call charout 'StdOut:',copies('',ScreenCols)  /* line */

call lineout 'StdOut:',''  /* empty line */
call lineout 'StdOut:',''  /* empty line */

return 1  /* end of ClearScreen */

/**********************************************************************/
LoadRexxUtils: procedure expose Global.  /* Load the REXX utility functions if necessary */
/**********************************************************************/

if (rxfuncquery('SysLoadFuncs') >< 0) then  /* if we have to load the REXX utility functions */
do

 if (rxfuncadd('SysLoadFuncs','RexxUtil','SysLoadFuncs') = 0) then  /* if we can register the general loading function */
 do
  call sysloadfuncs  /* call the general loading function */
 end
 else  /* if we cannot register the general loading function */
 do
  return 0  /* no success */
 end

end

return 1  /* end of LoadRexxUtils */

/**********************************************************************/
Syntax:  /* handles syntax errors */
/**********************************************************************/

call FatalError 'REXX did not recognize the instruction',Sigl  /* start with this */
call Halt  /* and quit via Halt */

/**********************************************************************/
Error:  /* handles general errors */
/**********************************************************************/

call FatalError 'External program error',Sigl  /* start with this */
call Halt  /* and quit via Halt */

/**********************************************************************/
Failure:  /* handles external program errors */
/**********************************************************************/

call FatalError 'External program failure',Sigl  /* start with this */
call Halt  /* and quit via Halt */

/**********************************************************************/
NoValue:  /* handles initialization errors */
/**********************************************************************/

call FatalError 'Initialization error',Sigl  /* start with this */
call Halt  /* and quit via Halt */

/**********************************************************************/
FatalError: procedure expose Global.  /* reports a fatal error */
/**********************************************************************/

parse arg ErrorStart,Sigl  /* get the argument */

call WriteLog '  ERROR : '||ErrorStart||' on line '||Sigl||':'||,  /* report */
              '    '||sourceline(Sigl)  /* report */

return 1  /* end of FatalError */

/*****************************************************************/
Halt:  /* we end up here if a halt condition occurred */
/*****************************************************************/

call beep 444,111  /* signal */
call beep 333,222  /* signal */

call WriteLog date('S')||' '||time('N')||' : Installation aborted.'  /* report */

call ClearScreen  /* clear the screen */

call lineout 'StdOut:','Installation aborted.'  /* show a message */

/*****************************************************************/
Quit:  /* we end up here at the end of it all */
/*****************************************************************/

call lineout 'StdOut:',''  /* empty line */
call lineout 'StdOut:','For details, see the install.log file in the RexxMail program directory.'  /* report */
call lineout 'StdOut:',''  /* empty line */
call lineout 'StdOut:',''  /* empty line */
call charout 'StdOut:','End of RexxMail installation - press any key to exit'  /* report */
call sysgetkey 'NOECHO'  /* wait for a key */

call ClearScreen  /* and wipe the screen */

exit  /* and exit */
