/**********************************************************************/
/*                                                                    */
/* Utility to set RexxMail folder attributes for current user         */
/*                                                                    */
/**********************************************************************/
/*                                                                    */
/* This program forms part of the RexxMail package, and may not be    */
/* distributed separately.                                            */
/*                                                                    */
/**********************************************************************/
/*                                                                    */
/* The latest version of RexxMail can be found at                     */
/*               www.degeus.com/rexx                                  */
/*                                                                    */
/**********************************************************************/
/*                                                                    */
/* This program is released under the terms of the GNU license, see   */
/*               www.gnu.org/copyleft/gpl.html                        */
/*                                                                    */
/**********************************************************************/
/*                                                                    */
/* (c) 2000-2002 Marcus de Geus                                       */
/*               marcus@degeus.com                                    */
/*               www.degeus.com                                       */
/*                                                                    */
/**********************************************************************/
/*                                                                    */
/* Use it if you like it. Don't if you don't. No legalese.            */
/*                                                                    */
/**********************************************************************/

signal on halt                                                                 /* handle halt condition */

if (\LoadRexxUtils()) then  /* if we cannot load the REXX utilities lib */
do
 call Halt /* quit */
end

/**********************************************************************/
/* See if we can find and open the location.txt file                  */
/**********************************************************************/

parse source . . ProgSpec                                                      /* get the program name */
ProgDir = filespec('D',ProgSpec)||filespec('P',ProgSpec)                       /* get the program directory */

LocFile = ProgDir||'location.txt'                                              /* the name of the locations file */

if (stream(LocFile,'C','QUERY EXISTS') = '') then                              /* if we cannot find the location file */
do
 call beep 333,333                                                             /* signal */
 call lineout StdOut,'Could not find "'||LocFile||'"'                                          /* report */
 call Halt                                                                     /* and abort */
end

if (stream(LocFile,'C','OPEN READ') >< 'READY:') then                          /* if we cannot open the location file */
do
 call beep 333,333                                                             /* signal */
 call lineout StdOut,'Could not open "'||LocFile||'"'                                          /* report */
 call Halt                                                                     /* and abort */
end

/**********************************************************************/
/* Retrieve the main folder location for the current user             */
/**********************************************************************/

parse upper arg UserName /* get any argument */

if (UserName = '') then  /* if we have no user name */
do

 UserName = translate(value('USER',,'OS2ENVIRONMENT')) /* get the user from the OS/2 environemnt, in upper case */

 if (UserName = '') then /* if we still have no user name */
 do
  UserName = 'DEFAULT' /* use the default */
 end

end

Location = ''                                                                /* start with nothing */

do while ((Location = '') & (lines(LocFile)))                                                      /* as long as we have no location, and there are lines left */

 NextLine = linein(LocFile)                                                    /* get the next line */

 if (NextLine >< '') then                                                      /* if the line is not empty */
 do

  if (substr(NextLine,1,1) >< '#') then                                        /* if it is not a comment */
  do

   parse upper var NextLine Location ' = ' NextUser ';'            /* get the bits we want */

   if (NextUser >< UserName) then  /* if this is not the right user */
   do
    Location = '' /* clear the location variable to continue the loop */
   end

  end

 end

end

call stream LocFile,'C','CLOSE'                                                /* close the locations file */

if (Location = '') then  /* if we found no location */
do
 call lineout StdOut,'No location found for user "'||UserName||'"' /* report */
 call Usage /* and tell 'em how to do it */
end

/**********************************************************************/
/* Get user input                                                     */
/**********************************************************************/

GoBack = 0 /* no need to go back yet */

do until (GoBack) /* go on until the user quits */

 call ClearScreen

 call lineout StdOut,'  Main RexxMail folder for user "'||UserName||'" = "'||Location||'"' /* info */
 call lineout StdOut,'' /* empty line */
 call lineout StdOut,'' /* empty line */
 call lineout StdOut,'  Select one of the following for more information:' /* info */
 call lineout StdOut,'' /* empty line */
 call lineout StdOut,'    [L]ocked user folders' /* info */
 call lineout StdOut,'    [U]nlocked user folders' /* info */
 call lineout StdOut,'    [I]nvisible user folders' /* info */
 call lineout StdOut,'    [V]isible user folders' /* info */
 call lineout StdOut,'    [B]itmap RexxMail folder background' /* info */
 call lineout StdOut,'    [S]ystem default folder background' /* info */
 call lineout StdOut,'    [M]onospaced (System VIO) icon text font' /* info */
 call lineout StdOut,'    [P]roportional (Warp Sans) icon text font' /* info */
 call lineout StdOut,'    [O]ther (user-defined) icon text font' /* info */
 call lineout StdOut,'' /* empty line */
 call lineout StdOut,'    [Q]uit' /* info */
 call lineout StdOut,'' /* empty line */
 call lineout StdOut,'' /* empty line */
 call charout StdOut,'  Action [L/U/I/V/B/S/M/P/O/Q] : Q'||d2c(8)  /* user prompt */
 Reply = sysgetkey('ECHO')  /* get the user's reply */

 select /* do one of the following */

  when (pos(Reply,'Ll') > 0) then /* if the reply is one of these */
  do
   call DoLock Location /* tell the user how to lock the user folders */
  end

  when (pos(Reply,'Uu') > 0) then /* if the reply is one of these */
  do
   call DoUnlock Location /* tell the user how to unlock the user folders */
  end

  when (pos(Reply,'Ii') > 0) then /* if the reply is one of these */
  do
   call DoInvisible Location /* tell the user how to make the folders invisible */
  end

  when (pos(Reply,'Vv') > 0) then /* if the reply is one of these */
  do
   call DoVisible Location /* tell the user how to make the folders visible */
  end

  when (pos(Reply,'Bb') > 0) then /* if the reply is one of these */
  do
   call DoBitmap Location,ProgDir /* tell the user how to give the folders a bitmap background */
  end

  when (pos(Reply,'Ss') > 0) then /* if the reply is one of these */
  do
   call DoSystem Location /* tell the user how to give the folders a system default background */
  end

  when (pos(Reply,'Mm') > 0) then /* if the reply is one of these */
  do
   call DoMonoText Location /* tell the user how to get a monospaced text font */
  end

  when (pos(Reply,'Pp') > 0) then /* if the reply is one of these */
  do
   call DoPropText Location /* tell the user how to get a proportional text font */
  end

  when (pos(Reply,'Oo') > 0) then /* if the reply is one of these */
  do
   call DoOtherText Location /* tell the user how to get a user-defined text font */
  end

  when (pos(Reply,'Qq'||d2c(13)||d2c(27)) > 0) then /* if the reply is one of these */
  do
   GoBack = 1  /* go back, i.e. quit */
  end

  otherwise /* if the reply is none of the above */
  do
   nop /* do nothing */
  end

 end

end

call Halt /* that's all, folks! */

/*****************************************************************/
DoLock: procedure                                                                  /* locks folders */
/*****************************************************************/

parse arg Location /* get the argument */

call ClearScreen /* clear the screen */
call lineout StdOut,'  This option will set the attributes of the main folder and its subfolders' /* report */
call lineout StdOut,'  to prevent their being copied, deleted, dragged, moved, or shadowed.' /* report */
call lineout StdOut,'  Also, folder properties notebooks will become inaccessible, and objects' /* report */
call lineout StdOut,'  cannot be dropped on these folders, with the exception of mail and address' /* report */
call lineout StdOut,'  folders.'  /* report */
call lineout StdOut,''  /* report */
call lineout StdOut,'  Note that this applies only to WPS operations; command-line operations are' /* report */
call lineout StdOut,'  not affected.' /* report */

if (UserQuery('Lock main folder and subfolders?')) then
do
 call LockUnLock Location,'YES'
end

return /* end of DoLock */

/*****************************************************************/
DoUnlock: procedure                                                                  /* unlocks folders */
/*****************************************************************/

parse arg Location /* get the argument */

call ClearScreen /* clear the screen */
call lineout StdOut,'  This option will set the attributes of the main folder and its subfolders' /* report */
call lineout StdOut,'  to allow their being copied, deleted, dragged, moved, or shadowed.' /* report */
call lineout StdOut,'  Also, folder properties notebooks will become accessible, and objects' /* report */
call lineout StdOut,'  can be dropped into these folders.' /* report */
call lineout StdOut,''  /* report */
call lineout StdOut,'  Note that this applies only to WPS operations; command-line operations are' /* report */
call lineout StdOut,'  not affected.' /* report */

if (UserQuery('Unlock main folder and subfolders?')) then
do
 call LockUnLock Location,'NO'
end

return /* end of DoUnlock */

/*****************************************************************/
DoInvisible: procedure                                                                  /* makes folders invisible */
/*****************************************************************/

parse arg Location /* get the argument */

call ClearScreen /* clear the screen */
call lineout StdOut,'  This option will make the folder objects in the main user folder invisible,' /* report */
call lineout StdOut,'  with the exception of the Accessories folder.' /* report */
call lineout StdOut,''  /* report */
call lineout StdOut,'  Note that this applies only to WPS operations; command-line operations are' /* report */
call lineout StdOut,'  not affected.' /* report */
call lineout StdOut,''  /* report */
call lineout StdOut,'  If you have not already done so, you may first want to create a shadow of' /* report */
call lineout StdOut,'  the Configuration folder in the Accessories folder for easier access.'  /* report */

if (UserQuery('Make folder objects in main folder invisible?')) then
do
 call ShowHide Location,'YES'
end

return /* end of DoInvisible */

/*****************************************************************/
DoVisible: procedure                                                                  /* makes folders visible */
/*****************************************************************/

parse arg Location /* get the argument */

call ClearScreen /* clear the screen */
call lineout StdOut,'  This option will make the folder objects in the main user folder visible.' /* report */
call lineout StdOut,''  /* report */
call lineout StdOut,'  Note that this applies only to WPS operations; command-line operations are' /* report */
call lineout StdOut,'  not affected.' /* report */

if (UserQuery('Make folder objects in main folder visible?')) then
do
 call ShowHide Location,'NO'
end

return /* end of DoVisible */

/*****************************************************************/
DoBitmap: procedure                                                                  /* sets bitmap background */
/*****************************************************************/

parse arg Location,ProgDir /* get the argument */

GoBack = 0 /* no need to go back yet */

do until (GoBack) /* go on until the user quits */

 call ClearScreen /* clear the screen */
 call lineout StdOut,'  This option sets the background of the user folders to use the FolderBG.BMP'  /* report */
 call lineout StdOut,'  file in the "Icons" subdirectory of the RexxMail program directory.' /* report */
 call lineout StdOut,'' /* empty line */
 call lineout StdOut,'  Select one of the following:' /* info */
 call lineout StdOut,'' /* empty line */
 call lineout StdOut,'    [S]ingle bitmap background' /* info */
 call lineout StdOut,'    [T]iled bitmap background' /* info */
 call lineout StdOut,'' /* empty line */
 call lineout StdOut,'    [Q]uit' /* info */
 call lineout StdOut,'' /* empty line */
 call lineout StdOut,'' /* empty line */
 call charout StdOut,'  Action [S/T/Q] : Q'||d2c(8)  /* user prompt */
 Reply = sysgetkey('ECHO')  /* get the user's reply */

 select /* do one of the following */

  when (pos(Reply,'Ss') > 0) then /* if the reply is one of these */
  do
   call SetBackground Location,ProgDir,'N' /* do it */
   GoBack = 1 /* and quit */
  end

  when (pos(Reply,'Tt') > 0) then /* if the reply is one of these */
  do
   call SetBackground Location,ProgDir,'T' /* do it */
   GoBack = 1 /* and quit */
  end

  when (pos(Reply,'Qq'||d2c(13)||d2c(27)) > 0) then /* if the reply is one of these */
  do
   GoBack = 1  /* go back, i.e. quit */
  end

  otherwise /* if the reply is none of the above */
  do
   nop /* do nothing */
  end

 end

end

return /* end of DoBackground */

/*****************************************************************/
DoSystem: procedure                                                                  /* sets system default folder background */
/*****************************************************************/

parse arg Location /* get the argument */

call ClearScreen /* clear the screen */
call lineout StdOut,'  This option sets the background of the user folders to the system default.'  /* report */
call SetBackground Location /* do it */

return /* end of DoBackground */

/*****************************************************************/
DoMonoText: procedure                                                                  /* sets monospaced text font */
/*****************************************************************/

parse arg Location /* get the argument */

GoBack = 0 /* no need to go back yet */

do until (GoBack) /* go on until the user quits */

 call ClearScreen /* clear the screen */
 call lineout StdOut,'  This option sets the icon text font to System VIO.'  /* report */
 call lineout StdOut,'' /* empty line */
 call lineout StdOut,'' /* empty line */
 call charout StdOut,'  Please enter the font size (2-18, or none to quit): ' /* info */
 parse pull FontSize

 select /* do one of the following */

  when (datatype(FontSize,'W')) then /* if the reply is a whole number */
  do

   if ((FontSize > 1) & (FontSize < 19)) then  /* if the reply is in the right range */
   do
    call SetTextFont Location,FontSize||'.System VIO' /* do it */
    GoBack = 1 /* and quit */
   end

  end

  when (FontSize = '') then /* if the reply is nothing */
  do
   GoBack = 1 /* quit */
  end

  otherwise /* if the reply is none of the above */
  do
   nop /* do nothing */
  end

 end

end

return /* end of DoMonoText */

/*****************************************************************/
DoPropText: procedure                                                                  /* sets proportional text font */
/*****************************************************************/

parse arg Location /* get the argument */

call ClearScreen /* clear the screen */
call lineout StdOut,'  This option sets the icon text font to 9. Warp Sans.'  /* report */
call SetTextFont Location,'9.WarpSans' /* do it */

return /* end of DoPropText */

/*****************************************************************/
DoOtherText: procedure                                                                  /* sets user-defined text font */
/*****************************************************************/

parse arg Location /* get the argument */

GoBack = 0 /* no need to go back yet */

do until (GoBack) /* go on until the user quits */

 call ClearScreen /* clear the screen */
 call lineout StdOut,'  This option sets the icon text font to a user-defined value.'  /* report */
 call lineout StdOut,'' /* empty line */
 call lineout StdOut,'' /* empty line */
 call charout StdOut,'  Please enter the font definition (e.g. "10.Helvetica"): ' /* info */
 parse pull FontSize
 parse var FontSize FontSize '.' FontName  /* get the components */

 select /* do one of the following */

  when (datatype(FontSize,'W')) then /* if the font size at least is a whole number, we have a chance of success */
  do
   call SetTextFont Location,FontSize||'.'||FontName /* do it */
   GoBack = 1 /* and quit */
  end

  when (FontSize = '') then /* if the reply is nothing */
  do
   GoBack = 1 /* quit */
  end

  otherwise /* if the reply is none of the above */
  do
   nop /* do nothing */
  end

 end

end

return /* end of DoOtherText */

/*****************************************************************/
SetTextFont: procedure /* sets the mail folder text font */
/*****************************************************************/

parse arg Location,FontSize /* get the arguments */

Settings = 'ICONFONT='||FontSize||';;'  /* use these parameters */
Message = 'Set icon text font to '||FontSize||'?'

if (UserQuery(Message)) then /* if we get the user's permission */
do
 call ProcessFolders Location,Settings  /* do it */
end

return /* end of SetBackGround */

/*****************************************************************/
SetBackGround: procedure /* sets the user folder background */
/*****************************************************************/

parse arg Location,ProgDir,Action /* get the arguments */

if (Action = '') then  /* if we have no action parameter */
do
 Settings = 'BACKGROUND=,,,,'  /* no background parameters */
 Message = 'Set user folder background to system default?'
end
else  /* if we have an action parameter */
do
 Settings = 'BACKGROUND='||ProgDir||'icons\folderbg.bmp,'||Action||',,I,255 255 255'  /* use these parameters */
 BMType = word('single tiled',pos(Action,'NT')) /* get the right bitmap type message string */
 Message = 'Set user folder background to '||BMType||' bitmap?'
end

if (UserQuery(Message)) then /* if we get the user's permission */
do
 call ProcessFolders Location,Settings  /* do it */
end

return /* end of SetBackGround */

/*****************************************************************/
LockUnlock: procedure                                                                  /* does the actual locking/unlocking */
/*****************************************************************/

parse arg Location,Action /* get the arguments */

MessLoc = Location||'\messages'  /* define the messages subdir */

call sysfiletree MessLoc,'Dirs.','DO'  /* look for the dir */

if (Dirs.0 = 0) then  /* if we do not find it */
do
 MessLoc = Location  /* use the original location */
end

Settings = 'NODELETE='||Action||';'||,  /* include the NoDelete setting for the next bunch of folders */
           'NOSETTINGS='||Action||';'||,  /* and NoSettings */
           'NOSHADOW='||Action||';'||,  /* and NoShadow */
           'NOCOPY='||Action||';'||,  /* and NoCopy */
           'NODRAG='||Action||';'||,  /* and NoDRag */
           'NOMOVE='||Action||';'  /* and NoMove */

call syssetobjectdata MessLoc||'\in',Settings  /* process the In folder */
call syssetobjectdata MessLoc||'\in archive',Settings  /* process the In Archive folder */
call syssetobjectdata MessLoc||'\out',Settings  /* process the Out folder */
call syssetobjectdata MessLoc||'\out archive',Settings  /* process the Out Archive folder */
call syssetobjectdata MessLoc||'\addresses',Settings  /* process the Addresses folder */

call sysfiletree MessLoc||'\addresses\*','Folders.','DOS'  /* look for (sub)folders in the Addresses folder */

do Number = 1 to Folders.0  /* take each one we find */
 call syssetobjectdata Folders.Number,Settings  /* and process it */
end

call sysfiletree Location||'\configuration\*','Folders.','DOS'  /* look for (sub)folders in the Configuration folder */

do Number = 1 to Folders.0  /* take each one we find */
 call syssetobjectdata Folders.Number,Settings  /* and process it */
end

Settings = Settings||'NODROP='||Action||';'  /* include NoDrop for the next bunch of folders */

call syssetobjectdata Location,Settings  /* process the main folder */
call syssetobjectdata Location||'\accessories',Settings  /* process the Accessories folder */
call syssetobjectdata Location||'\configuration',Settings  /* process the Configuration folder */
call syssetobjectdata Location||'\temp',Settings  /* process the Temp folder */

return                                                                         /* end of LockUnlock */

/*****************************************************************/
ShowHide: procedure                                                                  /* does the actual hiding/showing */
/*****************************************************************/

parse arg Location,Action /* get the arguments */

Settings = 'NOTVISIBLE='||Action||';'   /* the setting to use */
call sysfiletree Location||'\*','Folders.','DO'  /* look for folders in the main folder */

do Number = 1 to Folders.0  /* take each one we find */

 if (filespec('N',Folders.Number) >< 'Accessories') then /* unless it is the Accessories folder */
 do
  call syssetobjectdata Folders.Number,Settings  /* process it */
 end

end

return                                                                         /* end of LockUnlock */

/*****************************************************************/
ProcessFolders: procedure /* processes folder settings */
/*****************************************************************/

parse arg Location,Settings /* get the arguments */

call syssetobjectdata Location,Settings  /* process the main folder */
call sysfiletree Location||'\*','Folders.','DOS'  /* look for (sub)folders in the main folder */

do Number = 1 to Folders.0  /* take each one we find */
 call syssetobjectdata Folders.Number,Settings  /* process the folder */
end

return /* end of ProcessFolders */

/**********************************************************************/
UserQuery: procedure expose Global.  /* get a reply from the user */
/**********************************************************************/

parse arg Prompt  /* get the argument */

call lineout StdOut,''  /* start a new line */
call lineout StdOut,''  /* start a new line */
call charout StdOut,'  '||Prompt||' [Y/N] Y'||d2c(8)  /* user prompt */
Reply = (pos(sysgetkey('ECHO'),'Yy+'||d2c(13)) > 0)  /* affirmative = 1 */

return Reply  /* end of UserQuery */

/**********************************************************************/
ClearScreen: procedure expose Global.  /* clears the screen ("syscls" does not preserve screen colours) */
/**********************************************************************/

ScreenSize = systextscreensize()  /* get the screen dimensions */
ScreenRows = word(ScreenSize,1)  /* the screen height */
ScreenCols = word(ScreenSize,2)  /* the screen width */

do Index = 1 to ScreenRows  /* for each row on the screen */
 call syscurpos Index,0  /* move to the row start */
 call charout StdOut,copies(' ',ScreenCols)  /* clear the line */
end

call syscurpos 0,0  /* move to the top left-hand corner of the screen */
call charout StdOut,copies('_',ScreenCols)  /* low line */
call lineout StdOut,'  RexxMail User Folders Utility' /* screen info */
call charout StdOut,copies('',ScreenCols)  /* high line */
call lineout StdOut,''  /* empty line */

return 1  /* end of ClearScreen */

/**********************************************************************/
LoadRexxUtils: procedure expose Global.  /* loads REXX utility functions if necessary */
/**********************************************************************/

if (rxfuncquery('SysLoadFuncs') >< 0) then  /* if we have to load the REXX utility functions */
do

 if (rxfuncadd('SysLoadFuncs','RexxUtil','SysLoadFuncs') = 0) then  /* if we can register the general loading function */
 do
  call sysloadfuncs  /* call the general loading function */
 end
 else  /* if we cannot register the general loading function */
 do
  Message = '  ERROR: Cannot load Rexx utility functions.'  /* report */
  call WriteLog Global.LogFile,Message  /* report */
  return 0  /* no success */
 end

end

return 1  /* end of LoadRexxUtils */

/*****************************************************************/
Usage:                                                                          /* we end up here */
/*****************************************************************/

call lineout StdOut,'Usage : MailLock [username]'

/*****************************************************************/
Halt:                                                                          /* we end up here */
/*****************************************************************/

exit                                                                           /* and exit */
