/**********************************************************************/
/*                                                                    */
/* RexxMail is a REXX program that uses the Workplace Shell, the      */
/* graphical user interface of the IBM OS/2 operating system, to      */
/* create an easy to use, object-oriented, highly flexible e-mail     */
/* message processing system.                                         */
/*                                                                    */
/**********************************************************************/
/*                                                                    */
/* This program forms part of the RexxMail package, and may not be    */
/* distributed separately.                                            */
/*                                                                    */
/**********************************************************************/
/*                                                                    */
/* The latest version of RexxMail can be found at                     */
/*               www.degeus.com/rexx                                  */
/*                                                                    */
/**********************************************************************/
/*                                                                    */
/* This program is released under the terms of the GNU license, see   */
/*               www.gnu.org/copyleft/gpl.html                        */
/*                                                                    */
/**********************************************************************/
/*                                                                    */
/* (c) 2000-2004 Marcus de Geus                                       */
/*               marcus@degeus.com                                    */
/*               www.degeus.com                                       */
/*                                                                    */
/**********************************************************************/
/*                                                                    */
/* Use it if you like it. Don't if you don't. No legalese.            */
/*                                                                    */
/**********************************************************************/

/**********************************************************************/
/* Main                                                               */
/**********************************************************************/

signal on Error  /* handles general error condition */
signal on Failure  /* handles external program errors */
signal on Halt  /* handles halt condition */
signal on NoValue  /* handles initialization errors */
signal on Syntax  /* handles syntax errors */

signal off NotReady  /* we handle I/O errors */

/**********************************************************************/
/* initialize some global strings                                     */
/**********************************************************************/

drop Global.  /* start with nothing */

Global.Build = '20040119.094720'  /* define the build number */
Global.BuildMess = 'RexxMail e-mail processor for OS/2 - build number '||Global.Build  /* a build number message */
Global.Copyright = '(c) 2000-2004 Marcus de Geus - marcus@degeus.com - www.degeus.com'  /* a copyright line */
Global.Abort = '(Try [Ctrl][C] to abort)'  /* a bit of help */
Global.CRLF = d2c(13)||d2c(10)  /* define CRLF */
Global.EmptyLine = Global.CRLF||Global.CRLF  /* define an empty line */
Global.Warning = ''  /* the warning string to use in message headers and object titles */
Global.Errors = ''  /* no error messages yet */
Global.Hidden.0 = 0  /* no files being processed yet */
Global.ExtraCharacters = '(){}[]<>/\-_=+|"*;:,.'  /* extra formatting characters allowed in some settings strings */

Global.ESCtoASC = '_**HDCS__'||d2c(9)||d2c(10)||'mf'||d2c(13)||'_*'||,  /* bytes 00 to 0F of the escape characters => ASCII conversion table */
                  '><_!ps__^_><__^_'  /* bytes 10 to 1F */

Global.850toASC = 'CueaaaaceeeiiiAA'||,  /* bytes 80 to 8F of the PC850 => ASCII conversion table */
                  'EaAooouuyOUoLOxf'||,  /* bytes 90 to 9F */
                  'aiounNao?r___!__'||,  /* bytes A0 to AF */
                  '_____AAAc____cY_'||,  /* bytes B0 to BF */
                  '______aA_______*'||,  /* bytes C0 to CF */
                  'dDEEEeIII____|I_'||,  /* bytes D0 to DF */
                  "O_OOoO___UUUyY_'"||,  /* bytes E0 to EF */
                  '-_____/__"-132*_'  /* bytes F0 to FF */

Global.850toISO = ''||,  /* bytes 80 to 8F of the PC850 => ISO-8859-1 conversion table */
                  'f'||,  /* bytes 90 to 9F */
                  'Ѫ'||,  /* bytes A0 to AF */
                  '__________'||,  /* bytes B0 to BF */
                  '_____________'||,  /* bytes C0 to CF */
                  'Ȁ_____'||,  /* bytes D0 to DF */
                  'յݯ'||,  /* bytes E0 to EF */
                  '_*'  /* bytes F0 to FF */

Global.ISOto850 = '______________'||,  /* bytes 80 to 8F of the ISO-8859-1 => PC850 conversion table */
                  '"&<>____________'||,  /* bytes 90 to 9F */
                  'Ͼ'||,  /* bytes A0 to AF */
                  ''||,  /* bytes B0 to BF */
                  'ǎԐ'||,  /* bytes C0 to CF */
                  'ѥ噞'||,  /* bytes D0 to DF */
                  'Ƅ'||,  /* bytes E0 to EF */
                  'Ф'  /* bytes F0 to FF */

/**********************************************************************/
/* Define the various operational settings                            */
/**********************************************************************/

Settings. = ''  /* empty all settings */

/**********************************************************************/
/* Define the text settings options                                   */
/**********************************************************************/

Index = 0  /* start at 0 */
StartIndex = Index + 1  /* the start index for later */

Index = Index + 1  /* up the counter */
Settings.Index.!Name = 'ADDRESS'  /* defines the default e-mail address of the sender for outgoing messages */
Settings.Index.!Single = 1  /* this is a single word entry */

Index = Index + 1  /* up the counter */
Settings.Index.!Name = 'CLOSINGTEXT'  /* defines the closing text file to use */

Index = Index + 1  /* up the counter */
Settings.Index.!Name = 'COLLECTACTION'  /* the action to take after collecting mail */
Settings.Index.!Options = 'OPENFOLDERS WARNICON'  /* the available options */
Settings.Index.!Single = 1  /* this is a single word entry  */

Index = Index + 1  /* up the counter */
Settings.Index.!Name = 'DATETIMEBODY'  /* defines how message times are included in the body text of messages */
Settings.Index.!Value = 'ORIGINAL'  /* use this as the default value */
Settings.Index.!Options = 'ORIGINAL SYSTEM UNIVERSAL UTC ORIGINALISO SYSTEMISO UNIVERSALISO UTCISO'  /* the available options */
Settings.Index.!Fuzzy = 1  /* this entry can contain additional formatting */

Index = Index + 1  /* up the counter */
Settings.Index.!Name = 'DATETIMEHEADER'  /* defines how message times are shown in view and edit files headers */
Settings.Index.!Value = 'ORIGINAL'  /* use this as the default value */
Settings.Index.!Options = 'ORIGINAL SYSTEM UNIVERSAL UTC ORIGINALISO SYSTEMISO UNIVERSALISO UTCISO'  /* the available options */
Settings.Index.!Fuzzy = 1  /* this entry can contain additional formatting */

Index = Index + 1  /* up the counter */
Settings.Index.!Name = 'DATETIMETITLE'  /* defines how message times are shown in in object titles */
Settings.Index.!Value = 'ORIGINAL'  /* use this as the default value */
Settings.Index.!Options = 'ORIGINAL SYSTEM UNIVERSAL UTC ORIGINALISO SYSTEMISO UNIVERSALISO UTCISO'  /* the available options */
Settings.Index.!Fuzzy = 1  /* this entry can contain additional formatting */

Index = Index + 1  /* up the counter */
Settings.Index.!Name = 'EDITOR'  /* defines the command to run on editing outgoing messages */
Settings.Index.!Value = 'E.EXE'  /* use the OS/2 system editor as the default value */

Index = Index + 1  /* up the counter */
Settings.Index.!Name = 'FORWARDTEXT'  /* defines the intro text for forwarded messages */

Index = Index + 1  /* up the counter */
Settings.Index.!Name = 'JUNKMAILHEADER'  /* defines the junk mail header to look for */

Index = Index + 1  /* up the counter */
Settings.Index.!Name = 'NAME'  /* defines the default "real" name of the sender for outgoing messages */

Index = Index + 1  /* up the counter */
Settings.Index.!Name = 'OBJECTTITLEIN'  /* defines the parts to be included in incoming mail message object titles */
Settings.Index.!Value = 'DATE FROM TO CC SUBJECT'  /* use this as the default value */
Settings.Index.!Options = 'DATE FROM TO CC SUBJECT'  /* the available options */

Index = Index + 1  /* up the counter */
Settings.Index.!Name = 'OBJECTTITLEOUT'  /* defines the parts to be included in outgoing mail message object titles */
Settings.Index.!Value = 'DATE FROM TO CC BCC SUBJECT'  /* use this as the default value */
Settings.Index.!Options = 'DATE FROM TO CC BCC SUBJECT'  /* the available options */

Index = Index + 1  /* up the counter */
Settings.Index.!Name = 'OPENFOLDERS'  /* defines the RexxMail folders to open on the desktop when the /open switch is used */
Settings.Index.!Value = 'IN INARCHIVE OUT OUTARCHIVE'  /* use this as the default value */
Settings.Index.!Options = 'ACCESSORIES ADDRESSES CONFIGURATION IN INARCHIVE OUT OUTARCHIVE'  /* the available options */

Index = Index + 1  /* up the counter */
Settings.Index.!Name = 'POP3PASSWORD'  /* defines the POP3 password to use */
Settings.Index.!Single = 1  /* this is a single word entry  */

Index = Index + 1  /* up the counter */
Settings.Index.!Name = 'POP3SERVER'  /* defines the POP3 server to contact, if any */
Settings.Index.!Single = 1  /* this is a single word entry  */

Index = Index + 1  /* up the counter */
Settings.Index.!Name = 'POP3USER'  /* defines the POP3 user name to use */
Settings.Index.!Single = 1  /* this is a single word entry  */

Index = Index + 1  /* up the counter */
Settings.Index.!Name = 'READER'  /* defines the command to run for reading the text part of incoming and sent messages */
Settings.Index.!Value = 'E.EXE'  /* use this as the default value */

Index = Index + 1  /* up the counter */
Settings.Index.!Name = 'READERHEADERIN'  /* defines which message header entries of incoming messages are to be included in the reader file */
Settings.Index.!Value = 'DATE FROM TO CC SUBJECT'  /* use this as the default value */

Index = Index + 1  /* up the counter */
Settings.Index.!Name = 'READERHEADEROUT'  /* defines which message header entries of sent messages are to be included in the reader file */
Settings.Index.!Value = 'DATE FROM TO CC BCC SUBJECT'  /* use this as the default value */

Index = Index + 1  /* up the counter */
Settings.Index.!Name = 'REPLYADDRESS'  /* defines the default reply e-mail address of the sender for outgoing messages */
Settings.Index.!Single = 1  /* this is a single word entry */

Index = Index + 1  /* up the counter */
Settings.Index.!Name = 'REPLYTEXT'  /* defines the intro text for reply messages */

Index = Index + 1  /* up the counter */
Settings.Index.!Name = 'RUNAFTERCOLLECT'  /* defines the command to run after collecting mail */

Index = Index + 1  /* up the counter */
Settings.Index.!Name = 'RUNAFTEROPEN'  /* defines the command to run after opening the RexxMail user folders */

Index = Index + 1  /* up the counter */
Settings.Index.!Name = 'RUNAFTERSEND'  /* defines the command to run after sending mail */

Index = Index + 1  /* up the counter */
Settings.Index.!Name = 'RUNATTACHIN'  /* defines a command to run on incoming mail attachments right after they have been unpacked from a mesage file */

Index = Index + 1  /* up the counter */
Settings.Index.!Name = 'RUNATTACHOUT'  /* defines a command to run on outgoing attachments before they are added to the sendable message file */

Index = Index + 1  /* up the counter */
Settings.Index.!Name = 'RUNBEFORECOLLECT'  /* defines the command to run before collecting mail */

Index = Index + 1  /* up the counter */
Settings.Index.!Name = 'RUNBEFOREOPEN'  /* defines the command to run before opening the RexxMail user folders */

Index = Index + 1  /* up the counter */
Settings.Index.!Name = 'RUNBEFORESEND'  /* defines the command to run before sending mail */

Index = Index + 1  /* up the counter */
Settings.Index.!Name = 'RUNCOLLECT'  /* defines a command to run for collecting mail */

Index = Index + 1  /* up the counter */
Settings.Index.!Name = 'RUNRECEIVED'  /* defines a command to run on received message files */

Index = Index + 1  /* up the counter */
Settings.Index.!Name = 'RUNSEND'  /* defines a command to run on sendable message files */

Index = Index + 1  /* up the counter */
Settings.Index.!Name = 'SORTADDRESS'  /* address type to use for sorting */
Settings.Index.!Value = 'FULL'  /* use this as the default value */
Settings.Index.!Single = 1  /* this is a single word entry  */
Settings.Index.!Options = 'ADDRESS FULL NAME'  /* the available options */

Index = Index + 1  /* up the counter */
Settings.Index.!Name = 'SMTPSERVER'  /* The SMTP server to use, if any */
Settings.Index.!Single = 1  /* this is a single word entry  */

Index = Index + 1  /* up the counter */
Settings.Index.!Name = 'STATIONERY'  /* defines the name of the stationery folder */
Settings.Index.!Value = 'BLUE_WHITE'  /* use this as the default value */

Index = Index + 1  /* up the counter */
Settings.Index.!Name = 'TIMESERVERS'  /* the names of time servers to contact for constructing a time zone string */

Index = Index + 1  /* up the counter */
Settings.Index.!Name = 'TIMEZONE'  /* a static time zone setting to use instead of contacting a time server */
Settings.Index.!Value = '-0000'  /* use this as the default value */
Settings.Index.!Single = 1  /* this is a single word entry  */

Index = Index + 1  /* up the counter */
Settings.Index.!Name = 'TOCCBCC'  /* address insertion folder(s) to show in attachments folder when editing */
Settings.Index.!Value = 'To-Cc-Bcc'  /* use this as the default value */
Settings.Index.!Options = 'TO CC BCC'  /* the available options */
Settings.Index.!Fuzzy = 1  /* this entry can contain additional formatting */

Index = Index + 1  /* up the counter */
Settings.Index.!Name = 'TITLEADDRESS'  /* Address type to use for message title */
Settings.Index.!Value = 'FULL'  /* use this as the default value */
Settings.Index.!Single = 1  /* this is a single word entry  */
Settings.Index.!Options = 'ADDRESS FULL NAME'  /* the available options */

Index = Index + 1  /* up the counter */
Settings.Index.!Name = 'VERSION'  /* The build number of the last RexxMail update */
Settings.Index.!Value = 0  /* use this as the default value */
Settings.Index.!Single = 1  /* this is a single word entry  */

Index = Index + 1  /* up the counter */
Settings.Index.!Name = 'VIRUSMAILHEADER'  /* defines the virus mail header to look for */

Index = Index + 1  /* up the counter */
Settings.Index.!Name = 'WORDSEPARATORS'  /* defines the word separators to use in parsing a message's subject line */

do NewIndex = StartIndex to Index  /* take each of the above settings */
 Settings.NewIndex.!Type = 'T'  /* and make it a text type */
end

/**********************************************************************/
/* Define the signal settings options                                 */
/**********************************************************************/

StartIndex = Index + 1  /* a new start index */

Index = Index + 1  /* up the counter */
Settings.Index.!Name = 'SIGNALRECEIVED'  /* defines the signal for incoming mail */

Index = Index + 1  /* up the counter */
Settings.Index.!Name = 'SIGNALSENT'  /* defines the signal for sent mail */

do NewIndex = StartIndex to Index  /* take each of the above settings */
 Settings.NewIndex.!Type = 'S'  /* and make it a signal type */
end

/**********************************************************************/
/* Define the numerical settings options                              */
/**********************************************************************/

StartIndex = Index + 1  /* a new start index */

Index = Index + 1  /* up the counter */
Settings.Index.!Name = 'ASCIILINELENGTH'  /* defines the maximum length of ASCII message lines */
Settings.Index.!Value = 76  /* use this as the default value */
Settings.Index.!Min = 1  /* use this as the minimum value */
Settings.Index.!Max = 998  /* use this as the maximum value */

Index = Index + 1  /* up the counter */
Settings.Index.!Name = 'HTMLLINELENGTH'  /* defines the length of text lines extracted from HTML files (0 = unwrapped) */
Settings.Index.!Value = 76  /* use this as the default value */
Settings.Index.!Min = 0  /* use this as the minimum value */
Settings.Index.!Max = 998  /* use this as the maximum value */

Index = Index + 1  /* up the counter */
Settings.Index.!Name = 'LOGACTIONLINES'  /* defines the number of action log lines we want to keep */
Settings.Index.!Value = ''  /* use this as the default value */
Settings.Index.!Min = 0  /* use this as the minimum value */

Index = Index + 1  /* up the counter */
Settings.Index.!Name = 'LOGERRORLINES'  /* defines the number of error log lines we want to keep */
Settings.Index.!Value = ''  /* use this as the default value */
Settings.Index.!Min = 0  /* use this as the minimum value */

Index = Index + 1  /* up the counter */
Settings.Index.!Name = 'LOGMAILLINES'  /* defines the number of mail log lines we want to keep */
Settings.Index.!Value = ''  /* use this as the default value */
Settings.Index.!Min = 0  /* use this as the minimum value */

Index = Index + 1  /* up the counter */
Settings.Index.!Name = 'MAXADDRESSES'  /* defines the maximum number of addresses to show in a recpients list in incoming messages */
Settings.Index.!Value = 5  /* use this as the default value */
Settings.Index.!Min = 0  /* use this as the minimum value */
              
Index = Index + 1  /* up the counter */
Settings.Index.!Name = 'MAXCOLLECTSIZE'  /* defines the maximum number of bytes of messages to collect automatically from the POP3 server */
Settings.Index.!Value = ''  /* use this as the default value */
Settings.Index.!Min = 0  /* use this as the minimum value */

Index = Index + 1  /* up the counter */
Settings.Index.!Name = 'POP3ATTEMPTS'  /* defines the maximum number of attempts to connect to the POP3 server */
Settings.Index.!Value = 10  /* use this as the default value */
Settings.Index.!Min = 0  /* use this as the minimum value */

Index = Index + 1  /* up the counter */
Settings.Index.!Name = 'REPLYLINELENGTH'  /* defines the maximum length of quoted reply message lines */
Settings.Index.!Value = 76  /* use this as the default value */
Settings.Index.!Min = 0  /* use this as the minimum value */
Settings.Index.!Max = 998  /* use this as the maximum value */

Index = Index + 1  /* up the counter */
Settings.Index.!Name = 'SMTPATTEMPTS'  /* defines the maximum number of attempts to connect to the SMTP server */
Settings.Index.!Value = 10  /* use this as the default value */
Settings.Index.!Min = 0  /* use this as the minimum value */

Index = Index + 1  /* up the counter */
Settings.Index.!Name = 'TIMEATTEMPTS'  /* defines the maximum number of attempts to connect to the time server */
Settings.Index.!Value = 10  /* use this as the default value */
Settings.Index.!Min = 0  /* use this as the minimum value */

do NewIndex = StartIndex to Index  /* take each of the above settings */
 Settings.NewIndex.!Type = 'N'  /* and make it a numerical type */
end

/**********************************************************************/
/* Define the Boolean settings options                                */
/**********************************************************************/

StartIndex = Index + 1  /* a new start index */

Index = Index + 1  /* up the counter */
Settings.Index.!Name = 'ASCIITEXT'  /* ASCIIText = YES: convert outgoing messages to ASCII, word-wrapped at ASCIILineLength characters */

Index = Index + 1  /* up the counter */
Settings.Index.!Name = 'CLOSEATTACHAFTEREDIT'  /* CloseAttachAfterEdit = YES: close outgoing attachments folders after editing messages */

Index = Index + 1  /* up the counter */
Settings.Index.!Name = 'CLOSEATTACHAFTERVIEW'  /* CloseAttachAfterView = YES: close incoming attachments folders after viewing messages */

Index = Index + 1  /* up the counter */
Settings.Index.!Name = 'ERRORMAIL'  /* ErrorMail = YES: send error messages to the user as mail */

Index = Index + 1  /* up the counter */
Settings.Index.!Name = 'HTMLLINES'  /* HTMLLINES = YES: insert separator lines into text extracted from HTML content */

Index = Index + 1  /* up the counter */
Settings.Index.!Name = 'HTMLSAFE'  /* HTMLSAFE = YES: rewrite HTML content to prevent unwanted net access */

Index = Index + 1  /* up the counter */
Settings.Index.!Name = 'HTMLTEXT'  /* HTMLTEXT = YES: extract text from HTML content into message body text */

Index = Index + 1  /* up the counter */
Settings.Index.!Name = 'HTMLURLLIST'  /* HTMLURLLIST = YES: rewrite HTML content with URLs at the end of the text content */

Index = Index + 1  /* up the counter */
Settings.Index.!Name = 'OPENATTACHBEFOREEDIT'  /* OpenAttachBeforeEdit = YES: open outgoing attachments folders when editing messages */

Index = Index + 1  /* up the counter */
Settings.Index.!Name = 'OPENATTACHBEFOREVIEW'  /* OpenAttachBeforeView = YES: open incoming attachments folders when viewing messages */

Index = Index + 1  /* up the counter */
Settings.Index.!Name = 'POP3INTERACTIVE'  /* POP3Interactive = YES: prompt the user for POP3 message retrieval and deletion */

Index = Index + 1  /* up the counter */
Settings.Index.!Name = 'POP3KEEPMESSAGES'  /* POP3KeepMessages = YES: do not delete messages from the POP3 server after retrieval */

Index = Index + 1  /* up the counter */
Settings.Index.!Name = 'SENDIFNOTREADY'  /* SendIfNotReady = YES: outgoing messages will be sent without checking to see if they are set "ready to send" */

Index = Index + 1  /* up the counter */
Settings.Index.!Name = 'SHOWPROGRESS'  /* ShowProgress = YES: show a byte counter and progress bar when sending or collecting mail */

Index = Index + 1  /* up the counter */
Settings.Index.!Name = 'SIGNALS'  /* Signals = YES: sound signals after certain operations */

Index = Index + 1  /* up the counter */
Settings.Index.!Name = 'TITLEKEYWORDS'  /* TitleKeywords= YES: include keywords in message object title lines */

do NewIndex = StartIndex to Index  /* take each of the above settings */
 Settings.NewIndex.!Type = 'B'  /* make it a Boolean type */
 Settings.NewIndex.!Value = 0  /* and make it FALSE */
end

Settings.0 = Index  /* the total number of text, numerical, and Boolean settings */

/**********************************************************************/
/* Define the various command switches                                */
/**********************************************************************/

Commands. = ''  /* empty all commands */

/**********************************************************************/
/* Define the valid command entries that require a mail file argument */
/**********************************************************************/

StartIndex = Index + 1  /* a new start index */

Index = Index + 1  /* up the counter */
Commands.Index.!Name = 'ADDADDRESSES'  /* this one always needs a mail file spec */

Index = Index + 1  /* up the counter */
Commands.Index.!Name = 'CLOSEATTACH'  /* this one always needs a mail file spec */

Index = Index + 1  /* up the counter */
Commands.Index.!Name = 'EDIT'  /* this one always needs a mail file spec */

Index = Index + 1  /* up the counter */
Commands.Index.!Name = 'FORWARD'  /* this one always needs a mail file spec */

Index = Index + 1  /* up the counter */
Commands.Index.!Name = 'OPENATTACH'  /* this one always needs a mail file spec */

Index = Index + 1  /* up the counter */
Commands.Index.!Name = 'RAW'  /* this one always needs a mail file spec */

Index = Index + 1  /* up the counter */
Commands.Index.!Name = 'REPLY'  /* this one always needs a mail file spec */

Index = Index + 1  /* up the counter */
Commands.Index.!Name = 'REPLYTOALL'  /* this one always needs a mail file spec */

Index = Index + 1  /* up the counter */
Commands.Index.!Name = 'REPLYTOORIGINALSENDER'  /* this one always needs a mail file spec */

Index = Index + 1  /* up the counter */
Commands.Index.!Name = 'REPLYTORECIPIENTS'  /* this one always needs a mail file spec */

Index = Index + 1  /* up the counter */
Commands.Index.!Name = 'REPLYTOSENDER'  /* this one always needs a mail file spec */

Index = Index + 1  /* up the counter */
Commands.Index.!Name = 'SEND'  /* this command accepts an optional argument */

Index = Index + 1  /* up the counter */
Commands.Index.!Name = 'SENDBARE'  /* this command accepts an optional multiple file argument */

Index = Index + 1  /* up the counter */
Commands.Index.!Name = 'SETASCII'  /* this one always needs a mail file spec */

Index = Index + 1  /* up the counter */
Commands.Index.!Name = 'SETASCIIQP'  /* this one always needs a mail file spec */

Index = Index + 1  /* up the counter */
Commands.Index.!Name = 'SETNOTOKTOSEND'  /* this one always needs a mail file spec */

Index = Index + 1  /* up the counter */
Commands.Index.!Name = 'SETOKNOTOKTOSEND'  /* this one always needs a mail file spec */

Index = Index + 1  /* up the counter */
Commands.Index.!Name = 'SETOKTOSEND'  /* this one always needs a mail file spec */

Index = Index + 1  /* up the counter */
Commands.Index.!Name = 'SETQP'  /* this one always needs a mail file spec */

Index = Index + 1  /* up the counter */
Commands.Index.!Name = 'SETNEW'  /* this one always needs a mail file spec */

Index = Index + 1  /* up the counter */
Commands.Index.!Name = 'SETRECEIVED'  /* this one always needs a mail file spec */

Index = Index + 1  /* up the counter */
Commands.Index.!Name = 'SETSENT'  /* this one always needs a mail file spec */

Index = Index + 1  /* up the counter */
Commands.Index.!Name = 'SETVIEWED'  /* this one always needs a mail file spec */

Index = Index + 1  /* up the counter */
Commands.Index.!Name = 'VIEW'  /* this one always needs a mail file spec */

do NewIndex = StartIndex to Index  /* take each of the above commands */
 Commands.NewIndex.!Type = 'C'  /* set the type */
end

/**********************************************************************/
/* Define the valid command entries that require a filespec argument  */
/**********************************************************************/

StartIndex = Index + 1  /* a new start index */

Index = Index + 1  /* up the counter */
Commands.Index.!Name = 'ATTACHTOALLINOUT'  /* this one always needs a filespec */

Index = Index + 1  /* up the counter */
Commands.Index.!Name = 'ATTACHTOOPEN'  /* this one always needs a filespec */

Index = Index + 1  /* up the counter */
Commands.Index.!Name = 'MAKECWMF'  /* this one always needs a filespec */

Index = Index + 1  /* up the counter */
Commands.Index.!Name = 'REGISTER'  /* this one always needs a filespec */

Index = Index + 1  /* up the counter */
Commands.Index.!Name = 'TITLE'  /* this one always needs a filespec */

do NewIndex = StartIndex to Index  /* take each of the above commands */
 Commands.NewIndex.!Type = 'R'  /* set the type */
end

/**********************************************************************/
/* Define the valid command entries that take no argument             */
/**********************************************************************/

StartIndex = Index + 1  /* a new start index */

Index = Index + 1  /* up the counter */
Commands.Index.!Name = 'COLLECT'  /* this one takes no argument */

Index = Index + 1  /* up the counter */
Commands.Index.!Name = 'HELP'  /* this one takes no argument */

Index = Index + 1  /* up the counter */
Commands.Index.!Name = 'OPEN'  /* this one takes no argument */

Index = Index + 1  /* up the counter */
Commands.Index.!Name = 'RECURSE'  /* this one takes no argument */

Index = Index + 1  /* up the counter */
Commands.Index.!Name = 'SENDREADY'  /* this one takes no argument */

do NewIndex = StartIndex to Index  /* take each of the above commands */
 Commands.NewIndex.!Type = 'N'  /* set the type */
end

/**********************************************************************/
/* Define the valid command entries that don't care what they get     */
/**********************************************************************/

StartIndex = Index + 1  /* a new start index */

Index = Index + 1  /* up the counter */
Commands.Index.!Name = 'ADDRESSES'  /* this command accepts an optional multiple file argument */

Index = Index + 1  /* up the counter */
Commands.Index.!Name = 'IMPORT'  /* this command accepts an optional multiple file argument */

do NewIndex = StartIndex to Index  /* take each of the above commands */
 Commands.NewIndex.!Type = 'M'  /* set the type */
end

/**********************************************************************/
/* Define the valid command entries that accept single files only     */
/**********************************************************************/

StartIndex = Index + 1  /* a new start index */

Index = Index + 1  /* up the counter */
Commands.Index.!Name = 'EXPORT'  /* this one accepts an optional single file argument */

do NewIndex = StartIndex to Index  /* take each of the above commands */
 Commands.NewIndex._Type = 'S'  /* set the type */
end

/**********************************************************************/
/* Define the valid command entries that take a non-filespec argument */
/**********************************************************************/

StartIndex = Index + 1  /* a new start index */

Index = Index + 1  /* up the counter */
Commands.Index.!Name = 'NEWMESSAGE'  /* this command accepts an optional argument that is not a file spec */

Index = Index + 1  /* up the counter */
Commands.Index.!Name = 'NEWMESSAGEOPEN'  /* this command accepts an optional argument that is not a file spec */

Index = Index + 1  /* up the counter */
Commands.Index.!Name = 'NEWMESSAGESEND'  /* this command accepts an optional argument that is not a file spec */

Index = Index + 1  /* up the counter */
Commands.Index.!Name = 'NEWMESSAGEHERE'  /* this command accepts an optional argument that is not a file spec */

Index = Index + 1  /* up the counter */
Commands.Index.!Name = 'NEWMESSAGEHEREOPEN'  /* this command accepts an optional argument that is not a file spec */

Index = Index + 1  /* up the counter */
Commands.Index.!Name = 'NEWMESSAGEHERESEND'  /* this command accepts an optional argument that is not a file spec */

Index = Index + 1  /* up the counter */
Commands.Index.!Name = 'TOOLBARCREATE'  /* this command accepts an optional argument that is not a file spec */

Index = Index + 1  /* up the counter */
Commands.Index.!Name = 'TOOLBARDELETE'  /* this command accepts an optional argument that is not a file spec */

Index = Index + 1  /* up the counter */
Commands.Index.!Name = 'TOOLBAROPEN'  /* this command accepts an optional argument that is not a file spec */

do NewIndex = StartIndex to Index  /* take each of the above commands */
 Commands.NewIndex.!Type = 'O'  /* set the type */
end

/**********************************************************************/
/* Define the valid command entries that are checked immediately      */
/**********************************************************************/

Index = Index + 1  /* up the counter */
Commands.Index.!Name = 'USER'  /* this command requires a single argument */

Index = Index + 1  /* up the counter */
Commands.Index.!Name = 'SETTINGSFILE'  /* this one requires a single file argument */

Commands.0 = Index  /* the total number of available commands */

/**********************************************************************/
/* Get the operating system, program directory and program name       */
/**********************************************************************/

parse source OS . Global.ProgSpec  /* get the OS and ProgSpec */

Global.ProgDir = strip(filespec('D',Global.ProgSpec)||filespec('P',Global.ProgSpec),'T','\')  /* the program directory name */
Global.ProgName = filespec('N',Global.ProgSpec)  /* the program name */

/**********************************************************************/
/* initialize some more stuff based on the program dir                */
/**********************************************************************/

Global.SettingsUpdate = Global.ProgDir||'\settupda.txt'  /* the settings update text file name */
Global.ErrorLog = Global.ProgDir||'\error.log'  /* the error log name for the time being (until we get our configuration sorted out) */

Global.IconDir = Global.ProgDir||'\Icons'  /* the icon dir */

if (\DirCheck(Global.IconDir)) then  /* if we cannot find it */
do
 call Quit  /* quit */
end

/**********************************************************************/
/* put out an identifying message                                     */
/**********************************************************************/

call lineout 'CON:',Global.BuildMess||Global.CRLF||Global.Copyright||Global.CRLF||Global.Abort||Global.CRLF  /* report to the console */

/**********************************************************************/
/* Make sure this is OS/2                                             */
/**********************************************************************/

if (OS >< 'OS/2') then  /* if the OS is not OS/2 */
do
 call AddError 'Fatal error: this program requires OS/2'  /* report */
 call Quit  /* and quit */
end

/**********************************************************************/
/* Try to load the REXX utilities                                     */
/**********************************************************************/

if (LoadRexxUtil() = 2) then  /* if we cannot load the Rexx utilities library */
do
 call AddError 'Fatal error: cannot load the REXX utilities library'  /* report */
 call Quit  /* and quit */
end

/**********************************************************************/
/* Try to load the REXX socket functions                              */
/**********************************************************************/

if (LoadRexxSock() = 2) then  /* if we cannot load the Rexxx socket functions library */
do
 call AddError 'Fatal error: cannot load the REXX socket functions library'  /* report */
 call Quit  /* and quit */
end

/**********************************************************************/
/* Make sure we have the utilities level we need                      */
/**********************************************************************/

if (rxfuncquery('sysutilversion')) then  /* if we cannot find the SysUtilVersion function */
do
 RexxVersionOK = 0  /* all is not well */
end
else  /* if we can find the SysUtilVersion function */
do
 RexxVersionOK = (sysutilversion() >= '2.00')  /* if the utilities level is at least 2.00, all is well */
end

if (\RexxVersionOK) then  /* if we do not have the right REXX utilities version */
do
 call AddError 'Fatal error: REXXUTIL.DLL is not at the required level'  /* report */
 call Quit  /* and quit */
end

/**********************************************************************/
/* Try to load code page 850                                          */
/**********************************************************************/

OriginalCP = sysqueryprocesscodepage()  /* get the current code page */

if (syssetprocesscodepage(850) >< 0) then  /* if we cannot load code page 850 */
do
 call AddError 'Fatal error: cannot load code page 850'  /* report */
 call Quit  /* and quit */
end

/**********************************************************************/
/* See if the CWMail* classes have been registered                    */
/**********************************************************************/

Global.CWMailFile = 0  /* no CWMailFile class found yet */
Global.CWMailFolder = 0  /* no CWMailFolder class found yet */

if (sysqueryclasslist('ClassList.') = 0) then  /* if we can get the class list */
do Index = 1 to ClassList.0  /* look at each entry */

 if (word(ClassList.Index,1) = 'CWMailFile') then  /* if it is what we want */
 do
  Global.CWMailFile = 1  /* we have the CWMailFile class */
 end

 if (word(ClassList.Index,1) = 'CWMailFolder') then  /* if it is what we want */
 do
  Global.CWMailFolder = 1  /* we have the CWMailFolder class */
 end

end

/**********************************************************************/
/* Get the command-line arguments                                     */
/**********************************************************************/

parse arg Commands  /* get the command-line arguments */

Commands = translate(Commands,' ',d2c(9))  /* convert any TAB characters to spaces */
Commands = strip(Commands,'B',' ')  /* remove any excess whitespace */
Global.StoreCommands = Commands  /* save the command-line arguments for later */

if (left(Commands,1) = '/') then  /* if the arguments start with a switch delimiter */
do
 parse var Commands '/' Commands  /* get rid of the first switch delimiter */
end

else  /* if the arguments do not start with a switch delimiter */

do

 parse var Commands PreCommands '/' Commands  /* get the bit before the first switch delimiter and get rid of the delimiter */

 PreCommands = strip(PreCommands,'B',' ')  /* get rid of excess space */

 if (PreCommands >< '') then  /* if we have something */
 do
  call AddError 'Missing command switch before "'||PreCommands||'"'  /* report an error */
 end

end

NewCommands. = ''  /* start with nothing */
CommCount = 0  /* start at 0 */

do while (Commands >< '')  /* as long as we have commands left */

 parse var Commands Command '/' Commands  /* get the next command sequence */
 Command = strip(Command,'B',' ')  /* remove any excess blanks */
 parse var Command Switch Argument  /* get any bits separated by the first space, if any */
 parse var Command PreEqual '=' PostEqual  /* get any bits separated by an equal sign , if any */

 if (length(Switch) > length(PreEqual)) then  /* if the equal sign came first */
 do
  Switch = strip(PreEqual,'B',' ')  /* use this */
  Argument = strip(PostEqual,'B',' ')  /* use this */
 end
 else  /* if the blank space came before any equal sign */
 do
  Argument = strip(Argument,'B',' ')  /* get rid of excess blanks */
  Argument = strip(Argument,'L','=')  /* get rid of any leading "=" */
  Argument = strip(Argument,'L',' ')  /* get rid of any leading blanks */
 end

 Switch = translate(Switch)  /* make the switch upper case */

 Index = 0  /* start at 0 */

 do until ((Switch = Settings.Index.!Name) | (Index > Settings.0))  /* go on until we find the switch in the settings or run out of options */
  Index = Index + 1  /* up the index */
 end

 if (Index <= Settings.0) then  /* if we found a match, we have a valid setting name */
 do

  select  /* do one of the following */

   when (Settings.Index.!Type = 'T') then  /* if it is a text setting */
   do
    Global.Switch = CheckValText(Switch,Argument,Settings.Index.!Options,Settings.Index.!Single,Settings.Index.!Fuzzy)  /* check and set the value */
   end

   when (Settings.Index.!Type = 'N') then  /* if it is a numerical setting */
   do
    Global.Switch = CheckValNum(Switch,Argument,Settings.Index.!Value,Settings.Index.!Min,Settings.Index.!Max)  /* check and set the value */
   end

   when (Settings.Index.!Type = 'S') then  /* if it is a signal setting */
   do
    Global.Switch = CheckValSignal(Switch,Argument)  /* check and set the value */
   end

   when (Settings.Index.!Type = 'B') then  /* if it is a Boolean setting */
   do

    if (Argument = '') then  /* if we have no argument */
    do
     Global.Switch = 1  /* the setting is TRUE */
    end
    else  /* if we have an argument */
    do
     Global.Switch = CheckValBool(Switch,Argument)  /* check and set the value */
    end

   end

   otherwise  /* if none of the above (which should not occur) */
   do
    call AddError 'Invalid switch type for '||Switch||': '||Settings.Index.!Type  /* report a coding error */
   end

  end

 end

 else  /* if we did not find a match */

 do

  Index = 0  /* start at 0 */

  do until ((Switch = Commands.Index.!Name) | (Index > Commands.0))  /* go on until we find the switch in the commands or run out of options */
   Index = Index + 1  /* up the index */
  end

  if (Index <= Commands.0) then  /* if we found a match */
  do

   select  /* do one of the following */

    when (Switch = 'SETTINGSFILE') then  /* if the switch is this one */
    do
     Global.SettingsFile = strip(Argument,'B','"')  /* remove any quotation marks and this will be our settings file (which may be empty) */
    end

    when (Switch = 'USER') then  /* if the switch is this one */
    do
     Global.User = strip(Argument,'B','"')  /* remove any quotation marks and this will be the user name (which may be empty) */
    end

    when (Switch = 'HELP') then  /* if the switch is this one */
    do
     Global.Errors = ''  /* skip any errors */
     call lineout 'CON:','Usage: '||Global.ProgName||' /<switch> [=] [<argument>] [...]'  /* show a minimal help message */
     call lineout 'CON:',''  /* show a minimal help message */
     call lineout 'CON:','The program must be called with at least one command switch, which may be'  /* show a minimal help message */
     call lineout 'CON:','followed by one or more arguments and/or any combination of additional command'  /* show a minimal help message */
     call lineout 'CON:','and/or settings switches and arguments.'  /* show a minimal help message */
     call lineout 'CON:',''  /* empty line */
     call lineout 'CON:','Switches and arguments must be separated by whitespace and/or a single "=".'  /* show a minimal help message */
     call lineout 'CON:','Switch/argument combinations must be separated by whitespace.'  /* show a minimal help message */
     call lineout 'CON:',''  /* empty line */
     call lineout 'CON:','For more information, see the RexxMail Reference Guide and RexxMail Tutorial.'  /* show a minimal help message */
     call Quit  /* and quit */
    end

    otherwise  /* if the switch is something else */
    do
     CommCount = CommCount + 1  /* up the command counter */
     NewCommands.CommCount.!Switch = Switch  /* store the switch */
     NewCommands.CommCount.!Argument = Argument  /* store the argument */
     NewCommands.CommCount.!Type = Commands.Index.!Type  /* store the command argument type */
    end

   end

  end

  else  /* if we did not find a match */

  do
   call AddError 'Invalid switch: '||Switch  /* report */
  end

 end

end

/**********************************************************************/
/* Look for the location file                                         */
/**********************************************************************/

Global.LocationFile = Global.ProgDir||'\location.txt'  /* the user location text file name */

if (\FileCheck(Global.LocationFile,1)) then  /* if we cannot find a location file */
do
 call Quit  /* quit */
end

/**********************************************************************/
/* Determine the user name if we need it. and the mail dir location   */
/**********************************************************************/

if (symbol('Global.User') >< 'VAR') then  /* if the user name has not yet been set through the command line (which may have set it empty) */
do
 Global.User = value('USER',,'OS2ENVIRONMENT')  /* try to get the user name from the OS/2 environment */
end

if (Global.User = '') then  /* if we still have no user name */
do
 Global.User = 'DEFAULT'  /* use this */
end

Global.MainDir = GetFileEntry(Global.LocationFile,Global.User)  /* look for the user's location in the location file */

if (Global.MainDir = '') then  /* if we have no directory spec */
do
 call AddError 'No user location defined for user "'||Global.User||'"'  /* report */
 call Quit  /* and quit */
end

if (\DirCheck(Global.MainDir)) then  /* if the directory does not exist */
do
 call Quit  /* quit */
end

/**********************************************************************/
/* Set the user's configuration dir                                   */
/**********************************************************************/

Global.ConfDir = Global.MainDir||'\Configuration'  /* the directory for the configuration files */

if (\DirCheck(Global.ConfDir)) then  /* if the dir does not exist */
do
 call Quit  /* quit */
end

/**********************************************************************/
/* Set a few file names in the Configuration dir                      */
/**********************************************************************/

Global.ControlColl = Global.ConfDir||'\collect.txt'  /* the collect control file */
Global.ControlRegi = Global.ConfDir||'\register.txt'  /* the register control file */
Global.ControlSend = Global.ConfDir||'\send.txt'  /* the send control file */
Global.ControlView = Global.ConfDir||'\view.txt'  /* the view control file */

Global.Addresses = Global.ConfDir||'\addresses.txt'  /* the default addresses list file */
Global.MIMETypes = Global.ConfDir||'\mimetype.txt'  /* the list of MIME types */

Global.ActionLog = Global.ConfDir||'\action.log'  /* the action log name */
Global.ErrorLog = Global.ConfDir||'\error.log'  /* the error log name */
Global.MailLog = Global.ConfDir||'\mail.log'  /* the mail log name */

/**********************************************************************/
/* initialize more global directory variables based on Global.MainDir */
/**********************************************************************/

Global.MessagesDir = Global.MainDir||'\Messages'  /* try this for the directory containing the primary message folders */

if (\DirCheck(Global.MessagesDir,1)) then  /* if the "Messages" subdir does not exist (quiet check) */
do
 Global.MessagesDir = Global.MainDir  /* the directory containing the primary message folders is the main dir */
end

Global.InDir = Global.MessagesDir||'\In'  /* the directory for incoming mail */

if (\DirCheck(Global.InDir)) then  /* if the dir does not exist */
do
 call Quit  /* quit */
end

Global.OutDir = Global.MessagesDir||'\Out'  /* the directory for outgoing mail */

if (\DirCheck(Global.OutDir)) then  /* if the dir does not exist */
do
 call Quit  /* quit */
end

Global.InArchDir = Global.MessagesDir||'\In Archive'  /* the directory for the incoming mail archive */

if (\DirCheck(Global.InArchDir)) then  /* if the dir does not exist */
do
 call Quit  /* quit */
end

Global.OutArchDir = Global.MessagesDir||'\Out Archive'  /* the directory for the outgoin mail archive */

if (\DirCheck(Global.OutArchDir)) then  /* if the dir does not exist */
do
 call Quit  /* quit */
end

Global.AccessDir = Global.MainDir||'\Accessories'  /* the directory for the accessories */

if (\DirCheck(Global.AccessDir)) then  /* if the dir does not exist */
do
 call Quit  /* quit */
end

Global.AddrDir = Global.MessagesDir||'\Addresses'  /* the directory for the address templates and subfolders */

if (\DirCheck(Global.AddrDir)) then  /* if the dir does not exist */
do
 call Quit  /* quit */
end

Global.TempDir = Global.MainDir||'\Temp'  /* use this as the directory for temporary files */

if (\DirCheck(Global.TempDir)) then  /* if the dir does not exist */
do
 call Quit  /* quit */
end

/**********************************************************************/
/* Define a directory location for mailing list address files         */
/**********************************************************************/

Global.ListDir = Global.ConfDir||'\Lists'  /* the lists directory */

/**********************************************************************/
/* Set the settings file name if we did not get it yet                */
/**********************************************************************/

if (symbol('Global.SettingsFile') >< 'VAR') then  /* if we have no settings file variable defined yet (and it may be empty if we do not want to use a settings file) */
do
 Global.SettingsFile = Global.ConfDir||'\settings.txt'  /* use the default RexxMail settings file name */
end

/**********************************************************************/
/* Get more global variables from the settings file                   */
/**********************************************************************/

if (Global.SettingsFile >< '') then  /* if we have a settings file name */
do

 InvalidSettings = 0  /* we have no invalid settings yet */

 if (\FileCheck(Global.SettingsFile,1)) then  /* if the file does not exist */
 do

  call AddError 'Creating new settings file "'||Global.SettingsFile||'"'  /* report an error to block further command processing */

  if (CommCount > 0) then  /* if we have any commands */
  do
   call AddError 'Aborting command processing'  /* report */
  end

 end

 else  /* if the file exists */

 do

  if (FileOpen(Global.SettingsFile,'READ')) then  /* if we can open the RexxMail settings file for reading */
  do

   SettingsCont = charin(Global.SettingsFile,1,chars(Global.SettingsFile))  /* get the settings file contents */
   call FileClose Global.SettingsFile  /* close the RexxMail settings file */
   SettingsCont = strip(SettingsCont,'T',d2c(26))  /* remove any EOF marker */

   do while (SettingsCont >< '')  /* go on until we run out of content */

    parse var SettingsCont Line (Global.CRLF) SettingsCont  /* get the next line */
    Line = translate(Line,d2c(32),d2c(9))  /* convert TABs to spaces */
    Line = strip(Line,'B',' ')  /* get rid of excess spaces */

    if (Line >< '') then  /* if the line is not empty */
    do

     if (left(Line,1) >< '#') then  /* if it is not a comment */
     do

      parse var Line Switch '=' Argument  /* get the parts we need */
      Switch = translate(strip(Switch))  /* get rid of superfluous spaces and make it upper case so the variable tail will work */
      Argument = strip(Argument)  /* get rid of superfluous spaces */
      Index = 0  /* start at 0 */

      do until ((Switch = Settings.Index.!Name) | (Index > Settings.0))  /* go on until we find the switch in the settings or run out of options */
       Index = Index + 1  /* up the index */
      end

      if (Index <= Settings.0) then  /* if we found a match, we have a valid setting name */
      do

       if (symbol('Global.'||Switch) >< 'VAR') then  /* if this one was not set through the command line */
       do

        select  /* do one of the following */

         when (Settings.Index.!Type = 'T') then  /* if it is a text setting */
         do
          Global.Switch = CheckValText(Switch,Argument,Settings.Index.!Options,Settings.Index.!Single,Settings.Index.!Fuzzy)  /* check and set the value */
         end

         when (Settings.Index.!Type = 'N') then  /* if it is a numerical setting */
         do
          Global.Switch = CheckValNum(Switch,Argument,Settings.Index.!Value,Settings.Index.!Min,Settings.Index.!Max)  /* check and set the value */
         end

         when (Settings.Index.!Type = 'S') then  /* if it is a signal setting */
         do
          Global.Switch = CheckValSignal(Switch,Argument)  /* check and set the value */
         end

         when (Settings.Index.!Type = 'B') then  /* if it is a Boolean setting */
         do

          if (Argument = '') then  /* if we have no argument */
          do
           Global.Switch = 0  /* this setting is FALSE */
          end
          else  /* if we have an argument */
          do
           Global.Switch = CheckValBool(Switch,Argument)  /* check and set the value */
          end

         end

         otherwise  /* if none of the above (which should not occur) */
         do
          call AddError 'Invalid switch type for '||Switch||': '||Settings.Index.!Type  /* report a coding error */
         end

        end

       end

      end

      else  /* if we did not find a match, it was not a valid entry */

      do
       InvalidSettings = 1  /* we have at least one invalid setting */
      end

     end

    end

   end

  end

 end

end

/**********************************************************************/
/* Set any remaining global variables                                 */
/**********************************************************************/

do Index = 1 to Settings.0  /* take each of the possible settings */

 if (symbol('Global.'||Settings.Index.!Name) >< 'VAR') then  /* if this one was not set through the command line or the settings file */
 do
  SettingName = Settings.Index.!Name  /* get the name */
  Global.SettingName = Settings.Index.!Value  /* and set the default value of the relevant global var (which may be empty) */
 end

end

/**********************************************************************/
/* Set some defaults that require dir info                            */
/**********************************************************************/

Global.PaperDir = Global.IconDir||'\'||Global.Stationery  /* the stationery directory to use */

if (Global.ClosingText >< '') then  /* if a closing text file has been specified */
do

 if (filespec('D',Global.ClosingText) = '') then  /* if we have no drive spec */
 do
  Global.ClosingText = Global.ConfDir||'\'||Global.ClosingText  /* use this */
 end

 if (\FileCheck(Global.ClosingText,1)) then  /* if we cannot find the file */
 do
  Global.ClosingText = ''  /* use nothing */
 end

end


/**********************************************************************/
/* See if we need to update the configuration file                    */
/**********************************************************************/

if (Global.SettingsFile >< '') then  /* if we have a settings file */
do

 if (Global.Version = '') then  /* if we found no version number */
 do
  Global.Version = 0  /* assume 0 */
 end

 if (Global.Build >> Global.Version) then  /* if we are running a later build of RexxMail */
 do

  if (\UpdateSettings(Global.SettingsFile,Global.SettingsUpdate)) then  /* if we cannot update the configuration file */
  do
   call Adderror 'Cannot update the configuration file'  /* report */
  end

  GotCWMFC = 0  /* we have no CWMailFile class yet */
  call sysqueryclasslist('ClassList.')  /* get the class list */

  do Index = 1 to ClassList.0  /* look at each entry */

   if (word(ClassList.Index,1) = 'CWMailFile') then  /* if it is what we want */
   do
    GotCWMFC = 1  /* we have a CWMailFile class */
   end

  end

  if (GotCWMFC) then  /* if the class is registered */
  do

   call LogAction 'Setting mail folder sorting and details class to CWMailFile'  /* report */
   call sysfiletree Global.MainDir||'\*','Folders.','DOS'  /* get all the mail folder names */

   do Index = 1 to Folders.0  /* take each one found */
    call syssetobjectdata Folders.Index,'DETAILSCLASS=CWMailFile;SORTCLASS=CWMailFile'  /* set the sort and details class to CWMailFile */
   end

  end

 end

end

/**********************************************************************/
/* Check the command-line commands we collected earlier               */
/**********************************************************************/

Recurse = ''  /* nothing means recurse = off */

if (CommCount = 0) then  /* if we have no valid commands at all */
do
 call AddError 'Missing command'  /* report */
end

else  /* if we have one or more commands */

do Index = 1 to CommCount  /* run through the commands we collected */

 NewCommands.Index.!Files. = ''  /* start with nothing */
 FileCount = 0  /* start a file counter at 0 */

 if (NewCommands.Index.!Argument = '') then  /* if we have no argument */
 do

  select  /* do one of the following */

   when (NewCommands.Index.!Switch = 'RECURSE') then  /* if the switch is this special case */
   do
    Recurse = 'S'  /* use this in the sysfiletree command later on when we go looking for files */
   end

   when (wordpos(NewCommands.Index.!Switch,'EXPORT IMPORT') > 0) then  /* if the switch is one of these special cases */
   do
    NewCommands.Index.!Argument = Global.Addresses  /* the address file to use */
   end

   when (NewCommands.Index.!Switch = 'SENDREADY') then  /* if the switch is this special case */
   do
    NewCommands.Index.!Argument = Global.OutDir||'\*'  /* use any files that may be in the Out dir */
   end

   when (pos(NewCommands.Index.!Type,'CR') > 0) then  /* if the switch is a command that requires an argument */
   do

    call AddError NewCommands.Index.!Switch||' requires an argument'  /* start a message */

    if (NewCommands.Index.!Type = 'C') then  /* if the switch is a command that requires a mail file spec */
    do
     call AddError 'If you double-clicked a program object, drop a mail file on it instead'  /* finish the message */
    end

   end

   otherwise  /* if none of the above apply */
   do
    nop  /* nothing to do */
   end

  end

 end

 else  /* if we have an argument */
  
 do

  if (NewCommands.Index.!Type = 'N') then  /* if the switch is a command that cannot take an argument */
  do
   call AddError NewCommands.Index.!Switch||' cannot take an argument'  /* report */
   NewCommands.Index.!Argument = ''  /* we no longer have an argument */
  end

 end
  
 if ((NewCommands.Index.!Argument >< '') & (NewCommands.Index.!Type >< 'O')) then  /* if we have an argument and it should not be the "other" type, assume it is a file spec */
 do

  do while (NewCommands.Index.!Argument >< '')  /* as long as we have something left */

   NewCommands.Index.!Argument = strip(NewCommands.Index.!Argument)  /* get rid of excess blanks */

   if (left(NewCommands.Index.!Argument,1) = '"') then  /* if the specs start with a double quotation mark */
   do
    NewCommands.Index.!Argument = strip(NewCommands.Index.!Argument,'L','"')  /* remove all leading quotation marks (in case we have double pairs) */
    parse var NewCommands.Index.!Argument FileLoc '"' NewCommands.Index.!Argument  /* get the bit before the next double quotation marks */
    NewCommands.Index.!Argument = strip(NewCommands.Index.!Argument,'L','"')  /* remove all leading quotation marks (in case we had double pairs) */
   end
   else  /* if the specs do not start with a double quotation mark */
   do
    parse var NewCommands.Index.!Argument FileLoc NewCommands.Index.!Argument  /* get the next space-delimited bit */
   end

   if (FileLoc >< '') then  /* if we have something */
   do

    if (NewCommands.Index.!Type = 'S') then  /* if the switch accepts a single (output) file spec only (which need not exist) */
    do

     if (verify(FileLoc,'?*','M') > 0) then  /* if the filespec contains wildcards */
     do
      call AddError NewCommands.Index.!Switch||' does not accept wildcards'  /* report an error */
     end
     else  /* if the filespec does not contain wildcards */
     do
      FileCount = FileCount + 1  /* up the file counter */
      NewCommands.Index.!Files.FileCount = FileLoc  /* store the file name */
     end

    end

    else  /* if the switch is another type, taking one or more existing files */

    do

     call sysfiletree FileLoc,'LocFiles.','FO'||Recurse  /* look for files */

     if (LocFiles.0 = 0) then  /* if we find nothing */
     do

      if (NewCommands.Index.!Switch >< 'SENDREADY') then  /* if it is not this special case */
      do
       call AddError 'Cannot find file "'||FileLoc||'"'  /* report an error */
      end

     end

     else  /* if we find files */

     do FileLocCount = 1 to LocFiles.0  /* take each of the files found */
      FileCount = FileCount + 1  /* up the file counter */
      NewCommands.Index.!Files.FileCount = LocFiles.FileLocCount  /* store the file name */
     end

    end

   end

  end

  if (FileCount = 0) then  /* if we found no files at all */
  do

   if (NewCommands.Index.!Switch >< 'SENDREADY') then  /* unless the switch is this special case */
   do
    call AddError NewCommands.Index.!Switch||': no files found'  /* report an error */
   end

  end

  else  /* if we did find files */

  do

   if ((FileCount > 1) & (NewCommands.Index.!Type = 'S')) then  /* if we found more than one file and the command is one that only takes a single file spec */
   do
    call AddError NewCommands.Index.!Switch||' cannot process multiple files'  /* report an error */
   end

  end

 end

 NewCommands.Index.!Files.0 = FileCount  /* store the file counter */

end

/**********************************************************************/
/* If we have collected any errors, abort further processing          */
/**********************************************************************/

if (Global.Errors >< '') then  /* if we have errors */
do
 call Quit  /* report and quit */
end

/**********************************************************************/
/* Run the command-line commands                                      */
/**********************************************************************/

do Index = 1 to CommCount  /* run through the commands we collected */

 select  /* do one of the following */

  when (NewCommands.Index.!Switch = 'ADDRESSES') then  /* if the switch is "addresses" */
  do
   if (NewCommands.Index.!Files.0 > 0) then  /* if we have files to process */
   do
    call ProcessFiles 'AddressEntry'  /* call the address book procedure */
   end
   else  /* if we have no files to process */
   do
    call sysopenobject Global.AddrDir,0,1  /* open the addresses folder */
   end
  end

  when (NewCommands.Index.!Switch = 'ADDADDRESSES') then  /* if the switch is "addaddresses" */
  do
   call ProcessFiles 'AddressAdd'  /* call the add addresses procedure */
  end

  when (left(NewCommands.Index.!Switch,8) = 'ATTACHTO') then  /* if the switch starts with "AttachTo" */
  do
   call ProcessFiles 'CopyAttachment "'||NewCommands.Index.!Switch||'"',1  /* call the attachment copy procedure with the switch as argument and show progress */
  end

  when (NewCommands.Index.!Switch = 'CLOSEATTACH') then  /* if the switch is "closeattach" */
  do
   call ProcessFiles 'AttDirShut'  /* call the shut attachments folder procedure */
  end

  when (NewCommands.Index.!Switch = 'COLLECT') then  /* if the switch is "collect" */
  do

   if (MailCollect() > 0) then  /* if we collected any messages from the POP3 server */
   do

    if (wordpos('WARNICON',translate(Global.CollectAction)) > 0) then  /* if we want a warning icon */
    do
     call MailWarning  /* show the mail warning icon */
    end

    if (wordpos('OPENFOLDERS',translate(Global.CollectAction)) > 0) then  /* if we want the user folders opened */
    do
     call OpenFolders  /* open the mail folders on the desktop */
    end

   end

  end

  when (NewCommands.Index.!Switch = 'EDIT') then  /* if the switch is  "edit" */
  do
   call ProcessFiles 'EditMessage'  /* edit the message (or view it if it was sent before) */
  end

  when (NewCommands.Index.!Switch = 'EXPORT') then  /* if the switch is "export" */
  do
   call AddressExport NewCommands.Index.!Files.1  /* call the mail address book export procedure with the output file name, if any */
  end

  when (NewCommands.Index.!Switch = 'FORWARD') then  /* if the switch is "forward" */
  do
   call ProcessFiles 'MakeForwardReplyMessage "'||NewCommands.Index.!Switch||'"'  /* call the message forward/reply procedure with the switch as argument */
  end

  when (NewCommands.Index.!Switch = 'IMPORT') then  /* if the switch is "import" */
  do
   call ProcessFiles 'AddressImport'  /* call the mail address book import procedure */
  end

  when (NewCommands.Index.!Switch = 'MAKECWMF') then  /* if the switch is "makecwmf" */
  do
   call ProcessFiles 'MakeCWMFMessage',1  /* convert the message and show progress */
  end

  when (left(NewCommands.Index.!Switch,10) = 'NEWMESSAGE') then  /* if the switch starts with "newmessage" */
  do
   call MakeNewMessage NewCommands.Index.!Argument,NewCommands.Index.!Switch  /* make a new message, passing the original argument and the switch */
  end

  when (NewCommands.Index.!Switch = 'OPEN') then  /* if the switch is "open" */
  do
   call OpenFolders  /* call the open mail folders procedure */
  end

  when (NewCommands.Index.!Switch = 'OPENATTACH') then  /* if the switch is "openattach" */
  do
   call ProcessFiles 'AttDirShow'  /* call the show attachments folder procedure */
  end

  when (NewCommands.Index.!Switch = 'RAW') then  /* if the switch is "raw" */
  do
   call ProcessFiles 'ViewRawMessage'  /* call the raw message viewing procedure */
  end

  when (NewCommands.Index.!Switch = 'REGISTER') then  /* if the switch is "register" */
  do
   call ProcessFiles 'RegisterMessage',1  /* call the mail register procedure and show progress */
  end

  when (left(NewCommands.Index.!Switch,5) = 'REPLY') then  /* if the switch starts with "reply" */
  do
   call ProcessFiles 'MakeForwardReplyMessage "'||NewCommands.Index.!Switch||'"'  /* call the message forward/reply procedure with the switch as argument */
  end

  when (left(NewCommands.Index.!Switch,4) = 'SEND') then  /* if the switch starts with "send" */
  do
   call ProcessFiles 'SendMessage '||(NewCommands.Index.!Switch = 'SENDBARE')  /* send the message, setting a "bare" flag if necessary */
  end

  when (left(NewCommands.Index.!Switch,3) = 'SET') then  /* if the switch starts with "set" */
  do
   call ProcessFiles 'SetMessageType "'||NewCommands.Index.!Switch||'"',1  /* call the message settings procedure with the switch as argument and show progress */
  end

  when (NewCommands.Index.!Switch = 'TITLE') then  /* if the switch is "title" */
  do
   call ProcessFiles 'SetTitle',1  /* call the title set procedure and show progress */
  end

  when (left(NewCommands.Index.!Switch,7) = 'TOOLBAR') then  /* if the switch starts with "toolbar" */
  do
   call Toolbar NewCommands.Index.!Switch,NewCommands.Index.!Argument  /* call the toolbar create procedure with the full switch and argument */
  end

  when (NewCommands.Index.!Switch = 'VIEW') then  /* if the switch was "view" */
  do
   call ProcessFiles 'ViewMessage'  /* call the message viewing procedure */
  end

  otherwise  /* if none of the above applies */
  do
   call AddError 'Command parsing error: "'||NewCommands.Index.!Switch||'"'  /* report */
   call Quit  /* quit -- but this should never arise */
  end

 end

end

call Quit  /* that's all, folks! */

/**********************************************************************/
ProcessFiles:  /* processes a number of files in succession and show progress */
/**********************************************************************/

parse arg ProcessName,Show  /* get the process name to use */

Show = (Show = 1)  /* show if Show is 1 */

ProcessName = 'call '||ProcessName  /* add the call keyword */

if (Show) then  /* if we are to show progress */
do
 call syscurstate 'OFF'  /* hide the cursor */
 call charout 'CON:','Files to process: '||NewCommands.Index.!Files.0  /* show the total number of files */
end

do LocalIndex = 1 to NewCommands.Index.!Files.0  /* take each of the files we found */

 Global.ProcFile = NewCommands.Index.!Files.LocalIndex  /* copy the file name to a global var (interpret does not work with long file names) */

 interpret ProcessName  /* run the process */

 if (Show) then  /* if we are to show progress */
 do
  call charout 'CON:',copies(d2c(8),length(NewCommands.Index.!Files.0))||,  /* report */
                      right(NewCommands.Index.!Files.0 - LocalIndex,length(NewCommands.Index.!Files.0),' ')  /* show the remaining no. of files to process */
 end

end

if (Show) then  /* if we are to show progress */
do
 call lineout 'CON:',' - ready.'  /* show this */
 call syscurstate 'ON'  /* show the cursor */
end

return  /* end of ProcessFiles */

/**********************************************************************/
Syntax:  /* handles syntax errors */
/**********************************************************************/

call FatalError 'REXX did not recognize the instruction',Sigl  /* start with this */
call Quit  /* and quit */

/**********************************************************************/
Error:  /* handles general errors */
/**********************************************************************/

call FatalError 'External program error',Sigl  /* start with this */
call Quit  /* and quit */

/**********************************************************************/
Failure:  /* handles external program errors */
/**********************************************************************/

call FatalError 'External program failure',Sigl  /* start with this */
call Quit  /* and quit */

/**********************************************************************/
NoValue:  /* handles initialization errors */
/**********************************************************************/

call FatalError 'Initialization error',Sigl  /* start with this */
call Quit  /* and quit */

/**********************************************************************/
FatalError: procedure expose Global.  /* reports a fatal error */
/**********************************************************************/

parse arg ErrorStart,Sigl  /* get the argument */

call AddError 'Fatal error: '||ErrorStart||' on line '||Sigl||':'||Global.CRLF,  /* report */
              '    '||sourceline(Sigl)  /* report */

return 1  /* end of FatalError */

/**********************************************************************/
Halt:  /* handles halt condition */
/**********************************************************************/

call AddError 'Program aborted by user'  /* report */
call Quit  /* and quit */

/**********************************************************************/
Quit:  /* handles normal and premature exits */
/**********************************************************************/

do Index = 1 to Global.Hidden.0  /* take each of the file objects that were hidden by other RexxMail processes */

 if (\UnhideObject(Global.Hidden.Index)) then  /* if we cannot unhide it */
 do
  call AddError 'Cannot unhide "'||Global.Hidden.Index||'"'  /* report */
 end

end

Errors = LogErrors(Global.Errors)  /* show and log any waiting error messages */

if (symbol('OriginalCP') = 'VAR') then  /* if the original code page is defined (it may not be if we did not even get that far) */
do
 call syssetprocesscodepage OriginalCP  /* restore the original code page */
end

exit Errors  /* quit */

/**********************************************************************/
AddError: procedure expose Global.  /* add an error string */
/**********************************************************************/

parse arg Message  /* get the argument */

Global.Errors = Global.Errors||'- '||Message||Global.CRLF  /* add the message to the existing error string */

return  /* end of AddError */

/**********************************************************************/
AddressAdd: procedure expose Global.  /* copy (address template) mail file to addresses folder(s) of open RexxMail messages */
/**********************************************************************/

AddrFile = Global.ProcFile  /* the file to process */

if (sysqueryswitchlist('ListItems.') = 0) then  /* if we can get a switch list */
do

 do Index = 1 to ListItems.0  /* take each of the items found  */

  FileName = filespec('N',ListItems.Index)  /* try to extract a file name from the list item */

  if ((left(FileName,4) = 'RXML') & (right(FileName,5) = '.EDIT') & (datatype(substr(FileName,5,4),'W'))) then  /* if we find a RexxMail edit file name */
  do

   AttDir = Global.TempDir||'\'||left(FileName,8)  /* the corresponding attachments dir */

   if (AttDirGet(AddrFile) >< AttDir) then  /* if it is not our own attachments dir */
   do

    MessAddDir = AttDir||'\'||Global.ToCcBcc  /* this should be the messages's additional addresses folder */

    if (syscopyobject(AddrFile,MessAddDir)) then  /* if we can copy the file to the additional addresses folder */
    do

     NewAddrFile = MessAddDir||'\'||filespec('N',AddrFile)  /* this should be the new addresses file spec */

     if (MessageContents(NewAddrFile)) then  /* if we can get the new address file contents */
     do
      NewTitle = MakeTitle(NewAddrFile,0,0,1,0)  /* get a title with just the addresses in it */
      call syssetobjectdata NewAddrFile,'TITLE='||NewTitle  /* change the file title */
     end

     call LogAction 'Address file copied to message addresses folder "'||MessAddDir||'"',1  /* report, quietly */

    end

   end

  end

 end

end

return 1  /* end of AddressAdd */

/**********************************************************************/
AddressCheck: procedure expose Global.  /* check address components */
/**********************************************************************/

parse arg FirstBit,LastBit  /* get the arguments */

Group = (pos(':;',FirstBit) > 0)  /* have we got a group here? */

if (Group && (words(LastBit) = 0)) then  /* if we have a group and the last bit is not empty, or vice versa */
do
 return ''  /* return nothing */
end

BadRange = '<>()[]@,"'||,  /* we can't have these in a bare (unbracketed) e-mail address */
           xrange('00'x,'1F'x)||,  /* nor these */
           xrange('7F'x,'FF'x)  /* nor these */

if (\Group) then  /* if it is not a group */
do

 BadRange = BadRange||';:\'  /* we cannot have these either */
 GotStartBracket = (left(FirstBit,1) = '<')  /* have we got a leading angle bracket on the first bit? */
 GotEndBracket = (right(LastBit,1) = '>')  /* have we got a trailing angle bracket on the last bit? */

 if (GotStartBracket >< GotEndBracket) then  /* if the start of the first bit and the end of the last bit do not match */
 do
  return ''  /* return nothing */
 end

 if (GotStartBracket) then  /* if the first bit has a leading angle bracket */
 do

  FirstBit = strip(FirstBit,'L','<')  /* get rid of the leading angle bracket on the first bit */

  if (words(FirstBit) = 0) then  /* if the first bit no longer contains a word */
  do
   return ''  /* return nothing */
  end

 end

 if (GotEndBracket) then  /* if we have a trailing angle bracket on the last bit */
 do

  LastBit = strip(LastBit,'T','>')  /* get rid of the trailing angle bracket */

  if (words(LastBit) = 0) then  /* if the last bit no longer contains a word */
  do
   return ''  /* return nothing */
  end

 end

end

if (verify(FirstBit||LastBit,BadRange,'M',1) > 0) then  /* if the first bit and/or the last bit contains illegal characters */
do
 return ''  /* return nothing */
end

if (Group) then  /* if we are dealing with a group */
do
 return FirstBit  /* this is the complete address */
end

return FirstBit||'@'||LastBit  /* end of AddressCheck */

/**********************************************************************/
AddressExport: procedure expose Global.  /* exports address book contents and edits address text file */
/**********************************************************************/

parse arg ExpFile  /* get the file name to process */

BakFile = ExpFile||'.bak'  /* the backup file */

call LogAction 'Exporting address data to "'||ExpFile||'"'  /* report */

if (FileCheck(ExpFile)) then  /* if the old address file already exists */
do

 if (FileCheck(BakFile)) then  /* if the backup file already exists */
 do

  if (sysfiledelete(BakFile) > 0) then  /* if we cannot delete the old backup file */
  do
    call AddError 'Cannot delete "'||BakFile||'"'  /* report */
    return 0  /* and quit */
  end

 end

 if (\FileOpen(ExpFile,'READ')) then  /* if we can't open the export file for reading */
 do
  return 0  /* return without success */
 end

 if (\FileOpen(BakFile)) then  /* if we can't open a new backup file for writing */
 do
  return 0  /* return without success */
 end

 call charout BakFile,charin(ExpFile,1,chars(ExpFile))  /* get the export file contents and write them to the backup file */
 call FileClose BakFile  /* close the backup file */
 call FileClose ExpFile  /* close the export file */

 if (sysfiledelete(ExpFile) > 0) then  /* if we cannot delete the existing export file */
 do
   call AddError 'Cannot delete "'||ExpFile||'"'  /* report */
   return 0  /* and quit */
 end

end

OutText = ''  /* start with nothing */

call sysfiletree Global.AddrDir||'\*','Files.','FOS'  /* get the address book contents */

do Index = 1 to Files.0  /* take each of the objects */

 if (\FileOpen(Files.Index,'READ')) then  /* try to open the file for reading, and if we cannot do so */
 do
  return 0  /* and quit */
 end

 Lines = charin(Files.Index,1,chars(Files.Index))  /* store the contents of the file in an input buffer */

 call FileClose Files.Index  /* close the file */

 parse var Files.Index (Global.AddrDir) '\' Name  /* get the bit of the file spec we want */
 OutText = OutText||'<TITLE>'||Global.CRLF||Name||Global.CRLF  /* add the file name to the output buffer with a header */

 if (Lines >< '') then  /* if we have lines */
 do
  OutText = OutText||'<TEXT>'||Global.CRLF||Lines||Global.CRLF  /* add them to the output buffer with a header */
 end

 Comments = CommentsGet(Files.Index)  /* get the file's comments, if any */

 if (Comments >< '') then  /* if we have something */
 do
  OutText = OutText||'<COMMENTS>'||Global.CRLF||Comments||Global.CRLF  /* add it to the output buffer with a header */
 end

 OutText = OutText||'<END>'||Global.EmptyLine  /* add an ending to the output buffer, and an empty line */

end

if (\FileOpen(ExpFile,'WRITE')) then  /* if we cannot open the address file for writing */
do
 return 0  /* and quit */
end

call charout ExpFile,OutText  /* write the contents of the output buffer to the file */
call FileClose ExpFile  /* close the file */

return 1  /* end of AddressExport */

/**********************************************************************/
AddressFormat: procedure expose Global.  /* formats a string of one or more @ddresses and optionally checks address syntax */
/**********************************************************************/

parse arg Addresses,Indent,Check  /* get the arguments */

First = 1  /* a flag to mark the first address */
NewAddresses = ''  /* start with no new addresses */
Check = (Check = 1)  /* if the check arg is 1, we need to check the addresses */

if (Indent = '') then  /* if we have no indent value, we want bare addresses */
do
 Spacer = ' '  /* the spacer is a blank space */
end
else  /* if we have an indent value (which may be 0) */
do
 Spacer = ','||Global.CRLF||copies(' ',Indent)  /* the spacer is an (indented) new line after a comma */
end

do while (Addresses >< '')  /* as long as we have addresses */

 Addresses = strip(Addresses,'L',' ')  /* remove leading spaces */

 do while (left(word(Addresses,1),1) = ',')  /* as long as the next address starts with a comma */
  Addresses = strip(Addresses,'L',',')  /* remove leading commas */
  Addresses = strip(Addresses,'L',' ')  /* remove leading spaces */
 end

 if (Addresses >< '') then  /* if we have something left */
 do

  if (pos(Global.Warning,Addresses) = 1) then  /* if the next address starts with a marker */
  do
   Addresses = substr(Addresses,length(Global.Warning) + 1)  /* use the rest */
  end

  OrgAddresses = Addresses  /* save the original addresses string */
  Comment = ''  /* we have no leading comment yet */
  EndComment = ''  /* we have no trailing comment yet */

  if (left(Addresses,1) = '"') then  /* if it starts with a quotation mark, it must be a comment preceding the actual address */
  do
   parse var Addresses '"' Comment '"' Addresses  /* get the comment bit */
   Comment = strip(Comment,'B',"'")  /* get rid of any single quotes around the comment */
   Addresses = strip(Addresses,'L',' ')  /* remove any leading blanks from the addresses string */
  end

  GroupPos = pos(':;',Addresses)  /* the first occurrence of a group terminator */
  AttPos = pos('@',Addresses)  /* the first occurrence of a normal address indicator */

  if ((GroupPos > 0) & ((AttPos = 0) | (GroupPos < AttPos))) then  /* if we have a leading group */
  do
   parse var Addresses FirstBit ':;' Addresses  /* get the group bit */
   FirstBit = FirstBit||':;'  /* restore the terminator */
  end
  else  /* if we do not have a leading group */
  do
   parse var Addresses FirstBit '@' Addresses  /* look for the first bit of the actual address */
  end

  LastLeft = lastpos('<',FirstBit)  /* the position in the first address bit of the last left angle bracket, if any */

  if (LastLeft > 0) then  /* if we find one */
  do

   LastSpace = lastpos(' ',FirstBit)  /* the position of the last space in the first bit */

   if (LastLeft > (LastSpace + 1)) then  /* if the last < is more than one character past the last space, we have an @ddress with a comment stuck to it */
   do
    parse var FirstBit LeftBit =(LastLeft) +1 RightBit  /* get the two bits before and after the last < */
    FirstBit = LeftBit||' <'||RightBit  /* and stick them back together with a space in between */
   end

  end

  if ((Comment = '') & (words(FirstBit) > 1)) then  /* if we have no comment yet, and the first address bit contains more than one word, part of it must be a leading comment */
  do
   Comment = subword(FirstBit,1,words(FirstBit)-1)  /* use all but the last word of the first bit for the comment */
   Comment = strip(Comment,'T','"')  /* get rid of any trailing quotation mark to correct asymmetry */
   FirstBit = word(FirstBit,words(FirstBit))  /* use the last word for the first address bit */
  end

  LastBit = ''  /* start with nothing in the last bit of the address */
  PrevChar = ''  /* start with no previous character */
  GotLastBit = 0  /* we have no last bit of the address yet */

  do until (GotLastBit)  /* go on until we have what we need */

   if (Addresses = '') then  /* if we have no addresses string left */
   do
    GotLastBit = 1  /* this is all we get */
   end
   else  /* if we have addresses stuff left */
   do

    NextChar = left(Addresses,1)  /* get the next character from the remaining addresses string */

    if (pos(NextChar,', (') > 0) then  /* if it is a comma, this is where a new address starts; if it is a space, an end comment may follow, if it is a left parenthesis, we have an end comment */
    do
     GotLastBit = 1  /* we have all we need */
    end
    else  /* if the next character is not a comma, a space, or a left parenthesis */
    do

     LastBit = LastBit||NextChar  /* add it to what we have */
     Addresses = substr(Addresses,2)  /* remove the first character from the remaining addresses string */

     if ((NextChar = ';') & (PrevChar = ':')) then  /* if we have a complete group terminator */
     do
      GotLastBit = 1  /* we have all we need */
     end
     else  /* if we have something else */
     do
      PrevChar = NextChar  /* store the character */
     end

    end

   end

  end

  if (left(word(Addresses,1),1) = '(') then  /* if the first word of the remaining addresses string starts with a left parenthesis, we have an end comment */
  do
   parse var Addresses '(' EndComment ')' Addresses  /* extract the end comment */
  end

  Address = AddressCheck(FirstBit,LastBit)  /* see if the address components are O.K., i.e. they make up a new address */

  if (Address = '') then  /* if the address was zapped (because it was bad) */
  do

   CutPoint = length(OrgAddresses) - length(Addresses) + 1  /* where the next original address, if any, starts */
   parse var OrgAddresses Address =(CutPoint) OrgAddresses  /* get the original address */

   if (Check) then  /* if we are checking the address */
   do
    Address = Global.Warning||Address  /* add a marker */
   end

  end

  else  /* if all is well */

  do

   if (LastBit = '') then  /* if we have no last bit, i.e. it is a group */
   do

    if (pos(Spacer||Address,Spacer||NewAddresses) > 0) then  /* or if we already have the address in the list of new addresses */
    do
     Address = ''  /* zap it */
    end

   end

   else  /* if it is not a group */

   do

    if (pos('<'||Address||'>',NewAddresses) > 0) then  /* if we already have the address in the list of new addresses */
    do
     Address = ''  /* zap it */
    end

   end

   if (Address >< '') then  /* if we still have something, i.e. if it is a new address */
   do

    if (Indent >< '') then  /* if we have an indent value (even if it is 0), we want more than just the bare address */
    do

     if (EndComment >< '') then  /* if we have an end comment */
     do

      if (Comment >< '') then  /* if we have a front comment */
      do
       Comment = Comment||' - '  /* stick on a separator */
      end

      EndComment = translate(EndComment,' ','"')  /* remove any double quotation marks */
      Comment = Comment||EndComment  /* append the end comment to the front comment, if any */

     end

     if (Comment >< '') then  /* if we have a comment */
     do

      if (Comment >< Address) then  /* it's not the same as the address */
      do
       Comment = '"'||Comment||'" '  /* add double quotes and a space */
      end
      else  /* if it is the same as the address (some brain-dead mailers will do this) */
      do
       Comment = ''  /* use nothing */
      end

     end

     if (LastBit >< '') then  /* if we have a last bit, i.e. it is not a group */
     do
      Address = '<'||Address||'>'  /* add angle brackets to the bare address */
     end
    
     Address = Comment||Address  /* stick any comment we have on the front */

    end

   end

   else  /* if it is not a new address */

   do
    Address = ''  /* zap it */
   end

  end

  if (Address >< '') then  /* if we have something  */
  do

   if (First) then  /* if this was the first address */
   do
    First = 0  /* the next is no longer the first */
   end
   else  /* if it wasn't the first address */
   do
     Address = Spacer||Address  /* add a spacer */
   end

   NewAddresses = NewAddresses||Address  /* add the result to what we already have */

  end

 end

end

return NewAddresses  /* end of AddressFormat */

/**********************************************************************/
AddressImport: procedure expose Global.  /* imports address book entries */
/**********************************************************************/

ImpFile = Global.ProcFile  /* get the argument */

if (\FileCheck(ImpFile,1)) then  /* if the address text file does not exist */
do
 return 0  /* and quit */
end

if (\FileOpen(ImpFile,'READ')) then  /* if we cannot open the address file for writing */
do
 return 0  /* and quit */
end

call LogAction 'Importing address data from "'||ImpFile||'"'  /* report */

InBuffer = Global.CRLF||charin(ImpFile,1,chars(ImpFile))  /* read the contents of the address file into an input buffer and add a CRLF to make sure the first block marker is recognized */

call FileClose ImpFile  /* close the file */

Sep_1 = Global.CRLF||'<TITLE>'  /* define a separator */
Sep_2 = Global.CRLF||'<TEXT>'  /* define a separator */
Sep_3 = Global.CRLF||'<COMMENTS>'  /* define a separator */
Sep_4 = Global.CRLF||'<END>'  /* define a separator */

EntryNo = 0  /* start at 0 */

do while (InBuffer >< '')  /* go on while we have contents left */

 EntryNo = EntryNo + 1  /* up the entry number */
 parse var InBuffer (Sep_1) Block (Sep_4) InBuffer  /* get the next block */

 if (Block = '') then  /* if the block is empty */
 do

  if (InBuffer >< '') then  /* if it not because we have run out of contents */
  do
   call AddError 'Entry number '||EntryNo||' contains no data'  /* report */
  end

 end

 else  /* if the block is not empty */

 do
  Name = ''  /* start with nothing */
  Text = ''  /* start with nothing */
  Comments = ''  /* start with nothing */
  parse var Block Block (Sep_3) Comments  /* extract any comments */
  parse var Block Name (Sep_2) Text  /* extract any name and text contents */

  if (Name >< '') then  /* if we have a name bit */
  do

   do while (left(Name,2) = Global.CRLF)  /* as long as the name starts with a CRLF */
    Name = substr(Name,3)  /* get rid of it */
   end

   do while (right(Name,2) = Global.CRLF)  /* as long as the name ends with a CRLF */
    Name = substr(Name,1,length(Name) - 2)  /* get rid of it */
   end

  end

  if (Name = '') then  /* if we have no name */
  do
   call AddError 'Entry number '||EntryNo||' contains no name'  /* report, not fatal */
  end

  else  /* if we have a name */

  do

   if (Text >< '') then  /* if we have a text bit */
   do

    do while (left(Text,2) = Global.CRLF)  /* as long as the text starts with a CRLF */
     Text = substr(Text,3)  /* get rid of it */
    end

    do while (right(Text,2) = Global.CRLF)  /* as long as the text ends with a CRLF */
     Text = substr(Text,1,length(Text) - 2)  /* get rid of it */
    end

   end

   if (Text = '') then  /* if we have no text */
   do
    call AddError 'Entry number '||EntryNo||' contains no text'  /* report */
   end

   Path = strip(filespec('D',Name)||filespec('P',Name),'T','\')  /* get the path */
   Target = Global.AddrDir  /* we start here */

   do while (Path >< '')  /* as long as the path continues */
    parse var Path Folder '\' Path  /* get the first part of the path */
    Settings = 'ALWAYSSORT=YES;ICONVIEW=FLOWED,MINI;WORKAREA=YES;BACKGROUND='||Global.IconDir||'\FolderBG.BMP'  /* the settings to use for the folder object */
    call syscreateobject 'WPFolder',Folder,Target,Settings,'FAIL'  /* try to create the folder, but fail if it already exists */
    Target = Target||'\'||Folder  /* move up the path for any next bit */
   end

   NewName = ''  /* start with nothing */

   do while (Name >< '')  /* as long as we have bits of name left */
    parse var Name NextBit (Global.CRLF) Name  /* get the bit up to the next CRLF */
    NewName = NewName||NextBit||' '  /* store the next bit */
   end

   OutFile = Global.AddrDir||'\'||NewName  /* the name of the output file */
   OutFile = translate(OutFile,'  ___________'||"''",'<>[]~?%&*|/;,"'||"`")  /* replace any awkward characters (do NOT use CheckCommLine!) */
   PathPart = strip(filespec('D',OutFile)||filespec('P',OutFile),'T','\')  /* the path part */
   NamePart = filespec('N',OutFile)  /* the name part */
   call sysdestroyobject OutFile  /* remove any existing instance of this object */
   call syscreateobject 'CWMailFile',NamePart,PathPart,,'FAIL'  /* create a CWMAILFile class object in the address dir using the name part */
   call HideObject OutFile  /* make the file invisible */

   if (FileOpen(OutFile,'WRITE')) then  /* if we can open the output file */
   do

    call charout OutFile,Text  /* write any text content to the file */
    call FileClose OutFile  /* close the file */
    call MessageSettings OutFile,'00000000','CHANGE'  /* make it a fresh outgoing mail message file */

    if (Comments >< '') then  /* if we have comments content */
    do

     do while (left(Comments,2) = Global.CRLF)  /* as long as the comments start with a CRLF */
      Comments = substr(Comments,3)  /* get rid of it */
     end

     do while (right(Comments,2) = Global.CRLF)  /* as long as the comments end with a CRLF */
      Comments = substr(Comments,1,length(Comments) - 2)  /* get rid of it */
     end

     if (Comments >< '') then  /* if we still have comments */
     do
      call CommentsPut OutFile,Comments  /* add the comments */
     end

    end

    call syssetobjectdata OutFile,'REXXMAILREFRESH=YES;TEMPLATE=YES;'  /* make the file a template etc. */
    call UnhideObject OutFile  /* make the file visible */

   end

  end

 end

end

return 1  /* end of AddressImport */

/**********************************************************************/
AttDirClose: procedure expose Global.  /* closes an attachments dir and returns its name if it contains attachments, otherwise deletes it and returns an empty string */
/**********************************************************************/

parse arg MessFile  /* get the arguments */

AttDir = AttDirGet(MessFile,0)  /* get the attachments dir, if any (do not create a new one) */

if (AttDir = '') then  /* if there is none */
do
 return ''  /* return with nothing */
end

call sysfiletree AttDir||'\*','Objects.','BO'  /* is there anything in the attachments folder? */

if (Objects.0 = 0) then  /* if we have no contents */
do
 call AttDirUnlink MessFile  /* get rid of the attachments folder EA */
 call sysdestroyobject AttDir  /* get rid of the attachments folder itself (i.e. "closing" it if open on the desktop) */
 return ''  /* return with nothing */
end

if (pos('4OS2.EXE',value('COMSPEC',,'OS2ENVIRONMENT')) > 0) then  /* if we are running 4OS2 */
do
 signal off Error  /* error signalling off */
 address cmd 'activate "'||filespec('N',AttDir)||'" close'  /* if 4OS2 is the default command processor, this should close the attachments folder */
 signal on Error  /* error signalling on */
 return AttDir  /* return the attachments dir name */
end

NewAttDir = AttDirCreate(MessFile)  /* get a new attachments folder name and link it to the message file instead of the old attachments folder */

do Index = 1 to Objects.0  /* take each of the objects in the old attachments folder */

 if (\sysmoveobject(Objects.Index,NewAttDir)) then  /* if we cannot move it to the new folder */
 do

  call AttDirLink MessFile,AttDir  /* restore the old attachments folder link */
  call AddError 'Cannot move "'||Objects.Index||'" to "'||NewAttDir||'"'  /* report */
  call sysfiletree NewAttDir||'\*','Objects.','BO'  /* look for any objects that have already been moved to the new folder */
  Restored = 1  /* assume complete success in putting things back */

  do Index = 1 to Objects.0  /* take each moved object we find */

   if (\sysmoveobject(Objects.Index,AttDir)) then  /* if we cannot move it back to the original attachments folder */
   do
    call AddError 'Cannot move "'||Objects.Index||'" back to "'||AttDir||'"'  /* report */
    Restored = 0  /* no success */
   end

  end

  if (Restored) then  /* if all files were restored */
  do
   call sysdestroyobject NewAttDir  /* get rid of the now empty new folder */
  end
  else  /* if not */
  do
   call sysopenobject NewAttDir,0,1  /* open the new folder on the desktop so the user can restore things manually */
  end

  return AttDir  /* return with the old attachments name */

 end

end

call sysdestroyobject AttDir  /* get rid of the original attachments folder */

return NewAttDir  /* end of AttDirClose */

/**********************************************************************/
AttDirCreate: procedure expose Global.  /* create an attachment dir and EA */
/**********************************************************************/

parse arg MessFile  /* get the argument, if any */

AttDir = TempFileName()  /* get a unique directory name in the temp files directory */
AttDirName = filespec('N',AttDir)  /* the name bit */

if (\syscreateobject('WPFolder',AttDirName,Global.TempDir,'NODELETE=YES;NODRAG=YES;NOMOVE=YES;NORENAME=YES;NOSETTINGS=YES;BACKGROUND='||Global.IconDir||'\AttachBG.BMP,T,,I;WORKAREA=YES;ALWAYSSORT=YES;','FAIL')) then  /* if we cannot create an attachments folder in the temp dir */
do
 call LogAction 'Cannot create attachments folder'  /* report */
 return ''  /* return with nothing */
end

if (MessFile >< '') then  /* if we have a file spec */
do

 if (\AttDirLink(MessFile,AttDir)) then  /* if we cannot link the attachments dir to the message file */
 do
  return ''  /* return with nothing */
 end

end

return AttDir  /* end of AttDirCreate */

/**********************************************************************/
AttDirGet: procedure expose Global.  /* get an attachment directory name, optionally creating the dir */
/**********************************************************************/

parse arg MessFile,Create  /* get the arguments */

Create = (Create = 1)  /* 1 = true */
AttDir = ''  /* start with zilch */

if (sysgetea(MessFile,'RXMLATTDIR','AttDirEA') >< 0) then  /* if we cannot get the attachments dir EA */
do
 call AddError 'Cannot retrieve attachments folder setting'  /* report */
 return ''  /* and quit */
end

if (AttDirEA >< '') then  /* if we found something */
do

 parse var AttDirEA EAType 3 . 5 AttDir  /* get the bits we want */

 if (EAType >< 'FDFF'x) then  /* if it is not ASCII text */
 do
  AttDir = ''  /* we have no attachments directory either */
 end

end

if (AttDir >< '') then  /* if we have something */
do

 call sysfiletree AttDir,'Dirs.','DO'  /* look for the actual directory */

 if (Dirs.0 = 0) then  /* if no such directory was found (i.e. it was deleted) */
 do
  AttDir = ''  /* we have nothing */
 end

end

if ((Create) & (AttDir = '')) then  /* if we still have nothing, and we want one created */
do
 AttDir = AttDirCreate(MessFile)  /* create a new one and link it to the message file */
end


return AttDir  /* end of AttDirGet */

/**********************************************************************/
AttDirLink: procedure expose Global.  /* link an attachment dir to a message file */
/**********************************************************************/

parse arg MessFile,AttDir  /* get the arguments */

EALength = reverse(right(d2c(length(AttDir)),2,'00'x))  /* set the EA length */
AttDirEA = 'FDFF'x||EALength||AttDir  /* prepare the attachments dir EA */

if (sysputea(MessFile,'RXMLATTDIR',AttDirEA) >< 0) then  /* if we cannot set the EA */
do
 call AddError 'Cannot link attachments folder to message file'  /* report */
 return 0  /* return with an error */
end

if (sysputea(MessFile,'.SUBJECT','') >< 0) then  /* if we cannot reset the old EA */
do
 call AddError 'Cannot reset old attachments folder link'  /* report */
 return 0  /* return with an error */
end

return 1  /* end of AttDirLink */

/**********************************************************************/
AttDirOpen: procedure expose Global.  /* opens an attachments dir on the desktop */
/**********************************************************************/

parse arg Messfile  /* get the argument */

if ((MessageSettings(MessFile,'1*******','MATCH')) | (MessageSettings(MessFile,'0*1*****','MATCH'))) then  /* if it is an incoming or sent message */
do
 AttDir = AttDirGet(MessFile,0)  /* get any existing attachments dir (i.e. only if the message is open on the desktop) */
end
else  /* if it is an outgoing message */
do
 AttDir = AttDirGet(MessFile,1)  /* get the attachments dir and create it if necessary */
end

if (AttDir >< '') then  /* if we have something to show */
do

 if (\sysopenobject(AttDir,0,1)) then  /* if we cannot open the attachments folder */
 do
  call AddError 'Cannot open attachments folder'  /* report */
 end

end

return AttDir  /* end of AttDirOpen */

/**********************************************************************/
AttDirShow: procedure expose Global.  /* shows an attachments dir on the desktop */
/**********************************************************************/

return (AttDirOpen(Global.ProcFile))  /* end of AttDirShow */

/**********************************************************************/
AttDirShut: procedure expose Global.  /* shuts an attachments dir for the main routine */
/**********************************************************************/

return (AttDirClose(Global.ProcFile) >< '')  /* end of AttDirShut */

/**********************************************************************/
AttDirUnlink: procedure expose Global.  /* delete the attachments directory EA */
/**********************************************************************/

parse arg MessFile  /* get the argument */

if (sysputea(MessFile,'RXMLATTDIR','') >< 0) then  /* if we cannot reset the EA */
do
 call AddError 'Cannot unlink attachments dir'  /* report */
 return 0  /* and quit with no succes */
end

return 1  /* end of AttDirUnlink */

/**********************************************************************/
CheckCommLine: procedure  /* replaces awkward characters in a string for command-line use and removes double spaces */
/**********************************************************************/

parse arg String  /* get the argument */

String = translate(String,' ()()__________..'||"''",d2c(9)||'<>[]~^?%&*=\|/;,"'||"`")  /* remove difficult characters from the string */

NewString = ''  /* start with nothing */

do while (words(String) > 0)  /* as long as we have words left */
 parse var String NextWord String  /* get the next word */
 NewString = NewString||' '||NextWord  /* and add to what we've got */
end

NewString = strip(NewString,'L',' ')  /* get rid of the leading blank */

return NewString  /* end of CheckCommLine */

/**************************************************************************/
CheckHeader: procedure expose Global.  /* checks for specific header content */
/**************************************************************************/

parse upper arg CheckBit,HeaderBit  /* get the arguments, in upper case */

do while (CheckBit >< '')  /* go on until we run out of steam */

 parse var CheckBit NextBit ';' CheckBit  /* get the next bit to check */
 parse var NextBit KeyWord ':' Value  /* get the components */
 KeyWord = strip(KeyWord)  /* get rid of excess whitespace */
 Value = strip(Value)  /* get rid of excess whitespace */

 if (Keyword >< '') then  /* if we have a keyword */
 do

  Entry = GetHeaderEntry(HeaderBit,KeyWord||':')  /* look for a matching entry */

  if (Entry >< '') then  /* if we have an entry */
  do

   if (Value = '') then  /* if we have no value */
   do
    return 1  /* return TRUE */
   end

   if (pos(Value,Entry) > 0) then  /* if we find a match */
   do
    return 1  /* return TRUE */
   end

  end

 end

end

return 0  /* end of CheckHeader */

/**********************************************************************/
CheckValBool: procedure expose Global.  /* checks the value of a Boolean argument */
/**********************************************************************/

parse arg Name,Value  /* get the arguments */

SetTrue = 'TRUE T YES Y ON 1 +'  /* these all count as true */
SetFalse = 'FALSE F NO N OFF 0 -'  /* these all count as false */

Value = translate(Value)  /* make the value ipper case */

if (wordpos(Value,SetTrue) > 0) then  /* if the setting is TRUE */
do
 return 1  /* return this */
end

if (wordpos(Value,SetFalse) > 0) then  /* if the setting is FALSE */
do
 return 0  /* return this */
end

call AddError 'The "'||Name||'" setting requires a TRUE or FALSE argument ('||SetTrue||' or '||SetFalse||')'  /* report */

return 0  /* end of CheckValBool */

/**********************************************************************/
CheckValNum: procedure expose Global.  /* checks the value of a parameter and sets a default value if necessary */
/**********************************************************************/

parse arg Name,Value,Default,Low,High  /* get the arguments */

if (Value = '') then  /* if the value is empty */
do
 return Default  /* return the default value */
end

if (\datatype(Value,'W')) then  /* if the value is not a whole number */
do
 call AddError 'The "'||Name||'" setting requires a whole number instead of '||Value  /* report */
 return Default  /* return the default value */
end

if ((Low >< '') & (Value < Low)) then  /* if it is out of the low range */
do
 call AddError 'The "'||Name||'" setting '||Value||' is out of range (< '||Low||')'  /* report */
 return Low  /* return the low  value */
end

if ((High >< '') & (Value > High)) then  /* if it is out of the high range */
do
 call AddError 'The "'||Name||'" setting '||Value||' is out of range (> '||High||')'  /* report */
 return High  /* return the high value */
end

return Value  /* end of CheckValNum */

/**********************************************************************/
CheckValSignal: procedure expose Global.  /* check the value of a beep sequence signal string */
/**********************************************************************/

parse arg Name,SignalString  /* get the argument */

if (SignalString = '') then  /* if there is no signal string (which is O.K.) */
do
 return ''  /* return with nothing */
end

ErrorString = ''  /* start with nothing */

if (translate(word(SignalString,1)) = 'BEEP') then  /* if it looks like a beep sequence, check it (if not, it must be an external command) */
do

 SignalString = translate(SignalString)  /* make the signal string upper case for use by the "SoundSignal" procedure */
 BeepSequence = subword(SignalString,2)  /* get the beep sequence */

 do while (BeepSequence >< '')  /* go on until we run out of steam */

  parse var BeepSequence Frequency ',' Duration BeepSequence  /* get the components */

  if (datatype(Frequency,'W')) then  /* if it is a whole number */
  do

   if ((Frequency < 37) | (Frequency > 32767)) then  /* if it is out of range */
   do
    ErrorString = ErrorString||Global.CRLF||'  Frequency out of range (37-32767): '||Frequency  /* add this */
    SignalString = ''  /* we have an error */
   end

  end
  else  /* if it is not a whole number */
  do
   ErrorString = ErrorString||Global.CRLF||'  Not a whole number: '||Frequency  /* add this */
   SignalString = ''  /* we have an error */
  end
   
  if (Duration >< '') then  /* if we ahev a duration value */
  do

   if (datatype(Duration,'W')) then  /* if this is a whole number */
   do
    
    if ((Duration < 1) | (Duration > 60000)) then  /* if it is out of range */
    do
     ErrorString = ErrorString||Global.CRLF||'  Duration out of range (1-60000): '||Duration  /* add this */
     SignalString = ''  /* we have an error */
    end

   end
   else  /* if it is not a whole number */
   do
    ErrorString = ErrorString||Global.CRLF||'  Not a whole number: '||Duration  /* add this */
    SignalString = ''  /* we have an error */
   end
   
  end
  else  /* if we have no duration value */
  do
   ErrorString = ErrorString||Global.CRLF||'  Missing duration value'  /* add this */
   SignalString = ''  /* we have an error */
  end

 end
 
end

if (ErrorString >< '') then  /* if we have one or more errors */
do
 call AddError 'Invalid "'||Name||'" setting: '||ErrorString  /* report */
end

return SignalString  /* end of CheckValSignal */

/**********************************************************************/
CheckValText: procedure expose Global.  /* checks the value of a text parameter */
/**********************************************************************/

parse arg Name,Value,OKValues,Single,Fuzzy  /* get the arguments */

if (Value = '') then  /* if the value is empty */
do
 return ''  /* return it */
end

Single = (Single = 1)  /* 1 = True */
Fuzzy = (Fuzzy = 1)  /* 1 = True */

if (Fuzzy) then  /* if additional formatting characters are allowed */
do
 CheckValue = translate(Value,' ',Global.ExtraCharacters)  /* turn extra formatting characters into blanks */
end
else  /* if no additional characters are allowed */
do
 CheckValue = Value  /* use the original */
end

if ((Single) & (words(CheckValue) > 1)) then  /* if the value should be single, and contains more than one word */
do
 call AddError 'The "'||Name||'" setting requires a single value instead of "'||Value||'"'  /* report */
 return ''  /* return with nothing */
end

if (OKValues >< '') then  /* if we have a list of OK values */
do

 do while (CheckValue >< '')  /* go on while we have stuff */

  parse var CheckValue CheckWord CheckValue  /* get the next word */

  if (wordpos(translate(CheckWord),translate(OKValues)) = 0) then  /* if the word is not in the OK values */
  do
   call AddError 'Invalid "'||Name||'" setting: "'||CheckWord||'"'  /* report */
   return ''  /* return with nothing */
  end

 end

end

return Value  /* end of CheckValText */

/**********************************************************************/
CommentsGet: procedure expose Global.  /* gets comments from an address book entry */
/**********************************************************************/

parse arg FileName  /* get the argument */

OutLines = ''  /* nothing yet */

call sysgetea FileName,'.COMMENTS','Comments'  /* look for key phrases */

if (Comments >< '') then  /* if we found comments */
do

 parse var Comments EAType 3 CodePage 5 Count 7 Contents  /* get the bits we want */

 if (EAType = 'DFFF'x) then  /* if it is a multiple value, multiple type entry */
 do

  Count = c2d(reverse(Count))  /* extract the number of comment lines */

  do Index = 1 to Count  /* take each line entry */

   parse var Contents EAType 3 EALength 5 Contents  /* get the bits we want */

   if (EAType = 'FDFF'x) then  /* it it is ASCII text (which it should be) */
   do
    EALength = c2d(reverse(EALength))  /* get the length of the next line */
    parse var Contents Text +(EALength) Contents  /* get the bits we want */
    OutLines = OutLines||Text||Global.CRLF  /* add the text and a CRLF */
   end

  end

 end

end

if (OutLines >< '') then  /* if we found something */
do
 OutLines = left(OutLines,length(OutLines) - 2)  /* get rid of the excess CRLF at the end */
end

return OutLines  /* end of CommentsGet */

/**********************************************************************/
CommentsPut: procedure expose Global.  /* adds comments to an address book entry */
/**********************************************************************/

parse arg FileName,Comments  /* get the arguments */

Contents = ''  /* start with nothing */
Count = 0  /* start at 0 */

do while (length(Comments) > 0)  /* run through the comments */
 parse var Comments NextLine (Global.CRLF) Comments  /* get the next line of the comments */
 EALength = reverse(right(d2c(Length(NextLine)),2,'00'x))  /* set the EA length */
 Contents = Contents||'FDFF'x||EALength||NextLine  /* add another ASCII string to the EA contents */
 Count = Count + 1  /* up the counter */
end

Count = reverse(right(d2c((Count)),2,'00'x))  /* set the entries count */
Comments = 'DFFF'x||'0000'x||Count||Contents  /* prepare the comments EA */

if (sysputea(FileName,'.COMMENTS',Comments) >< 0) then  /* if we cannot attach them to the file */
do
 call AddError 'Cannot attach comments to "'||FileName||'"'  /* report */
 return 0  /* return an error */
end

return 1  /* end of CommentsPut */

/**********************************************************************/
CopyAttachment: procedure expose Global.  /* attach file to RexxMail message(s) */
/**********************************************************************/

parse upper arg Switch  /* get the argument, in upper case */

if (Switch = '') then  /* if we have no switch */
do
 return 0  /* quit */
end

FileName = Global.ProcFile  /* the file to process */

select  /* do one of the following */

 when (left(Switch,12) = 'ATTACHTOOPEN') then  /* if we want to use all the open messages on the desktop */
 do

  if (sysqueryswitchlist('ListItems.') = 0) then  /* if we can get a switch list */
  do Index = 1 to ListItems.0  /* take each of the items found  */

   EditFile = filespec('N',ListItems.Index)  /* try to extract a file name from the list item */

   if ((left(EditFile,4) = 'RXML') & (right(EditFile,5) = '.EDIT') & (datatype(substr(EditFile,5,4),'W'))) then  /* if we find a RexxMail edit file name */
   do
    call MakeAttCopy FileName,Global.TempDir||'\'||left(EditFile,8)  /* attempt the actual copy */
   end

  end

 end

 when (Switch = 'ATTACHTOALLINOUT') then  /* if we want to use the attachment folder of all messages in the Out folder */
 do

  call sysfiletree Global.Outdir||'\*','Files.','FO'  /* look for RexxMail files in the 'Out' folder */

  do Index = 1 to Files.0  /* take each of the files found */

   if (MessageSettings(Files.Index,,'CHECK')) then  /* if the file is a RexxMail message */
   do

    if (MessageSettings(Files.Index,'0*******','MATCH')) then  /* if the message is outgoing */
    do

     AttDir = AttDirGet(Files.Index,1)  /* get an attachments dir, creating it if necessary */

     if (AttDir >< '') then  /* if we got one */
     do
      call MakeAttCopy FileName,AttDir,Files.Index  /* attempt the actual copy */
     end

    end

   end

  end

 end

 otherwise  /* if none of the above */
 do
  call AddError 'Invalid switch value: "'||Switch||'"'  /* report */
  return 0  /* return with no success */
 end

end

return 1  /* end of CopyAttachment */

/**********************************************************************/
MakeAttCopy: procedure expose Global.  /* do the actual copying etc. */
/**********************************************************************/

parse arg FileName,AttDir,MessFile  /* get the arguments */

if (syscopyobject(FileName,AttDir)) then  /* if we can copy the file to the attachments folder */
do

 if (MessFile >< '') then  /* if we have a message file spec */
 do
  call MessageSettings MessFile,'****1***','CHANGE'  /* change the message settings to show an attachment */
  call syssetobjectdata MessFile,'REXXMAILATTACHMENT=Yes'  /* set the attachment indicator for the CWMailFile class */
  call LogAction 'File "'||FileName||'" attached to "'||MessFile||'"',1  /* report, quietly */
 end
 else  /* if we do not have a message file spec */
 do
  call LogAction 'File "'||FileName||'" copied to attachments folder "'||AttDir||'"',1  /* report, quietly */
 end

end
else  /* if we cannot copy the file to it */
do

 if (MessFile >< '') then  /* if we have a message file spec */
 do
  call AddError 'Cannot attach file "'||FileName||'" to message file; attachment may already exist'  /* report an error */
 end
 else  /* if we do not have a message file spec */
 do
  call AddError 'Cannot copy file "'||FileName||'" to attachments folder; attachment may already exist'  /* report an error */
 end

end

return  /* end of MakeAttCopy */

/**********************************************************************/
DateTimeDisplay: procedure expose Global.  /* rewrites a date/time stamp string if necessary */
/**********************************************************************/

parse arg OrgStamp,TimeTypes  /* get the arguments */

if (OrgStamp = '') then  /* if we have no date/time stamp */
do
 return ''  /* return nothing */
end

DayNames = 'Mon Tue Wed Thu Fri Sat Sun'  /* the names of the weekdays */
PrevDayName = ''  /* we have no previous day name yet */
MonthNames = 'Jan Feb Mar Apr May Jun Jul Aug Sep Oct Nov Dec'  /* the names of the months */

parse var OrgStamp DayName Rest  /* get the day name */

if (right(DayName,1) = ',') then  /* if a day name reference was included, it should be followed by a comma */
do

 DayName = left(DayName,3)  /* get the first 3 characters of the day name */
 DayPos = wordpos(translate(DayName),translate(DayNames))  /* look for a semblance of the day name in the day names */

 if (DayPos > 0) then  /* if we find the day name */
 do
  DayName = word(DayNames,DayPos)  /* get the valid name */
 end
 else  /* if we do not find the day name */
 do
  DayName = ''  /* use nothing */
 end

 parse var Rest Day Rest  /* get the numerical day of the month */

end
else  /* if there was no day name reference */
do
 Day = DayName  /* the first bit was the numerical day of the month */
 DayName = ''  /* we have no day name */
end

parse var Rest MonthName Year Time OffsetStr  /* extract the other bits we need */

MonthName = left(MonthName,3)  /* get the first 3 characters of the month name */
Month = wordpos(translate(MonthName),translate(MonthNames))  /* look for a semblance of the month name in the month names */

if (Month = 0) then  /* if we do not find the month name */
do
 return OrgStamp  /* simply return the orignal */
end

if (\datatype(Year||Day,'W')) then  /* if the year or the day is not a whole number, the d/t stamp is a dud */
do
 return OrgStamp  /* simply return the orignal */
end

if (Year < 1000) then  /* if the year has less than 4 digits */
do

  if (Year < 50) then  /* if it's less than 50 */
  do
   Year = Year + 2000  /* add 2000 */
  end
  else  /* if it's 50 or over */
  do
   Year = Year + 1900  /* add 1900 */
  end

end

DateTimeStamp = ''  /* start with nothing */

do while (TimeTypes >< '')  /* go on while we have types left */

 if (DateTimeStamp >< '') then  /* if we already have something */
 do
  DateTimeStamp = DateTimeStamp||' '  /* add a space */
 end

 NewYear = Year  /* we have no new year value yet */
 NewMonth = Month  /* we have no new month value yet */
 NewDay = Day  /* we have no new day value yet */
 NewDayName = DayName  /* we have no new day name value yet */
 NewTime = Time  /* we have no new time value yet */

 parse var TimeTypes TimeType TimeTypes  /* get the next one */

 do while (pos(left(TimeType,1),Global.ExtraCharacters) > 0)  /* as long as the first character is an extra */
  parse var TimeType AddChar +1 TimeType  /* separate the first character */
  DateTimeStamp = DateTimeStamp||AddChar  /* and add it to the output string */
 end

 if (TimeType >< '') then  /* if we have something left */
 do

  AddDateTimeStamp = ''  /* start with nothing */

  do while (pos(right(TimeType,1),Global.ExtraCharacters) > 0)  /* as long as the last character is an extra */
   CutPoint = length(TimeType) - 1  /* where to cut */
   parse var TimeType TimeType +(CutPoint) AddChar  /* get the last character */
   AddDateTimeStamp = AddDateTimeStamp||AddChar  /* and add it to the additional output string */
  end

  ISODate = (right(TimeType,3) = 'ISO')  /* see if we want an ISO sorted date */
  TimeType = left(TimeType,1)  /* use just the initial of the time type */

  if (TimeType >< 'O') then  /* if we want UTC or system time */
  do

   if (OffsetStr >< '') then  /* if we have a time offset and perhaps more */
   do

    parse var OffsetStr Offset Rest  /* get what should be the time offset */

    if ((length(Offset) >< 5) | (pos(left(Offset,1),'+-') = 0) | (\datatype(right(Offset,4),'W'))) then  /* if the offset indicator is not of the [+/-]nnnn type */
    do

     OffsetN = '  GMT    UT   UTC   EDT   EST   CDT   CST   MDT   MST   PDT   PST'  /* non-numerical time offset indicators */
     Offsets = '+0000 +0000 +0000 -0400 -0500 -0500 -0600 -0600 -0700 -0700 -0800'  /* the equivalent offsets */
     OffsetPos = wordpos(Offset,OffsetN)  /* look for the offset string in our array */

     if (OffsetPos > 0) then  /* if we find it */
     do
      Offset = word(Offsets,OffsetPos)  /* convert the alpha offset to a numerical offset */
     end
     else  /* if we do not find it */
     do
      Offset = '-0000'  /* use this */
     end

    end

   end
   else  /* if we have no time offset indicator at all */
   do
    Offset = '-0000'  /* use this */
   end

   if (\((Offset = 0) & (left(Offset,1) = '-'))) then  /* if we have a valid offset */
   do

    if (TimeType = 'S') then  /* if we want local system time */
    do

     Offset = Offset + GetTimeZone(0)  /* get the local system time zone, contacting a time server if necessary, and add it to the message offset */

     if (Offset < 0) then  /* if the offset is negative */
     do
      Offset = '-'||right(substr(Offset,2),4,'0')  /* use this */
     end
     else  /* if the offset is zero or positive */
     do
      Offset = '+'||right(Offset,4,'0')  /* use this */
     end

    end

    if (Offset >< 0) then  /* if we have an actual offset */
    do

     MonthDays = '31 28 31 30 31 30 31 31 30 31 30 31'  /* the no. of days in each month */
     parse var Time Hour ':' Minute ':' Second  /* get the time stamp components */
     parse var Offset OffsetSign +1 OffsetHour +2 OffsetMinute .  /* get the offset components */

     if (OffsetSign = '-') then  /* if the offset is negative */
     do

      Hour = Hour + OffsetHour  /* add the offset hours */
      Minute = Minute + OffsetMinute  /* add the offset minutes */

      do while (Minute > 59)  /* as long as we run over the minute limit */
       Minute = Minute - 60  /* rewrite the minute */
       Hour = Hour + 1  /* add an hour */
      end

      do while (Hour > 23)  /* as long as we run over the hour limit */

       Hour = Hour - 24  /* rewrite the hour */
       NewDay = NewDay + 1  /* add a day */

       if (NewDayName >< '') then  /* if we have a day name */
       do
        NewDayName = word(DayNames||' Mon',wordpos(NewDayName,DayNames) + 1)  /* get the new day name */
       end

       MonthLen = word(MonthDays,NewMonth)  /* the no. of days in the date stamp month */

       if ((NewMonth = 2) & (LeapYear(NewYear))) then  /* if it is Feb in a leap year */
       do
        MonthLen = MonthLen + 1  /* add a day */
       end

       if (NewDay > MonthLen) then  /* if we run over the day limit */
       do

        if (NewMonth = 12) then  /* if it is in December */
        do
         NewMonth = 1  /* skip to the first month */
         NewYear = NewYear + 1  /* move on a year */
        end
        else  /* if it is not in Dec */
        do
         NewMonth = NewMonth + 1  /* just move up a month */
        end

        NewDay = 1  /* reset the day to 1 */

       end

      end

     end

     else  /* if the offset is positive */

     do

      Hour = Hour - OffsetHour  /* subtract the offset hours */
      Minute = Minute - OffsetMinute  /* subtract the offset minutes */

      do while (Minute < 0)  /* as long as we run below the minute limit */
       Minute = 60 + Minute  /* rewrite the minute */
       Hour = Hour - 1  /* subtract an hour */
      end

      do while (Hour < 0)  /* as long as we run below the hour limit */

       Hour = 24 + Hour  /* rewrite the hour */
       NewDay = NewDay - 1  /* subtract a day */

       if (NewDayName >< '') then  /* if we have a day name */
       do
        NewDayName = word('Sun '||DayNames,wordpos(NewDayName,DayNames))  /* get the new day name this way */
       end

       if (NewDay < 1) then  /* if we run below the day limit */
       do

        if (NewMonth = 1) then  /* if the month is Jan */
        do
         NewMonth = 12  /* skip to Dec */
         NewYear = NewYear - 1  /* go back a year */
        end
        else  /* if it is not Jan */
        do
         NewMonth = NewMonth - 1  /* just go back a month */
        end

        NewDay = word(MonthDays,NewMonth)  /* the no. of days in the new date stamp month */

        if ((NewMonth = 2) & (LeapYear(NewYear))) then  /* if it is Feb in a leap year */
        do
         NewDay = NewDay + 1  /* add a day */
        end

       end

      end

     end

     NewTime = right(Hour,2,'0')||':'||right(Minute,2,'0')||':'||right(Second,2,'0')  /* the new time stamp */

    end

   end

  end

  if (ISODate) then  /* if we want an ISO date */
  do
   NewStamp = NewYear||'-'||right(NewMonth,2,'0')||'-'||right(NewDay,2,'0')||' '||NewTime  /* this is our new d/t stamp */
  end
  else  /* if we want an RFC date */
  do
   NewStamp = NewDay||' '||word(MonthNames,NewMonth)||' '||NewYear||' '||NewTime  /* this is our new d/t stamp */
  end

  select  /* do one of the following */

   when (TimeType = 'O') then  /* if we want the original time */
   do

    if (OffsetStr >< '') then  /* if we have an offset (and perhaps more) */
    do
     NewStamp = NewStamp||' '||OffsetStr  /* add it as it is */
    end

   end

   when (TimeType = 'U') then  /* if we want UTC time */
   do

    if ((Offset = 0) & (left(Offset,1)= '-')) then  /* if we have an unknown offset */
    do
     NewStamp = NewStamp||' (?)'  /* add this */
    end
    else  /* if we have a known offset */
    do
     NewStamp = NewStamp||' UTC'  /* add this */
    end

   end

   otherwise  /* if none of the above */
   do
    nop  /* nothing */
   end

  end

  if (NewDayName >< PrevDayName) then  /* if we have a new day name that is not the same as the previous day name (always, the first time round) */
  do

   if (ISODate) then  /* if we want an ISO date */
   do
    NewStamp = NewStamp||' ('||NewDayName||')'  /* add it */
   end
   else  /* if we want an RFC date */
   do
    NewStamp = NewDayName||', '||NewStamp  /* add it */
   end

   PrevDayName = NewDayName  /* save the value for later use */

  end

  DateTimeStamp = DateTimeStamp||NewStamp||AddDateTimeStamp  /* add the results to what we have */

 end

end

return DateTimeStamp  /* end of DateTimeDisplay */

/**********************************************************************/
DateTimeRFC: procedure expose Global.  /* returns message date string */
/**********************************************************************/

parse arg NoZone  /* get the argument */

Global.TimeZone = GetTimeZone(NoZone)  /* get and set the local time zone if necessary */
DateString = left(date('W'),3)||', '||date('N')||' '||time('N')||' '||Global.TimeZone  /* prepare the message date string */

return DateString  /* end of DateTimeRFC */

/**********************************************************************/
DateTimeSys: procedure expose Global.  /* returns a date/time stamp string */
/**********************************************************************/

parse arg LongTime  /* get the argument */

LongTime = (LongTime = 1)  /* 1 is true */

if (LongTime) then  /* if we want a long time */
do
 TimeType = 'L'  /* use this */
end
else  /* if not */
do
 TimeType = 'N'  /* use this */
end

DateStr = date('S')  /* get the sorted date */

Stamp = left(DateStr,4)||'-'||,  /* start the date & time stamp with the year in 4 digits */
        substr(DateStr,5,2)||'-'||,  /* followed by the month in 2 digits */
        substr(DateStr,7,2)||' '||,  /* followed by the day of the month in 2 digits */
        time(TimeType)  /* followed by the normal or long time */

return Stamp  /* end of DateTimeSys */

/**********************************************************************/
DecodeB64: procedure expose Global.  /* decodes a B64 string */
/**********************************************************************/

parse arg B64Str  /* get the argument */

B64Chars = xrange('A','Z')||xrange('a','z')||xrange('0','9')||'+/'  /* define the base64 character set */
BitStr = ''  /* start with nothing */

do while (B64Str >< '')  /* as long as we have something left */

 parse var B64Str NextChar 2 B64Str  /* get the next character */

 if (NextChar >< '=') then  /* if it it is not padding */
 do
  CharVal = pos(NextChar,B64Chars) - 1  /* the value represented by the B64 character */
  CharBits = right(x2b(d2x(CharVal)),6,'0')  /* convert the value to 6 bits */
  BitStr = BitStr||CharBits  /* add the result to the bit string */
 end

end

TextStr = ''  /* start with nothing */

do while(length(BitStr) > 7)  /* as long as we have at least a whole byte left */
 parse var BitStr NextChar 9 BitStr  /* get the first 8 bits */
 TextStr = TextStr||x2c(b2x(NextChar))  /* convert to a character and add to the result */
end

return TextStr  /* end of DecodeB64 */

/**********************************************************************/
DecodeHeader: procedure expose Global.  /* decodes header lines if neccessary (and if possible) */
/**********************************************************************/

parse arg Text  /* get the argument */

CodeStart = '=?'  /* define the start of an encoded string */
CodeEnd = '?='  /* define the end of an encoded string */
GotCode = 0  /* we have not seen any encoded stuff yet */
NewText = ''  /* start with nothing */

do while (Text >< '')  /* as long as we have a bit of string left */

 parse var Text NextBit Text  /* get the next bit */
 parse var NextBit FrontBit (CodeStart) CharSet '?' Encoding '?' CodedBit (CodeEnd) BackBit  /* extract the bits we want */

 if (CharSet >< '') then  /* if we found a character set code, we have encoded text */
 do

  select  /* do one of the following */

   when (translate(CharSet) = 'US-ASCII') then  /* if the character set is US-ASCII */
   do

    if (translate(Encoding) = 'Q') then  /* if it uses quoted-printable encoding */
    do
     NewStr = translate(CodedBit,' ','_')  /* decode underscores into spaces */
    end
    else  /* if it uses b64 encoding */
    do
     NewStr = DecodeB64(CodedBit)  /* decode it */
    end

   end

   when (translate(CharSet) = 'ISO-8859-1') then  /* if the character set is ISO-8859-1 */
   do

    if (translate(Encoding) = 'Q') then  /* if it uses quoted-printable encoding */
    do

     NewStr = ''  /* start with nothing */

     do while (CodedBit >< '')  /* as long as we have coded content */

      parse var CodedBit NextChar 2 CodedBit  /* get the first character of what we have left */

      if (NextChar = '=') then  /* if it is a "=" */
      do
       parse var CodedBit NextChar 3 CodedBit  /* get the next two characters */
       NewBit = x2c(NextChar)  /* convert them from hex to character */
      end
      else  /* if it is not a "=" */
      do
       NewBit = translate(NextChar,' ','_')  /* turn an underscore into a space */
      end

      NewStr = NewStr||NewBit  /* add the result to what we have */

     end

    end

    else  /* if it uses b64 encoding */

    do
     NewStr = DecodeB64(CodedBit)  /* decode it */
    end

    NewStr = translate(NewStr,Global.ISOto850,xrange('80'x,'FF'x))  /* convert high bits from ISO 8859-1 to PC850 */

   end

   otherwise  /* if none of the above apply */
   do
    NewStr = '(...)'  /* we have nothing to show until this bit gets added */
   end

  end

  NewStr = FrontBit||NewStr||BackBit  /* stick any front and back bits back on */

  if (\GotCode) then  /* if the previous string contained no code */
  do
   NewStr = ' '||NewStr  /* start with a space */
   GotCode = 1  /* set the code flag */
  end

 end

 else  /* if it is not an encoded string */

 do
  NewStr = ' '||NextBit  /* use it as it is, preceded by a space */
  GotCode = 0  /* reset the code flag */
 end

 NewText = NewText||NewStr  /* add the new bit to the new string */

end

NewText = strip(NewText,'B',' ')  /* remove excess spaces */

return NewText  /* end of DecodeHeader */

/**********************************************************************/
DirCheck: procedure expose Global.  /* checks the existence of a directory */
/**********************************************************************/

parse arg DirSpec,Quiet  /* get the argument */

Quiet = (Quiet = 1)  /* 1 = TRUE */

CurDir = directory()  /* store the current dir */

if (directory(DirSpec) = '') then  /* if we cannot change to the specified dir */
do

 if (\Quiet) then  /* unless we want a quiet check */
 do
  call AddError 'Cannot find directory "'||DirSpec||'"'  /* report an error */
 end

 return 0  /* return no result */

end
else  /* if we did change to the specified dir */
do
 call directory CurDir  /* change back to the original dir */
end

return 1  /* end of DirCheck */

/**********************************************************************/
DeleteDir: procedure expose Global.  /*  deletes a directory and its contents (sysdestroyobject causes problems) */
/**********************************************************************/

parse arg Directory  /* get the argument */

call sysfiletree Directory||'\*','Files.','FOS'  /* get any files in the directory and its subdirectories */

do Index = 1 to Files.0  /* take each of the files found */
 call sysfiledelete Files.Index  /* and delete it */
end

call sysfiletree Directory||'\*','Dirs.','DOS'  /* get any directories in the directory and its subdirectories */

do Index = Dirs.0 to 1 by -1  /* take each of the directories found, starting at the bottom */
 call sysrmdir Dirs.Index  /* and delete it */
end

call sysrmdir Directory  /* zap the topmost directory */

return  /* end of DeleteDir */

/**********************************************************************/
EditMessage: procedure expose Global.  /* edits mail file and sets mail message comment */
/**********************************************************************/

MessFile = Global.ProcFile  /* get the name of the file to edit */

if (\MessageSettings(MessFile,'0*******','MATCH')) then  /* if the file is not an outgoing message */
do
 call AddError 'Invalid message type'  /* report */
 return 0  /* and quit */
end

if (MessageSettings(MessFile,'**1*****','MATCH')) then  /* if the file has been sent, we are dealing with an old version */
do
 call ViewMessage MessFile  /* call the viewer routine */
 return 1  /* and quit */
end

MessageDir = strip(filespec('D',MessFile)||filespec('P',MessFile),'T','\')  /* the directory the message file is in */
IsAddress = (pos(translate(Global.AddrDir),translate(MessageDir)) > 0)  /* if it is an address folder file, set a flag */

if (IsAddress) then  /* if it's an address folder file */
do
 EditFile = MessFile  /* we will just edit the file later on */
end

else  /* if it's not an address folder file */

do

 OKToSend = MessageSettings(MessFile,'*1******','MATCH')  /* see if the original message is OK to send */
 call MessageSettings MessFile,'*0******','CHANGE'  /* make the file not O.K. to send */

 if (Global.ASCIIText) then  /* if all mail is to be sent as ASCII text */
 do
  call MessageSettings MessFile,'***1****','CHANGE'  /* set the ASCII flag */
 end

 if (\MessageContents(MessFile)) then  /* if we cannot get the message contents */
 do
  return 0  /* and quit */
 end

 if (GetHeaderEntry(Global.MessHead,'FROM:') = '') then  /* if we have no "From:" line */
 do

  if ((GetHeaderEntry(Global.MessHead,'REPLY-TO:') = '') & (Global.ReplyAddress >< '')) then  /* if we have no "Reply-to:" line, and we have a default reply address */
  do
   Global.MessHead = 'Reply-To: <'||Global.ReplyAddress||'>'||Global.CRLF||Global.MessHead  /* Add a "Reply-To: " line */
  end

  Global.MessHead = 'From: "'||Global.Name||'" <'||Global.Address||'>'||Global.CRLF||Global.MessHead  /* Add a "From: " line */

  if (Global.ClosingText >< '') then  /* if we have a closing text file */
  do

   if (FileOpen(Global.ClosingText,'READ')) then  /* if we can open the file, i.e. it exists and is ready */
   do
    Global.MessBody = Global.MessBody||Global.EmptyLine||charin(Global.ClosingText,1,chars(Global.ClosingText))  /* add the closing text to the message text after an empty line */
    call FileClose Global.ClosingText  /* close the closing phrase file */
   end

  end


 end

 call HeadCheck  /* check the header entries */
 EditText = MakeHeaderLines(Global.MessHead,,0)||Global.CRLF||Global.MessBody  /* the text to write to the edit file */

 AttDir = AttDirGet(MessFile,1)  /* get the attachments dir, creating it if necessary  */

 if (AttDir = '') then  /* if we still have no att dir */
 do
  return 0  /* quit */
 end

 if (\syscreateobject('WPFolder',Global.ToCcBcc,AttDir,'NODELETE=YES;NODRAG=YES;NOMOVE=YES;NORENAME=YES;NOSETTINGS=YES;ICONFILE='||Global.IconDir||'\MailToCc.ICO;BACKGROUND='||Global.IconDir||'\FolderBG.BMP,T,,I;','FAIL')) then  /* if we cannot create an address insertion folder in the attachments folder */
 do
  call LogAction 'Using existing address insertion folder in "'||AttDir||'"'  /* report */
 end

 if (Global.OpenAttachBeforeEdit) then  /* if we want the folder open on the desktop */
 do

  if (\sysopenobject(AttDir,0,1)) then  /* if we cannot open the attachments folder */
  do
   call AddError 'Cannot open attachments folder'  /* report */
  end

 end

 EditFile = AttDir||'.EDIT'  /* the matching edit file name */

 if (\FileOpen(EditFile,'WRITE')) then  /* if we cannot open the file for writing */
 do
  return 0  /* just quit */
 end

 call charout EditFile,EditText  /* write the edit text to the edit file */
 call FileClose EditFile  /* close the edit file */

end

if (\IsAddress) then  /* if it's not an address folder file */
do
 call sysfiletree EditFile,'Files.','FO','+****','-****'  /* reset the archive attribute of the edit file */
end

call RunCommand Global.Editor,EditFile  /* try to run the edit command */

if (IsAddress) then  /* if it's an address folder file */
do
 return 1  /* we're done */
end

if ((\FileCheck(MessFile)) | (\FileCheck(EditFile))) then  /* if either file has ceased to exist */
do
 return 1  /* we're done */
end

ToAdds = ''  /* start with nothing */
CcAdds = ''  /* start with nothing */
BccAdds = ''  /* start with nothing */
KeepAdds = 0  /* assume we can remove the address insertion folder afterwards */

call sysfiletree AttDir||'\'||Global.ToCcBcc||'\*','Addresses.','FO'  /* look for address files in the address insertion folder */

if (Addresses.0 > 0) then  /* if we found any */
do

 do Index = 1 to Addresses.0  /* run through the files we found */

  if (MessageSettings(Addresses.Index,,'CHECK')) then  /* if the file is a RexxMail message */
  do

   call MessageContents Addresses.Index,'Global.AttHead','Global.AttBody'  /* get the file contents */

   if (MessageSettings(Addresses.Index,'1*******','MATCH')) then  /* if the message is incoming */
   do

    Adds = GetHeaderEntry(Global.AttHead,'From:')  /* look for this entry in the message header */

    if (Adds >< '') then  /* if we find something */
    do
     ToAdds = ToAdds||', '||Adds  /* add it to what we already have */
    end

   end

   else  /* if the message is outgoing */

   do

    Adds = GetHeaderEntry(Global.AttHead,'To:')  /* look for this entry in the message header */

    if (Adds >< '') then  /* if we find something */
    do
     ToAdds = ToAdds||', '||Adds  /* add it to what we already have */
    end

    Adds = GetHeaderEntry(Global.AttHead,'Cc:')  /* look for this entry in the message header */

    if (Adds >< '') then  /* if we find something */
    do
     CcAdds = CcAdds||', '||Adds  /* add it to what we already have */
    end

    Adds = GetHeaderEntry(Global.AttHead,'Bcc:')  /* look for this entry in the message header */

    if (Adds >< '') then  /* if we find something */
    do
     BccAdds = BccAdds||', '||Adds  /* add it to what we already have */
    end

   end

   call sysdestroyobject Addresses.Index  /* get rid of the message file */

  end

  else  /* if the file is not a RexxMail message */

  do

   if (FileOpen(Addresses.Index,'READ')) then  /* if we can open the file for reading */
   do
    AddsCont = charin(Addresses.Index,1,chars(Addresses.Index))  /* get the file content */
    call stream Addresses.Index,'C','CLOSE'  /* close the file */
   end

   AddsCont = AddressFormat(AddsCont,0,1)  /* format the content to get an unindented addresses string and insert warnings */

   if (pos(Global.Warning,AddsCont) = 0) then  /* if all is well */
   do
    ToAdds = ToAdds||', '||AddsCont  /* add the new stuff to what we already have */
    call sysdestroyobject Addresses.Index  /* get rid of the file */
   end
   else  /* if we have one or more invalid addresses */
   do
    KeepAdds = 1  /* keep the address insertion folder */
   end

  end

 end

end

if (\KeepAdds) then  /* if we are to remove the address insertion folder */
do
 call sysdestroyobject AttDir||'\'||Global.ToCcBcc  /* get rid of the folder -- no checks, it may have been deleted already by the user */
end

AttDir = AttDirClose(MessFile)  /* close the attachments dir; we get a (linked) name in return if there are any contents left */
GotAtt = (AttDir >< '')  /* if we have a name, we have contents */
Settings = MessageSettings(MessFile)  /* get the message settings from the original file (they may have been changed in the meantime) */
NewName = ''  /* we have no new name yet */

if (\MessageContents(EditFile)) then  /* if we cannot get the new edit message contents */
do
 return 0  /* quit with an error */
end

call sysfiledelete MessFile  /* zap the original message file */

if (\FileOpen(MessFile)) then  /* if we cannot open a new message file for writing */
do
 return 0  /* quit with an error */
end

call HeadCheck ToAdds,CcAdds,BccAdds  /* check the header entries and add any extra recipients */
call charout MessFile,MakeHeaderLines(Global.MessHead,,0)||Global.CRLF||Global.MessBody  /* write the new header, a separating empty line, and the body to the message file */
call FileClose MessFile  /* close the message file */

NewName = MakeTitle(MessFile,1,KeepAdds,0,0)  /* get a new name for the mail file, and insert warnings as necessary */
OKToSend = (pos(Global.Warning,NewName) = 0)  /* if the title contains no warnings, the new message is O.K. to send */

call sysfiledelete EditFile  /* get rid of the edit file */
call MessageSettings MessFile,left(Settings,1)||,  /* change the message settings: copy the first position */
                              OKToSend||,  /* set the second position according to the OKToSend flag */
                              substr(Settings,3,2)||,  /* copy the third and fourth positions */
                              GotAtt||,  /* set the fifth position according to the GotAtt flag */
                              right(Settings,3),'CHANGE'  /* copy the sixth, seventh, and eighth positions */
call syssetobjectdata MessFile,'REXXMAILATTACHMENT='||word('No Yes',GotAtt + 1)  /* set the attachment indicator for the CWMailFile class */

if (GotAtt) then  /* if we have attachments left */
do
 call AttDirLink MessFile,AttDir  /* link the attachments dir to the new message file */
end

if (NewName >< '') then  /* if we have a new name for the message file */
do
 call syssetobjectdata MessFile,'TITLE='||NewName  /* rename the file */
end

return 1  /* end of EditMessage */

/**********************************************************************/
FileCheck: procedure expose Global.  /* checks the existence of a file */
/**********************************************************************/

parse arg FileName,Warning  /* get the argument */

if (stream(FileName,'C','QUERY EXISTS') = '') then  /* if the file does not exist */
do

 if (Warning = 1) then  /* if we want a warning */
 do
  call AddError 'Cannot find file "'||FileName||'"'  /* report */
 end

 return 0  /* return no result */

end

return 1  /* end of FileCheck */

/**********************************************************************/
FileClose: procedure expose Global.  /* closes a file */
/**********************************************************************/

parse arg FileName  /* get the argument */

if (stream(FileName,'C','CLOSE') >< 'READY:') then  /* if we cannot close the file */
do
 call AddError 'Cannot close file "'||FileName||'"'  /* report */
 return 0  /* return with an error */
end

return 1  /* end of FileClose */

/**********************************************************************/
FileInUse: procedure expose Global.  /* checks if a file is in use */
/**********************************************************************/

parse arg FileName  /* get the argument */

InUse = (stream(FileName,'C','OPEN ') >< 'READY:')  /* if we cannot open the file for writing, it is in use */

if (\InUse) then  /* if the file was not in use */
do
 call stream FileName,'C','CLOSE'  /* close it */
end

return InUse  /* end of FileInUse */

/**********************************************************************/
FileOpen: procedure expose Global.  /* attempts to open a file */
/**********************************************************************/

parse arg FileName,Operation  /* get the arguments */

Attempts = 0  /* we haven't tried yet */

do until ((Success) | (Attempts = 10))  /* go on until we succeed or reach the maximum no. of attempts */

 Success = (stream(FileName,'C','OPEN '||Operation) = 'READY:')  /* if we can open the file, all is well */

 if (\Success) then  /* if we did not succeed */
 do

  Attempts = Attempts + 1  /* up the attempts counter */

  if (Attempts = 10) then  /* if we've reached the last try */
  do
   call AddError 'Cannot open file "'||FileName||'"'  /* report */
  end
  else  /* if we have more tries left */
  do
   call syssleep 0.1  /* wait a bit */
  end

 end

end

return Success  /* end of FileOpen */

/**********************************************************************/
GetAddressList: procedure expose Global.  /* extract a list of names and/or addresses from a formatted list of addresses */
/**********************************************************************/

parse arg Addresses,AddressType  /* get the arguments */

Names = ''  /* start with nothing */
Separator = ','||Global.CRLF  /* the separator between full addresses */

do while (Addresses >< '')  /* as long as we have bits left */

 if (Names >< '') then  /* if we already have something */
 do
  Names = Names||'. '  /* add a separator */
 end

 parse var Addresses Address (Separator) Addresses  /* get the next address */

 if (words(Address) > 1) then  /* if the address contains more than one word */
 do
  NamePart = strip(subword(Address,1,words(Address)-1),'B','"')  /* all but the last must be the name, and get rid of any quotation marks */
  Address = subword(Address,words(Address))  /* the last is the address */
 end
 else  /* if the address contains just a single word */
 do
  NamePart = ''  /* we have no name part */
 end

 AddressType = translate(left(AddressType,1))  /* use just the first character, in upper case */

 select  /* do one of the following */

  when (AddressType = 'F') then  /* if we want it all */
  do

   if (NamePart >< '') then  /* if we have a name part */
   do
    NewName = NamePart||' '||Address  /* use this */
   end
   else  /* if we do not have a name part */
   do
    NewName = Address  /* use this */
   end

  end

  when (AddressType = 'N') then  /* if we want just the name */
  do

   if (NamePart >< '') then  /* if we have a name part */
   do
    NewName = NamePart  /* use it */
   end
   else  /* if we do not have a name part */
   do
    NewName = Address  /* use the address */
   end

  end

  otherwise  /* if we want just the address */
  do
   Address = strip(Address,'L','<')  /* get rid of the leading angle bracket */
   Address = strip(Address,'T','>')  /* get rid of the trailing angle bracket */
   NewName = Address  /* use the address */
  end

 end

 Names = Names||NewName  /* add the result to the names */

end

return Names  /* end of GetAddressList */

/**********************************************************************/
GetFileEntry: procedure expose Global.  /* looks for key word in files and returns first line with matching content */
/**********************************************************************/

parse arg FileName,SearchString  /* get the arguments */

SearchString = ' '||SearchString||';'  /* add a space and semi-colon */

if (sysfilesearch(SearchString,FileName,'Found.') = 0) then  /* if we can search the file for the search string */
do

 do Index = 1 to Found.0  /* look through the finds */

  if (left(Found.Index,1) >< '#') then  /* if the stuff we found is not a comment */
  do

   parse var Found.Index Entry '=' .  /* we want the bit preceding the '=' */
   Entry = strip(translate(Entry,d2c(32),d2c(9)),'B',' ')  /* convert TABs into spaces and get rid of excess space */

   if (Entry >< '') then  /* if we have something */
   do
    return Entry  /* return with this */
   end

  end

 end

end

return ''  /* end of GetFileEntry */

/**********************************************************************/
GetHeaderEntry: procedure expose Global.  /* returns the first entry or all entries ("More") in a message header with a matching key word */
/**********************************************************************/

parse arg HeaderLines,SeekStr,More  /* get the header lines, the seek string, and any "MORE" parameter */

More = (More = 1)  /* 1= TRUE */
SeekStr = translate(SeekStr)  /* convert the seek string to upper case */
OutLine = ''  /* start with nothing */

do while ((HeaderLines >< '') & ((OutLine = '') | (More)))  /* go on until we run out of header content or get what we want */

 parse var HeaderLines KeyWord ':' Entry (Global.CRLF) HeaderLines  /* get the relevant bits */

 if (SeekStr = translate(KeyWord)||':') then  /* if the uppercased keyword is the one we're looking for */
 do

  Entry = strip(Entry,'B',' ')  /* get rid of excess space */

  if (Entry >< '') then  /* if there is something left */
  do

   if (OutLine >< '') then  /* if we already have something */
   do
    Entry = Global.CRLF||Entry  /* add a new line */
   end

   OutLine = OutLine||Entry  /* add the new bit to the output text */

  end

 end

end

return OutLine  /* end of GetHeaderEntry */

/**********************************************************************/
GetRedirect: procedure expose Global.  /* check for a destination or action in a destination/action specs file */
/**********************************************************************/

parse arg MoveFile,FromText,ToText,SubjectText,JunkMail,VirusMail  /* get the arguments */

SearchText. = ''  /* start with nothing */
Counter = 1  /* start at 1 */

SearchText.Counter = '*DEFAULT*'  /* start with this */

if (JunkMail = 1) then  /* if it is junk mail */
do
 Counter = Counter + 1  /* up the counter */
 SearchText.Counter = '*JUNKMAIL*'  /* add this */
end

if (VirusMail = 1) then  /* if it is virus mail */
do
 Counter = Counter + 1  /* up the counter */
 SearchText.Counter = '*VIRUSMAIL*'  /* add this */
end

if (FromText >< '') then  /* if we have a sender string */
do

 FromText = translate(AddressFormat(FromText,,0))  /* reformat the sender's address(es), bare, no check, upper case */

 do Index = 1 to words(FromText)  /* as long as we have sender addresses */
  NextWord = word(FromText,Index)  /* get the next word */
  parse var NextWord . '@' NextDomain  /* extract the sender's domain */
  Counter = Counter + 1  /* up the counter */
  SearchText.Counter = '<'||NextWord  /* store this */
  Counter = Counter + 1  /* up the counter */
  SearchText.Counter = '<'||NextDomain  /* store this */
  Counter = Counter + 1  /* up the counter */
  SearchText.Counter = NextWord  /* store this */
  Counter = Counter + 1  /* up the counter */
  SearchText.Counter = NextDomain  /* store this */
 end

end

if (ToText >< '') then  /* if we have a recipients string */
do

 ToText = translate(AddressFormat(ToText,,0))  /* reformat the recipient's address(es), bare, no check, upper case */

 do Index = 1 to words(ToText)  /* as long as we have recipient addresses */
  NextWord = word(ToText,Index)  /* get the next word */
  parse var NextWord . '@' NextDomain  /* extract the sender's domain */
  Counter = Counter + 1  /* up the counter */
  SearchText.Counter = '>'||NextWord  /* store this */
  Counter = Counter + 1  /* up the counter */
  SearchText.Counter = '>'||NextDomain  /* store this */
  Counter = Counter + 1  /* up the counter */
  SearchText.Counter = NextWord  /* store this */
  Counter = Counter + 1  /* up the counter */
  SearchText.Counter = NextDomain  /* store this */
 end

end

if (SubjectText >< '') then  /* if we have a subject string */
do

 SubjectText = translate(translate(SubjectText),' ',Global.WordSeparators||d2c(9))  /* upper case, and turn TABs and other word separators into spaces */
 NewText = ''  /* start with nothing (we are going to get rid of multiple spaces in the subject text string) */

 do while(SubjectText >< '')  /* as long as we have words left in the subject text string */
  parse var SubjectText NextWord SubjectText  /* get the next word, without any spaces */
  NewText = NewText||' '||NextWord  /* and add it to what we have, after a single space */
 end

 WordCount = words(NewText)  /* the number of words in the subject string */

 do Index = 1 to WordCount  /* for the number of words in the string */

  do SubIndex = Index to WordCount  /* for each of the successive words left */
   NextString = subword(NewText,Index,(SubIndex - Index + 1))  /* take this section */
   Counter = Counter + 1  /* up the counter */
   SearchText.Counter = '='||NextString  /* store this */
   Counter = Counter + 1  /* up the counter */
   SearchText.Counter = NextString  /* store this */
  end

 end

end

Entry = ''  /* start with nothing */
LastLineNo = 999999  /* start high */

do Index = 1 to Counter  /* walk through the list */

 SearchString = SearchText.Index||';'  /* look for this */

 if (sysfilesearch(SearchString,MoveFile,'Found.','N') = 0) then  /* if we can search the destination specs file */
 do

  do SubIndex = 1 to Found.0  /* look through the finds */

   parse var Found.SubIndex LineNo FoundLine  /* split it into the bits we want */

   if ((left(FoundLine,1) >< '#') & (LineNo < LastLineNo)) then  /* if the stuff we found is not a comment, and the line number precedes our last best find */
   do

    FoundLine = translate(FoundLine,' ',d2c(9))  /* change TABs into spaces */
    UpperLine = translate(FoundLine)  /* make it all upper case in this copy */

    parse var UpperLine UpperLine (SearchString) .  /* get the bits we want */

    select  /* do one of the following */

     when (right(UpperLine,1) = ' ') then  /* if the preceding character is a space */
     do
      parse var FoundLine Entry '=' .  /* we want the bit preceding the '=' */
      LastLineNo = LineNo  /* store the line number */
     end

     when (right(UpperLine,1) = '&') then  /* if the preceding character is an ampersand */
     do

      LastWord = word(UpperLine,words(UpperLine))  /* get the preceding upper-case word */

      do until ((LastWord = '') | (\Match))  /* go on until we have nothing left or bomb out */

       parse var LastWord NextWord '&' LastWord  /* get the next ANDed bit */

       Match = 0  /* no match yet */

       do SubSubIndex = 1 to Counter  /* run through the search strings */

        if (NextWord = SearchText.SubSubIndex) then  /* if the word is in the search text array */
        do
         Match = 1  /* we have a match */
         SubSubIndex = Counter  /* exit the subsubloop */
        end

       end

      end

      if (Match) then  /* if we were successful */
      do
       parse var FoundLine Entry '=' .  /* we want the bit preceding the '=' from the original line */
       LastLineNo = LineNo  /* store the line number */
      end

     end

     otherwise  /* if neither of the above */
     do
      nop  /* do nothing */
     end

    end

   end

  end

 end

end

if (Entry >< '') then  /* if we found something */
do
 Entry = strip(Entry,'B',' ')  /* get rid of excess space */
end

return Entry  /* end of GetRedirect */

/**********************************************************************/
GetTimeZone: procedure expose Global.  /* gets the local time zone */
/**********************************************************************/

if ((symbol('Global.TimeZone') = 'VAR') & (Global.TimeZone >< '')) then  /* if we already have a global time zone var */
do
 return Global.TimeZone  /* simply return it */
end

parse arg NoZone  /* get the argument */

NoZone = (NoZone = 1)  /* 1 = True */

if (NoZone) then  /* if we are not allowed to contact time servers */
do
 return '-0000'  /* return an unknown time zone */
end

TimeZone = ''  /* start with nothing */

do while ((TimeZone = '') & (Global.TimeServers >< ''))  /* as long as we have no result, and we have time servers left */

 parse var Global.TimeServers TimeServer Global.TimeServers  /* get the next entry */
 parse var TimeServer TimeServer ':' PortNumber  /* see if we have a port number */

 if (PortNumber = '') then  /* if not */
 do
  PortNumber = 37  /* default to this (NTP) */
 end

 Socket = ServerConnect(TimeServer,PortNumber,Global.TimeAttempts)  /* get the socket number for a server connection through the specified or default port */

 if (Socket = '') then  /* if we have no socket */
 do
  call LogAction 'Connection failed'  /* report a miss */
 end

 else  /* if we have a socket */

 do

  call LogAction 'Connection established'  /* report */
  SocketBytes = sockrecv(Socket,'TimeBytes',4)  /* see how many bytes we get from the server; look for 4 */
  call SocketClose Socket  /* close the socket */

  if (SocketBytes = 4) then  /* if we got 4 bytes */
  do

   call LogAction 'Time data retrieved'  /* report */
   SysTime = ((date('B') - 693595) * 86400) + time('S')  /* system seconds since midnight, 1900-01-01 */
   TimeBytes = reverse(TimeBytes)  /* reverse the time bytes */
   AbsTime = 0  /* start with nothing */

   do Index = 1 to 4  /* take each of the bytes, starting with the low order one */
    AbsTime = AbsTime + (c2d(substr(TimeBytes,Index,1)) * (256 ** (Index-1)))  /* absolute seconds since midnight, 1900-01-01 */
   end

   TimeDiff = SysTime - AbsTime  /* the time difference in seconds, plus or minus */

   if (sign(TimeDiff) = -1) then  /* if the time difference is negative */
   do
    TimeSign = '-'  /* we need a minus sign */
    TimeDiff = -1 * TimeDiff  /* reverse the sign of the time difference */
   end

   else  /* if the time difference is not negative */

   do
    TimeSign = '+'  /* we need a plus sign */
   end

   TimeDiff = (TimeDiff + 900) % 1800  /* the system offset, rounded to the nearest half hour */

   if ((TimeSign = '-') & (TimeDiff = 0)) then  /* if we have a negative time difference, and the 30-minute offset is 0, i.e. we are only a bit behind */
   do
    TimeSign = '+'  /* use a plus */
   end

   ZoneHours = (TimeDiff % 2) * 100  /* construct the hours number */
   ZoneHalfHours= (TimeDiff // 2) * 30  /* construct the half hours number */
   TimeZone = TimeSign||right((ZoneHours + ZoneHalfHours),4,'0')  /* the whole time zone */
   call LogAction 'UTC offset calculated as "'||TimeZone||'"'  /* report */

  end

  else  /* if we did not get 4 bytes */

  do
   call LogAction 'Time data retrieval failed'  /* report */
  end

 end

end

if (TimeZone = '') then  /* if we still have no time zone */
do
 TimeZone = '-0000'  /* use the "undefined" string instead */
end

return TimeZone  /* end of GetTimeZone */

/**********************************************************************/
HeadCheck: procedure expose Global.  /* checks header contents of outgoing messages */
/**********************************************************************/

parse arg ToAdds,CcAdds,BccAdds  /* get the arguments */

OKKeywords = 'FROM:',  /* allow a from string */
             'TO:',  /* allow a to string */
             'CC:',  /* allow a cc string */
             'BCC:',  /* allow a bcc string */
             'REPLY-TO:',  /* allow a reply to string */
             'SUBJECT:',  /* allow a subject string */
             'COMMENTS:',  /* allow a comments string */
             'IN-REPLY-TO:',  /* allow an in reply to string */
             'REFERENCES:',  /* allow a references string */
             'KEYPHRASES:'  /* allow a keyphrases string */

NewList = ''  /* start afresh */
NewHeader = ''  /* start with nothing */
HeadText = Global.MessHead  /* copy the unfolded header contents */

do while (HeadText >< '')  /* go on while we have bits of unfolded header left */

 parse var HeadText HeadLine (Global.CRLF) HeadText  /* get the next line */

 if (pos(Global.Warning,HeadLine) = 1) then  /* if the line starts with a marker */
 do
  HeadLine = substr(HeadLine,length(Global.Warning) + 1)  /* use the remainder */
 end

 FirstWord = word(HeadLine,1)  /* get the first word of the line */

 if (FirstWord >< '') then  /* if we have something */
 do

  UpFirstWord = translate(Firstword)  /* the uppercase version of the first word */

  if (wordpos(UpFirstword,NewList) > 0) then  /* if it is in the new keywords list, we already have this one */
  do
   HeadLine = Global.Warning||HeadLine  /* add a marker */
  end

  else  /* if it is a new keyword */

  do

   NewList = NewList||' '||UpFirstWord  /* add the keyword to the list of new keywords */

   if (wordpos(UpFirstword,OKKeywords) > 0) then  /* if it is in the O.K. keywords list */

   do

    select  /* do one of the following */

     when (UpFirstWord = 'TO:') then  /* if we are looking at the "TO:" line */
     do
      HeadLine = HeadLine||ToAdds  /* add any extra recipients */
      ToAdds = ''  /* get rid of the extra recipients */
     end

     when (UpFirstWord = 'CC:') then  /* if we are looking at the "CC:" line */
     do
      HeadLine = HeadLine||CcAdds  /* add any extra recipients */
      CcAdds = ''  /* get rid of the extra recipients */
     end

     when (UpFirstWord = 'BCC:') then  /* if we are looking at the "BCC:" line */
     do
      HeadLine = HeadLine||BccAdds  /* add any extra recipients */
      BccAdds = ''  /* get rid of the extra recipients */
     end

     otherwise  /* if none of the above */
     do
      nop  /* do nothing */
     end

    end

   end

   else  /* if it is not in the O.K. keywords list */

   do

    if ((left(UpFirstword,2) >< 'X-') | (right(Firstword,1) >< ':')) then  /* if it is not an X-string followed by a colon, it's a rogue entry */
    do
     HeadLine = Global.Warning||HeadLine  /* so add a marker */
    end

   end

  end

 end

 else  /* if we have nothing */

 do
  HeadLine = Global.Warning||HeadLine  /* add a marker */
 end

 NewHeader = NewHeader||HeadLine||Global.CRLF  /* add this line to the unfolded header */

end

if (ToAdds >< '') then  /* if we still have these extra recipients hanging about, there was no "To:" line in the header */
do
 NewHeader = NewHeader||'To: '||ToAdds||Global.CRLF  /* so create one */
end

if (CcAdds >< '') then  /* if we still have these extra recipients hanging about, there was no "Cc:" line in the header */
do
 NewHeader = NewHeader||'Cc: '||CcAdds||Global.CRLF  /* so create one */
end

if (BccAdds >< '') then  /* if we still have these extra recipients hanging about, there was no "Bcc:" line in the header */
do
 NewHeader = NewHeader||'Bcc: '||BccAdds||Global.CRLF  /* so create one */
end

Global.MessHead = NewHeader  /* copy the unfolded header back to the global variable */

return 1  /* end of HeadCheck */

/**********************************************************************/
HideObject: procedure expose Global.  /* hide an object from view */
/**********************************************************************/

parse arg Object  /* get the argument */

if (\syssetobjectdata(Object,'NOTVISIBLE=YES')) then  /* if we cannot make the object invisible */
do
 return 0  /* return an error */
end

HideCount = Global.Hidden.0 + 1  /* the new value for the hidden objects counter */
Global.Hidden.HideCount = Object  /* store the object name */
Global.Hidden.0 = HideCount  /* up the hidden objects counter */

return 1  /* end of HideObject */

/**********************************************************************/
HTMLFilter: procedure expose Global.  /* filters an HTML file */
/**********************************************************************/

parse arg HTMLFile,NewTitle  /* get the argument */

Changed = 0  /* we have not changed anything yet */

if (FileOpen(HTMLFile,'READ')) then  /* if we can open the file for reading */
do

 HTMLCont = ' '||charin(HTMLFile,1,chars(HTMLFile))  /* get the HTML content and stick a space in front to make sure we see the first tag */
 call FileClose HTMLFile  /* close the file */
 Title = ''  /* no title yet */
 HeaderTags = ''  /* start with nothing */
 NewHTMLCont = ''  /* start with nothing */

 if (pos('<HEAD>',translate(HTMLCont)) > 0) then  /* if we have a <HEAD> tag */
 do
  InHead = 1  /* assume we're in the <HEAD> part to begin with to make sure 'stray' tags get picked up */
 end
 else  /* if we have no <HEAD> tag */
 do
  InHead = 0  /* we're not in the <HEAD> part */
 end

 do while (HTMLCont >< '')  /* as long as we have something left */

  parse var HTMLCont FirstBit '<' TagContent '>' HTMLCont  /* get the next tag content and the part preceding it */

  if (TagContent >< '') then  /* if we have tag content */
  do

   NewContent = ''  /* start with nothing */

   do while (TagContent >< '')  /* run through the content */

    parse var TagContent BeforeBit (Global.CRLF) TagContent  /* look for a bit terminated by a CRLF pair (FrontPage appears to consider this formatting) */
    NewContent = NewContent||BeforeBit  /* always use this bit */

    if (TagContent >< '') then  /* if there is more to come */
    do
     NewContent = NewContent||' '  /* add a space */
    end

   end

   TagContent = NewContent  /* copy the results */
   TagWord = translate(word(TagContent,1))  /* get the first tag word in upper case */

   if (TagWord = 'A') then  /* if the tag is a normal anchor link */
   do
    FirstBit = FirstBit||'<'||TagContent||'>'  /* restore the tag */
   end

   else  /* if the tag is not a normal anchor link */

   do

    if (Global.HTMLSafe) then  /* if we want safe HTML */
    do

     URLPos = pos('HTTP://',translate(TagContent))  /* get the position of any external URL in the tag content (with or without --mandatory-- quotes) */

     if (URLPos > 0) then  /* if we have something */
     do
      parse var TagContent =(URLPos) URLContent .  /* get the URL content */
      TagContent = strip(TagContent,'T','"')  /* remove any double quotes that remain from the URL start */
      URLContent = strip(URLContent,'T','"')  /* remove any double quotes that remain at the end of the URL */

      if (URLContent >< '') then  /* if we have something */
      do

       if (TagWord = 'BODY') then  /* if this is the <BODY> tag, the URL must be an external background bitmap, so */
       do
        parse var TagContent FirstTagContent (URLContent) TagContent  /* split the tag content around the URL */
        TagContent = FirstTagContent||TagContent  /* rewrite the tag content without the URL */
       end

       else  /* if it is not the <BODY> tag */

       do
        URLString = '<FONT SIZE="-1"><I>External '||TagWord||' link:</I>'||Global.CRLF||,  /* small print explanatory text */
                    '<A HREF="'||URLContent||'">'||URLContent||'</A></FONT>'||Global.CRLF  /* followed by the URL link */

        if (InHead) then  /* if we are still inside the <HEAD> part */
        do
         HeaderTags = HeaderTags||'<BR>'||Global.CRLF||URLString  /* add a line break to HeaderTags, followed by the URL string */
        end
        else  /* if we are not inside the <HEAD> part */
        do
         FirstBit = FirstBit||Global.CRLF||,  /* start a new source line */
                    '<HR SIZE=2>'||Global.CRLF||URLString||,  /* add a horizontal rule followed by the URL string */
                    '<HR SIZE=2>'||Global.CRLF  /* and end with another horizontal rule */
        end

       end

      end

      TagContent = ''  /* delete the tag content to signal that we have finished processing this tag */
      Changed = 1  /* we have a change */

     end

    end

    if (TagContent >< '') then  /* if we still have tag content */
    do

     select  /* do one of the following */

      when (TagWord = 'HEAD') then  /* if the content starts with this tag */
      do
       FirstBit = FirstBit||'<HEAD>'||Global.CRLF||,  /* restore the header tag */
                  '<META NAME="MSSmartTagsPreventParsing" CONTENT="TRUE">'||Global.CRLF  /* add an anti-spam tag */
      end

      when (TagWord = '/HEAD') then  /* if the content starts with this tag */
      do
       FirstBit = FirstBit||'</HEAD>'||Global.CRLF  /* restore the tag */
       InHead = 0  /* we're no longer inside the <HEAD> part */
      end

      when (TagWord = 'IMG') then  /* if the content starts with this tag */
      do
       TagContent = LocalURLTag(TagContent,'SRC')  /* look for a URL filename and change the tag content if we find it */
       FirstBit = FirstBit||'<'||TagContent||'>'  /* restore or redo the tag */
      end

      when (TagWord = 'BODY') then  /* if this is the 'BODY' tag */
      do

       TagContent = LocalURLTag(TagContent,'BACKGROUND')  /* look for a URL filename and change the tag content if we find it */
       FirstBit = FirstBit||'<'||TagContent||'>'  /* restore or redo the tag */

       if (HeaderTags >< '') then  /* if we have converted header tags */
       do
        FirstBit = FirstBit||Global.CRLF||,  /* start a new source line */
                   '<HR SIZE=2>'||Global.CRLF||,  /* add a horizontal rule */
                   '<B>NOTE:</B>'||Global.CRLF||,  /* add bold text */
                   '<BR>'||Global.CRLF||,  /* add a line break */
                   '<I>RexxMail removed the following header content:</I>'||Global.CRLF||,  /* add italic text */
                   HeaderTags||,  /* add the rewritten header tags */
                   '<HR SIZE=2>'||Global.CRLF  /* and end with another horizontal rule */
       end

      end

      when ((TagWord = 'TITLE') & (NewTitle)) then  /* if this is the 'TITLE' tag and we want a new title */
      do

       NextTag = ''  /* no next tag yet */

       do while (translate(word(NextTag,1)) >< 'TITLE')  /* go on until we find the title end tag */
        parse var HTMLCont NextBit '</' NextTag '>' HTMLCont  /* get the next content bit and the next tag */
        Title = Title||NextBit  /* add the content bit to our title */
       end

       Title = translate(Title,'  ',d2c(10)||d2c(13))  /* turn any CRLFs into spaces */
       Title = CheckCommLine(Title)  /* remove any awkward characters */
       FirstBit = FirstBit||'<TITLE>'||Title||'</TITLE>'||Global.CRLF  /* restore the title part */

      end

      otherwise  /* if none of the above apply */
      do
       FirstBit = FirstBit||'<'||TagContent||'>'  /* restore the tag */
      end

     end

    end

   end

   NewHTMLCont = NewHTMLCont||FirstBit  /* add the first bit to the new content */

  end

 end

 if (sysfiledelete(HTMLFile) = 0) then  /* if we can delete the original HTML file */
 do

  if (FileOpen(HTMLFile,'WRITE')) then  /* if we can open a new file for writing */
  do
   NewHTMLCont = strip(NewHTMLCont,'B',' ')  /* get rid of excess spaces */
   call charout HTMLFile,NewHTMLCont  /* write the new content to the file */
   call FileClose HTMLFile  /* close the file */
  end

 end

 if (Title >< '') then  /* if we have a new title */
 do
  if (\syssetobjectdata(HTMLFile,'TITLE='||Title||'.HTML')) then  /* if we cannot set the new title */
  do
   call AddError 'Cannot set title'  /* report a non-fatal error */
  end
 end

end

return Changed  /* end of HTMLFilter */

/**********************************************************************/
HTMLToText: procedure expose Global.  /* converts (ISO 8859-1) HTML to PC-850 text */
/**********************************************************************/

parse arg HTMLText  /* get the argument */

HTMLText = strip(HTMLText,'B',' ')  /* get rid of excess blanks */

if (translate(right(HTMLText,7)) >< '</HTML>') then  /* if the HTML stuff does not end with the correct end tag */
do
 HTMLText = HTMLText||'</HTML>'  /* add it to make sure the text part gets bumped out */
end

HTMLCodes = 'euro',  /* euro sign */
            copies('_ ',15),  /* 15 blanks */
            'quot',  /* double quotation mark */
            'amp',  /* ampersand */
            'lt',  /* less-than sign */
            'gt',  /* greater-than sign */
            copies('_ ',12),  /* 12 blanks */
            'nbsp',  /* non-breaking space */
            'iexcl',  /* inverted exclamation mark */
            'cent',  /* cent sign */
            'pound',  /* pound sterling sign */
            'curren',  /* general currency sign */
            'yen',  /* yen sign */
            'brvbar',  /* broken vertical bar */
            'sect',  /* section sign */
            'uml',  /* umlaut (dieresis) */
            'copy',  /* copyright */
            'ordf',  /* feminine ordinal */
            'laquo',  /* left angle quote, guillemet left */
            'not',  /* not sign */
            'shy',  /* soft hyphen */
            'reg',  /* registered trademark */
            'macr',  /* macron accent */
            'deg',  /* degree sign */
            'plusmn',  /* plus or minus */
            'sup2',  /* superscript two */
            'sup3',  /* superscript three */
            'acute',  /* acute accent */
            'micro',  /* micro sign */
            'para',  /* paragraph sign */
            'middot',  /* middle dot */
            'cedil',  /* cedilla */
            'sup1',  /* superscript one */
            'ordm',  /* masculine ordinal */
            'raquo',  /* right angle quote, guillemet right */
            'frac14',  /* fraction one quarter */
            'frac12',  /* fraction one half */
            'frac34',  /* fraction three quarters */
            'iquest',  /* inverted question mark */
            'Agrave',  /* uppercase A, grave */
            'Aacute',  /* uppercase A, acute */
            'Acirc',  /* uppercase A, circumflex */
            'Atilde',  /* uppercase A, tilde */
            'Auml',  /* uppercase A, dieresis or Umlaut */
            'Aring',  /* uppercase A, ring */
            'AElig',  /* uppercase AE diphthong (ligature) */
            'Ccedil',  /* uppercase C, cedilla */
            'Egrave',  /* uppercase E, grave */
            'Eacute',  /* uppercase E, acute */
            'Ecirc',  /* uppercase E, circumflex */
            'Euml',  /* uppercase E, dieresis or Umlaut */
            'Igrave',  /* uppercase I, grave */
            'Iacute',  /* uppercase I, acute */
            'Icirc',  /* uppercase I, circumflex */
            'Iuml',  /* uppercase I, dieresis or Umlaut */
            'ETH',  /* uppercase Eth, Icelandic */
            'Ntilde',  /* uppercase N, tilde */
            'Ograve',  /* uppercase O, grave */
            'Oacute',  /* uppercase O, acute */
            'Ocirc',  /* uppercase O, circumflex */
            'Otilde',  /* uppercase O, tilde */
            'Ouml',  /* uppercase O, dieresis or Umlaut */
            'times',  /* multiply sign */
            'Oslash',  /* uppercase O, slash */
            'Ugrave',  /* uppercase U, grave */
            'Uacute',  /* uppercase U, acute */
            'Ucirc',  /* uppercase U, circumflex */
            'Uuml',  /* uppercase U, dieresis or Umlaut */
            'Yacute',  /* uppercase Y, acute */
            'THORN',  /* uppercase THORN, Icelandic */
            'szlig',  /* lowercase sharp s, sz ligature */
            'agrave',  /* lowercase a, grave */
            'aacute',  /* lowercase a, acute */
            'acirc',  /* lowercase a, circumflex */
            'atilde',  /* lowercase a, tilde */
            'auml',  /* lowercase a, dieresis or Umlaut */
            'aring',  /* lowercase a, ring */
            'aelig',  /* lowercase ae diphthong (ligature) */
            'ccedil',  /* lowercase c, cedilla */
            'egrave',  /* lowercase e, grave */
            'eacute',  /* lowercase e, acute */
            'ecirc',  /* lowercase e, circumflex */
            'euml',  /* lowercase e, dieresis or Umlaut */
            'igrave',  /* lowercase i, grave */
            'iacute',  /* lowercase i, acute */
            'icirc',  /* lowercase i, circumflex */
            'iuml',  /* lowercase i, dieresis or Umlaut */
            'eth',  /* lowercase eth, Icelandic */
            'ntilde',  /* lowercase n, tilde */
            'ograve',  /* lowercase o, grave */
            'oacute',  /* lowercase o, acute */
            'ocirc',  /* lowercase o, circumflex */
            'otilde',  /* lowercase o, tilde */
            'ouml',  /* lowercase o, dieresis or Umlaut */
            'divide',  /* division sign */
            'oslash',  /* lowercase o, slash */
            'ugrave',  /* lowercase u, grave */
            'uacute',  /* lowercase u, acute */
            'ucirc',  /* lowercase u, circumflex */
            'uuml',  /* lowercase u, dieresis or Umlaut */
            'yacute',  /* lowercase y, acute */
            'thorn',  /* lowercase thorn, Icelandic */
            'yuml'  /* lowercase y, dieresis or umlaut */

IndentTags = 'DL',  /* <DL> */
             'OL',  /* <OL> */
             'UL'  /* <UL> */

UnIndentTags = '/DL',  /* </DL> */
               '/OL',  /* </OL> */
               '/UL'  /* </UL> */

NewLineTags = 'BR',  /* <BR> */
              'DIV',  /* <DIV> */
              'P',  /* <P> */
              'HR',  /* <HR> */
              'DT',  /* <DT> */
              'DD',  /* <DD> */
              'LI',  /* <LI> */
              'H1',  /* <H1> */
              '/H1',  /* </H1> */
              'H2',  /* <H2> */
              '/H2',  /* </H2> */
              'H3',  /* <H3> */
              '/H3',  /* </H3> */
              'H4',  /* <H4> */
              '/H4',  /* </H4> */
              'H5',  /* <H5> */
              '/H5',  /* </H5> */
              'H6',  /* <H6> */
              '/H6',  /* </H6> */
              'TD',  /* <TD> */
              '/TD',  /* </TD> */
              'PRE',  /* <PRE> */
              '/PRE',  /* </PRE> */
              'BLOCKQUOTE',  /* <BLOCKQUOTE> */
              '/BLOCKQUOTE',  /* </BLOCKQUOTE> */
              'TITLE',  /* <TITLE> */
              '/TITLE',  /* </TITLE> */
              '/HTML'  /* </HTML> */

Text = ''  /* start with nothing */
AddText = ''  /* start with nothing to add */
TypeStack = ''  /* start with an empty type stack */
PreText = 0  /* no preformatted text yet */
Indent = 0  /* start with no indent */
URLCount = 0  /* start a counter */
URLList. = ''  /* start with no URLs */

if (Global.HTMLLineLength = 0) then  /* if we have a line length limit of 0, i.e. unwrapped text */
do
 SepLength = Global.ASCIILineLength  /* use this */
end
else  /* if we have a line wrap length */
do
 SepLength = Global.HTMLLineLength  /* use that instead */
end

do while (HTMLText >< '')  /* as long as we have HTML text left */

 parse var HTMLText NextPart '<' TagCont '>' HTMLText  /* get the next part before any tag, and keep the tag content */

 if (TagCont >< '') then  /* if we have tag content */
 do

  TagWord = translate(word(TagCont,1))  /* get the first word of the tag content in upper case */

  select  /* do one of the following */

   when (left(TagWord,3) = '!--') then  /* if it is the start of a comment */
   do

    if (right(TagCont,2) >< '--' ) then  /* if the tag content does not end with a comment terminator, the end tag we have must be part of the comment */
    do
     parse var HTMLText . '-->' HTMLText  /* so remove the remaining comment from the rest of the HTML text */
     TagCont = ''  /* get rid of the tag content to prevent further processing */
    end

   end

   when (TagWord = 'SCRIPT') then  /* if it is the start of a script */
   do

    do until (TagWord = '/SCRIPT')  /* go on until we get to the end of the script */
     parse var HTMLText . '<' TagCont '>' HTMLText  /* look for the next tag */
     TagWord = translate(word(TagCont,1))  /* get the first word of the tag content in upper case */
    end

    TagCont = ''  /* get rid of the tag content to prevent further processing */
    HTMLText = '<P>'||HTMLText  /* add a new paragraph -- scripts may insert whitespace */

   end

   otherwise  /* if none of the above apply */
   do
    nop  /* do nothing */
   end

  end

 end

 if (NextPart >< '') then  /* if we have something before the tag */
 do

  do while (NextPart >< '')  /* as long as we have something left */

   parse var NextPart BeforePart '&' CharCode ';' NextPart  /* get the bit before any character code, and keep the code */

   if (CharCode >< '') then  /* if we have a character code */
   do

    if ((left(CharCode,1) = '#') & (datatype(substr(CharCode,2),'W'))) then  /* if the code is a character number */
    do

     select  /* do one of the following */

      when (substr(CharCode,2) = '8212') then  /* if it is the emdash code */
      do
       Character = '--'  /* use a double dash */
      end

      when (substr(CharCode,2) = '8364') then  /* if it is the euro code */
      do
       Character = d2c(213)  /* use the euro symbol */
      end

      otherwise  /* if it is none of the above */
      do
       Character = translate(d2c(substr(CharCode,2)),Global.ISOto850,xrange('80'x,'FF'x))  /* convert the ISO-8859-1 code to the PC-850 equivalent */
      end

     end

    end
    else  /* if the character code is not a number */
    do

     Position = wordpos(CharCode,HTMLCodes)  /* look for the character code in the HTML codes string */

     if (Position = 0) then  /* if we do not find it */
     do
      Character = CharCode  /* use the original code; it's probably malformed text */
     end
     else  /* if we do find it */
     do
      Character = substr(Global.ISOto850,Position,1)  /* get the character we want from the ISOto850 array */
     end

    end

    BeforePart = BeforePart||Character  /* add the character to what we have */

   end

   AddText = AddText||BeforePart  /* add the result to what we have */

  end

 end

 if (TagCont >< '') then  /* if we have tag content */
 do

  if ((TagWord = 'A') & (Global.HTMLURLList)) then  /* if it is an anchor and we want URLs listed */
  do

   parse var TagCont . '=' URLCont  /* get the URL content */

   if (URLCont >< '') then  /* if we have URL content */
   do
    URLCount = URLCount + 1  /* up the URL counter */
    URLList.URLCount = URLCont  /* store the URL */
    AddText = AddText||' [URL '||URLCount||'] '  /* add a marker to the text */
   end

  end

  else  /* if we are not dealing with an anchor */

  do

   if (PreText) then  /* if we are dealing with preformatted text */
   do

    if (TagWord = '/PRE') then  /* if the tag signals the end of preformatted text */
    do

     do while (AddText >< '')  /* walk through the part we have collected */
      parse var AddText NextBit (Global.CRLF) AddText  /* get the bit up to the next CRLF */
      Text = Text||copies(' ',(Indent * 4))||NextBit||Global.CRLF  /* add it to what we have, with an indent */
     end

     PreText = 0  /* reset the preformatted text flag */

    end

    else  /* if it is not the end of the preformatted text */

    do
     AddText = AddText||'<'  /* add the left angle bracket to what we have */
     HTMLText = TagCont||'>'||HTMLText  /* restore the remainder to be processed next time around */
    end

   end

   else  /* if we are not dealing with preformatted text */

   do

    if (wordpos(TagWord,IndentTags||' '||UnIndentTags||' '||NewLineTags) > 0) then  /* if it is one of these */
    do

     NewText = ''  /* start with nothing */

     if (AddText >< '') then  /* if we have something to add */
     do

      AddText = strip(translate(AddText,'   ',d2c(9)||Global.CRLF),'B',' ')  /* take what we have, turn TAB and CRLF into space, and remove excess space */

      do while (AddText >< '')  /* as long as we have something left */
       parse var AddText NextWord AddText  /* get the next word (to get rid of excess whitespace) */
       NewText = NewText||NextWord||' '  /* and add it to what we have, followed by a space */
      end

      NewText = strip(NewText,'T',' ')  /* remove the trailing blank */

     end

     if (NewText >< '') then  /* if we have something to add */
     do

      if (Text >< '') then  /* if previous text exists */
      do

       if (right(Text,2) = Global.CRLF) then  /* if the previous text ends with a CRLF */
       do
        Text = Text||copies(' ',(Indent * 4))  /* add a full indent */
       end

      end

      if (Global.HTMLLineLength > 0) then  /* if we have a line length limit */
      do
       NewText = WordWrap(NewText,(Global.HTMLLineLength - (Indent * 4)),(Indent * 4))  /* word-wrap the new text part with a full indent */
      end

      Text = Text||NewText||Global.CRLF  /* add the new text to the existing text, followed by a new line */

     end

     else  /* if we have nothing to add */

     do

      if ((wordpos(TagWord,NewLineTags) > 0) & (Text >< '')) then  /* if it is one of these and we already have text */
      do

       if (right(Text,4) >< Global.CRLF||Global.CRLF) then  /* if the previous text does not end with a double CRLF */
       do
        Text = Text||Global.CRLF  /* add a new line */
       end

      end

     end

     select  /* do one of the following */

      when (wordpos(TagWord,IndentTags) > 0) then  /* if it is one of these */
      do

       TypeStack = TagWord||' '||TypeStack  /* push the tag word onto the type stack */
       Indent = Indent + 1  /* move up */

       if (TagWord = 'OL') then  /* if we are starting an ordered list */
       do
        OListCount.Indent = 0  /* start a new list counter */
       end

      end

      when (wordpos(TagWord,UnIndentTags) > 0) then  /* if it is one of these */
      do

       if (TagWord = '/OL') then  /* if we are ending an ordered list */
       do
        drop OListCount.Indent  /* drop the list counter */
       end

       Indent = Indent - 1  /* move back */
       TypeStack = subword(TypeStack,2)  /* remove the tag word from the type stack */

      end

      otherwise  /* it must be a new line tag */
      do

       select  /* do one of the following */

        when (TagWord = 'P') then  /* if it is a paragraph marker */
        do
         Text = Text||Global.CRLF  /* add another CRLF */
        end

        when (TagWord = 'PRE') then  /* if it is the start of preformatted text */
        do
         PreText = 1  /* set the preformatted text flag */
        end

        when (TagWord = 'DT') then  /* if it is a definition tag */
        do

         if (Indent > 0) then  /* if we have an indent */
         do
          TagIndent = Indent - 1  /* go back one */
         end
         else  /* if we have no indent */
         do
          TagIndent = 0  /* nothing */
         end

         Text = Text||copies(' ',(TagIndent * 4))||'  - '  /* tag indent plus a marker */

        end

        when (TagWord = 'DD') then  /* if it is a definition definition */
        do
         Text = Text||copies(' ',(Indent * 4))  /* full indent */
        end

        when (TagWord = 'LI') then  /* if it is a list item */
        do

         if (Indent > 0) then  /* if we have an indent */
         do
          TagIndent = Indent - 1  /* go back one */
         end
         else  /* if we have no indent */
         do
          TagIndent = 0  /* nothing */
         end

         if (word(TypeStack,1) = 'OL') then  /* if we are in an ordered list */
         do
          OListCount.Indent = OListCount.Indent + 1  /* up the current list counter */
          Text = Text||copies(' ',(TagIndent * 4))||right(OListCount.Indent,2,' ')||'. '  /* indent, the number, a full stop, and a space */
         end
         else  /* we must be in an unordered list */
         do
          Text = Text||copies(' ',(TagIndent * 4))||'   '  /* indent and a bullet */
         end

        end

        when ((wordpos(TagWord,'HR H1 H2 H3 H4 H5 H6 TD TITLE') > 0) & (Global.HTMLLines)) then  /* if it is one of these, and we want lines */
        do

         Text = Text||copies(' ',(Indent * 4))||copies('_',(SepLength - (Indent * 4)))||Global.CRLF  /* full indent, a starting line and a new line */

         if (TagWord = 'HR') then  /* if it is a separator line we want */
         do
          Text = Text||copies(' ',(Indent * 4))||copies('',(SepLength - (Indent * 4)))||Global.CRLF  /* add an ending line */
         end

        end

        when ((wordpos(TagWord,'/H1 /H2 /H3 /H4 /H5 /TD /TITLE') > 0) & (Global.HTMLLines)) then  /* if it is one of these, and we want lines */
        do
         Text = Text||copies(' ',(Indent * 4))||copies('',(SepLength - (Indent * 4)))||Global.CRLF  /* add an ending line */
        end

        otherwise  /* if it is none of the above */
        do
         nop  /* do nothing else */
        end

       end

      end

     end

    end

   end

  end

 end

end

do Index = 1 to URLCount  /* for each of the URLs we found in the text */
 Text = Text||Global.CRLF||'URL '||right(Index,length(URLCount),' ')||' = '||URLList.Index  /* add it to the text on a new line */
end

return Text  /* end of HTMLToText */

/**********************************************************************/
LeapYear: procedure expose Global.  /* check for leap year */
/**********************************************************************/

parse arg Year  /* get the year to analyze */

if (Year = '') then  /* if we have nothing */
do
 Year = substr(date('S'),1,4)  /* use the current year */
end

if (Year // 4 = 0) then  /* if the year is divisible by 4 */
do

 if (Year // 100 = 0) then  /* and if the year is divisible by 100 */
 do

  if (Year // 400 = 0) then  /* and if the year is divisible by 400 */
  do
   Leap = 1  /* it is a leap year */
  end
  else  /* if the year is divisible by 100, but not by 400 */
  do
   Leap = 0  /* it is not a leap year */
  end

 end
 else  /* if the year is divisible by 4, but not by 100 */
 do
  Leap = 1  /* it is a leap year */
 end

end
else  /* if the year is not divisible by 4 */
do
 Leap = 0  /* it is not a leap year */
end

return Leap  /* end of LeapYear */

/**********************************************************************/
LoadRexxSock: procedure expose Global.  /* try to load the REXX socket functions library if necessary */
/**********************************************************************/

if (rxfuncquery('SockLoadFuncs') >< 0) then  /* if the REXX socket functions lib is not already loaded */
do

 if (rxfuncadd('SockLoadFuncs','RxSock','SockLoadFuncs') = 0) then  /* if we can register the general loading function */
 do

  if (sockloadfuncs() = 0) then  /* if we cannot load the functions */
  do
   Result = 2  /* bad result */
  end
  else  /* if we can load them */
  do
   Result = 1  /* indicate we loaded the library */
  end

 end

end

else  /* if the library had already been loaded */

do
 Result = 0  /* indicate that the REXX socket library had already been loaded */
end

return Result  /* end of LoadRexxSock */

/**********************************************************************/
LoadRexxUtil: procedure expose Global.  /* try to load the REXX utilities library */
/**********************************************************************/

if (rxfuncquery('SysLoadFuncs')) then  /* if it is not already loaded */
do

 if (rxfuncadd('SysLoadFuncs','RexxUtil','SysLoadFuncs') = 0) then  /* if we can add it */
 do
  if (SysLoadFuncs() = 0) then  /* if we cannot load it */

  do
   Result = 2  /* bad result */
  end
  else  /* if we can load it */
  do
   Result = 1  /* indicate we loaded the library */
  end

 end

end

else  /* if the library had already been loaded */

do
 Result = 0  /* indicate that the REXX utilities library had already been loaded */
end

return Result  /* end of LoadRexxUtil */

/**********************************************************************/
LocalURLTag: procedure expose Global.  /* converts an URL to the filename of a local attachment file */
/**********************************************************************/

parse arg TagContent,URLString  /* get the arguments */

UpperTagContent = translate(TagContent)  /* make the tag content upper case */
URLString = translate(URLString)||'='  /* add this to the upper case URL string */
parse var UpperTagContent FirstTagContent (URLString) URLContent RestTagContent  /* get the bits we want */

if (URLContent >< '') then  /* if we have URL content */
do

 URLContent = strip(URLContent,'B','"')  /* remove any quotation marks */

 if (left(URLContent,4) = 'CID:') then  /* if it starts with CID: */
 do

  URLContent = substr(strip(URLContent,'B','"'),5)  /* get rid of any double quotation marks and remove the first four characters ('CID:') */
  SourceString = '<'||URLContent||'>'  /* add angle brackets to get a source name */
  parse upper var Global.MessBody FirstBodyPart (SourceString) LastBodyPart +513 .  /* get the body part before and a bit after the source name part, both in upper case */
  FirstBodyPart = left(reverse(FirstBodyPart),512)  /* reverse the first body part and take the first bit */
  parse var FirstBodyPart FileNamePart '=EMAN' Remainder  /* get the file name part that follows 'NAME=' */

  if (Remainder = '') then  /* if we have no remainder, we have no file name part, so */
  do
   parse var LastBodyPart 'NAME=' FileNamePart  /* look for it in the last body part */
  end
  else  /* if we do have something */
  do
   FileNamePart = reverse(FileNamePart)  /* turn the file name part the right way round */
  end

  parse var FileNamePart FileNamePart (Global.CRLF) .  /* remove any extra lines */
  parse var FileNamePart FileNamePart ';' .  /* remove any extra stuff (which would be after a semicolon) */
  FileNamePart = strip(FileNamePart,'B','"')  /* remove any double quotation marks */

 end

 else  /* if the URL does not start with CID: */

 do
  URLContent = reverse(URLContent)  /* turn the URL content back to front */
  parse var URLContent URLContent '/' .  /* dump everything after the first (i.e. preceding the last) slash to get rid of a subdir path */
  FileNamePart = reverse(URLContent)  /* turn the URL content back the right way */
 end

 TagContent = FirstTagContent||URLString||'"'||FileNamePart||'" '||RestTagContent  /* create new tag content */

end

return TagContent  /* end of LocalURLTag */

/**********************************************************************/
LogAction: procedure expose Global.  /* logs an action */
/**********************************************************************/

parse arg Action,Quiet  /* get the argument */

Quiet = (Quiet = 1)  /* a value of 1 indicates True */

if (\Quiet) then  /* unless we want to keep quiet */
do
 call lineout 'STDOUT:',Action  /* write the log text to the standard output device */
end

WriteLog = Global.LogActionLines  /* copy the global log lines value */

if (WriteLog = '') then  /* if we have an empty value */
do
 WriteLog = 1  /* use this */
end

if (WriteLog > 0) then  /* if we are to log errors */
do

 if (\FileOpen(Global.ActionLog,'WRITE')) then  /* if we cannot open the file for writing */
 do
  MessText = 'Cannot open log file "'||Global.ActionLog||'"'  /* the text to display/send */
  call lineout 'STDOUT:',MessText  /* report to the standard output device */
  call SystemMessage 'Action log file write error.',MessText  /* send a message to the user if required */
  return 0  /* return with no success */
 end

 ActionText = ''  /* start with nothing */

 do while (Action >< '')  /* as long as we have text left */

  parse var Action NextBit (Global.CRLF) Action  /* get the bit up to the next CRLF */

  if (NextBit >< '') then  /* if we have something */
  do
   ActionText = ActionText||' '||strip(NextBit,'B',' ')  /* remove excess space and add it to what we have */
  end

 end

 call lineout Global.ActionLog,DateTimeSys()||' :'||ActionText  /* write a date/time stamp and the new action text */
 call FileClose Global.ActionLog  /* close the log file */

end

return 1  /* end of LogAction */

/**********************************************************************/
LogErrors: procedure expose Global.  /* processes and logs collected error messages */
/**********************************************************************/

parse arg ErrorLines  /* get the argument */

if (ErrorLines = '') then  /* if we have no errors */
do
 return 0  /* return with no further action */
end

call beep 333,333  /* sound a simple signal */

ErrorText = ''  /* we have nothing yet */

if (symbol('Global.StoreCommands') = 'VAR') then  /* if the command-line store has been defined */
do

 if (Global.StoreCommands >< '') then  /* if we have command line arguments */
 do
  ErrorText = 'RexxMail received the following command-line arguments:'||Global.CRLF||,  /* the first line of the explanatory text */
              '  '||Global.StoreCommands||Global.CRLF  /* the second line of the explanatory text */
 end

end

if (lastpos(Global.CRLF,ErrorLines) > pos(Global.CRLF,ErrorLines)) then  /* if the error message contains more than one CRLF pair, i.e. if there is more than one message */
do
 Plural = 's'  /* we need a plural s */
end
else  /* if we have just the one entry */
do
 Plural = ''  /* no plural s needed */
end
 
ErrorText = ErrorText||'RexxMail detected the following error'||Plural||':'||Global.CRLF||ErrorLines  /* another line of explanatory text, followed by the error lines */

call charout 'STDOUT:',ErrorText  /* send the error text to standard output */

if ((pos('Fatal error: ',ErrorLines) = 0) & (pos('command',ErrorLines) > 0)) then  /* if we do not have a fatal error, and we do have one or more command errors */
do
 call lineout 'CON:',Global.CRLF||'Use the /help command switch for more information'  /* send instructions to the console */
end

if (symbol('Global.ErrorMail') = 'VAR') then  /* if the Global.ErrorMail settings has been set */
do

 if (Global.ErrorMail) then  /* if we want an error message to be sent to the user */
 do

  if (symbol('Global.InDir') = 'VAR') then  /* if the Global.InDir settings has been set */
  do

   if (Global.InDir >< '') then  /* if we have an incoming messages dir */
   do

    StatusMess = ''  /* start with no status message */

    if (symbol('Global.User') = 'VAR') then  /* if we have a user name */
    do
     StatusMess = StatusMess||'User          = "'||Global.User||'"'||Global.CRLF  /* report */
    end

    if (symbol('Global.MainDir') = 'VAR') then  /* if we have a main folder name */
    do
     StatusMess = StatusMess||'Main folder   = "'||Global.MainDir||'"'||Global.CRLF  /* report */
    end

    if (symbol('Global.SettingsFile') = 'VAR') then  /* if we have a settings file name */
    do
     StatusMess = StatusMess||'Settings file = "'||Global.SettingsFile||'"'||Global.CRLF  /* report */
    end

    call SystemMessage 'RexxMail error.',ErrorText||Global.CRLF||StatusMess  /* insert the error text and send it with the status text as a message to the user */

   end

  end

 end

end

if (symbol('Global.LogErrorLines') >< 'VAR') then  /* if the Global.LogErrorLines settings has not been set (i.e. we bombed out early) */
do
 Global.LogErrorLines = 1  /* set it to 1 */
end

if (Global.LogErrorLines >< 0) then  /* if we are to log errors */
do

 if (symbol('Global.ErrorLog') >< 'VAR') then  /* if the error log name has not been set (i.e. we bombed out very early -- this should not happen) */
 do
  call charout 'STDOUT:','No error log file defined.'  /* send the error text to standard output */
 end

 if (FileOpen(Global.ErrorLog,'WRITE')) then  /* if we can open the log file for writing */
 do

  call charout Global.ErrorLog,DateTimeSys()||' : '  /* write a date/time stamp to the log file */

  FirstLine = 1  /* this is the first line */

  do while (ErrorText >< '')  /* as long as we have error text left */

   parse var ErrorText NextError (Global.CRLF) ErrorText  /* get the next error message */

   if (FirstLine) then  /* if this is the first line */
   do
    FirstLine = 0  /* the next line is not */
   end
   else  /* if this is not the first line */
   do
    call charout Global.ErrorLog,copies(' ',22)  /* write an indent */
   end

   call lineout Global.ErrorLog,NextError  /* write the error text */

  end

  call FileClose Global.ErrorLog  /* close the log file */

 end
 else  /* if we cannot open the error log file */
 do
  call SystemMessage 'Error log write error.',ErrorText||Global.CRLF||,  /* report */
                     'Cannot write to error log file:'||Global.CRLF||,  /* report */
                     '  '||Global.ErrorLog  /* report */
 end

end

return 1  /* end of LogErrors */

/**********************************************************************/
LogMail: procedure expose Global.  /* log a sent or collected message */
/**********************************************************************/

WriteLog = Global.LogMailLines  /* copy the global log lines value */

if (WriteLog = '') then  /* if we have an empty value */
do
 WriteLog = 1  /* use this */
end

if (WriteLog > 0) then  /* if we are to log errors */
do

 if (\FileOpen(Global.MailLog,'WRITE')) then  /* if we cannot open the log file for writing */
 do
  call AddError 'Cannot open log file "'||Global.MailLog||'"'  /* report an error */
  return 0  /* return with no success */
 end

 parse arg Direction,Sender,Recipient,RecipientsTo,RecipientsCc,RecipientsBcc,Subject,MessID  /* get the arguments */

 OutLine = DateTimeSys()||' -- '||Direction||' -- From: '||Sender||' -- '  /* write a date/time stamp, the direction, the sender, and some separators */

 if (Recipient >< '') then  /* if we have a lone recipient (incoming mail only) */
 do
  OutLine = OutLine||'For: '||Recipient||' -- '  /* add the recipient and a separator */
 end

 OutLine = OutLine||'To: '||RecipientsTo||' -- '  /* add the To recipients and a separator */

 if (RecipientsCc >< '') then  /* if we have Cc recipients */
 do
  OutLine = OutLine||'Cc: '||RecipientsCc||' -- '  /* add the Cc recipients and a separator */
 end

 if (RecipientsBcc >< '') then  /* if we have Bcc recipients */
 do
  OutLine = OutLine||'Bcc: '||RecipientsBcc||' -- '  /* add the Bcc recipients and a separator */
 end

 OutLine = OutLine||'Subject: '||Subject||' -- Message-Id: '||MessID  /* add the subject and the message ID */

 call lineout Global.MailLog,OutLine  /* write the log entry */
 call FileClose Global.MailLog  /* close the log file */

end

return 1  /* end of LogMail */

/**********************************************************************/
MailCollect: procedure expose Global.  /* Gets waiting mail messages from POP3 server */
/**********************************************************************/

/**********************************************************************/
/* Have we got an external command to run?                            */
/**********************************************************************/

call RunCommand Global.RunBeforeCollect  /* see if we can run an external command before collecting mail */

if (RunCommand(Global.RunCollect) >< '') then  /* if we can run an external command to collect mail */
do
 call RunCommand Global.RunAfterCollect  /* see if we can run an external command after collecting mail */
 return 1  /* quit */
end

/**********************************************************************/
/* See if we have all the arguments we really need                    */
/**********************************************************************/

if (Global.POP3Server = '') then  /* if we have no server name */
do
 return 0  /* simply return, we are apparently using some other means of collecting mail */
end

if (Global.POP3User = '') then  /* if we have no user name */
do
 call AddError 'Missing configuration entry: POP3User'  /* report */
 return 0  /* return */
end

if (Global.POP3Password = '') then  /* if we have no password */
do
 call AddError 'Missing configuration entry: POP3Password'  /* report */
 return 0  /* return */
end

/**********************************************************************/
/* See if we have a non-standard port number specified                */
/**********************************************************************/

parse var Global.POP3Server Global.POP3Server ':' PortNumber  /* look for a port number */

if (PortNumber = '') then  /* if there is none */
do
 PortNumber = 110  /* default to this (POP3) */
end

/**********************************************************************/
/* See if we are the only collect process active at the moment        */
/**********************************************************************/

MutexName = Global.POP3Server||Global.Pop3User  /* construct the mutex semaphore name */
SemHandle = syscreatemutexsem('\SEM32\'||MutexName)  /* get a semaphore handle */

if (sysrequestmutexsem(SemHandle,1) >< 0) then  /* if we cannot grab the semaphore */
do
 call LogAction 'Another RexxMail process is already collecting mail for "'||,  /* report */
                Global.POP3User||'" from "'||Global.POP3Server||'"'  /* report */
 return 0  /* and quit */
end

/**********************************************************************/
/* Try to connect to the server                                       */
/**********************************************************************/

Socket = ServerConnect(Global.POP3Server,PortNumber,Global.POP3Attempts)  /* get the socket number for a server connection through the specified or default port */

if (Socket = '') then  /* if we have no socket */
do
 call sysreleasemutexsem SemHandle  /* release the semaphore */
 return 0  /* and quit with no success */
end

Global.Socket.SocketBuffer = ''  /* start with an empty socket buffer */

if (\SocketAnswer(Socket,'+OK')) then  /* if we get the wrong return code */
do
 call AddError 'No response from "'||Global.POP3Server||'"'  /* report */
 call sysreleasemutexsem SemHandle  /* release the semaphore */
 return 0  /* and quit with no success */
end

/**********************************************************************/
/* Try to log on to the server                                        */
/**********************************************************************/

call LogAction 'Logging on as user "'||Global.POP3User||'"'  /* report */

if (\SocketSendLine(Socket,'USER '||Global.POP3User,'+OK')) then  /* if we cannot send this and get the right reply */
do
 call LogAction 'User logon failed'  /* report */
 call AddError 'User not accepted'  /* report */
 call ServerDisconnect Socket,Global.POP3Server  /* disconnect from the POP3 server */
 call sysreleasemutexsem SemHandle  /* release the semaphore */
 return 0  /* and quit with no success */
end

call LogAction 'Sending password'  /* report */

if (\SocketSendLine(Socket,'PASS '||Global.POP3Password,'+OK')) then  /* if we cannot send this and get the right reply */
do
 call LogAction 'Password verification failed'  /* report */
 call AddError 'Password not accepted'  /* report */
 call ServerDisconnect Socket,Global.POP3Server  /* disconnect from the POP3 server */
 call sysreleasemutexsem SemHandle  /* release the semaphore */
 return 0  /* and quit with no success */
end

call LogAction 'Logon successful'  /* report */

Messages = POP3Process(Socket)  /* process any waiting mail */

call ServerDisconnect Socket,Global.POP3Server  /* disconnect */
call sysreleasemutexsem SemHandle  /* release the semaphore */
call LogAction 'Messages collected from "'||Global.POP3Server||'": '||Messages  /* report */

call RunCommand Global.RunAfterCollect  /* see if we can run an external command after collecting mail */

return Messages  /* end of MailCollect */

/**********************************************************************/
MailWarning: procedure expose Global.  /* shows that mail has arrived, or deletes/hides its own program object */
/**********************************************************************/

parse arg Hide  /* get the argument, if any */

Hide = (Hide = 1)  /* true if 1 */
ObjectID = '<RexxMail_Warning_Object>'  /* the "Mail!" warning program object ID */
Success = 0  /* no success yet */

if (Hide) then  /* if we are to hide the warning object */
do
 Success = (syssetobjectdata(ObjectID,'NOTVISIBLE=YES'))  /* if we can hide the warning object, all is well */
end

else  /* if we are to create or unhide the object */

do

 if (sysqueryswitchlist('Stem.') = 0) then  /* if we can store the window list contents in Stem. */
 do

  do Index = 1 to Stem.0  /* take each entry */

   Entry = translate(Stem.Index)  /* make the list entry upper case */

   if ((pos(translate(filespec('N',Global.MainDir)),Entry) = 1) & (pos(' VIEW',Entry) = (length(Entry) - 4))) then  /* if it looks identical to the main mail folder */
   do
    Success = 1  /* the mail folder must be open on the desktop, so set a flag */
   end

  end

 end

 if (\Success) then  /* if the main mail folder is not open on the desktop */
 do

  Success = (syssetobjectdata(ObjectID,'NOTVISIBLE=NO'))  /* see if we can unhide the object */

  if (\Success) then  /* if we cannot make make the object visible, we'll simply assume it does not exist */
  do
   Settings = 'EXENAME='||Global.ProgDir||'\'||Global.ProgName||';'||,  /* opening the object will call RexxMail */
              'PARAMETERS=/OPEN;'||,  /* use the "/Open" switch */
              'MINIMIZED=YES;'||,  /* the program will run minimized */
              'OBJECTID='||ObjectID||';'||,  /* use the object ID defined earlier */
              'ICONFILE='||Global.IconDir||'\mailwarn.ico;'  /* attach the MailWarn icon file */
   Success = syscreateobject('WPProgram','Mail!','<WP_DESKTOP>',Settings)  /* if we can create a new program object, all is well */
  end

 end

end

return Success  /* end of MailWarning */

/**********************************************************************/
AddressEntry: procedure expose Global.  /* extracts address from mail file and adds a template to address book folder */
/**********************************************************************/

MessFile = Global.ProcFile  /* get the file name to process */

if (\MessageContents(MessFile)) then  /* if we cannot get the message contents */
do
 return 0  /* and quit */
end

if (MessageSettings(MessFile,'1*******','MATCH')) then  /* if it is an incoming message */
do
 Address = GetHeaderEntry(Global.MessHead,'FROM:')  /* use this address */
end

else  /* if it is not an incoming message */

do

 if (MessageSettings(MessFile,'0*******','MATCH')) then  /* if it is an outgoing message */
 do
  Address = GetHeaderEntry(Global.MessHead,'TO:')  /* use this address */
 end
 else  /* if it is not an outgoing message either */
 do
  call AddError 'Not a valid mail message'  /* report */
  return 0  /* and quit */
 end

end

if (Address = '') then  /* if there is no address */
do
 call AddError 'No address found'  /* report */
 return 0  /* and quit */
end

Address = AddressFormat(Address,4,0)  /* reformat the address, indent to match "To: ", no check */

if (pos(Global.Warning,Address) > 0) then  /* if there is an error marker */
do
 call AddError 'Invalid address found'  /* report */
 return 0  /* and quit */
end

parse var Address Title (Global.CRLF) .  /* remove extra lines */

OutFile = Global.AddrDir||'\ '||CheckCommLine(Title)  /* change dangerous (command line!) characters for the object title and strip spaces, stick a single space in front */

if (FileCheck(OutFile)) then  /* if such a file already exists in the address directory */
do
 call AddError 'Address template already exists'  /* report */
 return 0  /* and quit */
end

PathPart = strip(filespec('D',OutFile)||filespec('P',OutFile),'T','\')  /* the path part */
NamePart = filespec('N',OutFile)  /* the name part */

call syscreateobject 'CWMailFile',NamePart,PathPart,,'FAIL'  /* create a CWMAILFile class object in the address dir using the name part */
call HideObject OutFile  /* make the file invisible */

if (\FileOpen(OutFile)) then  /* if we cannot open the file */
do
 call sysdestroyobject OutFile  /* get rid of the useless object */
 return 0  /* and quit */
end

OutText = 'To: '||Address||Global.CRLF||'Subject: '  /* line up the output text */
call charout OutFile,OutText  /* write the output text to the file */
call FileClose OutFile  /* close the file */

if (\MessageSettings(OutFile,'00000000','CHANGE')) then  /* if we can not make it a fresh outgoing mail message file */
do
 call AddError 'Cannot set address template message attributes'  /* report */
 return 0  /* and quit */
end

call UnhideObject OutFile  /* make the file visible */

if (\syssetobjectdata(OutFile,'REXXMAILREFRESH=YES;TEMPLATE=YES;')) then  /* if we cannot make the file a template etc. */
do
 call AddError 'Cannot set address template attribute'  /* report */
 return 0  /* and quit */
end

return 1  /* end of AddressEntry */

/**********************************************************************/
MakeCWMFMessage: procedure expose Global.  /* converts pre-CWMessFile RexxMail files */
/**********************************************************************/

MessFile = Global.ProcFile  /* get the file name to process */

if (\MessageSettings(MessFile,,'CHECK')) then  /* if this is not a RexxMail Mail message */
do
 call LogAction 'Skipping "'||MessFile||'"'  /* report */
 return 0  /* quit */
end

MessDir = strip(filespec('D',MessFile)||filespec('P',MessFile),'T','\')  /* get the file path and remove the trailing backslash */
TempFile = TempFileName('CONV',MessDir)  /* get a temp file name */
TempName = filespec('N',TempFile)  /* get the temporary file's name part */

if (\syscreateobject('CWMailFile',TempName,MessDir,,'FAIL')) then  /* if we cannot create a CWMAILFile class object in the destination dir using the name part */
do
 call AddError 'Cannot create CWMailFile object'  /* report */
 return 0  /* and quit */
end

call HideObject TempFile  /* make the file object invisible */

if (\FileOpen(TempFile)) then  /* if we cannot open the temp file for writing */
do
 call sysdestroyobject TempFile  /* get rid of the target file */
 return 0  /* and quit */
end

if (\MessageContents(MessFile)) then  /* if we cannot get the source file contents */
do
 return 0  /* and quit */
end

Remains = charout(TempFile,Global.MessHead||Global.EmptyLine||Global.MessBody,1)  /* see if we can write the message content to the temp file */
call FileClose TempFile  /* close the target file */

if (Remains > 0) then  /* if we had remains */
do
 call sysdestroyobject TempFile  /* get rid of the target file */
 call AddError 'Cannot rewrite file'  /* report */
 return 0  /* and quit */
end

Settings = MessageSettings(MessFile)  /* get the message settings */

if (\MessageSettings(TempFile,Settings,'CHANGE')) then  /* if we cannot restore the file settings */
do
 call sysdestroyobject TempFile  /* get rid of the target file */
 call AddError 'Cannot rewrite message settings'  /* report */
 return 0  /* and quit */
end

GotAtt = (substr(Settings,5,1) = 1)  /* set a flag if we had an attachment already marked */
NewName = MakeTitle(TempFile,1,0,0,(left(Settings,1) = 1))  /* get the new title and insert warnings */
call UnhideObject TempFile  /* make the file visible */

if (\syssetobjectdata(TempFile,'REXXMAILREFRESH=YES;REXXMAILATTACHMENT='||,  /* if we cannot set the file attributes */
                               word('No Yes',GotAtt + 1)||';TITLE='||NewName)) then  /* if we cannot set the file attributes */
do
 call sysdestroyobject TempFile  /* get rid of the target file */
 call AddError 'Cannot set title'  /* report */
 return 0  /* and quit */
end

if (\sysdestroyobject(MessFile)) then  /* if we cannot get rid of the source file */
do
 call sysdestroyobject TempFile  /* get rid of the target file */
 call AddError 'Cannot delete file'  /* report */
 return 0  /* and quit */
end

call LogAction 'Converted "'||MessFile||'"'  /* report */

return 1  /* end of MakeCWMFMessage */

/**********************************************************************/
MakeForwardReplyMessage: procedure expose Global.  /* generate a forward or reply message */
/**********************************************************************/

parse arg Switch  /* get the argument */

Reply = (Switch >< 'FORWARD')  /* if we are not forwarding, set a flag to show we are replying */
MessFile = Global.ProcFile  /* the file name to process */

if (\MessageSettings(MessFile,'1*******','MATCH')) then  /* if it is not an incoming message */
do
 call AddError 'Invalid message type'  /* report */
 return 0  /* and quit */
end

MessFileDir = strip(filespec('D',MessFile)||filespec('P',MessFile),'T','\')  /* get the path to the mail message */

if (pos(translate(Global.MainDir),translate(MessFileDir)) > 0) then  /* if it is anywhere in the RexxMail (sub)folders */
do
 OutDir = Global.OutDir  /* create the output file in the 'Out' folder */
end
else  /* if the original message is not in any of the RexxMail folders */
do
 OutDir = MessFileDir  /* we will create the output file in the same folder as the original */
end

AttDir = AttDirCreate()  /* get an attachments dir name, do not link it to the message */

call UnpackMessage MessFile,AttDir,0  /* unpack the message into the attachments dir; do not postprocess the contents */

ReadFile = AttDir||'.READ'  /* the read file name */

if (\FileOpen(ReadFile,'READ')) then  /* if we cannot open the message read file for reading */
do
 return 0  /* return with an error */
end

ReadText = charin(ReadFile,1,chars(ReadFile))  /* copy the contents of the file into a buffer */
call FileClose ReadFile  /* close the read file */
Separator = ''||Global.CRLF  /* this signals the start of the body text */
parse var ReadText . (Separator) BodyText  /* get the message body text */
Subject = GetHeaderEntry(Global.MessHead,'SUBJECT:')  /* try to get the subject from the original message */
MessageID = GetHeaderEntry(Global.MessHead,'Message-ID:')  /* try to get the message ID from the original message */

if (Subject = '') then  /* if we have no subject */
do
 Subject = '(no subject)'  /* substitute this */
end

if (Reply) then  /* if we are replying */
do

 FromSender = GetHeaderEntry(Global.MessHead,'FROM:')  /* the original sender */
 Sender = GetHeaderEntry(Global.MessHead,'X-LISTNAME:')  /* get the list name, if any */

 if (Sender = '') then  /* if we have nothing */
 do

  Sender = GetHeaderEntry(Global.MessHead,'RESENT-FROM:')  /* get the last sender, if any (it could still be from a list) */

  if (Sender = '') then  /* if we have nothing */
  do

   Sender = FromSender  /* use this */

   if (Sender = '') then  /* if we have nothing, we have a dud message, but we'll continue to look for a sender: */
   do

    Sender = GetHeaderEntry(Global.MessHead,'SENDER:')  /* look for the actual sender */

    if (Sender = '') then  /* if we have nothing */
    do

     Sender = GetHeaderEntry(Global.MessHead,'APPARENTLY-FROM:')  /* look for the apparent sender */

     if (Sender = '') then  /* if we have nothing */
     do

      Sender = GetHeaderEntry(Global.MessHead,'FROM')  /* look for the envelope sender -- you never know */

      if (Sender >< '') then  /* if we have something this time */
      do
       Sender = word(Sender,1)  /* use just the first word */
      end

     end

    end

    FromSender = Sender  /* if we started without a sender, we had no real sender either, so we will use what we have now */

   end

  end

 end

 select  /* do one of the following */

  when (Switch = 'REPLYTOALL') then  /* if we are replying to everybody we can find */
  do

   SendTo = Sender||', '||FromSender  /* start with the sender's and the real sender's addresses */
   ReplyToAddress = GetHeaderEntry(Global.MessHead,'REPLY-TO:')  /* look for "Reply-To:" addresses */

   if (ReplyToAddress >< '') then  /* if we have something */
   do
    SendTo = SendTo||', '||ReplyToAddress  /* add it to the recipients */
   end

   ToAddress = GetHeaderEntry(Global.MessHead,'TO:')  /* look for "To:" addresses */

   if (ToAddress >< '') then  /* if we have something */
   do
    SendTo = SendTo||', '||ToAddress  /* add it to the recipient list */
   end

   CcAddress = GetHeaderEntry(Global.MessHead,'CC:')  /* look for "Cc:" addresses */

   if (CcAddress >< '') then  /* if we have something */
   do
    SendTo = SendTo||', '||CcAddress  /* add it to the recipients list */
   end

  end

  when (Switch = 'REPLYTOSENDER') then  /* if we are replying to the real sender */
  do
   SendTo = FromSender  /* use the real sender's address */
  end

  when (Switch = 'REPLYTORECIPIENTS') then  /* if we are replying to the recipients addresses */
  do

   SendTo = GetHeaderEntry(Global.MessHead,'TO:')  /* get the "To:" addresses */

   if (SendTo = '') then  /* if there is no address */
   do
    SendTo = GetHeaderEntry(Global.MessHead,'APPARENTLY-TO:')  /* look for an "Apparently-To:" address */
   end

   CcAddress = GetHeaderEntry(Global.MessHead,'CC:')  /* look for "Cc:" addresses */

   if (CcAddress >< '') then  /* if we have something */
   do
    SendTo = SendTo||', '||CcAddress  /* add it to the recipient list */
   end

  end

  otherwise  /* if we have none of the above special cases */
  do

   SendTo = GetHeaderEntry(Global.MessHead,'REPLY-TO:')  /* try to find a "Reply-To:" address */

   if (SendTo = '') then  /* if there is none */
   do
    SendTo = Sender  /* use the sender address */
   end

  end

 end

 if (SendTo = '') then  /* if we have no address */
 do
  call AddError 'Cannot find a reply address'  /* report */
  return 0  /* and return with an error */
 end

 OutText = 'To: '||SendTo||Global.CRLF  /* start with a "To:" line */

 if (MessageID <> '') then  /* if there is a message ID */
 do
  OutText = OutText||'In-Reply-To: '||MessageID||Global.CRLF  /* add the reply info line */
 end

 if (translate(word(Subject,1)) >< 'RE:') then  /* if the subject does not start with "Re:" */
 do
  Subject = 'Re: '||Subject  /* add it to the header */
 end

 IntroText = Global.ReplyText  /* use this for the intro text setup */

 BodyCopy = BodyText  /* copy the body text */
 BodyText = ''  /* start with nothing */

 do while (BodyCopy >< '')  /* while we have something in the body copy */

  parse var BodyCopy NextLine (Global.CRLF) BodyCopy  /* take the next line */

  if (Global.ReplyLineLength = 0) then  /* if we have no maximum line length */
  do
   BodyText = BodyText||'> '||NextLine||Global.CRLF  /* add the text to the new body text, preceded by a quotation sign and terminated by CRLF */
  end
  else  /* if we have a maximum line length */
  do
   BodyText = BodyText||WordWrap('> '||NextLine,(Global.ReplyLineLength),1,0,'> ')||Global.CRLF  /* word-wrap the line with quotation signs, etc.*/
  end

 end

end
else  /* if we are forwarding */
do

 OutText = 'From: "'||Global.Name||'" <'||Global.Address||'>'||Global.CRLF||'To: '||Global.CRLF  /* start with a prefilled "From:" line and an empty "To:" line */

 if (MessageID <> '') then  /* if there is a message ID */
 do
  OutText = OutText||'References: '||MessageID||Global.CRLF  /* add a references line */
 end

 if (translate(word(Subject,1)) >< 'FWD:') then  /* if the subject does not start with "Fwd:" */
 do
  Subject = 'Fwd: '||Subject  /* add it */
 end

 IntroText = Global.ForwardText  /* use this for the intro text setup */

end

OutText = OutText||'Subject: '||Subject||Global.EmptyLine  /* add the subject line to the output text followed by an empty line */
OutText = OutText||MakeIntroText(IntroText,Global.MessHead)  /* add the intro text, if any */
OutText = OutText||BodyText  /* add the body text */

OutFile = TempFileName('FORREP',OutDir)  /* create a unique temporary file name */
OutName = filespec('N',OutFile)  /* get the name part */

call syscreateobject 'CWMailFile',OutName,OutDir,,'FAIL'  /* create a CWMAILFile class object in the destination dir using the name part */
call HideObject OutFile  /* make the new file object invisible */

if (\FileOpen(OutFile,'WRITE')) then  /* if we cannot open the output file for writing */
do
 return 0  /* and return with an error */
end

call charout OutFile,OutText  /* write the output text to the new message file */
call FileClose OutFile  /* close the output file */

if (Reply) then  /* if we are replying */
do
 call DeleteDir AttDir  /* get rid of the attachments folder -- we need only the read file for a reply */
 call MessageSettings OutFile,'00000000','CHANGE'  /* make the file a new outgoing mail message file */
end
else  /* if we are forwarding */
do

 if (\AttDirLink(OutFile,AttDir)) then  /* if we cannot link the attachments dir to the new message */
 do
  return 0  /* return with an error */
 end

 call sysfiletree AttDir||'\*','Files.','FOS'  /* have we got anything in the attachments dir(s)? */
 call MessageSettings OutFile,'0000'||(Files.0 > 0)||'000','CHANGE'  /* make it a new outgoing mail message file and set the attachments flag if necessary */

end

if (\MessageContents(OutFile)) then  /* get the contents of the new file */
do
 return 0  /* quit */
end

NewName = MakeTitle(OutFile,0,0,0,0)  /* get a new title and insert no warnings */
call UnhideObject OutFile  /* make the file visible */

if (\syssetobjectdata(OutFile,'REXXMAILREFRESH=YES;TITLE='||NewName)) then  /* if we cannot rename the file */
do
 call AddError 'Cannot rewrite title of output file'  /* report; not fatal */
end

if (FileCheck(ReadFile)) then  /* if the read file still exists */
do
 call sysfiledelete ReadFile  /* get rid of the read file -- no check, it may already have been deleted by another process */
end

return 1  /* end of MakeForwardReplyMessage */

/**********************************************************************/
MakeHeaderLines: procedure expose Global.  /* gets header info from original message file and creates indented and folded header text block */
/**********************************************************************/

parse arg MessHead,KeywordList,Shorten  /* get the arguments */

Shorten = (Shorten = 1)  /* if the Shorten argument is 1, it is true */
Header. = ''  /* start with nothing */
HeadCount = 0  /* start counting header lines at 0 */

if (KeywordList = '') then  /* if we do not have a list of keywords */
do

 Outgoing = 1  /* the message is outgoing */

 do while (MessHead >< '')  /* as long as we have a bit of header left */
  parse var MessHead HeadLine (Global.CRLF) MessHead  /* get the next header line */
  HeadCount = HeadCount + 1  /* up the counter */
  Header.HeadCount = HeadLine  /* store the header line */
 end

end

else  /* if we have a list of keywords */

do

 Outgoing = 0  /* the message is incoming */

 do while (KeywordList >< '')  /* while we have something left in this variable */

  parse var KeywordList Keyword KeywordList  /* get the next word from the list of keywords */

  if (left(Keyword,1) = '*') then  /* if it starts with "*" */
  do
   Keyword = substr(Keyword,2)  /* use the rest */
   More = 1  /* we want more than the first occurrence */
  end
  else  /* if the word does not start with "*" */
  do
   More = 0  /* we don't want more */
  end

  HeadLine = GetHeaderEntry(MessHead,Keyword||':',More)  /* look for this entry in the message header */

  if (HeadLine >< '') then  /* if we have something */
  do
   HeadCount = HeadCount + 1  /* up the counter */
   Header.HeadCount = Keyword||': '||HeadLine  /* store the header line */
  end

 end

end

Reformats = 'FROM:',  /* reformat addresses */
            'TO:',  /* reformat addresses */
            'REPLY-TO:',  /* reformat addresses */
            'CC:',  /* reformat addresses */
            'BCC:',  /* reformat addresses */
            'APPARENTLY-FROM:',  /* reformat addresses */
            'SENDER:',  /* reformat addresses */
            'RESENT-FROM:',  /* reformat addresses */
            'RESENT-TO:',  /* reformat addresses */

DateLines = 'RECEIVED:',  /* reformat date if required */
            'SENT:'  /* reformat date if required */

AddDLines = 'DATE:',  /* reformat date if required and add original date line if UTC */
            'RESENT:'  /* reformat date if required and add original date line if UTC */

Indent = 0  /* start with an indent of 0 */

do Index = 1 to HeadCount  /* run through the header lines */
 Indent = min(32,max(length(word(Header.Index,1)),Indent))  /* adjust the indent value if needed, but not to more than 32 */
end

Indent = Indent + 2  /* add extra space after the colon */

NewHeader = ''  /* start with nothing */

do Index = 1 to HeadCount  /* run through the header lines again */

 HeadLine = Header.Index  /* get the header line */
 parse var HeadLine Keyword NextEntry  /* get the two components */

 if (right(Keyword,1) = ':') then  /* if it is a real keyword ending with a colon */
 do

  if (wordpos(translate(Keyword),Reformats) > 0) then  /* if this is one we want to reformat */
  do

   NextEntry = AddressFormat(NextEntry,Indent,Outgoing)  /* get a formatted string of addresses, indented, checked if outgoing */

   if (Shorten) then  /* if we want long lists eliminiated */
   do

    ThisEntry = NextEntry  /* copy the entry */
    Counter = 0  /* start a counter */

    do while (ThisEntry >< '')  /* run through the entry */
     Counter = Counter + 1  /* up the counter */
     parse var ThisEntry . (Global.CRLF) ThisEntry  /* remove the bit before the next CRLF */
    end

    if (Counter > Global.MaxAddresses) then  /* if we have more addresses than we want to see */
    do
     NextEntry = '[list of '||Counter||' recipients]'  /* use this instead */
    end

   end

  end

  else  /* if this is not one we want to reformat */

  do

   if (wordpos(translate(Keyword),DateLines||' '||AddDLines) > 0) then  /* if it is one with dates in it */
   do

    EntryLines = NextEntry  /* copy the entry's lines */
    NextEntry = ''  /* start all over with nothing */
    AddLine = ''  /* nothing to add yet */

    do while (Entrylines >< '')  /* go on while we have stuff left */

     Receiver = ''  /* we have no receiver yet (we may be getting a "Received:" line) */

     parse var EntryLines NextLine (Global.CRLF) EntryLines  /* get the first entry */

     if (translate(Keyword) = 'RECEIVED:') then  /* if it is this one */
     do

      LastSemiPos = lastpos(';',NextLine)  /* the last semicolon position but one */

      parse var NextLine RouteInfo =(LastSemiPos) +1 NextLine  /* get the different bits */

      do while ((RouteInfo >< '') & (Receiver = ''))  /* as long as we have something left and no result */

       parse var RouteInfo NextWord RouteInfo  /* get the next word */

       if (translate(NextWord) = 'BY') then  /* if it is this */
       do
        Receiver = word(RouteInfo,1)  /* the receiver must be the second word, so store it till after we sort the date, if necessary */
       end

      end

     end

     if (Receiver >< '') then  /* if we have a receiver */
     do
      DateTimeType = word(Global.DateTimeHeader,1)  /* use just the first word of the date/time stamp type string */
     end
     else  /* if we do not have a receiver */
     do
      DateTimeType = Global.DateTimeHeader  /* use the whole date/time stamp type string */
     end

     NextLine = DateTimeDisplay(NextLine,DateTimeType)  /* get a formatted date/time stamp */
     
     if (Receiver >< '') then  /* if we have a receiver */
     do
      NextLine = NextLine||' by '||Receiver  /* add this */
     end

     if (NextEntry >< '') then  /* if this is not the first line */
     do
      NextLine = Global.CRLF||copies(' ',Indent)||NextLine  /* add a new line and an indent */
     end

     NextEntry = NextEntry||NextLine  /* add the result to what we already have */

    end

   end

   else  /* if it is not a date entry */

   do
    NextEntry = WordWrap(NextEntry,(76 - Indent),Indent,1)  /* word-wrap it at 76 - Indent chars, indent, no word breaks */
   end

  end

  IndentTail = Indent - length(Keyword)  /* the bit of indent space we still have to add */

  if (IndentTail < 0) then  /* if we have a negative indent tail, i.e. the keyword is longer than the maximum indent */
  do
   IndentTail = 2  /* use a double space */
  end

  NewHeader = NewHeader||Keyword||copies(' ',IndentTail)||NextEntry||Global.CRLF  /* add the keyword and entry to the new header */

 end

 else  /* if it is not a real keyword, it must be already marked as an error */

 do
  NewHeader = NewHeader||HeadLine||Global.CRLF  /* add the original line to the new header */
 end

end

return NewHeader  /* end of MakeHeaderLines */

/**************************************************************************/
MakeIntroText: procedure expose Global.  /* prepares intro text for reply/forwarded messages */
/**************************************************************************/

parse arg TextLine,Header  /* get the arguments */

NewText = ''  /* nothing yet */

do while (TextLine >< '')  /* as long as we have a reply formula */

 parse var TextLine NextBit '%' TextLine  /* look for the next bit and anything following a placeholder */
 NewText = NewText||NextBit  /* start by adding the first bit */

 if (TextLine >< '') then  /* if we found more, it must include a marker */
 do

  parse var TextLine Marker +1 TextLine  /* get the marker */

  select  /* do one of the following */

   when (Marker = 'D') then  /* if it is this */
   do

    AddText = GetHeaderEntry(Header,'DATE:')  /* try to get a message date */

    if (AddText >< '') then  /* if we have something */
    do
     AddText = DateTimeDisplay(AddText,Global.DateTimeBody)  /* turn it into the type of date display we need */
    end
    else  /* if we have no date */
    do
     AddText = 'an unknown date'  /* use this */
    end

   end

   when (Marker = 'F') then  /* if it is this */
   do

    AddText = GetHeaderEntry(Header,'FROM:')  /* try to get a sender */

    if (AddText >< '') then  /* if we have something */
    do
     AddText = AddressFormat(AddText,0,0)  /* reformat the address, no indent, no check */
    end
    else  /* if we have no sender */
    do
     AddText = 'an unknown sender'  /* use this */
    end

   end

   when (Marker = 'N') then  /* if it is this */
   do
    AddText = Global.CRLF  /* add a new line */
   end

   when (Marker = 'S') then  /* if it is this */
   do

    AddText = GetHeaderEntry(Header,'SUBJECT:')  /* try to get the subject */

    if (AddText = '') then  /* if we have nothing */
    do
     AddText = 'an unknown subject'  /* use this */
    end

   end

   when (Marker = 'T') then  /* if it is this */
   do

    AddText = GetHeaderEntry(Header,'TO:')  /* try to get the recipient(s) */

    if (AddText >< '') then  /* if we have something */
    do
     AddText = AddressFormat(AddText,0,0)  /* reformat the address, no indent, no check */
    end
    else  /* if we have no recipients */
    do
     AddText = 'an unknown recipient'  /* use this */
    end

   end

   otherwise  /* if none of the above */
   do
    call AddError 'Invalid macro %'||Marker||' in setup string'  /* report (not fatal) */
   end

  end

  NewText = NewText||AddText  /* add the result */

 end

end

if (NewText >< '') then  /* if we have a result */
do
 NewText = NewText||Global.EmptyLine  /* add an empty line */
end

return NewText  /* end of MakeIntroText */

/**********************************************************************/
MakeNewMessage: procedure expose Global.  /* generate a new message to a recipient */
/**********************************************************************/

parse arg Recipients,Switch  /* get the arguments */

if (pos('HERE',Switch) > 0) then  /* if we want the new message in the current dir */
do
 MessDir = strip(directory(),'T','\')  /* use the current dir name without a trailing backslash (we need that bit if we are in a root dir) */
end
else  /* if we are to put the message in the default dir */
do
 MessDir = Global.OutDir  /* we will create the new message in the 'Out' folder */
end

GotSubject = 0  /* no subject line yet */
GotAttach = 0  /* no attachments yet */
BodyPart = ''  /* no body part yet */
MessFile = TempFileName('NEW',MessDir)  /* get a unique temporary file name */
MessName = filespec('N',MessFile)  /* get the name part */
call syscreateobject 'CWMailFile',MessName,MessDir,,'FAIL'  /* create a CWMAILFile class object in the destination dir using the name part */
call HideObject MessFile  /* make the file invisible */

if (\FileOpen(MessFile,'WRITE')) then  /* if we cannot open the message file for writing */
do
 call sysdestroyobject MessFile  /* get rid of the useless object */
 return 0  /* and return with an error */
end

call FileClose MessFile  /* close the Mess file */

AttDir = AttDirCreate(MessFile)  /* create an attachments dir and link it to the message file */

if (Recipients >< '') then  /* if we have recipients */
do

 if ((left(Recipients,2) = '\"') & (right(Recipients,2) = '\"')) then  /* if the parameters start and end with this (Mozilla URLs!) */
 do
  Recipients = substr(Recipients,3,length(Recipients) - 4)  /* get rid of them */
 end
 else  /* if we do not have a Mozilla URL */
 do
  Recipients = strip(Recipients,'B','"')  /* remove any quotation marks */
 end

 Recipients = strip(Recipients,'B',' ')  /* remove any excess whitespace */

 if (translate(left(Recipients,7)) = 'MAILTO:') then  /* if the upper case recipients string starts with this */
 do
  parse var Recipients . ':' Recipients  /* get rid of that bit */
 end

 parse var Recipients Recipients '?' Parameters  /* look for parameters */
 Recipients = translate(Recipients,',',';')  /* convert any semicolons in the recipients list to commas */
 MessCont = 'To: '||Recipients||Global.CRLF  /* add the recipients list to the top of the message */

 do while (Parameters >< '')  /* as long as we have parameters left */

  parse var Parameters ParamName '=' ParamValue '&' Parameters  /* get the next bit */
  ParamName = translate(ParamName)  /* make the parameter name upper case */

  select  /* do one of the following */

   when (ParamName = 'FROM') then  /* if it is this one */
   do
    MessCont = MessCont||'From: '||ParamValue||Global.CRLF  /* add this line to the message header */
   end

   when (ParamName = 'REPLY-TO') then  /* if it is this one */
   do
    MessCont = MessCont||'Reply-To: '||ParamValue||Global.CRLF  /* add this line to the message header */
   end

   when (ParamName = 'SUBJECT') then  /* if it is this one */
   do
    MessCont = MessCont||'Subject: '||ParamValue||Global.CRLF  /* add this line to the message header */
    GotSubject = 1  /* we have a subject line */
   end

   when (ParamName = 'CC') then  /* if it is this one */
   do
    MessCont = MessCont||'Cc: '||translate(ParamValue,',',';')||Global.CRLF  /* convert any semicolons to commas and add this line to the message header */
   end

   when (ParamName = 'BCC') then  /* if it is this one */
   do
    MessCont = MessCont||'Bcc: '||translate(ParamValue,',',';')||Global.CRLF  /* convert any semicolons to commas and add this line to the message header */
   end

   when (ParamName = 'BODY') then  /* if it is this one */
   do
    BodyPart = ParamValue  /* make it the body text */
   end

   when (ParamName = 'ATTACH') then  /* if it is this one */
   do

    do while (ParamValue >< '')  /* as long as we have files left */

     parse var ParamValue AttachFile ';' ParamValue  /* get the next file name in the list */

     call sysfiletree AttachFile,'Files.','FO'  /* look for the file spec */

     if (Files.0 = 0) then  /* if we do not find it */
     do
      call sysfiletree directory()||'\'||AttachFile,'Files.','FO'  /* look for the file spec in the current dir */
     end

     if (Files.0 = 0) then  /* if we still find nothing */
     do
      call AddError 'File not found "'||AttachFile||'"'  /* report */
     end

     do Index = 1 to Files.0  /* take each of the files we found */

      if (syscopyobject(Files.Index,AttDir)) then  /* if we can copy the attachment file to the attachment dir */
      do
       GotAttach = 1  /* set the attachment flag */
      end
      else  /* if we cannot copy it */
      do
       call AddError 'Cannot copy "'||Files.Index||'" to "'||AttDir||'"'  /* report */
      end

     end

    end

   end

   otherwise  /* if it is none of the above */
   do
    call AddError 'Unknown mailto: parameter '||ParamName||'='||ParamValue  /* report */
   end

  end

 end

end

else  /* if we have no recipients */

do
 MessCont = 'To: '||Global.CRLF  /* start with this */
end

if (\GotSubject) then  /* if we have no subject line */
do
 MessCont = MessCont||'Subject: '||Global.CRLF  /* add an empty one */
end

if (BodyPart >< '') then  /* if we have a body part */
do
 MessCont = MessCont||Global.CRLF||BodyPart  /* add it to the header, with an empty line in between */
end

if (\FileOpen(MessFile,'WRITE')) then  /* if we cannot open the message file for writing */
do
 call sysdestroyobject MessFile  /* get rid of the useless object */
 return 0  /* and return with an error */
end

call charout MessFile,MessCont  /* write the message contents to the message file */
call FileClose MessFile  /* close the Mess file */
call MessageSettings MessFile,'0000'||GotAttach||'000','CHANGE'  /* make it a new outgoing mail message file and set the attachments flag if necessary */

if (\MessageContents(MessFile)) then  /* if we cannot get the new message contents */
do
 return 0  /* quit */
end

Title = 'New Message'  /* start with this */
 
if (Recipients >< '') then  /* if we have recipients */
do
 parse var Recipients Recipients ',' .  /* use only the first one */
 Recipients = CheckCommLine(Recipients)  /* get rid of awkward characters */
 Title = Title||' to '||Recipients  /* use this as the new message object title */
end

call UnhideObject MessFile  /* make the file visible */
call syssetobjectdata MessFile,'REXXMAILREFRESH=YES;TITLE='||Title  /* set the title */

if (pos('OPEN',Switch) > 0) then  /* if we want the message opened on the desktop for editing */
do
 Global.ProcFile = MessDir||'\'||Title  /* specify the file to use */
 call EditMessage  /* open the message on the desktop */
end

if (pos('SEND',Switch) > 0) then  /* if we want the message sent */
do
 Global.ProcFile = MessDir||'\'||Title  /* specify the file to use */
 call MessageSettings Global.ProcFile,'*1******','CHANGE'  /* make the message ready to send */
 return SendMessage()  /* try to send the message and return with the result */
end

return 1  /* end of MakeNewMessage */

/**********************************************************************/
MakeSendFile: procedure expose Global.  /* prepares the actual file to send */
/**********************************************************************/

parse arg MessFile  /* get the argument */

if (\FileOpen(MessFile)) then  /* if we cannot open the message file */
do
 return ''  /* return with no result */
end

MessCont = charin(MessFile,1,chars(MessFile))  /* read the contents of the message file into MessCont */

call FileClose MessFile  /* close the message file */

parse var MessCont MessHead (Global.EmptyLine) MessBody  /* split the message content into a header and a body */

MessHead = translate(MessHead,d2c(32),d2c(9))  /* convert header TABs to spaces */
MessHead = translate(MessHead,Global.ESCtoASC,xrange('00'x,'1F'x))  /* convert header escape codes to ASCII */
MessBody = translate(MessBody,Global.ESCtoASC,xrange('00'x,'1F'x))  /* convert body text escape codes to ASCII */

ASCIIMessage = MessageSettings(MessFile,'0**1****','MATCH')  /* if we find an ASCII EA, this message should be sent as ASCII */

if (ASCIIMessage) then  /* if we want ASCII text */
do
 MessHead = translate(MessHead,Global.850toASC,xrange('80'x,'FF'x))  /* convert the header to ASCII*/
 MessBody = translate(MessBody,Global.850toASC,xrange('80'x,'FF'x))  /* convert the body text to ASCII */
 MessBody = translate(MessBody,d2c(32),d2c(9))  /* convert body TABs to spaces */
end
else  /* if we want quoted-printable ISO 8859-1 text */
do
 MessHead = translate(MessHead,Global.850toISO,xrange('80'x,'FF'x))  /* convert the header to ISO 8859-1 */
 MessBody = translate(MessBody,Global.850toISO,xrange('80'x,'FF'x))  /* convert the body text to ISO 8859-1 */
end

NewHead = ''  /* start with an empty new header */

do until (MessHead = '')  /* go on until we run out of header */

 MIMEPrevWord = 0  /* there is no previous word, so it cannot have been a MIME word */
 parse var MessHead NextLine (Global.CRLF) MessHead  /* get a line from the header */

 if (left(NextLine,1) = ' ') then  /* if the line starts with a space */
 do

  Indent = wordindex(NextLine,1) - 1  /* the indent goes up to where the first word starts, minus one space because the next word also gets a space in front of it */

  if (Indent < 0) then  /* if the indent is negative */
  do
   Indent = 1  /* use 1 */
  end

  parse var NextLine NewLine =(Indent) NextLine  /* start the new line with the indent */

 end

 else  /* if the line does not start with a space */

 do
  parse var NextLine NewLine NextLine  /* start the new line with the first word, which must be a keyword */
 end

 do while (NextLine >< '')  /* go on until we run out of line */

  parse var NextLine NextWord NextLine  /* get the next word in the line */

  if (verify(NextWord,xrange(x2c(00),x2c(7F)),'NOMATCH') > 0) then  /* if the current word contains any high-bit characters */

  do

   NewWord = ''  /* start with nothing */
   ISOStart = '=?iso-8859-1?Q?'  /* define the start of an ISO string */
   ISOEnd = '?='  /* define the end of an ISO string */

   if (left(NextWord,1) = '"') then  /* if the word starts with a quotation mark */
   do
    NextWord = substr(NextWord,2)  /* cut it off */
    ISOStart = '"'||ISOStart  /* add the quotation mark to the start of the ISO start code */
   end

   if (right(NextWord,1) = '"') then  /* if the word ends with a quotation mark */
   do
    NextWord = left(NextWord,length(NextWord) - 1)  /* cut it off */
    ISOEnd = ISOEnd||'"'  /* add the quotation mark to the end of the ISO end code */
   end

   do Index = 1 to length(NextWord)  /* run through the message text */

    NextChar = substr(NextWord,Index,1)  /* take each character */

    if ((c2d(NextChar) > 126) | (pos(NextChar,'_=') > 0)) then  /* if it's not in the OK range, or if we are in a MIME word and it is an underscore or equal sign */
    do
     NewWord = NewWord||'='||c2x(NextChar)  /* substitute an equals sign followed by the character's hex code */
    end
    else  /* if the character is in the OK range */
    do
     NewWord = NewWord||NextChar  /* simply add it to the output buffer */
    end

   end

   if (MIMEPrevWord) then  /* if the previous word was also a MIME word */
   do
    NewWord = '_'||NewWord  /* stick on an underscore */
   end
   else  /* if the previous word was not a MIME word */
   do
    NewWord = ' '||ISOStart||NewWord  /* stick on a space and an ISO start string */
    MIMEPrevWord = 1  /* set the MIME flag */
   end

   if (NextLine = '') then  /* if this is the last word */
   do
    NewWord = NewWord||ISOEnd  /* add the end code */
   end

  end

  else  /* if the word does not contain any high-bit characters */

  do

   if (MIMEPrevWord) then  /* if the previous word was a MIME word */
   do
    NewWord = ISOEnd||' '||NextWord  /* stick on an ISO end string and a space */
    MIMEPrevWord = 0  /* reset the MIME flag */
   end
   else  /* if the previous word was not a MIME word */
   do
    NewWord = ' '||NextWord  /* just stick on a space */
   end

  end

  NewLine = NewLine||NewWord  /* add it to the rest of the line */

 end

 NewHead = NewHead||NewLine||Global.CRLF  /* add the new line to the header */

end

parse var Global.Address . '@' Domain '>'  /* get the (real) sender's domain name */
HostName = value('HOSTNAME',,'OS2Environment')  /* try to get a host name from the OS/2 environment */

if (HostName >< '') then  /* if we have a host name */
do
 HostName = HostName||'.'||Domain  /* add the domain name to it */
end
else  /* if we have no host name */
do
 HostName = Domain  /* just use the domain name */
end

UniqueStr = date('S')||'.'||translate(left(time('L'),11),'.',':')||'.'||right(random(999),3,'0')  /* a unique string */

NewHead = NewHead||'Date: '||DateTimeRFC()||Global.CRLF||,  /* add a date line to the new header */
         'Message-Id: <'||UniqueStr||'@'||HostName||'>'||Global.CRLF  /* and a message ID line */

GotAttach = 0  /* no attachments yet */

AttDir = AttDirGet(MessFile,0)  /* get the attachments directory name, if any (do not create it) */

if (AttDir >< '') then  /* if there is an attachments storage directory */
do
 call RunCommand Global.RunAttachOut,,AttDir  /* see if we have to run a command in the attachments dir */
 call sysfiletree AttDir||'\*','Attach.','FOS'  /* look again for remaining attachment files in the attachments directory and any subdirectories */
 GotAttach = (Attach.0 > 0)  /* set a flag if we've got attachments */
end

if ((GotAttach) | (\ASCIIMessage)) then  /* if we've got attachments, or we want quoted-printable text */
do
 NewHead = NewHead||'Mime-version: 1.0'||Global.CRLF  /* add the MIME version header line */
end

if (GotAttach) then  /* if we've got MIME attachments */
do

 MIMEBound = '=='||UniqueStr||'=='  /* the MIME boundary to use */

 if (Attach.0 > 1) then  /* if we have more than one attachment */
 do
  Plural = 's'  /* we need a plural "s" */
 end
 else  /*if there is only one attachment */
 do
  Plural = ''  /* we don't need the "s" */
 end

 NewHead = NewHead||'Content-type: multipart/mixed; boundary="'||MIMEBound||'"'||Global.CRLF  /* add the MIME boundary header line */
 NewHead = NewHead||Global.CRLF  /* add an empty line */
 NewHead = NewHead||'This is a multi-part message in MIME format.'||Global.CRLF  /* add a plain text message line */
 NewHead = NewHead||'In addition to the message below it contains '||,  /* add an info line */
           Attach.0||' b64-encoded attachment'||Plural||'.'||Global.CRLF  /* rest of the info line */
 NewHead = NewHead||Global.CRLF||'--'||MIMEBound||Global.CRLF  /* add an empty line followed by a MIME boundary */

end

if (ASCIIMessage) then  /* if we want ASCII text */
do

 if (GotAttach) then  /* if this is a MIME-enhanced message */
 do
  NewHead = NewHead||'Content-Type: text/plain; charset="us-ascii"'||Global.CRLF  /* the first type line */
  NewHead = NewHead||'Content-Transfer-Encoding: 7bit'||Global.CRLF  /* another one */
 end

end

else  /* if we want quoted-printable text */

do
 NewHead = NewHead||'Content-Type: text/plain; charset="iso-8859-1"'||Global.CRLF  /* the first type line */
 NewHead = NewHead||'Content-Transfer-Encoding: quoted-printable'||Global.CRLF  /* another one */
end

NewBody = ''  /* start with nothing */

do while (length(MessBody) > 0)  /* go through the contents of the body text */

 parse var MessBody NextLine (Global.CRLF) MessBody  /* and each time, take the next line from the body text */

 if (ASCIIMessage) then  /* if we want ASCII text */
 do
  NewBody = NewBody||WordWrap(NextLine,Global.ASCIILineLength,,0)||Global.CRLF  /* add the word-wrapped (no indent, word breaks are allowed) line to the body text */
 end

 else  /* if we don't want ASCII text, we get quoted-printable */

 do
  OutLine = ''  /* start with an empty line */

  do while (length(NextLine) > 0)  /* as long as we have characters left in the line */
   parse var NextLine NextChar 2 NextLine  /* get the next character from the line */

   if ((c2d(NextChar) > 126) | (NextChar = '=')) then  /* if it's not in the OK range, or if it is an equal sign */
   do
    NextChar = '='||c2x(NextChar)  /* substitute an equals sign followed by the character's hex code */
   end

   if ((length(OutLine||NextChar) > 76) |,  /* if this line has more than 76 characters (the maximum according to the RFC spec) */
      ((length(OutLine||NextChar) = 76) & (length(NextLine) > 0))) then  /* or if we will end up with more than 76 characters */
   do
    NewBody = NewBody||OutLine||'='||Global.CRLF  /* add the line to the body text, end the line with = and start a new line */
    OutLine = ''  /* we now have an empty line */
   end

   OutLine = OutLine||NextChar  /* add the new character to the new line */

  end

  NewBody = NewBody||OutLine||Global.CRLF  /* add to the body, with a CRLF */

 end

end

SendFile = TempFileName('SEND')  /* prepare a unique file name */
NamePart = filespec('N',SendFile)  /* the name part of the send file */

call syscreateobject 'CWMailFile',NamePart,Global.TempDir,,'FAIL'  /* create a CWMAILFile class object in the temp dir using the name part */

if (\FileOpen(SendFile,'WRITE')) then  /* if we cannot open the temporary file for writing */
do
 return ''  /* no result */
end

call charout SendFile,NewHead||Global.CRLF||NewBody  /* write the new message header and body to the sendable file, separated by an empty line */
Success = 1  /* assume success */

if (GotAttach) then  /* if we've got one or more MIME attachments */
do

 Encoder = Global.ProgDir||'\mmencode.exe'  /* the encoder file spec */

 if (\FileCheck(Encoder,1)) then  /* if we can't find the encoder */
 do
  Success = 0  /* no success */
 end

 else  /* if we find the decoder */

 do Index = 1 to Attach.0  /* take each of the attachment files */

  AttachName = filespec('N',Attach.Index)  /* get the name of the attachment file */
  MIMEType = ''  /* no MIME type yet */
  DotPos = lastpos('.',AttachName)  /* look for the last dos position in the file name */

  if (DotPos > 0) then  /* if we have a last dot */
  do
   AttachExt = substr(AttachName,DotPos + 1)  /* get the extension */
   MIMEType = GetFileEntry(Global.MIMETypes,AttachExt)  /* try to match the extension with a MIME type */
  end

  if (MIMEType = '') then  /* if we cannot match it */
  do
   MIMEType = 'application/octet-stream'  /* use this */
  end

  call charout SendFile,Global.EmptyLine||,  /* start with an empty line */
                        '--'||MIMEBound||Global.CRLF||,  /* followed by a MIME boundary */
                        'Content-Type: '||MIMEType||'; name="'||AttachName||'"'||Global.CRLF||,  /* add a content line */
                        'Content-Transfer-Encoding: base64'||Global.CRLF||,  /* add another one */
                        'Content-Disposition: attachment; filename="'||AttachName||'"'||Global.EmptyLine  /* and another one, followed bij an empty line */

  call FileClose SendFile  /* close the new message file to allow MMENCODE.EXE access */

  Encoder||' < "'||Attach.Index||'" >> "'||SendFile||'"'  /* convert the attachment file to B64 code and add the result to SendFile [EXTERNAL] */

  if (RC >< 0) then  /* if we get an error */
  do
   Success = 0  /* we have no success overall */
  end

  call FileOpen SendFile,'WRITE'  /* reopen the new message file for more writing */

 end

 call charout SendFile,Global.EmptyLine||'--'||MIMEBound||'--'||Global.CRLF  /* write an empty line followed by a MIME end boundary and a new line */

end

call FileClose SendFile  /* close the new message file */

if (\Success) then  /* if something went wrong */
do
 call sysfiledelete SendFile  /* get rid of the temp file -- no check, it may already have been deleted by another process */
 SendFile = ''  /* return nothing */
end

return SendFile  /* end of MakeSendFile */

/**********************************************************************/
MakeTitle: procedure expose Global.  /* returns new title for mail file */
/**********************************************************************/

parse arg MessFile,Warnings,KeepAdds,Addresses,Incoming  /* get the parameters */

Warnings = (Warnings = 1)  /* 1 = true */
KeepAdds = (KeepAdds = 1)  /* 1 = true */
Addresses = (Addresses = 1)  /* 1 = true */
Incoming = (Incoming =1)  /* 1 = true */

if (Addresses) then  /* if we want just the recipients' addresses */
do
 Outgoing = 0  /* no outgoing flag, so we won't get unnecessary checks later on */
end
else  /* if we want more than just the recipients' addresses */
do

 OutGoing = (MessageSettings(MessFile,'0*******','MATCH'))  /* look for an outgoing message type flag */

 if (OutGoing & Warnings) then  /* if the message is outgoing and we want warnings */
 do
  WarnStart = Global.Warning||' '  /* start of a warning line */
  WarnEnd = ' '  /* end of a warning line */
 end
 else  /* if the message is incoming, or we do not want warnings */
 do
  WarnStart = '('  /* start of a warning line */
  WarnEnd = ')'  /* end of a warning line */
 end

 ObjectFrom = ''  /* start with nothing */
 ObjectTo = ''  /* start with nothing */
 ObjectCc = ''  /* start with nothing */
 ObjectBcc = ''  /* start with nothing */

 TitleDate = GetHeaderEntry(Global.MessHead,'DATE:')  /* get the date */

 if (TitleDate = '') then  /* if we have no date */
 do

  LastTime = sysgetfiledatetime(MessFile,'A')  /* get the last access time */

  parse var LastTime LastYear '-' LastMonth '-' LastDay ' ' LastTime  /* parse the date/time components */

  MonthName = word('Jan Feb Mar Apr May Jun Jul Aug Sep Oct Nov Dec',strip(LastMonth,'L','0'))  /* get the corresponding month abbreviation */
  TitleDate = left(date('W'),3)||', '||LastDay||' '||MonthName||' '||LastYear||' '||LastTime||' '||GetTimeZone(1)  /* the d/t stamp */

 end

 ObjectDate = DateTimeDisplay(TitleDate,'UTCISO')  /* use a simple ISO UTC date/time string for the object date */
 TitleDate = DateTimeDisplay(TitleDate,Global.DateTimeTitle)  /* rewrite the title date/time stamp if necessary */

 TitleFrom = GetHeaderEntry(Global.MessHead,'FROM:')  /* get the sender */

 if (TitleFrom = '') then  /* if we have nothing */
 do

  TitleFrom = GetHeaderEntry(Global.MessHead,'APPARENTLY-FROM:')  /* get the sender this way */

  if (TitleFrom = '') then  /* if we have no sender */
  do

   TitleFrom = GetHeaderEntry(Global.MessHead,'SENDER:')  /* get the sender this way */

   if (TitleFrom = '') then  /* if we have no sender */
   do
    TitleFrom = word(GetHeaderEntry(Global.MessHead,'FROM'),1)  /* get the sender this way */
   end

  end

 end

 if ((TitleFrom = '') | (TitleFrom = d2c(9))) then  /* if we still have no sender */
 do
  TitleFrom = WarnStart||'NO CONTENT'||WarnEnd  /* use this */
 end

 else  /* if we have something */

 do

  TitleFrom = AddressFormat(TitleFrom,0,(OutGoing & Warnings))  /* get a reformatted address string, no indent, check if outgoing and we want warnings */

  if ((Warnings) & (pos(Global.CRLF,TitleFrom) > 0)) then  /* if it contains more than one address and we want warnings */
  do
   TitleFrom = WarnStart||'MULTIPLE SENDERS'||WarnEnd  /* insert a warning */
  end

  else  /* if there is only one address */

  do

   if ((Warnings) & (pos(Global.Warning,TitleFrom) > 0)) then  /* if it contains a marker */
   do
    TitleFrom = WarnStart||'ERROR'||WarnEnd  /* insert a warning */
   end
   else  /* if there is no marker, or we do not want warnings */
   do
    ObjectFrom = left(GetAddressList(TitleFrom,Global.SortAddress),64)  /* extract a list of names according to the Global.SortAddress setting, and truncate the string for folder sorting use */
    TitleFrom = GetAddressList(TitleFrom,Global.TitleAddress)  /* extract a list of names according to the Global.TitleAddress setting */
   end

  end

 end

end

TitleTo = GetHeaderEntry(Global.MessHead,'TO:')  /* get the recipients */

if (TitleTo >< '') then  /* if we have one or more recipients */
do

 TitleTo = AddressFormat(TitleTo,0,(OutGoing & Warnings))  /* get a reformatted address string, no indent, check if outgoing and we want warnings */

 if ((Warnings) & (pos(Global.Warning,TitleTo) > 0)) then  /* if it contains a marker and we want warnings */
 do
  TitleTo = WarnStart||'ERROR'||WarnEnd  /* insert a warning */
 end
 else  /* if there is no marker, or we do not want warnings */
 do
  ObjectTo = left(GetAddressList(TitleTo,Global.SortAddress),64)  /* extract a list of names according to the Global.SortAddress setting, and truncate the string for folder sorting use */
  TitleTo = GetAddressList(TitleTo,Global.TitleAddress)  /* extract a list of names according to the Global.TitleAddress setting */
 end

end

if ((TitleTo = '') & (\Addresses)) then  /* if we still have no recipient, and we want more than just the addresses */
do
 TitleTo = WarnStart||'NO CONTENT'||WarnEnd  /* use this */
end

TitleCc = GetHeaderEntry(Global.MessHead,'CC:')  /* look for CC addresses */

if (TitleCc >< '') then  /* if CC addresses were found */
do

 TitleCc = AddressFormat(TitleCc,0,(OutGoing & Warnings))  /* get a reformatted address string, no indent, check if outgoing and we want warnings */

 if ((Warnings) & (pos(Global.Warning,TitleCc) > 0)) then  /* if it contains a marker and we want warnings */
 do
  TitleCc = WarnStart||'ERROR'||WarnEnd  /* create a warning line */
 end
 else  /* if there is no marker */
 do
  ObjectCc = left(GetAddressList(TitleCc,Global.SortAddress),64)  /* extract a list of names according to the Global.SortAddress setting, and truncate the string for folder sorting use */
  TitleCc = GetAddressList(TitleCc,Global.TitleAddress)  /* extract a list of names according to the Global.TitleAddress setting */
 end

end

TitleBcc = GetHeaderEntry(Global.MessHead,'BCC:')  /* look for Bcc addresses */

if (TitleBcc >< '') then  /* if BCC addresses were found */
do

 TitleBcc = AddressFormat(TitleBcc,0,(OutGoing & Warnings))  /* get a reformatted address string, no indent, check if outgoing and we want warnings */

 if ((Warnings) & (pos(Global.Warning,TitleBcc) > 0)) then  /* if it contains a marker and we want warnings */
 do
  AddLine = WarnStart||'ERROR'||WarnEnd  /* add a warning line */
 end
 else  /* if there is no marker */
 do
  ObjectBcc = left(GetAddressList(TitleBcc,Global.SortAddress),64)  /* extract a list of names according to the Global.SortAddress setting, and truncate the string for folder sorting use */
  TitleBcc = GetAddressList(TitleBcc,Global.TitleAddress)  /* extract a list of names according to the Global.TitleAddress setting */
 end

end

if (\Addresses) then  /* if we want more than just the addresses */
do

 TitleSubject = GetHeaderEntry(Global.MessHead,'SUBJECT:')  /* get the subject */

 if (TitleSubject = '') then  /* if we have no subject */
 do
  TitleSubject = WarnStart||'NO CONTENT'||WarnEnd  /* use this */
  ObjectSubject = ''  /* nothing */
 end
 else  /* if we have something */
 do
  ObjectSubject = left(TitleSubject,64)  /* get the object data subject */
 end

 Settings = 'REXXMAILDATE='||strip(ObjectDate)||';'||,  /* set the object date */
            'REXXMAILFROM='||strip(ObjectFrom)||';'||,  /* set the object sender */
            'REXXMAILTO='||strip(ObjectTo)||';'||,  /* set the object recipients */
            'REXXMAILCC='||strip(ObjectCc)||';'||,  /* set the object recipients */
            'REXXMAILBCC='||strip(ObjectBcc)||';'||,  /* set the object recipients */
            'REXXMAILSUBJECT='||strip(ObjectSubject)  /* set the object subject */
 call syssetobjectdata MessFile,Settings  /* set the object data strings */

end

TitleCount = 0  /* start at 0 */
Title. = ''  /* start with nothing */

if (Addresses) then  /* if we want just the recipients' addresses */
do
 TitleWords = 'TO CC BCC'  /* look for these */
end
else  /* if we want more than just the recipients' addresses */
do

 if (Incoming) then  /* if the message is incoming */
 do
  TitleWords = translate(Global.ObjectTitleIn)  /* copy the contents of the incoming global object title words string setting, upper case */
 end
 else  /* id the message is outgoing */
 do
  TitleWords = translate(Global.ObjectTitleOut)  /* copy the contents of the outgoing global object title words string setting, upper case */
 end

end

if (TitleWords = '') then  /* if we have no title words string setting */
do
 TitleCount = 1  /* up the counter */
 Title.TitleCount = DateTimeSys(1)  /* the file title to use is a date/time stamp with the long time */
end
else  /* if we have a title words string */
do

 TitleWordsCopy = TitleWords  /* copy the title words string */

 do while (TitleWordsCopy >< '')  /* as long as we have object title words left */

  TitleCount = TitleCount + 1  /* up the title lines counter */
  parse var TitleWordsCopy NextWord TitleWordsCopy  /* get the next word */

  select  /* do one of the following */

   when (NextWord = 'DATE') then  /* if the next word is 'DATE' */
   do

    TitleDate = CheckCommLine(TitleDate)  /* get rid of problem characters */

    if (Global.TitleKeywords) then  /* if we want title headers */
    do
     TitleDate = 'Date: '||TitleDate  /* add the header */
    end

    Title.TitleCount = TitleDate  /* use the title date */

   end

   when ((NextWord = 'FROM') | (NextWord = 'SENDER')) then  /* if the next word is 'FROM' or 'SENDER' */
   do

    TitleFrom = CheckCommLine(TitleFrom)  /* get rid of problem characters */

    if (Global.TitleKeywords) then  /* if we want title headers */
    do
     TitleFrom = 'From: '||TitleFrom  /* add the header */
    end

    Title.TitleCount = TitleFrom  /* use the sender */

   end

   when ((NextWord = 'TO') | (NextWord = 'RECIPIENTS')) then  /* if the next word is 'TO' or 'RECIPIENTS' */
   do

    TitleTo = CheckCommLine(TitleTo)  /* get rid of problem characters */

    if (Global.TitleKeywords) then  /* if we want title headers */
    do
     TitleTo = 'To  : '||TitleTo  /* add the header */
    end

    Title.TitleCount = TitleTo  /* use the recipients */

   end

   when (NextWord = 'CC') then  /* if the next word is 'CC' */
   do

    TitleCc = CheckCommLine(TitleCc)  /* get rid of problem characters */

    if (TitleCc >< '') then  /* if we have a Cc line */
    do

     if (Global.TitleKeywords) then  /* if we want title headers */
     do
      TitleCc = 'Cc  : '||TitleCc  /* add the header */
     end

     Title.TitleCount = TitleCc  /* use the Cc line */

    end

    else  /* if we have no Cc line */

    do
     TitleCount = TitleCount - 1  /* lower the title lines counter */
    end

   end

   when (NextWord = 'BCC') then  /* if the next word is 'BCC' */
   do

    if (TitleBcc >< '') then  /* if we have a Bcc line */
    do

     TitleBcc = CheckCommLine(TitleBcc)  /* get rid of problem characters */

     if (Global.TitleKeywords) then  /* if we want title headers */
     do
      TitleBcc = 'Bcc : '||TitleBcc  /* add the header */
     end

     Title.TitleCount = TitleBcc  /* use the Bcc line */

    end

    else  /* if we have no Bcc line */

    do
     TitleCount = TitleCount - 1  /* lower the title lines counter */
    end

   end

   when (NextWord = 'SUBJECT') then  /* if the next word is 'SUBJECT' */
   do

    TitleSubject = CheckCommLine(TitleSubject)  /* get rid of problem characters */

    if (Global.TitleKeywords) then  /* if we want title headers */
    do
     TitleSubject = 'Subj: '||TitleSubject  /* add the header */
    end

    Title.TitleCount = TitleSubject  /* use the subject */

   end

   otherwise  /* if it is not one of the above */
   do
    call AddError 'Illegal keyword in '||word('ObjectTitleOut ObjectTitleIn',(Incoming + 1))||' setting: '||NextWord  /* report a non-fatal error */
    TitleCount = TitleCount - 1  /* reduce the counter back to its previous value */
   end

  end

 end

end

if (\Addresses) then  /* if we want not just the recipients' addresses */
do

 if ((wordpos('FROM',TitleWords) = 0) & (wordpos('SENDER',TitleWords) = 0)) then  /* if we did not include either of these in the title */
 do

  if (pos(Global.Warning,TitleFrom) > 0) then  /* if the relevant header entry contains a marker */
  do
   TitleCount = TitleCount + 1  /* up the title lines counter */
   Title.TitleCount = WarnStart||'From: ERROR'||WarnEnd  /* add a warning */
  end

 end

 if ((wordpos('TO',TitleWords) = 0) & (wordpos('RECIPIENTS',TitleWords) = 0)) then  /* if we did not include either of these in the title */
 do

  if (pos(Global.Warning,TitleTo) > 0) then  /* if the relevant header entry contains a marker */
  do
   TitleCount = TitleCount + 1  /* up the title lines counter */
   Title.TitleCount = WarnStart||'To: ERROR'||WarnEnd  /* add a warning */
  end

 end

 if (wordpos('CC',TitleWords) = 0) then  /* if we did not include this one in the title */
 do

  if (pos(Global.Warning,TitleCc) > 0) then  /* if the relevant header entry contains a marker */
  do
   TitleCount = TitleCount + 1  /* up the title lines counter */
   Title.TitleCount = WarnStart||'Cc: ERROR'||WarnEnd  /* add a warning */
  end

 end

 if (wordpos('BCC',TitleWords) = 0) then  /* if we did not include this one in the title */
 do

  if (pos(Global.Warning,TitleBcc) > 0) then  /* if the relevant header entry contains a marker */
  do
   TitleCount = TitleCount + 1  /* up the title lines counter */
   Title.TitleCount = WarnStart||'Bcc: ERROR'||WarnEnd  /* add a warning */
  end

 end

 if (wordpos('SUBJECT',TitleWords) = 0) then  /* if we did not include this one in the title */
 do

  if (pos(Global.Warning,TitleSubject) > 0) then  /* if the relevant header entry contains a marker */
  do
   TitleCount = TitleCount + 1  /* up the title lines counter */
   Title.TitleCount = WarnStart||'NO SUBJECT'||WarnEnd  /* add a warning */
  end

 end

 if (KeepAdds) then  /* if the address insertion folder is still there, it must contain invalid files */
 do
  TitleCount = TitleCount + 1  /* up the title lines counter */
  Title.TitleCount = WarnStart||'INVALID To-Cc-Bcc CONTENT'||WarnEnd  /* add a warning */
 end

 if (pos(Global.CRLF||Global.Warning,Global.CRLF||Global.MessHead) > 0) then  /* if the header contains a line that starts with a marker */
 do
  TitleCount = TitleCount + 1  /* up the title lines counter */
  Title.TitleCount = WarnStart||'ILLEGAL KEYWORD'||WarnEnd  /* add a warning */
 end

 if ((Global.MessBody = '') & Warnings) then  /* if we have no message body and we want warnings */
 do
  TitleCount = TitleCount + 1  /* up the title lines counter */
  Title.TitleCount = WarnStart||'NO MESSAGE BODY'||WarnEnd  /* add a warning */
 end

end

do Index = 2 to TitleCount  /* for each line in the title, except the first one */
 Title.Index = d2c(10)||Title.Index  /* add a leading new line */
end

NewTitle. = ''  /* start with nothing */

do Index = 1 to TitleCount  /* for each line in the title */
 NewTitle.Index = Title.Index  /* copy the contents */
end

FullTitle = ''  /* start with nothing */
MaxLineLength = 100  /* the maximum no. of characters in a file name line */

do Index = 1 to TitleCount  /* for each line in the title */

 if (length(NewTitle.Index) > MaxLineLength) then  /* if this part exceeds the maximum line length */
 do
  NewTitle.Index = left(NewTitle.Index,MaxLineLength)  /* reduce this one to the maximum line length */
 end

 FullTitle = FullTitle||NewTitle.Index  /* add it to the full title */

end

MaxTitleLength = 200 - length(filespec('D',MessFile)||filespec('P',MessFile))  /* the maximum no. of characters in the title */

if (length(FullTitle) > MaxTitleLength) then  /* if the resulting title is too long */
do

 CutLength = MaxTitleLength  /* start by reducing all the strings to the maximum allowable total length */

 do while (length(FullTitle) > MaxTitleLength)  /* as long as the full title is too long */

  do Index = 1 to TitleCount  /* for each line in the title */

   if (length(NewTitle.Index) > CutLength) then  /* if this part exceeds the cutting length */
   do
    NewTitle.Index = left(NewTitle.Index,CutLength)  /* reduce this one to the cutting length */
   end

  end

  FullTitle = ''  /* start with nothing once more */

  do Index = 1 to TitleCount  /* for each line in the title */
   FullTitle = FullTitle||NewTitle.Index  /* add it to the full title */
  end

  CutLength = CutLength - 1  /* reduce the cutting length by one */

 end

end

FullTitle = ''  /* start with nothing once more */

do Index = 1 to TitleCount  /* for each line in the title */

 NewTitle.Index = strip(NewTitle.Index,'T',' ')  /* get rid of trailing blanks */

 if (NewTitle.Index >< Title.Index) then  /* if the contents are not what we originally had */
 do
  NewTitle.Index = NewTitle.Index||'...'  /* add an ellipsis */
 end

 FullTitle = FullTitle||NewTitle.Index  /* add it to the full title */

end

if (right(FullTitle,1) >< '.') then  /* if the full title does not end in a full stop */
do
 FullTitle = FullTitle||'.'  /* add a dot */
end

return FullTitle  /* end of MakeTitle */

/**********************************************************************/
MessageContents: procedure expose Global.  /* gets data from mail file */
/**********************************************************************/

parse arg MessFile,HeadName,BodyName  /* get the arguments */

if (HeadName = '') then  /* if we have no head name spec */
do
 HeadName = 'Global.MessHead'  /* use this default */
end

if (BodyName = '') then  /* if we have no body name spec */
do
 BodyName = 'Global.MessBody'  /* use this default */
end

HeadPart = ''  /* reset the message header content var */
BodyPart = ''  /* reset the message body content var */
call value HeadName,HeadPart  /* set the header variable to contain nothing */
call value BodyName,BodyPart  /* set the body variable to contain nothing */

if (\FileOpen(MessFile,'READ')) then  /* if we cannot open the message file */
do
 return 0  /* return with an error */
end

BodyPart = charin(MessFile,1,chars(MessFile))  /* copy the contents */
call FileClose MessFile  /* and close the file */
parse var BodyPart Header (Global.EmptyLine) BodyPart  /* split the contents into a header and a body */

if (Header >< '') then  /* if we have a header */
do

 do while (right(Header,2) = Global.CRLF)  /* as long as the header ends with a CRLF */
  Header = left(Header,length(Header) - 2)  /* get rid of the CRLF */
 end

 Header = translate(Header,' ',D2C(9))  /* convert any TABs to SPACEs */
 FoldPoint = Global.CRLF||' '  /* define a folding point */

 do until (Header = '')  /* go on util we have no header left */

  Header = strip(Header,'L',' ')  /* remove excess leading spaces */
  parse var Header Line (FoldPoint) Header  /* get the next line ending in a folding point, if any */
  Line = DecodeHeader(Line)  /* decode the line if necessary */
  HeadPart = HeadPart||Line  /* add the line to the new header */

  if (Header >< '') then  /* if we have any header content left, the next line tacks onto the end of the previous one */
  do
   HeadPart = HeadPart||' '  /* add a single space to the new header */
  end

 end

end

else  /* if there is no header */

do
 HeadPart = ''  /* we have nothing to show */
end

call value HeadName,HeadPart  /* set the header variable to contain the header contents */
call value BodyName,BodyPart  /* set the body variable to contain the body contents */

return 1  /* end of MessageContents */

/**********************************************************************/
MessageSettings: procedure expose Global.  /* handles EA settings */
/**********************************************************************/

/**********************************************************************/
/* Flags:                                                             */
/*   Position 0:                                                      */
/*     UNSET = outgoing message                                       */
/*     SET   = incoming message                                       */
/*   Position 1:                                                      */
/*     UNSET = not O.K. to send                                       */
/*     SET   = O.K. to send                                           */
/*   Position 2:                                                      */
/*     UNSET = not processed                                          */
/*     SET   = processed                                              */
/*   Position 3:                                                      */
/*     UNSET = non-ASCII (i.e. quoted-printable text)                 */
/*     SET   = ASCII (i.e. 7-bit, word-wrapped, no TABs)              */
/*   Position 4:                                                      */
/*     UNSET = no attachments                                         */
/*     SET   = attachments                                            */
/**********************************************************************/
/* Switch options:                                                    */
/*   CHANGE:   Changes the file settings to the settings provided,    */
/*             with "1" indicating "set", "0" indicating "reset", and */
/*             "*" indicating "leave unchanged".                      */
/*   MATCH :   Looks for a match between the file settings and the    */
/*             settings provided. A "*" character serves as a "don't  */
/*             care" placeholder.                                     */
/*   CHECK :   See if we are dealing with a RexxMail message file.    */
/**********************************************************************/

parse upper arg MessFile,Setting,Action  /* get the arguments */

if (sysgetea(MessFile,'RXMLSETTINGS','SettingsEA') >< 0) then  /* if we cannot get the current settings */
do
 call AddError 'Cannot retrieve message settings'  /* report */
 return 0  /* and quit with an error */
end

if (SettingsEA = '') then  /* if we find no current setting, it could be an old-style message */
do

 if (sysgetea(MessFile,'.VERSION','SettingsEA') >< 0) then  /* if we cannot get the current settings */
 do
  call AddError 'Cannot retrieve old-style message settings'  /* report */
  return 0  /* and quit with an error */
 end

 if (sysputea(MessFile,'.VERSION','') >< 0) then  /* if we cannot remove the current old-style settings */
 do
  call AddError 'Cannot remove old-style message settings'  /* report */
  return 0  /* and quit with an error */
 end

 parse var SettingsEA EATypeLength 5 SettingsEA  /* get the two main parts */
 SettingsEA = EATypeLength||translate(SettingsEA,'01','-+')  /* recode -+ into 01 and stick the two parts together again */

 if (sysputea(MessFile,'RXMLSETTINGS',SettingsEA) >< 0) then  /* if we cannot set the current settings */
 do
  call AddError 'Cannot set new-style settings'  /* report */
  return 0  /* and quit with an error */
 end

end

parse var SettingsEA EAType 3 . 5 CurSettings  /* get the bits we want */

if (EAType >< 'FDFF'x) then  /* if it is not an ASCII EA */
do
 CurSettings = ''  /* we have no setting either */
end

if (Action = 'CHECK') then  /* if all we are doing is checking to see if we are dealing with a RexxMail mail file */
do

 if (CurSettings >< '') then  /* if we find a CurSettings string */
 do
  return (verify(CurSettings,'10','NOMATCH') = 0)  /* if CurSettings contains nothing but 1 and 0, it must be a RexxMail message file */
 end
 else  /* if we find nothing */
 do
  return 0  /* this cannot be a RexxMail message file */
 end

end

CurSettings = left(CurSettings,8,'0')  /* pad to 8 characters with 0 (i.e. "not set") */

if (Setting = '') then  /* if we received no action argument */
do
 return CurSettings  /* return the current value */
end

Setting = left(Setting,8,'*')  /* pad to 8 characters with asterisks */

if (verify('Setting','*10','M') >< 0) then  /* if Setting contains anything but *, 1, 0 */
do
 call AddError 'Invalid characters in message status setting: '||Setting  /* report */
 return 0  /* return without success */
end

NewSettings = ''  /* start with nothing */

do Index = 1 to 8  /* take each of the characters in the action setting */

 if (substr(Setting,Index,1) = '*') then  /* if the action setting is * */
 do
  NewSettings = NewSettings||substr(CurSettings,Index,1)  /* copy the current setting */
 end
 else  /* if the new setting is not * (but 1 or 0) */
 do
  NewSettings = NewSettings||substr(Setting,Index,1)  /* copy the new setting */
 end

end

select  /* selects the action to take */

 when (Action = 'CHANGE') then  /* if we're asked to change the settings */
 do

  if (left(NewSettings,1) = 0) then  /* if the first position, i.e. the "Received" bit is not set */
  do

   if (substr(NewSettings,3,1) = 1) then  /* if the third position, i.e. the "Processed" bit is set */
   do
    TypeText = 'Mail Message Sent'  /* make it a sent mail message file type */
   end
   else  /* if the third position bit is unset */
   do
    TypeText = 'Mail Message Out'  /* make it an outgoing mail message file type */
   end

  end
  else  /* if the "Received" bit is set */
  do
   TypeText = 'Mail Message In'  /* make it an incoming mail message file type */
  end

  EALength = reverse(right(d2c(Length(TypeText)),2,'00'x))  /* set the EA length */
  Type = 'DFFF'x||'0000'x||'0100'x||'FDFF'x||EALength||TypeText  /* set the message type */

  if (sysputea(MessFile,'.TYPE',Type) >< 0) then  /* if we cannot attach it to the file */
  do
   call AddError 'Cannot attach mail file type'  /* report */
   return 0  /* and quit with an error */
  end

  EALength = reverse(right(d2c(Length(NewSettings)),2,'00'x))  /* set the EA length */
  Version = 'FDFF'x||EALength||NewSettings  /* the ASCII version EA */

  if (sysputea(MessFile,'RXMLSETTINGS',Version) >< 0) then  /* if we cannot attach it to the file */
  do
   call AddError 'Cannot attach mail file version'  /* report */
   return 0  /* and quit with an error */
  end

  if (\syssetobjectdata(MessFile,'ICONFILE='||Global.PaperDir||'\'||NewSettings||'.ICO')) then  /* if we cannot set the icon */
  do
   call AddError 'Cannot set icon'  /* report */
   return 0  /* and quit with an error */
  end

  return 1  /* return with success */

 end

 when (Action = 'MATCH') then  /* if the action is to check for a match */
 do
  return (NewSettings = CurSettings)  /* return O.K. if they match */
 end

 otherwise  /* if we have an unexpected action parameter */
 do
  return 0  /* return an error status */
 end

end

return 0  /* end of MessageSettings */

/**********************************************************************/
MoveMessage: procedure expose Global.  /* moves a mail message file after sending or viewing, if necessary */
/**********************************************************************/

parse arg MessFile,MoveFile,Incoming  /* get the parameters */

MoveDir = MoveMessageDir(MessFile,MoveFile,(Incoming = 1))  /* see if we need to move the file to another folder */

if (MoveDir = '') then  /* if we are not to move the message, or we have nowhere to move it to */
do
 return ''  /* return with no move */
end

if (\sysmoveobject(MessFile,MoveDir)) then  /* if we cannot move the message file to the directory, another one of the same name already exists */
do

 Title = MakeTitle(MessFile,,,,Incoming)  /* get the message title */
 MessDir = filespec('D',MessFile)||filespec('P',MessFile)  /* the message dir */
 TempFile = TempFileName('MOVE',MessDir)  /* get a temp file name in the message dir */
 TempName = filespec('N',TempFile)  /* get the name part */

 if (\syssetobjectdata(MessFile,'TITLE='||TempName)) then  /* if we cannot rename the file */
 do
  call AddError 'Cannot rename message to move it to '||MoveDir  /* report */
  return ''  /* return with nothing */
 end

 if (\sysmoveobject(TempFile,MoveDir)) then  /* if we cannot move the temp name message file to the directory */
 do

  call AddError 'Cannot move message to '||MoveDir  /* report */

  if (\syssetobjectdata(TempFile,'TITLE='||Title)) then  /* if we cannot restore the file title */
  do
   call AddError 'Cannot restore message title'  /* report */
  end

  return ''  /* return with nothing */

 end

 call syssetobjectdata MoveDir||'\'||TempName,'TITLE='||Title  /* restore the moved file title */

end

call LogAction 'Message moved to "'||MoveDir||'"'  /* report */

return MoveDir  /* end of MoveMessage */

/**************************************************************************/
MoveMessageDir: procedure expose Global.  /* returns a redirect action for a mail message file */
/**************************************************************************/

parse arg MessFile,MoveFile,Incoming  /* get the parameters */

if (\FileCheck(MoveFile)) then  /* if we can find no move list file */
do
 return ''  /* return with nothing */
end

Incoming = (Incoming = 1)  /* if the Incoming argument is 1, it is true */

FromStr = GetHeaderEntry(Global.MessHead,'FROM:')  /* get the sender's address(es) from the 'From:' line */

if (FromStr = '') then  /* if we did not find a sender */
do

 FromStr = GetHeaderEntry(Global.MessHead,'APPARENTLY-FROM:')  /* get the sender's address(es) from the 'Apparently-from:' line*/

 if (FromStr = '') then  /* if we did not find a sender */
 do
  FromStr = GetHeaderEntry(Global.MessHead,'SENDER:')  /* get the sender's address(es) from the 'Sender:' line */
 end

end

ToStr = GetHeaderEntry(Global.MessHead,'TO:')||', '||GetHeaderEntry(Global.MessHead,'CC:')  /* get the recipient address(es) */
SubjectStr = GetHeaderEntry(Global.MessHead,'SUBJECT:')  /* get the subject */
JunkMail = (Incoming & (CheckHeader(Global.JunkMailHeader,Global.MessHead)))  /* if the message is incoming and matches the junk mail header definition, set the junk mail flag */
VirusMail = (Incoming & (CheckHeader(Global.VirusMailHeader,Global.MessHead)))  /* if the message is incoming and matches the virus mail header definition, set the virus mail flag */

Action = GetRedirect(MoveFile,FromStr,ToStr,SubjectStr,JunkMail,VirusMail)  /* try to find a matching action */

if (Action = '') then  /* if we have nothing */
do
 return ''  /* nothing to do */
end

MoveDir = ''  /* we have no destination dir yet */

do while (Action >< '')  /* go on until we run out of steam */

 parse var Action NextPart '&' Action  /* get the next bit */

 select  /* do one of the following */

  when (word(NextPart,1) = '*RUN*') then  /* if it is a command to run */
  do
   call RunCommand subword(NextPart,2),MessFile  /* run the string as a command */
  end

  when (NextPart = '*SKIP*') then  /* if it is the special leave string */
  do
   MoveDir = ''  /* there is no need to move this one */
  end

  when (NextPart = '*DELETE*') then  /* if it is the special delete string */
  do
   MoveDir = Global.TempDir  /* use the temp dir to move the file out of sight to be deleted the next time the mail folders are opened */
  end

  otherwise  /* if it is none of the above */
  do

   MoveDir = strip(NextPart,'T','\')  /* strip off any trailing backslash */

   if (left(MoveDir,1) = '*') then  /* if it starts with this, we need to add the In or Out Archive dir */
   do

    MoveDir = strip(substr(MoveDir,2),'L','\')  /* use the rest and strip off any leading backslash */

    if (Incoming) then  /* if the message is incoming */
    do
     MoveDir = Global.InArchDir||'\'||MoveDir  /* use this destination */
    end
    else  /* if it is outgoing */
    do
     MoveDir = Global.OutArchDir||'\'||MoveDir  /* use this destination */
    end

   end

  end

 end

end

if (MoveDir = '') then  /* if we still have no destination dir */
do
 return ''  /* quit */
end

call sysfiletree MoveDir,'MoveDir.','DO'  /* see if the dir exists */

if (MoveDir.0 = 0) then  /* if it does not */
do

 NextPos = 3  /* start looking at position 3 (well, actually 4, since we add 1 later) */

 do until (NextPos = 0)  /* go on until we reach the end of the line */

  NextPos = pos('\',MoveDir,(NextPos + 1))  /* get the next backslash position */

  if (NextPos = 0) then  /* if we have reached the end of the line */
  do
   NextDir = MoveDir  /* we have the whole dir spec */
  end
  else  /* if not */
  do
   NextDir = left(MoveDir,(NextPos - 1))  /* get the next dir level */
  end

  call sysfiletree NextDir,'NextDir.','DO'  /* see if the dir exists */

  if (NextDir.0 = 0) then  /* if it does not */
  do

   if (sysmkdir(NextDir) >< 0) then  /* if we cannot create the dir */
   do
    say 'Cannot create "'||NextDir||'"'  /* report */
    return ''  /* no result */
   end

  end

 end

end
else  /* if it exists */
do

 if (translate(MoveDir)||'\' = translate(filespec('D',MessFile)||filespec('P',MessFile))) then  /* if it is the original message dir */
 do
  return ''  /* forget it */
 end

end

return MoveDir  /* end of MoveMessageDir */

/**********************************************************************/
OpenFolders: procedure expose Global.  /* opens the RexxMail message folders */
/**********************************************************************/

call sysopenobject '<REXXMAIL_TOOLBAR>',121,0  /* always try to open the default toolbar */

AlreadyOpen = 0  /* assume the main folder was not already open */

Keywords = 'ACCESSORIES',  /* a keyword */
           'ADDRESSES',  /* a keyword */
           'CONFIGURATION',  /* a keyword */
           'IN',  /* a keyword */
           'INARCHIVE',  /* a keyword */
           'OUT',  /* a keyword */
           'OUTARCHIVE'  /* a keyword */

Folders = 'Global.AccessDir',  /* a folder corresponding to a keyword */
          'Global.AddrDir',  /* a folder corresponding to a keyword */
          'Global.ConfDir',  /* a folder corresponding to a keyword */
          'Global.InDir',  /* a folder corresponding to a keyword */
          'Global.InArchDir',  /* a folder corresponding to a keyword */
          'Global.OutDir',  /* a folder corresponding to a keyword */
          'Global.OutArchDir'  /* a folder corresponding to a keyword */

if (sysqueryswitchlist('ListItems.') = 0) then  /* if we can get a switch list */
do Index = 1 to ListItems.0  /* take each of the items found  */

 if (translate(ListItems.Index) = translate(filespec('N',Global.MainDir))) then  /* if we find the main dir */
 do
  AlreadyOpen = 1  /* the main dir is open (or minimized) on the desktop */
 end

end

if (\AlreadyOpen) then  /* if the main dir was not already open */
do
 call RunCommand Global.RunBeforeOpen  /* see if we can run an external command before opening the folders */
end

do while (Global.OpenFolders >< '')  /* as long as we have folder values */

 parse upper var Global.OpenFolders NextKeyword Global.OpenFolders  /* get the next value in upper case */

 OpenFolder = word(Folders,wordpos(NextKeyword,Keywords))  /* get the corresponding folder */

 if (OpenFolder >< '') then  /* if we found it */
 do

  if (\sysopenobject(value(OpenFolder),0,1)) then  /* if we cannot open the folder on the desktop */
  do
   call AddError 'Cannot open folder "'||OpenFolder||'"'  /* report */
  end

 end
 else  /* if we did not find it */
 do
  call AddError 'Invalid OpenFolders argument: "'||NextKeyWord||'"'  /* report */
 end

end

if (\syssetobjectdata(Global.MainDir,'WORKAREA=NO;')) then  /* if we cannot make the main folder a 'no work area' folder */
do
 call AddError 'Cannot set RexxMail folder "'||Global.MainDir||'" to "no work area"'  /* report, not fatal */
end

if (Global.MessagesDir >< Global.MainDir) then  /* if the "messages" subdir is not the same as the main dir */
do

 if (\sysopenobject(Global.MessagesDir,0,1)) then  /* if we cannot open the messages dir */
 do
  call AddError 'Cannot open RexxMail folder "'||Global.MessagesDir||'"'  /* report */
  return 0  /* and quit */
 end

end

if (\sysopenobject(Global.MainDir,0,1)) then  /* if we cannot open the main dir */
do
 call AddError 'Cannot open RexxMail folder "'||Global.MainDir||'"'  /* report */
 return 0  /* and quit */
end

if (\syssetobjectdata(Global.MainDir,'WORKAREA=YES;')) then  /* if we cannot make the main folder a 'work area' folder */
do
 call AddError 'Cannot set RexxMail folder "'||Global.MainDir||'" to "work area"'  /* report, not fatal */
end

call MailWarning 1  /* destroy the mail warning program object, if any */

if (\AlreadyOpen) then  /* if the main dir was not already open */
do

 AllAttDirs = ''  /* start with nothing */
 call sysfiletree Global.Outdir||'\*','Files.','FO'  /* look for RexxMail files in the 'Out' folder */

 do Index = 1 to Files.0  /* take each of the objects found */
  AllAttDirs = AllAttDirs||' '||AttDirGet(Files.Index,0)  /* get the file's attachment folder name, if any, and add it to the list (do not create it) */
 end

 call sysfiletree Global.TempDir||'\rxml????','TempDirs.','DO'  /* look for attachment folders in the temporary storage directory */

 do Index = 1 to TempDirs.0  /* take each of the objects found */

  if (wordpos(TempDirs.Index,AllAttDirs) = 0) then  /* if its name is not in the current attachment folders list */
  do

   ZapDir = 1  /* assume we can get rid of the dir to begin with */
   DirName = filespec('N',TempDirs.Index)  /* get the name part of the dir spec */

   do SubIndex = 1 to ListItems.0  /* take each of the task list items found  */

    if (ListItems.SubIndex = DirName) then  /* if we find the dir in the task list, it is open on the desktop */
    do
     ZapDir = 0  /* no need to zap the dir yet */
    end

   end

   if (ZapDir) then  /* if we are to remove the dir */
   do
    call DeleteDir TempDirs.Index  /* get rid of it */
   end

  end

 end

 call sysfiletree Global.TempDir||'\*','Files.','FO'  /* look for any remaining files */

 do Index = 1 to Files.0  /* take each of the files found */

  if (MessageSettings(Files.Index,,'CHECK')) then  /* if the file is a RexxMail message */
  do

   if (\FileInUse(Files.Index)) then  /* if the file is not in use */
   do
    call sysfiledelete Files.Index  /* delete it */
   end

  end

  else  /* if the file is not a RexxMail message file */

  do

   if ((left(filespec('N',Files.Index),4) = 'RXML') & (datatype(substr(filespec('N',Files.Index),5,4),'W'))) then  /* if it is a temporary RexxMail file */
   do

    if (right(Files.Index,4) = 'POP3') then  /* if it is a POP3 file that somehow got left behind */
    do

     if (\FileInUse(Files.Index)) then  /* if the file is not in use */
     do
      call RegisterMessage Files.Index  /* register the message */
     end

    end
    else  /* if it is not a POP3 file */
    do
     call sysfiledelete Files.Index  /* delete it */
    end

   end

  end

 end

 call RunCommand Global.RunAfterOpen  /* see if we can run an external command after opening the folders */
 call TrimLogFile Global.ActionLog,Global.LogActionLines  /* trim if necessary */
 call TrimLogFile Global.ErrorLog,Global.LogErrorLines  /* trim if necessary */
 call TrimLogFile Global.MailLog,Global.LogMailLines  /* trim if necessary */

end

return 1  /* end of OpenFolders */

/**********************************************************************/
POP3DeleteMess:  procedure expose Global.  /* delete a message from the POP3 server */
/**********************************************************************/

parse arg Socket,Index  /* get the arguments */

if (\SocketSendLine(Socket,'DELE '||Index,'+OK')) then  /* if we cannot send this and get the right reply */
do
 call AddError 'Cannot delete message (server no. '||Index||')'  /* report */
 return 0  /* return with an error */
end

call LogAction 'Deleted message (server no. '||Index||')'  /* report */

return 1  /* end of POP3DeleteMess */

/**********************************************************************/
POP3GetStatus: procedure expose Global.  /* get the number of waiting messages from the POP3 server */
/**********************************************************************/

parse arg Socket  /* get the argument */

Reply = SocketSendLine(Socket,'STAT','?')  /* send this and get the reply */

parse var Reply Status Messages Bytes  /* get the status, the number of messages and the total byte count */

if (Status >< '+OK') then  /* if we did not get the right status reply */
do
 call AddError 'Cannot get message status from server'  /* report */
 return 0  /* quit with nothing */
end

if (Messages = 0) then  /* if there are no messages */
do
 call LogAction 'There are no waiting messages'  /* report */
end
else  /* if we have waiting messages */
do
 call lineout 'CON:',''  /* skip a line on the display */
 call LogAction 'No. of waiting messages = '||Messages||'; total size = '||Bytes||' bytes'  /* report */
end

return Messages  /* end of POP3GetStatus */

/**********************************************************************/
POP3Process: procedure expose Global.  /* handles mail retrieval from POP3 server */
/**********************************************************************/

parse arg Socket  /* get the argument */

if (\Global.POP3Interactive) then  /* if we are not interactive */
do

 AutoProcess = (FileCheck(Global.ControlColl))  /* if we can find the action file, we want automated processing */

 if (AutoProcess) then  /* if we have an action specs file, i.e. we are collecting messages automatically (even if the file is empty!) */
 do
  call LogAction 'Using collect action file "'||Global.ControlColl||'"'  /* report */
 end

end

Collected = 0  /* we have not collected any messages yet */
Deleted = 0  /* we have not deleted any messages yet */

do until (\Global.POP3Interactive)  /* go on looping if we are working interactively */

 if (POP3GetStatus(Socket) = 0) then  /* if we have no waiting messages */
 do

  if ((Deleted = 0) | (\Global.POP3Interactive)) then  /* if we have not deleted any messages either, or if we are not processing interactively */
  do
   return Collected  /* we're done */
  end

  else  /* if we have deleted messages, and we are processing interactively */

  do

   if (Deleted = 1) then  /* if we have only one deleted message */
   do
    Plural = ''  /* no plural s */
   end
   else  /* if we have more than one deleted message */
   do
    Plural = 's'  /* we need a plural s */
   end

   ValidKeys = '1Uu0Qq'||d2c(13)||d2c(27)  /* we will accept these keystrokes */
   call lineout 'CON:',''  /* empty line */
   call lineout 'CON:','             1 = Undelete '||Deleted||' message'||Plural||' on server'  /* info */
   call lineout 'CON:','             0 = Quit'  /* info */
   call lineout 'CON:',''  /* empty line */
   call charout 'CON:','             Your choice (1 U [Enter] | 0 Q [Esc]): U'||d2c(8)  /* prompt */

   do until (pos(Reply,ValidKeys) > 0)  /* go on until we get a valid keystroke */
    Reply = sysgetkey('NOECHO')  /* get the reply */
   end

   if (pos(Reply,'1Uu'||d2c(13)) > 0) then  /* if it is one of these */
   do

    call lineout 'CON:','U'  /* echo the key */
    call lineout 'CON:',''  /* empty line */

    if (POP3Reset(Socket)) then  /* if we can reset the server */
    do
     Deleted = 0  /* we have no deleted messages now */
    end

   end

   else  /* if it was another key, we'll quit */

   do
    call lineout 'CON:','Q'  /* complete the line */
    call lineout 'CON:',''  /* empty line */
    return Collected  /* we're done */
   end

  end

 end

 else  /* if we have waiting messages */

 do

  call LogAction 'Retrieving message list'  /* report */

  if (\SocketSendLine(Socket,'LIST','+OK')) then  /* if we cannot send this and get the right reply */
  do
   call AddError 'Cannot retrieve message list from server'  /* report */
   return Collected  /* return with no success */
  end

  SortList. = ''  /* start with an empty sorting list */
  MessList. = ''  /* start with an empty message list */
  MessCount = 0  /* start with a message count of 0 */
  NextLine = SocketGetLine(Socket)  /* get a line from the POP3 socket */

  do while ((NextLine >< '.') & (NextLine >< d2c(0)))  /* as long as it is not a full stop or a null byte */
   MessCount = MessCount + 1  /* up the message counter */
   parse var NextLine Number Bytes .  /* get the ingredients */
   SortList.MessCount = right(Bytes,12,'0')||right(Number,6,'0')  /* this will be used for sorting to size later */
   MessList.Number.!Bytes = Bytes  /* store the byte count */
   NextLine = SocketGetLine(Socket)  /* get the next line */
  end

  if (NextLine >< '.') then  /* if we did not get to the end of the list */
  do
   call AddError 'Cannot retrieve message list from server'  /* report */
   return Collected  /* return with no success */
  end

  if (\Global.Pop3Interactive) then  /* if we are not working interactively */
  do

   if (MessCount > 0) then  /* if we have something */
   do

    SortList.0 = MessCount  /* copy the message counter */

    if (sysstemsort('SortList.') = 0) then  /* if we can sort the list in ascending order */
    do
     call LogAction 'Message list sorted; retrieving messages in order of size.'  /* report (failure to sort is non-fatal) */
    end

   end

  end

  call LogAction 'Retrieving message headers'  /* report */

  do Index = 1 to MessCount  /* run through the list */

   parse var SortList.Index =13 MessNumber  /* retrieve the message number */

   MessNumber = strip(MessNumber,'L','0')  /* get rid of leading zeroes */

   if (\SocketSendLine(Socket,'TOP '||MessNumber||' 0','+OK')) then  /* if we cannot send this and get the right reply */
   do
    call AddError 'Cannot retrieve header of message '||Index||' (no. '||MessNumber||' on server)'  /* report */
    return Collected  /* return with no success */
   end

   SortList.Index = MessNumber  /* replace the sort list contents (bytes/number) with the message number */
   MessList.MessNumber.!For = '[unknown]'  /* the default recipient text */
   MessList.MessNumber.!JunkMail = 0  /* no junk mail yet */
   MessList.MessNumber.!VirusMail = 0  /* no virus mail yet */
   GotRecipient = 0  /* we have no recipient yet */
   NextLine = ''  /* nothing yet */
   StoreLine = SocketGetLine(Socket)  /* get a line from the POP3 socket */

   do while ((NextLine >< '.') & (NextLine >< d2c(0)))  /* as long as we do not encounter a full stop or a null byte by itself */

    if (NextLine >< '') then  /* if we have a next line */
    do

     if (pos(left(NextLine,1),d2c(9)||d2c(32)) > 0) then  /* if the next line starts with whitespace */
     do
      StoreLine = StoreLine||' '||strip(translate(NextLine,d2c(32),d2c(9)),'B',' ')  /* add it to what we have, removing any excess leading or trailing whitespace */
     end
     else  /* if the next line does not start with whitespace */
     do

      parse var StoreLine FirstWord ':' RestOfLine  /* get the bits we want from what we have in store */
      StoreLine = NextLine  /* start a new stored line */
      FirstWord = translate(strip(FirstWord,'T',' '))  /* remove any trailing blanks from the first word, and make it upper case */
      RestOfLine = strip(RestOfLine)  /* get rid of excess whitespace */

      select  /* do one of the following */

       when (FirstWord = 'DATE') then  /* if it is this */
       do
        MessList.MessNumber.!Date = DateTimeDisplay(RestOfLine,'UTCISO')  /* rewrite the date if necessary and store it */
       end

       when (FirstWord = 'FROM') then  /* if it is this */
       do
        Sender = DecodeHeader(RestOfLine)  /* decode if necessary */
        MessList.MessNumber.!From = AddressFormat(Sender,0,0)  /* format the address to our liking */
        MessList.MessNumber.!FromAddress = AddressFormat(Sender,,0)  /* also get the bare address */
       end

       when (FirstWord = 'TO') then  /* if it is this */
       do
        MessList.MessNumber.!To = AddressFormat(DecodeHeader(RestOfLine),,0)  /* decode if necessary, and store it bare */
       end

       when (FirstWord = 'CC') then  /* if it is this */
       do
        MessList.MessNumber.!Cc = AddressFormat(DecodeHeader(RestOfLine),,0)  /* decode if necessary, and store it bare */
       end

       when (FirstWord = 'BCC') then  /* if it is this (which should not occur, but who knows what lurks out there) */
       do
        MessList.MessNumber.!Bcc = AddressFormat(DecodeHeader(RestOfLine),,0)  /* decode if necessary, and store it bare */
       end

       when (FirstWord = 'SUBJECT') then  /* if it is this */
       do

        MessList.MessNumber.!Subject = DecodeHeader(RestOfLine)  /* decode if necessary, and store it */

        if (length(MessList.MessNumber.!Subject) > 60) then  /* if there is too much of it */
        do
         MessList.MessNumber.!Subject = left(MessList.MessNumber.!Subject,60)||'...'  /* trim it and add an ellipsis */
        end

       end

       when (FirstWord = 'MESSAGE-ID') then  /* if it is this */
       do
        MessList.MessNumber.!MessageID = RestOfLine  /* store it */
       end

       when ((FirstWord = 'RECEIVED') & (\GotRecipient)) then  /* if it is this and we have no recipient yet */
       do

        parse var RestOfLine . ' for ' Recipient .  /* get the bit we want */
        Recipient = strip(Recipient,'T',';')  /* remove any trailing semicolon */

        if (Recipient >< '') then  /* if we have something */
        do
         MessList.MessNumber.!For = AddressFormat(Recipient,,0)  /* store the bare address */
         GotRecipient = 1  /* we have our recipient */
        end

       end

       otherwise  /* if it is none of the above */
       do

        if (\MessList.MessNumber.!JunkMail) then  /* if we have no junk mail flag yet */
        do
         MessList.MessNumber.!JunkMail = CheckHeader(Global.JunkMailHeader,FirstWord||': '||RestOfLine)  /* set the junk mail flag if necessary */
        end

        if (\MessList.MessNumber.!VirusMail) then  /* if we have no virus mail flag yet */
        do
         MessList.MessNumber.!VirusMail = CheckHeader(Global.VirusMailHeader,FirstWord||': '||RestOfLine)  /* set the virus mail flag if necessary */
        end

       end

      end

     end

    end

    NextLine = SocketGetLine(Socket)  /* get the next line */

   end

   if (NextLine >< '.') then  /* if we did not get to the termination sequence */
   do
    call AddError 'Cannot retrieve header of message '||Index||' (no. '||MessNumber||' on server)'  /* report */
    return Collected  /* return with no success */
   end

  end

  do Index = 1 to MessCount  /* run through the message info we collected */

   MessNumber = SortList.Index  /* get the message number */
   LogText = ' message no. '||MessNumber||' on server ('||MessList.MessNumber.!Bytes||' bytes)'  /* a line for the action log file */

   call lineout 'CON:',''  /* empty line on the display */
   call lineout 'STDOUT:','Message no.: '||Index||' of '||MessCount||' (no. '||MessNumber||' on server)'  /* report to standard output */
   call lineout 'STDOUT:','Message ID : '||MessList.MessNumber.!MessageID  /* report to standard output */
   call lineout 'STDOUT:','Size       : '||MessList.MessNumber.!Bytes||' bytes'  /* report to standard output */
   call lineout 'STDOUT:','Date       : '||MessList.MessNumber.!Date  /* report to standard output */
   call lineout 'STDOUT:','From       : '||MessList.MessNumber.!From  /* report to standard output */
   call lineout 'STDOUT:','For        : '||MessList.MessNumber.!For  /* report to standard output */
   call lineout 'STDOUT:','Subject    : '||MessList.MessNumber.!Subject  /* report to standard output */

   if (MessList.MessNumber.!JunkMail = 1) then  /* if it was flagged as junk mail */
   do
    call lineout 'STDOUT:','Note       : This message has been flagged as junk mail'  /* report to standard output */
   end

   if (MessList.MessNumber.!VirusMail = 1) then  /* if it was flagged as virus mail */
   do
    call lineout 'STDOUT:','Note       : This message has been flagged as virus mail'  /* report to standard output */
   end

   if (Global.POP3Interactive) then  /* if we are processing interactively */
   do

    ValidKeys = '1Gg2Kk3Dd4Ss0Qq'||d2c(13)||d2c(27)  /* we will accept these keystrokes */
    UndelKey = ''  /* do not show the undelete key */
    call lineout 'CON:',''  /* empty line */
    call lineout 'CON:','             1 = Get message and delete from server'  /* info */
    call lineout 'CON:','             2 = get message, but Keep on server'  /* info */
    call lineout 'CON:','             3 = Delete message from server'  /* info */
    call lineout 'CON:','             4 = Skip message'  /* info */

    if (Deleted > 0) then  /* if we have deleted messages */
    do

     if (Deleted = 1) then  /* if we have only one deleted message */
     do
      Plural = ''  /* no plural s */
     end
     else  /* if we have more than one deleted message */
     do
      Plural = 's'  /* we need a plural s */
     end

     call lineout 'CON:','             5 = Undelete '||Deleted||' message'||Plural||' on server'  /* info */
     ValidKeys = ValidKeys||'5Uu'  /* we will now also accept these keystrokes */
     UndelKey = ' | 5 U'  /* show the undelete key */

    end

    call lineout 'CON:','             0 = Quit'  /* info */
    call lineout 'CON:',''  /* empty line */
    call charout 'CON:','             Your choice (1 G [Enter] | 2 K | 3 D | 4 S'||UndelKey||' | 0 Q [Esc]): G'||d2c(8)  /* prompt */

    do until (pos(Reply,ValidKeys) > 0)  /* go on until we get a valid keystroke */
     Reply = sysgetkey('NOECHO')  /* get the reply */
    end

    select  /* do one of the following */

     when (pos(Reply,'1Gg'||d2c(13)) > 0) then  /* if it is one of these */
     do
      call lineout 'CON:','G'  /* echo the key */
      Action = '*GET*'  /* use this */
     end

     when (pos(Reply,'2Kk') > 0) then  /* if it is one of these */
     do
      call lineout 'CON:','K'  /* echo the key */
      Action = '*KEEP*'  /* use this */
     end

     when (pos(Reply,'3Dd') > 0) then  /* if it is one of these */
     do
      call lineout 'CON:','D'  /* echo the key */
      Action = '*DELETE*'  /* use this */
     end

     when (pos(Reply,'4Ss') > 0) then  /* if it is one of these */
     do
      call lineout 'CON:','S'  /* just echo the key */
      Action = '*SKIP*'  /* use this */
     end

     when (pos(Reply,'5Uu') > 0) then  /* if it is one of these */
     do
      call lineout 'CON:','U'  /* echo the key */
      Action = '*UNDELETE*'  /* use this */
     end

     when (pos(Reply,'0Qq'||d2c(27)) > 0) then  /* if it is one of these */
     do
      call lineout 'CON:','Q'  /* echo the key */
      Action = '*QUIT*'  /* use this */
     end

     otherwise  /* this shouldn't occur */
     do
      Action = '*SKIP*'  /* use this to be on the safe side */
     end

    end

    call lineout 'CON:',''  /* empty line */

   end

   else  /* if we are not processing interactively */

   do

    if (AutoProcess) then  /* if we are processing using an action specs file */
    do

     Action = GetRedirect(Global.ControlColl,MessList.MessNumber.!FromAddress,MessList.MessNumber.!For,MessList.MessNumber.!Subject,MessList.MessNumber.!JunkMail,MessList.MessNumber.!VirusMail)  /* look for a matching action spec */

     if (Action = '') then  /* if we found nothing */
     do
      Action = '*GET*'  /* use this */
     end

    end

    else  /* if we are simply getting all waiting mail */

    do
     Action = '*GET*'  /* use this */
    end

   end

   select  /* do one of the following */

    when ((Action = '*GET*') | (Action = '*KEEP*')) then  /* if we have one of these */
    do

     if (Global.MaxCollectSize >< '') then  /* if we have a size limit */
     do

      if (MessList.MessNumber.!Bytes > Global.MaxCollectSize) then  /* if this one is over the limit */
      do
       Action = ''  /* no more action required */
       Subject = 'Message '||MessList.MessNumber.!MessageID||' exceeds size limit.'  /* a message subject */
       MessText = 'The following message was left on the mail server because it exceeds'||Global.CRLF||,  /* start a message text */
                  'the maximum collect size setting of '||Global.MaxCollectSize||' bytes:'||Global.EmptyLine||,  /* start a message text */
                  '    Message ID : '||MessList.MessNumber.!MessageID||Global.CRLF||,  /* add this */
                  '    Size       : '||MessList.MessNumber.!Bytes||' bytes'||Global.CRLF||,  /* add this */
                  '    Date       : '||MessList.MessNumber.!Date||Global.CRLF||,  /* add this */
                  '    From       : '||MessList.MessNumber.!From||Global.CRLF||,  /* add this */
                  '    For        : '||MessList.MessNumber.!For||Global.CRLF||,  /* add this */
                  '    Subject    : '||MessList.MessNumber.!Subject  /* add this */
       call SystemMessage Subject,MessText  /* report to the user */
       call LogMail '===',MessList.MessNumber.!FromAddress,MessList.MessNumber.!For,MessList.MessNumber.!To,MessList.MessNumber.!Cc,MessList.MessNumber.!Bcc,MessList.MessNumber.!Subject,MessList.MessNumber.!MessageID  /* log the message */
      end

     end

     if (Action >< '') then  /* if we are to carry on */
     do

      MailFile = POP3RetrieveMessage(Socket,MessNumber,MessList.MessNumber.!Bytes)  /* if we can collect the message, we should get back a file name */

      if (MailFile >< '') then  /* if we do get back a file name */
      do

       if ((Action = '*GET*') & (\Global.POP3KeepMessages)) then  /* unless we want to always keep the messages on the server */
       do

        if (POP3DeleteMess(Socket,MessNumber)) then  /* if we can delete the message */
        do
         Deleted = Deleted + 1  /* up the deleted messages counter */
        end

       end

       Collected = Collected + 1  /* up the collected messages counter */
       call LogMail '<==',MessList.MessNumber.!FromAddress,MessList.MessNumber.!For,MessList.MessNumber.!To,MessList.MessNumber.!Cc,MessList.MessNumber.!Bcc,MessList.MessNumber.!Subject,MessList.MessNumber.!MessageID  /* log the message */
       call SoundSignal Global.SignalReceived  /* signal if required */
       call RunCommand Global.RunReceived,MailFile  /* run an optional external command on the message file */

       if (FileCheck(MailFile)) then  /* if the mail file is still there */
       do
        call RegisterMessage MailFile  /* register the message */
       end

      end

     end

    end

    when (Action = '*DELETE*') then  /* if we have this */
    do

     if (\Global.POP3KeepMessages) then  /* unless we want to keep the messages on the server */
     do

       if (POP3DeleteMess(Socket,MessNumber)) then  /* if we can delete the message */
       do
        call LogMail '=X=',MessList.MessNumber.!FromAddress,MessList.MessNumber.!For,MessList.MessNumber.!To,MessList.MessNumber.!Cc,MessList.MessNumber.!Bcc,MessList.MessNumber.!Subject,MessList.MessNumber.!MessageID  /* log the message as deleted */
        Deleted = Deleted + 1  /* up the deleted messages counter */
       end

     end

    end

    when (Action = '*SKIP*') then  /* if we have this */
    do
     call POP3SkipMess MessNumber  /* skip the message */
    end

    when (Action = '*UNDELETE*') then  /* if we have this */
    do

     if (POP3Reset(Socket)) then  /* if we can reset the server */
     do
      Deleted = 0  /* we have no deleted messages now */
     end

     Index = MessCount  /* make sure we exit the current counted loop to rescan the list */

    end

    when (Action = '*QUIT*') then  /* if we have this */
    do
     return Collected  /* quit */
    end

    otherwise  /* if none of the above */
    do
     call AddError 'Incorrect entry in collect action file: "'||Action||'"'  /* report */
     call POP3SkipMess MessNumber  /* skip the message */
    end

   end

  end

 end

 call lineout 'CON:',''  /* empty line on the display */

end

return Collected  /* end of POP3Process */

/**********************************************************************/
POP3Reset: procedure expose Global.  /* resets the POP3 server */
/**********************************************************************/

parse arg Socket  /* get the argument */

if (\SocketSendLine(Socket,'RSET','+OK')) then  /* if we cannot reset the server */
do
 call AddError 'Cannot undelete messages on server'  /* report */
 return 0  /* return with an error */
end

call LogAction 'Messages undeleted'  /* report */

return 1  /* end of POP3Reset */

/**********************************************************************/
POP3RetrieveMessage: procedure expose Global.  /* retrieve a message from the POP3 server */
/**********************************************************************/

parse arg Socket,Index,ByteCount  /* get the arguments */

if (\SocketSendLine(Socket,'RETR '||Index,'+OK')) then  /* if we cannot send this and get the right reply */
do
 call AddError 'Cannot retrieve message (server no. '||Index||') from server'  /* report */
 return ''  /* return with no success */
end

MailFile = TempFileName('POP3')  /* get a unique file name in the temp folder */

if (\FileOpen(MailFile,'WRITE')) then  /* if we cannot open the file for writing */
do
 call AddError 'Cannot open '||MailFile  /* report */
 return ''  /* return with no success */
end

HostName = value('HOSTNAME',,'OS2Environment')  /* try to get a host name from the OS/2 environment */

if (HostName >< '') then  /* if we have a host name */
do
 parse var Global.Address . '@' Domain '>'  /* get the (real) sender's domain name */
 HostName = HostName||'.'||Domain  /* add the domain name to it */
end
else  /* if we have no host name */
do
 HostName = 'localhost'  /* just use this */
end

call lineout MailFile,'Received: by '||HostName||'; '||DateTimeRFC()  /* start with a "Received:" line */
call time 'R'  /* reset the timer */

InBytes = 0  /* we have collected nothing yet */

if (Global.ShowProgress) then  /* if we want stats */
do
 call syscurstate 'OFF'  /* switch off the cursor */
 LastTime = ProgressBar('Bytes retrieved = ',InBytes,ByteCount,0,0)  /* start a progress bar */
end

call syssleep 0.00001  /* wait a fraction of a second to make sure the timer gets started */

NextLine = SocketGetLine(Socket)  /* get the next line */

do while ((NextLine >< '.') & (NextLine >< d2c(0)))  /* go on until we collect a full stop or a null byte */

 if (left(NextLine,1) = '.') then  /* if the line starts with a period */
 do
  NextLine = substr(NextLine,2)  /* skip the first character */
 end

 call lineout MailFile,NextLine  /* write the line to the file */

 InBytes = InBytes + length(NextLine) + 2  /* up the byte count */

 if (Global.ShowProgress) then  /* if we want stats */
 do
  LastTime = ProgressBar('Bytes retrieved = ',InBytes,ByteCount,time('E'),LastTime)  /* show the status */
 end

 NextLine = SocketGetLine(Socket)  /* get the next line */

end

Elapsed = time('E')  /* store the elapsed time */

if (InBytes < ByteCount) then  /* if we have fewer bytes than expected, we may have a rogue message on our hands */
do
 call lineout MailFile,'[Warning: message terminator arrived sooner than expected'  /* insert a warning line in the message */
 call lineout MailFile,' no. of bytes expected = '||ByteCount  /* insert a warning line in the message */
 call lineout MailFile,' no. of bytes received = '||InBytes||']'  /* insert a warning line in the message */
end

call FileClose MailFile  /* close the file */

if (Global.ShowProgress) then  /* if we wanted stats */
do
 call ProgressBar 'Bytes retrieved = ',InBytes,ByteCount,0,-1  /* show the last status and clear the status display */
 call syscurstate 'ON'  /* switch the cursor back on */
end

if (NextLine >< '.') then  /* if we did not get to the termination sequence */
do
 call AddError 'Cannot retrieve message (server no. '||Index||') from server'  /* report */
 call SysFileDelete MailFile  /* remove the file */
 return ''  /* return with no success */
end

call LogAction 'Retrieved message (server no. '||Index||'; '||ByteCount||' bytes; '||format(ByteCount / Elapsed,,0)||' bytes/second)'  /* report */

return MailFile  /* end of POP3RetrieveMessage */

/**********************************************************************/
POP3SkipMess:  procedure expose Global.  /* skips a message on the POP3 server (i.e. does nothing; just reports) */
/**********************************************************************/

parse arg Index  /* get the argument */

call LogAction 'Skipping message (server no. '||Index||')'  /* report */

return 1  /* end of POP3SkipMess */

/**********************************************************************/
ProgressBar: procedure expose Global.  /* displays a byte counter, timer and progress bar, and optionally erases the lot */
/**********************************************************************/

parse arg Text,CurCount,TotalCount,ElapsedTime,LastTime  /* get the arguments */

if (((ElapsedTime - LastTime) > 0.5) | (ElapsedTime = 0)) then  /* if more than half a second went by since the last update, or if we are just starting */
do

 CurRow = word(syscurpos(),1)  /* get the current row number */
 ScreenWidth = word(systextscreensize(),2)  /* get the screen width */

 if (TotalCount >< '') then  /* if we have a maximum byte count */
 do

  if (CurRow = word(systextscreensize(),1) - 1) then  /* if we are at the bottom row of the screen */
  do
   call lineout 'CON:',''  /* skip to the next line to move the existing text up one line */
   CurRow = CurRow - 1  /* position the byte counter a row up */
   call syscurpos CurRow,0  /* reposition the cursor */
  end

  CountLength = length(TotalCount)  /* the maximum length of the byte count string */
  Filler = '0'  /* pad with zeroes */

 end
 else  /* if we have no maximum byte count */
 do
  CountLength = 9  /* use this (enough for 999,999,999 bytes) */
  Filler = ' '  /* pad with blanks */
 end

 call charout 'CON:',Text||right(CurCount,CountLength,Filler)  /* show the current byte count */

 if (TotalCount >< '') then  /* if we have a max. byte count */
 do

  call charout 'CON:','/'||TotalCount||'  Time left = '  /* report */

  if (CurCount = 0) then  /* if we have no bytes yet, we are just starting out */
  do
   call charout 'CON:','__:__:__'  /* report nothing */
  end
  else  /* if we are already under way */
  do
   SecsLeft = format((TotalCount - CurCount) * ElapsedTime / CurCount,,0)  /* calculate the number of whole seconds left */
   HoursLeft = right(SecsLeft % 3600,2,'0')  /* calculate the hours left */
   SecsLeft = SecsLeft // 3600  /* calculate the seconds left without the hours */
   MinsLeft = right(SecsLeft % 60,2,'0')  /* calculate the minutes left */
   SecsLeft = right(SecsLeft // 60,2,'0')  /* calculate the seconds left without the minutes */
   call charout 'CON:',HoursLeft||':'||MinsLeft||':'||SecsLeft  /* report the time left */
  end

  if (ScreenWidth > 20) then  /* if we have room for a progress bar */
  do

   if (CurCount <= TotalCount) then  /* if we are still within a useful range */
   do
    ProgCount = format((CurCount / TotalCount) * (ScreenWidth - 4),,0)  /* the number of progress blocks to show on the progress bar */
    call syscurpos CurRow + 1,0  /* move to the next row */
    call charout 'CON:','  '||copies(d2c(178),ProgCount)||copies(d2c(176),ScreenWidth - 4 - ProgCount)  /* show the progress bar */
   end

  end

 end

 if (LastTime < 0) then  /* if we have a negative value for LastTime, we want a cleanup */
 do

  call syssleep 0.1  /* wait a bit to show the progress display */
  call syscurpos CurRow,0  /* return the cursor to its original position */
  call charout 'CON:',copies(' ',ScreenWidth - 1)  /* erase the byte counter line */

  if (ScreenWidth > 20) then  /* if we have a progress bar */
  do
   call syscurpos CurRow + 1,0  /* move to the start of the progress bar line */
   call charout 'CON:',copies(' ',ScreenWidth - 1)  /* erase the line */
  end

 end
 else  /* if we have a normal value for the last time */
 do
  LastTime = ElapsedTime  /* make sure we return the elapsed time so the last time counter gets updated */
 end

 call syscurpos CurRow,0  /* return the cursor to its original position */

end

return LastTime  /* end of ProgressBar */

/**********************************************************************/
ProgSpecPut: procedure expose Global.  /* attaches a program spec EA to a file */
/**********************************************************************/

parse arg MessFile  /* get the argument */

EALength = reverse(right(d2c(length(Global.ProgSpec)),2,'00'x))  /* set the EA length */
AttDirEA = 'FDFF'x||EALength||Global.ProgSpec  /* prepare the attachments dir EA */

if (sysputea(MessFile,'RXMLPROGSPEC',AttDirEA) >< 0) then  /* if we cannot set the EA */
do
 call AddError 'Cannot link program spec to message file'  /* report */
 return 0  /* return with an error */
end

return 1  /* end of ProgSpecPut */

/**********************************************************************/
RegisterMessage: procedure expose Global.  /* registers incoming messages, optionally moving them to a special folder and setting the folder's icon */
/**********************************************************************/

parse arg MessFile  /* get the argument */

if (MessFile = '') then  /* if we have none */
do
 MessFile = Global.ProcFile  /* get the file name to process */
end

if (\MessageContents(MessFile)) then  /* if we cannot get get the file contents */
do
 return 0  /* quit with an error */
end

DestDir = MoveMessageDir(MessFile,Global.ControlRegi,1)  /* see if the message needs to be moved to anywhere special */

if (DestDir = '') then  /* if we get no special destination */
do
 DestDir = Global.InDir  /* simply use the In dir */
end

if (\DirCheck(DestDir,1)) then  /* if the destination dir does not exist */
do
 DestDir = ProgDir  /* use the program dir */
end

MailFile = TempFileName('REGISTER',DestDir)  /* get a unique file name */

if (MailFile = '') then  /* if we get no file name */
do
 call AddError 'Cannot create mail file'  /* report */
 return 0  /* return with an error */
end

NamePart = filespec('N',MailFile)  /* extract the name part */
call syscreateobject 'CWMailFile',NamePart,DestDir,,'FAIL'  /* create a CWMAILFile class object in the destination dir using the name part */

call HideObject MailFile  /* make the file invisible */

if (\FileOpen(MailFile)) then  /* if we cannot open the new file for writing */
do
 call sysdestroyobject MailFile  /* get rid of the now useless file object */
 return 0  /* quit with an error */
end

if (\FileOpen(MessFile,'READ')) then  /* if we cannot open the original file for reading */
do
 call sysdestroyobject MailFile  /* get rid of the now useless file object */
 return 0  /* quit with an error */
end

if (charout(MailFile,charin(MessFile,1,chars(MessFile))) > 0) then  /* if we can not write the original message contents to the new mail file */
do
 call sysdestroyobject MailFile  /* get rid of the now useless file object */
 return 0  /* quit with an error */
end

call FileClose MailFile  /* close the new mail file */
call FileClose MessFile  /* close the original mail file */
call sysfiledelete MessFile  /* get rid of the original file */

call MessageSettings MailFile,'10000000','CHANGE'  /* set the incoming message status: "received & unread" mail message status, no attachments, and complete with unset bits */
NewName = MakeTitle(MailFile,1,0,0,1)  /* get the new title and insert warnings */

call UnhideObject MailFile  /* make the file visible again */

call syssetobjectdata MailFile,'REXXMAILREFRESH=YES;REXXMAILATTACHMENT=No;TITLE='||NewName  /* set the file attributes */

if (pos(translate(Global.MessagesDir),translate(DestDir)) = 1) then  /* if the folder is a RexxMail user folder */
do
 call syssetobjectdata DestDir,'ICONFILE='||Global.IconDir||'\foldred0.ico'  /* set the normal Mail Waiting folder icon */
 call syssetobjectdata DestDir,'ICONNFILE=1,'||Global.IconDir||'\foldred1.ico'  /* set the animated Mail Waiting folder icon */
end

return 1  /* end of RegisterMessage */

/**********************************************************************/
RunCommand: procedure expose Global.  /* runs an external command if necessary */
/**********************************************************************/

parse arg CommLine,CommFile,CommDir  /* get the arguments */

if (CommLine = '') then  /* if we have no command definition */
do
 return ''  /* quit with nothing */
end

if (CommFile >< '') then  /* if we have a file spec */
do

 if (pos('%N',CommLine) > 0) then  /* if we have a file name placeholder */
 do
  parse var CommLine FirstBit '%N' LastBit  /* look for the filename macro in the command line definition */
  FirstBit = strip(FirstBit,'T','"')  /* remove any trailing " */
  LastBit = strip(LastBit,'L','"')  /* remove any leading " */
  CommLine = FirstBit||'"'||CommFile||'"'||LastBit  /* this is the complete command */
 end
 else  /* if not */
 do
  CommLine = CommLine||' "'||CommFile||'"'  /* this is the complete command */
 end

end

call LogAction 'Command configuration option parsed as:'||Global.CRLF||,  /* report */
               '  '||CommLine  /* report */

if (CommDir >< '') then  /* if we have a command directory name */
do

 OrgDir = directory()  /* store the current directory name */

 if (directory(CommDir) = '') then  /* if we cannot change to the command directory */
 do
  call AddError 'Cannot change to "'||CommDir||'"'  /* report */
  return 0  /* and quit */
 end

end

signal off Error  /* we'll handle errors ourselves */
signal off Failure  /* we'll handle failures ourselves */

address cmd CommLine  /* run the command */

signal on Error  /* back to where we were */
signal on Failure  /* back to where we were */

if (RC = '') then  /* if we get nothing in return */
do
 RC = 0  /* use this */
end

if (CommDir >< '') then  /* if we have a command directory name */
do

 if (directory(OrgDir) = '') then  /* if we cannot change back to the original directory */
 do
  call AddError 'Cannot change back to "'||OrgDir||'"'  /* report */
 end

end

return RC  /* end of RunCommand */

/**********************************************************************/
SendMessage: procedure expose Global.  /* sends an outgoing message */
/**********************************************************************/

parse arg Bare  /* get the argument */

Bare = (Bare = 1)  /* 1 = TRUE */

MessFile = Global.ProcFile  /* get the name of the file to send */

if (\FileCheck(MessFile)) then  /* if the file no longer exists */
do
 return 0  /* quit with no result */
end

if (\Bare) then  /* unless we are sending a bare message */
do

 if (\MessageSettings(MessFile,'0*0*****','MATCH')) then  /* if the file is not outgoing and unprocessed */
 do
  return 0  /* quit with no result */
 end

 if (\Global.SendIfNotReady) then  /* unless we are sending regardless of a "ready" marker */
 do

  if (\MessageSettings(MessFile,'*1******','MATCH')) then  /* if the file is not O.K. for processing */
  do
   return 0  /* quit with no result */
  end

 end

end

MessSett = MessageSettings(MessFile)  /* get the message settings */

if (\Bare) then  /* unless we're sending a bare message */
do
 call HideObject MessFile  /* make the file invisible and record the settings */
end

if (\MessageContents(MessFile)) then  /* if we cannot get the message file contents */
do
 return 0  /* quit */
end

SendFile = MakeSendFile(MessFile)  /* try to get a sendable file */

Sent = 0  /* nothing has been sent yet */

if (SendFile >< '') then  /* if we get back a sendable file name */
do

 call RunCommand Global.RunBeforeSend  /* see if we can run an external command before sending */

 Sent = (RunCommand(Global.RunSend,SendFile) >< '')  /* see if we can run an external send command */

 if (Sent = 0) then  /* if the external command did not work, i.e. does not exist */
 do
  Sent = SMTPSendMessage(SendFile)  /* use the internal routine to send the file */
 end

 call RunCommand Global.RunAfterSend  /* see if we can run an external command after sending */

end

if (Sent) then  /* if the message was sent OK */
do

 if (\Bare) then  /* if the message was sent "not bare" */
 do

  AttDir = AttDirGet(MessFile,0)  /* get the attachments folder name, if any (do not create it) */

  if (AttDir >< '') then  /* if we had attachments */
  do
   call DeleteDir AttDir  /* get rid of the attachments folder */
  end

  call sysfiledelete MessFile  /* get rid of the original message file */
  Global.Hidden.0 = Global.Hidden.0 - 1  /* move the hidden objects counter back one */

  call MessageSettings SendFile,left(MessSett,1)||'01'||right(MessSett,5),'CHANGE'  /* new sent mail message setting: Processed and Not O.K. for processing */

  NewDir = ''  /* the message has not been moved yet */
  MessDir = strip(filespec('D',MessFile)||filespec('P',MessFile),'T','\')  /* get the file path and remove the trailing backslash */

  if (translate(MessDir) = translate(Global.OutDir)) then  /* if the file is in the Out folder */
  do
   NewDir = (MoveMessage(SendFile,Global.ControlSend,0))  /* see if the send file was moved to another folder */
  end

  if (NewDir = '') then  /* if the send file is still in the temp dir */
  do

   if (sysmoveobject(SendFile,MessDir)) then  /* if we can move the send file to the original message folder */
   do
    NewDir = MessDir  /* the new dir */
   end
   else  /* if we cannot move the send file */
   do
    NewDir = Global.TempDir  /* use the old file dir */
    call AddError 'Cannot restore sent file to Out folder; see Temp folder'  /* report */
   end

  end

  NewFile = NewDir||'\'||filespec('N',SendFile)  /* the new file name */
  call SetTitle NewFile  /* set the file title */

 end

end

else  /* if an error occurred */

do

 if (\Bare) then  /* unless we're sending "bare" */
 do
  call UnhideObject MessFile  /* make the file visible again */
 end

end

call lineout 'CON:',''  /* skip a line on the display */

return Sent  /* end of SendMessage */

/**********************************************************************/
ServerConnect: procedure expose Global.  /* connects to a server */
/**********************************************************************/

parse arg Server,Port,MaxAttempts  /* get the arguments */

if (MaxAttempts = 0) then  /* if no attempts are to be made */
do
 return ''  /* return with no socket */
end

if (sockinit() >< 0) then  /* if we cannot initialize the socket buffers */
do
 call AddError 'Socket initialization error'  /* report */
 return ''  /* return with no socket */
end

Socket = socksocket('AF_INET','SOCK_STREAM','IPPROTO_TCP')  /* create a socket for TCP protocol */

if (Socket < 0) then  /* if we did not get a socket */
do
 call AddError 'Cannot get socket'  /* report */
 return ''  /* return with no socket */
end

if (verify(Server,'1234567890.','NOMATCH') = 0) then  /* if we have nothing but digits and dots, assume it was a dotted address */
do
 Address.!addr = Server  /* use the server address as it is */
end
else  /* if not, asssume it was a mnemonic name */
do

 call LogAction 'Resolving host name "'||Server||'"'  /* report */

 Attempts = 0  /* no try yet */
 GotHost = 0  /* no host contacted yet */

 do while ((\GotHost) & (Attempts < MaxAttempts))  /* go on until we get a result, or get timed out */

  GotHost = sockgethostbyname(Server,'Host.!')  /* if we can get the SMTP server's IP address by name, all is well */

  if (\GotHost) then  /* if we did not get the right result */
  do
   call syssleep 1  /* wait a second */
   Attempts = Attempts + 1  /* up the Attempts counter */
   call LogAction 'Retrying'  /* report */
  end

 end

 if (\GotHost) then  /* if we did not get the right result */
 do
  call AddError 'Cannot resolve host name "'||Server||'"'  /* report */
  return ''  /* return with no socket */
 end

 Address.!addr = Host.!addr  /* define the connect address */

end

Address.!family = 'AF_INET'  /* define the connection type */
Address.!port = Port  /* define the port to use */

if (Server >< Address.!addr) then  /* if we have a menomnic name in addition to a dotted address */
do
 AddMessage = '"'||Server||'" at '  /* add this to the next message */
end
else  /* if not */
do
 AddMessage = ''  /* add nothing */
end

call LogAction 'Connecting to host '||AddMessage||Address.!addr||' through port '||Port  /* report */
Attempts = 0  /* no attempts yet */
GotConn = 0  /* no connection yet */

do while ((\GotConn) & (Attempts < MaxAttempts))  /* go on until we get a result, or get timed out */

 GotConn = (sockconnect(Socket,'Address.!') = 0)  /* if we can open the socket connection, all is well */

 if (\GotConn) then  /* if we did not get the right result */
 do
  call syssleep 1  /* wait a second */
  Attempts = Attempts + 1  /* up the attempts counter */
  call LogAction 'Retrying'  /* report */
 end

end

if (\GotConn) then  /* if we did not get the right result */
do
 call AddError 'Cannot connect to host "'||Server||'"'  /* report */
 return ''  /* return with no success */
end

return Socket  /* end of ServerConnect */

/**********************************************************************/
ServerDisconnect: procedure expose Global.  /* disconnect from a server */
/**********************************************************************/

parse arg Socket,Server  /* get the arguments */

call LogAction 'Disconnecting from "'||Server||'"'  /* report */

if (\SocketSendLine(Socket,'QUIT','221 +OK')) then  /* if we do not get 221 or +OK in return when we send this line */
do
 call AddError 'Error while disconnecting from "'||Server||'"'  /* report */
end

call SocketClose Socket  /* close the socket */

return 1  /* end of ServerDisconnect */

/**********************************************************************/
SetMessageType: procedure expose Global.  /* sets message type attributes */
/**********************************************************************/

parse upper arg Switch  /* get the argument, in upper case */

if (Switch = '') then  /* if we have no switch */
do
 return 0  /* quit */
end

MessFile = Global.ProcFile  /* the file name to process */
MessSettings = (MessageSettings(MessFile))  /* get the message file's status settings */
NewName = ''  /* no new name yet */

if (substr(MessSettings,1,1) = 0) then  /* if it is an outgoing message */
do

 if (substr(MessSettings,3,1) = 0) then  /* if the message has not yet been sent */
 do

  if (Switch = 'SETASCIIQP') then  /* if we are to toggle the ASCII/Q-P setting */
  do

   if (substr(MessSettings,4,1) = 1) then  /* if the ASCII flag has already been set */
   do
    Switch = 'SETQP'  /* reset it */
   end
   else  /* if the ASCII flag has not been set yet */
   do
    Switch = 'SETASCII'  /* set it */
   end

  end

  else  /* if we are not to toggle the ASCII/Q-P setting */

  do

   if (Switch = 'SETOKNOTOKTOSEND') then  /* if we are to toggle the new/sendable setting */
   do

    if (substr(MessSettings,2,1) = 1) then  /* if the sendable flag has already been set */
    do
     Switch = 'SETNOTOKTOSEND'  /* reset it */
    end
    else  /* if the sendable flag has not been set yet */
    do
     Switch = 'SETOKTOSEND'  /* set it */
    end

   end

  end

  if (Switch = 'SETOKTOSEND') then  /* if we are to set the "OK to send" flag */
  do

   if (\MessageContents(MessFile)) then  /* if we cannot get the message contents */
   do
    return 0  /* quit */
   end

   NewName = MakeTitle(MessFile,1,0,0,0)  /* get a new name for the mail file, and insert warnings */

   if (NewName = '') then  /* if have no new name */
   do
    call AddError 'Cannot check syntax of '||MessFile  /* report */
    return 0  /* quit */
   end

   if (pos(Global.Warning,NewName) > 0) then  /* if the message is not OK to send */
   do
    call AddError 'Syntax errors found in header of '||MessFile  /* report */
    return 0  /* quit */
   end

  end

 end

end

SwitchString = 'SETNEW SETNOTOKTOSEND SETOKTOSEND SETSENT SETASCII SETQP SETRECEIVED SETVIEWED'  /* the available types */
SettingsString = '00000000 000***** 010***** 001***** ***1**** ***0**** 10000000 101*****'  /* the corresponding settings strings */
SwitchPos = wordpos(Switch,SwitchString)  /* look for the switch in the switch string */

if (SwitchPos = 0) then  /* if the switch is not there */
do
 call AddError 'Invalid switch : "'||Switch||'"'  /* report */
 return 0  /* quit */
end

if (\UnhideObject(MessFile)) then  /* if we cannot make the file visible etc. */
do
 call AddError 'Cannot set file attributes'  /* report */
 return 0  /* quit */
end

Settings = word(SettingsString,SwitchPos)  /* get the settings that correspond with the message type */

if (\MessageSettings(MessFile,Settings,'CHANGE')) then  /* if we cannot attach the mail message settings */
do
 call AddError 'Cannot change file settings'  /* report */
 return 0  /* quit */
end

if (NewName >< '') then  /* if we have a new name */
do

 if (\syssetobjectdata(MessFile,'TITLE='||NewName)) then  /* if we cannot set the new title */
 do
  call AddError 'Cannot change file title'  /* report */
  return 0  /* and quit */
 end

end

return 1  /* end of SetMessageType */

/**********************************************************************/
SetTitle: procedure expose Global.  /* sets message title */
/**********************************************************************/

parse arg MessFile  /* look for an argument */

if (MessFile = '') then  /* if there is none */
do
 MessFile = Global.ProcFile  /* the file name to process */
end

if (\MessageSettings(MessFile,,'CHECK')) then  /* if it is not a RexxMail message */
do
 return 0  /* quit */
end

if (\MessageContents(MessFile)) then  /* if we cannot get the file contents */
do
 return 0  /* quit */
end

NewName = MakeTitle(MessFile,0,0,0,MessageSettings(MessFile,'1*******','MATCH'))  /* get a new name for the mail file, do not insert warnings */
Result = (syssetobjectdata(MessFile,'TITLE='||NewName))  /* if we can set the title, all is well */

if (\Result) then  /* if all is not well */
do
 call AddError 'Cannot set title'  /* report */
end

return Result  /* end of SetTitle */

/**********************************************************************/
SMTPSendMessage: procedure expose Global.  /* sends mail to an SMTP "smart host" mail relay */
/**********************************************************************/

parse arg SendFile  /* get the argument */

/**********************************************************************/
/* See if we have all the arguments we really need                    */
/**********************************************************************/

if (Global.SMTPServer = '') then  /* if we have no server name */
do
 call AddError 'Missing configuration entry: SMTPServer'  /* report */
 return 0  /* and quit with no success */
end

/**********************************************************************/
/* Get the message contents                                           */
/**********************************************************************/

if (\FileOpen(SendFile,'READ')) then  /* if we cannot open the file for reading */
do
 return 0  /* quit with no success */
end

MessCont = charin(SendFile,1,chars(SendFile))  /* get the sendable message file contents */
call FileClose SendFile  /* close the message file */
parse var MessCont MessageHead (Global.EmptyLine) MessageBody  /* split the message into its header and body */

/**********************************************************************/
/* Get some header info from the message header                       */
/**********************************************************************/

Sender = ''  /* start with nothing */
Recipients = ''  /* we have no recipients yet */
RecipientsTo = ''  /* we have no To recipients yet */
RecipientsCc = ''  /* we have no Cc recipients yet */
RecipientsBcc = ''  /* we have no Bcc recipients yet */
Subject = ''  /* we have no subject yet */
LookFold = 0  /* no need to start looking for folded lines yet */
GotBcc = 0  /* we are not in a "Bcc:" bit yet */
Header = MessageHead  /* copy the header contents */
MessageHead = ''  /* start again */
RecpHead = ''  /* start with nothing */

do while (Header >< '')  /* as long as we have header contents left */

 parse var Header NextLine (Global.CRLF) Header  /* get the next line */
 NextLine = translate(NextLine,d2c(32),d2c(9))  /* convert TABs to spaces */

 if ((LookFold) & (left(NextLine,1) = d2c(32))) then  /* if we are looking for folded lines, and the line starts with whitespace */
 do

  NewRecipient = AddressFormat(NextLine,,0)  /* extract the bare address from the line, no check */

  if (wordpos(NewRecipient,Recipients) = 0) then  /* if the address is not in the list */
  do
   Recipients = Recipients||' '||NewRecipient  /* add it */
  end

 end
 else  /* if we aren't looking for folded lines, or if the line is not a folded bit */
 do

  FirstWord = translate(word(NextLine,1))  /* get the keyword in upper case */

  if (wordpos(FirstWord,'TO: CC: BCC:') > 0) then  /* if it is a recipient line */
  do

   NewRecipient = AddressFormat(subword(NextLine,2),,0)  /* extract the bare e-mail addresses from the second and any following words, no check */

   if (wordpos(NewRecipient,Recipients) = 0) then  /* if the address is not in the list */
   do
    Recipients = Recipients||' '||NewRecipient  /* add it */
   end

   LookFold = 1  /* look out for folded lines that may follow */
   GotRec = 1  /* we have a recipient line */
   GotBcc = (FirstWord = 'BCC:')  /* if it is a "Bcc:" line, set a flag */

   select  /* do one of the following */

    when (FirstWord = 'TO:') then  /* if it is this kind of recipient */
    do
     RecipientsTo = RecipientsTo||' '||NewRecipient  /* add it to what we have */
    end

    when (FirstWord = 'CC:') then  /* if it is this kind of recipient */
    do
     RecipientsCc = RecipientsCc||' '||NewRecipient  /* add it to what we have */
    end

    when (FirstWord = 'BCC:') then  /* if it is this kind of recipient */
    do
     RecipientsBcc = RecipientsBcc||' '||NewRecipient  /* add it to what we have */
    end

    otherwise  /* if none of the above (which should not occur) */
    do
     nop  /* nothing */
    end

   end

  end

  else  /* if this is not a recipient line */

  do

   if (translate(word(NextLine,1)) = 'FROM:') then  /* if it is a "From:" line */
   do

    if (Sender = '') then  /* if we have no sender yet */
    do
     Sender = AddressFormat(subword(NextLine,2),,0)  /* extract the bare e-mail address from the second and any following words, no check */
    end
    else  /* if we already have a sender line */
    do
     call AddError 'Invalid header line: '||NextLine  /* report */
     return 0  /* and quit */
    end

   end

   if (translate(word(NextLine,1)) = 'SENDER:') then  /* if it is a "Sender:" line */
   do
    call AddError 'Invalid header line: '||NextLine  /* report */
    return 0  /* and quit */
   end

   if (translate(word(NextLine,1)) = 'SUBJECT:') then  /* if this is a subject line */
   do

    if (Subject = '') then  /* if we have no subject yet */
    do

     Subject = subword(NextLine,2)  /* the rest of the line must be the subject */

     if (length(Subject) > 60) then  /* if there is too much of it */
     do
      Subject = left(Subject,60)||'...'  /* trim it and add an ellipsis */
     end

    end
    else  /* if we already have a subject line */
    do
     call AddError 'Invalid header line: '||NextLine  /* report */
     return 0  /* and quit */
    end

   end

   LookFold = 0  /* no need to look (further) for folded lines */
   GotRec = 0  /* we are not in a recipient line */
   GotBcc = 0  /* it is not a "Bcc:" line */

  end

 end

 if (\GotBcc) then  /* if we are not in a "Bcc:" bit */
 do

  if (GotRec) then  /* if this is a recipient line */
  do
   RecpHead = RecpHead||NextLine||Global.CRLF  /* add the line to the new recipient header lines */
  end
  else  /* if it is not a recipient line */
  do
   MessageHead = MessageHead||NextLine||Global.CRLF  /* add the line to the new normal header lines */
  end

 end

end

if (Recipients = '') then  /* if we have no recipients */
do
 call AddError 'Missing recipient'  /* report */
 return 0  /* and quit */
end

if (Sender = '') then  /* if we have no sender */
do
 call AddError 'Missing sender'  /* report */
 return 0  /* and quit */
end

if (Sender = Global.Address) then  /* if the sender is the local user */
do
 SenderLine = ''  /* no sender line needed */
 RealSenderBit = ''  /* nothing here either */
end
else  /* if the two do not two match */
do
 SenderLine = 'Sender: "'||Global.Name||'" <'||Global.Address||'>'||Global.CRLF  /* construct a "Sender:" line */
 RealSenderBit = ' ('||Global.Address||')'  /* we will insert this later in screen and log messages */
end

/**********************************************************************/
/* Expand the list of recipients                                      */
/**********************************************************************/

NewRecipients = ''  /* start with nothing */

do while (Recipients >< '')  /* as long as we have recipients left */

 parse var Recipients NextAddress Recipients  /* get the next address from the list */

 if (pos(':;',NextAddress) = 0) then  /* if the address does not contain a group indicator */
 do
  NewRecipients = NewRecipients||' '||NextAddress  /* just add the address to the new list */
 end

 else  /* if the address contains a group indicator */

 do

  parse var NextAddress ListName ':;'.  /* get the list name */
  ListFile = Global.ListDir||'\'||ListName  /* the list file to look for */

  if (FileCheck(ListFile)) then  /* if there is a mail list file of that name */
  do

   call LogAction 'Including mailing list file "'||ListFile||'"'  /* report */

   if (\FileOpen(ListFile,'READ')) then  /* if we cannot open the mailing list file for reading */
   do
    return 0  /* quit with an error */
   end

   ListCont = charin(ListFile,1,chars(ListFile))  /* get the contents of the list file */

   do while (ListCont >< '')  /* as long as we have content left */

    parse var ListCont InLine (Global.CRLF) ListCont  /* get the next line */

    if (InLine >< '') then  /* if it is not an empty line */
    do

     if (left(InLine,1) >< '#') then  /* if it is not a comment */
     do

      AddRecipients = AddressFormat(InLine,,1)  /* get a properly formatted and checked bare address */

      if (pos(Global.Warning,AddRecipients) > 0) then  /* if the new address contains a warning marker */
      do
       call AddError 'Malformed recipient address "'||InLine||'" in "'||ListFile||'"'  /* report */
       return 0  /* and quit with an error */
      end

      else  /* if all went well */

      do
       NewRecipients = NewRecipients||' '||AddRecipients  /* add it to the expanded list */
      end

     end

    end

   end

   call FileClose ListFile  /* close the mailing list file */

  end

  else  /* if there is no mail list file of that name */

  do

   ListFile = Global.AddrDir||'\'||ListName||'\*'  /* the list file spec to look for */

   if (\FileCheck(ListFile,1)) then  /* if the list dir contains nothing, or does not exist */
   do
    return 0  /* quit with an error */
   end

   else  /* if all is well */

   do

    call LogAction 'Including mail group files "'||ListFile||'"'  /* report */
    call sysfiletree ListFile,'Files.','FO'  /* look for (template) files in the list dir */

    do Index = 1 to Files.0  /* take each one */

     if (\MessageContents(Files.Index,'Global.ListHead','Global.ListBody')) then  /* if we cannot get the unwrapped header contents (and the body, if any) */
     do
      return 0  /* and quit with an error */
     end

     MessRecipients = GetHeaderEntry(Global.ListHead,'TO:')  /* get the recipients from the header */
     AddRecipients = AddressFormat(MessRecipients,,1)  /* get a list of properly formatted and checked bare addresses */

     if (pos(Global.Warning,AddRecipients) > 0) then  /* if the new addresses contains a warning marker */
     do
      call AddError 'Malformed recipient address "'||AddRecipients||'" in "'||Files.Index||'"'  /* report */
      return 0  /* and quit with an error */
     end

     else  /* if all went well */

     do
      NewRecipients = NewRecipients||' '||AddRecipients  /* add it to the expanded list */
     end

    end

   end

  end

 end

end

/**********************************************************************/
/* Look for the host name  name to pass to the server                 */
/**********************************************************************/

MessID = GetHeaderEntry(MessageHead,'MESSAGE-ID:')  /* get the message ID */

if (MessID = '') then  /* if there is no message ID */
do
 call AddError 'Message ID not found'  /* report */
 return 0  /* and quit */
end

parse var MessID . '@' HostName '>'  /* get the host name */

/**********************************************************************/
/* Connect to the server                                              */
/**********************************************************************/

parse var Global.SMTPServer Global.SMTPServer ':' PortNumber  /* look for a port number */

if (PortNumber = '') then  /* if there is none */
do
 PortNumber = 25  /* use this (SMTP) */
end

Socket = ServerConnect(Global.SMTPServer,PortNumber,Global.SMTPAttempts)  /* get the socket number for a server connection through the specified or default port */

if (Socket = '') then  /* if we have no socket */
do
 return 0  /* quit without success */
end

Global.Socket.SocketBuffer = ''  /* start with an empty socket buffer */

if (\SocketAnswer(Socket,220)) then  /* if we don't get a 220 return code */
do
 call AddError 'No response from '||Global.SMTPServer  /* report */
 call SocketClose Socket  /* close the socket */
 return 0  /* return with no success */
end

if (\SocketSendLine(Socket,'HELO '||HostName,250)) then  /* if don't get 250 in return when we send this line */
do
 call AddError HostName||' not accepted'  /* report */
 call ServerDisconnect Socket,Global.SMTPServer  /* disconnect */
 return 0  /* return with no success */
end

/**********************************************************************/
/* Send the message(s)                                                */
/**********************************************************************/

LogRecipients = strip(NewRecipients)  /* copy the list of recipients, and get rid of superfluous blanks, for the log message later on */

if (\SocketSendLine(Socket,'MAIL FROM:<'||Global.Address||'>',250)) then  /* if we don't get 250 in return when we send this line */
do
 call AddError 'MAIL FROM:<'||Global.Address||'> not accepted by server'  /* report */
 call ServerDisconnect Socket,Global.SMTPServer  /* disconnect */
 return 0  /* quit without success */
end

LogText = 'Sending message from '||Sender||RealSenderBit||Global.CRLF||,  /* start a log text with this */
          '                  to '  /* and this */

RecipientCount = 0  /* we have no successful recipients yet */

do while (NewRecipients >< '')  /* as long as we have recipients left in the "normal" list */

 parse var NewRecipients NextRecipient NewRecipients  /* get the next recipient from the list */

 if (RecipientCount > 0) then  /* if we are not on the first line anymore */
 do
  LogText = LogText||Global.CRLF||'                     '  /* add a new line and an indent to the log text */
 end

 if (SocketSendLine(Socket,'RCPT TO:<'||NextRecipient||'>',250 251)) then  /* if we get 250 or 251 in return when we send this line */
 do
  RecipientCount = RecipientCount + 1  /* up the counter */
  LogText = LogText||NextRecipient  /* add the recipient to the log text */
 end
 else  /* if we failed to enter the recipient */
 do
  call AddError 'RCPT TO:<'||NextRecipient||'> not accepted by server'  /* report */
  call ServerDisconnect Socket,Global.SMTPServer  /* disconnect */
  return 0  /* quit without success */
 end

end

call LogAction LogText  /* report */

SMTPFile = TempFileName('SMTP')  /* get a temp file name */

if (\FileOpen(SMTPFile)) then  /* if we cannot open the file */
do
 call AddError 'Cannot write SMTP file'  /* report */
 return 0  /* and quit */
end

DataBlock = RecpHead||,  /* the complete block of data to be sent contains the remaining recipients header */
            MessageHead||,  /* the rest of the original message header */
            SenderLine||,  /* the sender line (if any) */
            'X-Mailer: '||Global.BuildMess||Global.CRLF||,  /* an "X-Mailer" entry to the header to show the build number */
            '          '||Global.CopyRight||Global.EmptyLine||,  /* a copyright line added to the X-mailer entry */
            MessageBody  /* and ends with the message body */

if (charout(SMTPFile,DataBlock) > 0) then  /* if we cannot write all the text to the SMTP file */
do
 call AddError 'Cannot write SMTP file'  /* report */
 return 0  /* and quit */
end

call FileClose SMTPFile  /* close the SMTP file */

if (\SocketSendLine(Socket,'DATA',354)) then  /* if we don't get 354 in return when we send this line */
do
 return 0  /* return with an error */
end

if (\FileOpen(SMTPFile,'READ')) then  /* if we cannot open the file for reading */
do
 call AddError 'Cannot open SMTP file'  /* report */
 return 0  /* and quit */
end

ByteCount = length(DataBlock)  /* get the length of the block to send  */

if (Global.ShowProgress) then  /* if we want stats */
do
 call syscurstate 'OFF'  /* switch off the cursor */
 OutBytes = 0  /* we have sent nothing yet */
 LastTime = ProgressBar('Bytes sent = ',OutBytes,ByteCount,0,0)  /* start a progress bar */
end

call time 'R'  /* reset the timer */
call syssleep 0.0001  /* wait a fraction of a second to start the timer */

StatusOK = 1  /* all is well */

do while ((lines(SMTPFile)) & (StatusOK))  /* as long as we have data to send and all is well */

 DataLine = linein(SMTPFile)  /* get the next line */

 if (DataLine <> '') then  /* if it is not an empty line */
 do

  if (left(DataLine,1) = '.') then  /* if the line starts with a period */
  do
   DataLine = '.'||DataLine  /* stick an extra period on the front */
  end

 end

 StatusOK = SocketSendLine(Socket,DataLine)  /* if we can send this to the socket, all is well */

 if (StatusOK) & (Global.ShowProgress) then  /* if we want stats and all is still well */
 do
  OutBytes = OutBytes + length(DataLine) + 2  /* up the line bytes count (plus the terminating CRLF) */
  LastTime = ProgressBar('Bytes sent = ',OutBytes,ByteCount,time('E'),LastTime)  /* show the status */
 end

end

Elapsed = time('E')  /* store the elpased time */

call FileClose SMTPFile  /* close the SMTP file */

if (Global.ShowProgress) then  /* if we wanted stats */
do
 call ProgressBar 'Bytes sent = ',OutBytes,ByteCount,0,-1  /* show the last status and erase the status display */
 call syscurstate 'ON'  /* switch the cursor back on */
end

if (\StatusOK) then  /* if we had an error */
do
 call AddError 'Error sending message to <'||NextRecipient||'>'  /* report */
 call ServerDisconnect Socket,Global.SMTPServer  /* disconnect */
 return 0  /* quit without success */
end

if (\SocketSendLine(Socket,'.',250)) then  /* if we don't get 250 in return when we send this line */
do
 return 0  /* quit */
end

if (RecipientCount = 1) then  /* if we have exactly 1 recipient */
do
 Plural = ''  /* we need no plural 's' */
end
else  /* if we have 0 or multiple recipients */
do
 Plural = 's'  /* we need a plural 's' */
end

call ServerDisconnect Socket,Global.SMTPServer  /* disconnect */
call LogAction 'Message sent to '||RecipientCount||' recipient'||Plural||' ('||ByteCount||' bytes; '||format((ByteCount / Elapsed),,0)||' bytes/second)'  /* report */
call SoundSignal Global.SignalSent  /* signal if required */

RecipientsTo = strip(RecipientsTo,'L',' ')  /* get rid of the leading space */
RecipientsCc = strip(RecipientsCc,'L',' ')  /* get rid of the leading space */
RecipientsBcc = strip(RecipientsBcc,'L',' ')  /* get rid of the leading space */

call LogMail '==>',Sender||RealSenderBit,,RecipientsTo,RecipientsCc,RecipientsBcc,Subject,MessID  /* log the message */

return 1  /* end of SMTPSendMessage */

/**********************************************************************/
SocketAnswer: procedure expose Global.  /* see if we got the right answer */
/**********************************************************************/

parse arg Socket,ReplyCodes  /* get the arguments */

Answer = ''  /* we have nothing yet */

do until (Answer >< '')  /* go on until we get a result */

 Answer = SocketGetLine(Socket)  /* get the next line from the socket (buffer) */

 if (Answer >< '') then  /* if we have something */
 do

  if (substr(Answer,4,1) = '-') then  /* if the fourth character is a hyphen, we have part of a multiline reply */
  do
   Answer = ''  /* dump this line */
  end

  else  /* if we have the final line of a multiline reply, or a single-line reply */

  do

   if (ReplyCodes >< '') then  /* if we have reply codes to match */
   do

    if (wordpos(word(Answer,1),ReplyCodes) > 0) then  /* if the first word of the answer is in of the reply codes we want */
    do
     Answer = 1  /* return success */
    end
    else  /* if the first word is not in the list of reply codes */
    do
     call AddError 'Error from server: '||subword(Answer,2)  /* report */
     Answer = 0  /* no success */
    end

   end

  end

 end

 else  /* if we get no answer */

 do
  Answer = 0  /* no success */
 end

end

return answer  /* end of SocketAnswer */

/**********************************************************************/
SocketClose: procedure expose Global.  /* closes the IP socket */
/**********************************************************************/

parse arg Socket  /* get the argument */

if (SockSoclose(Socket) >< 0) then  /* close the socket */
do
 call AddError 'Error closing socket'  /* report */
end

return 1  /* end of SocketClose */

/**********************************************************************/
SocketGetLine: procedure expose Global.  /* get a line from a socket and optionally see if we got the right answer */
/**********************************************************************/

parse arg Socket  /* get the argument */

do while (pos(Global.CRLF,Global.Socket.SocketBuffer) = 0)  /* as long as we have no CRLF in the buffer */

 if (sockrecv(Socket,'SocketData',8096) > 0) then  /* if we can get data from the socket */
 do
  Global.Socket.SocketBuffer = Global.Socket.SocketBuffer||SocketData  /* add the results to the buffer */
 end
 else  /* if we cannot get anything from the socket */
 do
  call AddError 'Connection closed'  /* report */
  return d2c(0)  /* and return a null byte */
 end

end

parse var Global.Socket.SocketBuffer NextLine (Global.CRLF) Global.Socket.SocketBuffer  /* take the bit before the CRLF from the buffer */

return NextLine  /* end of SocketGetLine */

/**********************************************************************/
SocketSendLine: procedure expose Global.  /* send a line to a socket */
/**********************************************************************/

parse arg Socket,DataToSend,ReplyCodes  /* get the argument */

Answer = 1  /* assume success */
DataToSend = DataToSend||Global.CRLF  /* add a CRLF to the data to be sent */
DataLength = length(DataToSend)  /* get the length of the data to be sent */

do while (DataLength > 0)  /* as long as we have data left to send */

 DataLength = socksend(Socket,DataToSend)  /* see how many bytes we can send */

 if (DataLength > 0) then  /* if we sent data */
 do
  DataToSend = substr(DataToSend,DataLength + 1)  /* get rid of the bit we sent */
  DataLength = length(DataToSend)  /* get the length of the remaining data to be sent */
 end
 else  /* if we sent no data or got an error */
 do
  call AddError 'Connection closed by server'  /* report */
  Answer = 0  /* all is not well */
 end

end

if ((Answer) & (ReplyCodes >< '')) then  /* if all went well and if we have reply codes to match */
do

 if (ReplyCodes = '?') then  /* if we want to see the answer itself */
 do
  ReplyCodes = ''  /* go on without the reply code */
 end

 Answer = SocketAnswer(Socket,ReplyCodes)  /* if we get the correct answer, all is well */

end

return Answer  /* end of SendString */

/**********************************************************************/
SoundSignal: procedure expose Global.  /* sound a signal (or not, as the case may be) */
/**********************************************************************/

if (\Global.Signals) then  /* if we want no signals at all */
do
 return 0  /* return with nothing */
end

parse arg SignalString  /* get the argument */

if (SignalString = '') then  /* if there is no signal string (which is O.K.) */
do
 return 0  /* return with nothing */
end

if (word(SignalString,1) = 'BEEP') then  /* if it starts with this, it is a beep sequence */
do

 SignalString = subword(SignalString,2)  /* use the rest */

 do while (SignalString >< '')  /* as long as we have something left */
  parse var SignalString NextSignal SignalString  /* get the next beep bit */
  interpret 'call beep '||NextSignal  /* and do it */
 end

end

else  /* if it is not a beep string, it must be an external command */

do
 call RunCommand SignalString  /* try to run the command */
end

return 1  /* end of SoundSignal */

/**********************************************************************/
SystemMessage: procedure expose Global.  /* sends a system message to the current user */
/**********************************************************************/

parse arg Subject,MessText  /* get the arguments */

MessFile = TempFileName('SYSTEM')  /* get a unique file name */

if (\FileOpen(MessFile)) then  /* if we cannot open the file */
do
 return 0  /* quit */
end

if (Global.Name = '') then  /* if we do not have a user name */
do
 Global.Name = 'RexxMail user'  /* use this */
end

if (Global.Address = '') then  /* if we do not have a user name */
do
 Global.Address = 'rexxmail-user@localhost'  /* use this */
end

call lineout MessFile,'Date: '||DateTimeRFC(1)  /* start with a date line, but do not contact a time server */
call lineout MessFile,'From: "RexxMail system messages" <rexxmail-messages@localhost>'  /* add a sender line */
call lineout MessFile,'To: "'||Global.Name||'" <'||Global.Address||'>'  /* add a recipient line */
call lineout MessFile,'Subject: '||Subject  /* add a subject line */
call lineout MessFile,''  /* add an empty line */
call charout MessFile,MessText  /* add the message text */
call FileClose MessFile  /* close the message file */
call RegisterMessage MessFile  /* register the message */

return 1  /* end of SystemMessage */

/**********************************************************************/
TempFileName: procedure expose Global.  /* returns a unique file name */
/**********************************************************************/

parse arg Extension,Location  /* get the arguments */

if (Location = '') then  /* if we have no location */
do
 Location = Global.TempDir  /* use the temp dir */
end
else  /* if we have a location */
do
 Location = strip(Location,'T','\')  /* remove any trailing backslash */
end

if (Extension >< '') then  /* if we have an extension */
do
 Extension = '.'||Extension  /* if needs a leading dot */
end

TempFile = systempfilename(Location||'\RXML????'||Extension)  /* this is the file name */

return TempFile  /* end of TempFileName */

/**********************************************************************/
TimeAttList: procedure expose Global.  /* gets a send time and attachment listing from an old-style 'sent' message file */
/**********************************************************************/

parse arg MessFile  /* get the argument */

AttList = ''  /* start with zilch */
call sysgetea MessFile,'.COMMENTS','Comments'  /* look for Comments */

if (Comments >< '') then  /* if we found comments */
do

 parse var Comments EAT 3 . 5 Count 7 Contents  /* get the bits we want */

 if (EAT = 'DFFF'x) then  /* if it is a multi-value, multi-type entry */
 do

  Count = c2d(reverse(Count))  /* restore the count value to the right order */

  do Index = 1 to Count  /* run through them all */

   parse var Contents EAT 3 Len 5 Contents  /* get the bits we want  */

   if (EAT = 'FDFF'x) then  /* if it is ASCII text */
   do
    Next = c2d(reverse(Len)) + 1  /* where the next entry starts */
    parse var Contents Text =(Next) Contents  /* get the bits we need */
    AttList = AttList||Text||Global.CRLF  /* add the text to the list */
   end

  end

 end

end

return AttList  /* end of TimeAttList */

/**************************************************************************/
Toolbar: procedure expose Global.  /* creates/deletes/opens a RexxMail toolbar */
/**************************************************************************/

parse arg Action,ToolbarName  /* get the argument */

if (ToolbarName >< '') then  /* if we have a name */
do
 ToolbarName = translate(ToolbarName,' ',d2c(9))  /* convert TABs to spaces */
 ObjectID = '<REXXMAIL_TOOLBAR_'||translate(translate(ToolbarName,'_',' '))||'>'  /* add the upper case name, with underscores instead of spaces, to the object ID */
end
else  /* if not */
do
 ToolbarName = 'RexxMail Toolbar'  /* use this */
 ObjectID = '<REXXMAIL_TOOLBAR>'  /* use this */
end

select  /* do one of the following */

 when (Action = 'TOOLBARCREATE') then  /* if we have this */
 do

  Settings = 'OBJECTID='||ObjectID||';'||,  /* the object ID */
             'LPACTIONSTYLE=OFF;'||,  /* no system action buttons */
             'LPCLOSEDRAWER=YES;'||,  /* close drawers after use */
             'LPDRAWERTEXT=NO;'||,  /* no text in drawers */
             'LPFLOAT=NO;'||,  /* do not float to top */
             'LPHIDECTLS=YES;'||,  /* hide the window controls */
             'LPSMALLICONS=NO;'||,  /* use normal icons */
             'LPTEXT=NO;'||,  /* no text on buttons */
             'LPVERTICAL=NO;'||,  /* show horizontal */
             'FPOBJECTS=<REXXMAIL_REFERENCE>'  /* include this object button */

  if (\syscreateobject('WPLaunchPad',ToolbarName,Global.MainDir,Settings,'FAIL')) then  /* if we cannot create the toolbar object */
  do
   call AddError 'Cannot create RexxMail toolbar: "'||ToolBarName||'"'  /* report */
   return 0  /* and return with no success */
  end

 end

 when (Action = 'TOOLBARDELETE') then  /* if we have this */
 do

  if (\sysdestroyobject(ObjectID)) then  /* if we cannot delete the toolbar */
  do
   call AddError 'Cannot delete RexxMail toolbar: "'||ToolbarName||'"'  /* report */
   return 0  /* return with no success */
  end

 end

 when (Action = 'TOOLBAROPEN') then  /* if we have this */
 do

  if (\sysopenobject(ObjectID,121,0)) then  /* if we cannot open the toolbar */
  do
   call AddError 'Cannot open RexxMail toolbar: "'||ToolbarName|'"'  /* report */
   return 0  /* return with no success */
  end

 end

 otherwise  /* this should not occur */
 do
  nop  /* nothing to do */
 end

end

return 1  /* end of Toolbar */

/**********************************************************************/
TrimLogFile: procedure expose Global.  /* rewrites a log file if necessary */
/**********************************************************************/

parse arg LogFile,KeepLines  /* get the arguments */

if (KeepLines = '') then  /* if we have an empty value here, we do not have to trim this log file (there is no size limit) */
do
 return 1  /* quit */
end

if (KeepLines = 0) then  /* if we have a zero value here, we do not have to trim this log file (there is no new log writing) */
do
 return 1  /* quit */
end

if (\FileOpen(LogFile)) then  /* if we cannot open the log file */
do
 return 0  /* quit */
end

Counter = 0  /* start a counter */
LogLines. = ''  /* start with nothing */

do while (lines(LogFile))  /* as long as we have log lines left */
 Counter = Counter + 1  /* up the counter */
 LogLines.Counter = linein(LogFile)  /* store the next line */
end

call FileClose LogFile  /* close the file */

StartLine = max(1,(Counter - KeepLines + 1))  /* if the first line we want to keep is less than 1, start at the first line */

if (StartLine > 1) then  /* if we want a new file */
do

 call sysfiledelete LogFile  /* get rid of the old file */

 if (\FileOpen(LogFile)) then  /* if we cannot open a new log file */
 do
  return 0  /* quit */
 end

 do Index = StartLine to Counter  /* take each of the lines we want */
  call lineout LogFile,LogLines.Index  /* write it to the log file */
 end

 call FileClose LogFile  /* close the file */

end

return 1  /* end of TrimLogFile */

/**********************************************************************/
UnhideObject: procedure expose Global.  /* unhide an object */
/**********************************************************************/

parse arg Object  /* get the argument */

if (\syssetobjectdata(Object,'NOTVISIBLE=NO;NODELETE=NO;NODRAG=NO;NOMOVE=NO')) then  /* if we cannot make the object visible (and some other things thrown in for good measure) */
do
 return 0  /* return an error */
end

Global.Hidden.0 = Global.Hidden.0 - 1  /* move the hidden objects counter back one */

return 1  /* end of HideObject */

/**********************************************************************/
UnpackMessage: procedure expose Global.  /* unpacks received/sent mail files */
/**********************************************************************/

parse arg MessFile,AttDir,Process  /* get the arguments */

if (\FileOpen(MessFile,'READ')) then  /* if we cannot open the message file for reading */
do
 return 0  /* quit with nothing */
end

MessCont = charin(MessFile,1,chars(MessFile))  /* get the message file contents */
call FileClose MessFile  /* close the message file */
CopyFile = AttDir||'.COPY'  /* the copy file name */

call sysfiledelete CopyFile  /* get rid of any existing file */

if (\FileOpen(CopyFile)) then  /* if we cannot open the message file for writing */
do
 return 0  /* quit with nothing */
end

call charout CopyFile,MessCont,1  /* write the message file content to the new file */
call FileClose CopyFile  /* close the new message file */

if (\MessageSettings(CopyFile,MessageSettings(MessFile),'CHANGE')) then  /* if we cannot copy the settings of the original file to the copy file */
do
 return 0  /* quit */
end

if (\ProgSpecPut(CopyFile)) then  /* if we cannot attach the program spec EA to the copy file */
do
 return 0  /* quit */
end

if (\MessageContents(CopyFile)) then  /* if we cannot extract the message contents */
do
 return 0  /* quit with nothing */
end

InComing = (MessageSettings(MessFile,'1*******','MATCH'))  /* set a flag if the message is incoming */

if (InComing) then  /* if the message is incoming */
do
 HeaderEntries = Global.ReaderHeaderIn  /* use this */
end
else  /* if the message is outgoing, i.e. sent */
do
 HeaderEntries = Global.ReaderHeaderOut  /* use this */
end

HeadText = MakeHeaderLines(Global.MessHead,HeaderEntries,1)  /* get the necessary header text from the mail file */
BodyText = ''  /* start with nothing */
InfoText = ''  /* we have no info text yet */
AttFiles.0 = 0  /* no attachment files yet */

SeparatorLength = 76  /* use the RFC default */

if (Global.MessBody >< '') then  /* if there is a message body */
do

 if (GetHeaderEntry(Global.MessHead,'MIME-Version:') = '') then  /* if we have no MIME content */
 do

  EncLine = translate(GetHeaderEntry(Global.MessHead,'Encoding:'))  /* look for an encoding line in the header */

  if (pos('UUENCODE',EncLine) > 0) then  /* if it looks like we have UU-encoded content */
  do

   UUDecoder = Global.ProgDir||'\uudecode.exe'  /* the UU decoder program spec */

   if (\FileCheck(UUDecoder,1)) then  /* if the UU decoder does not exist */
   do

    if (Process) then  /* if we are to process contents */
    do
     InfoText = InfoText||'WARNING: cannot find UU decoder "'||UUDecoder||'"'||Global.CRLF  /* add a warning to the body text */
    end

    UUDecoder = ''  /* we have no UU decoder */

   end

   do while (EncLine >< '')  /* as long as we have something left */

    parse var EncLine ContLines ContType ',' EncLine  /* get the first content code bit */
   
    if ((datatype(ContLines,'W')) & (ContType >< '')) then  /* if we find what we expected */
    do

     NextPos = 1  /* start at the first character of the body content */

     do while (substr(Global.MessBody,NextPos,2) = Global.CRLF)  /* as long as the next pair is a CRLF */
      NextPos = NextPos + 2  /* skip over it */
     end

     do ContLines  /* for each content line */
      NextPos = pos(Global.CRLF,Global.MessBody,NextPos) + 2  /* look for the next CRLF and skip over it */
     end

     parse var Global.MessBody NextPart =(NextPos) Global.MessBody  /* get the next part */

    end

    else  /* if we find something unexpected */

    do
     NextPart = Global.MessBody  /* use the lot */
     ContType = 'UUENCODE'  /* assume UU encoded content */
    end

    select  /* do one of the following */

     when (ContType = 'TEXT') then  /*  if we have a text bit */
     do
      BodyText = BodyText||NextPart||Global.CRLF  /* simply add the part and an extra CRLF to our body text */
     end

     when (ContType = 'UUENCODE') then  /*  if we have a UUE bit */
     do

      if ((Process) & (UUDecoder >< '')) then  /* if we have a UU decoder and we want the contents processed */
      do

       TempFile = TempFileName('UUE',AttDir)  /* create a temporary file */

       if (FileOpen(TempFile)) then  /* if we can open the temp file */
       do
        call charout TempFile,NextPart  /* and write it to the temp file */
        call FileClose TempFile  /* close the temp file */
        call directory AttDir  /* change to the attachments dir */
        address cmd UUDecoder||' "'||TempFile||'"'  /* decode the temp file [EXTERNAL] */
        call directory Global.TempDir  /* change to the temp dir */
        call sysfiledelete TempFile  /* remove the temp file */
       end

      end

     end

     otherwise  /* if it is none of the above */
     do
      BodyText = BodyText||Global.CRLF||'[RexxMail cannot decode "'||ContType||'" content]'||Global.CRLF  /* report */
     end

    end

   end

   call sysfiletree AttDir||'\*','AttFiles.','FOS'  /* have we got any files in the attachment dir (or in newly created subdirs)? */

  end

  else  /* if we appear to have a normal text body */

  do
   BodyText = Global.MessBody  /* simply use the original message body */
  end

 end

 else  /* if we have MIME content */

 do

  CTLine = translate(GetHeaderEntry(Global.MessHead,'Content-type:'))  /* look for a content type line in the header */

  if (CTLine >< '') then  /* if we have a content line */
  do

   if (pos('MULTIPART',CTLine) = 0) then  /* if the message is not multi-part */
   do

    if (pos('TEXT/',CTLine) > 0) then  /* if it is a text message */
    do

     CTELine = translate(GetHeaderEntry(Global.MessHead,'Content-transfer-encoding:'))  /* look for a content transfer encoding line */

     if ((CTELine = '') | ((pos('7BIT',CTELine) > 0) | (pos('8BIT',CTELine) > 0))) then  /* if we have no special encoding */
     do

      if (pos('TEXT/HTML',CTLine) > 0) then  /* if it is an HTML message */
      do

       HTMLFile = TempFileName('HTML',AttDir)  /* get a unique name to use */

       if (FileOpen(HTMLFile,'WRITE')) then  /* if we can open the file for writing */
       do

        call charout HTMLFile,Global.MessBody  /* write the body text to the file */
        call FileClose HTMLFile  /* close the file */

        if (Process) then  /* are we are to process contents */
        do
         InfoText = InfoText||'HTML content written to attached file'||Global.CRLF  /* the additional text */
        end

       end

      end

      else  /* if it is not HTML */

      do

       if (pos('ISO-8859-1',CTLine) > 0) then  /* if it uses the ISO 8859-1 character set */
       do
        BodyText = translate(Global.MessBody,Global.ISOto850,xrange('80'x,'FF'x))  /* convert the ISO-8859-1 message body content to PC850 and use it */
       end
       else  /* if it uses no coding, or something we can't decode anyway */
       do
        BodyText = Global.MessBody  /* use the original body content */
       end

      end

     end

    end

   end

  end

  else  /* if we do not have a content line, i.e. a malformed message, any message body must be simple text */

  do
   BodyText = Global.MessBody  /* so use what we have */
  end

  if ((BodyText = '') & (InfoText = '')) then  /* if we still have nothing */
  do

   call directory Global.TempDir  /* change to the user's temp dir -- this is a kludge to make munpack.exe work and to make sure we can 'sysdestroyobject' the att dir later */
   MIMEDecoder = Global.ProgDir||'\munpack.exe'  /* the MIME decoder program spec */

   if (\FileCheck(MIMEDecoder,1)) then  /* if the MIME decoder does not exist */
   do
    return 0  /* quit with nothing */
   end

   address cmd MIMEDecoder||' -C "'||AttDir||'" -e -q -t "'||CopyFile||'"'  /* unpack the mail file into the attachments directory, adding MIME type EAs, quietly, and including text parts [EXTERNAL] */
   call sysfiletree AttDir||'\*','AttFiles.','FOS'  /* have we got any files in the attachment dir (or in newly created subdirs)? */

   do Index = 1 to AttFiles.0  /* run through the lot */

    AttFileName = filespec('N',AttFiles.Index)  /* get the file name bit */

    if (sysgetea(AttFiles.Index,'MIME-TYPE','TypeInfo') = 0) then  /* if we can get the MIME type from the file's EAs */
    do
     parse upper var TypeInfo 5 MIMEType  /* get the bit we want, in upper case */
    end
    else  /* if we cannot get the MIME type */
    do
     MIMEType = 'TEXT/PLAIN'  /* use this */
    end

    select  /* do one of the following */

     when (MIMEType = 'TEXT/PLAIN') then  /* if it's plain text */
     do

      NamePart = filespec('N',AttFiles.Index)  /* get the name part */

      if ((translate(left(NamePart,4)) = 'PART') & (datatype(substr(NamePart,5),'W'))) then  /* if it's a message part file */
      do

       if (FileOpen(AttFiles.Index,'READ')) then  /* if we can open the file */
       do

        Buffer = charin(AttFiles.Index,1,chars(AttFiles.Index))  /* copy the file contents to a buffer */
        call FileClose AttFiles.Index  /* close the part file */

        do while (left(Buffer,2) = Global.CRLF)  /* as long as the text starts with a CRLF */
         Buffer = substr(Buffer,3)  /* skip the CRLF and take the rest */
        end

        if (Buffer >< '') then  /* if we have anything left */
        do
         Buffer = translate(Buffer,Global.ISOto850,xrange('80'x,'FF'x))  /* just assume it's ISO-8859-1 text and convert any high-bit characters to PC850 */
         BodyText = BodyText||Buffer||Global.CRLF  /* add the file contents to the body text */
        end

       end

       call sysfiledelete AttFiles.Index  /* zap the part file */
       AttFiles.0 = AttFiles.0 - 1  /* one less attachment file */

      end

     end

     when (MIMEType = 'TEXT/HTML') then  /* if it's HTML */
     do

      FileName = filespec('N',AttFiles.Index)  /* get the file name */

      if (pos('.HTM',translate(right(FileName,5))) = 0) then  /* if the file extension is not (*)HTM(L) */
      do
       FileName = FileName||'.HTML'  /* add it */
      end

      call syssetobjectdata AttFiles.Index,'TITLE='||FileName  /* set the (new) object title */

      if (Process) then  /* are we are to process contents */
      do
       InfoText = InfoText||'HTML content written to attached file'||Global.CRLF  /* the message to insert as body text */
      end

     end

     otherwise  /* if neither of the above apply */
     do

      if ((left(AttFileName,4) = 'part') & (datatype(substr(AttFileName,5),'W'))) then  /* if it is a "part*" file, with "*" a whole number */
      do
       call syssetobjectdata AttFiles.Index,'TITLE='||translate(MIMEType,'..','/\')  /* change the title to show what's in it */
      end

     end

    end

   end

  end

 end

 if (BodyText >< '') then  /* if we have something */
 do

  if (pos('<HTML>',translate(left(BodyText,120))) > 0) then  /* if it looks like HTML */
  do

   HTMLFile = TempFileName('HTML',AttDir)  /* get a unique name to use */

   if (FileOpen(HTMLFile,'WRITE')) then  /* if we can open the file for writing */
   do

    call charout HTMLFile,BodyText  /* write the body text to the file */
    call FileClose HTMLFile  /* close the file */
    BodyText = ''  /* no body text */

    if (Process) then  /* are we are to process contents */
    do
     InfoText = InfoText||'HTML content written to attached file'||Global.CRLF  /* the additional text */
    end

   end

  end

 end

 call sysfiletree AttDir||'\*.htm?','HTMLFiles.','FO'  /* have we got any HTM(L) files? */

 do Index = 1 to HTMLFiles.0  /* run through the lot */

  if ((Global.HTMLText) & (BodyText = '')) then  /* if we want text content for the message body */
  do

   if (FileOpen(HTMLFiles.Index,'READ')) then  /* if we can open the HTML file for reading */
   do

    HTMLTextBit = charin(HTMLFiles.Index,1,min(chars(HTMLFiles.Index),10000))  /* get the file content, or just the first 10000 characters, whichever is the smallest */

    call FileClose HTMLFiles.Index  /* close the file */

    HTMLTextBit = translate(HTMLTextBit,Global.ISOto850,xrange('80'x,'FF'x))  /* translate any high bits */
    BodyText = BodyText||copies('_',SeparatorLength)||Global.CRLF||,  /* add a line to the message body */
               'Text extract from HTML file:'||Global.CRLF||,  /* add info to the message body */
               copies('',SeparatorLength)||Global.CRLF  /* add a line to the message body */
    BodyText = BodyText||HTMLToText(HTMLTextBit)||Global.CRLF  /* extract the text from the HTML file and add it to the body */

   end

  end

  if (Process) then  /* if we want content processed */
  do

   HTMLName = filespec('N',HTMLFiles.Index)  /* get the bare file name */
   NewTitle = (((left(HTMLName,4) = 'RXML') | (left(HTMLName,4) = 'part')) & (datatype(substr(HTMLName,5,1),'NUM')))  /* if it is a temporary file name, let HTMLFilter look for a new title */

   if (HTMLFilter(HTMLFiles.Index,NewTitle)) then  /* if filtering the file resulted in "safe" changes */
   do
    InfoText = InfoText||'HTML content made safe'||Global.CRLF  /* add info */
   end

   Type = 'DFFF'x||'0000'x||'0200'x||'FDFF'x||'0400'x||'HTML'||'FDFF'x||'0900'x||'text/html'  /* set the file types */
   call sysputea HTMLFiles.Index,'.TYPE',Type  /* attach them to the file */

  end

 end

end

HeadText = HeadText||copies('',SeparatorLength)||Global.CRLF  /* add a separator line to the header text */

if (InfoText >< '') then  /* if we have info text */
do
 HeadText = HeadText||copies('_',SeparatorLength)||Global.CRLF||InfoText||copies('',SeparatorLength)||Global.CRLF  /* add the info text and separators */
end

if (BodyText = '') then  /* if we have no body text */
do
 BodyText = copies('_',SeparatorLength)||Global.CRLF||'[no message text]'||Global.CRLF||copies('',SeparatorLength)||Global.CRLF  /* use this */
end

ReadFile = AttDir||'.READ'  /* the name for the reader file */

if (FileOpen(ReadFile,'WRITE')) then  /* if we can open the new read file for writing */
do
 call charout ReadFile,HeadText||BodyText  /* write the contents */
 call FileClose ReadFile  /* close the file */
end
else  /* if we cannot open the new read file for writing */
do
 ReadFile = ''  /* we have no read file */
end

if ((Process) & (AttFiles.0 > 0)) then  /* if we have attachment files left and we want the contents processed */
do

 if (Incoming) then  /* if it is an incoming message */
 do
  call RunCommand Global.RunAttachIn,,AttDir  /* run any external command in the attachments dir */
 end

 call sysfiletree AttDir||'\*','AttFiles.','FOS','*****','-****'  /* see what's in the attachments dir now, resetting all archive attributes */

end

if ((Process) & (AttFiles.0 > 0)) then  /* if we still have attachment files left and we wanted contents processed */
do
 call MessageSettings MessFile,'****1***','CHANGE'  /* change the message settings to show the presence of attachments */
 call syssetobjectdata MessFile,'REXXMAILATTACHMENT=Yes'  /* set the attachment indicator for the CWMailFile class */
end
else  /* if we have no (more) attachments, or we don't want any processing anyway */
do
 return 0  /* return with nothing */
end

return 1  /* end of UnpackMessage */

/**********************************************************************/
UpdateSettings: procedure expose Global.  /* updates the current user's configuration file */
/**********************************************************************/

parse arg OldSettingsFile,NewSettingsFile  /* get the arguments */

call LogAction 'Updating configuration file "'||OldSettingsFile||'"'  /* report */
MessText = ''  /* start with no message text */

if (FileCheck(OldSettingsFile)) then  /* if the RexxMail configuration file exists */
do

 if (\FileOpen(OldSettingsFile,'READ')) then  /* if we cannot open the old settings file for reading */
 do
  return 0  /* quit */
 end

 OldSettingsCont = charin(OldSettingsFile,1,chars(OldSettingsFile))  /* get the contents of the old settings file */

 call FileClose OldSettingsFile  /* close the old settings file */

 OldSettingsCont = strip(OldSettingsCont,'T',d2c(26))  /* remove any EOF marker */

end

else  /* if we have no file */

do
 OldSettingsCont = ''  /* we have no content */
end

if (\FileOpen(NewSettingsFile,'READ')) then  /* if we cannot open the new settings file for reading */
do
 return 0  /* quit */
end

NewSettingsCont = charin(NewSettingsFile,1,chars(NewSettingsFile))  /* get the contents of the new settings file */
call FileClose NewSettingsFile  /* close the new settings file */
OldSettings. = ''  /* start with nothing */
Counter = 1  /* start a counter */

do while(OldSettingsCont >< '')  /* as long as we have old settings content left */

 parse var OldSettingsCont NextLine (Global.CRLF) OldSettingsCont  /* get the next line */
 StripLine = strip(translate(NextLine,' ',d2c(9)),'B',' ')  /* turn TABs into blanks, then get rid of extra blanks at either end */

 if (StripLine = '') then  /* if we have an empty line */
 do
  OldSettings.Counter._Comment = OldSettings.Counter._Comment||Global.CRLF  /* add an empty line to the comment part */
 end

 else  /* if it is not an empty line */

 do

  if (left(StripLine,1) = '#') then  /* if it is a comment */
  do
   OldSettings.Counter._Comment = OldSettings.Counter._Comment||NextLine||Global.CRLF  /* add the original line to the comment part */
  end
  else  /* if it is not a comment */
  do
   OldSettings.Counter._Entry = NextLine  /* store the original entry */
   OldSettings.Counter._KeyWord = translate(word(StripLine,1))  /* the keyword is the first word of the stripped line, in upper case */
   Counter = Counter + 1  /* up the counter to start a new entry */
  end

 end

end

OldSettings.0 = Counter  /* save  the counter */
NewSettings. = ''  /* start with nothing */
Counter = 0  /* restart the counter, this time at 0 */

do while(NewSettingsCont >< '')  /* as long as we have new settings content left */

 Counter = Counter + 1  /* up the counter */
 parse var NewSettingsCont NextPart (Global.EmptyLine) NewSettingsCont  /* get the next part */
 NewSettings.Counter._Part = NextPart  /* save the part */

 do while (NextPart >< '')  /* as long as we have something left */
  parse var NextPart NextLine (Global.CRLF) NextPart  /* get the next line, so we end up with just the last line */
 end

 NewSettings.Counter._Keyword = translate(word(NextLine,1))  /* the keyword is the first word; use upper case */

end

NewSettings.0 = Counter  /* save  the counter */
NewSettings = ''  /* start with nothing */
InvalidStuff = Global.CRLF||copies('#',72)||Global.CRLF||,  /* the first line of the warning to add before an invalid or obsolete entry */
               '# This setting is invalid or obsolete; please remove it '||copies('#',16)||Global.CRLF||,  /* the second line of the warning to add before an invalid or obsolete entry */
               copies('#',72)||Global.CRLF||,  /* the third line of the warning to add before an invalid or obsolete entry */
               '# '  /* the comment before an invalid or obsolete entry */
GotVersion = 0  /* we have no version setting yet */
GotUpdate = 0  /* we are not updating as far as we know (we may be correcting an invalid setting) */
GotInvalid = 0  /* we have no invalid or obsolete entries yet */

do Counter = 1 to OldSettings.0  /* run through the old settings */

 Found = 0  /* we haven't found anything yet */

 do Index = 1 to NewSettings.0  /* run through the new settings */

  if (OldSettings.Counter._KeyWord = NewSettings.Index._KeyWord) then  /* if the two keywords are the same */
  do
   NewSettings.Index._Keyword = ''  /* get rid of the new keyword */
   Found = 1  /* we've found it */
  end

 end

 if (Found) then  /* if the old keyword was found in the new file */
 do
  NewSettings = NewSettings||OldSettings.Counter._Comment||OldSettings.Counter._Entry||Global.CRLF  /* add the comment and entry to the new settings */
 end

 else  /* if the old keyword was not found in the new file */

 do

  NewSettings = NewSettings||OldSettings.Counter._Comment  /* always add the user's own comment part */

  if (OldSettings.Counter._KeyWord >< '') then  /* if we have an old keyword */
  do

   if (OldSettings.Counter._KeyWord = 'VERSION') then  /* if it is the 'Version' setting */
   do

    GotVersion = 1  /* we have a version setting */

    if (Global.Build >> Global.Version) then  /* if we are updating */
    do
     OldSettings.Counter._Entry = left(OldSettings.Counter._Entry,length(OldSettings.Counter._Entry) - 15)  /* get the user's indent for the version entry */
     NewSettings = NewSettings||OldSettings.Counter._Entry||Global.Build||Global.CRLF  /* insert the version string, retaining the original formatting */
     GotUpdate = 1  /* we have an update on our hands */
    end
    else  /* if we are not updating */
    do
     NewSettings = NewSettings||OldSettings.Counter._Entry||Global.CRLF  /* insert the original version string */
    end

   end

   else  /* if it is not the 'Version' setting */

   do
    NewSettings = NewSettings||InvalidStuff||OldSettings.Counter._Entry||Global.EmptyLine  /* add the commented-out entry to the new settings with a comment */
    GotInvalid = 1  /* we have at least one invalid or obsolete entry */
   end

  end

 end

end

if (GotInvalid) then  /* if we have commented out one or more invalid or obsolete entries */
do
 MessText = '- One or more invalid or obsolete entries have been commented out.'||Global.CRLF  /* this is the message text */
end

if (GotVersion) then  /* if we already had a version setting */
do

 if (GotUpdate) then  /* if we have performed an update */
 do
  MessText = '- The version number entry has been changed to '||Global.Build||'.'||Global.CRLF||MessText  /* add this to the start of the message text */
 end

end

else  /* if we have no version number */

do
 VersionStuff = copies('#',72)||Global.CRLF||,  /* start with a separator */
                '# RexxMail build number '||copies('#',48)||Global.CRLF||,  /* add a warning */
                '#'||Global.CRLF||,  /* add an empty comment line */
                '# Do not remove this entry!'||Global.CRLF||,  /* add a warning */
                '#'||Global.CRLF||,  /* add an empty comment line */
                copies('#',72)||Global.CRLF||,  /* add another separator */
                'Version = '||Global.Build  /* add the version entry */
 NewSettings = VersionStuff||Global.EmptyLine||NewSettings  /* add the lot to what we have */
 MessText = '- A version number entry has been added.'||Global.CRLF||MessText  /* add this to the start of the message text */
end

NewStuff = ''  /* no new stuff yet */

do Index = 1 to NewSettings.0  /* run through the new settings again */

 if (NewSettings.Index._KeyWord >< '') then  /* if the keyword still exists */
 do
  NewStuff = NewStuff||Global.EmptyLine||NewSettings.Index._Part  /* add the complete entry to the new stuff */
 end

end

if (NewStuff >< '') then  /* if we have new stuff */
do
 NewSettings = NewSettings||Global.CRLF||copies('#',72)||Global.CRLF  /* add a separator */
 NewSettings = NewSettings||'# New settings '||copies('#',57)||Global.CRLF  /* add a header */
 NewSettings = NewSettings||'#'||Global.CRLF  /* and an empty comment line */
 NewSettings = NewSettings||'# The following entries are new for this build of RexxMail, or'||Global.CRLF  /* add text */
 NewSettings = NewSettings||'# they were not found in the existing configuration file.'||Global.CRLF  /* add text */
 NewSettings = NewSettings||copies('#',72)||NewStuff  /* and another separator and the new stuff */
 MessText = MessText||'- One or more new configuration entries have been added;'||Global.CRLF  /* add this to the message text */
 MessText = MessText||'  please edit the new configuration entries now.'||Global.CRLF  /* add this to the message text */
end

call sysfiledelete OldSettingsFile  /* delete the old settings file */

if (\FileOpen(OldSettingsFile,'WRITE')) then  /* if we cannot open a new RexxMail settings file */
do
 return 0  /* quit */
end

do while (right(NewSettings,2) = Global.CRLF)  /* as long as the new settings end with a CRLF */
 NewSettings = left(NewSettings,length(NewSettings) - 2)  /* remove it */
end

call charout OldSettingsFile,NewSettings||Global.CRLF  /* write the new settings to the RexxMail settings file with a CRLF at the end */
call FileClose OldSettingsFile  /* close the updated settings file */

if (MessText >< '') then  /* if we have something to report */
do
 MessText = 'Your configuration file '||Global.CRLF||,  /* add this to the front of the message text */
            '  '||OldSettingsFile||Global.CRLF||,  /* followed by this */
            'has been updated:'||Global.CRLF||MessText  /* and this */
 call SystemMessage 'RexxMail configuration updated.',MessText  /* send a message to the user */
end

return 1  /* end of UpdateSettings */

/**********************************************************************/
ViewMessage: procedure expose Global.  /* views received mail messages; also resets folder icon if no unread messages remain */
/**********************************************************************/

MessFile = Global.ProcFile  /* the file name to process */

MessDir = strip(filespec('D',MessFile)||filespec('P',MessFile),'T','\')  /* the message dir */

if (MessageSettings(MessFile,'1*0*****','MATCH')) then  /* if it is an unprocessed incoming message */
do
 call MessageSettings MessFile,'**1*****','CHANGE'  /* new read mail message setting: Processed */
end

AttDir = AttDirGet(MessFile,1)  /* get the attachments dir, creating it if necessary */

if (AttDir = '') then  /* if we still have no attachments dir */
do
 return 0  /* quit */
end

GotAtt = UnpackMessage(MessFile,AttDir,1)  /* unpack the mail file into the attachment dir; process the attachments folder contents */
ReadFile = AttDir||'.READ'  /* the read file */

if (\FileCheck(ReadFile,0)) then  /* if we have no read file (no warning required) */
do
 return 0  /* quit */
end

if (GotAtt) then  /* if we have attachments */
do

 if (Global.OpenAttachBeforeView) then  /* if we want the attachments folder opened */
 do
  call sysopenobject AttDir,0,1  /* open the attachments folder on the desktop */
 end

 call sysfiletree AttDir||'\*','PreFiles.','FOS'  /* see what's in the attachments dir */

end
else  /* if we have no attachments */
do
 call DeleteDir AttDir  /* get rid of the attachments folder */
 call AttDirUnlink MessFile  /* get rid of the attachments folder link */
end

call RunCommand Global.Reader,ReadFile  /* see if we can view the read file */

if (GotAtt) then  /* if we had attachments */
do

 if (Global.CloseAttachAfterView) then  /* if we normally want the attachments folder closed */
 do

  Changed = 0  /* no changes yet */

  call sysfiletree AttDir||'\*','PostFiles.','FOS'  /* what have we got (left) in the attachments dir? */

  do Index = 1 to PostFiles.0  /* run through the list of files we have now */

   Existing = 0  /* assume the file is not an existing one (i.e. it was not there before) */

   do Counter = 1 to PreFiles.0  /* run through the list of files we had before */

    if (PostFiles.Index = PreFiles.Counter) then  /* if we have a match */
    do
     call sysfiletree PostFiles.Index,'ArchFiles.','FO','+****'  /* see if the archive attribute of the file was set */
     Existing = (ArchFiles.0 = 0)  /* if it has not, the file hasn't changed */
    end

   end

   if (\Existing) then  /* if the file was not an existing one */
   do
    Changed = 1  /* we have at least 1 change */
   end

  end

  if (\Changed) then  /* if we have no important changes */
  do
   call sysdestroyobject AttDir  /* get rid of the attachments folder */
  end

 end

 if (FileCheck(MessFile,0)) then  /* if the message file still exists */
 do
  call AttDirUnlink MessFile  /* get rid of the attachments folder link */
 end

end

if (FileCheck(ReadFile)) then  /* if the read file still exists */
do
 call sysfiledelete ReadFile  /* get rid of the read file -- no check, it may already have been deleted by another process */
end

if (FileCheck(MessFile)) then  /* if the message file still exists */
do

 CheckPath = translate(filespec('D',MessFile)||strip(filespec('P',MessFile),'T','\'))  /* the upper case path of the message file */

 if (CheckPath = translate(Global.InDir)) then  /* if the file is in the In folder */
 do
  call MoveMessage MessFile,Global.ControlView,1  /* move the file to another folder if necessary */
 end

end

call sysfiledelete AttDir||'.COPY'  /* get rid of the copy file created by the unpack routine */

if (pos(translate(Global.MessagesDir),translate(MessDir)) = 1) then  /* if the message folder is a RexxMail user folder */
do

 ResetIcon = 1  /* assume we are to reset the message folder icon to begin with */

 call sysfiletree MessDir||'\*','Files.','FO'  /* look for files in the message dir */

 do Index = 1 to Files.0  /* run through the lot */

  if (MessageSettings(Files.Index,,'CHECK')) then  /* if it is a RexxMail message file */
  do

   if (MessageSettings(Files.Index,'**0*****','MATCH')) then  /* if the file is unprocessed, i.e. unread */
   do
    ResetIcon = 0  /* the icon stays put */
   end

  end

 end

 if (ResetIcon) then  /* if we are to reset the message dir icon */
 do
  call syssetobjectdata MessDir,'ICONFILE='||Global.IconDir||'\folddef0.ico'  /* specify the default normal icon */
  call syssetobjectdata MessDir,'ICONNFILE=1,'||Global.IconDir||'\folddef1.ico'  /* specify the default animated icon */
 end

end

return 1  /* end of ViewMessage */

/**********************************************************************/
ViewRawMessage: procedure expose Global.  /* shows the raw message contents */
/**********************************************************************/

MessFile = Global.ProcFile  /* get the file name to process */

Tempfile = TempFileName('RAW')  /* get a temp file name */

if (\FileOpen(TempFile,'WRITE')) then  /* if we cannot open the temp file for writing */
do
 return 0  /* and quit */
end

if (\FileOpen(MessFile,'READ')) then  /* if we cannot open the message file for reading */
do
 return 0  /* and quit */
end

call charout TempFile,charin(MessFile,1,chars(MessFile))  /* copy the whole contents of the message file into the temp file */
call FileClose MessFile  /* close the message file */
call FileClose TempFile  /* close the temp file */
call MessageSettings MessFile,'**1*****','CHANGE'  /* new edited mail message setting: Processed */
call RunCommand Global.Reader,TempFile  /* see if we can view the file */
call sysfiledelete TempFile  /* get rid of the temporary file */

return 1  /* end of ViewRawMessage */

/**********************************************************************/
WordWrap: procedure expose Global.  /* word-wraps a line of text */
/**********************************************************************/

parse arg TextLine,MaxLength,Indent,NoBreak,IndentString  /* get the arguments */

if (IndentString = '') then  /* if we have no indent string */
do
 IndentString = ' '  /* use a single space */
end

NoBreak = (NoBreak = 1)  /* this one is true if the argument is 1 */

if (Indent = '') then  /* if we have no indent */
do
 Indent = 0  /* we have no indent */
end

if (MaxLength = '') then  /* if we have no maximum line length value */
do
 MaxLength = 76  /* use this */
end

CutPoint = MaxLength + 1  /* start the cut at ASCIILineLength plus one */
NewText = ''  /* start with nothing */
TextLine = strip(TextLine,'T',' ')  /* remove any trailing spaces */

do while (length(TextLine) > MaxLength)  /* as long as the line is longer than the limit */

 parse var TextLine NextBit =(CutPoint) TextLine  /* get the next bit of the maximum length */

 if ((right(NextBit,1) = ' ') | (left(TextLine,1) = ' ')) then  /* if we cut the line just before or after a space, i.e. we cut between two words */
 do
  TextLine = strip(TextLine,'L',' ')  /* get rid of leading spaces on the remaining bit of the line */
 end

 else  /* if we cut through a word */

 do

  LastSpace = lastpos(' ',NextBit)  /* look for the last space in this bit */

  if (LastSpace >< 0) then  /* if we found a space */
  do
   parse var NextBit NextBit =(LastSpace) MoreBit  /* cut off the bit after the space and save it for the next line */
   TextLine = strip(MoreBit,'L',' ')||TextLine  /* get rid of the leading space and stick this bit back on to what's left of the original text line */
  end

  else  /* if we found no space */

  do

   if (NoBreak) then  /* if we must not break words */
   do
    parse var TextLine MoreBit TextLine  /* get the first word of the remaining bit of the line */
    NextBit = NextBit||MoreBit  /* and add it to what we have */
   end

  end

 end

 NextBit = strip(NextBit,'T',' ')  /* get rid of trailing spaces */
 NewText = NewText||NextBit  /* add the bit we have to the new text */

 if (TextLine >< '') then  /* if there is more to come */
 do
  NewText = NewText||Global.CRLF||copies(IndentString,Indent)  /* add a new line with an indent */
 end

end

NewText = NewText||TextLine  /* add the remaining bit of the line to the text */

return NewText  /* end of WordWrap */
