%{
/**********************************************************************\
*
*  Copyright (c) 1994  Carnegie Mellon University
*  All rights reserved.
*  
*  Use and copying of this software and preparation of derivative
*  works based on this software are permitted, including commercial
*  use, provided that the following conditions are observed:
*  
*  1. This copyright notice must be retained in full on any copies
*     and on appropriate parts of any derivative works.
*  2. Documentation (paper or online) accompanying any system that
*     incorporates this software, or any part of it, must acknowledge
*     the contribution of the Gwydion Project at Carnegie Mellon
*     University.
*  
*  This software is made available "as is".  Neither the authors nor
*  Carnegie Mellon University make any warranty about the software,
*  its performance, or its conformity to any specification.
*  
*  Bug reports, questions, comments, and suggestions should be sent by
*  E-mail to the Internet address "gwydion-bugs@cs.cmu.edu".
*
***********************************************************************
*
* $Header: /afs/cs.cmu.edu/project/gwydion/hackers/nkramer/mindy/mindy-1.3/comp/RCS/lexer.l,v 1.14 95/03/13 21:43:23 nkramer Exp $
*
* This file is the lexical analizer.
*
\**********************************************************************/

#include <stdio.h>
#include "lexer.h"
#include "src.h"
#include "parser-tab.h"

extern int isatty();
extern void warn();

int line_count = 1;

#define is(type) return (yylval.token = make_token(yytext, yyleng)), type;

static void skip_multi_line_comment(void);
static int make_header_key(void);
static int make_header_val(void);
static int make_header_end(void);
  
%}
  
%x ini key val etc

D	[0-9]
E	[esdx][-+]?{D}+

A	[a-z]
G	[!&*<=>|^$%@_]
S	[-+~?/]

DGS	({D}|{G}|{S})
ADGS	({A}|{DGS})

N	((({G}{DGS}*)?{A}{ADGS}*)|({D}{DGS}*({A}{DGS}+)*{A}{A}{ADGS}*))

O	":="|"+"|"-"|"*"|"/"|"="|"=="|"<"|">"|"<="|">="|"~="|"&"|"|"|"^"

STR	\"(([ !#-\[\]-~])|(\\["\\abefnrt0]))*\"

%%

<INITIAL>.			BEGIN(ini); yyless(0);

<ini>^#!.*\n			BEGIN(key); warn(line_count, "igoring initial #! interpreter comment\n"); line_count++; 

<ini,key>^[a-z][-a-z0-9]*:	BEGIN(val); return make_header_key();
<ini,key>^[ \t]*\n		BEGIN(etc); return make_header_end();

<val>.*\n([ \t]+.+\n)*		BEGIN(key); return make_header_val();

<etc>[ \t\f]+			;
<etc>[\n]			line_count++;

<etc>"//".*			;
<etc>"/*"			skip_multi_line_comment();

<etc>abstract			is(ABSTRACT);
<etc>above			is(ABOVE);
<etc>begin			is(DBEGIN);
<etc>below			is(BELOW);
<etc>block			is(BLOCK);
<etc>by				is(BY);
<etc>case			is(CASE);
<etc>class			is(CLASS);
<etc>cleanup			is(CLEANUP);
<etc>concrete			is(CONCRETE);
<etc>constant			is(CONSTANT);
<etc>define			is(DEFINE);
<etc>else			is(ELSE);
<etc>elseif			is(ELSEIF);
<etc>end			is(END);
<etc>exception			is(EXCEPTION);
<etc>finally			is(FINALLY);
<etc>for			is(FOR);
<etc>free			is(FREE);
<etc>from			is(FROM);
<etc>generic			is(GENERIC);
<etc>handler			is(HANDLER);
<etc>if				is(IF);
<etc>in				is(IN);
<etc>inherited			is(INHERITED);
<etc>instance			is(INSTANCE);
<etc>keyed-by			is(KEYED_BY);
<etc>keyword			is(KEYWORD_RESERVED_WORD);
<etc>let			is(LET);
<etc>local			is(LOCAL);
<etc>method			is(METHOD);
<etc>open			is(OPEN);
<etc>otherwise			is(OTHERWISE);
<etc>primary			is(PRIMARY);
<etc>required			is(REQUIRED);
<etc>seal			is(SEAL);
<etc>sealed			is(SEALED);
<etc>select			is(SELECT);
<etc>slot			is(SLOT);
<etc>subclass			is(SUBCLASS);
<etc>then			is(THEN);
<etc>to				is(TO);
<etc>unless			is(UNLESS);
<etc>until			is(UNTIL);
<etc>variable			is(VARIABLE);
<etc>virtual			is(VIRTUAL);
<etc>while			is(WHILE);

<etc>module			is(MODULE);
<etc>library			is(LIBRARY);
<etc>export			is(EXPORT);
<etc>create			is(CREATE);
<etc>use			is(USE);
<etc>all			is(ALL);

<etc>"prefix:"			is(PREFIX_KEYWORD);
<etc>"import:"			is(IMPORT_KEYWORD);
<etc>"exclude:"			is(EXCLUDE_KEYWORD);
<etc>"export:"			is(EXPORT_KEYWORD);
<etc>"rename:"			is(RENAME_KEYWORD);
<etc>"while:"			is(WHILE_KEYWORD);
<etc>"until:"			is(UNTIL_KEYWORD);

<etc>"("			is(LPAREN);
<etc>")"			is(RPAREN);
<etc>","			is(COMMA);
<etc>"."			is(DOT);
<etc>";"			is(SEMI);
<etc>"["			is(LBRACKET);
<etc>"]"			is(RBRACKET);
<etc>"{"			is(LBRACE);
<etc>"}"			is(RBRACE);
<etc>"::"			is(COLON_COLON);
<etc>"-"			is(MINUS);
<etc>"~"			is(TILDE);
<etc>"="			is(EQUAL);
<etc>"=="			is(EQUAL_EQUAL);
<etc>"=>"			is(ARROW);
<etc>"#("			is(SHARP_PAREN);
<etc>"#["			is(SHARP_BRACKET);
<etc>"#t"			is(SHARP_T);
<etc>"#f"			is(SHARP_F);
<etc>"#next"			is(NEXT);
<etc>"#rest"			is(REST);
<etc>"#key"			is(KEY);
<etc>"#all-keys"		is(ALL_KEYS);

<etc>[-+]?{D}+			is(INTEGER);
<etc>#x[0-9a-f]+		is(INTEGER);
<etc>#o[0-7]+			is(INTEGER);
<etc>#b[01]+			is(INTEGER);

<etc>[-+]?{D}*\.{D}+{E}?	is(FLOAT);
<etc>[-+]?{D}+\.{D}*{E}?	is(FLOAT);
<etc>[-+]?{D}+{E}		is(FLOAT);

<etc>'[ -&(-\[\]-~]'		is(CHARACTER);
<etc>'\\['\\abefnrt0]'		is(CHARACTER);

<etc>{STR}			is(STRING);

<etc>{O}			is(BINARY_OPERATOR);
<etc>{N}			is(SYMBOL);
<etc>\\{O}			is(SYMBOL);
<etc>\\{N}			is(SYMBOL);
<etc>{N}:			is(KEYWORD);
<etc>#{STR}			is(SYMBOL_LITERAL);

<ini,key,val,etc>.		is(BOGUS);

%%

static void skip_multi_line_comment(void)
{
    int depth = 1;
    int c, prev = '\0';
    
    while (1) {
	c = input();
	switch (c) {
	  case EOF:
	    return;
	  case '\n':
	    line_count++;
	    prev = c;
	    break;
	  case '/':
	    if (prev == '*')
		if (--depth == 0)
		    return;
		else
		    prev = 0;
	    else
		prev = c;
	    break;
	  case '*':
	    if (prev == '/') {
		depth++;
		prev = 0;
	    }
	    else
		prev = c;
	    break;
	  default:
	    prev = c;
	    break;
	}
    }
}

static int make_header_key()
{
  yylval.token = make_token(yytext, yyleng-1);
  return HEADER_KEY;
}
static int make_header_val()
{
  char *p1, *p2;
  int skipped = 0;

  for (p1 = p2 = yytext; p1 < yytext + yyleng; ) {
    /* skip initial spaces */
    while (p1 < yytext + yyleng && (*p1 == ' ' || *p1 == '\t')) {
      p1 += 1;
      skipped += 1;
    }
    /* copy to end of line */
    while (p1 < yytext + yyleng && *p1 != '\n') {
      *p2++ = *p1++;
    }
    /* copy and count newline */
    *p2++ = *p1++;
    line_count += 1;
  }
  /* make the token, dropping the last newline */
  yylval.token = make_token(yytext, yyleng - skipped - 1);
  return HEADER_VAL;
}
static int make_header_end()
{
  line_count += 1;
  yylval.token = yylval.token = make_token(yytext, yyleng);
  return HEADER_END;
}

int yywrap() { return 1; }
