/***************************************************************/
/***************** Oberon-2 RunTime System *********************/
/***************************************************************/
/*  Author : Diane Corney                                      */
/*  Date:    November, 1992                                    */
/*  Oberon specific functions - to be used with generic        */
/*  runtime system - gprts.c                                   */
/*  modifications:  9/8/93 new assExTrp for ASSERT(bool,num)   */
/*                 29/9/94 un_chr = signed char for Solaris    */
/***************************************************************/

#include <stdio.h>
#include <memory.h>
#include <string.h>

#define recordMagic 0x55555500
#define objArrMagic 0x5555FF00
#define arrayMagic  0x55FF5500
#define magicOffset 2
#define descOffset  1
#define sizeOffset  1

typedef unsigned char un_chr;
typedef un_chr **bundle; /* pointer to array of pointer to char */

extern void _catcher(bundle);

/* variables required by the garbage collector */

int _o2heapLo, _o2heapHi;
int *_gp_mainStackBase;
extern int _gp_heapSize;   /* set by modbase */

char ProgArgs_FP_Overflow; /* required by gprts.c */
                           /* should be defined there */

/* error message components */

static un_chr ttErr[] = "type guard failure ";
static un_chr asErr[] = "record assign error ";
static un_chr wiErr[] = "with statement failure ";
static un_chr inMsg[] = "in module <";
static un_chr liMsg[] = "> at line ";
static un_chr meErr[] = "memory error - out of oberon heap space ";
static un_chr endM1[] = "coroutine ended without TRANSFER";

static un_chr *ttMsg[] = {ttErr, inMsg, 0, liMsg, 0, 0};
static un_chr *asMsg[] = {asErr, inMsg, 0, liMsg, 0, 0};
static un_chr *wiMsg[] = {wiErr, inMsg, 0, liMsg, 0, 0};
static un_chr *meMsg[] = {meErr, 0};
static un_chr *endMS[] = {endM1, 0};

/************************ Traps ************************************/

static un_chr numStr[16];
    
void Recurse (int num, un_chr **str)
{
  if (num >= 10)  Recurse(num/10,str);
  **str = num % 10 + 48;
  ++(*str);
}

void NumToStr (int num)
{
  un_chr *ptr = numStr;
  Recurse(num,&ptr);
  *ptr = 0;
}

void _gp_memTrp()
{
  _catcher(meMsg);
}

void _gp_assignTrp(un_chr *mod, int line)
{
  asMsg[2] = mod;
  NumToStr(line);
  asMsg[4] = numStr;
  _catcher(asMsg);
}

void _gp_typeTrp(un_chr *mod, int line) 
{
  ttMsg[2] = mod;
  NumToStr(line);
  ttMsg[4] = numStr;
  _catcher(ttMsg);
}

void _gp_withTrp(un_chr *mod, int line)
{
  wiMsg[2] = mod;
  NumToStr(line);
  wiMsg[4] = numStr;
  _catcher(wiMsg);
}

void _gp_assExTrp(un_chr *mod, int line, int exNum)
{
  fputs("**** gp.rts: assert error in module <",stderr);
  fputs(mod,stderr);
  fputs("> at line ",stderr);
  NumToStr(line);
  fputs(numStr,stderr);
  fputs(" ****\n",stderr);
  exit(exNum);
}

void _gpo_endTrp(void)    /* known to coroutines.s */
{
    _catcher(endMS);
}

void _gp_strCopy (char *dst, char *src, int dstLength, int srcLength)
{
  int copyLength;
  if (dstLength > srcLength)
    copyLength = srcLength;
  else
    copyLength = dstLength - 1;
  for ( ; 0 < copyLength; --copyLength, ++dst, ++src) 
    *dst = *src;
  *dst = 0;
}

char _gp_strComp (char *left, int leftHigh, char *right, int rightHigh)
/* returns 0 if less, 1 if equal, 2 if greater */
{
  int i,max;

  if (leftHigh > rightHigh) 
    max = rightHigh;
  else
    max = leftHigh;
  for (i = 0; *left != 0 && *right != 0; left++, right++, i++) 
  {
    if (*left > *right) return 2;
    if (*left < *right) return 0;
    if (i == max) {
      if (leftHigh > rightHigh) {
        if (*(++left) == 0) return 1;
        return 2;
      }
      if (leftHigh < rightHigh) {
        if (*(++right) == 0) return 1;
        return 0;
      }
      return 1;
    }
  }
  if (*left > *right) return 2;
  if (*left < *right) return 0;
  return 1;
}
      
/**************** Memory Allocation *************************/

int * _gp_rtsAllocObj(int size)
{
  int *ptr;
#ifndef NOGC
  O2Memory_ALLOCATE(&ptr,size);
#else
  ptr = (int *)malloc(size);
#endif
  return (ptr + 2);
}

int * _gp_rtsAlloc(int size)
{
  return (int *)malloc(size);
}

/******************** Constructors ****************************/

typedef void (*RecConstruct)(int);

void _gp_recArrConstruct (int * arrPtr, unsigned * info, unsigned dimNo)
/*    info =  size
              num
              ...   (repeated for each dimension)
              descAdr
*/  
{
  int i,j,cpySize;
  int * ptr;
  unsigned * firstSize;

  ptr = arrPtr;
  firstSize = info;
  firstSize += 1;

  /* fill tags */

  for (i = 0; i < dimNo; i++)
  {
    *(ptr - magicOffset) = objArrMagic;
    *(ptr - sizeOffset) = *info;
    ptr += 2;
    info += 2;
  }

  /* call record constructor for first element */
  /* info now points to record descriptor address */

  if (*firstSize > 0)
  {

    (*(RecConstruct)(*(int *)((*info) - 4)))((int)ptr); 

    /* copy elements */

    for (i = 0; i < dimNo; i++)
    {
      ptr -= 2;
      info -= 2;
      cpySize = (*info - 8) / (*(info + 1));

      for (j = 1; j < *(info + 1); j++)
      {
        memcpy(ptr + j * (cpySize/sizeof(int)),ptr,cpySize); 
      }
    }
  }
}

void _gp_arrayConstruct (int * arrPtr, unsigned * info, unsigned dimNo)
/*    info =  size
              num
              ...   (repeated for each dimension)
              pointerOff
*/  
{
  int i,j,cpySize;
  int * ptr;

  ptr = arrPtr;

  /* fill tags */

  for (i = 0; i < dimNo-1; i++) {
    *(ptr - magicOffset) = objArrMagic;
    *(ptr - sizeOffset) = *info;
    ptr += 2;
    info += 2;
  }
  *(ptr - sizeOffset) = *info;
  info += 2;
  *(ptr - magicOffset) = arrayMagic + *info;

  /* info now points to ptrOffsets */
  
  if (*info > 0) { 
    /* set pointer fields to nil */
    for (i = 0; i < *(info - 1); i += *info) 
      *(ptr + i) = 0;
  }

  /* copy elements */
  for (i = 0; i < dimNo-1; i++) {
    ptr  -= 2;
    info -= 2; 
    for (j = 1; j < *(info - 1); j++)
      memcpy(ptr + j * (*info / sizeof(int)),ptr,*info); 
  }
}

/**************** Memory Deallocation *************************/

typedef void (*MarkProc) (int *,unsigned);
typedef void (*RecDestruct) (int *,MarkProc);

void _gp_Ptr_Destruct(int *ptr, MarkProc mProc);

#define magicMask 0xFFFFFFF0
#define ptrMask   0x0000000F

void _gp_Array_Destruct(int *ptr, MarkProc mProc)
{
  unsigned ptrOff, size, i;

  if (ptr)  /* check for nil pointer */ 
  {

/* clear magic number */

    *ptr = 0;
    size = (unsigned) *(ptr - 1);
    ptrOff = (unsigned) ((*(ptr - 2)) & ptrMask);
    (*(MarkProc)(mProc))(ptr-2,size+8);
    if (ptrOff)
    {
      for (i = 0; i < size; i += ptrOff)
      {
        _gp_Ptr_Destruct((int *) *(ptr + i),mProc); 
      }
    }
    *ptr = arrayMagic;
  }
}


void _gp_RecArray_Destruct(int *ptr, MarkProc mProc, unsigned numElem)
{
  unsigned recSize, offset;
  int *desc, i, magicStore;
  

  if (ptr)  /* check for nil pointer */
  {
/* clear magic number */

    magicStore = *ptr;
    *ptr = 0;
    (*(MarkProc)(mProc))(ptr-2,8);
    desc = (int *) *(ptr + 1);
    recSize = *desc;
    if (! numElem)
    {
      numElem = ((unsigned) *(ptr - 1))/recSize;
    } 
    recSize /= sizeof(int); 
    ptr += 2;
    for (i = 0, offset = 0; i < numElem; i++, offset += recSize)
      (*(RecDestruct)(*(desc - 2)))((int *)(ptr + offset),mProc); 
    *ptr = magicStore;
  }
}  

void _gp_Ptr_Destruct(int *ptr, MarkProc mProc) 
{
  int magic;
  unsigned numElems, recSize;
  int *desc;

  if (ptr)  /* check for nil pointer */
  {
    magic = *(ptr - 2);
    if (magic == recordMagic) {
      desc = (int *) *(ptr - 1);
      (*(RecDestruct)(*(desc - 2)))(ptr,mProc); 
    } else if (magic == objArrMagic) {
      desc = (int *) *(ptr + 1);
      recSize = *desc;
      numElems = ((unsigned) *(ptr - 1))/recSize;
      _gp_RecArray_Destruct(ptr,mProc,numElems);
    } else if ((magic & magicMask) == arrayMagic)
      _gp_Array_Destruct(ptr,mProc);
  }
}

/****************************************************************/

void InitO2RTS(int dummy)
{
#ifdef NOGC
  _gp_heapSize = _o2heapHi = _o2heapLo = 0;
#else
  int ptr;

  /* allocate oberon heap */

  _gp_mainStackBase = &dummy;
  _gp_heapSize += 8 - (_gp_heapSize % 8);
  ptr = (int)malloc(_gp_heapSize+8);
  if (!ptr)
  {
    fputs("**** gprts: heap error: cannot allocate oberon heap ****\n",stderr);
    exit(1);
  } else {
    if (ptr % 8)
      _o2heapLo = ptr + 8 - (ptr % 8); 
    else
      _o2heapLo = ptr;

    _o2heapHi = _o2heapLo + _gp_heapSize;
  }
#endif
}


