IMPLEMENTATION MODULE Graphics;

(* Revision record:

   30 Aug 90  John Hynd		Portable colour table initialisation
				Add PlaceGraphWindow
*)

FROM Xglobal IMPORT unsignedShort, unsignedInt, Bool, long;
FROM Xlib IMPORT DisplayPtr, Window, GC, XSetForeground,
                XOpenDisplay, XDefaultScreen, XWhitePixel, XBlackPixel,
                XCreateSimpleWindow, XDefaultRootWindow, XStoreName,
                XEvent, XEventsQueued, QueuedAfterFlush, XPending,
		XSelectInput, ExposureMask,
		XQueryPointer, XClearWindow,
		XAllocColor, Status, Fail,
		XAllocColorCells, XQueryColor,
		XCopyColormapAndFree, XInstallColormap,
		XCreateColormap,
		XDefaultVisual, AllocAll, AllocNone,
		ButtonPressMask, XNextEvent, XButtonEvent, ButtonPress,
		XDrawPoint, XFillRectangle, XDrawRectangle, XDefaultGC,
                XDrawString, Colormap, XColor, XStoreColor, XDrawLine,
                XMapRaised, XFlush, XDefaultColormap,
                XDestroyWindow, XCloseDisplay;
FROM StdError IMPORT WriteString;
FROM SYSTEM IMPORT CAST;
FROM InOut IMPORT WriteCard, WriteLn;

CONST
  numColours = 255;

TYPE
  DrawDotProcedure = PROCEDURE (CARDINAL, CARDINAL, CARDINAL);

VAR
  mydisplay : DisplayPtr;
  mywindow : Window;
  myscreen : INTEGER;
  myforeground, mybackground : CARDINAL ;
  mygc : GC;
  myColourMap : Colormap;
  opened : BOOLEAN;
  colourTable : ARRAY [0..255] OF CARDINAL;

  PROCEDURE ForegroundColour() : CARDINAL; 
  BEGIN RETURN myforeground END ForegroundColour;

  PROCEDURE BackgroundColour() : CARDINAL; 
  BEGIN RETURN mybackground END BackgroundColour;

  PROCEDURE EventHappened () : BOOLEAN;

  (* Returns TRUE iff an event is waiting in the queue *)

  BEGIN (* EventHappened *)
    RETURN (XPending (mydisplay) > 0);
  END EventHappened;

  PROCEDURE ThrowEvent;

  (* Throws away the first event in the queue *)

  VAR
    throw : XEvent;

  BEGIN (* ThrowEvent *)
    XNextEvent(mydisplay, throw);
  END ThrowEvent;

  PROCEDURE Assert (cond : BOOLEAN; message : ARRAY OF CHAR);

  BEGIN (* Assert *)
    IF NOT cond THEN
      WriteString (message);
      IF opened THEN CloseScreen; END;
      HALT;
    END; (* IF *)
  END Assert;

  PROCEDURE SetUpColours (VAR colourMap : Colormap; VAR display : DisplayPtr);

  VAR
    colour, r, g, b, count : INTEGER;
    xColour : XColor;
    throw : Status;

    (*
     *  this procedure produces a wild palette of 
     *  colours with big changes between each neighbour
     * 
    PROCEDURE NextColour (VAR r, g, b : INTEGER);
    BEGIN (* NextColour *)

      colour := colour + 3 * 64 + 1 * 8 + 1;

      r := colour MOD 8;
      g := (colour DIV 8) MOD 8;
      b := (colour DIV (8 * 8)) MOD 8;
      r := r * 32;
      g := g * 32;
      b := b * 32;
    END NextColour;
     *)

    (*
     *  this procedure produces a ramp of primaries
     *  and secondary colours
     *) 
    PROCEDURE NextColour (VAR r, g, b : INTEGER);
      VAR thing : BITSET;
	  lsbs  : CARDINAL;
    BEGIN (* NextColour *)

      thing := CAST(BITSET,count DIV 16);
      lsbs  := count MOD 16;

      IF 0 IN thing THEN r := lsbs * 16 ELSE r := 0 END;
      IF 1 IN thing THEN g := lsbs * 16 ELSE g := 0 END;
      IF 2 IN thing THEN b := lsbs * 16 ELSE b := 0 END;

    END NextColour;

  BEGIN (* SetUpColours *)
    colour := 0;
    WITH xColour DO
      flags := CHR(7);
      r := 0;
      g := 0;
      b := 0;
      FOR count := 20 TO 255 DO
	red := r * 256;
	green := g * 256;
	blue := b * 256;
	throw := XAllocColor (display, colourMap, xColour);
	colourTable [count] := pixel;
	NextColour (r, g, b);
(*
	WriteCard (r, 10); WriteCard (g, 10); WriteCard (b, 10); 
	WriteLn; 
 *)
      END; (* FOR *)
    END; (* WITH *)
    (* Ensure that background maps to background *)
    colourTable [mybackground] := mybackground;
  END SetUpColours;

  PROCEDURE PlaceGraphWindow (name : ARRAY OF CHAR; top, left : CARDINAL;
						    xSize, ySize : CARDINAL);

  VAR
    throw : INTEGER;

  BEGIN
    IF opened THEN
      XSetForeground (mydisplay, mygc, mybackground);
      XFillRectangle (mydisplay, mywindow, mygc, top, left, xSize, ySize);
      RETURN;
    END; (* IF *)
    (* Connect to the server: *)
    mydisplay := XOpenDisplay("");
    IF mydisplay = NIL THEN
      WriteString ("Cannot connect to server");
      WriteLn;
      HALT;
    END;
    (* Get a screen: *)
    myscreen := XDefaultScreen(mydisplay);
    (* Look up "black" and "white": *)
    mybackground := XWhitePixel(mydisplay, myscreen);
    myforeground := XBlackPixel(mydisplay, myscreen);
    mywindow := XCreateSimpleWindow (mydisplay,
                        XDefaultRootWindow(mydisplay), top, left, xSize, ySize, 2,
                        myforeground, mybackground);
    (* Give the window manager a window name hint *)
    XStoreName(mydisplay, mywindow, name);
    (* Pop this window up on the screen: *)
    mygc := XDefaultGC (mydisplay, myscreen);
    myColourMap := XDefaultColormap (mydisplay, myscreen);
(*    myColourMap := XCreateColormap (mydisplay, mywindow,
				XDefaultVisual (mydisplay, myscreen),
				AllocAll); *)
    SetUpColours (myColourMap, mydisplay);
    XMapRaised(mydisplay, mywindow);
    XSelectInput(mydisplay, mywindow, ExposureMask);
    REPEAT
    UNTIL EventHappened ();
    ThrowEvent;
    XSelectInput (mydisplay, mywindow, ButtonPressMask);
    opened := TRUE;
  END PlaceGraphWindow;

  PROCEDURE GraphWindow (name : ARRAY OF CHAR; xSize, ySize : CARDINAL);

  BEGIN
    PlaceGraphWindow (name, 0,0, xSize, ySize);
  END GraphWindow;

  PROCEDURE GraphMode (name : ARRAY OF CHAR);

  BEGIN (* GraphMode *)
    GraphWindow (name, 640, 400);
  END GraphMode;

  PROCEDURE DrawBlock (x, y, width, height, colour : CARDINAL);

  BEGIN (* DrawBlock *)
    Assert (opened, "DrawBlock: graph screen not opened");
    XSetForeground (mydisplay, mygc, colourTable [colour]);
    XFillRectangle (mydisplay, mywindow, mygc, x, y, width, height);
  END DrawBlock;

  PROCEDURE Dot (x, y, colour : CARDINAL);

  BEGIN (* Dot *)
    Assert (opened, "Dot: graph screen not opened");
    (* Draw a rectangle on mywindow at (x,y), width 1, height 1 *)
    DrawBlock (x * 2, y * 2, 2, 2, colour);
  END Dot;

  PROCEDURE HiResDot (x, y, colour : CARDINAL);

  BEGIN (* Dot *)
    Assert (opened, "Dot: graph screen not opened");
    (* Draw a rectangle on mywindow at (x,y), width 1, height 1 *)
    XSetForeground (mydisplay, mygc, colourTable [colour] );
    XDrawPoint (mydisplay, mywindow, mygc, x, y);
  END HiResDot;

  PROCEDURE Flush;

  BEGIN (* Flush *)
    Assert (opened, "Flush: graph screen not opened");
    XFlush (mydisplay);
  END Flush;

  PROCEDURE GenericLine (x0, y0, x1, y1, colour : CARDINAL;
			 dotProcedure : DrawDotProcedure);

  CONST
    Infinity = MAX (CARDINAL);

  VAR
    xCount, yCount, xInc, yInc, xThreshold, yThreshold, dotCount : CARDINAL;
    yStep : INTEGER;

    PROCEDURE Swap (VAR n1, n2 : CARDINAL);

    VAR
      temp : CARDINAL;

    BEGIN (* Swap *)
      temp := n2;
      n2 := n1;
      n1 := temp;
    END Swap;

  BEGIN (* GenericLine *)
    Assert (opened, "Line: graph screen not opened");
    IF x1 < x0 THEN
      Swap (x0, x1); Swap (y0, y1);	(* Always draw left to right *)
    END; (* IF *)
    IF y1 > y0 THEN
      xThreshold := y1 - y0;
      yStep := 1;
    ELSE
      xThreshold := y0 - y1;
      yStep := -1;
    END; (* IF *)
    yInc := xThreshold;
    yThreshold := x1 - x0;
    xInc := yThreshold;
    xCount := xThreshold DIV 2;
    yCount := yThreshold DIV 2;
    IF yInc > xInc THEN
      dotCount := yInc + 1
    ELSE
      dotCount := xInc + 1
    END; (* IF *)
    WHILE (dotCount > 0) DO
      dotProcedure (x0, y0, colour);
      DEC (dotCount);
      xCount := xCount + xInc;
      yCount := yCount + yInc;
      IF xCount > xThreshold THEN
        xCount := xCount - xThreshold;
        INC (x0);
      END; (* IF *)
      IF yCount > yThreshold THEN
        yCount := yCount - yThreshold;
        y0 := CAST (CARDINAL, CAST (INTEGER, y0) + yStep);
      END; (* IF *)
    END; (* WHILE *)
  END GenericLine;

  PROCEDURE Line (x0, y0, x1, y1, colour : CARDINAL);

  BEGIN (* Line *)
    GenericLine (x0, y0, x1, y1, colour, Dot);
  END Line;

  PROCEDURE HiResLine (x0, y0, x1, y1, colour : CARDINAL);

  BEGIN (* Line *)
    XSetForeground (mydisplay, mygc, colourTable [colour]);
    XDrawLine (mydisplay, mywindow, mygc, x0, y0, x1, y1);
    GenericLine (x0, y0, x1, y1, colour, HiResDot);
  END HiResLine;

  PROCEDURE CloseScreen; (* Kills the graphics window for cleanup purposes *)

  BEGIN (* CloseScreen *)
    Assert (opened, "CloseScreen: graph screen not opened");
    XDestroyWindow (mydisplay, mywindow);
    XCloseDisplay (mydisplay);
  END CloseScreen;

  PROCEDURE InitColourTable;

  VAR
    count : CARDINAL;

  BEGIN (* InitColourTable *)
    FOR count := 0 TO numColours DO colourTable [count] := count; END;
(* Omit special (fire-specific?) mappings: *)
    colourTable [0] := 1;
    colourTable [1] := 3;
    colourTable [2] := 7;
    colourTable [3] := 0;
    colourTable [4] := 6;
    colourTable [5] := 8;
    colourTable [6] := 2;
    colourTable [7] := 10;
(* But ensure that background is correct - see above *)
  END InitColourTable;

  PROCEDURE GetMouseClick (VAR x, y, button : INTEGER);

  VAR
    myevent : XEvent;

  BEGIN (* GetMouseClick *)
    XSelectInput(mydisplay, mywindow, ButtonPressMask);
    LOOP
      (* Get the next event in the event queue: *)
      XNextEvent(mydisplay, myevent);
      IF myevent.type = ButtonPress THEN
        x := myevent.xbutton.x;
        y := myevent.xbutton.y;
        button := myevent.xbutton.state;
        RETURN; (* RETURN here!! *)
      END; (* IF *)
    END; (* LOOP *)
  END GetMouseClick;

  PROCEDURE GetBox (VAR x0, y0, x1, y1, button : INTEGER);

  VAR
    throw : Bool;
    xRoot, yRoot, x, y : INTEGER;
    buttons : unsignedInt;
    rootwin, childwin : Window;

  BEGIN (* GetBox *)
    GetMouseClick (x0, y0, button);
    REPEAT
      throw := XQueryPointer (mydisplay, mywindow, rootwin, childwin,
				xRoot, yRoot, x, y, buttons);
    UNTIL (CAST (BITSET, button) * CAST (BITSET, buttons)) = BITSET {};
    HiResLine (x0, y0, x1, y0, 3);
    HiResLine (x0, y0, x0, y1, 3);
    HiResLine (x0, y1, x1, y1, 3);
    HiResLine (x1, y1, x1, y0, 3);
  END GetBox;

BEGIN (* Graphics *)
  InitColourTable;
  opened := FALSE;
END Graphics.
