(****************************************************************************)
(*                  Oberon-2 Memory Management module                       *)
(*                                                                          *)
(*      This modules performs the heap memory allocation and garabage       *)
(*      collection for Oberon 2                                             *)
(*                                                                          *)
(*      Author :   Brian Meilak    Started 30/08/92                         *)
(*		   John Gough      modified 2-Mar-92			    *)
(*                                                                          *)
(*                                                                          *)
(*      THIS MODULE MUST BE COMPILED WITH RANGE CHECKING TURNED OFF!        *)
(*                                                                          *)
(****************************************************************************)

IMPLEMENTATION MODULE O2Memory;

IMPORT O2RTS;

(*
IMPORT Storage;
*)
(* -------------------------- 
FROM Graphics IMPORT GetMouseClick, PlaceGraphWindow, CloseScreen, 
	HiResDot, Flush, DrawBlock, Dot, EventHappened;
 -------------------------- *)
IMPORT SYSTEM;
IMPORT ProgArgs;
IMPORT Types;

(****************************************************************************)
(*                           Bitmap Declarations                            *)
(****************************************************************************)
(* $R- *)
(* $I- *)
(* $T- *)

  CONST	bitsPerByte	= 8;		(* beware with Alpha, etc !!! *)
	bitsPerHalf	= 16;
	bitsPerWord	= 32;
	bytesPerHalf	= 2;
	bytesPerWord	= 4;
	bytesPerReal	= 8;

  CONST	bytesPerMapBit	= 8;
	bytesPerMapByte	= bytesPerMapBit * bitsPerByte;
	bytesPerMapHalf	= bytesPerMapBit * bitsPerHalf;
	bytesPerMapWord	= bytesPerMapBit * bitsPerWord;

  CONST limit		= 1000000H;  (* 16 megabytes ... *)

  TYPE  CardPtr    = POINTER TO CARDINAL;
        CardArrPtr = POINTER TO ARRAY [0 .. MAX(CARDINAL)] OF CARDINAL;
	BitRange   = [0..31];
  	ByteMask   = ARRAY BitRange OF CARDINAL;  (* arrays to help in the  *)
                                                  (* byte/half/word masking *)
                                                  (* process                *)

      (* our Dummy HEAP! *)
	HeapPointer	= POINTER TO 
                             ARRAY [0 .. limit DIV bytesPerReal] 
				OF REAL;

        BitMapTypes  = (byte, half, word, bits);
        BitMap = POINTER TO
                   RECORD 
                    CASE (* blank *) : BitMapTypes OF
		    | byte : byteArr : ARRAY [0..limit] OF Types.BYTECARD;
		    | half : halfArr : ARRAY [0..limit] OF Types.SHORTCARD;
		    | word : wordArr : ARRAY [0..limit] OF CARDINAL;
		    | bits : bitsArr : ARRAY [0..limit] OF BITSET;
                    END;
                   END;


  CONST (* Little-endian *)
	lowMask     = ByteMask{000000001H,000000003H,000000007H,00000000FH,
                               00000001FH,00000003FH,00000007FH,0000000FFH,
                               0000001FFH,0000003FFH,0000007FFH,000000FFFH,
                               000001FFFH,000003FFFH,000007FFFH,00000FFFFH,
                               00001FFFFH,00003FFFFH,00007FFFFH,0000FFFFFH,
                               0001FFFFFH,0003FFFFFH,0007FFFFFH,000FFFFFFH,
                               001FFFFFFH,003FFFFFFH,007FFFFFFH,00FFFFFFFH,
                               01FFFFFFFH,03FFFFFFFH,07FFFFFFFH,0FFFFFFFFH};

	highMask    = ByteMask{080000000H,0C0000000H,0E0000000H,0F0000000H,
			       0F8000000H,0FC000000H,0FE000000H,0FF000000H,
			       0FF800000H,0FFC00000H,0FFE00000H,0FFF00000H,
			       0FFF80000H,0FFFC0000H,0FFFE0000H,0FFFF0000H,
			       0FFFF8000H,0FFFFC000H,0FFFFE000H,0FFFFF000H,
			       0FFFFF800H,0FFFFFC00H,0FFFFFE00H,0FFFFFF00H,
			       0FFFFFF80H,0FFFFFFC0H,0FFFFFFE0H,0FFFFFFF0H,
			       0FFFFFFF8H,0FFFFFFFCH,0FFFFFFFEH,0FFFFFFFFH};

  VAR
	occupied	 : BitMap;	(* the currently used bitmap  *)
	spareMap	 : BitMap;	(* the map under construction *)

	realHeapSize	 : CARDINAL;	(* in bytes *)
	realBitmapSize	 : CARDINAL;	(* in bytes *)

	lastByteAlloc	 : CARDINAL;
	lastHalfAlloc	 : CARDINAL;
	lastWordAlloc	 : CARDINAL;

	maxByteIndex	 : CARDINAL;
	maxHalfIndex	 : CARDINAL;
	maxWordIndex	 : CARDINAL;

	garbageCollected : BOOLEAN;

(****************************************************************************)
(*                     Garbage Collector Code                               *)
(****************************************************************************)

   (* --------------------------------------------------------------- *
    * PRE: adr represents an address that is mapped by the Bitmap. 
    * Mark off in the bitmap at address adr , the area represented by
    * 'size' bytes in "bytesPerBitMap" byte chunks. 
    *)
    PROCEDURE Mark(adr : CARDINAL; size : CARDINAL);
    VAR  
	bitOffset   : CARDINAL; (* bit index of first *)
	extraBits   : CARDINAL;	(* number of bits - 1 *)
	firstBitset : CARDINAL; (* index of first set *)
	firstBitPos : CARDINAL; (* bit index in word1 *)
	lastBitset  : CARDINAL; (* index of last set  *)
	lastBitPos  : CARDINAL; (* bit index in wordN *)
        ix          : CARDINAL;
        tmp         : CARDINAL;
    BEGIN

     (* offset from start of bitmap in bits ... *)
      bitOffset := (adr - O2RTS._o2heapLo) DIV bytesPerMapBit;

      extraBits   := size DIV bytesPerMapBit - 1; (* assert size mod 8 = 0 *)
      firstBitset := bitOffset DIV bitsPerWord;
      firstBitPos := bitOffset MOD bitsPerWord;
      lastBitset  := (bitOffset + extraBits) DIV bitsPerWord;
      lastBitPos  := (bitOffset + extraBits) MOD bitsPerWord;

     (* clear the bit in the occupied map for this *)
      EXCL(occupied^.bitsArr[firstBitset], firstBitPos);
 
(* -------------------------- 
      FOR ix := bitOffset TO bitOffset + extraBits DO
        HiResDot(ix MOD 256, ix DIV 256, 1);
      END;
      Flush();
 -------------------------- *)

     (*
      *  Mark it in spareMap! 
      *
      *  Now first bit to mark is bitOffset,
      *  the last bit to mark is bitOffset + extraBits
      *
      *
      *)
      IF lastBitset = firstBitset THEN
        spareMap^.bitsArr[firstBitset] := spareMap^.bitsArr[firstBitset] +
                                          BITSET{firstBitPos .. lastBitPos};


      ELSE (* multiple words *)
	ix := firstBitset;
	tmp := highMask[31-firstBitPos];
        spareMap^.bitsArr[ix] := spareMap^.bitsArr[ix] + 
                                 SYSTEM.CAST(BITSET,tmp);
        INC(ix);
        tmp := lowMask[bitsPerWord-1];
        WHILE ix < lastBitset DO
          spareMap^.bitsArr[ix] := spareMap^.bitsArr[ix] +
                                   SYSTEM.CAST(BITSET,tmp);
          INC(ix);
        END;
        tmp := lowMask[lastBitPos]; 
	spareMap^.bitsArr[ix] := spareMap^.bitsArr[ix] + 
                                 SYSTEM.CAST(BITSET,tmp);
      END;
    END Mark;


   (* -------------------------------------------------------- *
    *  Register this object as non-garbage, if it has magic.
    *  Assert that ptr is not NIL, and points to heap. If
    *  the magic is known, then use one of CollectXXXX();
    *)
    PROCEDURE RegisterThisObject(adr : CARDINAL);
      VAR magicPtr  : CardPtr;
	  protoMagic : CARDINAL;
    BEGIN
      ProgArgs.Assert(adr # 0);
       (* look for the magic ...   *)
	magicPtr   := SYSTEM.CAST(CardPtr,adr - O2RTS._magicOffset);
	protoMagic := magicPtr^;

        IF protoMagic = O2RTS._recordMagic THEN
	  CollectRecord(adr,Mark);
        ELSIF protoMagic = O2RTS._objArrMagic THEN
	  CollectObjArray(adr,Mark);
        ELSIF (protoMagic DIV 256) = (O2RTS._arrayMagic DIV 256) THEN
	  CollectArray(adr,Mark);
     (* ELSE do nothing *)
        END
    END RegisterThisObject;

   (* -------------------------------------------------------- *
    *  Collect this non-garbage object. It is known to 
    *  be a record object, with descriptor at adr - 4.
    *)
    PROCEDURE CollectRecord(adr : CARDINAL;
			    mrk : O2RTS.MarkerType);
      VAR valPointer : CardPtr;
	  pointerVal : CARDINAL;
	  destructor : O2RTS.DThunkType;
      VAR bitOffset : CARDINAL;
          bitset    : CARDINAL;
          bitPos    : CARDINAL;
    BEGIN
      ProgArgs.Assert(adr # 0);
      bitOffset := (SYSTEM.CAST(CARDINAL,adr) - 
			O2RTS._magicOffset -
			O2RTS._o2heapLo) DIV bytesPerMapBit;

      bitset := bitOffset DIV bitsPerWord;
      bitPos := bitOffset MOD bitsPerWord;
     (*
      *  first check, is location occupied?
      *)

      IF NOT(bitPos IN occupied^.bitsArr[bitset]) THEN RETURN END;

     (* get the descriptor pointer address ... *)
      valPointer := SYSTEM.CAST(CardPtr,adr - O2RTS._descOffset);

     (* get the descriptor pointer ... *)
      pointerVal := valPointer^;

     (* get the destructor pointer address ... *)
      valPointer := SYSTEM.CAST(CardPtr,pointerVal - O2RTS._destOffset);

     (* get the destructor address ... *)
      destructor := SYSTEM.CAST(O2RTS.DThunkType,valPointer^);

     (* now call it ... *)
      destructor(adr,mrk);
    END CollectRecord;

   (* -------------------------------------------------------- *
    *  Collect this non-garbage object. It is known to be
    *  a record array object, with descriptor at adr - 4.
    *)
    PROCEDURE CollectObjArray(adr : CARDINAL;
			      mrk : O2RTS.MarkerType);
      VAR valPointer : CardPtr;
	  sizPointer : CardPtr;
	  pointerVal : CARDINAL;
	  destructor : O2RTS.DThunkType;
	  structSize : CARDINAL;
	  index      : CARDINAL;
      VAR bitOffset : CARDINAL;
          bitset    : CARDINAL;
          bitPos    : CARDINAL;
    BEGIN
      ProgArgs.Assert(adr # 0);

      bitOffset := (SYSTEM.CAST(CARDINAL,adr) - 
			O2RTS._magicOffset -
			O2RTS._o2heapLo) DIV bytesPerMapBit;

      bitset := bitOffset DIV bitsPerWord;
      bitPos := bitOffset MOD bitsPerWord;
     (*
      *  first check, is location occupied?
      *)

      IF NOT(bitPos IN occupied^.bitsArr[bitset]) THEN RETURN END;

     (* get the structure size ... *)
      valPointer := SYSTEM.CAST(CardPtr,adr - O2RTS._descOffset);
      structSize := valPointer^ - O2RTS._tagSize;

     (* first collect the whole structure *)
      mrk(adr-O2RTS._tagSize,structSize);

     (*
      *   Now we must recursively collect elements ---
      *   get a pointer to the descriptor for element[0]
      *
      *			|=======| __ recArray magic		(adr - 8)
      *			|-------| __ total struct size		(adr - 4)
      *			|=======| __ magic of zero-th elem	(adr + 0)
      *		adr --> |-------| __ descriptor of element	(adr + 4)
      *			|-------|
      *			| elem 	| \
      *			| body	| /  zero-th element ...
      *			|=======|
      *			v  ...  v increasing addresses
      *	
      *   first, how large are the elements?
      *)
      
      INC(adr,O2RTS._tagSize);				(* addr of el[0] *)

      valPointer := SYSTEM.CAST(CardPtr,adr-O2RTS._magicOffset);
      IF valPointer^ = O2RTS._recordMagic THEN
       (*
	*   each element is a record, each of which has 
	*   an identical descriptor, and hence destructor
	*)

       (* get the descriptor pointer *)
        valPointer := SYSTEM.CAST(CardPtr,adr-O2RTS._descOffset);
        sizPointer := SYSTEM.CAST(CardPtr,valPointer^); (* addr of descr *)

       (* get the destructor pointer *)
        valPointer := SYSTEM.DECADR(sizPointer,O2RTS._destOffset);
	destructor := SYSTEM.CAST(O2RTS.DThunkType,valPointer^);

        FOR index := 0 TO structSize DIV sizPointer^ - 1 DO
  	  destructor(adr + index * sizPointer^,mrk);
        END;

      ELSIF valPointer^ = O2RTS._objArrMagic THEN
       (*
	*   each element is an array of objects,
	*   each of which has its own tag-field
	*)
        sizPointer := SYSTEM.CAST(CardPtr,adr-O2RTS._descOffset);

        FOR index := 0 TO structSize DIV sizPointer^ - 1 DO
  	  CollectObjArray(adr + index * sizPointer^,mrk);
        END;

      ELSE
       (*
	*   each element is an array of simple 
	*   objects, or an array of open arrays
	*)
        sizPointer := SYSTEM.CAST(CardPtr,adr-O2RTS._descOffset);

        FOR index := 0 TO structSize DIV sizPointer^ - 1 DO
  	  CollectArray(adr + index * sizPointer^,mrk);
        END;
      END;
    END CollectObjArray;

   (* -------------------------------------------------------- *
    *  Collect this non-garbage object. It is known to be
    *  an array object, with size at adr - 4, magic at -8.
    *)
    PROCEDURE CollectArray(adr : CARDINAL;
			   mrk : O2RTS.MarkerType);
      VAR valPointer : CardPtr;
	  stepSize   : CARDINAL;
	  destructor : O2RTS.DThunkType;
	  structSize : CARDINAL;
	  index      : CARDINAL;
      VAR bitOffset : CARDINAL;
          bitset    : CARDINAL;
          bitPos    : CARDINAL;
          i : CARDINAL;
          tmp : CardPtr;
    BEGIN
      ProgArgs.Assert(adr # 0);
      bitOffset := (SYSTEM.CAST(CARDINAL,adr) - 
			O2RTS._magicOffset -
			O2RTS._o2heapLo) DIV bytesPerMapBit;

      bitset := bitOffset DIV bitsPerWord;
      bitPos := bitOffset MOD bitsPerWord;
     (*
      *  first check, is location occupied?
      *)

      IF NOT(bitPos IN occupied^.bitsArr[bitset]) THEN RETURN END;

     (* get the structure size ... *)
      valPointer := SYSTEM.CAST(CardPtr,adr - O2RTS._descOffset);
      structSize := valPointer^;

     (* first collect the whole structure *)
      mrk(adr-O2RTS._tagSize,structSize);

     (* get the low byte of magic *)
      valPointer := SYSTEM.CAST(CardPtr,adr - O2RTS._magicOffset);
      stepSize   := valPointer^ MOD 256 * bytesPerWord;
      IF stepSize <> 0 THEN (* collect recursively *)
        FOR index := 0 TO structSize DIV stepSize - 1 DO
	  valPointer := SYSTEM.CAST(CardPtr,adr + index * stepSize);
	  IF valPointer^ <> 0 THEN 
	    RegisterThisObject(adr + index * stepSize);
	  END;
	END;
      END;
    END CollectArray;

(****************************************************************************)
(*                     End Garbage Collector Utilities                      *)
(****************************************************************************)

    PROCEDURE CollectStatic();
    (* Do the collect of memory for the static objects *)
      VAR ixOuter, 
	  ixInner : CARDINAL;
	  thisLst : O2RTS.StaticList;
	  thisPtr : CardPtr;
          ch : CHAR;
    BEGIN
      ixOuter := 0;
      thisLst := O2RTS._gp_global_Ptr_Lists[ixOuter];
     (*
      *  thisLst is the static pointer list for 
      *  one particular module of the program
      *)
      WHILE thisLst <> NIL DO
	ixInner := 0;
	thisPtr := thisLst^[ixInner];
       (*
        *  thisPtr is a member of the pointer list
        *  for a module. The value is the address
	*  of a pointer ...
        *)
	WHILE thisPtr <> NIL DO
	  IF thisPtr^ <> 0 THEN  (* ptr is not nil *)
	    RegisterThisObject(thisPtr^);
	  END;
	  INC(ixInner);
	  thisPtr := thisLst^[ixInner];
	END;
	INC(ixOuter);
	thisLst := O2RTS._gp_global_Ptr_Lists[ixOuter];
      END;
    END CollectStatic;


    PROCEDURE CollectStack();
    (* Do the collect of memory for pointers on stack *)
    VAR stackTop   : ARRAY [0 .. 0] OF CARDINAL;
	stackIndex : CARDINAL;
	maxIndex   : CARDINAL;
	thisPtr    : CARDINAL;
        ch : CHAR;
    BEGIN
      stackIndex := 0;
      maxIndex   := (O2RTS._gp_mainStackBase - 
			SYSTEM.CAST(CARDINAL,SYSTEM.ADR(stackTop))) DIV
			bytesPerWord;

      (* stack grows from high adr to low adr ie grows DOWN *)
      WHILE stackIndex <= maxIndex DO
	IF (stackTop[stackIndex] >= O2RTS._o2heapLo) AND
	   (stackTop[stackIndex] <= O2RTS._o2heapHi) THEN
          (* ptr to heap object *)

          thisPtr := stackTop[stackIndex];                       
	  RegisterThisObject(thisPtr);
        END;   (* IF IsO2HeapPtr *)
        INC(stackIndex);
      END;     (* WHILE *)
    END CollectStack;


    PROCEDURE GarbageCollect();
    (* Do the stack and pointer chain traversal *)
VAR ch : CHAR;
      VAR temp : BitMap;
    BEGIN

      IF NOT garbageCollected THEN
        ZeroSpareMap();
        CollectStatic();
        CollectStack();
       (*
	*  now we swap the occupied and spare maps!
	*)
	temp     := spareMap;
	spareMap := occupied;
	occupied := temp;

        garbageCollected := TRUE;
      ELSE
        O2RTS._gp_memTrp();
      END;
    END GarbageCollect;


    PROCEDURE InvokeGarbageCollect();
    (* Explicitly call Garbage Collector *)
    BEGIN
      garbageCollected := FALSE;
      GarbageCollect();
    END InvokeGarbageCollect;


(****************************************************************************)
(*                        Memory Map Code                                   *)
(****************************************************************************)

    PROCEDURE InitHeap();
    (* Initialise heap and bitmap. Must be explicitly called   *)
      VAR bits : CARDINAL;
    BEGIN
      garbageCollected := FALSE;

      (* calculate real array sizes *)
      realHeapSize   := O2RTS._o2heapHi - O2RTS._o2heapLo + 1; 
      realBitmapSize := realHeapSize DIV bytesPerMapWord * bytesPerWord;
      maxByteIndex  := realBitmapSize;
      maxHalfIndex  := realBitmapSize DIV bytesPerHalf;
      maxWordIndex  := realBitmapSize DIV bytesPerWord;

(* -------------------------- 
     (*
      *  open a graphical window of appropriate size ...
      *)
      bits := realBitmapSize * bitsPerByte;
      PlaceGraphWindow ("Memory Bitmap", 100, 100, 256, bits DIV 256 + 1);
 -------------------------- *)

     (* allocate memory map storage *)
      spareMap := O2RTS._gp_rtsAlloc(realBitmapSize);
      ZeroSpareMap();
      occupied := spareMap;
      spareMap := O2RTS._gp_rtsAlloc(realBitmapSize);
    END InitHeap;


    PROCEDURE ZeroSpareMap();
      VAR index : CARDINAL;
      VAR bits : CARDINAL;
    BEGIN
      FOR index := 0 TO realBitmapSize DIV bytesPerWord - 1 DO
	spareMap^.bitsArr[index] := BITSET{};
      END;
      lastByteAlloc := 0;
      lastHalfAlloc := 0;
      lastWordAlloc := 0;

(* -------------------------- 
      bits := realBitmapSize * bitsPerByte;
      DrawBlock (0,0,256,bits DIV 256 + 1,30);
      Flush;
 -------------------------- *)

    END ZeroSpareMap;

    PROCEDURE AllocateMulti(VAR adr : SYSTEM.ADDRESS;
			        siz : CARDINAL); (* size in bitMap *)
    (* Find multiple words                                         *)
    (* - try and find all complete words                           *)
    (* - if above fails, do a garbage collect and try again        *)
      VAR trial : CARDINAL;
	  extra : CARDINAL;
	  index : CARDINAL;
	  xBits : CARDINAL;
    BEGIN
      trial := lastWordAlloc;
      xBits := siz - 1;
      extra := xBits DIV bitsPerWord;
      LOOP
	LOOP
         (*  find a first free word *)
	  WHILE (occupied^.wordArr[trial] <> 0) AND
	        (trial < maxWordIndex - extra) DO INC(trial) END;
         (*  have we blown the limit *)
          IF trial >= maxWordIndex - extra THEN 
            (* time for garbage collect *)
            GarbageCollect();
            trial := 0;			
	    EXIT;			(* go round again *)
	  END;
         (*  now is there more ?    *)
	  FOR index := trial + 1 TO trial + extra DO
            IF occupied^.wordArr[index] <> 0 THEN
	      INC(trial); EXIT;
	    END;
          END;
	 (*
	  *  arriving here means a string of "extra + 1" zeros
	  *  has been found, starting at trial ...
	  *)
          FOR index := trial TO trial + extra - 1 DO
            occupied^.wordArr[index] := lowMask[31];
	  END;
          occupied^.wordArr[trial + extra] := lowMask[xBits MOD 32];
	  adr := SYSTEM.CAST(SYSTEM.ADDRESS,
			O2RTS._o2heapLo + trial * bytesPerMapWord);
	  lastWordAlloc := trial;
	  RETURN;
        END; (* inner loop *)
      END; (* outer loop *)
    END AllocateMulti;

    PROCEDURE AllocateWord(VAR adr : SYSTEM.ADDRESS;
			       siz : CARDINAL);		(* size in bitMap *)
      VAR trial : CARDINAL;
    BEGIN
      trial := lastWordAlloc;
      LOOP
	WHILE (occupied^.wordArr[trial] <> 0) AND
	      (trial < maxWordIndex) DO INC(trial) END;
        IF trial >= maxWordIndex THEN 
          (* time for garbage collect *)
          GarbageCollect();
          trial := 0;				(* wrap *)
        ELSE 
          occupied^.wordArr[trial] := lowMask[siz - 1];
	  adr := SYSTEM.CAST(SYSTEM.ADDRESS,
			O2RTS._o2heapLo + trial * bytesPerMapWord);
	  lastWordAlloc := trial;
	  EXIT;
        END;
      END;
    END AllocateWord;

    PROCEDURE AllocateHalf(VAR adr : SYSTEM.ADDRESS;
			       siz : CARDINAL);		(* size in bitMap *)
      VAR trial : CARDINAL;
    BEGIN
      trial := lastByteAlloc;
      LOOP
	WHILE (occupied^.halfArr[trial] <> 0) AND
	      (trial < maxHalfIndex) DO INC(trial) END;
        IF trial >= maxHalfIndex THEN 
          (* time for garbage collect *)
          GarbageCollect();
          trial := 0;				(* wrap *)
        ELSE 
          occupied^.halfArr[trial] := lowMask[siz - 1];
	  adr := SYSTEM.CAST(SYSTEM.ADDRESS,
			O2RTS._o2heapLo + trial * bytesPerMapHalf);
	  lastByteAlloc := trial;
	  EXIT;
        END;
      END;
    END AllocateHalf;

    PROCEDURE AllocateByte(VAR adr : SYSTEM.ADDRESS;
			       siz : CARDINAL);		(* size in bitMap *)
      VAR trial : CARDINAL;
    BEGIN
      trial := lastByteAlloc;
      LOOP
	WHILE (occupied^.byteArr[trial] <> 0) AND
	      (trial < maxByteIndex) DO INC(trial) END;
        IF trial >= maxByteIndex THEN 
          (* time for garbage collect *)
          GarbageCollect();
          trial := 0;				(* wrap *)
        ELSE 
          occupied^.byteArr[trial] := lowMask[siz - 1];
	  adr := SYSTEM.CAST(SYSTEM.ADDRESS,
			O2RTS._o2heapLo + trial * bytesPerMapByte);
	  lastByteAlloc := trial;
	  EXIT;
        END;
      END;
    END AllocateByte;

    PROCEDURE ALLOCATE(VAR ptr : SYSTEM.ADDRESS; size : CARDINAL);

    (* PRE: 'size' is the number of bytes required, size > 0          *)
    (* POST: ALLOCATE returns the start address of a block containing *)
    (*       N x 8 bytes                                              *)
    (*       where N = size DIV 8, IF size MOD 8 > 0 , THEN INC(N)    *) 

      CONST roundMask = bytesPerMapBit - 1;
(* -------------------------- *)
      VAR start, pixel, bits : CARDINAL;
(* -------------------------- *)
    BEGIN
     (* round up just in case *)
      bits := (size + roundMask) DIV bytesPerMapBit;
      IF    bits <= bitsPerByte THEN
        AllocateByte(ptr,bits);
      ELSIF bits <= bitsPerHalf THEN
        AllocateHalf(ptr,bits);
      ELSIF bits <= bitsPerWord THEN
        AllocateWord(ptr,bits);
      ELSE                                           (* Multiple words! *)
        AllocateMulti(ptr,bits);
      END;
(* -------------------------- 
      start := SYSTEM.DIFADR(ptr,
		SYSTEM.CAST(SYSTEM.ADDRESS,O2RTS._o2heapLo)) DIV 
			bytesPerMapBit;
      FOR pixel := start TO start + bits - 1 DO
        HiResDot (pixel MOD 256, pixel DIV 256, 1);
      END;
      Flush;
 -------------------------- *)
      garbageCollected := FALSE;

    END ALLOCATE;

(****************************************************************************)
(*                        O2Memory Main                                     *)
(****************************************************************************)


BEGIN (* main *)
  IF O2RTS._o2heapHi > O2RTS._o2heapLo THEN
    InitHeap();
  END;
END O2Memory.




