(**************************************************************************)
(*                                                                        *)
(*  Program to work out module dependencies in a Modula-2 program.        *)
(*  Copyright (C) 2019   Peter Moylan                                     *)
(*                                                                        *)
(*  This program is free software: you can redistribute it and/or modify  *)
(*  it under the terms of the GNU General Public License as published by  *)
(*  the Free Software Foundation, either version 3 of the License, or     *)
(*  (at your option) any later version.                                   *)
(*                                                                        *)
(*  This program is distributed in the hope that it will be useful,       *)
(*  but WITHOUT ANY WARRANTY; without even the implied warranty of        *)
(*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *)
(*  GNU General Public License for more details.                          *)
(*                                                                        *)
(*  You should have received a copy of the GNU General Public License     *)
(*  along with this program.  If not, see <http://www.gnu.org/licenses/>. *)
(*                                                                        *)
(*  To contact author:   http://www.pmoylan.org   peter@pmoylan.org       *)
(*                                                                        *)
(**************************************************************************)

MODULE InitOrder;

        (********************************************************)
        (*                                                      *)
        (*       Working out the module initialisation          *)
        (*            order of a Modula-2 program               *)
        (*                                                      *)
        (*  Programmer:         P. Moylan                       *)
        (*  Started:            25 May 2019                     *)
        (*  Last edited:        25 September 2019               *)
        (*  Status:             Working                         *)
        (*                                                      *)
        (*  This is a new version that ignores the .DEF files   *)
        (*  and works only with .MOD files.  The main reason    *)
        (*  for checking the definition modules would be to     *)
        (*  find circular dependencies, but the compiler checks *)
        (*  that, so an extra check here would be redundant.    *)
        (*                                                      *)
        (********************************************************)


IMPORT TextIO, Strings, IOChan, ChanConsts, SeqFile;

FROM IOChan IMPORT
    (* type *)  ChanId;

FROM STextIO IMPORT
    (* proc *)  WriteChar, WriteString, WriteLn;

FROM ProgramArgs IMPORT
    (* proc *)  ArgChan, IsArgPresent;

FROM Files IMPORT
    (* type *)  FilenameString,
    (* proc *)  CreatePathTable, LocateModule;

FROM Scanner IMPORT
    (* proc *)  StripSpaces, StartScan, Scan;

FROM Storage IMPORT
    (* proc *)  ALLOCATE, DEALLOCATE;

(********************************************************************************)

CONST
    Nul = CHR(0);

    (* An option controlling the order of the output.  The Modula-2     *)
    (* standard does not, as far as I know, specify the order of        *)
    (* initialisation of modules declared at the same level.  I would   *)
    (* naturally assume order of declaration, but the following option  *)
    (* also allows for reverse order of declaration.  Unfortunately it  *)
    (* appears that XDS uses yet another order.  This will take a bit   *)
    (* more investigation.                                              *)

    ReverseOrder = TRUE;

    (* Options used only during program testing. *)

    Testing = FALSE;
    TestParam = "test";                     (* used only when testing *)
    (*TestParam = "D:\myapps2\wsu\setup";      (* used only when testing *)*)

TYPE
    InfoIndex = [0..2047];
    ExtendedInfoIndex = [0..MAX(InfoIndex)+1];

    (* The fields in a ModuleData record are:                           *)
    (*     name         the name of the module                          *)
    (*     filename     the name in the file system                     *)
    (*     done         means we have no further need to deal with      *)
    (*                      this module.                                *)

    ModuleData = RECORD
                     name, filename: FilenameString;
                     done: BOOLEAN;
                 END (*RECORD*);

    (* We keep these records in an array (see below), in no particular  *)
    (* order. (The order is in fact the order of discovery.)            *)

    (* The dependency graph is kept in an upwardly-linked tree.  We     *)
    (* don't need a full global structure for this because, to save     *)
    (* space, we discard nodes once we're certain we won't need them    *)
    (* again.                                                           *)

    NodePtr = POINTER TO NodeData;

    NodeData =  RECORD
                    NodeNo: InfoIndex;
                    parent: NodePtr;
                    next: NodePtr;
                END (*RECORD*);

VAR
    ModuleInfo: ARRAY InfoIndex OF ModuleData;

    (* ModuleInfo[NextFree] is the first unused array element. *)

    NextFree: ExtendedInfoIndex;

(************************************************************************)
(*                    PICKING UP PROGRAM ARGUMENTS                      *)
(************************************************************************)

PROCEDURE GetParameters (VAR (*OUT*) directory, modname: FilenameString);

    (* Picks up an optional program argument from the command line. *)

    VAR args: ChanId;
        ParameterString: ARRAY [0..255] OF CHAR;
        pos1, pos2: CARDINAL;
        found1, found2: BOOLEAN;

    BEGIN
        IF Testing THEN
            ParameterString := TestParam;
        ELSE
            ParameterString := "";
            args := ArgChan();
            IF IsArgPresent() THEN
                TextIO.ReadString (args, ParameterString);
            END (*IF*);
        END (*IF*);

        StripSpaces (ParameterString);
        Strings.Assign (ParameterString, modname);

        directory := "";
        IF modname[0] <> Nul THEN

            (* Separate the result into "directory" and "module name".      *)
            (* We assume the directory separator is either '/' or '\'.      *)

            Strings.FindPrev ('/', modname, LENGTH(modname)-1, found1, pos1);
            Strings.FindPrev ('\', modname, LENGTH(modname)-1, found2, pos2);
            IF NOT found1 OR (found2 AND (pos2 < pos1)) THEN
                pos1 := pos2;
            END (*IF*);
            IF found1 OR found2 THEN
                Strings.Assign (modname, directory);
                directory[pos1] := Nul;
                Strings.Delete (modname, 0, pos1+1);
            END (*IF*);

        END (*IF*);

    END GetParameters;

(************************************************************************)
(*                  WORKING OUT THE MODULE DEPENDENCIES                 *)
(************************************************************************)

PROCEDURE AddModule (name: FilenameString): InfoIndex;

    (* Puts a new module into the ModuleInfo array, unless it's         *)
    (* already there.  In either case returns the position in the       *)
    (* array.                                                           *)

    VAR pos: ExtendedInfoIndex;

    BEGIN
        pos := 0;
        WHILE (pos < NextFree)
                     AND NOT Strings.Equal(ModuleInfo[pos].name, name) DO
            INC (pos);
        END (*WHILE*);
        IF (pos = NextFree) AND (pos < MAX(InfoIndex)) THEN
            ModuleInfo[pos].name := name;
            ModuleInfo[pos].done := FALSE;
            NextFree := pos + 1;
        END (*IF*);
        RETURN pos;
    END AddModule;

(************************************************************************)

PROCEDURE FindImportsBy (this: NodePtr): NodePtr;

    (* Reads the IMPORT lines in ModuleInfo[j].filename, adds new       *)
    (* entries to ModuleInfo if appropriate.   The return value points  *)
    (* to a linear list.                                                *)

    VAR cid: IOChan.ChanId;  res: ChanConsts.OpenResults;
        head, tail: NodePtr;
        token: FilenameString;
        j: InfoIndex;

    (********************************************************************)

    PROCEDURE SkipToSemicolon;

        (* Scans forward until token is ';' or end of file reached. *)

        VAR ch: CHAR;

        BEGIN
            REPEAT
                Scan (token);
                ch := token[0];
            UNTIL (ch = ';') OR (ch = Nul);
        END SkipToSemicolon;

    (********************************************************************)

    PROCEDURE AddNode (name: FilenameString);

        (* Adds a node to the head..tail list, also updates the  *)
        (* ModuleInfo array if needed.                           *)

        (* The Modula-2 standard does not, as far as I know, specify    *)
        (* the order of initialisation of modules declared at the same  *)
        (* level.  I would naturally assume order of declaration, but   *)
        (* it looks as if the XDS system uses reverse order of          *)
        (* declaration.  I allow for both possibilities below, using    *)
        (* the global option constant ReverseOrder.  If ReverseOrder is *)
        (* TRUE then we add the new nodes at the head of the list       *)
        (* rather than at the tail.                                     *)

        VAR p: NodePtr;

        BEGIN
            NEW (p);
            p^.NodeNo := AddModule (name);
            p^.parent := this;
            IF ReverseOrder THEN
                p^.next := head;
                head := p;
                IF tail = NIL THEN
                    tail := p;
                END (*IF*);
            ELSE
                p^.next := NIL;
                IF tail = NIL THEN
                    head := p;
                ELSE
                    tail^.next := p;
                END (*IF*);
                tail := p;
            END (*IF*);
        END AddNode;

    (********************************************************************)

    BEGIN
        j := this^.NodeNo;
        head := NIL;  tail := NIL;
        SeqFile.OpenRead (cid, ModuleInfo[j].filename, SeqFile.text, res);
        IF res = ChanConsts.opened THEN
            StartScan (cid);

            (* Scan past the module header line. *)

            SkipToSemicolon;

            (* Search for IMPORT or FROM lines. *)

            LOOP
                Scan (token);
                IF token[0] = Nul THEN
                    EXIT (*LOOP*);
                ELSIF Strings.Equal (token, "IMPORT") THEN

                    (* Handle IMPORT x, y, z, ... *)

                    LOOP
                        Scan (token);
                        AddNode (token);
                        Scan (token);

                        (* We're expecting a comma at this point. *)

                        IF token[0] <> ',' THEN
                            EXIT (*LOOP*);
                        END (*IF*);
                    END (*LOOP*);

                ELSIF Strings.Equal (token, "FROM") THEN

                    (* Handle FROM x IMPORT ... *)

                    Scan (token);
                    AddNode (token);
                    SkipToSemicolon;

                ELSE
                    EXIT (*LOOP*);
                END (*IF*);

            END (*LOOP*);

            SeqFile.Close (cid);

        END (*IF*);

        RETURN head;

    END FindImportsBy;

(************************************************************************)

(*
PROCEDURE DumpList (j: InfoIndex;  q: NodePtr);

    (* For debugging: lists the modules on the "next" chain. *)

    BEGIN
        WriteString (ModuleInfo[j].name);
        WriteString (" ->(");
        WHILE q <> NIL DO
            WriteString (ModuleInfo[q^.NodeNo].name);
            q := q^.next;
            IF q <> NIL THEN
                WriteChar (" ");
            END (*IF*);
        END (*WHILE*);
        WriteChar (")");  WriteLn;
    END DumpList;
*)

(************************************************************************)

PROCEDURE Expand (p: NodePtr);

    (* Assuming ModuleInfo[j].name is already set, i.e. the module      *)
    (* name is known: recursively finds the names of all imported       *)
    (* modules, and outputs the leaf names in the order in which the    *)
    (* modules should be initialised.                                   *)

    VAR q, next: NodePtr;
        j: InfoIndex;
        IsSource: BOOLEAN;

    BEGIN
        j := p^.NodeNo;

        IF NOT ModuleInfo[j].done THEN

            (* Setting the "done" flag at this point might seem to be   *)
            (* a little premature, given that we still have to process  *)
            (* everything below us in the import tree.  This, however,  *)
            (* seems to be how XDS does module initialisation.  A       *)
            (* side-effect of this strategy is that the circular        *)
            (* dependences are handled by arbitrarily deciding that     *)
            (* this module is the top one in the relevant subtree.      *)

            ModuleInfo[j].done := TRUE;

            (* Parse the source code of the current module to create a  *)
            (* linear list of imported modules.                         *)

            WITH ModuleInfo[j] DO
                LocateModule (name, FALSE, filename, IsSource);
            END (*WITH*);
            IF IsSource THEN
                (* We don't process modules without sources. *)

                q := FindImportsBy (p);
                (*DumpList (j, q);*)

                WHILE q <> NIL DO
                    Expand (q);
                    next := q^.next;
                    DISPOSE (q);
                    q := next;
                END (*WHILE*);

                (* The above recursion should have taken care of all    *)
                (* modules below us in the import tree, so it's now     *)
                (* time to put out the current module name.             *)

                WriteString (ModuleInfo[j].name);  WriteLn;

            END (*IF*);
        END (*IF*);
    END Expand;

(************************************************************************)

PROCEDURE FindAllImports;

    VAR TreeHead: NodePtr;

    BEGIN
        NextFree := 0;
        IF ModuleInfo[0].name[0] <> Nul THEN

            (* Create the root node of the import tree. *)

            ModuleInfo[0].done := FALSE;
            NextFree := 1;
            NEW (TreeHead);
            TreeHead^.NodeNo := 0;
            TreeHead^.next := NIL;
            TreeHead^.parent := NIL;

            Expand (TreeHead);

            DISPOSE (TreeHead);

        END (*IF*);

    END FindAllImports;

(************************************************************************)
(*                           MAIN PROGRAM                               *)
(************************************************************************)

VAR StartDirectory: FilenameString;

BEGIN
    GetParameters (StartDirectory, ModuleInfo[0].name);
    IF StartDirectory[0] = Nul THEN
        StartDirectory := '.';
    END (*IF*);
    CreatePathTable (FALSE, StartDirectory);
    FindAllImports;
END InitOrder.

