IMPLEMENTATION MODULE Mouse;

        (********************************************************)
        (*                                                      *)
        (*                  Mouse driver                        *)
        (*                                                      *)
        (*  Programmer:         P. Moylan                       *)
        (*  Last edited:        28 January 1998                 *)
        (*  Status:             Partly written                  *)
        (*                                                      *)
        (*      Faults:                                         *)
        (*          - some procedures still not implemented.    *)
        (*            I might decide to remove them.            *)
        (*                                                      *)
        (********************************************************)

FROM DumpFile IMPORT DumpString, DumpCard, DumpEOL;

IMPORT OS2;

FROM SYSTEM IMPORT
    (* type *)  CARD16;

FROM LowLevel IMPORT
    (* proc *)  IAND;

        (* I don't know why I have to import the following constants,   *)
        (* given that I've already imported the enumerated types.       *)
        (* Either this is a compiler bug or it's my misunderstanding    *)
        (* of the standard.                                             *)

FROM Mouse0 IMPORT
    (* const *)  LeftButton, RightButton, MiddleButton, Motion,
                 LeftDown, MiddleDown, RightDown, LeftUp, RightUp, MiddleUp;

FROM TaskControl IMPORT
    (* proc *)  CreateTask;

(************************************************************************)

TYPE ChangeMap = ARRAY Buttons OF Events;

CONST
    (* Mappings for translating button states to events. *)

    Map1 = ChangeMap {LeftDown, RightDown, MiddleDown};
    Map2 = ChangeMap {LeftUp, RightUp, MiddleUp};

VAR
    mouse: OS2.HMOU;
    HaveMouse: BOOLEAN;
    NumberOfButtons: CARDINAL;
    ButtonState: ButtonSet;

    (* UserHandler is a user-supplied procedure that we must call when  *)
    (* a mouse event is detected; and EventsToDetect is the set of      *)
    (* events the user wants to know about.  (Events not in this set    *)
    (* will be ignored, except to the extent that this module keeps     *)
    (* a private record of the button state.)                           *)

    UserHandler: EventHandler;
    EventsToDetect: EventSet;

    (* We don't let the mouse position go outside the following limits. *)

    Limits: RECORD
                top, bottom, left, right: CARDINAL;
            END (*RECORD*);

(************************************************************************)

PROCEDURE MouseAvailable (): BOOLEAN;

    (* Returns TRUE iff a mouse driver is loaded, a mouse exists, and   *)
    (* mouse operation is permitted in module ConfigurationOptions.     *)

    BEGIN
        RETURN HaveMouse AND (NumberOfButtons > 0);
    END MouseAvailable;

(************************************************************************)

PROCEDURE ResetMouse (VAR (*OUT*) MousePresent: BOOLEAN;
                        VAR (*OUT*) NumberOfButtons: CARDINAL);

    (* Initializes mouse, returning MousePresent as FALSE if no mouse   *)
    (* available and as TRUE if it is, and NumberOfButtons as the       *)
    (* number of buttons for the mouse if installed.                    *)

    VAR Nbuttons: CARD16;

    BEGIN
        MousePresent := HaveMouse;
        OS2.MouGetNumButtons (Nbuttons, mouse);
        NumberOfButtons := Nbuttons;
        SetMouseCursorLimits (0, 24, 0, 79);
    END ResetMouse;

(************************************************************************)

PROCEDURE CheckLimits (VAR (*INOUT*) X, Y: CARDINAL);

    (* Modifies X and Y, if necessary, to ensure that they are within   *)
    (* the currently active limits.                                     *)

    BEGIN
        WITH Limits DO
            IF X < left THEN X := left
            ELSIF X > right THEN X := right
            END (*IF*);
            IF Y < top THEN Y := top
            ELSIF Y > bottom THEN Y := bottom
            END (*IF*);
        END (*WITH*);
    END CheckLimits;

(************************************************************************)

PROCEDURE GetTextMousePosition (VAR (*OUT*) Xposition: CARDINAL;
                                VAR (*OUT*) Yposition: CARDINAL);

    (* Returns the current position of the mouse cursor. *)

    VAR position: OS2.PTRLOC;

    BEGIN
        OS2.MouGetPtrPos (position, mouse);
        Xposition := position.col;
        Yposition := position.row;
        CheckLimits (Xposition, Yposition);
    END GetTextMousePosition;

(************************************************************************)

PROCEDURE GetTextMouseStatus (VAR (*OUT*) buttons: ButtonSet;
                                VAR (*OUT*) Xposition: CARDINAL;
                                VAR (*OUT*) Yposition: CARDINAL);

    (* Returns the current mouse position and state of the buttons.     *)

    BEGIN
        GetTextMousePosition (Xposition, Yposition);
        buttons := ButtonState;
    END GetTextMouseStatus;

(************************************************************************)

PROCEDURE SetTextMousePosition (Xposition: CARDINAL; Yposition: CARDINAL);

    (* Initialises the mouse position. *)

    VAR position: OS2.PTRLOC;

    BEGIN
        CheckLimits (Xposition, Yposition);
        position.row := Yposition;  position.col := Xposition;
        OS2.MouSetPtrPos (position, mouse);
    END SetTextMousePosition;

(************************************************************************)

PROCEDURE SetTextMousePage (page: CARDINAL);

    (* Sets the hardware screen page where the mouse is visible. *)

    BEGIN
        (* Operation not supported. *)
    END SetTextMousePage;

(************************************************************************)

PROCEDURE SetMouseCursorLimits (top, bottom: CARDINAL;
                                        left, right: CARDINAL);

    (* Specifies a rectangular region outside which the mouse cursor    *)
    (* may not go.                                                      *)

    VAR PtrArea: OS2.NOPTRRECT;
        status: CARDINAL;

    BEGIN
        Limits.top := top;
        Limits.bottom := bottom;
        Limits.left := left;
        Limits.right := right;

        (* The following does not seem to have any effect. *)

        WITH PtrArea DO
            row := top;  col := left;  cRow := bottom;  cCol := right;
        END (*WITH*);
        status := OS2.MouRemovePtr (PtrArea, mouse);
        IF status <> 0 THEN
            DumpString ("MouRemovePtr error code ");
            DumpCard (status);  DumpEOL;
        END (*IF*);

    END SetMouseCursorLimits;

(************************************************************************)

PROCEDURE ShowMouseCursor;

    (* Makes the mouse cursor visible on the screen. *)

    BEGIN
        (* Operation not supported. *)
    END ShowMouseCursor;

(************************************************************************)

PROCEDURE HideMouseCursor;

    (* Makes the mouse cursor invisible. *)

    BEGIN
        (* Operation not supported. *)
    END HideMouseCursor;

(************************************************************************)

PROCEDURE InstallEventHandler (DetectedEvents: EventSet;
                                        Handler: EventHandler);

    (* Nominates the procedure to be called whenever an event in the    *)
    (* set DetectedEvents occurs.                                       *)

    BEGIN
        EventsToDetect := DetectedEvents;
        UserHandler := Handler;
    END InstallEventHandler;

(************************************************************************)

PROCEDURE DummyEventHandler (E: EventSet;  B: ButtonSet;  row, col: CARDINAL);

    (* This is a "do nothing" event handler, that is used only if       *)
    (* there has been no call to InstallEventHandler.                   *)

    BEGIN
    END DummyEventHandler;

(************************************************************************)

PROCEDURE ButtonEvents (oldbuttons, newbuttons: ButtonSet): EventSet;

    (* Calculates the events corresponding to a change in button state  *)
    (* from "oldbuttons" to "newbuttons".                               *)

    VAR B: Buttons;  result: EventSet;

    BEGIN
        result := EventSet {};
        FOR B := LeftButton TO MiddleButton DO
            IF B IN newbuttons-oldbuttons THEN
                INCL (result, Map1[B]);
            ELSIF B IN oldbuttons-newbuttons THEN
            (*ELSIF (B IN oldbuttons) AND NOT (B IN newbuttons) THEN*)
                INCL (result, Map2[B]);
            END (*IF*);
        END (*FOR*);
        RETURN result;
    END ButtonEvents;

(************************************************************************)
(*                       TASK TO PICK UP MOUSE EVENTS                   *)
(************************************************************************)

PROCEDURE EventTask;

    VAR Buffer: OS2.MOUEVENTINFO;  WaitOnEmpty: CARD16;
        CurrentFlags, NewRow, NewCol: CARDINAL;
        motion, B1down, B2down, B3down: BOOLEAN;
        NewButtonState: ButtonSet;
        NewEvents: EventSet;

    BEGIN
        WaitOnEmpty := 1;
        LOOP
            OS2.MouReadEventQue (Buffer, WaitOnEmpty, mouse);
            CurrentFlags := Buffer.fs;

            (* These are the flags as reported by the system call.  Now we      *)
            (* have to decode them into the notation that we are using.         *)

            motion := IAND (CurrentFlags, 2BH) <> 0;
            B1down := IAND (CurrentFlags, 06H) <> 0;
            B2down := IAND (CurrentFlags, 18H) <> 0;
            B3down := IAND (CurrentFlags, 60H) <> 0;
            NewButtonState := ButtonSet{};
            IF B1down THEN INCL(NewButtonState, LeftButton) END (*IF*);
            IF B2down THEN INCL(NewButtonState, RightButton) END (*IF*);
            IF B3down THEN INCL(NewButtonState, MiddleButton) END (*IF*);

            (* That tells us the current state of the buttons.  Now work out    *)
            (* which buttons changed state.                                     *)

            NewEvents := ButtonEvents (ButtonState, NewButtonState);
            IF motion THEN INCL (NewEvents, Motion) END(*IF*);
            ButtonState := NewButtonState;

            WITH Buffer DO
                NewRow := row;  NewCol := col
            END (*WITH*);
            CheckLimits (NewCol, NewRow);

            (* Call the user handler. *)

            IF EventsToDetect*NewEvents <> EventSet{} THEN
                UserHandler (NewEvents, NewButtonState, NewCol, NewRow);
            END (*IF*);

        END (*LOOP*);

    END EventTask;

(************************************************************************)

BEGIN
    ButtonState := ButtonSet{};
    EventsToDetect := EventSet{};
    UserHandler := DummyEventHandler;

    HaveMouse := OS2.MouOpen(NIL, mouse) = 0;
    IF HaveMouse THEN
        ResetMouse (HaveMouse, NumberOfButtons);
        CreateTask (EventTask, 6, "Mouse events");
    ELSE
        NumberOfButtons := 0;
    END (*IF*);

END Mouse.
