(********************************************************************************************
** PROGRAM     : wSpamFilter
** VERSION     : 1.0.0
** DESCRIPTION : Simple POP3 EMail Filter
** AUTHOR      : Stuart King
** COPYRIGHT   : Copyright (c) Irie Tools, 2002. All Rights Reserved.
** NOTES       :
**    This sample program is distributed with Irie Pascal, and was written to provide
** an example of how to write Windows Sockets API programs, and how to write a POP3 client
** with Irie Pascal. To make best use of this sample you should have a basic understanding
** of Pascal as well as a basic understanding of the Sockets API, and the POP3 protocol.
** The following RFC's are relevant to the POP3 protocol (rfc 1081, rfc 2821, and rfc 2822),
** and are provided in the 'info\rfc' subdirectory (as rfc1081.txt, rfc2821.txt, and
** rfc2822.txt). Further information is available in the configuration file (wspamfilter.cfg).
** An effort was made, while writing this program, to stick to the general
** Sockets API as much as possible, and to avoid the extensions added in the Windows Sockets
** API. However in three cases this was not possible. The Windows Sockets API, requires you
** to call function "WSAStartUp" before calling any other function, and to call the function
** "WSACleanUp" when you are finished. In addition the Windows Sockets API provies the
** function "WSAGetLastError" which returns an error code whenever an error occurs.
**
**    The file "windows.inc" is a system include file, that is distributed with
** Irie Pascal. Including "windows.inc" is an easy way to access hundreds of
** functions, procedures, types, and constants defined in the Windows API. In almost all
** cases, the names of entities included by "windows.inc" are identical to the names of
** the corresponding entity in the Windows API. However there are a few unavoidable
** cases where different names are used. For example the Windows API was created for the C
** programming language, and defines the type "SOCKET" and the function "socket". Although
** "SOCKET" and "socket" differ only in case, they refer to different entities. This is not
** a problem in C which is case sensitive. However since Pascal is not case sensitive,
** "SOCKET" and "socket" are the same identifier and can not refer to different entities. So
** "windows.inc" uses "SOCKET" for the type and "createsocket" for the function.
**
**   Notice that "windows.inc" is included before the keyword "program" which normally
** marks the start of a program. Starting in version 2.5, Irie Pascal allows declarations
** to occur before the keyword "program". When this is done, the declarations are placed in
** the same invisible outermost scope, as the declarations for the built-in identifiers
** (e.g. writeln). As you probably know, if you declare an identifier your declaration will
** override any other declarations of that identifier in outer scopes. This means that the
** program below can use any declarations in "windows.inc" it wants to, and pretend that
** the other declarations don't exist. For this reason, it is recommended that you also
** include system include files before the keyword "program".
**********************************************************************************************)

(*$I windows.inc *)
program wSpamFilter;
const
	DEFAULT_MAX_MESSAGES = 200;
	MaxLine = 2016;		//Maximum length of a line.
	POP3_PORT = 110;	//TCP/IP port used to connect to the POP3 server
	CR = #13;			//Carriage Return character
	LF = #10;			//Line Feed character
	BS = #8;			//Back Space character
type
	positive = 0..maxint;
	LineType = cstring[MaxLine];

	FieldKind = (
		FieldError, FieldReceived, FieldFrom, FieldSender, FieldReplyTo,
		FieldReturnPath, FieldTo, FieldSubject, FieldMessageId, FieldXMailer
	);
	FieldBodyType = LineType;

	//FieldType is the type used to represent the POP3 email header fields.
	FieldType = record
		kind : FieldKind;
		body : FieldBodyType;
	end;

	RuleKind = (RuleKeep, RuleDelete);

	//RuleType is the type used to represent the applications filtering rules.
	RuleType = record
		kind : RuleKind;
		field : FieldType;
		ToList : LineType;
	end;
var
	bNoPeriod : boolean;		//Whether the POP server sends a PERIOD to indicate no more data
	Rules : list of RuleType;	//List of filtering rules.
	MailServerURL : LineType;	//Mail Server's URL as read from the configuration file or entered by the user.
	User: string;				//POP3 User as read from the configuration file or entered by the user.
	Password : string;			//POP3 User Password as entered by the user.
	MailServerName : cstring;	//Mail Server's name as returned by the network.
	MailServerAddress : in_addr;	//Mail Server's IP address
	DataSocket : SOCKET;		//TCP/IP socket used to send and receive data
	InData, OutData : LineType;	//Input and Output buffers used when sending and receiving data
	NumMailMessages : positive; //Number of mail messages to be read from the Mail Server
	QuitRequired : boolean;		//Whether the application should send the POP3 QUIT command
								//before exiting.
	MaxMessages : positive;

	procedure ClientShutdown; forward;

	//PURPOSE: Appends a message to the application's log file.
	//ARGUMENT(s):
	//   1. msg - The message to append to the log
	procedure LogMessage(msg : LineType);
	const
		LOG_FILENAME = 'wSpamFilter.log';
	var
		fLog : text;
	begin (* LogMessage *)
		traperrors(false);
		append(fLog, LOG_FILENAME);
		writeln(fLog, msg);
		close(fLog);
		traperrors(true);
	end; (* LogMessage *)

	//PURPOSE: Reports errors that occur when using sockets.
	//ARGUMENT(s):
	//   1. msg - The error message to report
	//NOTES:
	//   This program does not attempt to recover from socket errors so these
	// errors are fatal. This procedure will shut the server down, after
	// reporting the error. The function Windows API function "MessageBox" is called
	// to display a Windows message box on the screen.
	procedure FatalSocketError(msg : string);
	var
		iLastWSAError : integer;
		iRet : integer;
		strError : string[16];
		caption, txt : cstring;
	begin (* FatalSocketError *)
		iLastWSAError := WSAGetLastError;
		str(iLastWSAError:1, strError);
		caption := 'WinServ';
		txt := 'ERROR: ' + strError + ' ' + msg;
		writeln(txt);
		LogMessage(txt);
		iRet := MessageBox(NULL, addr(txt), addr(caption), MB_OK);
		ClientShutDown;
	end; (* FatalSocketError *)

	//PURPOSE: Waits for a socket to become ready for reading or for a socket
	//         in the listening state to become ready to accept a connection
	//         without blocking.
	//ARGUMENT(s):
	//   1. s - The socket to wait on
	//   2. wMaxWait - The maximum amount of time to wait (in milliseconds).
	//RETURNS:
	//   TRUE - If the socket is ready for reading
	//   FALSE - If the socket is not ready for reading
	//NOTES:
	//   The Windows Socket API function "select" is called to wait on the
	// socket.
	function SocketReadyForReading(s : SOCKET; wMaxWait : word) : boolean;
	var
		fdRead : fd_set;
		iRet : integer;
		//strRet : string[16];
		t : timeval;
	begin (* SocketReadyForReading *)
		fdRead.fd_count := 1;
		fdRead.fd_array[0] := s;
		t.tv_sec := wMaxWait div 1000;
		t.tv_usec := wMaxWait mod 1000;
		iRet := select(0, fdRead, null, null, t);
		//str(iRet:1, strRet);
		//LogMessage('select returned ' + strRet);
		SocketReadyForReading := (iRet=1);
	end; (* SocketReadyForReading *)

	//PURPOSE: Initializes the application
	procedure Initialize;
	var
		data : WSADATA;
		iRet : integer;

		//PURPOSE: Returns the current date and time in a format suitable for
		//         logging.
		function GetTimeStamp : string;
		var
			d : record
				day : integer;
				month : integer;
				year : integer;
				DayOfWeek : integer
			end;
			t : record
				hour : integer;
				minute : integer;
				second : integer;
			end;
			
			//PURPOSE: Converts an integer into a zero-padded string
			//ARGUMENT(s):
			//   1. i - The integer to convert
			//   2. len - The length of the string to return
			//RETURNS:
			//   The zero padded string
			function Int2String(i : integer; len : integer) : string;
			var
				S : string;
			begin (* Int2String *)
				str(i:1, s);
				while length(s) < len do
					s := '0' + s;
				Int2String := s
			end; (* Int2String *)

		begin (* GetTimeStamp *)
			getdate(d.year, d.month, d.day, d.DayOfWeek);
			gettime(t.hour, t.minute, t.second);
			GetTimeStamp := Int2String(d.year, 4) + '-' +
							Int2String(d.month, 2) + '-' +
							Int2String(d.day, 2) + ' ' +
							Int2String(t.hour, 2) + ':' +
							Int2String(t.minute, 2) + ':' +
							Int2String(t.second, 2); 
		end; (* GetTimeStamp *)

	begin (* Initialize *)
		bNoPeriod := false;
		new(Rules);
		MailServerURL := '';
		User := '';
		Password := '';
		MaxMessages := DEFAULT_MAX_MESSAGES;
		fill(MailServerAddress, 0);
		DataSocket := INVALID_SOCKET;
		InData := '';
		QuitRequired := false;

		iRet := WSAStartUp($0202, data);
		if iRet <> 0 then
			begin
				writeln('WSAStartUp call failed. Return Code=', iRet);
				ClientShutdown;
			end;
		LogMessage('----- Start POP3 Session ' + GetTimeStamp + ' -----');
	end; (* Initialize *)

	procedure ReadCommandLineOptions;
	var
		i : positive;
	begin (* ReadCommandLineOptions *)
		for i := 1 to paramcount do
			begin
				if lowercase(paramstr(i)) = '-noperiod' then
					bNoPeriod := true
			end
	end; (* ReadCommandLineOptions *)

	//PURPOSE: Reads a password from the keyboard.
	//RETURNS: The string entered from the keyboard.
	//NOTES:
	//    This procedure is used instead of "readln" because "readln"
	//echoes the characters typed to the screen, which is undesirably
	//when a password is being entered. The built-in function "readkey"
	//is used to read from the keyboard without echoing to the screen.
	function ReadPassword : string;
	const
		ENTER_KEY = 7168;
	var
		strPassword : string;
		key : integer;

		//PURPOSE: Gets the next key from the keyboard buffer
		function GetKey : integer;
		var
			k : integer;
		begin (* GetKey *)
			k := ord(readkey);
			if k=0 then
				k := ord(readkey) *256;
			GetKey := k;
		end; (* GetKey *)

		//PURPOSE: Flushes the keyboard buffer (i.e. reads all waiting keys)
		procedure FlushKeyboard;
		var
			key : integer;
		begin (* FlushKeyboard *)
			while keypressed do
				key := GetKey;
		end; (* FlushKeyboard *)

	begin (* ReadPassword *)
		FlushKeyboard;
		strPassword := '';
		repeat
			key := GetKey;
			if (key=ord(BS)) and (strPassword<>'') then
				begin
					strPassword := copy(strPassword, 1, length(strPassword)-1);
					write(BS, ' ', BS);
					flush(output);
				end
			else if (key<=ord(maxchar)) and (isalphanum(chr(key))) and (length(strPassword)<255) then
				begin
					strPassword := strPassword + chr(key);
					write('*');
					flush(output);
				end;
		until (key=ord(CR)) or (key=ENTER_KEY);
		writeln;
		flush(output);
		ReadPassword := strPassword;
	end; (* ReadPassword *)

	//PUPOSE: Gets a username from the user if none is defined.
	//GLOBAL(s):
	//    1. User - Stores the password entered by the user.
	procedure GetUser;
	begin (* GetUser *)
		if User='' then
			begin
				write('Please enter User: ');
				readln(User);
				User := trim(User);
			end;
		if User='' then
			begin
				writeln('You must enter a User');
				ClientShutdown;
			end;
	end; (* GetUser *)

	//PUPOSE: Gets a password from the user.
	//GLOBAL(s):
	//    1. Password - Stores the password entered by the user.
	// NOTES:
	//    This procedure prompts the user to enter a password and then
	//uses the "ReadPassword" procedure above to actually read the password.
	//Since a password is mandatory this procedure exits if the user does not
	//enter a password.
	procedure GetUserPassword;
	begin (* GetUserPassword *)
		write('Please enter password ');
		flush(output);
		Password := trim(ReadPassword);
		if Password='' then
			begin
				writeln('You must enter a Password');
				flush(output);
				LogMessage('No password entered');
				ClientShutdown;
			end;
	end; (* GetUserPassword *)

	//PURPOSE: Parses the field part of a rule.
	//PARAMETER(s):
	//    1. f - is used to output the result.
	//    2. line - contains the field part of the rule.
	procedure ParseField(var f : FieldType; var line : LineType);
	var
		iPos : positive;
		strFieldName : LineType;

		//PURPOSE: Examines the name of a field and identifies what
		//         kind of field it is.
		//PARAMETER(s):
		//    1. name - The name of the field
		//RETURNS:
		//    The kind of field.
		//NOTES: The name of the field must be in lowercase.
		function ParseFieldName(name : string) : FieldKind;
		var
			kind : FieldKind;
		begin (* ParseFieldName *)
			if name='received:' then
				kind := FieldReceived
			else if name='from:' then
				kind := FieldFrom
			else if name='sender:' then
				kind := FieldSender
			else if name='reply-to:' then
				kind := FieldReplyTo
			else if name='return-path:' then
				kind := FieldReturnPath
			else if (name='to:') or (name='cc:') then
				kind := FieldTo
			else if name='subject:' then
				kind := FieldSubject
			else if name='message-id:' then
				kind := FieldMessageId
			else if name='x-mailer:' then
				kind := FieldXMailer
			else
				kind := FieldError;
			ParseFieldName := kind
		end; (* ParseFieldName *)

	begin (* ParseField *)
		line := lowercase(line); //Convert field to lowercase for easy comparison
		f.kind := FieldError;	//Initialize kind of field
		strFieldName := '';		//Initialize name of field
		iPos := pos(':', line);	//Find end of name part of field.
		//
		//If a name part of the field was found then
		//   extract the name part
		//   parse it
		//   and store the rest of the field as the field body.
		//Else
		//   This field is invalid but do nothing since the kind of field
		//   has already been initialzed to indicate an error parsing the field.
		if iPos<>0 then
			begin
				strFieldName := trim(copy(line, 1, iPos));
				f.kind := ParseFieldName(strFieldName);
				line := trim(copy(line, iPos+1));
				f.body := line;
			end;
	end; (* ParseField *)

	//PURPOSE: Reads and processes the application's configuration file.
	procedure ReadConfigInfo;
	const
		ConfigFileName = 'wSpamFilter.cfg';	//Name of configuration file
		CommentChar = ';';					//Char used to indicate a comment
	var
		line : LineType;		//Buffer used to store each line of the configuration file
		fConfig : text;			//File used to read the configuration file
		iLineNo : positive;		//Number of current line in the configuration file.
								//Used when reporting errors in the configuration file
								//to indicate which line contains the error.

		//PURPOSE: Processes a (non-empty, and non-comment) line of the
		//         configuration file.
		//PARAMETER(s):
		//    1. line - contains the line to be processed.
		procedure ProcessConfigLineInfo(var line : LineType);
		var
			iPos : positive;
			strCommand : string;

			//PURPOSE: Reports an error in the application's configuration file.
			procedure ReportConfigError(msg : string);
			begin (* ReportConfigError *)
				writeln('Error processing ''', ConfigFileName, ''' line ', iLineNo:1, ' ', msg);
			end; (* ReportConfigError *)

			//PURPOSE: Finds the first whitespace character in the current line.
			//RETURNS:
			//     The position in the line of the first whitespace character
			//     or 0 if there are no white space characters in the line.
			//NOTES:
			//    This function is similar to "pos" but it searches for more
			// than one character.
        	function FindSpace : positive;
			var
				iPos, iLen : positive;
				blnDone : boolean;
			begin (* FindSpace *)
				iPos := 1;
				iLen := length(line);
				blnDone := false;
				repeat
					if iPos<=iLen then
						begin
							if isspace(line[iPos]) then
								blnDone := true
							else
								inc(iPos);
						end
					else
						blnDone := true;
				until blnDone;
				if iPos > iLen then
					iPos := 0;
				FindSpace := iPos;
			end; (* FindSpace *)

			//PURPOSE: Remove trailing comments from the current line.
			//NOTES:
			//   This procedure is called before processing the MailServer and
			// User commands, however it is not called before the other commands
			// because they may contain data that looks like a comment.
			procedure PreProcessCommand;
			var
				iSpace, iComment, iPos : positive;
			begin (* PreProcessCommand *)
				iPos := 0;
				iSpace := FindSpace;
				iComment := pos(CommentChar, line);
				if iSpace<>0 then
					iPos := iSpace;
				if (iComment<>0) and (iComment<iPos) then
					iPos := iComment;
				if iPos<>0 then
					line := copy(line, 1, iPos-1);
				line := trim(line);
			end; (* PreProcessCommand *)

			//PURPOSE: Processes a MailServer command from the config file
			procedure ProcessMailServerCommand;
			begin (* ProcessMailServerCommand *)
				PreProcessCommand;
				MailServerURL := line;
			end; (* ProcessMailServerCommand *)

			//PURPOSE: Processes a User command from the config file
			procedure ProcessUserCommand;
			begin (* ProcessUserCommand *)
				PreProcessCommand;
				User := line;
			end; (* ProcessUserCommand *)

			//PURPOSE: Processes a MaxMessages command from the config file
			procedure ProcessMaxMessagesCommand;
			var
				iVal, iErr : integer;
			begin (* ProcessMaxMessagesCommand *)
				PreProcessCommand;
				val(line, iVal, iErr);
				if (iErr = 0) and (iVal > 0) then
					MaxMessages := iVal;
			end; (* ProcessMaxMessagesCommand *)

			//PURPOSE: Processes a rule from the application's config file.
			//PARAMETER(s):
			//    1. rule - Stores the rule being processed.
			//NOTES:
			//    When the procedure is called the caller should already
			// have specified what type of rule it is and indicated this
			// in the parameter 'rule'. Also the current line should have
			// already been modified so that it contains only the field
			// part of the rule.
			procedure ProcessRule(var rule : RuleType);
			begin (* ProcessRule *)
				ParseField(rule.field, line);
				if rule.field.kind <> FieldError then
					insert(rule, Rules)
				else
					ReportConfigError('Invalid rule');
			end; (* ProcessRule *)

			//PURPOSE: Processes a delete rule.
			//NOTES:
			//    Sets the type of rule then call "ProcessRule" to
			//actully process the rule
			procedure ProcessDeleteCommand;
			var
				rule : RuleType;
			begin (* ProcessDeleteCommand *)
				rule.kind := RuleDelete;
				rule.ToList := '';
				ProcessRule(rule);
			end; (* ProcessDeleteCommand *)

			//PURPOSE: Processes a keep rule.
			//NOTES:
			//    Sets the type of rule then call "ProcessRule" to
			//actully process the rule
			procedure ProcessKeepCommand;
			var
				rule : RuleType;
			begin (* ProcessKeepCommand *)
				rule.kind := RuleKeep;
				rule.ToList := '';
				ProcessRule(rule);
			end; (* ProcessKeepCommand *)

			//PURPOSE: Processes a ToList command from the config file
			procedure ProcessToListCommand;
			var
				rule : RuleType;
			begin (* ProcessToListCommand *)
				rule.kind := RuleDelete;
				rule.ToList := line;
				rule.field.kind := FieldError;
				insert(rule, rules);
			end; (* ProcessToListCommand *)

		begin (* ProcessConfigLineInfo *)
			//
			//The first word of each line of the configuration file is the command.
			//Extract the command from the line and then call the approprate procedure
			//to process the command
			//
			iPos := FindSpace;
			if iPos>0 then
				strCommand := trim(copy(line, 1, iPos-1))
			else
				strCommand := '';
			//writeln('COMMAND ', strCommand);
			line := trim(copy(line, iPos));
			if strCommand='mailserver:' then
				ProcessMailServerCommand
			else if strCommand='user:' then
				ProcessUserCommand
			else if strCommand='maxmessages:' then
				ProcessMaxMessagesCommand
			else if strCommand='tolist:' then
				ProcessToListCommand
			else if strCommand='delete' then
				ProcessDeleteCommand
			else if strCommand='keep' then
				ProcessKeepCommand
			else
				ReportConfigError('Invalid command '+ strCommand);
		end; (* ProcessConfigLineInfo *)

	begin (* ReadConfigInfo *)
		//
		//Open the configuration and read each line until the end of the file.
		//For each line update the line number and call "ProcessConfigLineInfo"
		//to process each line that is not empty or a comment.
		//
		reset(fConfig, ConfigFileName);
		iLineNo := 0;
		while not eof(fConfig) do
			begin
				readln(fConfig, line);
				iLineNo := iLineNo + 1;
				line := lowercase(trim(line));
				if (line<>'') and (line[1]<>CommentChar) then
					ProcessConfigLineInfo(line);
			end;
		close(fConfig);
	end; (* ReadConfigInfo *)

	//PURPOSE: Finds the address of the MailServer from it's URL.
	//NOTES:
	//     It also finds the cononical name of the mail server.
	procedure FindMailServerAddress;
	var
		iRet : integer;
		pHostEnt : p_hostent;
		pAddress : ^address;
		url : cstring;
	begin (* FindMailServerAddress *)
		if (length(MailServerURL)+1) > sizeof(url) then
			begin
				writeln('Mail Server URL ''', MailServerURL, ''' is too long');
				ClientShutdown;
			end;
		url := MailServerURL;
		pHostEnt := gethostbyname(addr(url));
		if pHostEnt = nil then
			begin
				writeln('Can not find ''', url, '''');
				FatalSocketError('CAll to gethostbyname failed');
			end;
		if pHostEnt^.h_addrtype <> AF_INET then
			begin
				writeln('Invalid mail server address type');
				ClientShutdown;
			end;
		if pHostEnt^.h_length <> 4 then
			begin
				writeln('Invalid mail server address length');
				ClientShutdown;
			end;
	
		//Copy mail server name from HostEnt structure
		MailServerName := url;
		if (lstrlen(pHostEnt^.h_name)+1) < sizeof(MailServerName) then
			iRet := lstrcpy(addr(MailServerName), pHostEnt^.h_name);

		//Copy mail server address from HostEnt structure
		pAddress := pHostEnt^.h_addr_list;
		if pAddress^ = null then
			begin
				writeln('Can not find mail server address');
				ClientShutdown;
			end
		else
			move(pAddress^, MailServerAddress, sizeof(MailServerAddress));
	end; (* FindMailServerAddress *)

	//PURPOSE: Displays Mail Server information
	procedure DisplayMailServerInfo;
	const
		MAX_ASCII_ADDRESS = 15;
	var
		AsciiAddress : cstring[MAX_ASCII_ADDRESS];
		addrRet : address;
		iRet : integer;
	begin (* DisplayMailServerInfo *)
		writeln('Mail Server Name: ', MailServerName);
		write('Mail Server Address: ');
		addrRet := inet_ntoa(MailServerAddress);
		if (addrRet=null) or ((lstrlen(addrRet)+1)>sizeof(AsciiAddress)) then
			writeln('[ERROR]')
		else
			begin
				iRet := lstrcpy(addr(AsciiAddress), addrRet);
				writeln(AsciiAddress);
			end;
	end; (* DisplayMailServerInfo *)

	//PURPOSE: Creates a client data socket.
	//PARAMETER(s):
	//    1. port - The TCP/IP port to connect the socket to
	//    2. a - The IP address to connect the socket to
	//RETURNS:
	//   The created socket.
	function CreateDataSocket : socket;
	var
		s : socket;
	begin (* CreateDataSocket *)
		s := createsocket(AF_INET, SOCK_STREAM, 0);
		if s = INVALID_SOCKET then
			FatalSocketError('CreateSocket call failed');
		CreateDataSocket := s;
	end; (* CreateDataSocket *)

	//PURPOSE: Connects a client data socket
	//PARAMETER(s):
	//    1. s    - The socket to be connected
	//    2. port - The TCP/IP port to connect the socket to
	//    3. a    - The IP address to connect the socket to
	procedure ConnectDataSocket(s : socket; port : integer; a : in_addr);
	var
		sa : sockaddr_in;
		iRet : integer;
	begin (* ConnectDataSocket *)
		fill(sa, 0);
		sa.sin_family := AF_INET;
		sa.sin_port := htonl(port) shr 16;
		sa.sin_addr := a;
		iRet := connect(s, addr(sa), sizeof(sa));
		if iRet <> 0 then
			FatalSocketError('Call to connect failed');
	end; (* ConnectDataSocket *)

	//PURPOSE: Closes a socket.
	//PARAETER(s):
	//    1. s - The socket to close.
	//NOTES:
	//    First it shuts down both reading and writing (that's what SD_BOTH means),
	// and then it actually closes the socket.
	procedure CloseClientSocket(var s : socket);
	var
		i : integer;
	begin (* CloseClientSocket *)
		if s <> INVALID_SOCKET then
			begin
				writeln('Closing socket');
				i := ShutDown(s, SD_BOTH);
				i := CloseSocket(s);
				s := INVALID_SOCKET;
			end
	end; (* CloseClientSocket *)

	//PURPOSE: Reads data from a socket
	//PARAMETER(S):
	//    1. buffer - used to store the data read
	//    2. wMaxWait - maximum number of ms to wait for the socket to become ready.
	//GLOBAL(s):
	//    1. DataSocket - The socket to read from.
	//NOTES:
	//    The socket is checked to make sure it is ready for reading
	//    before attempting to read. This makes it unlikely that the application
	//    will become blocked waiting for data to come in.
	procedure ReadDataSocket(var buffer : LineType; wMaxWait : word);
	var
		iDataRead : integer;
	begin (* ReadDataSocket *)
		buffer := '';
		if SocketReadyForReading(DataSocket, wMaxWait) then
			begin
				writeln('Reading...');
				LogMessage('Reading...');
				iDataRead := recv(DataSocket, addr(buffer), sizeof(buffer)-2, 0);
				if iDataRead=SOCKET_ERROR then
					FatalSocketError('Call to recv failed')
				else if iDataRead > 0 then
					buffer[iDataRead+1] := chr(0);
			end
	end; (* ReadDataSocket *)

	//PURPOSE: Reads and logs a line of data from the data socket
	//PARAMETER(s):
	//    1. buffer - Stores the data read from the socket
	//NOTES:
	//    This procedure should only be called when exactly one line of data
	// is expected. If no lines are received the application will wait a fairly
	// long time before timing out. If more than one line is received only the
	// first line will be logged.
	procedure ReadLineDataSocket(var buffer : LineType);
	const
		MAX_WAIT_FOR_RESP = 25000;
	var
		iPos : positive;
	begin (* ReadLineDataSocket *)
		ReadDataSocket(buffer, MAX_WAIT_FOR_RESP);
		iPos := pos(CR, buffer);
		if (buffer<>'') and (iPos<>0) then
			begin
				buffer[iPos] := chr(0);
				writeln(buffer);
				LogMessage('IN ' + buffer);
				buffer[iPos] := CR
			end
		else
			begin
				writeln(buffer);
				LogMessage('IN ' + buffer);
			end;
	end; (* ReadLineDataSocket *)

	//PURPOSE: Writes and logs a line to the data socket.
	//PARAMETER(s):
	//    1. buffer - Contains the data to write to the socket.
	//NOTES:
	//    A CR/LF pair is appended to the line before writing to the socket.
	procedure WriteDataSocket(var buffer : LineType);
	var
		iRet : integer;
	begin (* WriteDataSocket *)
		writeln('Writing ', buffer);
		LogMessage('OUT ' + buffer);
		buffer := buffer + CR + LF;
		iRet := send(DataSocket, addr(buffer), length(buffer), 0);
	end; (* WriteDataSocket *)

	//PURPOSE: Checks whether a line indicates a POP3 success status
	//PARAMETER(s):
	//    1. buffer - Contains the line to be checked.
	//RETURNS:
	//    TRUE - If the line indicates success
	// or
	//    FALSE - If the line indicates failure
	//NOTES:
	//    This procedure should be called after sending a command to the
	// mail server to check whether the command was successful. The line
	// received from the mail server in response to the command should be
	// passed as the parameter.
	function CheckSuccessIndicator(var buffer : LineType) : boolean;
	const
		SUCCESS_INDICATOR = '+';
	begin (* CheckSuccessIndicator *)
		CheckSuccessIndicator := (buffer<>'') and (buffer[1]=SUCCESS_INDICATOR);
	end; (* CheckSuccessIndicator *)

	//PURPOSE: Logs on to the POP3 Mail Server.
	procedure LogOn;
	const
		MAX_ATTEMPTS=3;
	var
		attempts : positive;
		success : boolean;

		//PURPOSE: Sends the USER command to the POP3 Mail Server
		//GLOBAL(s):
		//   1. OutData - Stores the USER command to send
		//   2. InData - Contains the response from the Mail Server
		//   3. QuitRequired - Set to true once the USER command is sent
		//      because once that is done the application needs to send
		//      the QUIT command before exiting.
		//   4. User - Contains the current username
		procedure SendUserCommand;
		begin (* SendUserCommand *)
			OutData := 'USER ' + User;
			WriteDataSocket(OutData);
			QuitRequired := true;
			ReadLineDataSocket(InData);
			if not CheckSuccessIndicator(InData) then
				begin
					writeln('User ''', User, ''' rejected by server');
					ClientShutdown;
				end;
		end; (* SendUserCommand *)

		//PURPOSE: Sends the PASS command to the POP3 Mail Server
		//PARAMETER(s):
		//    1. pwd - The Users Password
		//GLOBAL(s):
		//   1. OutData - Stores the PASS command to send
		//   2. InData - Contains the response from the Mail Server
		//NOTES:
		//    This procedure calls the Windows Socket API function "send"
		// directly rather than calling "WriteDataSocket" to avoid
		// writting the users password to the log file.
		procedure SendPassCommand(pwd : string);
		var
			iRet : integer;
			buffer : LineType;
		begin (* SendPassCommand *)
			writeln('Writing PASS command');
			buffer := 'PASS ' + pwd + CR + LF;
			iRet := send(DataSocket, addr(buffer), length(buffer), 0);
			LogMessage('OUT PASS <secret>');
		end; (* SendPassCommand *)

	begin (* LogOn *)
		attempts := 0;
		repeat
			GetUser;
			GetUserPassword;

			attempts := attempts + 1;

			SendUserCommand;
			SendPassCommand(Password);

			//Read the response to the PASS command
			ReadLineDataSocket(InData);
			success := CheckSuccessIndicator(InData);
			if not success then
				writeln('Invalid user or password');
		until (attempts=MAX_ATTEMPTS) or success;
		if not success then
			ClientShutdown;
	end; (* LogOn *)

	//PURPOSE: Finds the number of messages waiting on the POP3 mail server.
	//GLOBAL(s):
	//   1. OutData - Stores the PASS command to send
	//   2. InData - Contains the response from the Mail Server
	//   3. NumMailMessages - Stores the number of mail messages
	//NOTES:
	//   The STAT command is used to determine the number of waiting messages
	// If the STAT command is successful the second word of the response
	// is the number of waiting messages. The third word is the size of
	// the messages (this is not used by this procedure).
	procedure FindNumberOfMessages;
	var
		iErr : positive;
	begin (* FindNumberOfMessages *)
		OutData := 'STAT';
		WriteDataSocket(OutData);
		ReadLineDataSocket(InData);
		if not CheckSuccessIndicator(InData) then
			ClientShutdown;
		NumMailMessages := 0;
		if countwords(InData)<2 then
			begin
				writeln('Invalid response to STAT');
				ClientShutdown;
			end;
		val(copyword(InData, 2), NumMailMessages, iErr);
		if iErr <> 0 then
			begin
				writeln('Invalid response to STAT');
				ClientShutdown;
			end;
	end; (* FindNumberOfMessages *)

	//PURPOSE:
	//    Retrieves the headers of the waiting messages and applies the rules
	// to determine whether to keep or delete each message.
	procedure ScanMessages;
	const
		MAX_WAIT_FOR_DATA = 25000;
		PERIOD = '.';
	var
		iMsgId : positive;
		strMsgId : string[16];
		bLineTruncated : boolean;

		//PURPOSE:
		//    Reads the headers of the current message and then
		// applies the filtering rules to decide whether the message should
		// be kept or deleted.
		//RETURNS:
		//    TRUE - If the message should be kept
		// or
		//    FALSE - If the message should be delted
		function KeepMessage : boolean;
		var
			line : LineType;
			MsgStatus : (Checking, DeleteIt, KeepIt);
			CurrFieldKind : FieldKind;
			header : FieldType;
			headers : list of FieldType;

			//PURPOSE:
			//    Extracts the next header from the current message.
			// applies the filtering rules to decide whether the message should
			// be kept or deleted.
			//PARAMETER(s):
			//    1. hdr - Stores the next header extracted from "InData" or an
			//           empty string if there are no more headers to extract.
			//GLOBAL(S):
			//    1. InData - Contains zero or more headers waiting to be extracted.
			//NOTES:
			//   This procedure removes the first header from "InData" and stores
			// it in the parametger "hdr".
			procedure GetHeader(var hdr : LineType);
			var
				iPos, iPosCR, iPosLF : positive;
			begin (* GetHeader *)
				iPosCR := pos(CR, InData);
				iPosLF := pos(LF, InData);
				if iPosCR = 0 then
					iPosCR := iPosLF;
				if iPosLF = 0 then
					iPosLF := iPosCR;
				if iPosCR = iPosLF then
					iPos := iPosCR
				else if iPosCR < iPosLF then
					iPos := iPosCR
				else
					iPos := iPosLF;

				if iPos=0 then
					begin
						hdr := InData;
						InData := '';
						bLineTruncated := true
					end
				else
					begin
						hdr := copy(InData, 1, iPos-1);
						InData := copy(InData, iPos+1);
						if (InData<>'') and ((InData[1]=LF) or (InData[1]=CR))then
							InData := copy(InData, 2);
						//If the POP3 server does not send a period after the headers and
						// a blank line is at the end of the data read in
						// then assume that this is the last header and insert a period
						if not bLineTruncated and bNoPeriod and (hdr='') and (InData='') then
							hdr := PERIOD;
						bLineTruncated := false;
					end;
				//hdr := trim(hdr);
				if (length(hdr)>1) and (hdr[1]=PERIOD) and (hdr[2]=PERIOD) then
					hdr := copy(hdr, 2);
			end; (* GetHeader *)

			//PURPOSE: Appends the current header to the list of headers
			//NOTES:
			//  The first character in the header is checked to see if it
			// is a white space character. If the first char is not a white space
			// char then the header is parsed to deterine what kind of header it is.
			// If the first char is a white space char then this header is actually
			// a continuation of the previous header. In this case the application
			// does not try to join the continuation header to the previous one,
			// instead this header is appended to the list as a seperate header
			// of the same kind as the previous header. This is done because:
			// 1. The joined header may be too long (perhaps that is why it was split up
			//     in the first place).
			// 2. It is easier this way and this is just a sample program.
			// 3. It probably won't make a different (unless the header was split
			//    in the middle of text that would match one of the filtering rules).
			procedure AppendHeader(var hdr : LineType);
			begin (* AppendHeader *)
				if (hdr<>'') and not isspace(hdr[1]) then
					begin
						ParseField(header, hdr);
						CurrFieldKind := header.kind;
					end
				else
					begin
						header.kind := CurrFieldKind;
						header.body := trim(hdr);
					end;
				if header.kind <> FieldError then
					insert(header, headers);
			end; (* AppendHeader *)

			//PURPOSE: Searches the rules for the first one that matches
			//         one of the headers.
			//RETURNS: The kind of rule found.
			//NOTES: This particular rule found is not important, what is
			// important is the kind of rule found (if any).
			function SearchRules : RuleKind;
			var
				bEnd : boolean;
				kind : RuleKind;
				CurrRule : RuleType;
				ToListRule : RuleType;
				iRule, iHeader : positive;
				iToList : positive;

				//PURPOSE: Searches the headers for one that matches a
				//  particular rule.
				//PARAMETER(s):
				//    1. r - The rule to match headers against.
				//RETURNS:
				//    TRUE - If a matching header is found
				// or
				//    FALSE - If a matching header is not found.
				function SearchHeaders(var r : RuleType) : boolean;
				var
					bEnd : boolean;
					CurrHeader : FieldType;
					//iHeader : positive;
				begin (* SeachHeaders *)
					iHeader := 0;
					bEnd := false;
					repeat
						iHeader := iHeader + 1;
						if iHeader > length(headers) then
							bEnd := true
						else
							begin
								CurrHeader := headers[iHeader];
								if r.field.kind = CurrHeader.kind then
									if pos(r.field.body, lowercase(CurrHeader.body)) <> 0 then
										bEnd := true
							end;
					until bEnd;
					if iHeader > length(headers) then
						iHeader := 0;
					SearchHeaders := (iHeader<>0);
				end; (* SeachHeaders *)

			begin (* SearchRules *)
				kind := RuleKeep;
				bEnd := false;
				iRule := 0;
				repeat
					iRule := iRule + 1;
					if iRule > length(rules) then
						bEnd := true
					else
						begin
							CurrRule := Rules[iRule];
							if CurrRule.ToList <> '' then
								begin
									kind := RuleDelete;
									for iToList := 1 to countwords(CurrRule.ToList, ',') do
										begin
											ToListRule.kind := RuleKeep;
											ToListRule.field.kind := FieldTo;
											ToListRule.field.body := trim(copyword(CurrRule.ToList, iToList, ','));
											if SearchHeaders(ToListRule) then
												kind := RuleKeep;
										end;
									if kind = RuleDelete then
										bEnd := true
								end
							else if SearchHeaders(CurrRule) then
								begin
									kind := CurrRule.kind;
									bEnd := true;
								end;
						end;
				until bEnd;
				SearchRules := kind;
			end; (* SearchRules *)

		begin (* KeepMessage *)
			//Discard first line which is the response to the TOP command.
			GetHeader(line);
			MsgStatus := Checking;
			new(headers);
			CurrFieldKind := FieldError;

			repeat
				if InData='' then
					ReadDataSocket(InData, MAX_WAIT_FOR_DATA);
				repeat
					GetHeader(line);
					writeln(line);
					LogMessage(line);
					if (line<>'') and (line <> PERIOD) then
						AppendHeader(line)
				until (line='') or (line=PERIOD);
			until (line=PERIOD) or (not SocketReadyForReading(DataSocket, MAX_WAIT_FOR_DATA));

			//Determine whether we need to delete or keep the message
			if SearchRules = RuleDelete then
				MsgStatus := DeleteIt
			else
				MsgStatus := KeepIt;

			//Release the list of headers.
			dispose(headers);

			KeepMessage := MsgStatus <> DeleteIt;
		end; (* KeepMessage *)

	begin (* ScanMessages *)
		bLineTruncated := false;

		//Ensure that the application doesn't find itself trying to scan
		//a huge number of messages.
		if NumMailMessages > MaxMessages then
			NumMailMessages := MaxMessages;

		//For each message send the TOP command to retrieve the headers
		// and if successful call "KeepMessage" to read the headers and
		// determine whether to keep the message. If the decision is not
		// to keep the message then send the DELE command to delete it.
		for iMsgId := 1 to NumMailMessages do
			begin
				str(iMsgId:1, strMsgId);
				if bNoPeriod then
					OutData := 'TOP ' + strMsgId + ' 1'
				else
					OutData := 'TOP ' + strMsgId + ' 0';
				WriteDataSocket(OutData);
				ReadLineDataSocket(InData);
				if not CheckSuccessIndicator(InData) then
					begin
						LogMessage(InData);
						ClientShutdown
					end
				else
					if not KeepMessage then
						begin
							OutData := 'DELE ' + strMsgId;
							WriteDataSocket(OutData);
							ReadLineDataSocket(InData);
						end;
			end;
	end; (* ScanMessages *)

	//PURPOSE: Shuts down the application.
	procedure ClientShutdown;
	var
		iRet : integer;
	begin (* ClientShutdown *)
		//If the DataSocket is open then send the QUIT command if required
		// and them close it.
		if DataSocket <> INVALID_SOCKET then
			begin
				if QuitRequired then
					begin
						OutData := 'QUIT';
						WriteDataSocket(OutData);
						ReadLineDataSocket(InData);
					end;
				CloseClientSocket(DataSocket);
			end;

		//Clean up the Windows Socket API environment
		iRet := WSACleanUp;

		//Release the list of rules
		dispose(Rules);

		//Exit the application
		halt
	end; (* ClientShutdown *)

begin
	Initialize;

	ReadCommandLineOptions;

	ReadConfigInfo;

	if MailServerURL='' then
		begin
			write('Please enter mail server URL: ');
			readln(MailServerURL);
			MailServerURL := trim(MailServerURL);
		end;
	if MailServerURL='' then
		begin
			writeln('You must enter a mail server');
			ClientShutdown;
		end;

	FindMailServerAddress;

	DisplayMailServerInfo;

	DataSocket := CreateDataSocket;
	ConnectDataSocket(DataSocket, POP3_PORT, MailServerAddress);

	ReadLineDataSocket(InData);

	LogOn;

	FindNumberOfMessages;

	ScanMessages;

	ClientShutdown;
end.
