(********************************************************************************************
** PROGRAM     : WinServS
** VERSION     : 1.0.0
** DESCRIPTION : Simple Windows Web Server
** AUTHOR      : Stuart King
** COPYRIGHT   : Copyright (c) Irie Tools, 2002. All Rights Reserved.
** NOTES       :
**    This sample program is distributed with Irie Pascal, and was written to provide
** an example of how to write Windows Sockets API programs with Irie Pascal. It is also
** an example of how to write a Windows NT/2000 service with Irie Pascal. To make best
** use of this sample you should have a basic understanding of Pascal as well as a basic
** understanding of the Sockets API, and Windows NT/2000 services. An effort was made,
** while writing this program, to stick to the general Sockets API as much as possible,
** and to avoid the extensions added in the Windows Sockets API. However in three cases
** this was not possible. The Windows Sockets API, requires you to call function
** "WSAStartUp" before calling any other function, and to call the function "WSACleanUp"
** when you are finished. In addition the Windows Sockets API provies the function
** "WSAGetLastError" which returns an error code whenever an error occurs.
**
**    This is just a sample program, so do not use this program as a real webserver, or
** for anu purpose where security, reliability, or performance is important.
**
**    The file "winsock2.inc" is a system include file, that is distributed with
** Irie Pascal and contains declarations for the functions, procedures, types, and
** constants defined in the Windows Sockets API. In almost all cases, the names of
** entities declared in "winsock2.inc" are identical to the names of the
** corresponding entity in the Sockets API. However there are a few unavoidable
** cases where different names are used. The Sockets API was created for the C
** programming language, and defines the type "SOCKET" and the function "socket".
** Although "SOCKET" and "socket" differ only in case, this is not a problem in C
** which is case sensitive. However since Pascal is not case sensitive, "SOCKET"
** and "socket" are the same identifier and can not refer to different entities. So
** "winsock2.inc" uses "SOCKET" for the type and "createsocket" for the function.
**
**   Notice that "winsock2.inc" and "winuser.inc" are included before the keyword
** "program" which normally marks the start of a program. Starting in version 2.5,
** Irie Pascal allows declarations to occur before the keyword "program". When this is done,
** the declarations are placed in the same invisible outermost scope, as the declarations
** for the built-in identifiers (e.g. writeln). As you probably know, if you declare an
** identifier your declaration will override any other declarations of that identifier in
** outer scopes. This means that the program below can use any declarations in
** "winsock2.inc" and "winuser.inc" it wants to, and pretend that the other declarations
** don't exist. For this reason, it is recommended that you also include system include
** files before the keyword "program".
**
**    As stated above this program is an example of how to create a Windows NT/2000
** service application. Before explaining how to write your program so that it will
** be a well behaved service application, here's what you need to know to make Irie Pascal
** generate a service application. You need to do two things, first you need to tell
** Irie Pascal to generate a .exe executable, and second you need to tell Irie Pascal that
** this .exe should be a service. Starting in version 2.5 all you need to do to create
** a .exe executable is generate an executable with a name that ends with .exe. Prior
** to version 2.5 you could create an executable with a name ending with .exe but internally
** the executable would still be a .ivm executable, so Windows would not execute it.
**
**    If you are compiling from the IDE then you would select 'Options...' from the
** 'Project' menu and then select the 'Miscellaneous' tab. Then make sure than the name of
** the executable ends with '.exe'. Then select the 'Code Generation' tab and make sure that
** the 'Generate service application' option is selected. If you are using the command-line
** version of the compiler then you need to use the -oName option to specify an executable
** name that ends with .exe, and you need to use the -gs option to tell the compiler to
** generate a service. For example to compile this program as a service using the
** command-line version of the compiler you would enter:
**
**    -ipc -owinservs.exe -gs winservs.pas
**
**     Now in order to write your program so that it will be a well behaved service
** application, you need to make sure your program regularly checks for the stop service
** event. The stop service event gets sent to your service application when a user or
** Windows wants to stop your service. Irie Pascal provides two built-in functions that
** can be useful in check for the stop service event. The first built-in function is called
** 'StopServiceEvent', and either returns the handle to the stop service event object that
** was passed to your service application, or it returns zero if your application is not
** running as a service. The second built-in function is called 'wait' and will wait for
** a specified period of time for one or more events or other syncronization objects to
** become signalled (in the case of an event object the event object becomes signalled when
** the event occurs). For further details please consult the Irie Pascal Programmer's Reference
** Manual for information on the 'wait' function. For now all you have to know is that 'wait'
** will return the handle to the first synchronization object that became signalled before the
** maximum wait interval expired or zero if no synchronization object became signalled
** before the wait interval expired. If a wait interval of zero is specified then 'wait'
** doesn't actually wait but will check that status of the synchronization objects and
** return immediately. If a wait interval of maxword is pecified then 'wait' will
** wait forever for a synchronization object to become signalled.
**
**    This program demonstrates a recommended technique for writing services. The program
** always checks that 'StopServiceEvent' is non-zero before calling 'wait' so that
** if it is not running as a service it will not wait for the stop service event. As
** a result the program can run both as a normal console application and as service
** application. This can be especially useful during debugging because console applications
** are easier to debug than services. So when debugging you can compile the application
** as a normal console application, and when you are confident that the basic operation of
** the applicatin is sound then you can compile it as a service for final testing and
** deployment.
**
**     Finally, service applications generated by Irie Pascal include the ability to
** install, delete and comfigure themselves and other service applications. To see
** a help screen you need to go to a console window and enter:
**
**     name -help
**
** where 'name' is the name of your service application. To see more help on the -change
** option you need to enter:
**
**     name -change
**
** where 'name' is the name of your service application.
**
**    Further information is available in the configuration file 'winservs.cfg' and in
** the following RFC's (rfc2616, rfc2396, and rfc2145).
*********************************************************************************************)

(*$I winsock2.inc *)
(*$I winuser.inc *)
program WinServS;
const
	CR = #13; //Carriage return character
	LF = #10; //Line feed character

	CONFIG_FILENAME = 'winservs.cfg'; //The name of the configuration file

	//The configuration file can have three entries.
	//root - which specifies where the web server should start looking for files.
	//errors - which specifies the subdirectory of root that contains the files that
	//         are returned when various errors occur. You can modify the files in this
	//         subdirectory to customize the files returned when errors occur.
	//port = which specifies which port the web server should listen for requests on.
	DEFAULT_ROOT_DIR = 'winservs';
	DEFAULT_ERRORS_DIR = 'errors';
	DEFAULT_PORT = 80;

	MAX_REQUEST_BUFFER = 4096;
	MAX_REQUEST_LINE = 800;
	MAX_URI = MAX_REQUEST_LINE;
	MAX_REASON_PHRASE = 80;
	EOF_FLAG = '[EOF]';
	MAX_METHOD = 16;
	MAX_OUTPUT_LINE = 4096;

	MAX_ACCEPT_DATA = 255;
	MAX_ACCEPT_CHAR_SET_DATA = 255;
	MAX_ACCEPT_ENCODING_DATA = 255;
	MAX_ACCEPT_LANG_DATA = 255;
	MAX_ALLOW_DATA = 16;
	MAX_AUTHORIZATION_DATA = 255;
	MAX_CACHE_CONTROL_DATA = 255;
	MAX_CONTENT_ENCODING_DATA = 255;
	MAX_CONTENT_LANG_DATA = 255;
	MAX_CONTENT_LENGTH_DATA = 16;
	MAX_CONTENT_LOCATION_DATA = MAX_URI;
	MAX_CONTENT_MD5_DATA = 32;
	MAX_CONTENT_TYPE_DATA = 255;
	MAX_DATE_DATA = 64;
	MAX_FROM_DATA = 255;
	MAX_HOST_URI = MAX_URI;
	MAX_IF_MATCH_DATA = 255;
	MAX_IF_MODIFIED_SINCE_DATA = MAX_DATE_DATA;
	MAX_IF_NONE_MATCH_DATA = 255;
	MAX_IF_RANGE_DATA = 255;
	MAX_IF_UNMODIFIED_SINCE_DATA = MAX_DATE_DATA;
	MAX_LAST_MODIFIED_DATA = MAX_DATE_DATA;
	MAX_PRAGMA_DATA = 255;
	MAX_RANGE_DATA = 80;
	MAX_REFERRER_DATA = MAX_URI;
	MAX_TE_DATA = 255;
	MAX_TRAILER_DATA = 255;
	MAX_TRANSFER_ENCODING_DATA = 255;
	MAX_UPGRADE_DATA = 255;
	MAX_USER_AGENT_DATA = 255;

	//These constants specify the values of various response codes.
	RC_OK = 200;
	RC_BAD_REQUEST = 400;
	RC_NOT_FOUND = 404;
	RC_REQUEST_TIMEOUT = 408;
	RC_REQUEST_URI_TOO_LARGE = 414;
	RC_EXPECTATION_FAILED = 417;
	RC_INTERNAL_SERVER_ERROR = 500;
	RC_NOT_IMPLEMENTED = 501;
	RC_HTTP_VERSION_NOT_SUPPORTED = 505;
type
	PathType = filename;
	positive = 0..maxint;
	CharQueue = packed list of char;
	RequestLine = string[MAX_REQUEST_LINE];
	OutputLineType = string[MAX_OUTPUT_LINE];
	RequestMethodKinds = (
		MethodUndef, MethodOptions, MethodGet, MethodHead, MethodPost, MethodPut,
		MethodDelete, MethodTrace, MethodConnect, MethodExt
	);
	RequestMethod = record
	case kind : RequestMethodKinds of
		MethodUndef, MethodOptions, MethodGet, MethodHead, MethodPost, MethodPut,
		MethodDelete, MethodTrace, MethodConnect : ();
		MethodExt : (ext : string)
	end;
	Request_URI = string[MAX_URI];
	HTTP_Version = record
		major : positive;
		minor : positive;
	end;
	AcceptDataType = string[MAX_ACCEPT_DATA];
	AcceptCharSetDataType = string[MAX_ACCEPT_CHAR_SET_DATA];
	AcceptEncodingDataType = string[MAX_ACCEPT_ENCODING_DATA];
	AcceptLangDataType = string[MAX_ACCEPT_LANG_DATA];
	AllowDataType = string[MAX_ALLOW_DATA];
	AuthorizationDataType = string[MAX_AUTHORIZATION_DATA];
	CacheControlDataType = string[MAX_CACHE_CONTROL_DATA];
	ConnectionDataType = record
		KeepAlive : boolean;
	end;
	ContentEncodingDataType = string[MAX_CONTENT_ENCODING_DATA];
	ContentLangDataType = string[MAX_CONTENT_LANG_DATA];
	ContentLengthDataType = string[MAX_CONTENT_LENGTH_DATA];
	ContentLocationDataType = string[MAX_CONTENT_LOCATION_DATA];
	ContentMD5DataType = string[MAX_CONTENT_MD5_DATA];
	ContentRangeDataType = record
		FirstBytePos : word;
		LastBytePos : word;
		InstanceLength : word;
	end;
	ContentTypeDataType = string[MAX_CONTENT_TYPE_DATA];
	DateDataType = string[MAX_DATE_DATA];
	FromDataType = string[MAX_FROM_DATA];
	HostDataType = record
		InternetHost : string[MAX_HOST_URI];
		Port : word;
	end;
	IfMatchDataType = string[MAX_IF_MATCH_DATA];
	IfModifiedSinceDataType = string[MAX_IF_MODIFIED_SINCE_DATA];
	IfNoneMatchDataType = string[MAX_IF_NONE_MATCH_DATA];
	IfRangeDataType = string[MAX_IF_RANGE_DATA];
	IfUnmodifiedSinceDataType = string[MAX_IF_UNMODIFIED_SINCE_DATA];
	LastModifiedDataType = string[MAX_LAST_MODIFIED_DATA];
	MaxForwardsDataType = word;
	PragmaDataType = string[MAX_PRAGMA_DATA];
	RangeDataType = string[MAX_RANGE_DATA];
	ReferrerDataType = string[MAX_REFERRER_DATA];
	TEDataType = string[MAX_TE_DATA];
	TrailerDataType = string[MAX_TRAILER_DATA];
	TransferEncodingDataType = string[MAX_TRANSFER_ENCODING_DATA];
	UpgradeDataType = string[MAX_UPGRADE_DATA];
	UserAgentDataType = string[MAX_USER_AGENT_DATA];
	HeaderKinds = (
		HeaderAccept, HeaderAcceptCharSet, HeaderAcceptEncoding, HeaderAcceptLang,
		HeaderAllow, HeaderAuthorization, HeaderCacheControl,
		HeaderContentEncoding, HeaderContentLang, HeaderContentLength,
		HeaderContentLocation, HeaderContentMD5, HeaderContentRange, HeaderContentType,
		HeaderDate, HeaderFrom, HeaderIfMatch, HeaderIfModifiedSince,
		HeaderIfNoneMatch, HeaderIfRange, HeaderIfUnmodifiedSince, HeaderLastModified,
		HeaderMaxForwards, HeaderPragma, HeaderRange, HeaderReferrer, HeaderTE,
		HeaderTrailer, HeaderTransferEncoding, HeaderUpgrade,
		HeaderUserAgent
	);
	Header = record
	case kind : HeaderKinds of
		HeaderAccept : (AcceptData : AcceptDataType);
		HeaderAcceptCharSet : (AcceptCharSetData : AcceptCharSetDataType);
		HeaderAcceptEncoding : (AcceptEncodingData : AcceptEncodingDataType);
		HeaderAcceptLang : (AcceptLangData : AcceptLangDataType);
		HeaderAllow : (AllowData : AllowDataType);
		HeaderAuthorization : (AuthorizationData : AuthorizationDataType);
		HeaderCacheControl : (CacheControlData : CacheControlDataType);
		HeaderContentEncoding : (ContentEncodingData : ContentEncodingDataType);
		HeaderContentLang : (ContentLangData : ContentLangDataType);
		HeaderContentLength : (ContentLengthData : ContentLengthDataType);
		HeaderContentLocation : (ContentLocationData : ContentLocationDataType);
		HeaderContentMD5 : (ContentMD5Data : ContentMD5DataType);
		HeaderContentRange : (ContentRangeData : ContentRangeDataType);
		HeaderContentType : (ContentTypeData : ContentTypeDataType);
		HeaderDate : (DateData : DateDataType);
		HeaderFrom : (FromData : FromDataType);
		HeaderIfMatch : (IfMatchData : IfMatchDataType);
		HeaderIfModifiedSince : (IfModifiedSinceData : IfModifiedSinceDataType);
		HeaderIfNoneMatch : (IfNoneMatchData : IfNoneMatchDataType);
		HeaderIfRange : (IfRangeData : IfRangeDataType);
		HeaderIfUnmodifiedSince : (IfUnmodifiedSinceData : IfUnmodifiedSinceDataType);
		HeaderLastModified : (LastModifiedData : LastModifiedDataType);
		HeaderMaxForwards : (MaxForwardsData : MaxForwardsDataType);
		HeaderPragma : (PragmaData : PragmaDataType);
		HeaderRange : (RangeData : RangeDataType);
		HeaderReferrer : (ReferrerData : ReferrerDataType);
		HeaderTE : (TEData : TEDataType);
		HeaderTrailer : (TrailerData : TrailerDataType);
		HeaderTransferEncoding : (TransferEncodingData : TransferEncodingDataType);
		HeaderUpgrade : (UpgradeData : UpgradeDataType);
		HeaderUserAgent : (UserAgentData : UserAgentDataType);
	end;
	//
	//This type describes the information collected about a HTTP request.
	RequestInfo = record
		method : RequestMethod;
		uri : Request_URI;
		ver : HTTP_Version;
		headers : list of header;
		ConnectionData : ConnectionDataType;
		HostData : HostDataType;
		resource : filename;
	end;
	StatusCodeClass = (Processing, Informational, Successful, Redirection, ClientError, ServerError);
	//
	//This type describes the information collected about a HTTP response.
	ResponseInfo = record
		StatusCode : integer;
		ReasonPhrase : string[MAX_REASON_PHRASE];
		ContentTypeData : ContentTypeDataType;
		ContentLengthData : ContentLengthDataType;
	end;
	ProgramStateType = (Starting, Running, Stopping);
var
	ListeningSocket : SOCKET; //This sockets listens for new connections.
	ConnectedSocket : SOCKET; //This socket transfers data (i.e. requests and responses).
	RootDir, ErrorsDir : PathType;
	ServerPort : integer;
	RequestQueue : CharQueue;
	ProgramState : ProgramStateType;
	AppDir : filename;

	procedure LogMessage(msg : string);
	var
		fLog : text;
	begin (* LogMessage *)
		traperrors(false);
		append(fLog, '\winservs.log');
		writeln(fLog, msg);
		close(fLog);
		traperrors(true);
	end; (* LogMessage *)

	//This procedure performs the necessary clean-up and shuts down the web server.
	//Basically it checks if the sockets are open and closes them it they are, and
	//then cleans up as required by the Windows Socket API, and finally halts.
	procedure ServerShutDown;
	var
		iRet : integer;
	begin (* ServerShutDown *)
		LogMessage('Shutting down');
		if ConnectedSocket <> INVALID_SOCKET then
			iRet := CloseSocket(ConnectedSocket);
		ConnectedSocket := INVALID_SOCKET;
		if ListeningSocket <> INVALID_SOCKET then
			iRet := CloseSocket(ListeningSocket);
		ListeningSocket := INVALID_SOCKET;
		iRet := WSACleanUp;
		if (ProgramState<>Stopping) and (StopServiceEvent<>0) then
			begin
				LogMessage('Waiting for Stop Event ');
				repeat
				until wait(maxword, StopServiceEvent)=StopServiceEvent;
				ProgramState := Stopping;
			end;
		LogMessage('Stopping');
		halt;
	end; (* ServerShutDown *)

	procedure CheckForStopServiceEvent;
	const
		WAIT_INTERVAL=0;
	begin
		if (ProgramState<>Stopping) and (StopServiceEvent<>0) then
			begin
				//LogMessage('Calling wait');
				if wait(WAIT_INTERVAL, StopServiceEvent)=StopServiceEvent then
					begin
						ProgramState := Stopping;
						ServerShutDown;
					end;
			end
	end;

	//This function checks whether a files exists. Actually it really checks
	//to see if a file can be opened for reading without causing an error,
	//which is usually the same thing. The function first turns off run-time error
	//trapping so that if an error occurs it can be retrieved with "getlasterror"
	//rather than causing the program to halt. Then the function attempts to
	//open the file for reading. If no errors occur then the file is closed and
	//the file must exist. If an error occurs then the function assumes the file
	//does not exist. In either case error trapping it turned back on before the
	//function exits.
	function FileExists(name : filename) : boolean;
	var
		f : text;
	begin (* FileExists *)
		traperrors(false);
		reset(f, name);
		if getlasterror=0 then
			begin
				close(f);
				FileExists := true;
			end
		else
			FileExists := false;
		traperrors(true)
	end; (* FileExists *)

	//The functions (CharQueueX) are worth a little explanation.
	//The HTTP/1.1 specification says that multiple requests can
	//occur over the same socket. So the web server has to deal with
	//the possibility that when it is reading a request from a socket it
	//might also read part of or all of another request. Remember that requests
	//have variable length and the web server can not know without parsing
	//each request when it reaches the end. The solution choosen to solve
	//this problem is to buffer each request in a queue, so that if the
	//web server reads part of the next request while processing the current
	//request, the next request is still stored in the queue and will be
	//read when the web server gets around to processing the next request.
	//
	//Queues are created when the ConnectedSocket is created and destroyed
	//when the ConnectedSocket is destroyed.

	//
	//This procedure creates a new queue
	procedure CharQueueCreate(var q : CharQueue);
	begin (* CharQueueCreate *)
		new(q);
	end; (* CharQueueCreate *)

	//
	//This function returns the length of a queue (or in other words, how many
	//characters are in the queue).
	function CharQueueLength(var q : CharQueue) : positive;
	begin (* CharQueueLength *)
		CharQueueLength := length(q);
	end; (* CharQueueLength *)

	//
	//This procedure add a character to the end of a queue
	procedure CharQueueAdd(var q : CharQueue; c : char);
	begin (* CharQueueAdd *)
		insert(c, q);
	end; (* CharQueueAdd *)

	//
	//This function return the character at the front of a queue,
	//without removing it from the queue.
	function CharQueuePeek(var q : CharQueue) : string;
	begin (* CharQueuePeek *)
		if CharQueueLength(q) > 0 then
			CharQueuePeek := q[1]
		else
			CharQueuePeek := '';
	end; (* CharQueuePeek *)

	//
	//This function returns the character at the front of a queue,
	//and removes it from the queue.
	function CharQueueRead(var q : CharQueue) : string;
	var
		s : string;
	begin (* CharQueueRead *)
		s := CharQueuePeek(q);
		if s <> '' then
			delete(q, 1, length(s));
		CharQueueRead := s;
	end; (* CharQueueRead *)

	//
	//This procedure destroys a queue.
	procedure CharQueueDestroy(var q : CharQueue);
	begin (* CharQueueDestroy *)
		dispose(q);
	end; (* CharQueueDestroy *)

	//
	//This procedure reports errors that occur when using sockets. This program does
	// not attempt to recover from socket errors so these errors are fatal. This
	// procedure will shut the server down, after reporting the error.
	//NOTE: The function "MessageBox" is called (the declaration is in "winuser.inc")
	// to display a Windows message box on the screen (if this program is not running
	// as a service).
	procedure FatalSocketError(msg : string);
	var
		iLastWSAError : integer;
		iRet : integer;
		strError : string[16];
		caption, txt : cstring;
	begin (* FatalSocketError *)
		iLastWSAError := WSAGetLastError;
		str(iLastWSAError:1, strError);
		caption := 'WinServS';
		txt := 'ERROR:' + strError + ' ' + msg;
		writeln(txt);
		LogMessage(txt);
		if StopServiceEvent = 0 then
			iRet := MessageBox(NULL, addr(txt), addr(caption), MB_OK);
		ServerShutDown;
	end; (* FatalSocketError *)

	//
	//This procedure reads the server configuration file, if it is present, and
	//changes the values of any of the default parameters specified.
	procedure ProcessConfigInfo;
	const
		MAX_CONFIG_LINE=800;
	type
		ConfigLine = string[MAX_CONFIG_LINE];
	var
		fConfig : text;
		line, key, value : ConfigLine;
		iPos, iErr : integer;
		blnConfigOpen : boolean;

		//This function is responsible for parsing string values.
		//In the configuration file string values must be enclosed
		//in single or double quotes, so what this procedure basically does
		//is extract the enclosed string. If the string is not properly enclosed
		//then the value returned is the empty string.
		procedure ParseStringValue(var v : ConfigLine);
		var
			iPos : integer;
		begin (* ParseStringValue *)
			if (v<>'') and ((v[1]='''')or(v[1]='"')) then
				begin
					iPos := pos(v[1], v, 2);
					if iPos>0 then
						v := copy(v, 2, iPos-2)
					else
						v := '';
				end
			else
				v := '';
		end; (* ParseStringValue *)

	begin (* ProcessConfigInfo *)
		blnConfigOpen := false;
		traperrors(false);
		reset(fConfig, AppDir+CONFIG_FILENAME);
		traperrors(true);
		blnConfigOpen := getlasterror=0;
		while blnConfigOpen and (not eof(fConfig)) do
			begin
				readln(fConfig, line);
				line:=lowercase(trim(line));
				if (line='') or ((line[1]=';')or(line[1]='''')or(line[1]='*')) then
					//This line is a comment so do nothing just ignore it
				else
					begin
						iPos := pos('=', line);
						if iPos>0 then
							begin
								key := lowercase(trim(copy(line, 1, iPos-1)));
								value := trim(copy(line, iPos+1));
								if key='root' then
									begin
										ParseStringValue(value);
										RootDir:=value;
										if RootDir='' then
											RootDir := DEFAULT_ROOT_DIR;
										if (RootDir<>'') and (RootDir[length(RootDir)]='\') then
											RootDir := copy(RootDir, 1, length(RootDir)-1);
										if (RootDir<>'') and (RootDir[length(RootDir)]='/') then
											RootDir := copy(RootDir, 1, length(RootDir)-1);
									end
								else if key='errors' then
									begin
										ParseStringValue(value);
										ErrorsDir := value;
										if ErrorsDir='' then
											ErrorsDir := DEFAULT_ERRORS_DIR;
										if (ErrorsDir<>'') and (ErrorsDir[length(ErrorsDir)]='\') then
											ErrorsDir := copy(ErrorsDir, 1, length(ErrorsDir)-1);
										if (ErrorsDir<>'') and (ErrorsDir[length(ErrorsDir)]='/') then
											ErrorsDir := copy(ErrorsDir, 1, length(ErrorsDir)-1);
										if (ErrorsDir<>'') and (ErrorsDir[1]='\') then
											ErrorsDir := copy(ErrorsDir, 2);
										if (ErrorsDir<>'') and (ErrorsDir[1]='/') then
											ErrorsDir := copy(ErrorsDir, 2);
									end
								else if key='port' then
									val(value, ServerPort, iErr);
								if iErr<>0 then
									ServerPort := DEFAULT_PORT
							end
					end
			end;
		if blnConfigOpen then
		close(fConfig);
	end; (* ProcessConfigInfo *)

	//This function creates a listening socket.
	//First it creates a socket. Then it binds the socket to
	//a particular port (i.e. the port we want the listen on).
	//The socket is not bound to any particular address (INADDR_ANY is used instead)
	//because connections will be accepted from any ip address.
	//Finally "listen" is called to convert the socket to a listing socket.
	function CreateListeningSocket(port : integer) : socket;
	var
		s : socket;
		sa : sockaddr_in;
		iRet : integer;
	begin (* CreateListeningSocket *)
		s := createsocket(AF_INET, SOCK_STREAM, 0);
		if s = INVALID_SOCKET then
			FatalSocketError('CreateSocket call failed');
		fill(sa, 0);
		sa.sin_family := AF_INET;
		sa.sin_port := htonl(port) shr 16;
		sa.sin_addr := htonl(INADDR_ANY) shr 16;
		iRet := bind(s, addr(sa), sizeof(sa));
		if iRet <> 0 then
			FatalSocketError('Call to bind failed');
		iRet := listen(s, SOMAXCONN);
		if iRet <> 0 then
			FatalSocketError('Call to listen failed');
		CreateListeningSocket := s;
	end; (* CreateListeningSocket *)

	//This procedure initializes the web server.
	//First initializing the sockets. Then intializes the server parameters
	//(root, errors, and port) to their default values. Then it reads the configuration
	//file to update the server parameters with any specified changes. Then it intializes
	//the Windows Socket API. And if all goes well it creates the listening socket.
	procedure Initialize;
	var
		data : WSADATA;
		iRet : integer;
		iLen : integer;
	begin (* Initialize *)
		ProgramState := Starting;
		fsplit(paramstr(0), AppDir,,);
		iLen := length(AppDir);
		if (iLen<>0) and (AppDir[iLen]<>'/') and (AppDir[iLen]<>'\') then
			AppDir := AppDir + '\';
		LogMessage('');
		LogMessage('Starting');
		ListeningSocket := INVALID_SOCKET;
		ConnectedSocket := INVALID_SOCKET;
		RootDir := DEFAULT_ROOT_DIR;
		ErrorsDir := DEFAULT_ERRORS_DIR;
		ServerPort := DEFAULT_PORT;
		ProcessConfigInfo;
		iRet := WSAStartUp($0202, data);
		if iRet <> 0 then
			begin
				writeln('WSAStartUp call failed. Return Code=', iRet);
				ServerShutDown;
			end;
		ListeningSocket := CreateListeningSocket(ServerPort);
		ProgramState := Running;
	end; (* Initialize *)

	//This procedure closes a socket. First it shuts down both reading and writing
	//(that's what SD_BOTH means), and then it actually closes the socket.
	//NOTE: Calling "shutdown" before closing the socket is probably not
	//  necessary but it does no harm.
	procedure CloseServerSocket(var s : socket);
	var
		i : integer;
	begin (* CloseServerSocket *)
		if s <> INVALID_SOCKET then
			begin
				writeln('Closing socket');
				i := ShutDown(s, SD_BOTH);
				i := CloseSocket(s);
				s := INVALID_SOCKET;
			end
	end; (* CloseServerSocket *)

	//This function waits for a socket to become ready for reading or for a socket in
	//the listening state to become ready to accept a connection without blocking.
	function SocketReadyForReading(s : SOCKET; WaitSecs, WaitMilliSecs : integer) : boolean;
	var
		fdRead : fd_set;
		iRet : integer;
		strRet : string[16];
		t : timeval;
	begin (* SocketReadyForReading *)
		fdRead.fd_count := 1;
		fdRead.fd_array[0] := s;
		t.tv_sec := WaitSecs;
		t.tv_usec := WaitMilliSecs;
		iRet := select(1, fdRead, null, null, t);
		str(iRet:1, strRet);
		//LogMessage('select returned ' + strRet);
		SocketReadyForReading := (iRet=1);
	end; (* SocketReadyForReading *)

	//This procedure waits for a new connection by calling "accept" on
	//the listening socket. The call to "accept" will block until a new
	//connection comes in. If all goes well the queue for request messages
	//is created.
	procedure WaitForConnection;
	var
		blnReady : boolean;

	begin (* WaitForConnection *)
		if ConnectedSocket = INVALID_SOCKET then
			begin
				blnReady := false;
				LogMessage('Waiting for connection');
				repeat
					CheckForStopServiceEvent;
					blnReady := SocketReadyForReading(ListeningSocket, 1, 0);
				until blnReady;
				LogMessage('Accepting connection');
				ConnectedSocket := accept(ListeningSocket, null, null);
			end;
		if ConnectedSocket = INVALID_SOCKET then
			FatalSocketError('Call to accept failed')
		else
			CharQueueCreate(RequestQueue);
	end; (* WaitForConnection *)

	//This procedure reads data (if any is available) into a queue.
	//The most interesting part of this procedure is the function "DataWaiting"
	//which is called before attempting to read data from the socket, because
	//we are using blocking sockets (the default kind), so if we called "recv"
	//to read data and none is waiting, the web server would block waiting
	//for data to come in. That would make the server very vulnerable to
	//denial of service attacks. All that an attacker would have to do is
	//open a connectin to the server and not send any data. Also if the server
	//is running a service it can't allow itself to block because then it wouldn't
	//be able to respond to the StopServiceEvent.
	procedure ReadIntoQueue(var q : CharQueue);
	var
		DataRead : integer;
		p : positive;
		buffer : packed array[1..MAX_REQUEST_BUFFER] of char;

		//This function determines if any data is waiting in the "ConnectedSocket".
		function DataWaiting : boolean;
		begin (* DataWaiting *)
			if ConnectedSocket = INVALID_SOCKET then
				DataWaiting := false
			else
				begin
					CheckForStopServiceEvent;
					DataWaiting := SocketReadyForReading(ConnectedSocket, 0, 0);
				end
		end; (* DataWaiting *)

	begin (* ReadIntoQueue *)
		if DataWaiting then
			begin
				LogMessage('Receiving');
				DataRead := recv(ConnectedSocket, addr(buffer), MAX_REQUEST_BUFFER, 0);
				if DataRead = SOCKET_ERROR then
					CloseServerSocket(ConnectedSocket)
				else if DataRead > 0 then
					for p := 1 to DataRead do
						CharQueueAdd(q, buffer[p]);
			end;
	end; (* ReadIntoQueue *)

	//This function is responsible for returning the next line in the request message.
	//If there is no data in the request queue then it calls "ReadIntoQueue" until
	//there is data or until it times out.
	function GetRequestLine : RequestLine;
	const
		MAX_REQUEST_WAIT_TIME = 10;
	var
		CurrRequestWaitTime : positive;
		line :RequestLine;
		done, EOL : boolean;
		s : string;
	begin (* GetRequestLine *)
		CurrRequestWaitTime := 0;
		line := '';

		//Read any data waiting in the socket into the request queue.
		ReadIntoQueue(RequestQueue);
		CurrRequestWaitTime := CurrRequestWaitTime + 1;
		while (CharQueueLength(RequestQueue)=0) and (CurrRequestWaitTime<=MAX_REQUEST_WAIT_TIME) do
			begin
				delay(100);
				CurrRequestWaitTime := CurrRequestWaitTime + 1;
				ReadIntoQueue(RequestQueue);
			end;

		//Now try and extract a line from the request queue.
		done := false;
		while not done do
			begin
				EOL := false;
				s := CharQueuePeek(RequestQueue);
				if s = CR then //This is the end of the line.
					begin
						EOL := true;
						s := CharQueueRead(RequestQueue); //Skip the CR
						//Check for a trailing LF and skip it if found.
						s := CharQueuePeek(RequestQueue);
						if s=LF then
							s := CharQueueRead(RequestQueue);
					end
				else if s=LF then //This is the end of the line.
					begin
						EOL := true;
						s := CharQueueRead(RequestQueue); // Skip the LF
					end
				else if s='' then //There is not more data in the queue
					begin
						done := true;
						line := EOF_FLAG;
					end
				else //this is just a normal character to add it to the line to be returned.
					line := line + CharQueueRead(RequestQueue);
				if EOL then
					begin
						//Since HTTP allows request lines to fold onto multiple lines
						//as long as continuation lines start with a space or horizontal tab
						//then although we are at the end of a line we need to make sure
						//that the beginning of the next line (if there is a next line)
						//does not begin with a space or HT. If it hasn't then
						//we are done if not we need to continue reading the next line.
						s := CharQueuePeek(RequestQueue);
						if (length(s)=0) or (not isspace(s[1])) then
							done := true;
					end
			end;
		GetRequestLine := line;
	end; (* GetRequestLine *)

	//This is the main procedure responsible for processing HTTP requests
	procedure ProcessHTTPRequests;
	const
		MAX_GET_ATTEMPTS = 20;
	var
		attempts : positive;
		line : RequestLine;
		done : boolean;
		ReqInfo : RequestInfo;
		ResInfo : ResponseInfo;

		//This procedure intializes the request information record
		procedure InitRequestInfo(var req : RequestInfo);
		begin (* InitRequestInfo *)
			req.method.kind := MethodUndef;
			req.uri := '';
			req.ver.major := 0;
			req.ver.minor := 0;
			new(req.headers);
			req.ConnectionData.KeepAlive := false;
			req.HostData.InternetHost := '';
			req.HostData.port := 80;
			req.resource := '';
		end; (* InitRequestInfo *)

		//This procedure intializes the response information record
		procedure InitResponseInfo(var res : ResponseInfo);
		begin (* InitResponseInfo *)
			res.StatusCode := 0;
			res.ReasonPhrase := '';
			res.ContentTypeData := '';
			res.ContentLengthData := '';
		end; (* InitResponseInfo *)

		//This procedure is responsible for making sense of the first line in
		//a request message. See RFC2616 for the syntax of request messages.
		procedure ParseRequestLine(var line : RequestLine; var req : RequestInfo; var res : ResponseInfo);
		var
			s : string;
			iPos : integer;
		begin (* ParseRequestLine *)
			//First identify the request method
			s := '';
			if length(CopyWord(line, 1)) <= MAX_METHOD then
				s := CopyWord(line, 1);
			if s = 'OPTIONS' then
				req.method.kind := MethodOptions
			else if s = 'GET' then
				req.method.kind := MethodGet
			else if s = 'HEAD' then
				req.method.kind := MethodHead
			else if s = 'POST' then
				req.method.kind := MethodPost
			else if s = 'PUT' then
				req.method.kind := MethodPut
			else if s = 'DELETE' then
				req.method.kind := MethodDelete
			else if s = 'TRACE' then
				req.method.kind := MethodTrace
			else if s = 'CONNECT' then
				req.method.kind := MethodConnect
			else
				begin
					req.method.kind := MethodExt;
					req.method.ext := s;
					res.StatusCode := RC_NOT_IMPLEMENTED;
					res.ReasonPhrase := 'Not Implemented';
					exit
				end;

			//Parse the request URI
			if length(CopyWord(line, 2)) <= MAX_URI then
				req.uri := CopyWord(line, 2)
			else
				begin
					res.StatusCode := RC_REQUEST_URI_TOO_LARGE;
					res.ReasonPhrase := 'Request-URI Too Large';
					exit
				end;

			//Parse the request version.
			if (length(CopyWord(line, 3))<=length('HTTP/')) or (length(CopyWord(line, 3))>255) then
				begin
					res.StatusCode := RC_BAD_REQUEST;
					res.ReasonPhrase := 'Bad Request';
					exit
				end;
			if copy(CopyWord(line, 3), 1 , length('HTTP/')) <> 'HTTP/' then
				begin
					res.StatusCode := RC_BAD_REQUEST;
					res.ReasonPhrase := 'Bad Request';
					exit
				end;
			s := copy(CopyWord(line, 3), length('HTTP/')+1);
			val(s, req.ver.major, iPos);
			if (iPos=0) or (s[iPos]<>'.') then
				begin
					res.StatusCode := RC_BAD_REQUEST;
					res.ReasonPhrase := 'Bad Request';
					exit
				end;
			s := copy(s, iPos+1);
			val(s, req.ver.minor, iPos);
			if req.ver.major <> 1 then
				begin
					res.StatusCode := RC_HTTP_VERSION_NOT_SUPPORTED;
					res.ReasonPhrase := 'HTTP Version Not Supported';
					exit
				end;
			if req.ver.minor <> 0 then
				req.ConnectionData.KeepAlive := true;
		end; (* ParseRequestLine *)

		//This function classifies HTTP response status codes into categories.
		//The first digit of status codes determine the category. See RFC2616.
		function ClassifyStatusCode(var res : ResponseInfo) : StatusCodeClass;
		var
			FirstDigit : '0'..'9';
			scc : StatusCodeClass;
		begin (* ClassifyStatusCode *)
			FirstDigit := '0';
			FirstDigit := chr((res.StatusCode div 100)+ord('0'));
			case FirstDigit of
			'1':
				scc := Informational;
			'2':
				scc := Successful;
			'3':
				scc := Redirection;
			'4':
				scc := ClientError;
			'5':
				scc := ServerError;
			otherwise
				scc := Processing;
			end;
			ClassifyStatusCode := scc;
		end; (* ClassifyStatusCode *)

		//This procedure parses request headers.
		//Most of the headers are in fact not used by the web server, they are just
		//stored and ignored.
		procedure ParseHeader(var line : RequestLine; var req : RequestInfo; var res : ResponseInfo);
		const
			MAX_HEADER_NAME=256;
		var
			iPos, iPortPos, iErr : integer;
			s : string;
			HeaderName : string[MAX_HEADER_NAME];
			h : Header;
		begin (* ParseHeader *)
			iPos := pos(':', line);
			if iPos=0 then
				begin
					res.StatusCode := RC_BAD_REQUEST;
					res.ReasonPhrase := 'Bad Request';
				end
			else if iPos>255 then
				begin
					res.StatusCode := RC_INTERNAL_SERVER_ERROR;
					res.ReasonPhrase := 'Internal Server Error';
				end
			else
				begin
					HeaderName := trim(lowercase(copy(line, 1, iPos-1)));
					if HeaderName='accept' then
						begin
							h.kind := HeaderAccept;
							h.AcceptData := trim(lowercase(copy(line, iPos+1)));
							insert(h, req.headers);
						end
					else if HeaderName = 'accept-charset' then
						begin
							h.kind := HeaderAcceptCharSet;
							h.AcceptCharSetData := trim(lowercase(copy(line, iPos+1)));
							insert(h, req.headers);
						end
					else if HeaderName = 'accept-encoding' then
						begin
							h.kind := HeaderAcceptEncoding;
							h.AcceptEncodingData := trim(lowercase(copy(line, iPos+1)));
							insert(h, req.headers);
						end
					else if HeaderName = 'accept-language' then
						begin
							h.kind := HeaderAcceptLang;
							h.AcceptLangData := trim(lowercase(copy(line, iPos+1)));
							insert(h, req.headers);
						end
					else if HeaderName = 'allow' then
						begin
							h.kind := HeaderAllow;
							h.AllowData := trim(lowercase(copy(line, iPos+1)));
							insert(h, req.headers);
						end
					else if HeaderName = 'authorization' then
						begin
							h.kind := HeaderAuthorization;
							h.AuthorizationData := trim(lowercase(copy(line, iPos+1)));
							insert(h, req.headers);
						end
					else if HeaderName = 'cache-control' then
						begin
							h.kind := HeaderCacheControl;
							h.CacheControlData := trim(lowercase(copy(line, iPos+1)));
							insert(h, req.headers);
						end
					else if HeaderName = 'connection' then
						begin
							//The connection header is used to help determine
							//whether to close the socket after processing the request.
							//The client may ask that the connection remain open by sending
							//Connectin: keep-alive
							line := trim(lowercase(copy(line, iPos+1)));
							for iPos := 1 to CountWords(line, ', ') do
								begin
									if CopyWord(line, iPos, ', ')='close' then
										req.ConnectionData.KeepAlive := false
									else if CopyWord(line, iPos, ', ')='keep-alive' then
										req.ConnectionData.KeepAlive := true
								end;
						end
					else if HeaderName = 'content-encoding' then
						begin
							h.kind := HeaderContentEncoding;
							h.ContentEncodingData := trim(lowercase(copy(line, iPos+1)));
							insert(h, req.headers);
						end
					else if HeaderName = 'content-language' then
						begin
							h.kind := HeaderContentLang;
							h.ContentLangData := trim(lowercase(copy(line, iPos+1)));
							insert(h, req.headers);
						end
					else if HeaderName = 'content-length' then
						begin
							h.kind := HeaderContentLength;
							h.ContentLengthData := trim(lowercase(copy(line, iPos+1)));
							insert(h, req.headers);
						end
					else if HeaderName = 'content-location' then
						begin
							h.kind := HeaderContentLocation;
							h.ContentLocationData := trim(lowercase(copy(line, iPos+1)));
							insert(h, req.headers);
						end
					else if HeaderName = 'content-md5' then
						begin
							h.kind := HeaderContentMD5;
							h.ContentMD5Data := trim(lowercase(copy(line, iPos+1)));
							insert(h, req.headers);
						end
					else if HeaderName = 'content-range' then
						begin
							iPos := pos('bytes', lowercase(line), iPos+1);
							if iPos=0 then
								begin
									res.StatusCode := RC_BAD_REQUEST;
									res.ReasonPhrase := 'Bad Request';
								end
							else
								begin
									s := copy(line, iPos+1);
									h.kind := HeaderContentRange;
									val(s, h.ContentRangeData.FirstBytePos, iPos);
									if (iPos=0) or (s[iPos]<>'-') then
										begin
											res.StatusCode := RC_BAD_REQUEST;
											res.ReasonPhrase := 'Bad Request';
										end
									else
										begin
											s := copy(s, iPos+1);
											val(s, h.ContentRangeData.LastBytePos, iPos);
											if (iPos=0) or (s[iPos]<>'/') then
												begin
													res.StatusCode := RC_BAD_REQUEST;
													res.ReasonPhrase := 'Bad Request';
												end
											else
												begin
													s := copy(s, iPos+1);
													if trim(s)='*' then
														begin
															h.ContentRangeData.InstanceLength := 0;
															insert(h, req.headers);
														end
													else
														begin
															val(s, h.ContentRangeData.InstanceLength, iPos);
															if iPos <> 0 then
																begin
																	res.StatusCode := RC_BAD_REQUEST;
																	res.ReasonPhrase := 'Bad Request';
																end
															else
																insert(h, req.headers);
														end
												end
										end
								end
						end
					else if HeaderName = 'content-type' then
						begin
							h.kind := HeaderContentType;
							h.ContentTypeData := trim(lowercase(copy(line, iPos+1)));
							insert(h, req.headers);
						end
					else if HeaderName = 'date' then
						begin
							h.kind := HeaderDate;
							h.DateData := trim(lowercase(copy(line, iPos+1)));
							insert(h, req.headers);
						end
					else if HeaderName = 'expect' then
						begin
							res.StatusCode := RC_EXPECTATION_FAILED;
							res.ReasonPhrase := 'Expectation Failed';
						end
					else if HeaderName = 'from' then
						begin
							h.kind := HeaderFrom;
							h.FromData := trim(lowercase(copy(line, iPos+1)));
							insert(h, req.headers);
						end
					else if (HeaderName='host') and (req.HostData.InternetHost='') then
						begin
							iPortPos := pos(':', line, iPos+1);
							if iPortPos <> 0 then
								begin
									req.HostData.InternetHost := trim(copy(line, iPos+1, iPortPos-iPos-1));
									val(copy(line, iPortPos+1), req.HostData.port, iErr);
									if iErr <> 0 then
										begin
											res.StatusCode := RC_BAD_REQUEST;
											res.ReasonPhrase := 'Bad Request';
										end
								end
							else
								begin
									req.HostData.InternetHost := trim(copy(line, iPos+1));
									req.HostData.port := DEFAULT_PORT;
								end
						end
					else if HeaderName = 'if-match' then
						begin
							h.kind := HeaderIfMatch;
							h.IfMatchData := trim(lowercase(copy(line, iPos+1)));
							insert(h, req.headers);
						end
					else if HeaderName = 'if-modified-since' then
						begin
							h.kind := HeaderIfModifiedSince;
							h.IfModifiedSinceData := trim(lowercase(copy(line, iPos+1)));
							insert(h, req.headers);
						end
					else if HeaderName = 'if-none-match' then
						begin
							h.kind := HeaderIfNoneMatch;
							h.IfNoneMatchData := trim(lowercase(copy(line, iPos+1)));
							insert(h, req.headers);
						end
					else if HeaderName = 'if-range' then
						begin
							h.kind := HeaderIfRange;
							h.IfRangeData := trim(lowercase(copy(line, iPos+1)));
							insert(h, req.headers);
						end
					else if HeaderName = 'if-unmodified-since' then
						begin
							h.kind := HeaderIfUnmodifiedSince;
							h.IfUnmodifiedSinceData := trim(lowercase(copy(line, iPos+1)));
							insert(h, req.headers);
						end
					else if HeaderName = 'last-modified' then
						begin
							h.kind := HeaderLastModified;
							h.LastModifiedData := trim(lowercase(copy(line, iPos+1)));
							insert(h, req.headers);
						end
					else if HeaderName = 'max-forwards' then
						begin
							h.kind := HeaderMaxForwards;
							val(copy(line, iPos+1), h.MaxForwardsData, iErr);
							if iErr <> 0 then
								begin
									res.StatusCode := RC_BAD_REQUEST;
									res.ReasonPhrase := 'Bad Request';
								end
							else
								insert(h, req.headers);
						end
					else if HeaderName = 'pragma' then
						begin
							h.kind := HeaderPragma;
							h.PragmaData := trim(lowercase(copy(line, iPos+1)));
							insert(h, req.headers);
						end
					else if HeaderName = 'range' then
						begin
							h.kind := HeaderRange;
							h.RangeData := trim(lowercase(copy(line, iPos+1)));
							insert(h, req.headers);
						end
					else if HeaderName = 'referer' then
						begin
							h.kind := HeaderReferrer;
							h.ReferrerData := trim(lowercase(copy(line, iPos+1)));
							insert(h, req.headers);
						end
					else if HeaderName = 'te' then
						begin
							h.kind := HeaderTE;
							h.TEData := trim(lowercase(copy(line, iPos+1)));
							insert(h, req.headers);
						end
					else if HeaderName = 'trailer' then
						begin
							h.kind := HeaderTrailer;
							h.TrailerData := trim(lowercase(copy(line, iPos+1)));
							insert(h, req.headers);
						end
					else if HeaderName = 'transfer-encoding' then
						begin
							h.kind := HeaderTransferEncoding;
							h.TransferEncodingData := trim(lowercase(copy(line, iPos+1)));
							insert(h, req.headers);
						end
					else if HeaderName = 'upgrade' then
						begin
							h.kind := HeaderUpgrade;
							h.UpgradeData := trim(lowercase(copy(line, iPos+1)));
							insert(h, req.headers);
						end
					else if HeaderName = 'user-agent' then
						begin
							h.kind := HeaderUserAgent;
							h.UserAgentData := trim(lowercase(copy(line, iPos+1)));
							insert(h, req.headers);
						end
				end
		end; (* ParseHeader *)

		//This procedure displays the request information.
		procedure DisplayRequestInfo(var req : RequestInfo);
		begin (* DisplayRequestInfo *)
			if req.uri <> '' then
				begin
					writeln('[URI]');
					writeln('URI=''', req.uri, '''');
					writeln('[ConnectionData]');
					writeln('KeepAlive=', req.ConnectionData.KeepAlive:1);
					writeln('[Host]');
					writeln('name=''', req.HostData.InternetHost, '''');
					writeln('Port=', req.HostData.port:1);
					writeln('[Resource]');
					writeln('resource=''', req.resource, '''');
				end
		end; (* DisplayRequestInfo *)

		//This procedure processes the request URI.
		procedure ProcessRequestURI(var req : RequestInfo);
		var
			iPos, iErr : integer;
		begin (* ProcessRequestURI *)
			req.uri := urldecode(req.uri); //Remove any encoding present

			//Convert all \'s to /'s in the URI for consistency.
			iPos := pos('\', req.uri);
			while iPos <> 0 do
				begin
					req.uri[iPos] := '/';
					iPos := pos('\', req.uri);
				end;

			//skip over the "http:" if present in the URI
			iPos := pos(':', req.uri);
			if (iPos<>0) and (lowercase(copy(req.uri, 1, iPos))='http:') then
				req.uri := copy(req.uri, iPos+1);

			//Extract and update the host information if any is present.
			//NOTE: The host information is not used for anything since this
			//web server does not support multiple hosts.
			if (length(req.uri)>=2) and (copy(req.uri, 1, 2)='//') then
				begin
					iPos := pos('/', req.uri, 3);
					if iPos<>0 then
						begin
							req.HostData.InternetHost := copy(req.uri, 3, iPos-3);
							req.uri := copy(req.uri, iPos);
							iPos := pos(':', req.HostData.InternetHost);
							if iPos <> 0 then
								begin
									val(copy(req.HostData.InternetHost, iPos+1), req.HostData.port, iErr);
									req.HostData.InternetHost := copy(req.HostData.InternetHost, 1, iPos-1);
								end;
						end;
				end;
		end; (* ProcessRequestURI *)

		//This procedure identifies the resource specified by the request.
		procedure IdentifyResource(var res : ResponseInfo; var req : RequestInfo);
		var
			iPos, iLen : integer;
		begin (* IdentifyResource *)
			//Convert all \'s to /'s in the Root directory
			req.resource := RootDir;
			iPos := pos('\', req.resource);
			while iPos > 0 do
				begin
					req.resource[iPos] := '/';
					iPos := pos('\', req.resource);
				end;

			//Append the URI to the Root directory to give the
			//full pathname of the resource.
			iLen := length(req.resource);
			if (iLen+1+length(req.uri)) > 255 then
				begin
					res.StatusCode := RC_REQUEST_URI_TOO_LARGE;
					res.ReasonPhrase := 'Request-URI Too Large';
				end
			else
				begin
					if (iLen>0) and (req.resource[iLen]='/') then
						req.resource := copy(req.resource, 1, iLen-1);
					req.resource := req.resource + req.uri
				end
		end; (* IdentifyResource *)

		//This procedure sends a line of response data.
		//Basically it justs adds a CR LF to the end of the data
		//and sends it through the ConnectedSocket".
		procedure SendResponseLine(var rl : OutputLineType);
		var
			iRet : integer;
		begin (* SendResponseLine *)
			writeln(rl);
			rl := rl + CR + LF;

			//Add code to check for errors and repeat call to 'send' until
			// all the data has been sent.
			LogMessage('Sending');
			iRet := send(ConnectedSocket, addr(rl[1]), length(rl), 0);
		end; (* SendResponseLine *)

		//This procedure sends the response headers.
		procedure SendResponseHeaders(var res : ResponseInfo; var req : RequestInfo);
		var
			sc : string;
			OutputLine : OutputLineType;

		begin (* SendResponseHeaders *)
			str(res.StatusCode:1, sc);
			OutputLine := 'HTTP/1.1 ' + sc + ' ' + res.ReasonPhrase;
			SendResponseLine(OutputLine);
			if not req.ConnectionData.KeepAlive then
				begin
					OutputLine := 'Connection: close';
					SendResponseLine(OutputLine);
				end;
			if res.ContentTypeData<>'' then
				begin
					OutputLine := 'Content-Type: '+res.ContentTypeData;
					SendResponseLine(OutputLine);
				end;
			if res.ContentLengthData <> '' then
				begin
					OutputLine := 'Content-Length: '+res.ContentLengthData;
					SendResponseLine(OutputLine);
				end;
			OutputLine := 'Server: IrieServ/1.0.0';
			SendResponseLine(OutputLine);
			OutputLine := '';
			SendResponseLine(OutputLine);
		end; (* SendResponseHeaders *)

		//This procedure sends the body of the response message
		procedure SendResponseBody(var res : ResponseInfo; var req : RequestInfo);
		const
			MAX_RESOURCE_BUFFER = 1024;
		var
			fResource : binary;
			buffer : packed array[1..MAX_RESOURCE_BUFFER] of char;
			wleft, wRead, wSent : word;
			iErr, iRet : integer;
		begin (* SendResponseBody *)
			wSent := 0;
			val(res.ContentLengthData, wLeft, iErr);
			if iErr<>0 then
				wLeft := 0;
			traperrors(false);
			reset(fResource, req.resource); //Attempt to open the resource.
			if getlasterror = 0 then
				begin
					while wLeft > 0 do
						begin
							//If the amount of data left to be sent is greater than
							//the size of the output buffer then just put in as much
							//as can hold. If the amount left is not greater than
							//the size of the output buffer then put in all that
							//is left.
							if wLeft > MAX_RESOURCE_BUFFER then
								rawread(fResource, buffer, MAX_RESOURCE_BUFFER, wRead)
							else
								rawread(fResource, buffer, wLeft, wRead);
							if (getlasterror<>0) or (wRead=0) then
								begin
									//If an error occured then signal we want to
									//close the socket and that we should try to send
									//any more data.
									req.ConnectionData.KeepAlive := false;
									wLeft := 0;
								end
							else
								begin
									//If everything is ok then send the data.
									LogMessage('Sending');
									iRet := send(ConnectedSocket, addr(buffer), wRead, 0);
									if (iRet=SOCKET_ERROR) or (iRet<=0) or (iRet>wRead) then
										begin
											//If an error occured then signal we want to
											//close the socket and that we should try to send
											//any more data.
											req.ConnectionData.KeepAlive := false;
											wLeft := 0;
										end
									else
										begin
											//If all went well then the amount left is less
											//by the amount that was read into the output buffer
											wLeft := wLeft - wRead;
											wSent := wSent + iRet
										end
								end
						end;
				end;
			close(fResource);
			traperrors(true);
			writeln(wSent:1, ' bytes sent');
		end; (* SendResponseBody *)

		//This procedure is responsible for sending a resource
		//(usually the one requested but can also by an error file).
		procedure SendResource(var res : ResponseInfo; var req : RequestInfo);
		var
			ext : filename;
			size : word;
			OutputLine : OutputLineType;
			sc : string[8];

			//This function returns the size of a file given it's name.
			//The built-in function "filesize" is used but it requires
			//that the file be open. So basically this function uses
			//the file's name to open the file and then calls "filesize".
			function GetFileSizeFromName(var fn : filename) : word;
			var
				f : binary;
			begin (* GetFileSizeFromName *)
				traperrors(false);
				reset(f, fn);
				if getlasterror = 0 then
					GetFileSizeFromName := filesize(f)
				else
					GetFileSizeFromName := 0;
				close(f);
				traperrors(true);
			end; (* GetFileSizeFromName *)

		begin (* SendResource *)
			//Get the size of the resource.
			size := GetFileSizeFromName(req.resource);
			str(size:1, res.ContentLengthData);
			if size=0 then
				begin
					req.ConnectionData.KeepAlive := false;
					res.ContentLengthData := '';
				end;
			fsplit(req.resource,,,ext); //Get the file's extension.

			//Try and determine the type of file based on it's extension.
			ext := lowercase(trim(ext));
			if (ext='.html') or (ext='.htm') or (ext='.shtml') then
				res.ContentTypeData := 'text/html'
			else if (ext='.jpeg') or (ext='.gif') then
				res.ContentTypeData := 'image/'+copy(ext, 2)
			else if (ext='.exe') or (ext='.com') or (ext='.ivm') then
				res.ContentTypeData := 'application/'+copy(ext, 2)
			else if ext='.txt' then
				res.ContentTypeData := 'text/plain';

			//Send the response headers
			SendResponseHeaders(res, req);

			//If the response has a body then send it.
			//Else send an error message as the body.
			if res.ContentLengthData <> '' then
				SendResponseBody(res, req)
			else
				begin
					OutputLine := '<html>';
					SendResponseLine(OutputLine);
					OutputLine := '<head>';
					SendResponseLine(OutputLine);
					OutputLine := '<title>Error</title>';
					SendResponseLine(OutputLine);
					OutputLine := '</head>';
					SendResponseLine(OutputLine);
					OutputLine := '<body>';
					SendResponseLine(OutputLine);
					OutputLine := '<p>The following error occured while processing your request:</p>';
					SendResponseLine(OutputLine);
					str(res.StatusCode:1, sc);
					OutputLine := '<p><strong>Error Code:</strong> '+sc+'</p>';
					SendResponseLine(OutputLine);
					OutputLine := '<p><strong>Description:</strong> '+res.ReasonPhrase+'</p>';
					SendResponseLine(OutputLine);
					OutputLine := '</body>';
					SendResponseLine(OutputLine);
					OutputLine := '</html>';
					SendResponseLine(OutputLine);
				end
		end; (* SendResource *)

		//This procedure sends a response when an error occurs.
		//It constructs a filename for the resource to send back from
		//the current StatusCode. It then concatenates the "Root" directory
		//and the "Errors" directory, and the constructed filename and
		//uses this as the pathname to the resource to send back.
		procedure SendErrorResponse(var res : ResponseInfo; var req : RequestInfo);
		var
			iPos, iLen : integer;
			fname : string[8];
		begin (* SendErrorResponse *)
			req.resource := RootDir;
			iPos := pos('\', req.resource);
			while iPos > 0 do
				begin
					req.resource[iPos] := '/';
					iPos := pos('\', req.resource);
				end;
			iLen := length(req.resource);
			if (iLen>0) and (req.resource[iLen]='/') then
				req.resource := copy(req.resource, 1, iLen-1);
			str(res.StatusCode:1, fname);
			fname := fname + '.html';
			req.resource := req.resource+'/'+ErrorsDir+'/'+fname;
			SendResource(res, req);
		end; (* SendErrorResponse *)

		//This function performs the method requested by the client.
		//Actually the only method currently implemented is the GET method.
		procedure PerformRequest(var res : ResponseInfo; var req : RequestInfo);

			//This function determines whether, adding the filename passed in to
			//the function to the current resource, identifies a file
			//that exists. This is used in cases where the current resource
			//is actually a directory and the web server should return one
			//of the files in the directory by default. Most of us have
			//browsed to a website with "http://domain.com" and had the
			//web server return a page even though none was specified.
			//What the web server did in that case was return a default file
			//from the root directory.
			function UseDefaultResource(fn : filename) : boolean;
			begin (* UseDefaultResource *)
				if FileExists(req.resource+'/'+fn) then
					begin
						req.resource := req.resource+'/'+fn;
						UseDefaultResource := true
					end
				else
					UseDefaultResource := false
			end; (* UseDefaultResource *)

			//This procedure performs a GET request.
			procedure PerformGetRequest(var res : ResponseInfo; var req : RequestInfo);
			var
				wFileMode : word;
				blnIsDir : boolean;
			begin (* PerformGetRequest *)
				//First try and determine if the requested resource is a directory
				//If the requested resource ends with a '/' then it must be a directory.
				if (length(req.resource)>0) and (copy(req.resource, length(req.resource),1)='/') then
					begin
						req.resource := copy(req.resource, 1, length(req.resource)-1);
						blnIsDir := true
					end
				else
					blnIsDir := false;

				//If the requested resource does not end with '/' then
				//use "getfilemode" to determine if it is a directory.
				if not blnIsDir then
					begin
						traperrors(false);
						getfilemode(req.resource, wFileMode);
						if (getlasterror=0) and ((wFileMode and dir_bit) <> 0) then
							blnIsDir := true
						else
							blnIsDir := false;
						traperrors(false);
					end;

				//If the request resource is a directory then try various
				//default files until we find one that exists.
				if blnIsDir then
					begin
						if UseDefaultResource('index.html') then
						else if UseDefaultResource('index.shtml') then
						else if UseDefaultResource('index.htm') then
						else if UseDefaultResource('index.shtm') then
						else if UseDefaultResource('default.html') then
						else if UseDefaultResource('default.shtml') then
						else if UseDefaultResource('default.htm') then
						else if UseDefaultResource('default.shtm') then
					end;

				//If the requested resource exists then set the response status to OK
				//and send the resource.
				if FileExists(req.resource) then
					begin
						res.StatusCode := RC_OK;
						res.ReasonPhrase := 'OK';
						SendResource(res, req)
					end
				else
					begin
						res.StatusCode := RC_NOT_FOUND;
						res.ReasonPhrase := 'Not Found';
						SendErrorResponse(res, req);
					end;
			end; (* PerformGetRequest *)

			procedure PerformPutRequest(var res : ResponseInfo; var req : RequestInfo);
			begin (* PerformPutRequest *)
				res.StatusCode := RC_NOT_IMPLEMENTED;
				res.ReasonPhrase := 'Not Implemented';
				SendErrorResponse(res, req);
			end; (* PerformPutRequest *)

			procedure PerformOptionsRequest(var res : ResponseInfo; var req : RequestInfo);
			begin (* PerformOptionsRequest *)
				res.StatusCode := RC_NOT_IMPLEMENTED;
				res.ReasonPhrase := 'Not Implemented';
				SendErrorResponse(res, req);
			end; (* PerformOptionsRequest *)

			procedure PerformHeadRequest(var res : ResponseInfo; var req : RequestInfo);
			begin (* PerformHeadRequest *)
				res.StatusCode := RC_NOT_IMPLEMENTED;
				res.ReasonPhrase := 'Not Implemented';
				SendErrorResponse(res, req);
			end; (* PerformHeadRequest *)

			procedure PerformPostRequest(var res : ResponseInfo; var req : RequestInfo);
			begin (* PerformPostRequest *)
				res.StatusCode := RC_NOT_IMPLEMENTED;
				res.ReasonPhrase := 'Not Implemented';
				SendErrorResponse(res, req);
			end; (* PerformPostRequest *)

		begin (* PerformRequest *)
			case req.method.kind of
			MethodGet:
				PerformGetRequest(res, req);
			MethodPut:
				PerformPutRequest(res, req);
			MethodOptions:
				PerformOptionsRequest(res, req);
			MethodHead:
				PerformHeadRequest(res, req);
			MethodPost:
				PerformPostRequest(res, req);
			MethodDelete, MethodTrace, MethodConnect, MethodExt:
				begin
					res.StatusCode := RC_NOT_IMPLEMENTED;
					res.ReasonPhrase := 'Not Implemented';
					SendErrorResponse(res, req);
				end;
			MethodUndef:
				begin
					res.StatusCode := RC_INTERNAL_SERVER_ERROR;
					res.ReasonPhrase := 'Internal Server Error';
					SendErrorResponse(res, req);
				end
			end
		end; (* PerformRequest *)

	begin (* ProcessHTTPRequests *)
		line := '';
		InitRequestInfo(ReqInfo);
		InitResponseInfo(ResInfo);
		attempts := 0;

		//Get the first line of the request message or time-out.
		done := false;
		while not done do
			begin
				line := GetRequestLine;
				attempts := attempts + 1;
				if (line=EOF_FLAG) or (line='') then
					begin
						if attempts>MAX_GET_ATTEMPTS then
							done := true
					end
				else
					done := true
			end;
		writeln(line);
		if line=EOF_FLAG then
			begin
				//We timed-out before we got the first line.
				ResInfo.StatusCode := RC_REQUEST_TIMEOUT;
				ResInfo.ReasonPhrase := 'Request Timeout';
				ReqInfo.ConnectionData.KeepAlive := false
			end
		else
			begin
				//We got the first line so parse it.
				//And if successful try and process the URI.
				ParseRequestLine(line, ReqInfo, ResInfo);
				if (ResInfo.StatusCode=0) and (ReqInfo.uri<>'') then
					ProcessRequestURI(ReqInfo);
			end;

		//If an ClientError or ServerError has occured make sure that the connection
		//will be closed after sending the response.
		if (ClassifyStatusCode(ResInfo)=ClientError) or (ClassifyStatusCode(ResInfo)=ServerError) then
			ReqInfo.ConnectionData.KeepAlive := false;
		if ResInfo.StatusCode=0 then
			begin
				//So far so good, so parse all the request headers.
				repeat
					line := GetRequestLine;
					writeln(line);
					if (line<>EOF_FLAG) and (line<>'') then
						ParseHeader(line, ReqInfo, ResInfo);
				until (line=EOF_FLAG) or (line='') or (ResInfo.StatusCode<>0)
			end;

		//If everything is still ok then identify the requested resource.
		if (ResInfo.StatusCode=0) and (ReqInfo.uri<>'') and (ReqInfo.uri<>'*') then
			IdentifyResource(ResInfo, ReqInfo);
		DisplayRequestInfo(ReqInfo);

		//If everything is ok then perform the request
		//else send an error response.
		if ResInfo.StatusCode=0 then
			PerformRequest(ResInfo, ReqInfo)
		else
			SendErrorResponse(ResInfo, ReqInfo);

		//If there are stored request headers then delete them so that
		//the request info structure can be re-used.
		if length(ReqInfo.headers) > 0 then
			delete(ReqInfo.headers, 1);

		//If the "ConnectedSocket" is defined and we are not going to keep alive connections
		//then close the "ConnectedSocket" and destroy the request queue.
		if (ConnectedSocket<>INVALID_SOCKET) and (not ReqInfo.ConnectionData.KeepAlive) then
			begin
				CloseServerSocket(ConnectedSocket);
				CharQueueDestroy(RequestQueue);
			end
	end; (* ProcessHTTPRequests *)

begin (* WinServS *)
	Initialize;
	repeat
		WaitForConnection;
		if ConnectedSocket <> INVALID_SOCKET then
			ProcessHTTPRequests;
	until ListeningSocket = INVALID_SOCKET;
	ServerShutDown;
end. (* WinServS *)
