/************************************************************************
** MODULE INFORMATION **
************************
** FILE NAME:          filterm.c
** SYSTEM NAME:        beholder
** MODULE NAME:        filter
** ORIGINAL AUTHOR(S): M.F.B. de Greeve
** VERSION NUMBER:     1.0
** CREATION DATE:      1992/9/2
** DESCRIPTION:        MIB-support of the filter - group
*************************************************************************/

#include <limits.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <mibsup.h>
#include <message.h>
#include <protocol.h>

#include "bitbyte.h"
#include "filtere.h"
#include "filterc.h"
#include "filter.h"


#define INDEXSIZE   1


static MIB_LOCAL    *filter = NULL;


static BOOLEAN RmonNext (SNMP_OBJECT *Obj, MIB_LOCAL **Local, WORD IdLen, WORD IdSze, void *Elm);



BOOLEAN FilterMInit(VOID)
{
    MessageConfig(FILTER_ERROR, "Filter");
    return TRUE;
}


/*****************************************************************
** NAME:        RmonNext
** SYNOPSIS:    BOOLEAN RmonNext (SNMP_OBJECT *Obj,
**                                  MIB_LOCAL **Local, WORD IdLen,
**                                  WORD IdSze, VOID **Elm)
** PARAMETERS:  Obj: requested object
**              Local: local datastructure: a collector
**              IdLen: identifier length known to the MIB
**              IdSze: size part application of identifier
**                     including the collector index !!
**              Elm: pointer to object if tables are used
** DESCRIPTION: application specific RMON NEXT function.
**              only called after MibRmon() for next requests 
**              searches the next object in the collector
** REMARKS:     ONLY FOR INTERNAL USE
** RETURNS:     TRUE: object found
**                    OPTIONAL: Elm -> pointer to object in table
*******************************************************************/

static BOOLEAN RmonNext(SNMP_OBJECT *Obj, MIB_LOCAL **Local, WORD IdLen, WORD IdSze, VOID **Elm)
{
    MIB_LOCAL    *local = *Local;

    if (local == NULL)
        return FALSE;
    if (Obj->IdLen == IdLen || local->Index > Obj->Id[IdLen])
    {
        /********************************************************************
        **  OPTIONAL: search function to find first obj in a table !! 
        **  if (not found)
        **      search next collector
        **  adjust object identifier to identifier first object
        **  Elm -> found table obj
        ********************************************************************/
        Obj->IdLen = IdLen + IdSze;
        Obj->Id[IdLen] = local->Index;
        return TRUE;
    }
    /********************************************************************
    **  OPTIONAL: search function to find next obj in a table !! 
    **  if (found)
    **      adjust object identifier to identifier next object
    **      Elm -> found table obj
    **      return TRUE
    ********************************************************************/
    *Local = local->Next;
    return (RmonNext(Obj, Local, IdLen, IdSze, Elm));
}


WORD filterIndex(SNMP_OBJECT *Obj, WORD IdLen)
{
    MIB_LOCAL    *local = NULL;

    if ((local = MibRmon (Obj, filter, IdLen, INDEXSIZE)) == NULL)
        return(SNMP_NOSUCHNAME);
    switch (Obj->Request)
    {
        case SNMP_PDU_NEXT:
            if (RmonNext (Obj, &local, IdLen, INDEXSIZE, NULL) == FALSE)
                return SNMP_NOSUCHNAME;
        case SNMP_PDU_GET:
            Obj->Syntax.LngInt = local->Index;
            return SNMP_NOERROR;
    }
}

WORD filterChannelIndex(SNMP_OBJECT *Obj, WORD IdLen)
{
    MIB_LOCAL   *local = NULL;
    FILTER      *data;

    if ((local = MibRmon(Obj, filter, IdLen, INDEXSIZE)) == NULL)
        return SNMP_NOSUCHNAME;

    switch (Obj->Request)
    {           
        case SNMP_PDU_NEXT:
            if (RmonNext (Obj, &local, IdLen, INDEXSIZE, NULL) == FALSE)
                return SNMP_NOSUCHNAME;
        case SNMP_PDU_GET:
            data = (FILTER*) local->Data;
            Obj->Syntax.LngInt = data->Channel;
            return SNMP_NOERROR;
        case SNMP_PDU_SET:
            data = (FILTER*) local->Data;
            data->Channel = Obj->Syntax.LngInt;
            return SNMP_NOERROR;
    }
}

WORD filterPktDataOffset(SNMP_OBJECT *Obj, WORD IdLen)
{
    MIB_LOCAL   *local = NULL;
    FILTER      *data;

    if ((local = MibRmon(Obj, filter, IdLen, INDEXSIZE)) == NULL)
        return SNMP_NOSUCHNAME;

    switch (Obj->Request)
    {
        case SNMP_PDU_NEXT:
            if (RmonNext (Obj, &local, IdLen, INDEXSIZE, NULL) == FALSE)
                return SNMP_NOSUCHNAME;
        case SNMP_PDU_GET:
            data = (FILTER*) local->Data;
            Obj->Syntax.LngInt = data->PktDataOffset;
            return SNMP_NOERROR;
        case SNMP_PDU_SET:
            data = (FILTER*) local->Data;
            data->PktDataOffset = Obj->Syntax.LngInt;
            return SNMP_NOERROR;
    }
}

WORD filterPktData(SNMP_OBJECT *Obj, WORD IdLen)
{
    MIB_LOCAL   *local = NULL;
    FILTER      *data;
    INT          count;

    if ((local = MibRmon(Obj, filter, IdLen, INDEXSIZE)) == NULL)
        return SNMP_NOSUCHNAME;

    switch (Obj->Request)
    {
        case SNMP_PDU_NEXT:
            if (RmonNext (Obj, &local, IdLen, INDEXSIZE, NULL) == FALSE)
                return SNMP_NOSUCHNAME;
        case SNMP_PDU_GET:
            data = (FILTER*) local->Data;
            count = Bits2Txt(Obj->Syntax.BufChr, data->PktData, data->PktDataLen,
                             data->Status == SNMP_VALID ? (WORD)data->PktDataOffset % BITS(BYTE) : 0,
                             data->PktDataBytes);
            if (count < 0)
            {
                Obj->SyntaxLen = 0;
                return SNMP_GENERROR;
            }
            else
                Obj->SyntaxLen = count;
            return SNMP_NOERROR;
        case SNMP_PDU_SET:
            data = (FILTER*) local->Data;
            if (data->Status != SNMP_UNDERCREATION)
                return SNMP_READONLY;
            if (Obj->SyntaxLen > SNMP_SIZE_BUFCHR)
                return SNMP_TOOBIG;
            count = Txt2Bits(&data->PktData, 0, &data->PktDataBytes, Obj->Syntax.BufChr, Obj->SyntaxLen);
            if (count < 0)
                return SNMP_BADVALUE;
            else
                data->PktDataLen = count;
            return SNMP_NOERROR;
    }
}

WORD filterPktDataMask(SNMP_OBJECT *Obj, WORD IdLen)
{
    MIB_LOCAL   *local = NULL;
    FILTER      *data;
    INT          count;

    if ((local = MibRmon(Obj, filter, IdLen, INDEXSIZE)) == NULL)
        return SNMP_NOSUCHNAME;

    switch (Obj->Request)
    {
        case SNMP_PDU_NEXT:
            if (RmonNext (Obj, &local, IdLen, INDEXSIZE, NULL) == FALSE)
                return SNMP_NOSUCHNAME;
        case SNMP_PDU_GET:
            data = (FILTER*) local->Data;
            count = Bits2Txt(Obj->Syntax.BufChr, data->PktDataMask, data->PktDataMaskLen,
                             data->Status == SNMP_VALID ? (WORD)data->PktDataOffset % BITS(BYTE) : 0,
                             data->PktDataMaskBytes);
            if (count < 0)
            {
                Obj->SyntaxLen = 0;
                return SNMP_GENERROR;
            }
            else
                Obj->SyntaxLen = count;
            return SNMP_NOERROR;
        case SNMP_PDU_SET:
            data = (FILTER*) local->Data;
            if (data->Status != SNMP_UNDERCREATION)
                return SNMP_READONLY;
            if (Obj->SyntaxLen > SNMP_SIZE_BUFCHR)
                return SNMP_TOOBIG;
            count = Txt2Bits(&data->PktDataMask, 0, &data->PktDataMaskBytes, Obj->Syntax.BufChr, Obj->SyntaxLen);
            if (count < 0)
                return SNMP_BADVALUE;
            else
                data->PktDataMaskLen = count;
            return SNMP_NOERROR;
    }
}

WORD filterPktDataNotMask(SNMP_OBJECT *Obj, WORD IdLen)
{
    MIB_LOCAL   *local = NULL;
    FILTER      *data;
    INT          count;

    if ((local = MibRmon(Obj, filter, IdLen, INDEXSIZE)) == NULL)
        return SNMP_NOSUCHNAME;

    switch (Obj->Request)
    {
        case SNMP_PDU_NEXT:
            if (RmonNext (Obj, &local, IdLen, INDEXSIZE, NULL) == FALSE)
                return SNMP_NOSUCHNAME;
        case SNMP_PDU_GET:
            data = (FILTER*) local->Data;
            count = Bits2Txt(Obj->Syntax.BufChr, data->PktDataNotMask, data->PktDataNotMaskLen,
                             data->Status == SNMP_VALID ? (WORD)data->PktDataOffset % BITS(BYTE) : 0,
                             data->PktDataNotMaskBytes);
            if (count < 0)
            {
                Obj->SyntaxLen = 0;
                return SNMP_GENERROR;
            }
            else
                Obj->SyntaxLen = count;
            return SNMP_NOERROR;
        case SNMP_PDU_SET:
            data = (FILTER*) local->Data;
            if (data->Status != SNMP_UNDERCREATION)
                return SNMP_READONLY;
            if (Obj->SyntaxLen > SNMP_SIZE_BUFCHR)
                return SNMP_TOOBIG;
            count = Txt2Bits(&data->PktDataNotMask, 0, &data->PktDataNotMaskBytes, Obj->Syntax.BufChr, Obj->SyntaxLen);
            if (count < 0)
                return SNMP_BADVALUE;
            else
                data->PktDataNotMaskLen = count;
            return SNMP_NOERROR;
    }
}


WORD filterPktStatus(SNMP_OBJECT *Obj, WORD IdLen)
{
    MIB_LOCAL   *local = NULL;
    FILTER      *data;

    if ((local = MibRmon(Obj, filter, IdLen, INDEXSIZE)) == NULL)
        return SNMP_NOSUCHNAME;

    switch (Obj->Request)
    {
        case SNMP_PDU_NEXT:
            if (RmonNext (Obj, &local, IdLen, INDEXSIZE, NULL) == FALSE)
                return SNMP_NOSUCHNAME;
        case SNMP_PDU_GET:
            data = (FILTER*) local->Data;
            Obj->Syntax.LngInt = data->PktStatus;
            return SNMP_NOERROR;
        case SNMP_PDU_SET:
            data = (FILTER*) local->Data;
            if (data->Status != SNMP_UNDERCREATION)
                return SNMP_READONLY;
            /* This is not supported: can not be set */
            return SNMP_GENERROR;
    }
}

WORD filterPktStatusMask(SNMP_OBJECT *Obj, WORD IdLen)
{
    MIB_LOCAL   *local = NULL;
    FILTER      *data;

    if ((local = MibRmon(Obj, filter, IdLen, INDEXSIZE)) == NULL)
        return SNMP_NOSUCHNAME;

    switch (Obj->Request)
    {
        case SNMP_PDU_NEXT:
            if (RmonNext (Obj, &local, IdLen, INDEXSIZE, NULL) == FALSE)
                return SNMP_NOSUCHNAME;
        case SNMP_PDU_GET:
            data = (FILTER*) local->Data;
            Obj->Syntax.LngInt = data->PktStatusMask;
            return SNMP_NOERROR;
        case SNMP_PDU_SET:
            data = (FILTER*) local->Data;
            if (data->Status != SNMP_UNDERCREATION)
                return SNMP_READONLY;
            /* This is not supported: can not be set */
            return SNMP_GENERROR;
    }
}

WORD filterPktStatusNotMask(SNMP_OBJECT *Obj, WORD IdLen)
{
    MIB_LOCAL   *local = NULL;
    FILTER      *data;

    if ((local = MibRmon(Obj, filter, IdLen, INDEXSIZE)) == NULL)
        return SNMP_NOSUCHNAME;

    switch (Obj->Request)
    {
        case SNMP_PDU_NEXT:
            if (RmonNext (Obj, &local, IdLen, INDEXSIZE, NULL) == FALSE)
                return SNMP_NOSUCHNAME;
        case SNMP_PDU_GET:
            data = (FILTER*) local->Data;
            Obj->Syntax.LngInt = data->PktStatusNotMask;
            return SNMP_NOERROR;
        case SNMP_PDU_SET:
            data = (FILTER*) local->Data;
            if (data->Status != SNMP_UNDERCREATION)
                return SNMP_READONLY;
            /* This is not supported: can not be set */
            return SNMP_GENERROR;
    }
}


WORD filterOwner(SNMP_OBJECT *Obj, WORD IdLen)
{
    MIB_LOCAL   *local = NULL;
    FILTER      *data;

    if ((local = MibRmon(Obj, filter, IdLen, INDEXSIZE)) == NULL)
        return SNMP_NOSUCHNAME;

    switch (Obj->Request)
    {
        case SNMP_PDU_NEXT:
            if (RmonNext (Obj, &local, IdLen, INDEXSIZE, NULL) == FALSE)
                return SNMP_NOSUCHNAME;
        case SNMP_PDU_GET:
            data = (FILTER*) local->Data;
            memcpy (Obj->Syntax.BufChr, data->Owner, data->OwnerLen);
            Obj->SyntaxLen = data->OwnerLen;
            return SNMP_NOERROR;
        case SNMP_PDU_SET:
            data = (FILTER*) local->Data;
            if (data->Status != SNMP_UNDERCREATION)
                return SNMP_READONLY;
            if (Obj->SyntaxLen > SNMP_SIZE_BUFCHR)
                return SNMP_TOOBIG;
            memcpy (data->Owner, Obj->Syntax.BufChr, Obj->SyntaxLen);
            data->OwnerLen = Obj->SyntaxLen;
            return SNMP_NOERROR;
    }
}

WORD filterStatus(SNMP_OBJECT *Obj, WORD IdLen)
{
    MIB_LOCAL   *local = NULL;
    FILTER      *data;

    local = MibRmon(Obj, filter, IdLen, INDEXSIZE);

    switch (Obj->Request)
    {
        case SNMP_PDU_GET:
            if (local == NULL)
            {
                Obj->Syntax.LngInt = SNMP_INVALID;
                return SNMP_NOSUCHNAME;
            }
            data = (FILTER*) local->Data;
            Obj->Syntax.LngInt = data->Status;
            return SNMP_NOERROR;
        case SNMP_PDU_NEXT:
            if (RmonNext (Obj, &local, IdLen, INDEXSIZE, NULL) == FALSE)
            {
                Obj->Syntax.LngInt = SNMP_INVALID;
                return SNMP_NOSUCHNAME;
            }
            data = (FILTER*) local->Data;
            Obj->Syntax.LngInt = data->Status;
            return SNMP_NOERROR;
        case SNMP_PDU_SET:
            if (local == NULL)
            {
                switch (Obj->Syntax.LngInt)
                {
                    case SNMP_CREATEREQUEST:
                        if ((local = MibInsert(Obj, &filter, IdLen, INDEXSIZE)) == NULL)
                            return SNMP_GENERROR;
                        if ((local->Data = DnpapMalloc(sizeof(FILTER))) == NULL)
                            return SNMP_GENERROR;
                        data = (FILTER*) local->Data;
                        memset(data, 0, sizeof(FILTER));
                        if (FilterCInit(data) == TRUE)
                        {
                            data->Status = SNMP_UNDERCREATION;
                            DnpapMessage(DMC_MESSAGE, FILTER_CREATE, "filter: collector %ld created", local->Index);
                            return SNMP_NOERROR;
                        }
                        DnpapFree(local->Data);
                        MibRemove (Obj, &filter, IdLen, INDEXSIZE);
                        return SNMP_GENERROR;
                    default:
                        return SNMP_NOSUCHNAME;
                }
            }
            data = (FILTER*) local->Data;
            switch (data->Status)
            {
                case SNMP_UNDERCREATION:
                    switch (Obj->Syntax.LngInt)
                    {
                        case SNMP_VALID:
                            if (FilterCStart(data) == TRUE)
                            {
                                data->Status = SNMP_VALID;
                                DnpapMessage(DMC_MESSAGE, FILTER_ACTIVE, "filter: collector %ld active", local->Index);
                                return SNMP_NOERROR;
                            }
                            return SNMP_GENERROR;
                        case SNMP_INVALID:
                            if (FilterCStop(data) == TRUE)
                            {
                                DnpapFree(local->Data);
                                DnpapMessage(DMC_MESSAGE, FILTER_DESTROY, "filter: collector %ld destroyed", local->Index);
                                if (MibRemove (Obj, &filter, IdLen, INDEXSIZE) == TRUE)
                                    return SNMP_NOERROR;
                            }
                            return SNMP_GENERROR;
                        default:
                            return SNMP_BADVALUE;
                    }
                case SNMP_VALID:
                    switch (Obj->Syntax.LngInt)
                    {
                        case SNMP_INVALID:
                            if (FilterCStop(data) == TRUE)
                            {
                                DnpapFree(local->Data);
                                DnpapMessage(DMC_MESSAGE, FILTER_DESTROY, "filter: collector %ld destroyed", local->Index);
                                if (MibRemove (Obj, &filter, IdLen, INDEXSIZE) == TRUE)
                                    return SNMP_NOERROR;
                            }
                            return SNMP_GENERROR;
                        case SNMP_VALID:
                            return SNMP_NOERROR;
                        default:
                            return SNMP_BADVALUE;
                    }
            }
    }
}


BOOLEAN FilterPkt(LONG channelIndex, PROT_PKT *pkt)
{
MIB_LOCAL *local;
FILTER *data;
PROT_OBJ len =   {0, {1,5}};
PROT_OBJ packet= {0, {1,7}};
BYTE scratch;
WORD bi, fb, lb;
BOOLEAN rule2fail, rule3fail, filterfound;

    if (ProtGetField(pkt, &len) == FALSE ||
        ProtGetField(pkt, &packet) == FALSE) 
    {
        DnpapMessage(DMC_ERROR, FILTER_BADPACKET,"filter: bad packet");
        return FALSE;
    }

    /*  All results of filters that apply are OR'ed. Actually lazy evaluation  */
    /*  is implemented: the first filter that matches returns TRUE.            */
    /*  In the heuristics below, a 'continue' is equivalent to a failing match,  */
    /*  and a 'break' is equivalent to a successful match.                       */
    filterfound = FALSE;
    for (local = filter; local != NULL; local = local->Next)
    {
        data = (FILTER*) local->Data;
        if (data->Status == SNMP_VALID && data->Channel == channelIndex)
        {
            filterfound = TRUE;
            if (data->PktDataLen == 0)
                continue;	/*  no data, no match  */
            if (data->PktDataLen + data->PktDataOffset > BITS(BYTE)*len.Syntax.LngInt)
                continue;       /*  rule 1 of packet matching  */
            fb = ((WORD)data->PktDataOffset)/BITS(BYTE);
            lb = ((WORD)data->PktDataOffset+data->PktDataLen+BITS(BYTE)-1)/BITS(BYTE);
            rule2fail = FALSE;
            if (data->PktDataNotMaskAllZero == TRUE)
                rule3fail = FALSE;
            else
                rule3fail = TRUE;
            for (bi = fb; bi < lb; bi++)
            {
                scratch = (((BYTE*)packet.Syntax.Ptr)[bi] ^ data->PktData[bi-fb]) & data->PktDataMask[bi-fb];
                if ((scratch & ~data->PktDataNotMask[bi-fb]) != 0)
                {
                    rule2fail = TRUE;
                    break;
                }
                if (rule3fail == TRUE && (scratch & data->PktDataNotMask[bi-fb]) != 0)
                    rule3fail = FALSE;
            }
            if (rule2fail == FALSE && rule3fail == FALSE)
            {
                return TRUE;
            }
            else
                continue;
        }
    }

    if (filterfound == TRUE)
        return FALSE;
    return TRUE;
}
