/************************************************************************
** MODULE INFORMATION*
**********************
**     FILE     NAME:       hostc.c
**     SYSTEM   NAME:       beholder
**     ORIGINAL AUTHOR(S):  Richard & Marc
**     VERSION  NUMBER:     0.99
**     CREATION DATE:       1992/9/29
**
** DESCRIPTION: host group of the RMON MIB: collector
**              
*************************************************************************
** CHANGES INFORMATION **
*************************
** REVISION:    $Revision$
** WORKFILE:    $Workfile$
** LOGINFO:     $Log$
*************************************************************************/
#if ! defined(PRD)
static char _pvcs_hdr[] =
"$Header$";
#endif

#include <stdlib.h>
#include <memory.h>
#include <dnpap.h>
#include <config.h>
#include <message.h>
#include <mac.h>
#include <hash.h>
#include <sys.h>
#include <protocol.h>
#include <topn.h>

#include "hoste.h"
#include "hostc.h"


static WORD HostMaxNrHosts = 2000;


static VOID HostCallback(MAC_COLL *collector, PROT_PKT *pkt);

static BOOLEAN HostAddLexList(HOST_CONTROL *hostcontrol, HOST *host);
static BOOLEAN HostRemoveLexList(HOST_CONTROL *hostcontrol, HOST *host);

static BOOLEAN HostAddTimeList(HOST_CONTROL *hostcontrol, HOST *host);
static BOOLEAN HostRemoveTimeList(HOST_CONTROL *hostcontrol, HOST *host);

static HOST* HostAddLRUList(HOST_CONTROL *hostcontrol, HOST *host);
static BOOLEAN HostUpdateLRUList(HOST_CONTROL *hostcontrol, HOST *host);
static BOOLEAN HostRemoveLRUList(HOST_CONTROL *hostcontrol, HOST *host);

static BOOLEAN HostUpdateTimeTable(HOST_CONTROL *hostcontrol);

static VOID DelHosts(HOST_CONTROL *hostcontrol);

#ifdef DEBUG
static VOID HostCheckListsConsistency(HOST_CONTROL *host);
static VOID HostCheckListsRemove(HOST_CONTROL *hostcontrol, HOST *host);
static VOID HostCheckListsRemove2(HOST_CONTROL *hostcontrol, HOST *host);
static VOID HostCheckListsAdd(HOST_CONTROL *hostcontrol, HOST *host);
static VOID HostCheckListsAdd2(HOST_CONTROL *hostcontrol, HOST *host);
#endif



BOOLEAN HostConfigInit(VOID)
{
	ConfigGetShort("beholder.host.maxnrhosts", &HostMaxNrHosts);

	if (HostMaxNrHosts < 2)
    {
        DnpapMessage(DMC_WARNING, HOST_MAX, "hostcontrol: beholder.host.maxnrhosts < 2, setting it to 2");
        HostMaxNrHosts = 2;
    }

	return TRUE;
}


BOOLEAN HostCInit(HOST_CONTROL *hostcontrol)
{
LONG source[] = {1,3,6,1,2,1,2,2,1,1,1};

    hostcontrol->Index = 0;
    memcpy(hostcontrol->Source, source, sizeof(source));
    hostcontrol->SourceLen = sizeof(source)/sizeof(source[0]);
    hostcontrol->TableSize = 0;
    hostcontrol->LastDeleteTime = 0;
    hostcontrol->Owner[0] = '\0';
    hostcontrol->OwnerLen = 0;
    hostcontrol->Status = SNMP_INVALID;
    
    if ((hostcontrol->Iface =
        MacIfaceGet((WORD) hostcontrol->Source[hostcontrol->SourceLen-1])) == NULL)
    {
        DnpapMessage(DMC_ERROR, HOST_NETINIT, "hostcontrol: network initialisation failed");
        return (FALSE);
    }
	
    hostcontrol->Table = NULL;
    
    hostcontrol->LexList = NULL;
    
    hostcontrol->TimeList = NULL;
    hostcontrol->TimeLast = NULL;
    hostcontrol->TimeTable = NULL;
    hostcontrol->TimeListUpdated = FALSE;

	hostcontrol->LRUList = NULL;
	hostcontrol->LRULast = NULL;
	
	return TRUE;
}


BOOLEAN HostCStart(HOST_CONTROL *hostcontrol)
{
    hostcontrol->Coll.Rcve       = HostCallback;
    hostcontrol->Coll.specific   = hostcontrol;
    if (!MacCollRegister(&(hostcontrol->Coll)))
    {
        DnpapMessage(DMC_ERROR, HOST_NETERR, "hostcontrol: network initialisation failed");
        return FALSE;
    }

    if ((hostcontrol->Table = NewHash(2011, NULL)) == NULL)
    {
        DnpapMessage(DMC_ERROR, HOST_HASHERR, "hostcontrol: can not create hashtable");
        return FALSE;
    }

    if ((hostcontrol->TimeTable = (HOST**)DnpapMalloc(HostMaxNrHosts*sizeof(HOST*))) == NULL)
    {
        DnpapMessage(DMC_ERROR, HOST_TIMER, "hostcontrol: can not create time ordered table");
        return FALSE;
    }

    return TRUE;
}


BOOLEAN HostCStop(HOST_CONTROL *hostcontrol)
{
    MacCollRemove(&(hostcontrol->Coll));

    DelHash(hostcontrol->Table);

    DelHosts(hostcontrol);

    DnpapFree(hostcontrol->TimeTable);

    return TRUE;
}


VOID HostCallback(MAC_COLL *collector, PROT_PKT *pkt)
{
HOST_CONTROL *hostcontrol = collector->specific;
HOST *host = NULL, *oldhost = NULL;
BYTE broadcast[HOST_SZEADDR] = {0xff,0xff,0xff,0xff,0xff,0xff};
PROT_OBJ Interface = {0, {1,2}};
PROT_OBJ Size = {0, {1,4}};
PROT_OBJ Dst = {1, {2,1}};
PROT_OBJ Src = {1, {2,2}};
LWORD size;

    if (ProtGetField(pkt,&Interface) == TRUE &&
        Interface.Syntax.LngInt == hostcontrol->Source[hostcontrol->SourceLen-1])
    {
        if (ProtGetField(pkt,&Size) == TRUE &&
            ProtGetField(pkt,&Src) == TRUE &&
            ProtGetField(pkt,&Dst) == TRUE)
        {
            size = Size.Syntax.LngUns + 4L;
            if (Src.SyntaxLen != HOST_SZEADDR || Dst.SyntaxLen != HOST_SZEADDR)
            {
                DnpapMessage(DMC_ERROR, HOST_INVPACKET, "hostcontrol: invalid packet");
                return;
            }

            if ((host = HashSearch(hostcontrol->Table, Src.Syntax.BufChr, HOST_SZEADDR)) == NULL)
            {
                /*  first try to add the new host  */
                if ((host = DnpapMalloc(sizeof(HOST))) != NULL)
                {
                    memset(host, 0, sizeof(HOST));
                    memcpy(host->Address, Src.Syntax.BufChr, HOST_SZEADDR);
#ifdef DEBUG
                    HostCheckListsAdd(hostcontrol, host);
#endif
                    if (HashAdd(hostcontrol->Table, host->Address, HOST_SZEADDR, host) == NULL)
                    {
                        DnpapMessage(DMC_WARNING, HOST_NADD, "hostcontrol: host could not be added to the hash table");
                        DnpapFree(host);
                        host = NULL;
                    }
                    else
                    {
#ifdef DEBUG
                        if (HashSearch(hostcontrol->Table, Src.Syntax.BufChr, HOST_SZEADDR) == NULL)
                            DnpapMessage(DMC_FATAL, HOST_ERROR, "host that was just added can not be retrieved");
#endif
                        HostAddLexList(hostcontrol, host);
                        HostAddTimeList(hostcontrol, host);
                        oldhost = HostAddLRUList(hostcontrol, host);
#ifdef DEBUG
                        HostCheckListsAdd2(hostcontrol, host);
#endif
                        hostcontrol->TableSize++;

                        if (hostcontrol->TableSize > (LONG)HostMaxNrHosts)
                        {
#ifdef DEBUG
                            HostCheckListsRemove2(hostcontrol, oldhost);
#endif
                            HostRemoveLRUList(hostcontrol, oldhost);
                            HostRemoveTimeList(hostcontrol, oldhost);
                            HostRemoveLexList(hostcontrol, oldhost);
                            HashRemove(hostcontrol->Table, oldhost->Address, HOST_SZEADDR);
#ifdef DEBUG
                            HostCheckListsRemove(hostcontrol, oldhost);
                            memset(oldhost->Address, 0, HOST_SZEADDR);
#endif
                            TopNEntryReset(hostcontrol->Index, oldhost);
                            DnpapFree(oldhost);
                            hostcontrol->LastDeleteTime = SysTime();
                            hostcontrol->TableSize--;
#ifdef DEBUG
                            if (HashSearch(hostcontrol->Table, Src.Syntax.BufChr, HOST_SZEADDR) == NULL)
                                DnpapMessage(DMC_FATAL, HOST_ERROR, "host that was just added can not be retrieved");
#endif
                        }
                    }
                }
            }
            if (host != NULL)
            {
                host->OutPkts++;
                host->OutOctets += size;
                if (!memcmp(Dst.Syntax.BufChr, &broadcast, HOST_SZEADDR))
                    host->OutBroadcastPkts++;
                else
                    if (Dst.Syntax.BufChr[0] & 0x01)
                        host->OutMulticastPkts++;
                        
                HostUpdateLRUList(hostcontrol, host);
                TopNEntryUpdate(hostcontrol->Index, host);
            }

#ifdef DEBUG
            HostCheckListsConsistency(hostcontrol);
#endif

            if ((host = HashSearch(hostcontrol->Table, Dst.Syntax.BufChr, HOST_SZEADDR)) == NULL)
            {
                if ((host = DnpapMalloc(sizeof(HOST))) != NULL)
                {
                    memset(host, 0, sizeof(HOST));
                    memcpy(host->Address, Dst.Syntax.BufChr, HOST_SZEADDR);
#ifdef DEBUG
                    HostCheckListsAdd(hostcontrol, host);
#endif
                    if (HashAdd(hostcontrol->Table, host->Address, HOST_SZEADDR, host) == NULL)
                    {
                        DnpapMessage(DMC_WARNING, HOST_NADD, "hostcontrol: host could not be added to the hash table");
                        DnpapFree(host);
                        host = NULL;
                    }
                    else
                    {
#ifdef DEBUG
                        if (HashSearch(hostcontrol->Table, Dst.Syntax.BufChr, HOST_SZEADDR) == NULL)
                            DnpapMessage(DMC_FATAL, HOST_ERROR, "host that was just added can not be retrieved");
#endif
                        HostAddLexList(hostcontrol, host);
                        HostAddTimeList(hostcontrol, host);
                        oldhost = HostAddLRUList(hostcontrol, host);
#ifdef DEBUG
                        HostCheckListsAdd2(hostcontrol, host);
#endif
                        hostcontrol->TableSize++;

                        if (hostcontrol->TableSize > (LONG)HostMaxNrHosts)
                        {
#ifdef DEBUG
                            HostCheckListsRemove2(hostcontrol, oldhost);
#endif
                            HostRemoveLRUList(hostcontrol, oldhost);
                            HostRemoveTimeList(hostcontrol, oldhost);
                            HostRemoveLexList(hostcontrol, oldhost);
                            HashRemove(hostcontrol->Table, oldhost->Address, HOST_SZEADDR);
#ifdef DEBUG
                            HostCheckListsRemove(hostcontrol, oldhost);
                            memset(oldhost->Address, 0, HOST_SZEADDR);
#endif
                            TopNEntryReset(hostcontrol->Index, oldhost);
                            DnpapFree(oldhost);
                            hostcontrol->LastDeleteTime = SysTime();
                            hostcontrol->TableSize--;
#ifdef DEBUG
                            if (HashSearch(hostcontrol->Table, Dst.Syntax.BufChr, HOST_SZEADDR) == NULL)
                                DnpapMessage(DMC_FATAL, HOST_ERROR, "host that was just added can not be retrieved");
#endif
                        }
                    }
                }
            }
            if (host != NULL)
            {
                host->InPkts++;
                host->InOctets += size;
                
                HostUpdateLRUList(hostcontrol, host);
                TopNEntryUpdate(hostcontrol->Index, host);
            }

#ifdef DEBUG
            HostCheckListsConsistency(hostcontrol);
#endif
        }
    }

    return;
}


BOOLEAN HostAddLexList(HOST_CONTROL *hostcontrol, HOST *host)
{
HOST *p, *q;

    if (hostcontrol->LexList == NULL)
    {
        hostcontrol->LexList = host;
        host->LexPrev = NULL;
        host->LexNext = NULL;
        return TRUE;
    }

    for (p = hostcontrol->LexList, q = NULL; p != NULL; q = p, p = p->LexNext)
    {
        if (memcmp(p->Address, host->Address, HOST_SZEADDR) >= 0)
            break;
    }

    if (q != NULL)
        q->LexNext = host;
    else
        hostcontrol->LexList = host;

    if (p != NULL)
        p->LexPrev = host;

    host->LexPrev = q;
    host->LexNext = p;

    return TRUE;
}


BOOLEAN HostAddTimeList(HOST_CONTROL *hostcontrol, HOST *host)
{
    if (hostcontrol->TimeList == NULL)
    {
        hostcontrol->TimeList = host;
        hostcontrol->TimeLast = host;
        host->TimePrev = NULL;
        host->TimeNext = NULL;
        
	    hostcontrol->TimeListUpdated = TRUE;

        return TRUE;
    }

    host->TimePrev = hostcontrol->TimeLast;
    host->TimeNext = NULL;
    hostcontrol->TimeLast->TimeNext = host;
    hostcontrol->TimeLast = host;
    
	hostcontrol->TimeListUpdated = TRUE;
	
    return TRUE;
}


HOST* HostAddLRUList(HOST_CONTROL *hostcontrol, HOST *host)
{
    if (hostcontrol->LRUList == NULL)
    {
        hostcontrol->LRUList = host;
        hostcontrol->LRULast = host;
        host->LRUPrev = NULL;
        host->LRUNext = NULL;
        
        return NULL;
    }

    host->LRUNext = hostcontrol->LRUList;
    host->LRUPrev = NULL;
    hostcontrol->LRUList->LRUPrev = host;
    hostcontrol->LRUList = host;

    return hostcontrol->LRULast;
}      


BOOLEAN HostUpdateLRUList(HOST_CONTROL *hostcontrol, HOST *host)
{
    if (hostcontrol->TableSize > 1 && hostcontrol->LRUList != host)
    {
        HostRemoveLRUList(hostcontrol, host);
        HostAddLRUList(hostcontrol, host);
    }
    return TRUE;
}


BOOLEAN HostRemoveLexList(HOST_CONTROL *hostcontrol, HOST *host)
{
    if (host->LexPrev != NULL)
        host->LexPrev->LexNext = host->LexNext;
    else
        hostcontrol->LexList = host->LexNext;
    if (host->LexNext != NULL)
        host->LexNext->LexPrev = host->LexPrev;

    host->LexPrev = NULL;
    host->LexNext = NULL;
    
    return TRUE;
}


BOOLEAN HostRemoveTimeList(HOST_CONTROL *hostcontrol, HOST *host)
{
    if (host->TimePrev != NULL)
        host->TimePrev->TimeNext = host->TimeNext;
    else
        hostcontrol->TimeList = host->TimeNext;
    if (host->TimeNext != NULL)
        host->TimeNext->TimePrev = host->TimePrev;
    else
        hostcontrol->TimeLast = host->TimePrev;

	hostcontrol->TimeListUpdated = TRUE;
	
    host->TimePrev = NULL;
    host->TimeNext = NULL;
    
    return TRUE;
}


BOOLEAN HostRemoveLRUList(HOST_CONTROL *hostcontrol, HOST *host)
{
    if (host->LRUPrev != NULL)
        host->LRUPrev->LRUNext = host->LRUNext;
    else
        hostcontrol->LRUList = host->LRUNext;
    if (host->LRUNext != NULL)
        host->LRUNext->LRUPrev = host->LRUPrev;
    else
        hostcontrol->LRULast = host->LRUPrev;

    host->LRUPrev = NULL;
    host->LRUNext = NULL;
    
    return TRUE;
}


VOID DelHosts(HOST_CONTROL *hostcontrol)
{
HOST *host1, *host2;

    for (host2 = hostcontrol->LexList; host2 != NULL; host2 = host1)
    {
        host1 = host2->LexNext;
        DnpapFree(host2);
        hostcontrol->TableSize--;
    }
    hostcontrol->LexList = NULL;
}


HOST* HostHashSearch(HOST_CONTROL *hostcontrol, HOST *host, WORD len)
{
HOST* p;

    if (len == HOST_SZEADDR &&
        (p = HashSearch(hostcontrol->Table, host->Address, HOST_SZEADDR)) != NULL)
        return p;
    else
    {
        for (p = hostcontrol->LexList; p != NULL; p = p->LexNext)
        {
            if (memcmp(p->Address, host->Address, len) >= 0)
                break;
        }
        return p;
    }
    return NULL;
}


HOST* HostHashSearch2(HOST_CONTROL *hostcontrol, SNMP_OBJECT *obj, WORD idlen)
{
WORD i;
BYTE address[HOST_SZEADDR];

    for (i = 0; i < HOST_SZEADDR; i++)
        address[i] = (BYTE)obj->Id[idlen+1+i];
    return HashSearch(hostcontrol->Table, address, HOST_SZEADDR);
}


HOST* HostTimeSearch(HOST_CONTROL *hostcontrol, SNMP_OBJECT *obj, WORD idlen)
{
	if (hostcontrol->TimeListUpdated == TRUE)
		HostUpdateTimeTable(hostcontrol);
    return hostcontrol->TimeTable[obj->Id[idlen+1]-1];
}


LONG HostSearchCreationOrder(HOST_CONTROL *hostcontrol, HOST *host)
{
	if (hostcontrol->TimeListUpdated == TRUE)
		HostUpdateTimeTable(hostcontrol);
    return host->TimeListPtr - hostcontrol->TimeTable + 1;
}


BOOLEAN HostUpdateTimeTable(HOST_CONTROL *hostcontrol)
{
LONG i;
HOST *p;

	for (p = hostcontrol->TimeList, i = 0; p != NULL; p = p->TimeNext, i++)
	{
		hostcontrol->TimeTable[i] = p;
		p->TimeListPtr = hostcontrol->TimeTable+i;
	}
	
	if (i != hostcontrol->TableSize)
	{
		DnpapMessage(DMC_FATAL, HOST_UPDTIME, "hostcontrol: internal error in time list and table");
		DnpapExit(HOST_UPDTIME);
	}
	
	hostcontrol->TimeListUpdated = FALSE;
	
	return TRUE;
}


#ifdef DEBUG

VOID HostCheckListsConsistency(HOST_CONTROL *hostcontrol)
{
HOST *p;
LONG count;

    for (p = hostcontrol->LexList, count = 0; p != NULL; p = p->LexNext, count++)
        if (HashSearch(hostcontrol->Table, p->Address, HOST_SZEADDR) != p)
            DnpapMessage(DMC_FATAL, HOST_ERROR, "hostcontrol: internal error in lexical list");
    if (count != hostcontrol->TableSize)
        DnpapMessage(DMC_FATAL, HOST_ERROR, "hostcontrol: wrong number of elements in lexical list");

    for (p = hostcontrol->TimeList, count = 0; p != NULL; p = p->TimeNext, count++)
        if (HashSearch(hostcontrol->Table, p->Address, HOST_SZEADDR) != p)
            DnpapMessage(DMC_FATAL, HOST_ERROR, "hostcontrol: internal error in time list");
    if (count != hostcontrol->TableSize)
        DnpapMessage(DMC_FATAL, HOST_ERROR, "hostcontrol: wrong number of elements in time list");

    for (p = hostcontrol->LRUList, count = 0; p != NULL; p = p->LRUNext, count++)
        if (HashSearch(hostcontrol->Table, p->Address, HOST_SZEADDR) != p)
            DnpapMessage(DMC_FATAL, HOST_ERROR, "hostcontrol: internal error in LRU list");
    if (count != hostcontrol->TableSize)
        DnpapMessage(DMC_FATAL, HOST_ERROR, "hostcontrol: wrong number of elements in time list");

    if (hostcontrol->TimeListUpdated == TRUE)
        HostUpdateTimeTable(hostcontrol);

    for (p = hostcontrol->TimeList, count = 0; p != NULL; p = p->TimeNext, count++)
        if (hostcontrol->TimeTable[count] != p || p->TimeListPtr != hostcontrol->TimeTable+count)
            DnpapMessage(DMC_FATAL, HOST_ERROR, "hostcontrol: internal error in time table");
}


VOID HostCheckListsAdd(HOST_CONTROL *hostcontrol, HOST *host)
{
HOST *p;

    if (HashSearch(hostcontrol->Table, host->Address, HOST_SZEADDR) != NULL)
        DnpapMessage(DMC_FATAL, HOST_ERROR, "host already in hash table");

    for (p = hostcontrol->LexList; p != NULL; p = p->LexNext)
    {
        if (memcmp(p->Address, host->Address, HOST_SZEADDR) == 0)
            DnpapMessage(DMC_FATAL, HOST_ERROR, "host already in lexical list");
    }
    for (p = hostcontrol->TimeList; p != NULL; p = p->TimeNext)
    {
        if (memcmp(p->Address, host->Address, HOST_SZEADDR) == 0)
            DnpapMessage(DMC_FATAL, HOST_ERROR, "host already in time list");
    }
    for (p = hostcontrol->LRUList; p != NULL; p = p->LRUNext)
    {
        if (memcmp(p->Address, host->Address, HOST_SZEADDR) == 0)
            DnpapMessage(DMC_FATAL, HOST_ERROR, "host already in LRU list");
    }
}


VOID HostCheckListsAdd2(HOST_CONTROL *hostcontrol, HOST *host)
{
HOST *p;

    if (HashSearch(hostcontrol->Table, host->Address, HOST_SZEADDR) == NULL)
        DnpapMessage(DMC_FATAL, HOST_ERROR, "host not added to hash table");

    for (p = hostcontrol->LexList; p != NULL; p = p->LexNext)
    {
        if (memcmp(p->Address, host->Address, HOST_SZEADDR) == 0)
            break;
    }             
    if (p == NULL)
        DnpapMessage(DMC_FATAL, HOST_ERROR, "host not added to lexical list");
    for (p = hostcontrol->TimeList; p != NULL; p = p->TimeNext)
    {
        if (memcmp(p->Address, host->Address, HOST_SZEADDR) == 0)
            break;
    }             
    if (p == NULL)
        DnpapMessage(DMC_FATAL, HOST_ERROR, "host not added to time list");
    for (p = hostcontrol->LRUList; p != NULL; p = p->LRUNext)
    {
        if (memcmp(p->Address, host->Address, HOST_SZEADDR) == 0)
            break;
    }   
    if (p == NULL)
        DnpapMessage(DMC_FATAL, HOST_ERROR, "host not added to LRU list");
}


VOID HostCheckListsRemove(HOST_CONTROL *hostcontrol, HOST *host)
{
HOST *p;

    if (HashSearch(hostcontrol->Table, host->Address, HOST_SZEADDR) != NULL)
        DnpapMessage(DMC_FATAL, HOST_ERROR, "host that was just removed can still be retrieved");

    for (p = hostcontrol->LexList; p != NULL; p = p->LexNext)
    {
        if (memcmp(p->Address, host->Address, HOST_SZEADDR) == 0)
            DnpapMessage(DMC_FATAL, HOST_ERROR, "host that was just removed can still be retrieved from lexical list");
    }
    for (p = hostcontrol->TimeList; p != NULL; p = p->TimeNext)
    {
        if (memcmp(p->Address, host->Address, HOST_SZEADDR) == 0)
            DnpapMessage(DMC_FATAL, HOST_ERROR, "host that was just removed can still be retrieved from time list");
    }
    for (p = hostcontrol->LRUList; p != NULL; p = p->LRUNext)
    {
        if (memcmp(p->Address, host->Address, HOST_SZEADDR) == 0)
            DnpapMessage(DMC_FATAL, HOST_ERROR, "host that was just removed can still be retrieved from LRU list");
    }
}


VOID HostCheckListsRemove2(HOST_CONTROL *hostcontrol, HOST *host)
{
HOST *p;

    if (HashSearch(hostcontrol->Table, host->Address, HOST_SZEADDR) == NULL)
        DnpapMessage(DMC_FATAL, HOST_ERROR, "host already removed");

    for (p = hostcontrol->LexList; p != NULL; p = p->LexNext)
    {
        if (memcmp(p->Address, host->Address, HOST_SZEADDR) == 0)
            break;
    }
    if (p == NULL)
        DnpapMessage(DMC_FATAL, HOST_ERROR, "host already removed from lexical list");
    for (p = hostcontrol->TimeList; p != NULL; p = p->TimeNext)
    {
        if (memcmp(p->Address, host->Address, HOST_SZEADDR) == 0)
            break;
    }
    if (p == NULL)
        DnpapMessage(DMC_FATAL, HOST_ERROR, "host already removed from time list");
    for (p = hostcontrol->LRUList; p != NULL; p = p->LRUNext)
    {
        if (memcmp(p->Address, host->Address, HOST_SZEADDR) == 0)
            break;
    }
    if (p == NULL)
        DnpapMessage(DMC_FATAL, HOST_ERROR, "host already removed from LRU list");
}

#endif

