/************************************************************************
** MODULE INFORMATION*
**********************
**     FILE     NAME:       matrix_c.c
**     SYSTEM   NAME:       beholder
**     ORIGINAL AUTHOR(S):  Richard Kooijman
**     VERSION  NUMBER:     0.99
**     CREATION DATE:       1992/9/29
**
** DESCRIPTION: matrix group of the RMON MIB: collector
**              
*************************************************************************
** CHANGES INFORMATION **
*************************
** REVISION:    $Revision$
** WORKFILE:    $Workfile$
** LOGINFO:     $Log$
*************************************************************************/
#if ! defined(PRD)
static char _pvcs_hdr[] =
"$Header$";
#endif

#include <stdlib.h>
#include <memory.h>
#include <dnpap.h>
#include <config.h>
#include <message.h>
#include <mac.h>
#include <hash.h>
#include <sys.h>
#include <protocol.h>

#include "matrix_d.h"
#include "matrix_e.h"
#include "matrix_c.h"


static WORD MatrixMaxNrSrcDsts = 8000;


static VOID MatrixCallback(MAC_COLL *collector, PROT_PKT *pkt);

static BOOLEAN MatrixAddSrcDstList(MATRIX_CONTROL *matrixcontrol, SRCDST *host);
static BOOLEAN MatrixRemoveSrcDstList(MATRIX_CONTROL *matrixcontrol, SRCDST *host);

static SRCDST* MatrixAddLRUList(MATRIX_CONTROL *matrixcontrol, SRCDST *srcdst);
static BOOLEAN MatrixUpdateLRUList(MATRIX_CONTROL *matrixcontrol, SRCDST *srcdst);
static BOOLEAN MatrixRemoveLRUList(MATRIX_CONTROL *matrixcontrol, SRCDST *srcdst);

static VOID DelSrcDsts(MATRIX_CONTROL *matrixcontrol);



BOOLEAN MatrixConfigInit(VOID)
{
	ConfigGetShort("beholder.matrix.maxnrsrcdsts", &MatrixMaxNrSrcDsts);

	if (MatrixMaxNrSrcDsts < 2)
    {
        DnpapMessage(DMC_WARNING, MATRIX_MAX, "matrixcontrol: beholder.matrix.maxnrsrcdsts < 2, setting it to 2");
        MatrixMaxNrSrcDsts = 2;
    }

	return TRUE;
}


BOOLEAN MatrixCInit(MATRIX_CONTROL *matrixcontrol)
{
LONG source[] = {1,3,6,1,2,1,2,2,1,1,1};

    memcpy(matrixcontrol->Source, source, sizeof(source));
    matrixcontrol->SourceLen = sizeof(source)/sizeof(source[0]);
    matrixcontrol->TableSize = 0;
    matrixcontrol->LastDeleteTime = 0;
    matrixcontrol->Owner[0] = '\0';
    matrixcontrol->OwnerLen = 0;
    matrixcontrol->Status = SNMP_INVALID;
    
    if ((matrixcontrol->Iface =
        MacIfaceGet((WORD) matrixcontrol->Source[matrixcontrol->SourceLen-1])) == NULL)
    {
        DnpapMessage(DMC_ERROR, MATRIX_NETINIT, "matrixcontrol: network initialisation failed");
        return (FALSE);
    }
	
    matrixcontrol->Table = NULL;
    matrixcontrol->SrcDstList = NULL;
    matrixcontrol->DstSrcList = NULL;

    matrixcontrol->LRUList = NULL;
    matrixcontrol->LRULast = NULL;

	return TRUE;
}


BOOLEAN MatrixCStart(MATRIX_CONTROL *matrixcontrol)
{
    matrixcontrol->Coll.Rcve       = MatrixCallback;
    matrixcontrol->Coll.specific   = matrixcontrol;
    if (!MacCollRegister(&(matrixcontrol->Coll)))
    {
        DnpapMessage(DMC_ERROR, MATRIX_NETERR, "matrixcontrol: network initialisation failed");
        return FALSE;
    }

    if ((matrixcontrol->Table = NewHash(5011, NULL)) == NULL)
    {
        DnpapMessage(DMC_ERROR, MATRIX_HASHERR, "matrixcontrol: can not create hashtable");
        return FALSE;
    }

    return TRUE;
}


BOOLEAN MatrixCStop(MATRIX_CONTROL *matrixcontrol)
{
    MacCollRemove(&(matrixcontrol->Coll));

    DelHash(matrixcontrol->Table);

    DelSrcDsts(matrixcontrol);

    return TRUE;
}


VOID MatrixCallback(MAC_COLL *collector, PROT_PKT *pkt)
{
MATRIX_CONTROL *matrixcontrol = collector->specific;
SRCDST *srcdst = NULL, *oldsrcdst = NULL;
BYTE broadcast[HOST_SZEADDR] = {0xff,0xff,0xff,0xff,0xff,0xff};
BYTE srcdstaddr[2*HOST_SZEADDR];
PROT_OBJ Interface = {0, {1,2}};
PROT_OBJ Size = {0, {1,4}};
PROT_OBJ Dst = {1, {2,1}};
PROT_OBJ Src = {1, {2,2}};
LWORD size;

    if (ProtGetField(pkt,&Interface) == TRUE &&
        Interface.Syntax.LngInt == matrixcontrol->Source[matrixcontrol->SourceLen-1])
    {
        if (ProtGetField(pkt,&Size) == TRUE &&
            ProtGetField(pkt,&Src) == TRUE &&
            ProtGetField(pkt,&Dst) == TRUE)
        {
            size = Size.Syntax.LngUns + 4L;
            if (Src.SyntaxLen != HOST_SZEADDR || Dst.SyntaxLen != HOST_SZEADDR)
            {
                DnpapMessage(DMC_ERROR, MATRIX_INVPACKET, "matrixcontrol: invalid packet");
                return;
            }

            memcpy(srcdstaddr, Src.Syntax.BufChr, HOST_SZEADDR);
            memcpy(srcdstaddr+HOST_SZEADDR, Dst.Syntax.BufChr, HOST_SZEADDR);
			
            if ((srcdst = HashSearch(matrixcontrol->Table, srcdstaddr, 2*HOST_SZEADDR)) == NULL)
            {
                /*  first try to add the new srcdst  */
                if ((srcdst = DnpapMalloc(sizeof(SRCDST))) != NULL)
                {
                    memset(srcdst, 0, sizeof(SRCDST));
                    memcpy(srcdst->SrcDst, srcdstaddr, 2*HOST_SZEADDR);
                    if (HashAdd(matrixcontrol->Table, srcdst->SrcDst, 2*HOST_SZEADDR, srcdst) == NULL)
                    {
                        DnpapMessage(DMC_WARNING, MATRIX_NADD, "matrixcontrol: srcdst could not be added to the hash table");
                        DnpapFree(srcdst);
                        srcdst = NULL;
                    }
                    else
                    {
                        MatrixAddSrcDstList(matrixcontrol, srcdst);
                        oldsrcdst = MatrixAddLRUList(matrixcontrol, srcdst);
                        matrixcontrol->TableSize++;

                        if (matrixcontrol->TableSize > (LONG)MatrixMaxNrSrcDsts)
                        {
                            MatrixRemoveLRUList(matrixcontrol, oldsrcdst);
                            MatrixRemoveSrcDstList(matrixcontrol, oldsrcdst);
                            HashRemove(matrixcontrol->Table, oldsrcdst->SrcDst, 2*HOST_SZEADDR);
                            DnpapFree(oldsrcdst);
                            matrixcontrol->LastDeleteTime = SysTime();
                            matrixcontrol->TableSize--;
                        }
                    }
                }
            }

            if (srcdst != NULL)
            {
                srcdst->Pkts++;
                srcdst->Octets += size;
                
                MatrixUpdateLRUList(matrixcontrol, srcdst);
            }
        }
    }

    return;
}


BOOLEAN MatrixAddSrcDstList(MATRIX_CONTROL *matrixcontrol, SRCDST *srcdst)
{
SRCDST *p, *q;
BYTE dstsrc1[2*HOST_SZEADDR], dstsrc2[2*HOST_SZEADDR];

    if (matrixcontrol->SrcDstList == NULL && matrixcontrol->DstSrcList == NULL)
    {
        matrixcontrol->SrcDstList = srcdst;
        srcdst->SrcDstPrev = NULL;
        srcdst->SrcDstNext = NULL;
        
        matrixcontrol->DstSrcList = srcdst;
        srcdst->DstSrcPrev = NULL;
        srcdst->DstSrcNext = NULL;
        
        return TRUE;
    }


    /*  update SrcDstList  */
    for (p = matrixcontrol->SrcDstList, q = NULL; p != NULL; q = p, p = p->SrcDstNext)
    {
        if (memcmp(p->SrcDst, srcdst->SrcDst, 2*HOST_SZEADDR) >= 0)
            break;
    }

    if (q != NULL)
        q->SrcDstNext = srcdst;
    else
        matrixcontrol->SrcDstList = srcdst;

    if (p != NULL)
        p->SrcDstPrev = srcdst;

    srcdst->SrcDstPrev = q;
    srcdst->SrcDstNext = p;


    /*  update DstSrcList  */
    memcpy(dstsrc2, srcdst->SrcDst+HOST_SZEADDR, HOST_SZEADDR);
    memcpy(dstsrc2+HOST_SZEADDR, srcdst->SrcDst, HOST_SZEADDR);
    for (p = matrixcontrol->DstSrcList, q = NULL; p != NULL; q = p, p = p->DstSrcNext)
    {
    	memcpy(dstsrc1, p->SrcDst+HOST_SZEADDR, HOST_SZEADDR);
    	memcpy(dstsrc1+HOST_SZEADDR, p->SrcDst, HOST_SZEADDR);
        if (memcmp(dstsrc1, dstsrc2, 2*HOST_SZEADDR) >= 0)
            break;
    }

    if (q != NULL)
        q->DstSrcNext = srcdst;
    else
        matrixcontrol->DstSrcList = srcdst;

    if (p != NULL)
        p->DstSrcPrev = srcdst;

    srcdst->DstSrcPrev = q;
    srcdst->DstSrcNext = p;
        

    return TRUE;
}


SRCDST* MatrixAddLRUList(MATRIX_CONTROL *matrixcontrol, SRCDST *srcdst)
{
    if (matrixcontrol->LRUList == NULL)
    {
        matrixcontrol->LRUList = srcdst;
        matrixcontrol->LRULast = srcdst;
        srcdst->LRUPrev = NULL;
        srcdst->LRUNext = NULL;
        
        return NULL;
    }

    srcdst->LRUNext = matrixcontrol->LRUList;
    srcdst->LRUPrev = NULL;
    matrixcontrol->LRUList->LRUPrev = srcdst;
    matrixcontrol->LRUList = srcdst;

    return matrixcontrol->LRULast;
}


BOOLEAN MatrixUpdateLRUList(MATRIX_CONTROL *matrixcontrol, SRCDST *srcdst)
{
    if (matrixcontrol->TableSize > 1)
    {          
        MatrixRemoveLRUList(matrixcontrol, srcdst);
        MatrixAddLRUList(matrixcontrol, srcdst);
    }
    return TRUE;
}


BOOLEAN MatrixRemoveSrcDstList(MATRIX_CONTROL *matrixcontrol, SRCDST *srcdst)
{
    if (srcdst->SrcDstPrev != NULL)
        srcdst->SrcDstPrev->SrcDstNext = srcdst->SrcDstNext;
    else
        matrixcontrol->SrcDstList = srcdst->SrcDstNext;
    if (srcdst->SrcDstNext != NULL)
        srcdst->SrcDstNext->SrcDstPrev = srcdst->SrcDstPrev;

    if (srcdst->DstSrcPrev != NULL)
        srcdst->DstSrcPrev->DstSrcNext = srcdst->DstSrcNext;
    else
        matrixcontrol->DstSrcList = srcdst->DstSrcNext;
    if (srcdst->DstSrcNext != NULL)
        srcdst->DstSrcNext->DstSrcPrev = srcdst->DstSrcPrev;

    return TRUE;
}


BOOLEAN MatrixRemoveLRUList(MATRIX_CONTROL *matrixcontrol, SRCDST *srcdst)
{
    if (srcdst->LRUPrev != NULL)
        srcdst->LRUPrev->LRUNext = srcdst->LRUNext;
    else
        matrixcontrol->LRUList = srcdst->LRUNext;
    if (srcdst->LRUNext != NULL)
        srcdst->LRUNext->LRUPrev = srcdst->LRUPrev;
    else
        matrixcontrol->LRULast = srcdst->LRUPrev;

    return TRUE;
}


VOID DelSrcDsts(MATRIX_CONTROL *matrixcontrol)
{
SRCDST *srcdst1, *srcdst2;

    for (srcdst2 = matrixcontrol->SrcDstList; srcdst2 != NULL; srcdst2 = srcdst1)
    {
        srcdst1 = srcdst2->SrcDstNext;
        DnpapFree(srcdst2);
        matrixcontrol->TableSize--;
    }
    matrixcontrol->SrcDstList = NULL;
    matrixcontrol->DstSrcList = NULL;
}


SRCDST* MatrixHashSearch(MATRIX_CONTROL *matrixcontrol, BYTE *srcdst, WORD len, BOOLEAN SDorder)
{
SRCDST* p;
BYTE srcdst2[2*HOST_SZEADDR], dstsrc[2*HOST_SZEADDR];

	if (SDorder == TRUE)
	{
	    if (len == 2*HOST_SZEADDR &&
	        (p = HashSearch(matrixcontrol->Table, srcdst, 2*HOST_SZEADDR)) != NULL)
	        return p;
	    else
	    {
	        for (p = matrixcontrol->SrcDstList; p != NULL; p = p->SrcDstNext)
	        {
	            if (memcmp(p->SrcDst, srcdst, len) >= 0)
	                break;
	        }
	        return p;
	    }
	    return NULL;
    }
    else
    {   /*  srcdst contains destination and source in that order, so switch them  */
    	memcpy(srcdst2, srcdst+HOST_SZEADDR, HOST_SZEADDR);
    	memcpy(srcdst2+HOST_SZEADDR, srcdst, HOST_SZEADDR);
	    if (len == 2*HOST_SZEADDR &&
	        (p = HashSearch(matrixcontrol->Table, srcdst2, 2*HOST_SZEADDR)) != NULL)
	        return p;
	    else
	    {
	        for (p = matrixcontrol->DstSrcList; p != NULL; p = p->DstSrcNext)
	        {
                /*  switch source and destination of the srcdst(p) under examination  */
		    	memcpy(dstsrc, p->SrcDst+HOST_SZEADDR, HOST_SZEADDR);
		    	memcpy(dstsrc+HOST_SZEADDR, p->SrcDst, HOST_SZEADDR);
		    
                /*  compare destination and source ordered 'addresses'  */
	            if (memcmp(dstsrc, srcdst, len) >= 0)
	                break;
	        }
	        return p;
	    }
	    return NULL;
	}
}


SRCDST* MatrixHashSearch2(MATRIX_CONTROL *matrixcontrol, SNMP_OBJECT *obj, WORD idlen, BOOLEAN SDorder)
{
WORD i;
BYTE srcdst[2*HOST_SZEADDR];

	if (SDorder == TRUE)
	{
	    for (i = 0; i < 2*HOST_SZEADDR; i++)
	        srcdst[i] = (BYTE)obj->Id[idlen+1+i];
	    return HashSearch(matrixcontrol->Table, srcdst, 2*HOST_SZEADDR);
    }
    else
    {
	    for (i = 0; i < HOST_SZEADDR; i++)
        {
	        srcdst[i] = (BYTE)obj->Id[idlen+1+HOST_SZEADDR+i];
	        srcdst[HOST_SZEADDR+i] = (BYTE)obj->Id[idlen+1+i];
        }
	    return HashSearch(matrixcontrol->Table, srcdst, 2*HOST_SZEADDR);
    }
}
