#include <dnpap.h>
#include <config.h>
#include <mib.h>
#include <mibsup.h>
#include <mibrqs.h>
#include <agent.h>
#include <trap.h>
#include <stdlib.h>
#include <string.h>
#include <timer.h>

#include "neuro_d.h"
#include "neuro_e.h"
#include "neuro_k.h"
#include "neuro_c.h"
#include "neuro_m.h"

			

static char MODULE[] =	"NeuroMIB";



static MIB_LOCAL*	NeuroDataInstance = NULL;

static WORD TotalNrNetworks = 0, RunningNrNetworks = 0;


static WORD         NeuroSetId(MIB_OBJECT* obj, WORD idc, WORD indexsize, LONG c, LONG l, LONG n, LONG i);

static MIB_LOCAL*	NeuroMibInsert(MIB_OBJECT* obj, MIB_LOCAL** local, WORD idlen, WORD indexsize);
static BOOLEAN		NeuroMibRemove(MIB_OBJECT* obj, MIB_LOCAL** local, WORD idlen, WORD indexsize);

static MIB_LOCAL*	NeuroMibRmon(MIB_OBJECT* obj, MIB_LOCAL* local, WORD idlen, WORD indexsize);
static BOOLEAN		NeuroMibNext(MIB_OBJECT* obj, MIB_LOCAL** local, WORD idlen, WORD indexsize);  
				
static WORD         MibName2ID(LONG* id, CHAR* name, WORD* n);
static WORD         MibID2String(const LONG* id, WORD nr, CHAR* name, WORD maxlen);

static BOOLEAN		NeuroMib2Neuron(NeuroData* data, InOutData* mibinout, LONG n, InOutput* inout);
static BOOLEAN		NeuroNeuron2Mib(InOutput* inout, LONG n, FLOAT* mibinout);

static FLOAT        NeuroMean(FLOAT value[], LONG nrvalues);
static FLOAT        NeuroHarmonic(FLOAT value[], LONG nrvalues);

static VOID         NeuroMCalcCallback(TIMER_DESCR *timer, ULONG now, VOID *param);

static BOOLEAN      NeuroCheckThresholds(NeuroData* data);


typedef struct MIB_PROFIL_
{
    BYTE*   Comm;
    WORD    CommLen;
} MIB_PROFIL;


/*  config data  */	     
#define MAXCONFIGNAME       81
#define MAXDATASRCNAME	    81

static SHORT		NrNetworks = 0;
				   


BOOLEAN NeuroMInit(int argc, char* argv[])
{
SHORT NrLayers, NrNeurons, DefaultNrLayers,
      DataSourceSteps, DataSourceAverageMethod, DataSourceTrap,
      DataSourceThresholdCheck;
DOUBLE LearningRate, Momentum, DefaultLearningRate, DefaultMomentum, 
       DataSourceValue, DataSourceThresholdValue;
MIB_PROFIL prf = { MODULE, sizeof(MODULE) };
MIB_OBJECT obj = { MIB_RQSSET, { NMIB_BASE }, NMIB_BASELEN, MIB_INTEGER, MIB_VALID };
CHAR configname[MAXCONFIGNAME], *DataSourceName;
SHORT c, l, n, i;

    MessageInit("Neuro", MSGFILE, MSGSTDIO, NULL);
	MessageConfig(NEURO_ERROFFSET, "Neuro");
	
	if (NeuroErrorInit() == FALSE)
		FATAL(MODULE, NEURO_ERRINIT);
		
    if (ConfigInit(argc, argv) == FALSE || ConfigLoad() == FALSE)
        ERROR(MODULE, NEURO_CONFIGINIT);

	if (ConfigGetShort("NeuroNrNetworks", &NrNetworks) == TRUE)
	{
		ConfigGetShort("DefaultNrLayers", &DefaultNrLayers);
		ConfigGetDouble("DefaultLearningRate", &DefaultLearningRate);
		ConfigGetDouble("DefaultMomentum", &DefaultMomentum);
		
		for (c = 1; c <= NrNetworks; c++)
		{
			/*  set Status to MIB_CREATEREQUEST  */
			obj.Type = MIB_INTEGER;
			obj.Syntax.LngInt = MIB_CREATEREQUEST;
			NeuroSetId(&obj, NMIB_STATUS, 1, c, 0, 0, 0);
			if (MibRequest(prf.Comm, prf.CommLen, &obj) != MIB_NOERROR)
				WARNING(MODULE, NEURO_MCREATE);
				
			/*  set Owner to MODULE  */
			obj.Type = MIB_OCTETSTR;
			memcpy(obj.Syntax.BufChr, MODULE, sizeof(MODULE));
			obj.SyntaxLen = sizeof(MODULE);
			NeuroSetId(&obj, NMIB_OWNER, 1, c, 0, 0, 0);
			if (MibRequest(prf.Comm, prf.CommLen, &obj) != MIB_NOERROR)
				WARNING(MODULE, NEURO_MOWNER);

			/*  set NrLayers to config value or default  */				
			obj.Type = MIB_INTEGER;
			sprintf(configname, "NrLayers[%d]", c);
			if (ConfigGetShort(configname, &NrLayers) == TRUE)
				obj.Syntax.LngInt = NrLayers;
			else
				obj.Syntax.LngInt = DefaultNrLayers;
			NeuroSetId(&obj, NMIB_NRLAYERS, 1, c, 0, 0, 0);
			if (MibRequest(prf.Comm, prf.CommLen, &obj) != MIB_NOERROR)
				WARNING(MODULE, NEURO_MNRLAYERS);
				
			/*  set LearningRate to config value or default  */				
			obj.Type = MIB_INTEGER;
			sprintf(configname, "LearningRate[%d]", c);
			if (ConfigGetDouble(configname, &LearningRate) == TRUE)
				obj.Syntax.LngInt = (LONG) FLOAT_SCALE*LearningRate;
			else
				obj.Syntax.LngInt = (LONG) FLOAT_SCALE*DefaultLearningRate;
			NeuroSetId(&obj, NMIB_LRATE, 1, c, 0, 0, 0);
			if (MibRequest(prf.Comm, prf.CommLen, &obj) != MIB_NOERROR)
				WARNING(MODULE, NEURO_MLRATE); 
				
			/*  set Momentum to config value or default  */				
			obj.Type = MIB_INTEGER;
			sprintf(configname, "Momentum[%d]", c);
			if (ConfigGetDouble(configname, &Momentum) == TRUE)
				obj.Syntax.LngInt = (LONG) FLOAT_SCALE*Momentum;
			else
				obj.Syntax.LngInt = (LONG) FLOAT_SCALE*DefaultMomentum;
			NeuroSetId(&obj, NMIB_MOMENTUM, 1, c, 0, 0, 0);
			if (MibRequest(prf.Comm, prf.CommLen, &obj) != MIB_NOERROR)
				WARNING(MODULE, NEURO_MMOMENTUM);
				
			for (l = 1; l <= NrLayers; l++)
			{
				obj.Type = MIB_INTEGER;
				sprintf(configname, "NrNeurons[%d,%d]", c, l);
				if (ConfigGetShort(configname, &NrNeurons) == TRUE)
					obj.Syntax.LngInt = NrNeurons;
				else					
				{
					ERROR2(MODULE, NEURO_MNRNEURONS, c, l);
					return FALSE;
				}
				NeuroSetId(&obj, NMIB_NRNEURONS, 2, c, l, 0, 0);
				if (MibRequest(prf.Comm, prf.CommLen, &obj) != MIB_NOERROR)
					ERROR(MODULE, NEURO_MSETNEURONS);
								
				if (l == 1 || l == NrLayers)
				{
					for (n = 1; n <= NrNeurons; n++)
					{	       
						obj.Type = MIB_OCTETSTR;
						sprintf(configname, "NeuroDataSource[%d,%d,%d]", c, l, n);
						if (ConfigGetString(configname, &DataSourceName) == TRUE)
							MibName2ID(obj.Syntax.BufInt, DataSourceName, &obj.SyntaxLen);
						else
						{
							ERROR3(MODULE, NEURO_MDATASRCNAME, c, l, n);
							return FALSE;
						}
						NeuroSetId(&obj, NMIB_DATASRC, 3, c, l, n, 0);
						if (MibRequest(prf.Comm, prf.CommLen, &obj) != MIB_NOERROR)
							ERROR(MODULE, NEURO_MSETDATASRC);
						
						obj.Type = MIB_INTEGER;
						sprintf(configname, "NeuroDataSourceMax[%d,%d,%d]", c, l, n);
						if (ConfigGetDouble(configname, &DataSourceValue) == TRUE)
							obj.Syntax.LngInt = (LONG) FLOAT_SCALE*DataSourceValue;
						else
						{
							ERROR3(MODULE, NEURO_MDATASRCMAX, c, l, n);
							return FALSE;
						}		     
						NeuroSetId(&obj, NMIB_DATASRCMAX, 3, c, l, n, 0);
						if (MibRequest(prf.Comm, prf.CommLen, &obj) != MIB_NOERROR)
							ERROR(MODULE, NEURO_MSETDATASRCMAX);
						
						obj.Type = MIB_INTEGER;
						sprintf(configname, "NeuroDataSourceMin[%d,%d,%d]", c, l, n);
						if (ConfigGetDouble(configname, &DataSourceValue) == TRUE)
							obj.Syntax.LngInt = (LONG) FLOAT_SCALE*DataSourceValue;
						else
						{
							ERROR3(MODULE, NEURO_MDATASRCMIN, c, l, n);
							return FALSE;
						}		     
						NeuroSetId(&obj, NMIB_DATASRCMIN, 3, c, l, n, 0);
						if (MibRequest(prf.Comm, prf.CommLen, &obj) != MIB_NOERROR)
							ERROR(MODULE, NEURO_MSETDATASRCMIN);
											       
						obj.Type = MIB_INTEGER;
						sprintf(configname, "NeuroDataSourceAverageMethod[%d,%d,%d]", c, l, n);
						if (ConfigGetShort(configname, &DataSourceAverageMethod) == TRUE)
							obj.Syntax.LngInt = DataSourceAverageMethod;
						else
						{
							ERROR3(MODULE, NEURO_MDATASRCAVERAGEMETHOD, c, l, n);
							return FALSE;
						}		     
						NeuroSetId(&obj, NMIB_DATASRCAVERAGEMETHOD, 3, c, l, n, 0);
						if (MibRequest(prf.Comm, prf.CommLen, &obj) != MIB_NOERROR)
							ERROR(MODULE, NEURO_MSETDATASRCAVERAGEMETHOD);
							
						obj.Type = MIB_INTEGER;
						sprintf(configname, "NeuroDataSourceAverageSteps[%d,%d,%d]", c, l, n);
						if (ConfigGetShort(configname, &DataSourceSteps) == TRUE)
							obj.Syntax.LngInt = DataSourceSteps;
						else
						{
							ERROR3(MODULE, NEURO_MDATASRCSTEPS, c, l, n);
							return FALSE;
						}		     
						NeuroSetId(&obj, NMIB_DATASRCSTEPS, 3, c, l, n, 0);
						if (MibRequest(prf.Comm, prf.CommLen, &obj) != MIB_NOERROR)
							ERROR(MODULE, NEURO_MSETDATASRCSTEPS);
							
						obj.Type = MIB_INTEGER;
						sprintf(configname, "NeuroDataSourceTrap[%d,%d,%d]", c, l, n);
						if (ConfigGetShort(configname, &DataSourceTrap) == TRUE)
							obj.Syntax.LngInt = DataSourceTrap;
						else
						{
							ERROR3(MODULE, NEURO_MDATASRCTRAP, c, l, n);
							return FALSE;
						}		     
						NeuroSetId(&obj, NMIB_DATASRCTRAP, 3, c, l, n, 0);
						if (MibRequest(prf.Comm, prf.CommLen, &obj) != MIB_NOERROR)
							ERROR(MODULE, NEURO_MSETDATASRCTRAP);
							
						obj.Type = MIB_INTEGER;
						sprintf(configname, "NeuroDataSourceThresholdCheck[%d,%d,%d]", c, l, n);
						if (ConfigGetShort(configname, &DataSourceThresholdCheck) == TRUE)
							obj.Syntax.LngInt = DataSourceThresholdCheck;
						else
						{
							ERROR3(MODULE, NEURO_MDATASRCCHECK, c, l, n);
							return FALSE;
						}		     
						NeuroSetId(&obj, NMIB_DATASRCTHRESHOLDCHECK, 3, c, l, n, 0);
						if (MibRequest(prf.Comm, prf.CommLen, &obj) != MIB_NOERROR)
							ERROR(MODULE, NEURO_MSETDATASRCCHECK);
							
						for (i = 0; i < MAXNRTHRESHOLDVALUES; i++)
						{
							obj.Type = MIB_INTEGER;
							sprintf(configname, "NeuroDataSourceThresholdValue[%d,%d,%d,%d]", c, l, n, i);
							if (ConfigGetDouble(configname, &DataSourceThresholdValue) == TRUE)
								obj.Syntax.LngInt = (LONG) FLOAT_SCALE*DataSourceThresholdValue;
							else
							{
								ERROR4(MODULE, NEURO_MDATASRCTHRESHOLDVALUE, c, l, n, i);
								return FALSE;
							}		     
							NeuroSetId(&obj, NMIB_DATASRCTHRESHOLDVALUE, 4, c, l, n, i);
							if (MibRequest(prf.Comm, prf.CommLen, &obj) != MIB_NOERROR)
								ERROR(MODULE, NEURO_MSETDATASRCTHRESHOLDVALUE);
						}
					}
				}
			}
		}
	}
	
	return TRUE;
}


BOOLEAN NeuroMStart(VOID)
{	 
MIB_PROFIL prf = { MODULE, sizeof(MODULE) };
MIB_OBJECT obj = { MIB_RQSSET, { NMIB_BASE }, NMIB_BASELEN, MIB_INTEGER, MIB_VALID };
SHORT c;

	for (c = 1; c <= NrNetworks; c++)
	{
		NeuroSetId(&obj, NMIB_STATUS, 1, c, 0, 0, 0);
		if (MibRequest(prf.Comm, prf.CommLen, &obj) != MIB_NOERROR)
			WARNING(MODULE, NEURO_MSTART);
	}
    return TRUE;
}


BOOLEAN NeuroMStop(VOID)
{	 
MIB_PROFIL prf = { MODULE, sizeof(MODULE) };
MIB_OBJECT obj = { MIB_RQSSET, { NMIB_BASE }, NMIB_BASELEN, MIB_INTEGER, MIB_INVALID };
SHORT c;

	for (c = 1; c <= NrNetworks; c++)
	{
		NeuroSetId(&obj, NMIB_STATUS, 1, c, 0, 0, 0);
		if (MibRequest(prf.Comm, prf.CommLen, &obj) != MIB_NOERROR)
			WARNING(MODULE, NEURO_MSTOP);
	}
    return TRUE;
}


WORD NeuroSetId(MIB_OBJECT* obj, WORD idc, WORD indexsize, LONG c, LONG l, LONG n, LONG i)
{
LONG baseindex[] = { NMIB_BASE, 0, 0, 0 };
			      
	switch (idc)
	{
    case NMIB_NRNETWORKS:
		NEUROSETID(baseindex, NMIB_NRNETWORKSID);
        break;
    case NMIB_INDEX:
		NEUROSETID(baseindex, NMIB_INDEXID);
        break;
    case NMIB_NRLAYERS:
		NEUROSETID(baseindex, NMIB_NRLAYERSID);
        break;
    case NMIB_NETREFRESH:
		NEUROSETID(baseindex, NMIB_NETREFRESHID);
        break;
    case NMIB_REQCOMM: 
		NEUROSETID(baseindex, NMIB_REQCOMMID);
        break;
    case NMIB_TRAIN:
		NEUROSETID(baseindex, NMIB_TRAINID);
        break;
    case NMIB_LRATE:
		NEUROSETID(baseindex, NMIB_LRATEID);
        break;
    case NMIB_MOMENTUM:
		NEUROSETID(baseindex, NMIB_MOMENTUMID);
        break;
    case NMIB_TOTALSTEPS:
		NEUROSETID(baseindex, NMIB_TOTALSTEPSID);
        break;
    case NMIB_TRAININGSTEPS: 
		NEUROSETID(baseindex, NMIB_TRAININGSTEPSID);
        break;
    case NMIB_OWNER:
		NEUROSETID(baseindex, NMIB_OWNERID);
        break;
    case NMIB_STATUS: 
		NEUROSETID(baseindex, NMIB_STATUSID);
        break;
    case NMIB_NRNEURONS: 
		NEUROSETID(baseindex, NMIB_NRNEURONSID);
        break;
    case NMIB_DATASRC:
		NEUROSETID(baseindex, NMIB_DATASRCID);
        break;
    case NMIB_DATASRCMAX:
		NEUROSETID(baseindex, NMIB_DATASRCMAXID);
        break;
    case NMIB_DATASRCMIN:
		NEUROSETID(baseindex, NMIB_DATASRCMINID);
        break;
    case NMIB_DATASRCAVERAGEMETHOD:
		NEUROSETID(baseindex, NMIB_DATASRCAVERAGEMETHODID);
        break;
    case NMIB_DATASRCSTEPS: 
		NEUROSETID(baseindex, NMIB_DATASRCSTEPSID);
        break;
    case NMIB_DATASRCTRAP:
		NEUROSETID(baseindex, NMIB_DATASRCTRAPID);
        break;
    case NMIB_DATASRCTRAPCOMM:
		NEUROSETID(baseindex, NMIB_DATASRCTRAPCOMMID);
        break;
    case NMIB_DATASRCTHRESHOLDCHECK:
		NEUROSETID(baseindex, NMIB_DATASRCTHRESHOLDCHECKID);
        break;
    case NMIB_DATASRCVALUE:
		NEUROSETID(baseindex, NMIB_DATASRCVALUEID);
        break;
    case NMIB_OUTPUTVALUE:
		NEUROSETID(baseindex, NMIB_OUTPUTVALUEID);
        break;
    case NMIB_DATASRCTHRESHOLDVALUE:
		NEUROSETID(baseindex, NMIB_DATASRCTHRESHOLDVALUEID);
        break;
	}							       
	
	memcpy(obj->Id, baseindex, (obj->IdLen = sizeof(baseindex))*sizeof(baseindex[0]));
    while (obj->Id[obj->IdLen-1] == 0)
        obj->IdLen--;
	
	switch (indexsize)
	{
	case NMIB_IOFFSET:				     
		if (i > 0)
			obj->Id[obj->IdLen-1+NMIB_IOFFSET] = i;
	case NMIB_NOFFSET:
		if (n > 0)
			obj->Id[obj->IdLen-1+NMIB_NOFFSET] = n;
	case NMIB_LOFFSET:
		if (l > 0)
			obj->Id[obj->IdLen-1+NMIB_LOFFSET] = l;
	case NMIB_COFFSET:
		if (c > 0)
			obj->Id[obj->IdLen-1+NMIB_COFFSET] = c;
    default:
        break;
	}	
	return MIB_NOERROR;
}


WORD MibName2ID(LONG* id, CHAR* name, WORD* n)
{
LONG tmp;
CHAR *p;
CHAR name2[10*MIB_SZEBUFINT];

		if (name == NULL)
			return MIB_NOSUCHNAME;
			
		strcpy(name2, name);
		
        *n = 0;

        p = strtok(name2, ".");
        while (p != NULL)
        {
                if (*n < MIB_SZEBUFINT && (tmp = atoi(p)) != 0)
                        id[*n++] = tmp;
                else
                        return MIB_NOSUCHNAME;
                p = strtok(NULL, ".");
        }

        return MIB_NOERROR;
}


WORD MibID2String(const LONG* id, WORD nr, CHAR* name, WORD maxlen)
{
WORD ni, i, n;
CHAR tmp[20];	/*  ample space for 1 number and a '.'  */

	ni = 0;
	name[0] = '\0';

	sprintf(tmp, "%ld%n", id[0], &n);	/*  initialize name  */
	if (ni+n > maxlen)
		return 0;
	strcpy(name+ni, tmp);
	ni += n;

	for (i = 1; i < nr; i++)
	{
		sprintf(tmp, ".%ld%n", id[i], &n);
		if (ni+n > maxlen)
			return 0;
		strcpy(name+ni, tmp);
		ni += n;
	}

	return(ni);
}


WORD NeuroIndex(MIB_OBJECT* obj, WORD idlen)
{
MIB_LOCAL* local = NULL;

	if ((local = NeuroMibRmon(obj, NeuroDataInstance, idlen, 1)) == NULL)
		return MIB_NOSUCHNAME;
		
	switch (obj->Request)
	{
	case MIB_RQSNXT:
		if (NeuroMibNext(obj, &local, idlen, 1) == FALSE)
			return MIB_NOSUCHNAME;
	case MIB_RQSGET:
		obj->Syntax.LngInt = local->Index;
		return MIB_NOERROR;
	case MIB_RQSSET:
		return MIB_READONLY;
	}
}


WORD NeuroLayerIndex(MIB_OBJECT* obj, WORD idlen)
{
MIB_LOCAL* local = NULL;
			
	if ((local = NeuroMibRmon(obj, NeuroDataInstance, idlen, 2)) == NULL)
		return MIB_NOSUCHNAME;
		
	switch (obj->Request)
	{
	case MIB_RQSNXT:
		if (NeuroMibNext(obj, &local, idlen, 2) == FALSE)
			return MIB_NOSUCHNAME;
	case MIB_RQSGET:
		obj->Syntax.LngInt = local->Index;
		return MIB_NOERROR;
	case MIB_RQSSET:
		return MIB_READONLY;
	}
}


WORD NeuroLayerIndexLayer(MIB_OBJECT* obj, WORD idlen)
{
MIB_LOCAL* local = NULL;
			
	if ((local = NeuroMibRmon(obj, NeuroDataInstance, idlen, 2)) == NULL)
		return MIB_NOSUCHNAME;
		
	switch (obj->Request)
	{
	case MIB_RQSNXT:
		if (NeuroMibNext(obj, &local, idlen, 2) == FALSE)
			return MIB_NOSUCHNAME;
	case MIB_RQSGET:
		obj->Syntax.LngInt = obj->Id[idlen-1+NMIB_LOFFSET];
		return MIB_NOERROR;
	case MIB_RQSSET:
		return MIB_READONLY;
	}
}


WORD NeuroLayerNeuronIndex(MIB_OBJECT* obj, WORD idlen)
{
MIB_LOCAL* local = NULL;
			
	if ((local = NeuroMibRmon(obj, NeuroDataInstance, idlen, 3)) == NULL)
		return MIB_NOSUCHNAME;
		
	switch (obj->Request)
	{
	case MIB_RQSNXT:
		if (NeuroMibNext(obj, &local, idlen, 3) == FALSE)
			return MIB_NOSUCHNAME;
	case MIB_RQSGET:
		obj->Syntax.LngInt = local->Index;
		return MIB_NOERROR;
	case MIB_RQSSET:
		return MIB_READONLY;
	}
}


WORD NeuroLayerNeuronIndexLayer(MIB_OBJECT* obj, WORD idlen)
{
MIB_LOCAL* local = NULL;
			
	if ((local = NeuroMibRmon(obj, NeuroDataInstance, idlen, 3)) == NULL)
		return MIB_NOSUCHNAME;
		
	switch (obj->Request)
	{
	case MIB_RQSNXT:
		if (NeuroMibNext(obj, &local, idlen, 3) == FALSE)
			return MIB_NOSUCHNAME;
	case MIB_RQSGET:
		obj->Syntax.LngInt = obj->Id[idlen-1+NMIB_LOFFSET];
		return MIB_NOERROR;
	case MIB_RQSSET:
		return MIB_READONLY;
	}
}


WORD NeuroLayerNeuronIndexNeuron(MIB_OBJECT* obj, WORD idlen)
{
MIB_LOCAL* local = NULL;
			
	if ((local = NeuroMibRmon(obj, NeuroDataInstance, idlen, 3)) == NULL)
		return MIB_NOSUCHNAME;
		
	switch (obj->Request)
	{
	case MIB_RQSNXT:
		if (NeuroMibNext(obj, &local, idlen, 3) == FALSE)
			return MIB_NOSUCHNAME;
	case MIB_RQSGET:
		obj->Syntax.LngInt = obj->Id[idlen-1+NMIB_NOFFSET];
		return MIB_NOERROR;
	case MIB_RQSSET:
		return MIB_READONLY;
	}
}


WORD NeuroLayerNeuronValueIndex(MIB_OBJECT* obj, WORD idlen)
{
MIB_LOCAL* local = NULL;
			
	if ((local = NeuroMibRmon(obj, NeuroDataInstance, idlen, 4)) == NULL)
		return MIB_NOSUCHNAME;
		
	switch (obj->Request)
	{
	case MIB_RQSNXT:
		if (NeuroMibNext(obj, &local, idlen, 4) == FALSE)
			return MIB_NOSUCHNAME;
	case MIB_RQSGET:
		obj->Syntax.LngInt = local->Index;
		return MIB_NOERROR;
	case MIB_RQSSET:
		return MIB_READONLY;
	}
}


WORD NeuroLayerNeuronValueIndexLayer(MIB_OBJECT* obj, WORD idlen)
{
MIB_LOCAL* local = NULL;
			
	if ((local = NeuroMibRmon(obj, NeuroDataInstance, idlen, 4)) == NULL)
		return MIB_NOSUCHNAME;
		
	switch (obj->Request)
	{
	case MIB_RQSNXT:
		if (NeuroMibNext(obj, &local, idlen, 4) == FALSE)
			return MIB_NOSUCHNAME;
	case MIB_RQSGET:
		obj->Syntax.LngInt = obj->Id[idlen-1+NMIB_LOFFSET];
		return MIB_NOERROR;
	case MIB_RQSSET:
		return MIB_READONLY;
	}
}


WORD NeuroLayerNeuronValueIndexNeuron(MIB_OBJECT* obj, WORD idlen)
{
MIB_LOCAL* local = NULL;
			
	if ((local = NeuroMibRmon(obj, NeuroDataInstance, idlen, 4)) == NULL)
		return MIB_NOSUCHNAME;
		
	switch (obj->Request)
	{
	case MIB_RQSNXT:
		if (NeuroMibNext(obj, &local, idlen, 4) == FALSE)
			return MIB_NOSUCHNAME;
	case MIB_RQSGET:
		obj->Syntax.LngInt = obj->Id[idlen-1+NMIB_NOFFSET];
		return MIB_NOERROR;
	case MIB_RQSSET:
		return MIB_READONLY;
	}
}


WORD NeuroLayerNeuronValueIndexValue(MIB_OBJECT* obj, WORD idlen)
{
MIB_LOCAL* local = NULL;
			
	if ((local = NeuroMibRmon(obj, NeuroDataInstance, idlen, 4)) == NULL)
		return MIB_NOSUCHNAME;
		
	switch (obj->Request)
	{
	case MIB_RQSNXT:
		if (NeuroMibNext(obj, &local, idlen, 4) == FALSE)
			return MIB_NOSUCHNAME;
	case MIB_RQSGET:
		obj->Syntax.LngInt = obj->Id[idlen-1+NMIB_IOFFSET];
		return MIB_NOERROR;
	case MIB_RQSSET:
		return MIB_READONLY;
	}
}


WORD NeuroOwner(MIB_OBJECT* obj, WORD idlen)
{
MIB_LOCAL* local = NULL;
NeuroData* data = NULL;
			
	if ((local = NeuroMibRmon(obj, NeuroDataInstance, idlen, 1)) == NULL)
		return MIB_NOSUCHNAME;
		
	switch (obj->Request)
	{
	case MIB_RQSNXT:
		if (NeuroMibNext(obj, &local, idlen, 1) == FALSE)
			return MIB_NOSUCHNAME;
	case MIB_RQSGET:
		data = (NeuroData*)local->Data;
		memcpy(obj->Syntax.BufChr, data->Owner.Name, obj->SyntaxLen = data->Owner.NameLen);
		return MIB_NOERROR;
	case MIB_RQSSET:
		data = (NeuroData*)local->Data;
		if (data->Status != MIB_UNDERCREATION)
			return MIB_READONLY;
		memcpy(data->Owner.Name, obj->Syntax.BufChr, data->Owner.NameLen = obj->SyntaxLen);
		return MIB_NOERROR;
	}
}


WORD NeuroNrNetworks(MIB_OBJECT* obj, WORD idlen)
{
	if (MibSimple(obj, idlen) == FALSE)
	 	return MIB_NOSUCHNAME;

	switch (obj->Request)
	{
    case MIB_RQSNXT:
	case MIB_RQSGET:
		obj->Syntax.LngInt = TotalNrNetworks;
		return MIB_NOERROR;
	case MIB_RQSSET:
		return MIB_READONLY;
	}
}


WORD NeuroRunningNrNetworks(MIB_OBJECT* obj, WORD idlen)
{
	if (MibSimple(obj, idlen) == FALSE)
	 	return MIB_NOSUCHNAME;

	switch (obj->Request)
	{
    case MIB_RQSNXT:
	case MIB_RQSGET:
		obj->Syntax.LngInt = RunningNrNetworks;
		return MIB_NOERROR;
	case MIB_RQSSET:
		return MIB_READONLY;
	}
}


WORD NeuroNrLayers(MIB_OBJECT* obj, WORD idlen)
{
MIB_LOCAL* local = NULL;
NeuroData* data = NULL;		       
InOutData* iodata = NULL;
LONG l;

	if ((local = NeuroMibRmon(obj, NeuroDataInstance, idlen, 1)) == NULL)
		return MIB_NOSUCHNAME;
		
	switch (obj->Request)
	{
	case MIB_RQSNXT:
		if (NeuroMibNext(obj, &local, idlen, 1) == FALSE)
			return MIB_NOSUCHNAME;
	case MIB_RQSGET:
		data = (NeuroData*)local->Data;
		obj->Syntax.LngInt = data->NrLayers;
		return MIB_NOERROR;
	case MIB_RQSSET:
		data = (NeuroData*)local->Data;
		if (data->Status != MIB_UNDERCREATION)
			return MIB_READONLY;	   
		if (obj->Syntax.LngInt > MAXNRLAYERS)
			return MIB_BADVALUE;
		if ((iodata = ReNewInOutData(data->TargetData, data->NrNeurons[data->NrLayers],
            data->NrLayers >= obj->Syntax.LngInt ? data->NrNeurons[obj->Syntax.LngInt] : 1)) == NULL)
			return MIB_GENERROR;
		data->TargetData = iodata;
		for (l = data->NrLayers; l < obj->Syntax.LngInt; l++)
			data->NrNeurons[l] = 1;
		data->NrLayers = obj->Syntax.LngInt;
		return MIB_NOERROR;
	}
}


WORD NeuroNrNeurons(MIB_OBJECT* obj, WORD idlen)
{
MIB_LOCAL* local = NULL;
NeuroData* data = NULL;		       
LONG layer;		       
InOutData* iodata = NULL;

	if ((local = NeuroMibRmon(obj, NeuroDataInstance, idlen, 2)) == NULL)
		return MIB_NOSUCHNAME;
		
	switch (obj->Request)
	{
	case MIB_RQSNXT:
		if (NeuroMibNext(obj, &local, idlen, 2) == FALSE)
			return MIB_NOSUCHNAME;
	case MIB_RQSGET:
		data = (NeuroData*)local->Data;
		layer = obj->Id[idlen-1+NMIB_LOFFSET];
		if (layer > data->NrLayers)
			return MIB_NOSUCHNAME;
		obj->Syntax.LngInt = data->NrNeurons[layer-1];
		return MIB_NOERROR;
	case MIB_RQSSET:
		data = (NeuroData*)local->Data;
		if (data->Status != MIB_UNDERCREATION)
			return MIB_READONLY;
		layer = obj->Id[idlen-1+NMIB_LOFFSET];
		if (layer > data->NrLayers)
			return MIB_NOSUCHNAME;
		if (obj->Syntax.LngInt < 1 || obj->Syntax.LngInt > MAXNRNEURONS)
			return MIB_BADVALUE;
		/*  input or output layer?  */
		if (layer == 1)
		{		       
			if ((iodata = ReNewInOutData(data->InputData, obj->Syntax.LngInt, data->NrNeurons[layer-1])) == NULL)
				return MIB_GENERROR;
			data->InputData = iodata;
		}
		else
		if (layer == data->NrLayers)
		{
			if ((iodata = ReNewInOutData(data->TargetData, obj->Syntax.LngInt, data->NrNeurons[layer-1])) == NULL)
				return MIB_GENERROR;
			data->TargetData = iodata;
		}	
		
		data->NrNeurons[layer-1] = obj->Syntax.LngInt;
		return MIB_NOERROR;
	}
}


WORD NeuroRequestComm(MIB_OBJECT* obj, WORD idlen)
{
MIB_LOCAL* local = NULL;
NeuroData* data = NULL;
			
	if ((local = NeuroMibRmon(obj, NeuroDataInstance, idlen, 1)) == NULL)
		return MIB_NOSUCHNAME;
		
	switch (obj->Request)
	{
	case MIB_RQSNXT:
		if (NeuroMibNext(obj, &local, idlen, 1) == FALSE)
			return MIB_NOSUCHNAME;
	case MIB_RQSGET:
		data = (NeuroData*)local->Data;
		memcpy(obj->Syntax.BufChr, data->Comm, 
		       (obj->SyntaxLen = data->CommLen)*sizeof(obj->Syntax.BufChr[0]));
		return MIB_NOERROR;
	case MIB_RQSSET:
		data = (NeuroData*)local->Data;
		if (data->Status != MIB_UNDERCREATION)
			return MIB_READONLY;
		memcpy(data->Comm, obj->Syntax.BufChr, 
		       (data->CommLen = obj->SyntaxLen)*sizeof(obj->Syntax.BufChr[0]));
		return MIB_NOERROR;
	}
}


WORD NeuroNetRefresh(MIB_OBJECT* obj, WORD idlen)
{
MIB_LOCAL* local = NULL;
NeuroData* data = NULL;		       

	if ((local = NeuroMibRmon(obj, NeuroDataInstance, idlen, 1)) == NULL)
		return MIB_NOSUCHNAME;
		
	switch (obj->Request)
	{
	case MIB_RQSNXT:
		if (NeuroMibNext(obj, &local, idlen, 1) == FALSE)
			return MIB_NOSUCHNAME;
	case MIB_RQSGET:
		data = (NeuroData*)local->Data;
		obj->Syntax.LngUns = data->NetRefresh;
		return MIB_NOERROR;
	case MIB_RQSSET:
		data = (NeuroData*)local->Data;
		if (data->Status == MIB_VALID)		/*  if running change timer  */
			if (NeuroCChangeTimer(data, obj->Syntax.LngUns) == FALSE)
				return MIB_GENERROR;
		data->NetRefresh = obj->Syntax.LngUns;
		return MIB_NOERROR;
	}
}


WORD NeuroTrain(MIB_OBJECT* obj, WORD idlen)
{
MIB_LOCAL* local = NULL;
NeuroData* data = NULL;		       

	if ((local = NeuroMibRmon(obj, NeuroDataInstance, idlen, 1)) == NULL)
		return MIB_NOSUCHNAME;
		
	switch (obj->Request)
	{
	case MIB_RQSNXT:
		if (NeuroMibNext(obj, &local, idlen, 1) == FALSE)
			return MIB_NOSUCHNAME;
	case MIB_RQSGET:
		data = (NeuroData*)local->Data;
		obj->Syntax.LngInt = data->Train;
		return MIB_NOERROR;
	case MIB_RQSSET:
		data = (NeuroData*)local->Data;
		data->NetRefresh = obj->Syntax.LngInt;
		return MIB_NOERROR;
	}
}


WORD NeuroLearningRate(MIB_OBJECT* obj, WORD idlen)
{
MIB_LOCAL* local = NULL;
NeuroData* data = NULL;		       

	if ((local = NeuroMibRmon(obj, NeuroDataInstance, idlen, 1)) == NULL)
		return MIB_NOSUCHNAME;
		
	switch (obj->Request)
	{
	case MIB_RQSNXT:
		if (NeuroMibNext(obj, &local, idlen, 1) == FALSE)
			return MIB_NOSUCHNAME;
	case MIB_RQSGET:
		data = (NeuroData*)local->Data;
		obj->Syntax.LngInt = (LONG) FLOAT_SCALE*data->LearningRate;
		return MIB_NOERROR;
	case MIB_RQSSET:
		data = (NeuroData*)local->Data;
		data->LearningRate = (LONG) obj->Syntax.LngInt/FLOAT_SCALE;
		return MIB_NOERROR;
	}
}


WORD NeuroMomentum(MIB_OBJECT* obj, WORD idlen)
{
MIB_LOCAL* local = NULL;
NeuroData* data = NULL;		       

	if ((local = NeuroMibRmon(obj, NeuroDataInstance, idlen, 1)) == NULL)
		return MIB_NOSUCHNAME;
		
	switch (obj->Request)
	{
	case MIB_RQSNXT:
		if (NeuroMibNext(obj, &local, idlen, 1) == FALSE)
			return MIB_NOSUCHNAME;
	case MIB_RQSGET:
		data = (NeuroData*)local->Data;
		obj->Syntax.LngInt = (LONG) FLOAT_SCALE*data->Momentum;
		return MIB_NOERROR;
	case MIB_RQSSET:
		data = (NeuroData*)local->Data;
		data->Momentum = (LONG) obj->Syntax.LngInt/FLOAT_SCALE;
		return MIB_NOERROR;
	}
}


WORD NeuroError(MIB_OBJECT* obj, WORD idlen)
{
MIB_LOCAL* local = NULL;
NeuroData* data = NULL;		       

	if ((local = NeuroMibRmon(obj, NeuroDataInstance, idlen, 1)) == NULL)
		return MIB_NOSUCHNAME;
		
	switch (obj->Request)
	{
	case MIB_RQSNXT:
		if (NeuroMibNext(obj, &local, idlen, 1) == FALSE)
			return MIB_NOSUCHNAME;
	case MIB_RQSGET:
		data = (NeuroData*)local->Data;
		obj->Syntax.LngInt = (LONG) FLOAT_SCALE*data->Error;
		return MIB_NOERROR;
	case MIB_RQSSET:
		return MIB_READONLY;
	}
}


WORD NeuroTotalSteps(MIB_OBJECT* obj, WORD idlen)
{
MIB_LOCAL* local = NULL;
NeuroData* data = NULL;		       

	if ((local = NeuroMibRmon(obj, NeuroDataInstance, idlen, 1)) == NULL)
		return MIB_NOSUCHNAME;
		
	switch (obj->Request)
	{
	case MIB_RQSNXT:
		if (NeuroMibNext(obj, &local, idlen, 1) == FALSE)
			return MIB_NOSUCHNAME;
	case MIB_RQSGET:
		data = (NeuroData*)local->Data;
		obj->Syntax.LngUns = data->TotalSteps;
		return MIB_NOERROR;
	case MIB_RQSSET:
		return MIB_READONLY;
	}
}


WORD NeuroTrainingSteps(MIB_OBJECT* obj, WORD idlen)
{
MIB_LOCAL* local = NULL;
NeuroData* data = NULL;		       

	if ((local = NeuroMibRmon(obj, NeuroDataInstance, idlen, 1)) == NULL)
		return MIB_NOSUCHNAME;
		
	switch (obj->Request)
	{
	case MIB_RQSNXT:
		if (NeuroMibNext(obj, &local, idlen, 1) == FALSE)
			return MIB_NOSUCHNAME;
	case MIB_RQSGET:
		data = (NeuroData*)local->Data;
		obj->Syntax.LngUns = data->TrainSteps;
		return MIB_NOERROR;
	case MIB_RQSSET:
		return MIB_READONLY;
	}
}


WORD NeuroOutputValue(MIB_OBJECT* obj, WORD idlen)
{
MIB_LOCAL* local = NULL;
NeuroData* data = NULL; 
InOutData* datasource = NULL;
LONG layer, neuron;
			
	if ((local = NeuroMibRmon(obj, NeuroDataInstance, idlen, 3)) == NULL)
		return MIB_NOSUCHNAME;
		
	switch (obj->Request)
	{
	case MIB_RQSNXT:
		if (NeuroMibNext(obj, &local, idlen, 3) == FALSE)
			return MIB_NOSUCHNAME;
	case MIB_RQSGET:
		data = (NeuroData*)local->Data;
		layer = obj->Id[idlen-1+NMIB_LOFFSET];
		neuron = obj->Id[idlen-1+NMIB_NOFFSET];
		if (layer != data->NrLayers)
			return MIB_NOSUCHNAME;
		if (neuron > data->NrNeurons[layer-1])
			return MIB_NOSUCHNAME;
		obj->Syntax.LngInt = (LONG) FLOAT_SCALE*data->OutputValue[neuron-1];
		return MIB_NOERROR;
	case MIB_RQSSET:
		return MIB_READONLY;
	}
}


WORD NeuroDataSource(MIB_OBJECT* obj, WORD idlen)
{
MIB_LOCAL* local = NULL;
NeuroData* data = NULL;
InOutData* datasource = NULL;
LONG layer, neuron;
			
	if ((local = NeuroMibRmon(obj, NeuroDataInstance, idlen, 3)) == NULL)
		return MIB_NOSUCHNAME;
		
	switch (obj->Request)
	{
	case MIB_RQSNXT:
		if (NeuroMibNext(obj, &local, idlen, 3) == FALSE)
			return MIB_NOSUCHNAME;
	case MIB_RQSGET:
		data = (NeuroData*)local->Data;
		layer = obj->Id[idlen-1+NMIB_LOFFSET];
		neuron = obj->Id[idlen-1+NMIB_NOFFSET];
		if (layer != 1 && layer != data->NrLayers)
			return MIB_NOSUCHNAME;
		if (neuron > data->NrNeurons[layer-1])
			return MIB_NOSUCHNAME;
		if (layer > 0)
			datasource = &data->TargetData[neuron-1];
		else
			datasource = &data->InputData[neuron-1];
		memcpy(obj->Syntax.BufInt, datasource->Id, 
		       (obj->SyntaxLen = datasource->IdLen)*sizeof(obj->Syntax.BufInt[0]));
		return MIB_NOERROR;
	case MIB_RQSSET:
		data = (NeuroData*)local->Data;
		if (data->Status != MIB_UNDERCREATION)
			return MIB_READONLY;
		layer = obj->Id[idlen-1+NMIB_LOFFSET];
		neuron = obj->Id[idlen-1+NMIB_NOFFSET];
		if (layer != 1 && layer != data->NrLayers)
			return MIB_NOSUCHNAME;
		if (neuron > data->NrNeurons[layer-1])
			return MIB_NOSUCHNAME;
		if (layer > 0)
			datasource = &data->TargetData[neuron-1];
		else
			datasource = &data->InputData[neuron-1];
		memcpy(datasource->Id, obj->Syntax.BufInt, 
		       (datasource->IdLen = obj->SyntaxLen)*sizeof(obj->Syntax.BufInt[0]));
		return MIB_NOERROR;
	}
}


WORD NeuroDataSourceValue(MIB_OBJECT* obj, WORD idlen)
{
MIB_LOCAL* local = NULL;
NeuroData* data = NULL; 
InOutData* datasource = NULL;
LONG layer, neuron;
			
	if ((local = NeuroMibRmon(obj, NeuroDataInstance, idlen, 3)) == NULL)
		return MIB_NOSUCHNAME;
		
	switch (obj->Request)
	{
	case MIB_RQSNXT:
		if (NeuroMibNext(obj, &local, idlen, 3) == FALSE)
			return MIB_NOSUCHNAME;
	case MIB_RQSGET:
		data = (NeuroData*)local->Data;
		layer = obj->Id[idlen-1+NMIB_LOFFSET];
		neuron = obj->Id[idlen-1+NMIB_NOFFSET];
		if (layer != 1 && layer != data->NrLayers)
			return MIB_NOSUCHNAME;
		if (neuron > data->NrNeurons[layer-1])
			return MIB_NOSUCHNAME;
		if (layer > 0)
			datasource = &data->TargetData[neuron-1];
		else
			datasource = &data->InputData[neuron-1];
		obj->Syntax.LngInt = (LONG) FLOAT_SCALE*datasource->TheValue;
		return MIB_NOERROR;
	case MIB_RQSSET:
		return MIB_READONLY;
	}
}


WORD NeuroDataSourceMax(MIB_OBJECT* obj, WORD idlen)
{
MIB_LOCAL* local = NULL;
NeuroData* data = NULL; 
InOutData* datasource = NULL;
LONG layer, neuron;
			
	if ((local = NeuroMibRmon(obj, NeuroDataInstance, idlen, 3)) == NULL)
		return MIB_NOSUCHNAME;
		
	switch (obj->Request)
	{
	case MIB_RQSNXT:
		if (NeuroMibNext(obj, &local, idlen, 3) == FALSE)
			return MIB_NOSUCHNAME;
	case MIB_RQSGET:
		data = (NeuroData*)local->Data;
		layer = obj->Id[idlen-1+NMIB_LOFFSET];
		neuron = obj->Id[idlen-1+NMIB_NOFFSET];
		if (layer != 1 && layer != data->NrLayers)
			return MIB_NOSUCHNAME;
		if (neuron > data->NrNeurons[layer-1])
			return MIB_NOSUCHNAME;
		if (layer > 0)
			datasource = &data->TargetData[neuron-1];
		else
			datasource = &data->InputData[neuron-1];
		obj->Syntax.LngInt = (LONG) FLOAT_SCALE*datasource->MaxValue;
		return MIB_NOERROR;
	case MIB_RQSSET:
		data = (NeuroData*)local->Data;
		if (data->Status != MIB_UNDERCREATION)
			return MIB_READONLY;
		layer = obj->Id[idlen-1+NMIB_LOFFSET];
		neuron = obj->Id[idlen-1+NMIB_NOFFSET];
		if (layer != 1 && layer != data->NrLayers)
			return MIB_NOSUCHNAME;
		if (neuron > data->NrNeurons[layer-1])
			return MIB_NOSUCHNAME;
		if (layer > 0)
			datasource = &data->TargetData[neuron-1];
		else
			datasource = &data->InputData[neuron-1];
		datasource->MaxValue = (LONG) obj->Syntax.LngInt/FLOAT_SCALE;
		return MIB_NOERROR;
	}
}


WORD NeuroDataSourceMin(MIB_OBJECT* obj, WORD idlen)
{
MIB_LOCAL* local = NULL;
NeuroData* data = NULL; 
InOutData* datasource = NULL;
LONG layer, neuron;
			
	if ((local = NeuroMibRmon(obj, NeuroDataInstance, idlen, 3)) == NULL)
		return MIB_NOSUCHNAME;
		
	switch (obj->Request)
	{
	case MIB_RQSNXT:
		if (NeuroMibNext(obj, &local, idlen, 3) == FALSE)
			return MIB_NOSUCHNAME;
	case MIB_RQSGET:
		data = (NeuroData*)local->Data;
		layer = obj->Id[idlen-1+NMIB_LOFFSET];
		neuron = obj->Id[idlen-1+NMIB_NOFFSET];
		if (layer != 1 && layer != data->NrLayers)
			return MIB_NOSUCHNAME;
		if (neuron > data->NrNeurons[layer-1])
			return MIB_NOSUCHNAME;
		if (layer > 0)
			datasource = &data->TargetData[neuron-1];
		else
			datasource = &data->InputData[neuron-1];
		obj->Syntax.LngInt = (LONG) FLOAT_SCALE*datasource->MinValue;
		return MIB_NOERROR;
	case MIB_RQSSET:
		data = (NeuroData*)local->Data;
		if (data->Status != MIB_UNDERCREATION)
			return MIB_READONLY;
		layer = obj->Id[idlen-1+NMIB_LOFFSET];
		neuron = obj->Id[idlen-1+NMIB_NOFFSET];
		if (layer != 1 && layer != data->NrLayers)
			return MIB_NOSUCHNAME;
		if (neuron > data->NrNeurons[layer-1])
			return MIB_NOSUCHNAME;
		if (layer > 0)
			datasource = &data->TargetData[neuron-1];
		else
			datasource = &data->InputData[neuron-1];
		datasource->MinValue = (LONG) obj->Syntax.LngInt/FLOAT_SCALE;
		return MIB_NOERROR;
	}
}


WORD NeuroDataSourceAverageMethod(MIB_OBJECT* obj, WORD idlen)
{
MIB_LOCAL* local = NULL;
NeuroData* data = NULL; 
InOutData* datasource = NULL;
LONG layer, neuron;
			
	if ((local = NeuroMibRmon(obj, NeuroDataInstance, idlen, 3)) == NULL)
		return MIB_NOSUCHNAME;
		
	switch (obj->Request)
	{
	case MIB_RQSNXT:
		if (NeuroMibNext(obj, &local, idlen, 3) == FALSE)
			return MIB_NOSUCHNAME;
	case MIB_RQSGET:
		data = (NeuroData*)local->Data;
		layer = obj->Id[idlen-1+NMIB_LOFFSET];
		neuron = obj->Id[idlen-1+NMIB_NOFFSET];
		if (layer != 1 && layer != data->NrLayers)
			return MIB_NOSUCHNAME;
		if (neuron > data->NrNeurons[layer-1])
			return MIB_NOSUCHNAME;
		if (layer > 0)
			datasource = &data->TargetData[neuron-1];
		else
			datasource = &data->InputData[neuron-1];
		obj->Syntax.LngInt = datasource->AverageMethod;
		return MIB_NOERROR;
	case MIB_RQSSET:
		data = (NeuroData*)local->Data;
		if (data->Status != MIB_UNDERCREATION)
			return MIB_READONLY;
		layer = obj->Id[idlen-1+NMIB_LOFFSET];
		neuron = obj->Id[idlen-1+NMIB_NOFFSET];
		if (layer != 1 && layer != data->NrLayers)
			return MIB_NOSUCHNAME;
		if (neuron > data->NrNeurons[layer-1])
			return MIB_NOSUCHNAME;
		if (layer > 0)
			datasource = &data->TargetData[neuron-1];
		else
			datasource = &data->InputData[neuron-1];
		datasource->AverageMethod = obj->Syntax.LngInt;
		return MIB_NOERROR;
	}
}


WORD NeuroDataSourceAverageSteps(MIB_OBJECT* obj, WORD idlen)
{
MIB_LOCAL* local = NULL;
NeuroData* data = NULL; 
InOutData* datasource = NULL;
LONG layer, neuron;
			
	if ((local = NeuroMibRmon(obj, NeuroDataInstance, idlen, 3)) == NULL)
		return MIB_NOSUCHNAME;
		
	switch (obj->Request)
	{
	case MIB_RQSNXT:
		if (NeuroMibNext(obj, &local, idlen, 3) == FALSE)
			return MIB_NOSUCHNAME;
	case MIB_RQSGET:
		data = (NeuroData*)local->Data;
		layer = obj->Id[idlen-1+NMIB_LOFFSET];
		neuron = obj->Id[idlen-1+NMIB_NOFFSET];
		if (layer != 1 && layer != data->NrLayers)
			return MIB_NOSUCHNAME;
		if (neuron > data->NrNeurons[layer-1])
			return MIB_NOSUCHNAME;
		if (layer > 0)
			datasource = &data->TargetData[neuron-1];
		else
			datasource = &data->InputData[neuron-1];
		obj->Syntax.LngInt = datasource->AverageSteps;
		return MIB_NOERROR;
	case MIB_RQSSET:
		data = (NeuroData*)local->Data;
		if (data->Status != MIB_UNDERCREATION)
			return MIB_READONLY;
		layer = obj->Id[idlen-1+NMIB_LOFFSET];
		neuron = obj->Id[idlen-1+NMIB_NOFFSET];
		if (layer != 1 && layer != data->NrLayers)
			return MIB_NOSUCHNAME;
		if (neuron > data->NrNeurons[layer-1])
			return MIB_NOSUCHNAME;
		if (layer > 0)
			datasource = &data->TargetData[neuron-1];
		else
			datasource = &data->InputData[neuron-1];
		datasource->AverageSteps = obj->Syntax.LngInt;
		return MIB_NOERROR;
	}
}


WORD NeuroDataSourceTrap(MIB_OBJECT* obj, WORD idlen)
{
MIB_LOCAL* local = NULL;
NeuroData* data = NULL; 
InOutData* datasource = NULL;
LONG layer, neuron;
			
	if ((local = NeuroMibRmon(obj, NeuroDataInstance, idlen, 3)) == NULL)
		return MIB_NOSUCHNAME;
		
	switch (obj->Request)
	{
	case MIB_RQSNXT:
		if (NeuroMibNext(obj, &local, idlen, 3) == FALSE)
			return MIB_NOSUCHNAME;
	case MIB_RQSGET:
		data = (NeuroData*)local->Data;
		layer = obj->Id[idlen-1+NMIB_LOFFSET];
		neuron = obj->Id[idlen-1+NMIB_NOFFSET];
		if (layer != 1 && layer != data->NrLayers)
			return MIB_NOSUCHNAME;
		if (neuron > data->NrNeurons[layer-1])
			return MIB_NOSUCHNAME;
		if (layer > 0)
			datasource = &data->TargetData[neuron-1];
		else
			datasource = &data->InputData[neuron-1];
		obj->Syntax.LngInt = datasource->Trap;
		return MIB_NOERROR;
	case MIB_RQSSET:
		data = (NeuroData*)local->Data;
		layer = obj->Id[idlen-1+NMIB_LOFFSET];
		neuron = obj->Id[idlen-1+NMIB_NOFFSET];
		if (layer != 1 && layer != data->NrLayers)
			return MIB_NOSUCHNAME;
		if (neuron > data->NrNeurons[layer-1])
			return MIB_NOSUCHNAME;
		if (layer > 0)
			datasource = &data->TargetData[neuron-1];
		else
			datasource = &data->InputData[neuron-1];
		datasource->Trap = (BOOLEAN)obj->Syntax.LngInt;
		return MIB_NOERROR;
	}
}


WORD NeuroDataSourceTrapComm(MIB_OBJECT* obj, WORD idlen)
{
MIB_LOCAL* local = NULL;
NeuroData* data = NULL;
InOutData* datasource = NULL;
LONG layer, neuron;
			
	if ((local = NeuroMibRmon(obj, NeuroDataInstance, idlen, 3)) == NULL)
		return MIB_NOSUCHNAME;
		
	switch (obj->Request)
	{
	case MIB_RQSNXT:
		if (NeuroMibNext(obj, &local, idlen, 3) == FALSE)
			return MIB_NOSUCHNAME;
	case MIB_RQSGET:
		data = (NeuroData*)local->Data;
		layer = obj->Id[idlen-1+NMIB_LOFFSET];
		neuron = obj->Id[idlen-1+NMIB_NOFFSET];
		if (layer != 1 && layer != data->NrLayers)
			return MIB_NOSUCHNAME;
		if (neuron > data->NrNeurons[layer-1])
			return MIB_NOSUCHNAME;
		if (layer > 0)
			datasource = &data->TargetData[neuron-1];
		else
			datasource = &data->InputData[neuron-1];
		memcpy(obj->Syntax.BufChr, datasource->TrapComm, 
		       (obj->SyntaxLen = datasource->TrapCommLen)*sizeof(obj->Syntax.BufChr[0]));
		return MIB_NOERROR;
	case MIB_RQSSET:
		data = (NeuroData*)local->Data;
		layer = obj->Id[idlen-1+NMIB_LOFFSET];
		neuron = obj->Id[idlen-1+NMIB_NOFFSET];
		if (layer != 1 && layer != data->NrLayers)
			return MIB_NOSUCHNAME;
		if (neuron > data->NrNeurons[layer-1])
			return MIB_NOSUCHNAME;
		if (layer > 0)
			datasource = &data->TargetData[neuron-1];
		else
			datasource = &data->InputData[neuron-1];
		memcpy(datasource->TrapComm, obj->Syntax.BufChr, 
		       (datasource->TrapCommLen = obj->SyntaxLen)*sizeof(obj->Syntax.BufChr[0]));
		return MIB_NOERROR;
	}
}


WORD NeuroDataSourceThresholdCheck(MIB_OBJECT* obj, WORD idlen)
{
MIB_LOCAL* local = NULL;
NeuroData* data = NULL; 
InOutData* datasource = NULL;
LONG layer, neuron;
			
	if ((local = NeuroMibRmon(obj, NeuroDataInstance, idlen, 3)) == NULL)
		return MIB_NOSUCHNAME;
		
	switch (obj->Request)
	{
	case MIB_RQSNXT:
		if (NeuroMibNext(obj, &local, idlen, 3) == FALSE)
			return MIB_NOSUCHNAME;
	case MIB_RQSGET:
		data = (NeuroData*)local->Data;
		layer = obj->Id[idlen-1+NMIB_LOFFSET];
		neuron = obj->Id[idlen-1+NMIB_NOFFSET];
		if (layer != 1 && layer != data->NrLayers)
			return MIB_NOSUCHNAME;
		if (neuron > data->NrNeurons[layer-1])
			return MIB_NOSUCHNAME;
		if (layer > 0)
			datasource = &data->TargetData[neuron-1];
		else
			datasource = &data->InputData[neuron-1];
		obj->Syntax.LngInt = datasource->ThresholdCheck;
		return MIB_NOERROR;
	case MIB_RQSSET:
		data = (NeuroData*)local->Data;
		layer = obj->Id[idlen-1+NMIB_LOFFSET];
		neuron = obj->Id[idlen-1+NMIB_NOFFSET];
		if (layer != 1 && layer != data->NrLayers)
			return MIB_NOSUCHNAME;
		if (neuron > data->NrNeurons[layer-1])
			return MIB_NOSUCHNAME;
		if (layer > 0)
			datasource = &data->TargetData[neuron-1];
		else
			datasource = &data->InputData[neuron-1];
		datasource->ThresholdCheck = obj->Syntax.LngInt;
		return MIB_NOERROR;
	}
}


WORD NeuroDataSourceThresholdValue(MIB_OBJECT* obj, WORD idlen)
{
MIB_LOCAL* local = NULL;
NeuroData* data = NULL; 
InOutData* datasource = NULL;
LONG layer, neuron;
LONG index;
			
	if ((local = NeuroMibRmon(obj, NeuroDataInstance, idlen, 4)) == NULL)
		return MIB_NOSUCHNAME;
		
	switch (obj->Request)
	{
	case MIB_RQSNXT:
		if (NeuroMibNext(obj, &local, idlen, 4) == FALSE)
			return MIB_NOSUCHNAME;
	case MIB_RQSGET:
		data = (NeuroData*)local->Data;
		layer = obj->Id[idlen-1+NMIB_LOFFSET];
		neuron = obj->Id[idlen-1+NMIB_NOFFSET];
		index = obj->Id[idlen-1+NMIB_IOFFSET];
		if (layer != 1 && layer != data->NrLayers)
			return MIB_NOSUCHNAME;
		if (neuron > data->NrNeurons[layer-1])
			return MIB_NOSUCHNAME;
		if (layer > 0)
			datasource = &data->TargetData[neuron-1];
		else
			datasource = &data->InputData[neuron-1];
		if (index > MAXNRTHRESHOLDVALUES)
			return MIB_NOSUCHNAME;
		obj->Syntax.LngInt = (LONG) datasource->ThresholdValue[index]*FLOAT_SCALE;
		return MIB_NOERROR;
	case MIB_RQSSET:
		data = (NeuroData*)local->Data;
		layer = obj->Id[idlen-1+NMIB_LOFFSET];
		neuron = obj->Id[idlen-1+NMIB_NOFFSET];
		index = obj->Id[idlen-1+NMIB_IOFFSET];
		if (layer != 1 && layer != data->NrLayers)
			return MIB_NOSUCHNAME;
		if (neuron > data->NrNeurons[layer-1])
			return MIB_NOSUCHNAME;
		if (layer > 0)
			datasource = &data->TargetData[neuron-1];
		else
			datasource = &data->InputData[neuron-1];
		if (index > MAXNRTHRESHOLDVALUES)
			return MIB_NOSUCHNAME;
		datasource->ThresholdValue[index] = (LONG) obj->Syntax.LngInt/FLOAT_SCALE;
		return MIB_NOERROR;
	}
}


WORD NeuroStatus(MIB_OBJECT* obj, WORD idlen)
{
MIB_LOCAL* local = NULL;
NeuroData* data = NULL; 
			
    TRACEMSG("NeuroStatus called");

	local = NeuroMibRmon(obj, NeuroDataInstance, idlen, 1);
	
	switch (obj->Request)
	{
	case MIB_RQSNXT:
		if (NeuroMibNext(obj, &local, idlen, 1) == FALSE)
		{
			obj->Syntax.LngInt = MIB_INVALID;
			return MIB_NOSUCHNAME;
		}
	case MIB_RQSGET:
		if (local == NULL)
		{
			obj->Syntax.LngInt = MIB_INVALID;
			return MIB_NOSUCHNAME;
		}
		data = (NeuroData*)local->Data;
		obj->Syntax.LngInt = data->Status;
		return MIB_NOERROR;
	case MIB_RQSSET:
		if (local == NULL)
		{
			switch (obj->Syntax.LngInt)
			{
			case MIB_CREATEREQUEST:
				if ((local = NeuroMibInsert(obj, &NeuroDataInstance, idlen, 1)) == NULL)
					return MIB_GENERROR;
				if ((local->Data = NewNeuroData()) == NULL)
					return MIB_GENERROR;
				data = (NeuroData*)local->Data;
				if (NeuroCInit(data) == TRUE)
				{
					data->Status = MIB_UNDERCREATION;
                    TotalNrNetworks++;
					return MIB_NOERROR;
				}
				else
				{
					NeuroCStop(data);
                    DelNeuroData(data);
					NeuroMibRemove(obj, &NeuroDataInstance, idlen, 1);
					return MIB_GENERROR;
				}
			default:
				return MIB_BADVALUE;
			}
		}
		else
		{
			data = (NeuroData*)local->Data;
			switch (data->Status)
			{
			case MIB_UNDERCREATION:
				switch (obj->Syntax.LngInt)
				{
				case MIB_VALID:		    
					if (NeuroCStart(data, NeuroMCalcCallback, local, data->NetRefresh) == TRUE)
					{
						data->Status = MIB_VALID;
                        RunningNrNetworks++;
						return MIB_NOERROR;
					}
				case MIB_INVALID:
					if (NeuroCStop(data) == TRUE)
					{
                        DelNeuroData(data);
						if (NeuroMibRemove(obj, &NeuroDataInstance, idlen, 1) == TRUE)
                        {
                            TotalNrNetworks--;
							return MIB_NOERROR;
                        }
					}
					return MIB_GENERROR;
				default:
					return MIB_BADVALUE;
				}
			case MIB_VALID:
				switch (obj->Syntax.LngInt)
				{
				case MIB_INVALID:	    
					if (NeuroCStop(data) == TRUE)
					{
                        DelNeuroData(data);
						if (NeuroMibRemove(obj, &NeuroDataInstance, idlen, 1) == TRUE)
                        {
                            RunningNrNetworks--;
                            TotalNrNetworks--;
							return MIB_NOERROR;
                        }
					}
					return MIB_GENERROR;
                case MIB_VALID:
                    return MIB_NOERROR;
				default:
					return MIB_BADVALUE;
				}
			}
		}
	}
}


MIB_LOCAL* NeuroMibInsert(MIB_OBJECT* obj, MIB_LOCAL** local, WORD idlen, WORD indexsize)
{
MIB_LOCAL* junk;

	if (obj->IdLen != idlen + indexsize)
		return NULL;

    switch(indexsize)
    {
    case NMIB_IOFFSET:
	    if (obj->Id[idlen-1+NMIB_IOFFSET] < 1)
    		return NULL;
    case NMIB_NOFFSET:
	    if (obj->Id[idlen-1+NMIB_NOFFSET] < 1)
		    return NULL;
    case NMIB_LOFFSET:
	    if (obj->Id[idlen-1+NMIB_LOFFSET] < 1)
	    	return NULL;
    case NMIB_COFFSET:
	    if (obj->Id[idlen-1+NMIB_COFFSET] < 1)
    		return NULL;
        break;
    }
		
	if ((junk = MALLOC(sizeof(MIB_LOCAL))) == NULL)
	    return NULL;
	    
	junk->Index = obj->Id[idlen-1+NMIB_COFFSET];
	
	if (*local == NULL || (*local)->Index > junk->Index)
	{
		junk->Next = (*local);
		*local = junk;
		return junk;
	}   
	
	while ((*local)->Next != NULL && (*local)->Next->Index <= junk->Index)
		*local = (*local)->Next;
		
	if ((*local)->Index == junk->Index)
	{
		FREE(junk);
		return *local;
	}	
	
	junk->Next = (*local)->Next;
	(*local)->Next = junk;
	
	return junk;
}


BOOLEAN NeuroMibRemove(MIB_OBJECT* obj, MIB_LOCAL** local, WORD idlen, WORD indexsize)
{
MIB_LOCAL* junk;
	
	if (*local == NULL)
		return FALSE;
		
	if (obj->IdLen != idlen + indexsize)
		return FALSE;
		
	if ((*local)->Index == obj->Id[idlen-1+NMIB_COFFSET])
	{
		junk = (*local)->Next;
		FREE(*local);
		*local = junk;
		return TRUE;
	}	
	
	while ((*local)->Next != NULL && (*local)->Next->Index < obj->Id[idlen-1+NMIB_COFFSET])
		*local = (*local)->Next;
		
	if ((*local)->Next->Index == obj->Id[idlen-1+NMIB_COFFSET])
	{
		junk = (*local)->Next->Next;
		FREE((*local)->Next);
		(*local)->Next = junk;
		return TRUE;
	}	
	
	return FALSE;
}


MIB_LOCAL* NeuroMibRmon(MIB_OBJECT* obj, MIB_LOCAL* local, WORD idlen, WORD indexsize)
{
	if (local == NULL || obj->IdLen > idlen + indexsize)
		return NULL;

{
    CHAR msg[81];
    TRACEMSG("Mib");
    MibID2String(obj->Id, obj->IdLen, msg, 80);
    TRACEMSG(msg);
}

	if (obj->Request != MIB_RQSNXT)
	{
		if (obj->IdLen != idlen + indexsize)
			return NULL;
		while (local->Index < obj->Id[idlen-1+NMIB_COFFSET] && local->Next != NULL)
			local = local->Next;
		if (local->Index == obj->Id[idlen-1+NMIB_COFFSET])
			return local;
		return NULL;
	}
	else
	{
		if (obj->IdLen == idlen)
			return local;
		while (local->Index < obj->Id[idlen-1+NMIB_COFFSET] && local->Next != NULL)
			local = local->Next;
		if (local->Index >= obj->Id[idlen-1+NMIB_COFFSET])
			return local;
		return NULL;
	}
}


BOOLEAN NeuroMibNext(MIB_OBJECT* obj, MIB_LOCAL** local, WORD idlen, WORD indexsize)
{	
NeuroData* data = NULL;
LONG layer, neuron;
WORD i;

    if (*local == NULL || indexsize < 1)
        return FALSE;
	
{
    CHAR msg[81];
    TRACEMSG("Next:");
    MibID2String(obj->Id, obj->IdLen, msg, 80);
    TRACEMSG(msg);
}

	if (obj->IdLen < idlen+indexsize)
	{				  
		for (i = obj->IdLen; i < idlen+indexsize; i++)
		{		      
			if (i == idlen-1+NMIB_COFFSET)
				obj->Id[i] = (*local)->Index;
			else				/*  MIB collector is initialized to index  */
				obj->Id[i] = 1;		/*  the rest can be initialized to 1       */
		}
		obj->IdLen = idlen+indexsize;
                return TRUE;
	}	
	else
	{
		switch (indexsize)
		{
		case NMIB_IOFFSET:
			data = (NeuroData*)(*local)->Data;
			/*  layer index must be 1 or NrLayers  */
			if (((layer = obj->Id[idlen-1+NMIB_LOFFSET]) == 1 || layer == data->NrLayers) &&
                            (neuron = obj->Id[idlen-1+NMIB_NOFFSET]) <= data->NrNeurons[layer-1] &&
			    obj->Id[idlen-1+NMIB_IOFFSET] < (layer == 1 ? data->InputData[neuron-1].AverageSteps : data->TargetData[neuron-1].AverageSteps))
			{
				obj->Id[idlen-1+NMIB_IOFFSET]++;
				return TRUE;
			}
		case NMIB_NOFFSET:
			data = (NeuroData*)(*local)->Data;
			/*  layer index must be 1 or NrLayers  */
			if (((layer = obj->Id[idlen-1+NMIB_LOFFSET]) == 1 || layer == data->NrLayers) &&
                            obj->Id[idlen-1+NMIB_NOFFSET] < data->NrNeurons[layer-1])
			{
				obj->Id[idlen-1+NMIB_NOFFSET]++;
				switch (indexsize)
				{	
				case NMIB_IOFFSET:
					obj->Id[idlen-1+NMIB_IOFFSET] = 1;
				}
				return TRUE;
			}
		case NMIB_LOFFSET:
			data = (NeuroData*)(*local)->Data;
			if (obj->Id[idlen-1+NMIB_LOFFSET] < data->NrLayers)
			{					   			   
				/*  layer index must be 1 or NrLayers in case   */
				/*  neuron index or subindex must be specified  */
				if (indexsize == NMIB_IOFFSET || indexsize == NMIB_NOFFSET)
					obj->Id[idlen-1+NMIB_LOFFSET] = data->NrLayers;
				else
					obj->Id[idlen-1+NMIB_LOFFSET]++;
				switch (indexsize)
				{	
				case NMIB_IOFFSET:
					obj->Id[idlen-1+NMIB_IOFFSET] = 1;
				case NMIB_NOFFSET:
					obj->Id[idlen-1+NMIB_NOFFSET] = 1;
				}
				return TRUE;
			}
		case NMIB_COFFSET:
			while ((*local)->Index <= obj->Id[idlen-1+NMIB_COFFSET] && (*local)->Next != NULL)
				*local = (*local)->Next;
			if ((*local)->Index > obj->Id[idlen-1+NMIB_COFFSET])
			{
				obj->Id[idlen-1+NMIB_COFFSET] = (*local)->Index;
				switch (indexsize)
				{
				case NMIB_IOFFSET:
					obj->Id[idlen-1+NMIB_IOFFSET] = 1;
				case NMIB_NOFFSET:
					obj->Id[idlen-1+NMIB_NOFFSET] = 1;
				case NMIB_LOFFSET:
					obj->Id[idlen-1+NMIB_LOFFSET] = 1;
				}
				return TRUE;
			}
			return FALSE;
		}
		return FALSE;
	}
	return FALSE;
}


VOID NeuroMCalcCallback(TIMER_DESCR *timer, ULONG now, VOID *param)
{      
NeuroData* data = param;
InOutput *in, *out;
FLOAT error;

	in = NewNeuronInOutput(data->NrNeurons[0]);
	out = NewNeuronInOutput(data->NrNeurons[data->NrLayers-1]);
			    
	if (in != NULL || out != NULL)
	{
		if (NeuroMib2Neuron(data, data->InputData, data->NrNeurons[0], in) == FALSE)
		{
			ERROR(MODULE, NEURO_NOINPUT);
			return;
		}
		if (data->Train && NeuroMib2Neuron(data, data->TargetData, data->NrNeurons[data->NrLayers-1], out) == FALSE)
		{
			ERROR(MODULE, NEURO_NOTARGET);
			return;
		}
			
		if (CalcNeuronNet(data->Net, 
			      in, data->NrNeurons[0], 
			      out, data->NrNeurons[data->NrLayers-1],
			      data->Train, TRUE,
			      data->LearningRate, data->Momentum) != NEURO_OK)
		{
			ERROR(MODULE, NEURO_CALCERR);
			return;
		}

		if (NeuroNeuron2Mib(out, data->NrNeurons[data->NrLayers-1], data->OutputValue) == FALSE)
		{
			ERROR(MODULE, NEURO_NORESULT);
			return;
		}
						  
		if (data->Train)
		{
			if (CalcNeuronNetError(data->Net, 
					   out, data->NrNeurons[data->NrLayers-1], 
					   &error) != NEURO_OK)
			{
				ERROR(MODULE, NEURO_CALCERR);
				return;
			}
				   
			data->Error = error;
			data->TrainSteps++;
		}
		else
		{
			data->Error = 0;
		}
		
		if (NeuroCheckThresholds(data) == FALSE)
			ERROR(MODULE, NEURO_THRESHOLD);
		
		data->TotalSteps++;
	}
 
	DelNeuronInOutput(out);
	DelNeuronInOutput(in);
}

		    
BOOLEAN NeuroMib2Neuron(NeuroData* data, InOutData* mibinout, LONG n, InOutput* inout)
{	 						  
MIB_PROFIL prf = { "public", 6 };
MIB_OBJECT obj = { MIB_RQSGET, { NMIB_BASE }, NMIB_BASELEN, MIB_INTEGER, MIB_VALID };
FLOAT value;
LONG i, s;

	for (i = 0; i < n; i++)
	{			 
		memcpy(prf.Comm, data->Comm, (prf.CommLen = data->CommLen));
		memcpy(obj.Id, mibinout[i].Id, (obj.IdLen = mibinout[i].IdLen)*sizeof(mibinout[i].Id[0]));
		if (MibRequest(prf.Comm, prf.CommLen, &obj) != MIB_NOERROR)
			return FALSE;
		else
		{
			inout[i].inoutnr = i;		 
			switch (obj.Type)
			{
			case MIB_INTEGER:
				value = (LONG) obj.Syntax.LngInt*FLOAT_SCALE;
				break;
			case MIB_COUNTER:
			case MIB_GAUGE:
			case MIB_TIMETICKS:
				value = (LONG) obj.Syntax.LngUns*FLOAT_SCALE;
				break;
			default:	     
				ERROR(MODULE, NEURO_WRONGTYPE);
				return FALSE;
			}
			
			value = NEURO_MAXMINOUTPUT*(value-mibinout[i].MinValue)/(mibinout[i].MaxValue-mibinout[i].MinValue)+NEURO_MINOUTPUT;
				
			if (mibinout[i].AverageMethod != SIMPLE_DIFF && mibinout[i].AverageMethod != SIMPLE_VALUE)
				for (s = 1; s < mibinout[i].AverageSteps; s++)
					mibinout[i].Value[s] = mibinout[i].Value[s-1];
				
			switch (mibinout[i].AverageMethod)
			{    
			case SIMPLE_DIFF: 
				value = value - mibinout[i].Value[0];
			case SIMPLE_VALUE:		       
				mibinout[i].Value[0] = value;
				mibinout[i].TheValue = value;
				break;
			case MEAN_DIFF:
				value = value - mibinout[i].Value[0];
			case MEAN_VALUE:
				mibinout[i].Value[0] = value;
				mibinout[i].TheValue = NeuroMean(mibinout[i].Value, mibinout[i].AverageSteps);
				break;
			case HARMONIC_DIFF:
				value = value - mibinout[i].Value[0];
			case HARMONIC_VALUE:
				mibinout[i].Value[0] = value;
				mibinout[i].TheValue = NeuroHarmonic(mibinout[i].Value, mibinout[i].AverageSteps);
				break;
			}
			
			inout[i].value = mibinout[i].TheValue;
		}
	}	 
	return TRUE;
}


FLOAT NeuroMean(FLOAT value[], LONG nrvalues)
{				      
FLOAT result = 0;
LONG i;

	for (i = 0; i < nrvalues; i++)
		result += value[i];
	return result/nrvalues;
}


FLOAT NeuroHarmonic(FLOAT value[], LONG nrvalues)
{				      
FLOAT result = 0;
LONG i;

	for (i = 0; i < nrvalues; i++)
		result += value[i];
	return result/nrvalues;
}


BOOLEAN NeuroNeuron2Mib(InOutput* inout, LONG n, FLOAT* out)
{
LONG i;

	for (i = 0; i < n; i++)
		out[inout->inoutnr] = inout->value;
    return TRUE;
}


BOOLEAN NeuroCheckThresholds(NeuroData* data)
{			
InOutData* datasource;
LONG l, n;   
FLOAT value;
BOOLEAN trap;

	for (l = 0; l < data->NrLayers; l += data->NrLayers-1)
	{				
		for (n = 0; n < data->NrNeurons[l]; n++)
		{			    
			if (l > 0)
			{
				datasource = &data->TargetData[n];
				if (data->Train)
					value = datasource->TheValue;
				else
					value = data->OutputValue[n];
			}
			else
			{
				datasource = &data->InputData[n];    
				value = datasource->TheValue;
			}
	
			if (datasource->Trap)
			{				  
				trap = FALSE;
				
				switch (datasource->ThresholdCheck)
				{	      
				case LESS_THAN:
					if (value < datasource->ThresholdValue[0])
						trap = TRUE;
					break;
				case GREATER_THAN:
					if (value > datasource->ThresholdValue[0])
						trap = TRUE;
					break;
				case IN_RANGE:
					if (value > datasource->ThresholdValue[0] &&
					    value < datasource->ThresholdValue[1])
						trap = TRUE;
					break;
				case OUT_RANGE:
					if (value < datasource->ThresholdValue[0] &&
					    value > datasource->ThresholdValue[1])
						trap = TRUE;
					break;
				case EQUAL:
					if (value == datasource->ThresholdValue[0])
						trap = TRUE;
					break;
				case NOT_EQUAL:
					if (value != datasource->ThresholdValue[0])
						trap = TRUE;
					break;
				case NO_THRESHOLD:
				default:
					break;
				}
				
				if (trap == TRUE)
					AgentSendTrap(datasource->TrapComm, datasource->TrapCommLen,
						    MIB_TRAP_ENTSPECIFIC, TRAP_SPECIFIC_NEUROTHRESHOLD, NULL, 0);
			}
		}
	}
	return TRUE;
}
