#ifdef PERFORMANCE_TEST


#include <dnpap.h>
#include <memory.h>
#include <string.h>
#include <config.h>
#include <timer.h>
#include <mac.h>

#include "perf.h"


static INT         mode;

static MAC_IFACE   *iface = NULL;

static LONG interval;
static TIMER_DESCR *timer = NULL;

static MAC_COLL    coll;

static ULONG timerprevnow, timeroldnow;
static MAC_STAT timerprevstat, timeroldstat;
static MAC_PERF timerprevperfstat;

static ULONG macprevnow, macoldnow;
static MAC_STAT macprevstat, macoldstat;
static MAC_PERF macprevperfstat;

static VOID PerfTimerInit(VOID);
static VOID PerfTimerCallback(TIMER_DESCR *timer, ULONG now, VOID *parm);

static VOID PerfMacInit(VOID);
static VOID PerfMacCallback(MAC_COLL *collector, PROT_PKT *pkt);

static VOID DisplayPerfStatistics(ULONG time, ULONG now, ULONG prevnow, ULONG corrtime,
                           MAC_STAT *stat, MAC_STAT *prevstat,
                           MAC_PERF *perfstat, MAC_PERF *prevperfstat);



BOOLEAN PerfInit(VOID)
{
SHORT       ifacenr;
CHAR        *modestr;

    if (ConfigGetShort("beholder.performance.iface", &ifacenr) == FALSE)
        ifacenr = PERFORMANCE_IFACE;
    if (ConfigGetLong("beholder.performance.interval", &interval) == FALSE)
        interval = PERFORMANCE_INTERVAL;
    if (ConfigGetString("beholder.performance.mode", &modestr) == FALSE)
        mode = PERF_TIMESAMPLE;
    else
    {
        if (stricmp(modestr, "time") == 0)
            mode = PERF_TIMESAMPLE;
        else
        if (stricmp(modestr, "packet") == 0)
            mode = PERF_PACKETSAMPLE;
        else
            mode = PERF_TIMESAMPLE;
    }

    if (interval <= 0)
    {
        DnpapMessage(DMC_WARNING, PERF_INTERVALFAIL, "perf: invalid interval value");
        return FALSE;
    }

    if ((iface = MacIfaceGet((WORD)ifacenr)) == NULL)
    {
        DnpapMessage(DMC_WARNING, PERF_NETFAIL, "perf: performance network initialization failed");
        return FALSE;
    }

    if (MacPerfSwitch(TRUE) == FALSE)
    {
        DnpapMessage(DMC_WARNING, PERF_SWITCHFAIL, "perf: performance statistics gathering initialisation failed");
        return FALSE;
    }

    if (mode == PERF_TIMESAMPLE)
    {
        if ((timer = TimerRegister(PerfTimerCallback, NULL, interval, TIMER_FOREVER, TIMER_TYPE_SKIP)) == NULL)
        {
            DnpapMessage(DMC_WARNING, PERF_TIMERFAIL, "perf: performance timer initialization failed");
            MacPerfSwitch(FALSE);
            return FALSE;
        }
        PerfTimerInit();
    }

    if (mode == PERF_PACKETSAMPLE)
    {
        coll.Rcve       = PerfMacCallback;
        coll.specific   = NULL;
        if (!MacCollRegister(&coll))
        {
            DnpapMessage(DMC_WARNING, PERF_MACBACKFAIL, "perf: network callback initialisation failed");
            TimerRemove(timer);
            MacPerfSwitch(FALSE);
            return FALSE;
        }
        PerfMacInit();
    }

    return TRUE;
}


VOID PerfTimerInit(VOID)
{
    timerprevnow = timeroldnow = 0;
    MacStatistics(iface, &timeroldstat);
    memset(&timerprevstat, 0, sizeof(MAC_STAT));
    MacPerfStatistics(&timerprevperfstat);
}


VOID PerfTimerCallback(TIMER_DESCR *timer, ULONG now, VOID *param)
{
MAC_STAT stat;
MAC_PERF perfstat;

    MacStatistics(iface, &stat);

    MacPerfStatistics(&perfstat);

    /*  baselining  */
    stat.LostPkts -= timeroldstat.LostPkts;
    stat.Pkts -= timeroldstat.Pkts;
    stat.DiscardedPkts -= timeroldstat.DiscardedPkts;
    stat.TruncatedPkts -= timeroldstat.TruncatedPkts;
    stat.StackedPkts -= timeroldstat.StackedPkts;

    DisplayPerfStatistics(now-timeroldnow, now, timerprevnow, 0,
                          &stat, &timerprevstat,
                          &perfstat, &timerprevperfstat);

    memcpy(&timerprevstat, &stat, sizeof(MAC_STAT));
    memcpy(&timerprevperfstat, &perfstat, sizeof(MAC_PERF));

    timerprevnow = now;
}


VOID PerfMacInit(VOID)
{
    macprevnow = macoldnow = 0;
    MacStatistics(iface, &macoldstat);
    memset(&macprevstat, 0, sizeof(MAC_STAT));
    MacPerfStatistics(&macprevperfstat);
}


VOID PerfMacCallback(MAC_COLL *coll, PROT_PKT *pkt)
{
ULONG now;

MAC_STAT stat;
MAC_PERF perfstat;

static LONG packets = 0;
static ULONG localtime = 0;

    if (++packets >= interval)
    {
        now = TimerNow();

        MacStatistics(iface, &stat);

        MacPerfStatistics(&perfstat);

        /*  baselining  */
        stat.LostPkts -= macoldstat.LostPkts;
        stat.Pkts -= macoldstat.Pkts;
        stat.DiscardedPkts -= macoldstat.DiscardedPkts;
        stat.TruncatedPkts -= macoldstat.TruncatedPkts;
        stat.StackedPkts -= macoldstat.StackedPkts;

        DisplayPerfStatistics(now-macoldnow, now, macprevnow, localtime,
                              &stat, &macprevstat,
                              &perfstat, &macprevperfstat);

        memcpy(&macprevstat, &stat, sizeof(MAC_STAT));
        memcpy(&macprevperfstat, &perfstat, sizeof(MAC_PERF));

        macprevnow = now;

        packets = 0;

        localtime = TimerNow()-now;
    }
}


VOID DisplayPerfStatistics(ULONG time, ULONG now, ULONG prevnow, ULONG corrtime,
                           MAC_STAT *stat, MAC_STAT *prevstat,
                           MAC_PERF *perfstat, MAC_PERF *prevperfstat)
{
ULONG    perfpkts, perfoctets, perftime;
ULONG    dtime;

    dtime = now - prevnow;

    if (perfstat->on)
    {
        perfpkts = perfstat->pkts - prevperfstat->pkts;
        perfoctets = perfstat->octets - prevperfstat->octets;
        perftime = perfstat->timeTotal - prevperfstat->timeTotal;
    }

    DnpapMessage(DMC_MESSAGE, PERF_MESSAGE,
                 "\n"
                 "Time (ms):     %10lu   Delta time (ms):    %10ld\n\n"
                 "LostPkts:      %10lu   LostPkts/s:         %10.2f\n"
                 "Pkts:          %10lu   Pkts/s:             %10.2f\n"
                 "DiscardedPkts: %10lu   DiscardedPkts/s:    %10.2f\n"
                 "TruncatedPkts: %10lu   TruncatedPkts/s:    %10.2f\n"
                 "BufferedPkts:  %10lu\n"
                 "StackedPkts:   %10lu   StackedPkts/s:      %10.2f\n\n",
                 time,                ((LONG)dtime),
                 stat->LostPkts,      ((FLOAT)stat->LostPkts-prevstat->LostPkts)/dtime*1000,
                 stat->Pkts,          ((FLOAT)stat->Pkts-prevstat->Pkts)/dtime*1000,
                 stat->DiscardedPkts, ((FLOAT)stat->DiscardedPkts-prevstat->DiscardedPkts)/dtime*1000,
                 stat->TruncatedPkts, ((FLOAT)stat->TruncatedPkts-prevstat->TruncatedPkts)/dtime*1000,
                 stat->BufferedPkts,
                 stat->StackedPkts,   ((FLOAT)stat->StackedPkts-prevstat->StackedPkts)/dtime*1000);

    if (perfstat->on)
        DnpapMessage(DMC_MESSAGE, PERF_MESSAGE,
                 "\n"
                 "PerfPkts:      %10lu   Delta PerfPkts:     %10ld\n"
                 "                            PerfPkts/s:         %10.2f\n"
                 "PerfOctets:    %10lu   Delta PerfOctets:   %10ld\n"
                 "                            PerfOctets/s:       %10.2f\n"
                 "PerfTimeTotal: %10lu   Delta PerfTimeTotal:%10ld\n"
                 "PerfTimeMin:   %10lu   Delta PerfTimeMin:  %10ld\n"
                 "PerfTimeMax:   %10lu   Delta PerfTimeMax:  %10ld\n\n"
                 "Average announce time: %10.2f\n\n",
                 perfstat->pkts,      perfpkts, ((FLOAT)perfpkts)/dtime*1000,
                 perfstat->octets,    perfoctets, ((FLOAT)perfoctets)/dtime*1000,
                 perfstat->timeTotal, perftime,
                 perfstat->timeMin,   ((LONG)perfstat->timeMin-prevperfstat->timeMin),
                 perfstat->timeMax,   ((LONG)perfstat->timeMax-prevperfstat->timeMax),
                 perfpkts == 0UL ? 0.0 : ((FLOAT)(perftime - 1000*corrtime))/perfpkts);
}


#else


#include <dnpap.h>
#include <mac.h>

#include "perf.h"


BOOLEAN PerfInit(VOID)
{
    if (MacPerfSwitch(FALSE) == FALSE)
    {
        DnpapMessage(DMC_WARNING, PERF_SWITCHFAIL, "perf: performance statistics gathering initialisation failed");
        return FALSE;
    }

    return TRUE;
}


#endif



