/**********************************************************************
** MODULE INFORMATION*
**********************
**      FILE     NAME:       maxmem.c
**      SYSTEM   NAME:       maxmem
**      ORIGINAL AUTHOR(S):  Alfred Kayser
**      VERSION  NUMBER:     1.0
**      CREATION DATE:       91/09/16
**
** DESCRIPTION: Memory functions.
***********************************************************************
** CHANGES INFORMATION **
*************************
** REVISION:   $Revision: $  
** AUTHOR:     $Author: $ 
** DATE:       $Date: $
** LOG:        $Log: $
**********************************************************************/

#define LIBRARY     /*  for #ifdef LIBRARY in dnpap.h  */

#include <limits.h>
#include <stdio.h>
#include <stdlib.h>
#include <malloc.h>
#include <dnpap.h>
#include <config.h>

#include "maxmem.h"
	

/* Default configuration */

#define MEMDEFMAX   LONG_MAX
#define MEMDEFCHUNK (UINT_MAX-100)
#define MAGIC2      0x15786179L
#define MAGIC3      0xABBADEDEL
#define TRICK(p,s)  (*(LONG*)((char*)p+s-sizeof(LONG)))


/* Prototypes for privat functions */

PRIVAT VOID *MemoryMalloc(ULONG size);
PRIVAT VOID *MemoryRealloc(VOID *memblock, ULONG size);
PRIVAT VOID  MemoryFree(VOID *memblock);


/* Local globals */

EXPORT LONG maxmemChunk = MEMDEFCHUNK; /* Max size under OS2 1.x */
EXPORT LONG maxmemMax   = MEMDEFMAX;   /* Unlimited! */
EXPORT LONG maxmemUsed  = 0;           /* Nothing used yet */


/* Export functions */

/**************************************************************
** NAME:        MemoryInit                                [API]
** SYNOPSIS:    VOID MemoryInit(LONG maxMem, LONG chunkSize)
** DESCRIPTION: Initializes the memory system.
**              This function installs a new DnpapMalloc,
**              DnpapRealloc and DnpapFree. The new functions
**              perform some sanity checks and restrict the
**              maximum amount of allocatable memory (maxMem)
**              and largest chunk of memory (chuckSize).
** RETURNS:     VOID
**************************************************************/
EXPORT VOID 
MemoryInit(LONG maxMem, LONG chunkSize)
{
    if (maxMem <= 0)
    {
        ConfigGetLong("maxmem.max", &maxMem);
        if (maxMem < 10000)
        {
            DnpapMessage(DMC_ERROR, MAXMEM_LOWMAX, "maxmem: maxmem.max < 10000, using default value");
            maxMem = 0;
        }
    }

    if (chunkSize <= 0)
    {
        ConfigGetLong("maxmem.chunk", &chunkSize);
        if (chunkSize < 100)
        {
            DnpapMessage(DMC_ERROR, MAXMEM_LOWCHUNK, "maxmem: maxmem.chunk < 100, using default value");
            chunkSize = 0;
        }
    }

    maxmemMax   = maxMem    ? maxMem    : MEMDEFMAX;
    maxmemChunk = chunkSize ? chunkSize : MEMDEFCHUNK;

    if (maxmemUsed > maxmemMax)
        DnpapMessage(DMC_FATAL, MEMORY_ERROR+1,
            "More memory (%ld bytes) was allocated than is allowed (%ld)",
                maxmemUsed, maxmemMax);
    DnpapSetMallocFun(MemoryMalloc);
    DnpapSetReallocFun(MemoryRealloc);
    DnpapSetFreeFun(MemoryFree);
}

/**************************************************************
** NAME:        MemoryGetAvail                            [API]
** SYNOPSIS:    LONG MemoryGetAvail()
** DESCRIPTION: Returns the max available memory.
** RETURNS:     LONG
**************************************************************/
EXPORT LONG
MemoryGetAvail()
{
    return maxmemMax - maxmemUsed;
}

/**************************************************************
** NAME:        MemoryGetMaxChunk                         [API]
** SYNOPSIS:    LONG MemoryGetMaxChunk()
** DESCRIPTION: Returns maximum allocatable size of memory.
** RETURNS:     LONG
**************************************************************/
EXPORT LONG
MemoryGetMaxChunk()
{
    return maxmemChunk - 2*sizeof(LONG);
}

/**************************************************************
** NAME:        MemoryGetMaxMemory                        [API]
** SYNOPSIS:    LONG MemoryGetMaxMemory()
** DESCRIPTION: Returns maximum allocatable amount of memory.
** RETURNS:     VOID
**************************************************************/
EXPORT LONG
MemoryGetMaxMemory()
{
    return maxmemMax - 2*sizeof(LONG);
}


PRIVAT VOID *
MemoryMalloc(ULONG siz)
{
    LONG *p;
    LONG size = siz;

    if (siz>(ULONG)maxmemChunk)
    {
        DnpapMessage(DMC_FATAL, MEMORY_ERROR+2,
            "Malloc: request is too large (size=%ld, max=%ld)",
                siz, maxmemChunk);
        return NULL;
    }
    if (maxmemUsed+size>maxmemMax)
    {
        DnpapMessage(DMC_FATAL, MEMORY_ERROR+3,
            "Malloc: out of memory (max=%ld, used=%ld, requested=%ld)",
                maxmemMax, maxmemUsed, siz);
        return NULL;
    }
    size = ((size+3)&(~3)) + 2*sizeof(LONG);
    p = malloc((size_t)size);
    if (p)
    {
        p[0] = ((LONG)size)<<8;
        TRICK(p,size) = MAGIC2;
        p++;
    }
    if (p)
        maxmemUsed += size;
    else
        DnpapMessage(DMC_FATAL, MEMORY_ERROR+4,
            "Malloc: failed to allocate %ld bytes", size);
    return (VOID *)p;
}
			 
   
PRIVAT VOID *
MemoryRealloc (VOID *memblock, ULONG siz)
{
    LONG oldSize;
    LONG size = siz;
    LONG *p;

    if (memblock==NULL)
        return MemoryMalloc(size);

    if (size==0)
    {
        MemoryFree(memblock);
        return NULL;
    }
    p=(LONG*)memblock;
    p--;
    oldSize = p[0]>>8;
    if (p[0]==MAGIC3)
    {
        DnpapMessage(DMC_FATAL, MEMORY_ERROR+12,
            "DnpapRealloc: memory at %lp was already freed", memblock);
        return NULL;
    }
    if (oldSize>maxmemChunk || (p[0]&0xFF))
    {
        DnpapMessage(DMC_FATAL, MEMORY_ERROR+7,
            "DnpapRealloc: size field before data was overwritten!");
        return NULL;
    }

    if (size>oldSize)
    {
        if (size>maxmemChunk)
        {
            DnpapMessage(DMC_FATAL, MEMORY_ERROR+5,
                "Malloc: request is too large (size=%ld)", size);
            return NULL;
        }
        if (maxmemUsed+size-oldSize>maxmemMax)
        {
            DnpapMessage(DMC_FATAL, MEMORY_ERROR+6,
                "Malloc: out of memory (max=%ld, used=%ld, requested=%ld)",
                    maxmemMax, maxmemUsed, size);
            return NULL;
        }
    }
    size = ((size+3)&(~3)) + 2*sizeof(LONG);
    if (TRICK(p,oldSize)!=MAGIC2)
        DnpapMessage(DMC_FATAL, MEMORY_ERROR+9,
            "DnpapRealloc: MAGIC word after data was overwritten!");
    p = realloc(p, (size_t)size);
    if (p)
    {
        maxmemUsed += size-oldSize;
        p[0] = ((LONG)size)<<8;
        TRICK(p,size) = MAGIC2;
        p++;
    }
    else
        DnpapMessage(DMC_FATAL, MEMORY_ERROR+7,
            "Malloc: failed to reallocate %ld bytes", size);
    return (VOID*)p;
}


PRIVAT VOID  
MemoryFree (VOID *memblock)                
{
    LONG size;
    LONG *p;

    if (memblock==NULL) return;
    p = (LONG *)memblock;
    p--;
    size = p[0]>>8;
    if (p[0]==MAGIC3)
    {
        DnpapMessage(DMC_FATAL, MEMORY_ERROR+12,
            "DnpapFree: memory at %lp was already freed", memblock);
        return;
    }

    if (size>maxmemChunk || (p[0]&0xFF))
    {
        DnpapMessage(DMC_FATAL, MEMORY_ERROR+10,
            "DnpapFree: size field before data was overwritten!");
        p[0] = MAGIC3;  /* Make sure it isn't freed again! */
        free(p);
        return;
    }

    if (TRICK(p,size) != MAGIC2)
        DnpapMessage(DMC_FATAL, MEMORY_ERROR+11,
            "DnpapFree: MAGIC word after data was overwritten!");
    maxmemUsed -= size;
    p[0] = MAGIC3;  /* Make sure it isn't freed again! */
    free(p);
}
