/************************************************************************
** MODULE INFORMATION*
**********************
**     FILE     NAME:       parse.c
**     SYSTEM   NAME:       mib
**     ORIGINAL AUTHOR(S):  Jan van Oorschot
**     VERSION  NUMBER:     1.00
**     CREATION DATE:       
**
** DESCRIPTION: parse an output file of mossy 
**              
*************************************************************************
** CHANGES INFORMATION **
*************************
** REVISION:    $Revision$
** WORKFILE:    $Workfile$
** LOGINFO:     $Log$
*************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <dnpap.h>
#include "mibinc.h"

typedef struct _dentry DENTRY;

struct _dentry
{
    char *name;
    int value;
};

static DENTRY mibType[] =
{
  {"TimeTicks",7},
  {"Counter",5},
  {"Aggregate",16},
  {"INTEGER", 1},
  {"Gauge",6},
  {"OctetString",2},
  {"DisplayString",2},		/* recognise DisplayString */
  {"ObjectID",3},
  {"NetworkAddress",18},
  {"IpAddress",4},
  {NULL,0}
};

static DENTRY mibMode[] =
{
  {"read-only",1},
  {"write-only",2},
  {"read-write",3},
  {"not-accessible",0},
  {NULL, 0}
};

static DENTRY mibStatus[] = 
{
  {"mandatory",1},
  {"deprecated",2},
  {"optional",3},
  {NULL,0}
};


MIBNODE *MibCreateNode(const char *name, char *position)
{
    char *iPtr, *fPtr,*ficName,fname[254];
    int index, dotCntr;
    MIBNODE *node,*fNode;

    /* first get the index from name and the father's name */
    for(iPtr=fPtr=position,dotCntr=0; *fPtr!='\0';fPtr++)
      if(*fPtr == '.') 
	{
	    iPtr = fPtr;
	    dotCntr++;
	}
    
    if(dotCntr==0)
      return NULL;
    index = atoi(iPtr+1);
    strncpy(fname, position, (iPtr - position));
    *(fname + (iPtr - position)) = '\0';

    /* try to get the pointer to the father node */
    fNode = MibFindNode(fname);
    if(fNode == NULL)
    {
	/* father doesn't exist */
	/* if there are more than one dot's, resolve it recursively */
	if(dotCntr>1)
	{
	    /* more dot's, create the father list using fictive names */
	    ficName = DnpapMalloc(strlen(fname)+1);
	    strcpy(ficName,fname);
	    fNode = MibCreateNode(ficName,fname);
	    if( fNode == NULL)
	    {
		free(ficName);
		return NULL;
	    }
	}
	else
	{
	    DnpapMessage(DMC_WARNING,1,"father <%s> couldn't be found\n",fname);
	    return NULL;
	}
    }


    /* we now have a pointer to father, a name and an index, create node */
    node =  DnpapMalloc( sizeof(MIBNODE));
    node->name = name;
    node->index = index;
    if( MibAddNode(node,fNode) != 0)
    {
	free(node);
	return NULL;
    }

    return node;
}

MIBNODE *MibScanLine(char *line)
{
    int nrChunks,clens[10];
    char *chunks[10];
    MIBNODE * node;
    int type,mode,status;
    char name[64],father[64],value[64],*cPtr;

    /* 
      
       The line should be a valid line from Mosy output. This restricts
       the output to the following:
       
       . blank line
       . comment line
       . declare line (name + father.index, no attributes)
       . full line (name + father.index + attributes)
    */

    nrChunks = MibFindChunks(line,chunks,clens);

    node = NULL;
    type = 0;
    mode = 0;
    status = 0;
    switch(nrChunks)
    {
	case 0: 
	  /* no chunks in this line, nothing to do */
	  break;
 	case 5:
	  /* full declaration, get parameters */
	  strncpy(value,chunks[2],clens[2]);
	  *(value + clens[2]) = '\0';
	  type = MibDecodeType(value);
	  strncpy(value,chunks[3],clens[3]);
	  *(value + clens[3]) = '\0';
	  mode = MibDecodeMode(value);
	  strncpy(value,chunks[4],clens[4]);
	  *(value + clens[4]) = '\0';
	  status = MibDecodeStatus(value);

	case 2:
	  /* declaration, dummy values */
	  strncpy(name,chunks[0],clens[0]);
	  *(name+clens[0]) = '\0';
	  strncpy(father,chunks[1],clens[1]);
	  *(father+clens[1]) = '\0';
	  /* create a new node struct */
	  cPtr = DnpapMalloc(strlen(name)+1);
	  strcpy(cPtr,name);
	  node = MibCreateNode(cPtr,father);
	  if(node == NULL)
	    free(cPtr);
	  break;
	default:
	  /* wrong number of args, ignore line */
	  break;
    }
    /* fill node with data */
    if(node != NULL)
    {
	node->type = type;
	node->mode = mode;
	node->status = status;
    }
    return node;
}

int MibFindChunks(char *line,char **chunks, int *clens)
{
    char *cPtr;
    int comment,curChunk,inChunk,chunkLen;
    /* parse the line for chunks */

    cPtr = line;
    curChunk = 0;
    comment = 0;
    while (!comment)
    {
	/* eat away white-space */
	while ( isspace(*cPtr))
	  cPtr++;
	inChunk = 1;
	chunkLen = 0;
	chunks[curChunk] = cPtr;
	while(inChunk)
	{
	    switch( *cPtr )
	    {
		case '\0':
		case '\n':
		   comment = 1;
		   inChunk = 0;
		   break;
		case ' ':
		case '\t':
		  inChunk=0;
		  break;
		case '-':
		  if (*(cPtr+1) == '-')
		  {
		    comment = 1;
		    inChunk = 0;
		  }
		default:
		  cPtr++;
		  chunkLen++;
	    }
	}
	clens[curChunk] = chunkLen;
	if(chunkLen != 0 )
	  curChunk++;
    }
    return curChunk;
}
	
int MibDecodeType(char *type)
{
    int tCntr;
    
    tCntr = 0;
    while( mibType[tCntr].name != NULL)
    {
	if(stricmp(mibType[tCntr].name,type)==0)
	  return mibType[tCntr].value;
	tCntr++;
    }
    DnpapMessage(DMC_WARNING,1,"<%s> unknown Mib Type\n",type);
    return mibType[tCntr].value;
}
int MibDecodeMode(char *mode)
{
    int tCntr;
    
    tCntr = 0;
    while( mibMode[tCntr].name != NULL)
    {
	if(stricmp(mibMode[tCntr].name,mode) == 0)
	  return mibMode[tCntr].value;
	tCntr++;
    }
    DnpapMessage(DMC_WARNING,1,"<%s> unknown Mib Mode\n",mode);
    return mibMode[tCntr].value;
}

int MibDecodeStatus(char *status)
{
    int tCntr;
    
    tCntr = 0;
    while( mibStatus[tCntr].name != NULL)
    {
	if(stricmp(mibStatus[tCntr].name,status) == 0)
	  return mibStatus[tCntr].value;
	tCntr++;
    }
    DnpapMessage(DMC_WARNING,1,"<%s> unknown Mib Status\n",status);
    return mibStatus[tCntr].value;
}

char *MibEncodeStatus(int status)
{
    int tCntr;
    
    tCntr = 0;
    while( mibStatus[tCntr].name != NULL)
    {
	if(mibStatus[tCntr].value == status)
	  return mibStatus[tCntr].name;
	tCntr++;
    }
    DnpapMessage(DMC_WARNING,1,"<%d> unknown Mib Status\n",status);
    return NULL;
}	
    
char *MibEncodeMode(int mode)
{
    int tCntr;
    
    tCntr = 0;
    while( mibMode[tCntr].name != NULL)
    {
	if(mibMode[tCntr].value==mode)
	  return mibMode[tCntr].name;
	tCntr++;
    }
    DnpapMessage(DMC_WARNING,1,"<%d> unknown Mib Mode\n",mode);
    return NULL;
}	
    
char *MibEncodeType(int type)
{
    int tCntr;
    
    tCntr = 0;
    while( mibType[tCntr].name != NULL)
    {
	if(mibType[tCntr].value==type)
	  return mibType[tCntr].name;
	tCntr++;
    }
    DnpapMessage(DMC_WARNING,1,"<%d> unknown Mib Type\n",type);
    return NULL;
}	
    



