/************************************************************************
** MODULE INFORMATION **
************************
** FILE NAME:          statv.c
** SYSTEM NAME:        beholder
** MODULE NAME:        stat
** ORIGINAL AUTHOR(S): M.F.B. de Greeve
** VERSION NUMBER:     1.0
** CREATION DATE:      1992/7/14
** DESCRIPTION:        etherStat-group: viewer
*************************************************************************/
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <dnpap.h>
#include <config.h>
#include <message.h>
#include <timer.h>
#include <mib.h>
#include <mgmt.h>
#include <vipinc.h>                                        

#define COMMUNITY       "viewer"
#define COMMUNITYLEN    6
#define TOTALVAR        17
#define ETH_TIM         0x01
#define ETH_VIP         0x02

static LWORD            statInterval;
static BYTE             remoteName[MIB_SZEBUFCHR];
static BYTE             statOwner[MIB_SZEBUFCHR];
static WORD             statOwnerLen;
static TIMER_DESCR      *statTimer = NULL;
static VIPINFO          *statFrame, *mainFrame, *modeFrame;
static VIPINFO          *graph1, *graph2, *graph3, *varFrame;
static VIPINFO          *configText, *configVar;
static LWORD            statClockOld, statVarOld[TOTALVAR];
static BOOLEAN          quitFlag = FALSE, showGraph = FALSE;

static MGMT             *mgmt = NULL;
static MIB_OBJECT       varObj[] =
{
    {MIB_RQSGET, {1,3,6,1,2,1,16,1,1,1,3,1}, 12},     /* dropevents */
    {MIB_RQSGET, {1,3,6,1,2,1,16,1,1,1,4,1}, 12},     /* octets */
    {MIB_RQSGET, {1,3,6,1,2,1,16,1,1,1,5,1}, 12},     /* pkts */
    {MIB_RQSGET, {1,3,6,1,2,1,16,1,1,1,6,1}, 12},     /* broadcast */
    {MIB_RQSGET, {1,3,6,1,2,1,16,1,1,1,7,1}, 12},     /* multicast */
    {MIB_RQSGET, {1,3,6,1,2,1,16,1,1,1,8,1}, 12},     /* CRCAlign */
    {MIB_RQSGET, {1,3,6,1,2,1,16,1,1,1,9,1}, 12},     /* undersize */
    {MIB_RQSGET, {1,3,6,1,2,1,16,1,1,1,10,1}, 12},    /* oversize */
    {MIB_RQSGET, {1,3,6,1,2,1,16,1,1,1,11,1}, 12},    /* fragments */
    {MIB_RQSGET, {1,3,6,1,2,1,16,1,1,1,12,1}, 12},    /* jabbers */
    {MIB_RQSGET, {1,3,6,1,2,1,16,1,1,1,13,1}, 12},    /* collisions */
    {MIB_RQSGET, {1,3,6,1,2,1,16,1,1,1,14,1}, 12},    /* 64 */
    {MIB_RQSGET, {1,3,6,1,2,1,16,1,1,1,15,1}, 12},    /* 65-127 */
    {MIB_RQSGET, {1,3,6,1,2,1,16,1,1,1,16,1}, 12},    /* 128-255 */
    {MIB_RQSGET, {1,3,6,1,2,1,16,1,1,1,17,1}, 12},    /* 256-511 */
    {MIB_RQSGET, {1,3,6,1,2,1,16,1,1,1,18,1}, 12},    /* 512-1023 */
    {MIB_RQSGET, {1,3,6,1,2,1,16,1,1,1,19,1}, 12}     /* 1024-1518 */
};


static VOID StatConfig (VOID);
static VOID StatInit (VOID);
static VOID TimerCall (TIMER_DESCR *timer, ULONG now, VOID *parm); 
static VOID QuitCall (VIPINFO *wip, VOID *ptr, BOOLEAN f);
static VOID ViewCall (VIPINFO *wip, VOID *ptr, BOOLEAN f);
static VOID ConfigCall (VIPINFO *wip, VOID *ptr, BOOLEAN f);
static VOID ModeCall (VIPINFO *wip, VOID *ptr, BOOLEAN f);
static VOID ResetCall (VIPINFO *wip, VOID *ptr, BOOLEAN f);
static VOID DrawConfig (MIB_OBJECT *obj, WORD ind);
static VOID DrawGraph (VOID);
static VOID DrawText (VOID);
static VOID UpdateText (ULONG now);
static VOID UpdateGraph (ULONG now);

VOID main (short argc, char *argv[])
{
    LWORD availableInterface = 0;

    MessageInit("UNKNOWN", MSGFILE, MSGVIP, NULL);
    MessageConfig(STATISTICS_ERROR, "etherStat");
    if (ConfigInit(argc, argv) == FALSE || ConfigLoad() == FALSE)
        DnpapMessage(DMC_ERROR, STATISTICS_ERROR, "etherStat: Config init");
    StatConfig();
    if (TimerInit() == FALSE)
        DnpapMessage(DMC_ERROR, STATISTICS_ERROR, "etherStat: timer init");
    else
        availableInterface |= ETH_TIM;
    if (VipInit (argc, argv) == FALSE)
        DnpapMessage(DMC_ERROR, STATISTICS_ERROR, "etherStat: VIP init");
    else
        availableInterface |= ETH_VIP;
    if ((mgmt = MgmtOpen (MGMT_PIPE, remoteName)) == NULL)
    {
        DnpapMessage(DMC_ERROR, STATISTICS_ERROR, "etherStat: init %s", remoteName);
        DnpapExit(1);
    }
    StatInit();
    while (!quitFlag)
    {
        if (availableInterface & ETH_TIM)
            if (TimerCheck() < 0)
            {
                DnpapMessage(DMC_ERROR, STATISTICS_ERROR, "etherStat: timer Check");
                MgmtClose (mgmt);
                DnpapExit(1);
            }
        if (availableInterface & ETH_VIP)
            if (VipCheck(FALSE) == FALSE)
            {
                DnpapMessage(DMC_ERROR, STATISTICS_ERROR, "etherStat: VIP Check");
                MgmtClose (mgmt);
                DnpapExit(1);
            }
    }
    MgmtClose (mgmt);
    TimerExit ();
    VipExit ();
    DnpapExit(0);
}


static VOID StatConfig (VOID)
{
    BYTE *p;

    if (!ConfigGetString("beholder.pipe.remote", &p) || strlen(p) > MIB_SZEBUFCHR)
        p="\\pipe\\beholder";
    strcpy(remoteName, p);
    if (ConfigGetString("beholder.stat.interval", &p))
        statInterval = atol(p);
    else
        statInterval = 2000L;
    if (!ConfigGetString("beholder.stat.owner", &p) || strlen(p) > MIB_SZEBUFCHR)
        p="monitorETHERSTAT";
    strcpy(statOwner, p);
    statOwnerLen = strlen(p);
    return;
}


static VOID StatInit (VOID)
{
    WORD status, index;

    MIB_OBJECT obj[] =
    {
        {MIB_RQSGET, {1,3,6,1,2,1,16,1,1,1,21,1}, 12},    /* status */
        {MIB_RQSSET, {1,3,6,1,2,1,16,1,1,1,21,1}, 12, MIB_INTEGER, MIB_CREATEREQUEST},
        {MIB_RQSSET, {1,3,6,1,2,1,16,1,1,1,20,1}, 12, MIB_DISPLAYSTR}
    };

    strcpy (obj[2].Syntax.BufChr, statOwner);
    obj[2].SyntaxLen = statOwnerLen;

    statFrame = VipOpenFrame (NULL, 5, 150, 430, 670);
    mainFrame = VipOpenFrame (statFrame, 0, 0, 1000, 1000);
    VipSetSystemMenu(statFrame, FALSE);


    if (MgmtRequest(mgmt, COMMUNITY, COMMUNITYLEN, MIB_RQSGET, &obj[0], 1, &status, &index) == FALSE)
    {
        DnpapMessage(DMC_ERROR, STATISTICS_ERROR,"etherStat: error %s", remoteName);
        QuitCall(NULL, NULL, FALSE);
        return;
    }
    if (status != MIB_NOERROR)
    {
        if (MgmtRequest(mgmt, COMMUNITY, COMMUNITYLEN, MIB_RQSSET, &obj[1], 2, &status, &index) == FALSE ||
            status != MIB_NOERROR)
        {
            DnpapMessage(DMC_ERROR, STATISTICS_ERROR,"etherStat: can't initialize");
            QuitCall(NULL, NULL, FALSE);
            return;
        }
    }
    VipShow (statFrame);
    ConfigCall(NULL, NULL, FALSE);
    return;
}


static VOID ConfigCall (VIPINFO *wip, VOID *ptr, BOOLEAN f)
{
    LONG                indexId[]= {1,3,6,1,2,1,16,1,1,1,1};
    WORD                status, index;
    WORD                ind = 0, IdLen = 11;
    VIPINFO             *quitButton, *viewButton, *configFrame;
    MIB_OBJECT          obj[] =
    {
        {MIB_RQSNXT, {1,3,6,1,2,1,16,1,1,1,1}, 11},     /* index */
        {MIB_RQSNXT, {1,3,6,1,2,1,16,1,1,1,20}, 11},    /* owner */
        {MIB_RQSNXT, {1,3,6,1,2,1,16,1,1,1,21}, 11}    /* status */
    };

    if (f == FALSE) /* RELEASED */
    {
        if (statTimer != NULL)
            TimerRemove(statTimer);
        VipDestroy(mainFrame);
        mainFrame = VipOpenFrame (statFrame, 0, 0, 1000, 1000);
        configFrame = VipOpenFrame (mainFrame, 0, 0, 1000, 880);
        configText = VipOpenText (configFrame, 0, 0, 500, 1000);
        configVar = VipOpenText (configFrame, 500, 0, 500, 1000);
        quitButton = VipOpenButton (mainFrame, 0, 890, 200, 100);
        viewButton = VipOpenButton (mainFrame, 200, 890, 200, 100);
        VipSetTitle (statFrame, "etherStat: Configuration Window");
        VipSetTitle(quitButton, "quit");
        VipSetTitle(viewButton, "view");
        VipSetButtonCallBack(quitButton, QuitCall, NULL);
        VipSetButtonCallBack(viewButton, ViewCall, NULL);
        while (MgmtRequest(mgmt, COMMUNITY, COMMUNITYLEN, MIB_RQSNXT, obj, 3, &status, &index) == TRUE &&
            status == MIB_NOERROR &&
            !memcmp(obj[0].Id, indexId, IdLen * sizeof(LONG)))
            DrawConfig (obj, ind++);
        VipShow (mainFrame);
    }
    return;
}

static VOID ViewCall (VIPINFO *wip, VOID *ptr, BOOLEAN f)
{
    WORD    status, index;
    VIPINFO *quitButton, *configButton, *modeButton, *resetButton;

    MIB_OBJECT obj[] =
    {
        {MIB_RQSGET, {1,3,6,1,2,1,16,1,1,1,21,1}, 12},
        {MIB_RQSSET, {1,3,6,1,2,1,16,1,1,1,21,1}, 12, MIB_INTEGER, MIB_VALID}
    };

    if (f == FALSE) /* RELEASED */
    {
        VipDestroy(mainFrame);
        mainFrame = VipOpenFrame (statFrame, 0, 0, 1000, 1000);
        modeFrame = VipOpenFrame (mainFrame, 0, 0, 1000, 880);
        quitButton = VipOpenButton (mainFrame, 0, 890, 200, 100);
        configButton = VipOpenButton (mainFrame, 200, 890, 200, 100);
        modeButton = VipOpenButton (mainFrame, 400, 890, 200, 100);
        resetButton = VipOpenButton (mainFrame, 600, 890, 200, 100);
        VipSetTitle(quitButton, "quit");
        VipSetTitle(configButton, "config");
        VipSetTitle(modeButton, "mode");
        VipSetTitle(resetButton, "reset");
        VipSetButtonCallBack(quitButton, QuitCall, NULL);
        VipSetButtonCallBack(configButton, ConfigCall, NULL);
        VipSetButtonCallBack(modeButton, ModeCall, NULL);
        VipSetButtonCallBack(resetButton, ResetCall, ptr);
        VipShow (mainFrame);
        if (MgmtRequest(mgmt, COMMUNITY, COMMUNITYLEN, MIB_RQSGET, &obj[0], 1, &status, &index) == FALSE)
            {
                DnpapMessage(DMC_ERROR, STATISTICS_ERROR,"etherStat: error %s", remoteName);
                QuitCall(NULL, NULL, FALSE);
                return;
            }
        if (status == MIB_NOERROR && obj[0].Syntax.LngInt == MIB_UNDERCREATION)
        {
            if (MgmtRequest(mgmt, COMMUNITY, COMMUNITYLEN, MIB_RQSSET, &obj[1], 1, &status, &index) == FALSE ||
                status != MIB_NOERROR)
            {
                DnpapMessage(DMC_ERROR, STATISTICS_ERROR,"etherStat: can't set status to valid");
                QuitCall(NULL, NULL, FALSE);
                return;
            }
        }
        if ((statTimer = TimerRegister(TimerCall, NULL, statInterval, 
            TIMER_FOREVER,TIMER_TYPE_SKIP)) == NULL)
        {
            DnpapMessage(DMC_ERROR, STATISTICS_ERROR,"etherStat: can't register timer");
            QuitCall(NULL, NULL, FALSE);
            return;
        }
        ResetCall (NULL, NULL, FALSE);
    }
    return;
}


static VOID QuitCall (VIPINFO *wip, VOID *ptr, BOOLEAN f)
{
    if (f == FALSE) /* RELEASED */
    {
        if (statTimer != NULL)
            TimerRemove(statTimer);
        VipDestroy (statFrame);
        quitFlag = TRUE;
    }
    return;
}

static VOID ResetCall (VIPINFO *wip, VOID *ptr, BOOLEAN f)
{
    WORD status, index;
    WORD ind;

    if (f == FALSE) /* RELEASED */
    {
        VipDestroy (modeFrame);
        modeFrame = VipOpenFrame (mainFrame, 0, 0, 1000, 880);
        if (showGraph == TRUE)
            DrawGraph();
        else
            DrawText();
        if (MgmtRequest(mgmt, COMMUNITY, COMMUNITYLEN, MIB_RQSGET, &varObj[0], 8, &status, &index) == FALSE ||
            status != MIB_NOERROR ||
            MgmtRequest(mgmt, COMMUNITY, COMMUNITYLEN, MIB_RQSGET, &varObj[8], 9, &status, &index) == FALSE ||
            status != MIB_NOERROR)
        {
            DnpapMessage(DMC_ERROR, STATISTICS_ERROR,"etherStat: can't get data");
            QuitCall(NULL, NULL, FALSE);
            return;
        }
        for (ind = 0; ind < TOTALVAR; ind++)
            statVarOld[ind] = varObj[ind].Syntax.LngUns;
        statClockOld = TimerNow();
        VipShow (modeFrame);
    }
    return;
}

static VOID ModeCall (VIPINFO *wip, VOID *ptr, BOOLEAN f)
{
    if (f == FALSE)
    {
        showGraph = 1 - showGraph;
        ResetCall (NULL, NULL, FALSE);
    }
    return;
}


static VOID TimerCall(TIMER_DESCR *timer, ULONG now, VOID *parm)
{
    WORD status, index;

    if (MgmtRequest(mgmt, COMMUNITY, COMMUNITYLEN, MIB_RQSGET, &varObj[0], 8, &status, &index) == FALSE ||
        status != MIB_NOERROR ||
        MgmtRequest(mgmt, COMMUNITY, COMMUNITYLEN, MIB_RQSGET, &varObj[8], 9, &status, &index) == FALSE ||
        status != MIB_NOERROR)
    {
        DnpapMessage(DMC_ERROR, STATISTICS_ERROR,"etherStat: can't get data");
        QuitCall(NULL, NULL, FALSE);
        return;
    }
    if (showGraph == TRUE)
        UpdateGraph(now);
    else
        UpdateText(now);
    return;
}


static VOID UpdateText (ULONG now)
{
    WORD            ind;
    LWORD           current;
    BYTE            str[MIB_SZEBUFCHR];

    for (ind = 0; ind < TOTALVAR; ind++)
    {
        current = varObj[ind].Syntax.LngUns - statVarOld[ind];
        sprintf(str, "..............................................................%lu", current);
        VipSetTextLine (varFrame, str, ind, VIP_RIGHT);
    }
    VipUpdate(varFrame, TRUE);
    return;
}

static VOID UpdateGraph (ULONG now)
{
    WORD ind;
    LWORD current, sec;

    sec = 1 + (now - statClockOld) / 1000L;
    statClockOld = now;
    for (ind = 0; ind < TOTALVAR; ind++)
    {
        current = varObj[ind].Syntax.LngUns - statVarOld[ind];
        statVarOld[ind] = varObj[ind].Syntax.LngUns;
        if (ind > 0)
        {
            if (ind < 5)
                VipSetGraphLast(graph2, ind - 1, (long) (current/sec));
            else if (ind < 11)
                VipSetGraphLast(graph1, ind - 4, (long) (current/sec));
            else
                VipSetGraphLast(graph3, ind - 11, (long) (current/sec));
        }
        else
            VipSetGraphLast(graph1, 0, (long) (current/sec));
    }
    VipUpdate(modeFrame, FALSE);
    return;
}

static VOID DrawGraph (VOID)
{
    VipSetTitle (statFrame, "etherStat: Graph Window");
    graph1 =      VipOpenGraph (modeFrame, 0, 600, 1000, 400);
        /* Drop, CRC, Under, Over, Fragment, Jabber, Collision */
    graph2 =       VipOpenGraph (modeFrame, 0, 300, 1000, 300);
        /* Pkt, Octet, Brc, Multi */
    graph3 =        VipOpenGraph (modeFrame, 0, 0, 1000, 300);
        /* 64, 128, 256, 512, 1024 */

    VipSetBackground (graph1, VIP_WHITE);
    VipSetBackground (graph2, VIP_WHITE);
    VipSetBackground (graph3, VIP_WHITE);
    VipSetGraphDataSet (graph1, 6, 50, 0, 100, "Collision", VIP_PINK);
    VipSetGraphDataSet (graph1, 5, 50, 0, 100, "Jabber", VIP_BLUE);
    VipSetGraphDataSet (graph1, 4, 50, 0, 100, "Fragment", VIP_CYAN);
    VipSetGraphDataSet (graph1, 3, 50, 0, 100, "OverSize", VIP_GREEN);
    VipSetGraphDataSet (graph1, 2, 50, 0, 100, "UnderSize", VIP_BROWN);
    VipSetGraphDataSet (graph1, 1, 50, 0, 100, "CRCAlign", VIP_DARKRED);
    VipSetGraphDataSet (graph1, 0, 50, 0, 500, "DropEvent", VIP_RED);
    VipSetGraphDataSet (graph2, 3, 50, 0, 500, "Multicast", VIP_PINK);
    VipSetGraphDataSet (graph2, 2, 50, 0, 500, "Broadcast", VIP_RED);
    VipSetGraphDataSet (graph2, 1, 50, 0, 500, "Pkt", VIP_DARKRED);
    VipSetGraphDataSet (graph2, 0, 50, 0, 250000L, "Octet", VIP_BLUE);
    VipSetGraphDataSet (graph3, 5, 50, 0, 200, "1024-1518Pkt", VIP_BROWN);
    VipSetGraphDataSet (graph3, 4, 50, 0, 200, "512-1023Pkt", VIP_CYAN);
    VipSetGraphDataSet (graph3, 3, 50, 0, 200, "256-511Pkt", VIP_PINK);
    VipSetGraphDataSet (graph3, 2, 50, 0, 200, "128-255Pkt", VIP_DARKRED);
    VipSetGraphDataSet (graph3, 1, 50, 0, 200, "65-127Pkt", VIP_RED);
    VipSetGraphDataSet (graph3, 0, 50, 0, 200, "64Pkt", VIP_BLUE);
    return;
}

static VOID DrawText (VOID)
{
    WORD        ind;
    VIPINFO     *textFrame;

    VipSetTitle (statFrame, "etherStat: Text Window");                                                 
    textFrame = VipOpenText (modeFrame, 0, 0, 1000, 1000);
    varFrame = VipOpenText (modeFrame, 700, 0, 300, 1000);
    VipSetTextLine (textFrame, "DropEvents...................................................................................................", 0, VIP_LEFT);
    VipSetTextLine (textFrame, "Octets.......................................................................................................", 1, VIP_LEFT);
    VipSetTextLine (textFrame, "Pkts.........................................................................................................", 2, VIP_LEFT);
    VipSetTextLine (textFrame, "BroadcastPkts................................................................................................", 3, VIP_LEFT);
    VipSetTextLine (textFrame, "MulticastPkts................................................................................................", 4, VIP_LEFT);
    VipSetTextLine (textFrame, "CRCAlignErrors...............................................................................................", 5, VIP_LEFT);
    VipSetTextLine (textFrame, "UndersizePkts................................................................................................", 6, VIP_LEFT);
    VipSetTextLine (textFrame, "OversizePkts.................................................................................................", 7, VIP_LEFT);
    VipSetTextLine (textFrame, "Fragments....................................................................................................", 8, VIP_LEFT);
    VipSetTextLine (textFrame, "Jabbers......................................................................................................", 9, VIP_LEFT);
    VipSetTextLine (textFrame, "Collisions...................................................................................................", 10, VIP_LEFT);
    VipSetTextLine (textFrame, "Pkts64Octets.................................................................................................", 11, VIP_LEFT);
    VipSetTextLine (textFrame, "Pkts65to127Octets............................................................................................", 12, VIP_LEFT);
    VipSetTextLine (textFrame, "Pkts128to255Octets...........................................................................................", 13, VIP_LEFT);
    VipSetTextLine (textFrame, "Pkts256to511Octets...........................................................................................", 14, VIP_LEFT);
    VipSetTextLine (textFrame, "Pkts512to1023Octets..........................................................................................", 15, VIP_LEFT);
    VipSetTextLine (textFrame, "Pkts1024to1518Octets.........................................................................................", 16, VIP_LEFT);
    for (ind = 0; ind < TOTALVAR; ind++)
    {
        VipSetTextLine (varFrame, "..........................................................................................................0", ind, VIP_RIGHT);
    }
    return;
}

static VOID DrawConfig (MIB_OBJECT *obj, WORD ind)
{
    BYTE                str[MIB_SZEBUFCHR];

    VipSetTextLine (configText, "Index:........................................................................................................", 0 + ind * 4, VIP_LEFT);
    VipSetTextLine (configText, "Owner:........................................................................................................", 1 + ind * 4, VIP_LEFT);
    VipSetTextLine (configText, "Status:.......................................................................................................", 2 + ind * 4, VIP_LEFT);
    sprintf(str, "..........................................................................................................................%ld", obj[0].Syntax.LngInt);
    VipSetTextLine (configVar, str, 0 + ind * 4, VIP_RIGHT);
    if (obj[1].SyntaxLen > 0)
        sprintf(str, "...........................................................................................................................%s", obj[1].Syntax.BufChr);
    else
        sprintf(str, "...........................................................................................................................");
    VipSetTextLine (configVar, str, 1 + ind * 4, VIP_RIGHT);
    switch (obj[2].Syntax.LngInt)
    {
        case MIB_INVALID:
            VipSetTextLine (configVar, "................................................................................................INVALID", 2 + ind * 4, VIP_RIGHT);
            break;
        case MIB_VALID:
            VipSetTextLine (configVar, "..................................................................................................VALID", 2 + ind * 4, VIP_RIGHT);
            break;
        case MIB_UNDERCREATION:
            VipSetTextLine (configVar, "..........................................................................................UNDERCREATION", 2 + ind * 4, VIP_RIGHT);
            break;
        default:
            VipSetTextLine (configVar, "................................................................................................UNKNOWN", 2 + ind * 4, VIP_RIGHT);
    }
    return;
}

