unit ui;

interface

uses
  Crt,
  DOS,
  List;

const
  MaxItems  = 8;

type
  TBounds   = record
                x, y: Byte;
              end;
  TListBox  = object
                _min,
                _max: TBounds;
                _items: TSortedList;
                _caption: String;
                _first,
                _last,
                _index: Word;
                procedure init(x1, y1, x2, y2: Byte; caption: String);
                procedure draw;
                procedure getListFromDir(path, mask: String);
                procedure getListFromPassage(fname, passage: String);
                procedure getListFromFile(fname: String);
                procedure getListFromString(list: String; sep: Char);
                function  selected: String;
                function  index: Word;
                procedure clearList;
              end;

  TEditBox  = object
                _row,
                _first,
                _last,
                _index: Byte;
                _drawn: Boolean;
                procedure init(row: Byte);
                procedure draw(str: String);
                procedure prompt(var str: String);
              end;

  TTextView = object
                _min,
                _max: TBounds;
                _items: TList;
                _caption: String;
                _first,
                _last,
                _size: Word;
                procedure init(x1, y1, x2, y2: Byte; caption: String);
                procedure draw;
                procedure getListFromFile(fname: String);
                procedure getListFromString(list: String; sep: Char);
                procedure show;
                procedure clearList;
              end;

  TMenue    = object
                _items: TList;
                _numItems: Byte;
                _sel: Byte;
                procedure init(list: String; sep: Char);
                procedure done;
                procedure draw;
                procedure show(var selection: Byte);
              end;

implementation

const
  UpArrow   = '';
  DownArrow = '';
  kbCtrl    = #0;
  kbCursor  = #224;
  kbUp      = #72;
  kbDown    = #80;
  kbLeft    = #75;
  kbRight   = #77;
  kbPgUp    = #73;
  kbPgDn    = #81;
  kbHome    = #71;
  kbEnd     = #79;
  kbEnter   = #13;
  kbESC     = #27;
  kbDel     = #83;
  kbBackSpc = #8;

procedure HideCursor;
var regs: Registers;
begin
  regs.ax := $100;
  regs.cX := 8192;;
  Intr($10,Regs);
end;

procedure ShowCursor;
var regs: Registers;
begin
  regs.ax := $100;
  regs.cX := 2828;;
  Intr($10,Regs);
end;

procedure HighlightOn;
begin
  TextColor(Black); TextBackground(LightGray);
end;

procedure HighLightOff;
begin
  TextColor(LightGray); TextBackground(Black);
end;


{*** TListBox ***}

procedure TListBox.init(x1, y1, x2, y2: Byte; caption: String);
begin
  _min.x:=x1; _min.y:=y1;
  _max.x:=x2; _max.y:=y2;
  _caption:=caption;
  _index:=1;
  _first:=1;
  _last:=_max.y - _min.y - 1;
end;

procedure TListBox.draw;
var x, y, maxX, maxY: Byte;
    fforward: Word;
begin
  window(_min.x, _min.y, _max.x, _max.y);
  HighlightOn;
  ClrScr;
  maxX:=_max.x - _min.x + 1;
  maxY:=_max.y - _min.y + 1;

  window(_min.x, _min.y, _max.x+1, _max.y);
  GotoXY(   1,    1); Write('');
  GotoXY(   1, maxY); Write('');
  GotoXY(maxX,    1); Write('');
  GotoXY(maxX, maxY); Write('');
  for x:=2 to maxX - 1 do begin
    GotoXY(x,    1); Write('');
    GotoXY(x, maxY); Write('');
  end;
  for y:=2 to maxY - 1 do begin
    GotoXY(   1, y); Write('');
    GotoXY(maxX, y); Write('');
  end;

  x:=(maxX div 2) - (length(_caption) div 2);
  GotoXY(x, 1); Write(_caption);
  if _first > 1 then begin
    GotoXY(maxX-2 , 1); Write(UpArrow);
  end;
  if _last < _items.size then begin
    GotoXY(maxX-1 , 1); Write(DownArrow);
  end;

  window(_min.x+1, _min.y+1, _max.x-1, _max.y-1);
  _items.reset;
  for fforward:=2 to _first do _items.next;
  for fforward:=_first to _last do begin
    Write(_items.current);
    GotoXY(1, WhereY + 1);
    _items.next;
  end;

  HighlightOff;
  _items.reset;
  GotoXY(1, _index - _first + 1);
  for fforward:=2 to _index do _items.next;
  Write(_items.current);

  window(1, 1, 80, 25);
end;

procedure TListBox.getListFromDir(path, mask: String);
var FInfo: SearchRec;
begin
  _items.create;
  FindFirst(path+'\'+mask, AnyFile, FInfo);
  while DosError = 0 do begin
    _items.insert(Finfo.name);
    FindNext(Finfo);
  end;
end;

procedure TListBox.getListFromPassage(fname, passage: String);
var t: Text;
    s: String;
    stop: Boolean;
begin
  _items.create;
  assign(t, fname);
  {$I-} reset(t) {$I+};
  if IOResult = 0 then begin
    s:='';
    while not eof(t) and (s <> '[' + passage + ']') do ReadLn(t, s);
    stop:=FALSE;
    s:='';
    while not (eof(t) or stop) do begin
      ReadLn(t, s);
      stop:=(s = '') or (pos('[', s) <> 0);
      if not stop then _items.insert(s);
    end;
    close(t);
  end;
end;

procedure TListBox.getListFromFile(fname: String);
var t: Text;
    s: String;
begin
  _items.create;
  assign(t, fname);
  {$I-} reset(t) {$I+};
  if IOResult = 0 then begin
    while not eof(t) do begin
      ReadLn(t, s);
      if s <> '' then _items.insert(s);
    end;
    close(t);
  end;
end;

procedure TListBox.getListFromString(list: String; sep: Char);
var item: String;
begin
  _items.create;
  while length(list) > 0 do begin
    item:=copy(list, 1, pos(sep, list)-1);
    delete(list, 1, pos(sep, list));
    _items.insert(item);
  end;
end;

function  TListBox.selected: String;
var ch: Char;
    numRows: Byte;
begin
  if _items.size > 0 then begin
    numRows:=_max.y - _min.y - 1;
    if _items._size < numRows then _last:=_items._size else _last:=numRows;
    HideCursor;
    repeat
      draw;
      ch:=ReadKey;
      case ch of
        kbESC  : selected:='';
        kbEnter: selected:=_items.current;
        kbCursor,
        kbCtrl : begin
                   ch:=ReadKey;
                   case ch of
                     kbHome: begin
                               _index:=1;
                               _first:=1;
                               if _items.size < numRows then _last:=_items.size
                                                 else _last:=numRows;
                             end;
                     kbUp  : if _index > 1 then dec(_index);
                     kbPgUp: if _first > numRows then begin
                               dec(_index, numRows);
                               dec(_first, numRows);
                               dec(_last , numRows);
                             end else _index:=1;
                     kbEnd : begin
                               _index:=_items.size;
                               _last:=_items.size;
                               if _items.size < numRows then _first:=1
                               else _first:=_items.size - numRows + 1;
                             end;
                     kbDown: if _index < _items.size then inc(_index);
                     kbPgDn: if (_last + numRows) <= _items.size then begin
                               inc(_index, numRows);
                               inc(_first, numRows);
                               inc(_last , numRows);
                             end else _index:=_items.size;
                   end; {of case}
                   while _index < _first do begin
                     dec(_first); dec(_last);
                   end;
                   while _index > _last do begin
                     inc(_first); inc(_last);
                   end;
                 end; {of kbCtrl}
      end; {of case}
    until ch in [kbESC, kbEnter];
    ShowCursor;
  end else selected:='';
end;

function TListBox.index: Word;
begin
  index:=_index;
end;

procedure TListBox.clearList;
begin
  _items.destruct;
end;


{*** TEditBox ***}

const
  BoxLen = 48;

procedure TEditBox.init(row: Byte);
begin
  _index:=1;
  _first:=1;
  _last:=1;
  _row:=row;
  _drawn:=FALSE;
end;

procedure TEditBox.draw(str: String);
var x: Byte;
begin
  if not _drawn then begin
    window(80 div 2 - BoxLen div 2 - 1, _row - 1,
           80 div 2 + BoxLen div 2    , _row + 1);
    HighlightOn;
    ClrScr;
    window(80 div 2 - BoxLen div 2 - 1, _row - 1,
           80 div 2 + BoxLen div 2 + 1, _row + 1);
    for x:=2 to BoxLen + 1 do begin
      GotoXY(x, 1); Write('');
      GotoXY(x, 3); Write('');
    end;

    GotoXY(         1, 1); Write('');
    GotoXY(         1, 2); Write('');
    GotoXY(         1, 3); Write('');
    GotoXY(BoxLen + 2, 1); Write('');
    GotoXY(BoxLen + 2, 2); Write('');
    GotoXY(BoxLen + 2, 3); Write('');
    _drawn:=TRUE;
  end;

  window(80 div 2 - BoxLen div 2, _row, 80 div 2 + BoxLen div 2 - 1, _row);
  ClrScr;
  write(copy(str, _first, _last));
  GotoXY(_index - _first + 1, 1);
end;

procedure TEditBox.prompt(var str: String);
var ch: Char;
begin
  _index:=length(str) + 1;
  _last:=length(str);
  _first:=1;
  repeat
    draw(str);
    ch:=ReadKey;
    case ch of
      kbESC    : begin
                   str:='';
                   _index:=1;
                   _first:=1;
                   _last:=0;
                 end;
      kbCursor,
      kbCtrl   : begin
                   ch:=ReadKey;
                   case ch of
                     kbLeft   : if _index > 1 then begin
                                  dec(_index);
                                  if _index < _first then begin
                                    dec(_first);
                                    dec(_last);
                                  end;
                                end;
                     kbRight  : if _index <= length(str) then begin
                                  inc(_index);
                                  {missing length(str) > BoxLen case}
                                end;
                     kbHome   : begin
                                  _index:=1;
                                  _first:=1;
                                  if length(str) < BoxLen then _last:=length(str)
                                  else _last:=BoxLen;
                                end;
                     kbEnd    : begin
                                  _index:=length(str) + 1;
                                  _last:=length(str);
                                  if length(str) < BoxLen then _first:=1
                                  else _first:=length(str) - BoxLen + 1;
                                end;
                     kbDel    : begin
                                  if _index <= length(str) then begin
                                    if _last = length(str) then dec(_last);
                                    delete(str, _index, 1);
                                  end;
                                end;
                   end;
                 end;
      kbBackSpc: if _index > 1 then begin
                   if _last = length(str) then dec(_last);
                   delete(str, _index - 1, 1);
                   dec(_index);
                 end;
      #32..#255: if length(str) < (BoxLen - 1) then begin
                   {missing length(str) > BoxLen case}
                   inc(_last);
                   insert(ch, str, _index);
                   inc(_index);
                 end;
    end;
  until ch = kbEnter;
  _drawn:=FALSE;

  HighlightOff;
  window(1, 1, 80, 25);
end;


{*** TTextView ***}

procedure TTextView.init(x1, y1, x2, y2: Byte; caption: String);
begin
  _min.x:=x1; _min.y:=y1;
  _max.x:=x2; _max.y:=y2;
  _caption:=caption;
  _size:=0;
  _first:=1;
  _last:=_max.y - _min.y - 1;
end;

procedure TTextView.draw;
var x, y, maxX, maxY: Byte;
    fforward: Word;
begin
  window(_min.x, _min.y, _max.x, _max.y);
  HighlightOn;
  ClrScr;
  maxX:=_max.x - _min.x + 1;
  maxY:=_max.y - _min.y + 1;

  window(_min.x, _min.y, _max.x+1, _max.y);
  GotoXY(   1,    1); Write('');
  GotoXY(   1, maxY); Write('');
  GotoXY(maxX,    1); Write('');
  GotoXY(maxX, maxY); Write('');
  for x:=2 to maxX - 1 do begin
    GotoXY(x,    1); Write('');
    GotoXY(x, maxY); Write('');
  end;
  for y:=2 to maxY - 1 do begin
    GotoXY(   1, y); Write('');
    GotoXY(maxX, y); Write('');
  end;

  x:=(maxX div 2) - (length(_caption) div 2);
  GotoXY(x, 1); Write(_caption);
  if _first > 1 then begin
    GotoXY(maxX-2 , 1); Write(UpArrow);
  end;
  if _last < _size then begin
    GotoXY(maxX-1 , 1); Write(DownArrow);
  end;

  window(_min.x+1, _min.y+1, _max.x-1, _max.y-1);
  _items.reset;
  for fforward:=2 to _first do _items.next;
  for fforward:=_first to _last do begin
    GotoXY(1, fforward - _first + 1);
    Write(_items.current);
    _items.next;
  end;

  HighlightOff;
  window(1, 1, 80, 25);
end;

procedure TTextView.getListFromFile(fname: String);
var t: Text;
    s: String;
begin
  _items.create;
  assign(t, fname);
  {$I-} reset(t) {$I+};
  if IOResult = 0 then begin
    while not eof(t) do begin
      ReadLn(t, s);
      _items.insertAfter(s);
      inc(_size);
    end;
    close(t);
  end;
end;

procedure TTextView.getListFromString(list: String; sep: Char);
var item: String;
begin
  _items.create;
  while length(list) > 0 do begin
    item:=copy(list, 1, pos(sep, list)-1);
    _items.insertAfter(item);
    delete(list, 1, pos(sep, list));
    inc(_size);
  end;
end;

procedure TTextView.show;
var ch: Char;
    numRows: Byte;
begin
  numRows:=_max.y - _min.y - 1;
  if _size < numRows then _last:=_size else _last:=numRows;
  HideCursor;
  repeat
    draw;
    ch:=ReadKey;
    if (ch in [kbCtrl, kbCursor]) and (_size > numRows) then begin
      ch:=ReadKey;
      case ch of
        kbHome: begin
                  _first:=1;
                  _last:=numRows;
                end;
        kbUp  : if _first > 1 then begin
                  dec(_first); dec(_last);
                end;
        kbPgUp: if _first > numRows then begin
                  dec(_first, numRows);
                  dec(_last, numRows);
                end else if _first > 1 then begin
                  _first:=1;
                  _last:=numRows;
                end;
        kbEnd : begin
                  _last:=_size;
                  _first:=_size - numRows + 1;
                end;
        kbDown: if _last < _size then begin
                  inc(_first); inc(_last);
                end;
        kbPgDn: if _size - numRows > _last then begin
                  inc(_first, numRows);
                  inc(_last, numRows);
                end else if _last < _size then begin
                  _last:=_size;
                  _first:=_size - numRows + 1;
                end;
      end;
    end;
  until ch in [kbESC, kbEnter];
  ShowCursor;
end;

procedure TTextView.clearList;
begin
  _items.destruct;
  _size:=0;
end;


{*** TMenue ***}

procedure TMenue.init(list: String; sep: Char);
var item: String;
begin
  _sel:=1;
  _numItems:=0;
  _items.create;
  while length(list) > 0 do begin
    item:=copy(list, 1, pos(sep, list)-1);
    delete(list, 1, pos(sep, list));
    _items.insertAfter(item);
    inc(_numItems);
  end;
  dec(_numItems);
end;

procedure TMenue.done;
begin
  _items.destruct;
  _numItems:=0;
end;

procedure TMenue.draw;
var x, y, i, ofs: Byte;
begin
  _items.reset;
  HighlightOff;
{ ClrScr;}
  ofs:=_numItems div 9;
  x:=40 - length(_items.current) div 2;
  GotoXY(x, 2 - ofs); Write(_items.current);
  GotoXY(x - 1, 3 - ofs);
  for i:=x to (x + length(_items.current) + 1) do Write('');

  y:=6 - 2 * ofs;
  while not _items.last do begin
    _items.next;
    GotoXY(14, y); Write(' ', _items.current, ' ');
    inc(y, 2); inc(i);
  end;

  _items.reset;
  for i:=1 to _sel do _items.next;
  HighlightOn;
  GotoXY(14, 6 + (-ofs + (_sel - 1)) * 2);
  Write(' ', _items.current, ' ');
  HighlightOff;
end;

procedure TMenue.show(var selection: Byte);
var ch: Char;
begin
  repeat
    draw;
    HideCursor;
    ch:=ReadKey;
    case ch of
      kbCtrl : begin
                 ch:=ReadKey;
                 case ch of
                   kbHome,
                   kbPgUp  : _sel:=1;
                   kbEnd,
                   kbPgDn  : _sel:=_numItems;
                   kbUp    : if _sel > 1 then dec(_sel) else _sel:=_numItems;
                   kbDown  : if _sel < _numItems then inc(_sel) else _sel:=1;
                 end;
               end;
      kbEnter: selection:=_sel;
      kbESC  : selection:=0;
      else if ch in ['1'..chr(_numItems+48)] then begin
        _sel:=ord(ch)-48;
        selection:=_sel;
      end;
    end;
  until ch in ['1'..chr(_numItems+48), kbESC, kbEnter];
  draw;
  GotoXY(15, 20);
  ShowCursor;
end;

end.
