// Bitmap conversions for DoomEd 4.0
// 
// Copyright  1995 by Geoff Allan
// All Rights Reserved. Unauthorised distribution of this source
// is a violation of Canadian and International Copyright laws.

#include "DoomEd40.hpp"

static BOOL     BitmapPaletteInitialized = FALSE;
static RGBQUAD  rgbQuad[256];

#define BACK    86
#define PaletteIntoHeader(val) {                                                     \
                                int   t;                                             \
                                if(!BitmapPaletteInitialized) {                      \
                                  LOGPALETTE   *pLogPalette;                         \
                                  pLogPalette = (LOGPALETTE *)GlobalLock(hgPalette); \
                                  for(t = 0; t < 256; t++) {                         \
                                    rgbQuad[t].rgbRed =                              \
                                      pLogPalette->palPalEntry[t].peRed;             \
                                    rgbQuad[t].rgbGreen =                            \
                                      pLogPalette->palPalEntry[t].peGreen;           \
                                    rgbQuad[t].rgbBlue  =                            \
                                      pLogPalette->palPalEntry[t].peBlue;            \
                                    }                                                \
                                  GlobalUnlock(hgPalette);                           \
                                  BitmapPaletteInitialized = TRUE;                   \
                                  }                                                  \
                                for(t = 0; t < 256; t++)                             \
                                  val->bmiColors[t] = rgbQuad[t];                    \
                                }


HBITMAP BitmapFromDoom(int which)
{
  HFILE     WadFile;
  HGLOBAL   hglb;
  BYTE    __huge *BufferBYTE;       // buffer
  HBITMAP   bmp;

  if(which == NotFound)
    return (HBITMAP)NULL;
  hglb = GlobalAlloc(GMEM_FIXED, Doom[which].Length);
  BufferBYTE = (BYTE __huge *)GlobalLock(hglb);
  
  WadFile = _lopen(szDoomWad, OF_READ);     // open for read
  _llseek(WadFile, Doom[which].Offset, 0);  // position to data
  _hread(WadFile, BufferBYTE, Doom[which].Length);// suck it into memory
  _lclose(WadFile);                         // close file
  
  bmp = ConvertImageToBMP(BufferBYTE);      // make the bitmap
  
  GlobalUnlock(hglb);                       // unlock memory
  GlobalFree(hglb);                         // give memory back
  return bmp;                               // send the bitmap back
}

HBITMAP BitmapFromDir(int which)
{
  HFILE     WadFile;
  HGLOBAL   hglb;
  BYTE    __huge *BufferBYTE;       // buffer
  HBITMAP   bmp;

  if(which == NotFound)
    return (HBITMAP)NULL;
  hglb = GlobalAlloc(GMEM_FIXED, Dir[which].Length);
  BufferBYTE = (BYTE __huge *)GlobalLock(hglb);
  
  WadFile = _lopen(szPrevFile1, OF_READ);   // open for read
  _llseek(WadFile, Dir[which].Offset, 0);   // position to data
  _hread(WadFile, BufferBYTE, Dir[which].Length); // suck it into memory
  _lclose(WadFile);                         // close file
  
  bmp = ConvertImageToBMP(BufferBYTE);            // make the bitmap
  
  GlobalUnlock(hglb);                       // unlock memory
  GlobalFree(hglb);                         // give memory back
  return bmp;                               // send the bitmap back
}

HBITMAP ConvertImageToBMP(BYTE __huge *BufferBYTE)
{
  // caller MUST Delete Bitmap when done with it.
  BYTE         *bm;
  BITMAPINFO   *bi;
  HBITMAP       hTBitmap;
  HDC           rDC;
  HGLOBAL       hgBitmap, hgBitmapInfo;
  WORD          i, j,
                PosX, PosY,
                ImageWidth, ImageHeight;
  WORD  __huge *BufferWORD  = (WORD  __huge *)BufferBYTE;
  DWORD __huge *BufferDWORD = (DWORD __huge *)BufferBYTE;
  DWORD         c;
  
  BitmapWidth  = ImageWidth  = BufferWORD[0];
  BitmapHeight = ImageHeight = BufferWORD[1];
  
  // make each scanline length a multiple of DWORD:
  while((BitmapWidth & 1) || (BitmapWidth & 2))
    BitmapWidth++;
    
  ViewerOffsetX = BufferWORD[2];
  ViewerOffsetY = BufferWORD[3];

  if((ImageWidth > 320) || 
     (ImageHeight > 200))      // dimensions too large,
    return NULL;                // probably not a bitmap

  // create DIB:
  hgBitmapInfo = GlobalAlloc(GHND, sizeof(BITMAPINFOHEADER) +
                            (256 * sizeof(RGBQUAD)));
  bi = (PBITMAPINFO)GlobalLock(hgBitmapInfo);
  bi->bmiHeader.biSize        = sizeof(BITMAPINFOHEADER);
  bi->bmiHeader.biWidth       = BitmapWidth;
  bi->bmiHeader.biHeight      = BitmapHeight;
  bi->bmiHeader.biPlanes      = 1;
  bi->bmiHeader.biBitCount    = 8;

  PaletteIntoHeader(bi);

  // allocate space for the bitmap
  hgBitmap = GlobalAlloc(GPTR, BitmapWidth * BitmapHeight);
  bm = (BYTE FAR *)GlobalLock(hgBitmap);
  
  memset(bm, BACK, BitmapWidth * BitmapHeight);    // background color
  
  for(PosX = 0; PosX < ImageWidth; PosX++) {
    c = BufferDWORD[PosX + 2]; // add 2 for header information
    while(BufferBYTE[c] != 0xff) {       // 0xff = done column
      PosY = BitmapHeight - (WORD)BufferBYTE[c++];
      j  = (WORD)BufferBYTE[c++];
      c++;  // skip unwanted value
      for(i = 0; i < j; i++)
        bm[PosX + (BitmapWidth * --PosY)] = BufferBYTE[c++];
      c++;  // skip unwanted value
      }     // wend
    }       // next PosX

  rDC = GetDC(hwnd);
  hTBitmap = CreateDIBitmap(rDC, (BITMAPINFOHEADER FAR *)bi, CBM_INIT,
                            bm, bi, DIB_RGB_COLORS);
  ReleaseDC(hwnd, rDC);

  GlobalUnlock(hgBitmapInfo);
  GlobalFree(hgBitmapInfo);
  GlobalUnlock(hgBitmap);
  GlobalFree(hgBitmap);
  return hTBitmap;  
}

HBITMAP TileFromDoom(int which)
{
  HFILE     WadFile;
  HGLOBAL   hglb;
  BYTE    __huge *BufferBYTE;       // buffer
  HBITMAP    bmp;

  if(which == NotFound)
    return (HBITMAP)NULL;
  hglb = GlobalAlloc(GMEM_FIXED, Doom[which].Length);
  BufferBYTE = (BYTE __huge *)GlobalLock(hglb);
  
  WadFile = _lopen(szDoomWad, OF_READ);     // open for read
  _llseek(WadFile, Doom[which].Offset, 0);  // position to data
  _hread(WadFile, BufferBYTE, Doom[which].Length);// suck it into memory
  _lclose(WadFile);                         // close file
  
  bmp = ConvertTileToBMP(BufferBYTE);             // make the bitmap
  
  GlobalUnlock(hglb);                       // unlock memory
  GlobalFree(hglb);                         // give memory back
  return bmp;                               // send the bitmap back
}

HBITMAP TileFromDir(int which)
{
  HFILE     WadFile;
  HGLOBAL   hglb;
  BYTE    __huge *BufferBYTE;       // buffer
  HBITMAP   bmp;

  if(which == NotFound)
    return (HBITMAP)NULL;
  hglb = GlobalAlloc(GMEM_FIXED, Dir[which].Length);
  BufferBYTE = (BYTE __huge *)GlobalLock(hglb);
  
  WadFile = _lopen(szPrevFile1, OF_READ);   // open for read
  _llseek(WadFile, Dir[which].Offset, 0);   // position to data
  _hread(WadFile, BufferBYTE, Dir[which].Length); // suck it into memory
  _lclose(WadFile);                         // close file
  
  bmp = ConvertTileToBMP(BufferBYTE);             // make the bitmap
  
  GlobalUnlock(hglb);                       // unlock memory
  GlobalFree(hglb);                         // give memory back
  return bmp;                               // send the bitmap back
}

HBITMAP ConvertTileToBMP(BYTE __huge *BufferBYTE)
{
  BYTE          buf[64];
  int           i;
  HDC           rDC;
  HBITMAP       hTBitmap;
  HGLOBAL       hgBitmapInfo;
  BITMAPINFO   *bi;

  BitmapWidth = BitmapHeight = 64;
  // reverse the image for DIB:
  for(i=0; i<32; i++) {
    memcpy(buf, &BufferBYTE[i * 64], 64);
    memcpy(&BufferBYTE[i * 64], &BufferBYTE[(63 - i) * 64], 64);
    memcpy(&BufferBYTE[(63 - i) * 64], buf, 64);
    }
  
  hgBitmapInfo = GlobalAlloc(GHND, sizeof(BITMAPINFOHEADER) + 256*sizeof(RGBQUAD));
  bi = (PBITMAPINFO)GlobalLock(hgBitmapInfo);
  bi->bmiHeader.biSize        = sizeof(BITMAPINFOHEADER);
  bi->bmiHeader.biWidth       = 64;
  bi->bmiHeader.biHeight      = 64;
  bi->bmiHeader.biPlanes      = 1;
  bi->bmiHeader.biBitCount    = 8;

  PaletteIntoHeader(bi);

  rDC = GetDC(hwnd);
  hTBitmap = CreateDIBitmap(rDC, (BITMAPINFOHEADER FAR *)bi, CBM_INIT,
                            BufferBYTE, bi, DIB_RGB_COLORS);
  ReleaseDC(hwnd, rDC);

  GlobalUnlock(hgBitmapInfo);
  GlobalFree(hgBitmapInfo);
  return hTBitmap;  
}

HBITMAP TextureImageFromDoom(int t)
{
// returns a handle to the bitmap image of a texture (# t)
// caller must free this bitmap.
  // caller MUST Delete Bitmap when done with it.
  HFILE         WadFile;
  int           panel;
  BYTE         *bm;
  BITMAPINFO   *bi;
  HBITMAP       hTBitmap;
  HDC           rDC;
  HGLOBAL       hgBitmap, hgBitmapInfo, hgPanel;
  WORD          i, j, k, inset, MaxInset,
                PosX, PosY,
                ImageWidth, ImageHeight,
                PanelWidth, PanelHeight;
  BYTE  __huge *BufferBYTE;
  WORD  __huge *BufferWORD;
  DWORD __huge *BufferDWORD;
  DWORD         c;
  TextureItem   ti[72];
  
  ViewerOffsetX = 0;
  ViewerOffsetY = 0;

  // read the required patch info:
  WadFile = _lopen(szDoomWad, OF_READ);
  _llseek(WadFile, Texture[t].offset, 0);
  for(i = 0; i < (WORD)Texture[t].num; i++)
    _lread(WadFile, &ti[i], sizeof(TextureItem));
  _lclose(WadFile);

  
  // create the texture bitmap:
  BitmapWidth  = ImageWidth  = Texture[t].x;
  BitmapHeight = ImageHeight = Texture[t].y;
  // make each scanline length a multiple of DWORD:
  while((BitmapWidth & 1) || (BitmapWidth & 2))
    BitmapWidth++;
  // create DIB info:
  MaxInset = BitmapWidth * BitmapHeight;
  hgBitmapInfo = GlobalAlloc(GHND, sizeof(BITMAPINFOHEADER) +
                            (256 * sizeof(RGBQUAD)));
  bi = (PBITMAPINFO)GlobalLock(hgBitmapInfo);
  bi->bmiHeader.biSize        = sizeof(BITMAPINFOHEADER);
  bi->bmiHeader.biWidth       = BitmapWidth;
  bi->bmiHeader.biHeight      = BitmapHeight;
  bi->bmiHeader.biPlanes      = 1;
  bi->bmiHeader.biBitCount    = 8;
  PaletteIntoHeader(bi);

  // allocate space for the texture bitmap
  hgBitmap = GlobalAlloc(GPTR, MaxInset);
  bm = (BYTE FAR *)GlobalLock(hgBitmap);
  // set to background color
  memset(bm, BACK, MaxInset);
  
  // draw for each panel:
  for(i = 0; i < (WORD)Texture[t].num; i++) {
    panel= DoomEntry(PName[ti[i].pname].name);
    hgPanel = GlobalAlloc(GMEM_FIXED, Doom[panel].Length);
    BufferBYTE = (BYTE __huge *)GlobalLock(hgPanel);
    BufferWORD  = (WORD  __huge *)BufferBYTE;
    BufferDWORD = (DWORD __huge *)BufferBYTE;
    WadFile = _lopen(szDoomWad, OF_READ);       // open for read
    _llseek(WadFile, Doom[panel].Offset, 0);    // position to data
    _hread(WadFile, BufferBYTE, Doom[panel].Length);  // suck it into memory
    _lclose(WadFile);                           // close file
  
    PanelWidth  = BufferWORD[0];
    PanelHeight = BufferWORD[1];

    for(PosX = 0; PosX < PanelWidth; PosX++) {
      c = BufferDWORD[PosX + 2]; // add 2 for header information
      while(BufferBYTE[c] != 0xff) {       // 0xff = done column
        PosY = BitmapHeight -  ti[i].y - (WORD)BufferBYTE[c++];
        j  = (WORD)BufferBYTE[c++];
        c++;  // skip unwanted value
        for(k = 0; k < j; k++) {
          inset = ti[i].x + PosX + (BitmapWidth * --PosY);
          if((inset >= 0) && (inset < MaxInset))
            bm[inset] = BufferBYTE[c];
          c++;
          }
        c++;  // skip unwanted value
        }     // wend
      }       // next PosX
    
    GlobalUnlock(hgPanel);                       // unlock memory
    GlobalFree(hgPanel);                         // give memory back
    }

  rDC = GetDC(hwnd);
  hTBitmap = CreateDIBitmap(rDC, (BITMAPINFOHEADER FAR *)bi, CBM_INIT,
                            bm, bi, DIB_RGB_COLORS);
  ReleaseDC(hwnd, rDC);

  GlobalUnlock(hgBitmapInfo);
  GlobalFree(hgBitmapInfo);
  GlobalUnlock(hgBitmap);
  GlobalFree(hgBitmap);
  return hTBitmap;  
}

// Required:
// BitmapToDoom(hBitmap) - convert edited bitmap to Doom format
// BitmapToTile(hBitmap) - convert edited tile to Doom format

