// 3d Display module for DoomEd 4.0
// Copyright  1995 by Geoff Allan

#include "DoomEd40.hpp"
#include <math.h>

// ToDo: 1) Replace ALL floating point with integer routines. This will increase
//          the speed to more nearly "real-time".
//       2) Experiment with drawing to a dib or WinG dib, to eliminate the
//          redrawing flicker.
//       3) Allow placement of textures on walls...

#define CAPTURING       (hDlg == GetCapture())
#define REDRAW          InvalidateRect(hDlg, NULL, TRUE);
#define PI              3.141592654
#define PI180           0.017453292

typedef struct {
  float a[4][4];
  } matrix;

typedef struct {
  float x;
  float y;
  float z;
  } point3d;

typedef struct {
  int   distance;
  int   element;
  } Sorter;

static  matrix  mat, t1;
static  POINT   newmouse, oldmouse;
static  BOOL    PointValid = TRUE, DoingRight = FALSE, DoingLeft = FALSE,
                Hidden = FALSE, OldHidden;
static  RECT    extent;         // map extents
static  SIZE    size;
static  HWND    gerb;
static  float   aspect;         // aspect ratio
static  int     vx = 1000, vy = 1000, vz = -200, cx, cy,
                ox = 0, oy = 0, oz = 0, zoom = 5;

void Do3d(void)
{
  FARPROC lpfnDlgProc;
  lpfnDlgProc = MakeProcInstance((FARPROC)Dialog3d, hinst);
  if(lpfnDlgProc) {
    DialogBox(hinst,
              MAKEINTRESOURCE(IDD_3D),
              hwnd,
              lpfnDlgProc);
    FreeProcInstance(lpfnDlgProc);
    }
}


int EXPORT Dialog3d(HWND hDlg, WORD wMsg, WORD wParam, DWORD lParam)
{
  PAINTSTRUCT ps;
  FARPROC     lpfnDlgProc;
  int         distance;
  int         dx, dy;
  int         idEnd;
  float       dd1, dd2; // dd1 = distance from center, dd2 = angle
  
  switch(wMsg)
    {
    case WM_INITDIALOG: {
      int       iLineDef, rNew;
      extent = GetMapExtents(); // get the current extents of the map
      ox = ((extent.right - extent.left) / 2) + extent.left;
      oy = ((extent.top - extent.bottom) / 2) + extent.bottom;
      oz = 0;
      lpfnDlgProc = MakeProcInstance((FARPROC)Dialog3dSetup, hinst);
      idEnd = DialogBox(hinst,
                        MAKEINTRESOURCE(IDD_3D_SETUP),
                        hDlg,
                        lpfnDlgProc);
      FreeProcInstance(lpfnDlgProc);
      if(idEnd == IDCANCEL)
        EndDialog(hDlg, IDCANCEL);
      // create and fill the Rect array:
      Rect.Flush();
      for(iLineDef = 0; iLineDef < LineDefsNum; iLineDef++) {
        if(eLineDef[iLineDef].Used &&
           (LineDef[iLineDef].sidedef1 != Nothing) &&
           !(LineDef[iLineDef].solidity & ML_TWOSIDED)) {
          // valid linedef: solid (can't pass)
          // strategy: draw vertical wall panel
          rNew = Rect.New();
          Rect[rNew]->p1.x = Vertex[LineDef[iLineDef].from].x;
          Rect[rNew]->p1.y = Vertex[LineDef[iLineDef].from].y;
          Rect[rNew]->p1.z = Sector[SideDef[LineDef[iLineDef].sidedef1].sector].floorZ;
          Rect[rNew]->p2.x = Vertex[LineDef[iLineDef].to].x;
          Rect[rNew]->p2.y = Vertex[LineDef[iLineDef].to].y;
          Rect[rNew]->p2.z = Sector[SideDef[LineDef[iLineDef].sidedef1].sector].ceilZ;
          Rect[rNew]->distance = 50;
          }
        if(eLineDef[iLineDef].Used &&
           (LineDef[iLineDef].sidedef1 != Nothing) &&
           (LineDef[iLineDef].sidedef2 != Nothing) &&
           (LineDef[iLineDef].solidity & ML_TWOSIDED)) {
          // two sided, two valid sidedefs
          // strategy: on two sided, draw the high and low panels if
          //           the two sector heights are not equal 
          if(Sector[SideDef[LineDef[iLineDef].sidedef1].sector].floorZ !=
             Sector[SideDef[LineDef[iLineDef].sidedef2].sector].floorZ) {
            // floors don't match:
            rNew = Rect.New();
            Rect[rNew]->p1.x = Vertex[LineDef[iLineDef].from].x;
            Rect[rNew]->p1.y = Vertex[LineDef[iLineDef].from].y;
            Rect[rNew]->p1.z = Sector[SideDef[LineDef[iLineDef].sidedef1].sector].floorZ;
            Rect[rNew]->p2.x = Vertex[LineDef[iLineDef].to].x;
            Rect[rNew]->p2.y = Vertex[LineDef[iLineDef].to].y;
            Rect[rNew]->p2.z = Sector[SideDef[LineDef[iLineDef].sidedef2].sector].floorZ;
            Rect[rNew]->distance = 50;
            }
          if(Sector[SideDef[LineDef[iLineDef].sidedef1].sector].ceilZ !=
             Sector[SideDef[LineDef[iLineDef].sidedef2].sector].ceilZ) {
            // ceilings don't match
            rNew = Rect.New();
            Rect[rNew]->p1.x = Vertex[LineDef[iLineDef].from].x;
            Rect[rNew]->p1.y = Vertex[LineDef[iLineDef].from].y;
            Rect[rNew]->p1.z = Sector[SideDef[LineDef[iLineDef].sidedef1].sector].ceilZ;
            Rect[rNew]->p2.x = Vertex[LineDef[iLineDef].to].x;
            Rect[rNew]->p2.y = Vertex[LineDef[iLineDef].to].y;
            Rect[rNew]->p2.z = Sector[SideDef[LineDef[iLineDef].sidedef2].sector].ceilZ;
            Rect[rNew]->distance = 50;
            }
          }
        }
      }
      return TRUE;
      break;
      
    case WM_SIZE:
      switch (wParam)
        {
        case SIZE_MAXIMIZED:
        case SIZE_RESTORED:
          REDRAW;
          return FALSE;
          break;
        default:
          return FALSE;
          break;
        }
      break;

    case WM_PAINT:
      gerb = hDlg;
      BeginPaint(hDlg, &ps);       // get paint information
      Draw3dMap(ps.hdc);
      EndPaint(hDlg, &ps);         // done painting
      return TRUE;                 // we processed, no default
      break;

    case WM_CLOSE:
      EndDialog(hDlg, IDOK);
      return FALSE;
      break;
    
    case WM_MOUSEMOVE:
      if(!CAPTURING) {
        return FALSE;
        }
      newmouse = MAKEPOINT(lParam);
      if(DoingLeft && DoingRight) {
        // intent: move closer to or farther from the
        //         look-toward point
        if(newmouse.x != oldmouse.x) {
          distance = newmouse.x - oldmouse.x;
          dx = vx - ox;
          dy = vy - oy;
          dd2 = (float) atan2((double)dy, (double)dx);    // current direction
          dd1 = (float)_hypot((double)dx, (double)dy);    // distance from center
          dd1 -= distance*10;
          vx = ox + (int)(cos(dd2) * dd1);
          vy = oy + (int)(sin(dd2) * dd1);            
          REDRAW;
          }
        oldmouse = newmouse;
        return FALSE;
        }

      if(DoingLeft) {
        if(newmouse.x != oldmouse.x) {
          // intent: rotate around the look-towards point
          //         by "distance" degrees (left or right)
          distance = newmouse.x - oldmouse.x;
          dx = vx - ox;
          dy = vy - oy;
          dd2 = (float)(atan2((double)dy, (double)dx) / PI180); // current direction
          dd1 = (float)_hypot((double)dx, (double)dy);          // distance from center
          dd2 -= (float)(distance)/2;                             // move around center
          dd2 *= (float)PI180;                                  // radians
          vx = ox + (int)(cos(dd2) * dd1);
          vy = oy + (int)(sin(dd2) * dd1);            
          REDRAW;
          }
        if(newmouse.y != oldmouse.y) {
          // intent: raise or lower the observer
          distance = newmouse.y - oldmouse.y;
          vz += distance*10;
          REDRAW;
          }
        oldmouse = newmouse;
        return FALSE;
        }
      if(DoingRight) {
        if(newmouse.x != oldmouse.x) {
          // intent: rotate look-towards point around the observer
          //         by "distance" degrees (left or right)
          distance = newmouse.x - oldmouse.x;
          dx = ox - vx;
          dy = oy - vy;
          dd2 = (float)(atan2((double)dy, (double)dx) / PI180); // current direction
          dd1 = (float)_hypot((double)dx, (double)dy);          // distance from center
          dd2 -= (float)(distance)/2;                           // move around center
          dd2 *= (float)PI180;                                  // radians
          ox = vx + (int)(cos(dd2) * dd1);
          oy = vy + (int)(sin(dd2) * dd1);            
          REDRAW;
          }
        if(newmouse.y != oldmouse.y) {
          // intent: raise or lower the look-toward point
          distance = newmouse.y - oldmouse.y;
          oz += distance*10;
          REDRAW;
          }
        oldmouse = newmouse;
        }
      return FALSE;
      break;

    case WM_LBUTTONDOWN:
      DoingLeft = TRUE;
      if(!DoingRight) {
        OldHidden = Hidden;
        Hidden = FALSE;
        }
      oldmouse = MAKEPOINT(lParam);
      SetCapture(hDlg);
      return FALSE;
      break;

    case WM_LBUTTONUP:
      DoingLeft = FALSE;
      ReleaseCapture();
      if(!DoingRight) {
        Hidden = OldHidden;
        if(Hidden) {
          hdc = GetDC(hDlg);
          Draw3dMap(hdc);
          ReleaseDC(hDlg, hdc);
          }
        }
      return FALSE;
      break;

    case WM_RBUTTONDOWN:
      DoingRight = TRUE;
      if(!DoingLeft) {
        OldHidden = Hidden;
        Hidden = FALSE;
        }
      oldmouse = MAKEPOINT(lParam);
      SetCapture(hDlg);
      return FALSE;
      break;

    case WM_RBUTTONUP:
      DoingRight = FALSE;
      ReleaseCapture();
      if(!DoingLeft) {
        Hidden = OldHidden;
        if(Hidden) {
          hdc = GetDC(hDlg);
          Draw3dMap(hdc);
          ReleaseDC(hDlg, hdc);
          }
        }
      return FALSE;
      break;
      
    case WM_KEYUP: {
      HDC   hdc;
      if(wParam == 'H') {
        Hidden = !Hidden;
        hdc = GetDC(hDlg);
        Draw3dMap(hdc);
        ReleaseDC(hDlg, hdc);
        }
      }
      return FALSE;
      break;

    default:
      return FALSE;
    }
}

int EXPORT Dialog3dSetup(HWND hDlg, WORD wMsg, WORD wParam, DWORD lParam)
{
  BOOL xlate;
  
  switch(wMsg) {
    case WM_INITDIALOG:
      SetDlgItemInt(hDlg, IDC_VX, vx, TRUE);
      SetDlgItemInt(hDlg, IDC_VY, vy, TRUE);
      SetDlgItemInt(hDlg, IDC_VZ, -vz, TRUE);
      SetDlgItemInt(hDlg, IDC_OX, ox, TRUE);
      SetDlgItemInt(hDlg, IDC_OY, oy, TRUE);
      SetDlgItemInt(hDlg, IDC_OZ, oz, TRUE);
      SetDlgItemInt(hDlg, IDC_ZOOM, zoom, FALSE);
      return TRUE;
      break;

    case WM_COMMAND:
      switch(wParam) {
        case IDOK:
          vx = GetDlgItemInt(hDlg, IDC_VX, &xlate, TRUE);
          vy = GetDlgItemInt(hDlg, IDC_VY, &xlate, TRUE);
          vz = -((int)GetDlgItemInt(hDlg, IDC_VZ, &xlate, TRUE));
          ox = GetDlgItemInt(hDlg, IDC_OX, &xlate, TRUE);
          oy = GetDlgItemInt(hDlg, IDC_OY, &xlate, TRUE);
          oz = GetDlgItemInt(hDlg, IDC_OZ, &xlate, TRUE);
          zoom = GetDlgItemInt(hDlg, IDC_ZOOM, &xlate, FALSE);
          EndDialog(hDlg, IDOK);
          return FALSE;
          break;
        
        case IDCANCEL:
          EndDialog(hDlg, IDCANCEL);
          return FALSE;
          break;
          
        default:
          return FALSE;
        }

    default:
      return FALSE;
    }
}

matrix InitMat(void)
{
  matrix xx;
  int    i, j;
  
  for(i=0; i<4; i++)
    for(j=0; j<4; j++) {
      if(i == j)
        xx.a[i][j] = 1;
      else
        xx.a[i][j] = 0;
      }
  return xx;
}

matrix MultMat(matrix m1, matrix m2)
{
  matrix xx;
  int    i, j;

  for(i=0; i<4; i++)
    for(j=0; j<4; j++)
      xx.a[i][j] = m1.a[i][0] * m2.a[0][j] +
                   m1.a[i][1] * m2.a[1][j] +
                   m1.a[i][2] * m2.a[2][j] +
                   m1.a[i][3] * m2.a[3][j];
  return xx;
}

POINT EyeToScreen(point3d pp)
{
  POINT xx;
  
  // cx, cy is center of view window...
  if(pp.z == 0) {
    PointValid = FALSE;
    return xx;
    }
  else
    PointValid = TRUE;
  xx.x = (int)(aspect * cx * (pp.x / pp.z)) + cx;
  xx.y = (int)(cy * (pp.y / pp.z)) + cy;
  if(abs(xx.x) > abs(size.cx * 2) ||
     abs(xx.y) > abs(size.cy * 2))
    PointValid = FALSE;
  return xx;
}

point3d Transform(point3d pp)
{
  point3d xx;
  
  xx.x = mat.a[0][0] * pp.x + mat.a[1][0] * pp.y + mat.a[2][0] * pp.z + mat.a[3][0];
  xx.y = mat.a[0][1] * pp.x + mat.a[1][1] * pp.y + mat.a[2][1] * pp.z + mat.a[3][1];
  xx.z = mat.a[0][2] * pp.x + mat.a[1][2] * pp.y + mat.a[2][2] * pp.z + mat.a[3][2];
  
  return xx;
}

void Draw3dLine(HDC rDC, int x1, int y1, int z1,
                         int x2, int y2, int z2)
{
  POINT from, to;
  point3d p1, p2;

  p1.x = x1;
  p1.y = y1;
  p1.z = -z1;
  p1 = Transform(p1);
  p2.x = x2;
  p2.y = y2;
  p2.z = -z2;
  p2 = Transform(p2);

  from = EyeToScreen(p1);
  if(!PointValid)
    return;
  to = EyeToScreen(p2);
  if(!PointValid)
    return;

  MoveTo(rDC, from.x + cx, from.y + cy);
  LineTo(rDC, to.x + cx, to.y + cy);
  return;
}

void SetViewMatrix(void)
{
  float fx = vx - ox, fy = vy - oy, fz = vz - oz;
  float d1, d2;

  // o? = look toward
  // v? = viewer
  
  // Translate origin to viewer location
  mat = InitMat();
  mat.a[3][0] = -vx;
  mat.a[3][1] = -vy;
  mat.a[3][2] = -vz;
  
  // Rotate around x-axis by 90 degrees
  t1 = InitMat();
  t1.a[1][1] = 0;
  t1.a[2][2] = 0;
  t1.a[2][1] = 1;
  t1.a[1][2] = -1;
  mat = MultMat(mat, t1);
  
  // Rotate around y-axis by angle dependent on look-toward point
  t1 = InitMat();
  d1 = (float)sqrt((double)(fx*fx + fy*fy));
  if(fabs(d1) > 0.0001) {
    t1.a[0][0] = -fy / d1;
    t1.a[2][2] = -fy / d1;
    t1.a[0][2] =  fx / d1;
    t1.a[2][0] = -fx / d1;
    mat = MultMat(mat, t1);
    }
  
  // Rotate around x-axis by angle dependent on look-toward point
  t1 = InitMat();
  d2 = (float)sqrt((double)(fx*fx + fy*fy + fz*fz));
  if(fabs(d2) > 0.0001) {
    t1.a[1][1] = d1 / d2;
    t1.a[2][2] = d1 / d2;
    t1.a[1][2] = fz / d2;
    t1.a[2][1] = -fz / d2;
    mat = MultMat(mat, t1);
    }

  t1 = InitMat();
  t1.a[2][2] = -1;      // invert z-axis
  t1.a[0][0] = zoom;
  t1.a[1][1] = zoom;
  mat = MultMat(mat, t1);
  
  // view transformation matrix now complete...
  
  return;
}

int compare(const void *arg1, const void *arg2)
{
  if(((Sorter *)arg1)->distance > ((Sorter *)arg2)->distance)
    return -1;
  if(((Sorter *)arg1)->distance < ((Sorter *)arg2)->distance)
    return 1;
  return 0;
}

void Draw3dMap(HDC rDC)
{ // this function draws the map in 3d (hidden lines removed)
  // on the given device context
  RECT          rc;
  int           i;
  HRGN          hRgn;
  HPEN          pOld;
  HGLOBAL       hglob;
  Sorter FAR   *sort;
  
  SetMapMode(rDC, MM_TEXT);
  SelectObject(rDC, hPenMapLines);
  GetClientRect(gerb, &rc);     // get display extents
  size.cx = rc.right;           // total display width
  size.cy = rc.bottom-rc.top;   // total display height
  cx = size.cx / 2;
  cy = size.cy / 2;
  aspect = (float)size.cy / (float)size.cx;
  SetViewMatrix();

  if(Hidden)
    Rect.PreCalc(vx, vy, vz); // calculate center points and observer distance

  // calculate the screen coordinates:
  for(i = 0; i < Rect.Entries(); i++)
    if(Rect.IsUsed(i)) {
      point3d   p1;
      p1.x = (float)Rect[i]->p1.x;
      p1.y = (float)Rect[i]->p1.y;
      p1.z = -(float)Rect[i]->p1.z;
      p1 = Transform(p1);
      Rect[i]->s1 = EyeToScreen(p1);
      if(!PointValid) {
        Rect[i]->distance = Nothing;
        continue;
        }
      p1.x = (float)Rect[i]->p1.x;
      p1.y = (float)Rect[i]->p1.y;
      p1.z = -(float)Rect[i]->p2.z;
      p1 = Transform(p1);
      Rect[i]->s2 = EyeToScreen(p1);
      if(!PointValid) {
        Rect[i]->distance = Nothing;
        continue;
        }
      p1.x = (float)Rect[i]->p2.x;
      p1.y = (float)Rect[i]->p2.y;
      p1.z = -(float)Rect[i]->p2.z;
      p1 = Transform(p1);
      Rect[i]->s3 = EyeToScreen(p1);
      if(!PointValid) {
        Rect[i]->distance = Nothing;
        continue;
        }
      p1.x = (float)Rect[i]->p2.x;
      p1.y = (float)Rect[i]->p2.y;
      p1.z = -(float)Rect[i]->p1.z;
      p1 = Transform(p1);
      Rect[i]->s4 = EyeToScreen(p1);
      if(!PointValid) {
        Rect[i]->distance = Nothing;
        continue;
        }
      }
  
  pOld = SelectPen(rDC, hPenMapLines);
  if(Hidden) {
    hglob = GlobalAlloc(GHND, Rect.Entries() * sizeof(Sorter));
    sort  = (Sorter FAR *)GlobalLock(hglob);
    for(i = 0; i < Rect.Entries(); i++)
      if(Rect.IsUsed(i)) {
        sort[i].element = i;
        sort[i].distance = Rect[i]->distance;
        }
      else {
        sort[i].element = i;
        sort[i].distance = Nothing;
        }
    qsort(sort, Rect.Entries(), sizeof(Sorter), compare);
    for(i = 0; i < Rect.Entries(); i++)
      if(sort[i].distance > 20) {
        hRgn = CreatePolygonRgn(&Rect[sort[i].element]->s1, 4, ALTERNATE);
        FillRgn(rDC, hRgn, hBrushBackground);
        MoveTo(rDC, Rect[sort[i].element]->s1.x, Rect[sort[i].element]->s1.y);
        LineTo(rDC, Rect[sort[i].element]->s2.x, Rect[sort[i].element]->s2.y);
        LineTo(rDC, Rect[sort[i].element]->s3.x, Rect[sort[i].element]->s3.y);
        LineTo(rDC, Rect[sort[i].element]->s4.x, Rect[sort[i].element]->s4.y);
        LineTo(rDC, Rect[sort[i].element]->s1.x, Rect[sort[i].element]->s1.y);
        DeleteRgn(hRgn);
        }
    GlobalUnlock(hglob);
    GlobalFree(hglob);
    }
  else {
    for(i = 0; i < Rect.Entries(); i++)
      if(Rect.IsUsed(i) && (Rect[i]->distance > 20)) {
        MoveTo(rDC, Rect[i]->s1.x, Rect[i]->s1.y);
        LineTo(rDC, Rect[i]->s2.x, Rect[i]->s2.y);
        LineTo(rDC, Rect[i]->s3.x, Rect[i]->s3.y);
        LineTo(rDC, Rect[i]->s4.x, Rect[i]->s4.y);
        LineTo(rDC, Rect[i]->s1.x, Rect[i]->s1.y);
        }
    }
  SelectPen(rDC, pOld);
  return;
}

void RectArray::PreCalc(int x, int y, int z)
{
  int       i;
  double    tx, ty, tz;
  for(i = 0; i < m_MaxEntries; i++)
    if(m_Data[i]) {
      // 3-d Pythagorean theorem:
      tx = (double)(x - ((Rect[i]->p2.x + Rect[i]->p1.x) / 2));
      ty = (double)(y - ((Rect[i]->p2.y + Rect[i]->p1.y) / 2));
      tz = (double)(z - ((Rect[i]->p2.z + Rect[i]->p1.z) / 2));
      Rect[i]->distance = (int)sqrt((tx * tx) + (ty * ty) + (tz * tz));
      }
}









