// Mouse Routines for DoomEd 4.0
// 
// Copyright  1995 by Geoff Allan
// All Rights Reserved. Unauthorised distribution of this source
// is a violation of Canadian and International Copyright laws.

#include "DoomEd40.hpp"

BOOL    DoingLeft = FALSE, DoingRight = FALSE;
BOOL    MightBeMoving = FALSE, MightBeInserting = FALSE;
BOOL    VertexSnap = FALSE, CreatingSector = FALSE;
DWORD   DownTime;   // time that the mouse was first clicked
POINT   StartPoint, LastPoint, Current, Gridded;
HDC     rDC;
int     oldROP;
int     tt;

void ExtractSectors(void);  // prototype

// cursors for various functions:
HCURSOR hCursorDrag, hCursorSel;

// some handy macros that only apply to mouse functions:
#define GetCurrent  Current = MAKEPOINT(lParam);                     \
                    DPtoLP(hdc, &Current, 1);                        \
                    Gridded.x = (Current.x / GridSize) * GridSize;   \
                    Gridded.y = (Current.y / GridSize) * GridSize;
#define GrabMouse   SetCapture(hwnd)
#define LoseMouse   ReleaseCapture()
#define HaveMouse   (hwnd == GetCapture())
#define TimeOut     (GetDoubleClickTime())

#define CURSOR_DRAGLS   hCursorDrag = LoadCursor(hinst,                 \
                                      MAKEINTRESOURCE(IDC_FOURWAY));    \
                        oldCursor = SetCursor(hCursorDrag);
#define CURSOR_DRAGTV   hCursorDrag = LoadCursor(hinst,                 \
                                      MAKEINTRESOURCE(IDC_FINGER));     \
                        oldCursor = SetCursor(hCursorDrag);
#define CURSOR_NOTDRAG  SetCursor(oldCursor);                           \
                        DestroyCursor(hCursorDrag);
#define CURSOR_SEL      hCursorSel  = LoadCursor(hinst,                 \
                                      MAKEINTRESOURCE(IDC_ARROW_BOX));  \
                        oldCursor = SetCursor(hCursorSel);
#define CURSOR_NOTSEL   SetCursor(oldCursor);                           \
                        DestroyCursor(hCursorSel);
#define CURSOR_MAG      hCursorMag  = LoadCursor(hinst,                 \
                                      MAKEINTRESOURCE(IDC_MAGGLASS1));  \
                        oldCursor = SetCursor(hCursorMag);
#define CURSOR_NOTMAG   SetCursor(oldCursor);                           \
                        DestroyCursor(hCursorMag);

void DoomEdLButtonDown(HDC hdc, WPARAM wParam, LPARAM lParam)
{
  if(DoingRight)
    return;
  else
    DoingLeft = TRUE;
  DownTime = GetTickCount();
  GetCurrent;
  StartPoint = Gridded;
  switch(MouseAction) {
    case MA_NORMAL:
      switch(Tool) {
        case T_THING:
          CurrentThing = NearestThing(Current);
          if(CurrentThing != Nothing)
            DefThing = Thing[CurrentThing];
          if(CurrentThing == Nothing) {
            if(!(wParam & MK_SHIFT)) {
              UnSelectAllThings();
              if(IsWindowVisible(hwndDialogThing))
                PutThingInDialog(DialogClear);
              }
            rDC = GetDC(hwnd);
            StartPoint = Current;
            SelectBrush(rDC, hBrushNull);
            SelectPen(rDC, hPenSelected);
            SelRect.top = SelRect.bottom = StartPoint.y;
            SelRect.left = SelRect.right = StartPoint.x;
            oldROP = SetROP2(rDC, R2_XORPEN);
            // draw new rectangle
            Rectangle(rDC, SelRect.left, SelRect.top,
                           SelRect.right, SelRect.bottom);
            SetROP2(rDC, oldROP);
            ReleaseDC(hwnd, rDC);
            MouseAction = MA_SELWINDOW;
            CURSOR_SEL;
            BottomMessage("Draw a rectangle enclosing the desired things");
            }
          else {
            if(wParam & MK_SHIFT) {
              SelectThing(CurrentThing);
              if(IsWindowVisible(hwndDialogThing))
                PutThingInDialog(CurrentThing);
              MightBeMoving = FALSE;
              }
            else {
              if(!eThing[CurrentThing].Selected) {
                UnSelectAllThings();
                SelectThing(CurrentThing);
                if(IsWindowVisible(hwndDialogThing))
                  PutThingInDialog(CurrentThing);
                }
              MightBeMoving = TRUE;
              }
            }
          break;

        case T_VERTEX:
          CurrentVertex = NearestVertex(Current);
          if(CurrentVertex == Nothing) {
            if(!(wParam & MK_SHIFT))
              UnSelectAllVertexes();
            rDC = GetDC(hwnd);
            StartPoint = Current;
            SelectBrush(rDC, hBrushNull);
            SelectPen(rDC, hPenSelected);
            SelRect.top = SelRect.bottom = StartPoint.y;
            SelRect.left = SelRect.right = StartPoint.x;
            oldROP = SetROP2(rDC, R2_XORPEN);
            // draw new rectangle
            Rectangle(rDC, SelRect.left, SelRect.top,
                           SelRect.right, SelRect.bottom);
            SetROP2(rDC, oldROP);
            ReleaseDC(hwnd, rDC);
            MouseAction = MA_SELWINDOW;
            CURSOR_SEL;
            BottomMessage("Draw a rectangle enclosing the desired vertices");
            }
          else {
            if(wParam & MK_SHIFT) {
              SelectVertex(CurrentVertex);
              MightBeMoving = FALSE;
              }
            else {
              if(!eVertex[CurrentVertex].Selected) {
                UnSelectAllVertexes();
                SelectVertex(CurrentVertex);
                }
              MightBeMoving = TRUE;
              }
            }
          break;

        case T_LINE:
          CurrentLineDef = NearestLineDef(Current);
          if(CurrentLineDef == Nothing) {
            if(!(wParam & MK_SHIFT)) {
              UnSelectAllLineDefs();
              if(IsWindowVisible(hwndDialogLineDef))
                PutLineDefInDialog(DialogClear);
              }
            rDC = GetDC(hwnd);
            StartPoint = Current;
            SelectBrush(rDC, hBrushNull);
            SelectPen(rDC, hPenSelected);
            SelRect.top = SelRect.bottom = StartPoint.y;
            SelRect.left = SelRect.right = StartPoint.x;
            oldROP = SetROP2(rDC, R2_XORPEN);
            // draw new rectangle
            Rectangle(rDC, SelRect.left, SelRect.top,
                           SelRect.right, SelRect.bottom);
            SetROP2(rDC, oldROP);
            ReleaseDC(hwnd, rDC);
            MouseAction = MA_SELWINDOW;
            CURSOR_SEL;
            BottomMessage("Draw a rectangle enclosing the desired lines");
            }
          else {
            if(wParam & MK_SHIFT) {
              SelectLineDef(CurrentLineDef);
              MightBeMoving = FALSE;
              }
            else {
              if(!eLineDef[CurrentLineDef].Selected) {
                UnSelectAllLineDefs();
                SelectLineDef(CurrentLineDef);
                }
              MightBeMoving = TRUE;
              }
            }
          break;

        case T_SECTOR:
          CurrentSector = NearestSector(Current);
          if(CurrentSector == Nothing) {
            if(!(wParam & MK_SHIFT)) {
              UnSelectAllSectors();
              if(IsWindowVisible(hwndDialogSector))
                PutSectorInDialog(DialogClear);
              }  
            rDC = GetDC(hwnd);
            StartPoint = Current;
            SelectBrush(rDC, hBrushNull);
            SelectPen(rDC, hPenSelected);
            SelRect.top = SelRect.bottom = StartPoint.y;
            SelRect.left = SelRect.right = StartPoint.x;
            oldROP = SetROP2(rDC, R2_XORPEN);
            // draw new rectangle
            Rectangle(rDC, SelRect.left, SelRect.top,
                           SelRect.right, SelRect.bottom);
            SetROP2(rDC, oldROP);
            ReleaseDC(hwnd, rDC);
            MouseAction = MA_SELWINDOW;
            CURSOR_SEL;
            BottomMessage("Draw a rectangle enclosing the desired sectors");
            }
          else {
            if(wParam & MK_SHIFT) {
              SelectSector(CurrentSector);
              if(IsWindowVisible(hwndDialogSector)) {
                PutSectorInDialog(CurrentSector);
                // update the start time - it can take awhile to
                // display the floor/ceiling bitmaps
                DownTime = GetTickCount();
                }
              MightBeMoving = FALSE;
              }
            else {
              // within a sector, not holding shift.
              if(!eSector[CurrentSector].Selected) {
                UnSelectAllSectors();
                SelectSector(CurrentSector);
                if(IsWindowVisible(hwndDialogSector)) {
                  PutSectorInDialog(CurrentSector);
                  // update the start time - it can take awhile to
                  // display the sector dialog
                  DownTime = GetTickCount();
                  }
                }
              MightBeMoving = TRUE;
              }
            }
          break;
        }
      GrabMouse;
      break;

    case MA_ZOOMWINDOW:
      // BottomMessage set in CtrlBars.c
//      CURSOR_MAG;
      rDC = GetDC(hwnd);
      SelectBrush(rDC, hBrushNull);
      SelectPen(rDC, hPenSelected);
      StartPoint = Current;
      SelRect.top = SelRect.bottom = StartPoint.y;
      SelRect.left = SelRect.right = StartPoint.x;
      oldROP = SetROP2(rDC, R2_XORPEN);
      // draw new rectangle
      Rectangle(rDC, SelRect.left, SelRect.top,
                     SelRect.right, SelRect.bottom);
      SetROP2(rDC, oldROP);
      ReleaseDC(hwnd, rDC);
      GrabMouse;
      break;

    default:
      break;
    }
}

void DoomEdLDouble    (HDC hdc, WPARAM wParam, LPARAM lParam)
{
  if(Tool == T_LINE)
    PutLineDefInDialog(CurrentLineDef);
  if(Tool == T_SECTOR)
    PutSectorInDialog(CurrentSector);
  if(Tool == T_THING)
    PutThingInDialog(CurrentThing);
}

void DoomEdLButtonUp  (HDC hdc, WPARAM wParam, LPARAM lParam)
{
  BOOL  InsideAnother = FALSE;
  int   i, m, n;

  DoingLeft = FALSE;
  if(HaveMouse) {
    GetCurrent;
    LoseMouse;

    switch(MouseAction) {
      // drop the selected objects at their new location:
      case MA_MOVESECTOR:
        // set cursor back to normal:
        CURSOR_NOTDRAG;
        rDC = GetDC(hwnd);
        oldROP = SetROP2(rDC, R2_XORPEN);
        // erase the old lines:
        SelectPen(rDC, hPenSelected);
        for(i = 0; i < LineDefsNum; i++)
          if(eLineDef[i].Moving) {
            MoveTo(rDC, Vertex[LineDef[i].from].x,
                        Vertex[LineDef[i].from].y);
            LineTo(rDC, Vertex[LineDef[i].to].x,
                        Vertex[LineDef[i].to].y);
            }       // endif movelinedef
        // reset drawing mode to normal
        SetROP2(rDC, oldROP);
        // now draw the new lines
        for(i = 0; i < LineDefsNum; i++)
          if(eLineDef[i].Moving) {
            MoveTo(rDC, Vertex[LineDef[i].from].x,
                        Vertex[LineDef[i].from].y);
            LineTo(rDC, Vertex[LineDef[i].to].x,
                        Vertex[LineDef[i].to].y);
            }       // endif movelinedef
        ReleaseDC(hwnd, rDC);
        // process the new position (in tools.c)
        DropSectors(Current, Gridded);
        // Join any vertexes added by DropSectors
        ProcessJoins();
        // stuff data in dialog if applicable
        if(IsWindowVisible(hwndDialogSector))
          PutSectorInDialog(CurrentSector);
        MouseAction = MA_NORMAL;
        MapChangeNode();
        break;

      case MA_MOVELINEDEF:
        CURSOR_NOTDRAG;
        // mostly the same code as MOVESECTOR
        rDC = GetDC(hwnd);
        oldROP = SetROP2(rDC, R2_XORPEN);
        // erase the old lines:
        SelectPen(rDC, hPenSelected);
        for(i = 0; i < LineDefsNum; i++)
          if(eLineDef[i].Moving) {
            MoveTo(rDC, Vertex[LineDef[i].from].x,
                        Vertex[LineDef[i].from].y);
            LineTo(rDC, Vertex[LineDef[i].to].x,
                        Vertex[LineDef[i].to].y);
            n = (Vertex[LineDef[i].from].x + Vertex[LineDef[i].to].x) / 2;
            m = (Vertex[LineDef[i].from].y + Vertex[LineDef[i].to].y) / 2;
            MoveTo(rDC, n, m); // this is the center of the line just drawn
            LineTo(rDC, n + (Vertex[LineDef[i].to].y - Vertex[LineDef[i].from].y) / 3,
                        m + (Vertex[LineDef[i].from].x - Vertex[LineDef[i].to].x) / 3);
            }       // endif movelinedef
        // reset drawing mode to normal
        SetROP2(rDC, oldROP);
        // now draw the new lines
        for(i = 0; i < LineDefsNum; i++)
          if(eLineDef[i].Moving) {
            MoveTo(rDC, Vertex[LineDef[i].from].x,
                        Vertex[LineDef[i].from].y);
            LineTo(rDC, Vertex[LineDef[i].to].x,
                        Vertex[LineDef[i].to].y);
            n = (Vertex[LineDef[i].from].x + Vertex[LineDef[i].to].x) / 2;
            m = (Vertex[LineDef[i].from].y + Vertex[LineDef[i].to].y) / 2;
            MoveTo(rDC, n, m); // this is the center of the line just drawn
            LineTo(rDC, n + (Vertex[LineDef[i].to].y - Vertex[LineDef[i].from].y) / 3,
                        m + (Vertex[LineDef[i].from].x - Vertex[LineDef[i].to].x) / 3);
            }       // endif movelinedef
        ReleaseDC(hwnd, rDC);
        // Join any vertexes at the same place:
        ProcessJoins();
        MapChangeNode();
        MouseAction = MA_NORMAL;
        break;

      case MA_MOVEVERTEX:
        CURSOR_NOTDRAG;
        rDC = GetDC(hwnd);
        oldROP = SetROP2(rDC, R2_XORPEN);
        // erase the old lines:
        SelectPen(rDC, hPenSelected);
        for(i = 0; i < LineDefsNum; i++)
          if(eLineDef[i].Moving) {
            MoveTo(rDC, Vertex[LineDef[i].from].x,
                        Vertex[LineDef[i].from].y);
            LineTo(rDC, Vertex[LineDef[i].to].x,
                        Vertex[LineDef[i].to].y);
            }       // endif movelinedef
        for(i = 0; i < VertexNum; i++)
          if(eVertex[i].Moving)
            Rectangle(rDC, Vertex[i].x - 4, Vertex[i].y - 4,
                           Vertex[i].x + 4, Vertex[i].y + 4);
        // reset drawing mode to normal
        SetROP2(rDC, oldROP);
        // now draw the new lines
        for(i = 0; i < LineDefsNum; i++)
          if(eLineDef[i].Moving) {
            if((LineDef[i].sidedef1 != Nothing) &&
               (LineDef[i].sidedef2 != Nothing))
              SelectPen(rDC, hPenMapInnerLines);
            else
              SelectPen(rDC, hPenMapLines);
            MoveTo(rDC, Vertex[LineDef[i].from].x,
                        Vertex[LineDef[i].from].y);
            LineTo(rDC, Vertex[LineDef[i].to].x,
                        Vertex[LineDef[i].to].y);
            }       // endif movelinedef
        SelectPen(rDC, hPenSelected);
        if(!VertexSnap)
          for(i = 0; i < VertexNum; i++)
            if(eVertex[i].Moving)
              Rectangle(rDC, Vertex[i].x - 4, Vertex[i].y - 4,
                             Vertex[i].x + 4, Vertex[i].y + 4);
        ReleaseDC(hwnd, rDC);
        // Join any vertexes at the same place:
        ProcessJoins();
        VertexSnap = FALSE;
        MapChangeNode();
        MouseAction = MA_NORMAL;
        break;

      case MA_MOVETHING:
        CURSOR_NOTDRAG;
        rDC = GetDC(hwnd);
        oldROP = SetROP2(rDC, R2_XORPEN);
        // erase the old things:
        SelectPen(rDC, hPenSelected);
        for(i = 0; i < ThingsNum; i++)
          if(eThing[i].Used && eThing[i].Selected)
            Rectangle(rDC, Thing[i].x - 8, Thing[i].y - 8,
                           Thing[i].x + 8, Thing[i].y + 8);
        // reset drawing mode to normal
        SetROP2(rDC, oldROP);
        // now draw the new things
        for(i = 0; i < ThingsNum; i++)
          if(eThing[i].Used && eThing[i].Selected)
            Rectangle(rDC, Thing[i].x - 8, Thing[i].y - 8,
                           Thing[i].x + 8, Thing[i].y + 8);
        ReleaseDC(hwnd, rDC);
        MapChange();
        MouseAction = MA_NORMAL;
        break;

      case MA_NORMAL:
        switch(Tool) {
          case T_THING:
            if(MightBeMoving)
              if((GetTickCount() - DownTime) < TimeOut)
                MightBeMoving = FALSE;
            break;

          case T_VERTEX:
            if(MightBeMoving)
              if((GetTickCount() - DownTime) < TimeOut)
                MightBeMoving = FALSE;
            break;

          case T_LINE:
            if(MightBeMoving)
              if((GetTickCount() - DownTime) < TimeOut)
                MightBeMoving = FALSE;
            break;

          case T_SECTOR:
            if(MightBeMoving)
              if((GetTickCount() - DownTime) < TimeOut)
                MightBeMoving = FALSE;
            break;
          }
        break;

      case MA_SELWINDOW:
        CURSOR_NOTSEL;
        switch(Tool) {
          case T_THING:
            rDC = GetDC(hwnd);
            oldROP = SetROP2(rDC, R2_XORPEN);
            SelectBrush(rDC, hBrushNull);
            SelectPen(rDC, hPenSelected);
            // erase selection rectangle
            Rectangle(rDC, SelRect.left, SelRect.top,
                           SelRect.right, SelRect.bottom);
            SetROP2(rDC, oldROP);
            ReleaseDC(hwnd, rDC);
            BottomMessage("");
            if((abs(SelRect.right - SelRect.left) +
                abs(SelRect.bottom - SelRect.top)) > 24) 
              SelectThing(Nothing);    // uses SelRect
            else
              UnSelectAllThings();
            MouseAction = MA_NORMAL;
            break;

          case T_VERTEX:
            rDC = GetDC(hwnd);
            oldROP = SetROP2(rDC, R2_XORPEN);
            SelectBrush(rDC, hBrushNull);
            SelectPen(rDC, hPenSelected);
            // erase old rectangle
            Rectangle(rDC, SelRect.left, SelRect.top,
                           SelRect.right, SelRect.bottom);
            SetROP2(rDC, oldROP);
            ReleaseDC(hwnd, rDC);
            BottomMessage("");
            if((abs(SelRect.right - SelRect.left) +
                abs(SelRect.bottom - SelRect.top)) > 24) 
              SelectVertex(Nothing);    // uses SelRect
            else
              UnSelectAllVertexes();
            MouseAction = MA_NORMAL;
            break;

          case T_LINE:
            rDC = GetDC(hwnd);
            oldROP = SetROP2(rDC, R2_XORPEN);
            SelectBrush(rDC, hBrushNull);
            SelectPen(rDC, hPenSelected);
            // erase old rectangle
            Rectangle(rDC, SelRect.left, SelRect.top,
                           SelRect.right, SelRect.bottom);
            SetROP2(rDC, oldROP);
            ReleaseDC(hwnd, rDC);
            BottomMessage("");
            if((abs(SelRect.right - SelRect.left) +
                abs(SelRect.bottom - SelRect.top)) > 24) 
              SelectLineDef(Nothing);    // uses SelRect
            else
              UnSelectAllLineDefs();
            MouseAction = MA_NORMAL;
            break;

          case T_SECTOR:
            rDC = GetDC(hwnd);
            oldROP = SetROP2(rDC, R2_XORPEN);
            SelectBrush(rDC, hBrushNull);
            SelectPen(rDC, hPenSelected);
            // erase old rectangle
            Rectangle(rDC, SelRect.left, SelRect.top,
                           SelRect.right, SelRect.bottom);
            SetROP2(rDC, oldROP);
            ReleaseDC(hwnd, rDC);
            BottomMessage("");
            if((abs(SelRect.right - SelRect.left) +
                abs(SelRect.bottom - SelRect.top)) > 24) 
              SelectSector(Nothing);    // uses SelRect
            else
              UnSelectAllSectors();
            MouseAction = MA_NORMAL;
            break;
          }
        break;

      case MA_ZOOMWINDOW:
        CURSOR_NOTMAG;
        rDC = GetDC(hwnd);
        oldROP = SetROP2(rDC, R2_XORPEN);
        SelectBrush(rDC, hBrushNull);
        SelectPen(rDC, hPenSelected);
        // erase old rectangle
        Rectangle(rDC, SelRect.left, SelRect.top,
                       SelRect.right, SelRect.bottom);
        SetROP2(rDC, oldROP);
        ReleaseDC(hwnd, rDC);
        MouseAction = MA_NORMAL;
        BottomMessage("");
        MapExtent = SelRect;
        RepaintMap();
        break;
      default:
        break;
      }     // end switch
    }       // endif HaveMouse
}

void DoomEdRButtonDown(HDC hdc, WPARAM wParam, LPARAM lParam)
{
  HWND  rDC;
  
  if(DoingLeft)
    return;
  else
    DoingRight = TRUE;
  GetCurrent;
  switch(Tool) {
    // right button functions: usually add something.
    case T_THING:
      // place a new thing at the mouse location, then allow it
      // to move as though it were already there and you just
      // selected it.
      CurrentThing = ThingNew();
      Thing[CurrentThing].x = Gridded.x;
      Thing[CurrentThing].y = Gridded.y;
      UnSelectAllThings();
      SelectThing(CurrentThing);
      if(IsWindowVisible(hwndDialogThing))
        PutThingInDialog(CurrentThing);
      MouseAction = MA_MOVETHING;
      LastPoint = Gridded;
      GrabMouse;
      break;

    case T_VERTEX:
      // if mouse is on a line somewhere, break the line and put a
      // new vertex. Then, drag the vertex into its new place. If the
      // new place joins another vertex, process auto-connect as required.
      if((CurrentLineDef = NearestLineDef(Current)) != Nothing) {
        int newl, newv, newsd;
        // Inserting a vertex into a line requires creating a new
        // linedef, and one or two new sidedefs.
        // The new line matches the old one.
        MouseAction = MA_MOVEVERTEX;
        UnSelectAllVertexes();
        rDC = GetDC(hwnd);
        // erase the old line:
        SelectPen(rDC, hPenBackground);
        MoveTo(rDC, Vertex[LineDef[CurrentLineDef].from].x,
                    Vertex[LineDef[CurrentLineDef].from].y);
        LineTo(rDC, Vertex[LineDef[CurrentLineDef].to].x,
                    Vertex[LineDef[CurrentLineDef].to].y);
        // get a new vertex:
        newv = VertexNew();
        // place new vertex where the mouse was clicked:
        Vertex[newv] = Gridded;
        eVertex[newv].Selected = TRUE;
        // adjust the lines to use the new vertex:
        newl = LineDefNew();          // get a new one
        LineDef[newl] = LineDef[CurrentLineDef];
        LineDef[newl].from         = newv;
        LineDef[CurrentLineDef].to = newv;
        // get a new sidedef:
        if(LineDef[CurrentLineDef].sidedef1 != Nothing) {
          newsd = SideDefNew();
          SideDef[newsd] = SideDef[LineDef[CurrentLineDef].sidedef1];
          LineDef[newl].sidedef1  = newsd;
          }
        // get another sidedef if required:
        if(LineDef[CurrentLineDef].sidedef2 != Nothing) {
          newsd = SideDefNew();
          SideDef[newsd] = SideDef[LineDef[CurrentLineDef].sidedef2];
          LineDef[newl].sidedef2  = newsd;
          }          
        // clear move arrays:
        for(tt = 0; tt < MAX_VERTEX; tt++)
          eVertex[tt].Moving = FALSE;
        for(tt = 0; tt < MAX_LINEDEF; tt++)
          eLineDef[tt].Moving = FALSE;
        // mark the appropriate objects as moving:
        eVertex[newv].Moving = TRUE;
        eLineDef[newl].Moving = TRUE;
        eLineDef[CurrentLineDef].Moving = TRUE;
        // now draw the new lines in XOR mode:
        oldROP = SetROP2(rDC, R2_XORPEN);
        SelectPen(rDC, hPenSelected);
        MoveTo(rDC, Vertex[LineDef[CurrentLineDef].from].x,
                    Vertex[LineDef[CurrentLineDef].from].y);
        LineTo(rDC, Vertex[LineDef[CurrentLineDef].to].x,
                    Vertex[LineDef[CurrentLineDef].to].y);
        MoveTo(rDC, Vertex[LineDef[newl].from].x,
                    Vertex[LineDef[newl].from].y);
        LineTo(rDC, Vertex[LineDef[newl].to].x,
                    Vertex[LineDef[newl].to].y);
        // draw the new vertex:
        Rectangle(rDC, Vertex[newv].x - 4, Vertex[newv].y - 4,
                       Vertex[newv].x + 4, Vertex[newv].y + 4);
        SetROP2(rDC, oldROP);
        ReleaseDC(hwnd, rDC);
        LastPoint = Gridded;
        GrabMouse;
        }     // endif valid line is near
      break;
    case T_LINE:
      // if mouse is near a vertex somewhere, then add a new line. Allow
      // the user to rubber band the new line to wherever, but auto-connect
      // it to the nearest other vertex. If there is not vertex within 64,
      // then cancel the operation.
      if((CurrentVertex = NearestVertex(Current)) != Nothing) {
        int newl, newv;
        // Inserting a vertex into a line requires creating a new
        // linedef, and one or two new sidedefs.
        // The new line matches the old one.
        MouseAction = MA_MOVEVERTEX;
        UnSelectAllLineDefs();
        CurrentSector = Nothing;
        // clear move arrays:
        for(tt = 0; tt < MAX_VERTEX; tt++)
          eVertex[tt].Moving = FALSE;
        for(tt = 0; tt < MAX_LINEDEF; tt++)
          eLineDef[tt].Moving = FALSE;
        rDC = GetDC(hwnd);
        newl = LineDefNew();                // get a new line
        CurrentLineDef = newl;
        eLineDef[newl].Moving = TRUE;
        LineDef[newl].sidedef1 = SideDefNew();
        LineDef[newl].sidedef2 = SideDefNew();
        SetLineBit(newl, ML_TWOSIDED);      // two sides, both blank
        SetLineBit(newl, ML_DONTDRAW);      // not visible on map
        ClearLineBit(newl, ML_BLOCKING);    // not blocking player
        LineDef[newl].from = CurrentVertex;
        newv = VertexNew();
        eVertex[newv].Moving = TRUE;
        Vertex[newv] = Gridded;
        LineDef[newl].to = newv;
        eVertex[newv].Selected = TRUE;
        // now draw the new line (and vertex) in XOR mode:
        oldROP = SetROP2(rDC, R2_XORPEN);
        SelectPen(rDC, hPenSelected);
        SelectBrush(rDC, hBrushNull);
        MoveTo(rDC, Vertex[LineDef[newl].from].x,
                    Vertex[LineDef[newl].from].y);
        LineTo(rDC, Vertex[LineDef[newl].to].x,
                    Vertex[LineDef[newl].to].y);
        Rectangle(rDC, Vertex[newv].x - 4, Vertex[newv].y - 4,
                       Vertex[newv].x + 4, Vertex[newv].y + 4);
        SetROP2(rDC, oldROP);
        ReleaseDC(hwnd, rDC);
        LastPoint = Gridded;
        VertexSnap = TRUE;  // snap moving vertex to nearest other vertex
        GrabMouse;
        }     // endif valid line is near
      break;
    case T_SECTOR:
      // If the user stretches out a box, that box becomes
      // the boundary for the new sector. If the user merely
      // right clicks on the sector, the sector attributes
      // can be edited.
      CurrentSector = NearestSector(Current);
      if(CurrentSector == Nothing) {
        rDC = GetDC(hwnd);
        StartPoint = Gridded;
        SelectBrush(rDC, hBrushNull);
        SelectPen(rDC, hPenSelected);
        SelRect.top = SelRect.bottom = StartPoint.y;
        SelRect.left = SelRect.right = StartPoint.x;
        oldROP = SetROP2(rDC, R2_XORPEN);
        // draw new rectangle
        Rectangle(rDC, SelRect.left, SelRect.top,
                       SelRect.right, SelRect.bottom);
        SetROP2(rDC, oldROP);
        ReleaseDC(hwnd, rDC);
        MouseAction = MA_SELWINDOW;
        BottomMessage("Stretch out the extent of a new sector (try Shift & Ctrl)");
        CreatingSector = TRUE;
        GrabMouse;
        }
      else {
        // CurrentSector is the one we are within:
        // multiples might be selected...
        if(!eSector[CurrentSector].Selected)
          SelectSector(CurrentSector);
        MightBeInserting = TRUE;
        }
      break;
    default:
      break;
    }       // end switch
}

void DoomEdRButtonUp  (HDC hdc, WPARAM wParam, LPARAM lParam)
{
  DoingRight = FALSE;
  switch(MouseAction) {
    case MA_MOVEVERTEX:
    case MA_MOVETHING:
      // use what's already tested...
      DoomEdLButtonUp(hdc, wParam, lParam);
      break;

    case MA_SELWINDOW:
      // adding a new sector, and we now know the extent
      // (SelRect)
      LoseMouse;
      CreatingSector = FALSE;
      if((abs(SelRect.right - SelRect.left) +
          abs(SelRect.bottom - SelRect.top)) > 24) {
        PopupNewSector(lParam);
        // handle new sector creation
        ShowingSectorRectangle = TRUE;
        CreatingSectorWithinSector = FALSE;
        }
      MightBeInserting = FALSE;
      ProcessJoins();
      MouseAction = MA_NORMAL;
      break;

    case MA_NORMAL:
      if(Tool == T_SECTOR) {
        if(CurrentSector != Nothing)
          PopupSelectedObject(lParam);    // edit attributes
        }
      break;

    default:
      break;
    }   // end switch MouseAction
}

void DoomEdRDouble    (HDC hdc, WPARAM wParam, LPARAM lParam)
{
}

void DoomEdMouseMove  (HDC hdc, WPARAM wParam, LPARAM lParam)
{
  int   i, dx, dy, m, n;
  
  MouseOnScreen = MAKEPOINT(lParam);    // save for other routines
  GetCurrent;
  if(!CreatingSector)
    DisplayCoords(Gridded.x, Gridded.y);
  if(ShowingSplash)
    SplashDown();
  if(HaveMouse) {
    switch(MouseAction) {

      case MA_ZOOMWINDOW:
      case MA_SELWINDOW:
        if(CreatingSector) {
          dx = abs(Gridded.x - StartPoint.x);
          dy = abs(Gridded.y - StartPoint.y);
          }
        else {
          dx = abs(Current.x - StartPoint.x);
          dy = abs(Current.y - StartPoint.y);
          }
        rDC = GetDC(hwnd);
        oldROP = SetROP2(rDC, R2_XORPEN);
        SelectBrush(rDC, hBrushNull);
        SelectPen(rDC, hPenDrag);
        // erase old rectangle
        Rectangle(rDC, SelRect.left, SelRect.top,
                       SelRect.right, SelRect.bottom);
        // adjust rectangle
        if(CreatingSector) {
          // when creating, show the size of the box, not the mouse location
          DisplayCoords(dx, dy);
          SelRect.top = max(StartPoint.y, Gridded.y);
          SelRect.left = min(StartPoint.x, Gridded.x);
          SelRect.right = max(StartPoint.x, Gridded.x);
          SelRect.bottom = min(StartPoint.y, Gridded.y);
          // only check shift & control when creating, not when selecting
          // both makes square centered on start
          if((wParam & MK_SHIFT) && (wParam & MK_CONTROL)) {
            int distance = max(dx, dy);
            SelRect.top = StartPoint.y + distance;
            SelRect.left = StartPoint.x - distance;
            SelRect.right = StartPoint.x + distance;
            SelRect.bottom = StartPoint.y - distance;
            }
          else {
            // shift key makes square
            if(wParam & MK_SHIFT) {
              if(dy > dx)
                Gridded.x = StartPoint.x + (StartPoint.y - Gridded.y);
              else
                Gridded.y = StartPoint.y + (StartPoint.x - Gridded.x);
              SelRect.top = max(StartPoint.y, Gridded.y);
              SelRect.left = min(StartPoint.x, Gridded.x);
              SelRect.right = max(StartPoint.x, Gridded.x);
              SelRect.bottom = min(StartPoint.y, Gridded.y);
              }
            // control key makes box centered on start
            if(wParam & MK_CONTROL) {
              SelRect.top = StartPoint.y + dy;
              SelRect.left = StartPoint.x - dx;
              SelRect.right = StartPoint.x + dx;
              SelRect.bottom = StartPoint.y - dy;
              }
            }       // endif not both
          }
        else {
          SelRect.top = max(StartPoint.y, Current.y);
          SelRect.left = min(StartPoint.x, Current.x);
          SelRect.right = max(StartPoint.x, Current.x);
          SelRect.bottom = min(StartPoint.y, Current.y);
          }
        // draw new rectangle
        Rectangle(rDC, SelRect.left, SelRect.top,
                       SelRect.right, SelRect.bottom);
        SetROP2(rDC, oldROP);
        ReleaseDC(hwnd, rDC);
        break;

      case MA_MOVEVERTEX:
      case MA_MOVESECTOR:
      case MA_MOVELINEDEF:
        rDC = GetDC(hwnd);
        oldROP = SetROP2(rDC, R2_XORPEN);
        // erase the old lines:
        SelectPen(rDC, hPenSelected);
        SelectBrush(rDC, hBrushNull);
        for(i = 0; i < LineDefsNum; i++)
          if(eLineDef[i].Used && eLineDef[i].Moving) {
            MoveTo(rDC, Vertex[LineDef[i].from].x,
                        Vertex[LineDef[i].from].y);
            LineTo(rDC, Vertex[LineDef[i].to].x,
                        Vertex[LineDef[i].to].y);
            if(MouseAction == MA_MOVELINEDEF) {
              n = (Vertex[LineDef[i].from].x + Vertex[LineDef[i].to].x) / 2;
              m = (Vertex[LineDef[i].from].y + Vertex[LineDef[i].to].y) / 2;
              MoveTo(rDC, n, m); // this is the center of the line just drawn
              LineTo(rDC, n + (Vertex[LineDef[i].to].y - Vertex[LineDef[i].from].y) / 3,
                          m + (Vertex[LineDef[i].from].x - Vertex[LineDef[i].to].x) / 3);
              }
            }       // endif movelinedef
        if(MouseAction == MA_MOVEVERTEX)
          for(i = 0; i < VertexNum; i++)
            if(eVertex[i].Used && eVertex[i].Moving)
              Rectangle(rDC, Vertex[i].x - 4, Vertex[i].y - 4,
                             Vertex[i].x + 4, Vertex[i].y + 4);
        // Update the vertex positions:
        dx = Gridded.x - LastPoint.x;
        dy = Gridded.y - LastPoint.y;
        for(i = 0; i < VertexNum; i++)
          if(eVertex[i].Used && eVertex[i].Moving) {
            if(VertexSnap) {
              int   x = NearestVertex(Current);
              if(CurrentSector == Nothing)
                CurrentSector = NearestSector(Current);
              if(x == Nothing)
                Vertex[i] = Gridded;
              else {
                Vertex[i] = Vertex[x];
                if(CurrentSector != Nothing) {
                  // we know there is both a sidedef1 and 2, since this
                  // is only called when adding a line inside a sector
                  SideDef[LineDef[CurrentLineDef].sidedef1].sector = CurrentSector;
                  SideDef[LineDef[CurrentLineDef].sidedef2].sector = CurrentSector;
                  }
                }
              }
            else {
              Vertex[i].x += dx;
              Vertex[i].y += dy;
              }
            }
            
        // now draw the new lines
        for(i = 0; i < LineDefsNum; i++)
          if(eLineDef[i].Used && eLineDef[i].Moving) {
            MoveTo(rDC, Vertex[LineDef[i].from].x,
                        Vertex[LineDef[i].from].y);
            LineTo(rDC, Vertex[LineDef[i].to].x,
                        Vertex[LineDef[i].to].y);
            if(MouseAction == MA_MOVELINEDEF) {
              n = (Vertex[LineDef[i].from].x + Vertex[LineDef[i].to].x) / 2;
              m = (Vertex[LineDef[i].from].y + Vertex[LineDef[i].to].y) / 2;
              MoveTo(rDC, n, m); // this is the center of the line just drawn
              LineTo(rDC, n + (Vertex[LineDef[i].to].y - Vertex[LineDef[i].from].y) / 3,
                          m + (Vertex[LineDef[i].from].x - Vertex[LineDef[i].to].x) / 3);
              }
            }       // endif movelinedef
        if(MouseAction == MA_MOVEVERTEX)
          for(i = 0; i < VertexNum; i++)
            if(eVertex[i].Used && eVertex[i].Moving)
              Rectangle(rDC, Vertex[i].x - 4, Vertex[i].y - 4,
                             Vertex[i].x + 4, Vertex[i].y + 4);
        SetROP2(rDC, oldROP);
        ReleaseDC(hwnd, rDC);
        break;

      case MA_MOVETHING:
        rDC = GetDC(hwnd);
        oldROP = SetROP2(rDC, R2_XORPEN);
        // erase the old things:
        SelectPen(rDC, hPenSelected);
        SelectBrush(rDC, hBrushNull);
        for(i = 0; i < ThingsNum; i++)
          if(eThing[i].Used && eThing[i].Selected)
            Rectangle(rDC, Thing[i].x - 8, Thing[i].y - 8,
                           Thing[i].x + 8, Thing[i].y + 8);
        // Update the thing positions:
        dx = Gridded.x - LastPoint.x;
        dy = Gridded.y - LastPoint.y;
        for(i = 0; i < ThingsNum; i++)
          if(eThing[i].Used && eThing[i].Selected) {
            Thing[i].x += dx;
            Thing[i].y += dy;
            }
        // now draw the new things
        for(i = 0; i < ThingsNum; i++)
          if(eThing[i].Used && eThing[i].Selected)
            Rectangle(rDC, Thing[i].x - 8, Thing[i].y - 8,
                           Thing[i].x + 8, Thing[i].y + 8);
        SetROP2(rDC, oldROP);
        ReleaseDC(hwnd, rDC);
        break;
      
      case MA_NORMAL:
        if(DoingRight) {
          if((GetTickCount() - DownTime) > TimeOut) {
            // mouse button held - move object with mouse
            if(Tool == T_THING) {
              MouseAction = MA_MOVETHING;
              rDC = GetDC(hwnd);
              // erase the old things:
              SelectBrush(rDC, hBrushNull);
              SelectPen(rDC, hPenBackground);
              for(i = 0; i < ThingsNum; i++)
                if(eThing[i].Used && eThing[i].Selected)
                  Rectangle(rDC, Thing[i].x - 8, Thing[i].y - 8,
                                 Thing[i].x + 8, Thing[i].y + 8);
              // snap the moving thing(s) to the grid in case
              // they weren't on the grid to start with
              for(i = 0; i < ThingsNum; i++)
                if(eThing[i].Used && eThing[i].Selected) {
                  Thing[i].x = (Thing[i].x / GridSize) * GridSize;
                  Thing[i].y = (Thing[i].y / GridSize) * GridSize;
                  }
              // user probably dragged while waiting for timeout,
              // so update the thing position(s):
              dx = Gridded.x - StartPoint.x;
              dy = Gridded.y - StartPoint.y;
              for(i = 0; i < ThingsNum; i++)
                if(eThing[i].Used && eThing[i].Selected) {
                  Thing[i].x += dx;
                  Thing[i].y += dy;
                  }
              // now draw the new things in XOR mode:
              oldROP = SetROP2(rDC, R2_XORPEN);
              SelectPen(rDC, hPenSelected);
              SelectBrush(rDC, hBrushNull);
              for(i = 0; i < ThingsNum; i++)
                if(eThing[i].Used && eThing[i].Selected)
                  Rectangle(rDC, Thing[i].x - 8, Thing[i].y - 8,
                                 Thing[i].x + 8, Thing[i].y + 8);
              SetROP2(rDC, oldROP);
              ReleaseDC(hwnd, rDC);
              LastPoint = Gridded;
              }     // endif tool == t_thing
            }       // endif timeout
          }         // endif doingright
        if(DoingLeft) {
          if((GetTickCount() - DownTime) > TimeOut) {
            // mouse button held - move object with mouse
            switch(Tool) {
              case T_THING:
                MouseAction = MA_MOVETHING;
                CURSOR_DRAGTV;
                rDC = GetDC(hwnd);
                // erase the old things:
                SelectPen(rDC, hPenBackground);
                SelectBrush(rDC, hBrushNull);
                for(i = 0; i < ThingsNum; i++)
                  if(eThing[i].Used && eThing[i].Selected)
                    Rectangle(rDC, Thing[i].x - 8, Thing[i].y - 8,
                                   Thing[i].x + 8, Thing[i].y + 8);
                // snap the moving thing(s) to the grid in case
                // they weren't on the grid to start with
                for(i = 0; i < ThingsNum; i++)
                  if(eThing[i].Used && eThing[i].Selected) {
                    Thing[i].x = (Thing[i].x / GridSize) * GridSize;
                    Thing[i].y = (Thing[i].y / GridSize) * GridSize;
                    }
                // user probably dragged while waiting for timeout,
                // so update the thing position(s):
                dx = Gridded.x - StartPoint.x;
                dy = Gridded.y - StartPoint.y;
                for(i = 0; i < ThingsNum; i++)
                  if(eThing[i].Used && eThing[i].Selected) {
                    Thing[i].x += dx;
                    Thing[i].y += dy;
                    }
                // now draw the new things in XOR mode:
                oldROP = SetROP2(rDC, R2_XORPEN);
                SelectPen(rDC, hPenSelected);
                SelectBrush(rDC, hBrushNull);
                for(i = 0; i < ThingsNum; i++)
                  if(eThing[i].Used && eThing[i].Selected)
                    Rectangle(rDC, Thing[i].x - 8, Thing[i].y - 8,
                                   Thing[i].x + 8, Thing[i].y + 8);
                SetROP2(rDC, oldROP);
                ReleaseDC(hwnd, rDC);
                LastPoint = Gridded;
                break;

              case T_VERTEX:
                MouseAction = MA_MOVEVERTEX;
                CURSOR_DRAGTV;
                // clear move arrays:
                for(tt = 0; tt < MAX_VERTEX; tt++)
                  eVertex[tt].Moving = FALSE;
                for(tt = 0; tt < MAX_LINEDEF; tt++)
                  eLineDef[tt].Moving = FALSE;
                // check each vertex
                for(i = 0; i < VertexNum; i++)
                  eVertex[i].Moving = (eVertex[i].Selected && eVertex[i].Used);
                // check each line
                for(i = 0; i < LineDefsNum; i++)
                  if(eLineDef[i].Used &&
                     (eVertex[LineDef[i].from].Selected || 
                      eVertex[LineDef[i].to].Selected))
                    eLineDef[i].Moving = TRUE;
                // get device context
                rDC = GetDC(hwnd);
                // erase the old lines:
                SelectPen(rDC, hPenBackground);
                for(i = 0; i < LineDefsNum; i++)
                  if(eLineDef[i].Used && eLineDef[i].Moving) {
                    MoveTo(rDC, Vertex[LineDef[i].from].x,
                                Vertex[LineDef[i].from].y);
                    LineTo(rDC, Vertex[LineDef[i].to].x,
                                Vertex[LineDef[i].to].y);
                    }       // endif movelinedef
                for(i = 0; i < VertexNum; i++)
                  if(eVertex[i].Used && eVertex[i].Moving)
                    Rectangle(rDC, Vertex[i].x - 4, Vertex[i].y - 4,
                                   Vertex[i].x + 4, Vertex[i].y + 4);
                // snap the moving vertex(es) to the grid in case
                // they weren't on the grid to start with
                for(i = 0; i < VertexNum; i++)
                  if(eVertex[i].Used && eVertex[i].Moving) {
                    Vertex[i].x = (Vertex[i].x / GridSize) * GridSize;
                    Vertex[i].y = (Vertex[i].y / GridSize) * GridSize;
                    }
                // user probably dragged while waiting for timeout,
                // so update the vertex positions:
                dx = Gridded.x - StartPoint.x;
                dy = Gridded.y - StartPoint.y;
                for(i = 0; i < VertexNum; i++)
                  if(eVertex[i].Moving) {
                    Vertex[i].x += dx;
                    Vertex[i].y += dy;
                    }
                // now draw the new lines in XOR mode:
                oldROP = SetROP2(rDC, R2_XORPEN);
                SelectPen(rDC, hPenSelected);
                for(i = 0; i < LineDefsNum; i++)
                  if(eLineDef[i].Used && eLineDef[i].Moving) {
                    MoveTo(rDC, Vertex[LineDef[i].from].x,
                                Vertex[LineDef[i].from].y);
                    LineTo(rDC, Vertex[LineDef[i].to].x,
                                Vertex[LineDef[i].to].y);
                    }       // endif movelinedef
                for(i = 0; i < VertexNum; i++)
                  if(eVertex[i].Used && eVertex[i].Moving)
                    Rectangle(rDC, Vertex[i].x - 4, Vertex[i].y - 4,
                                   Vertex[i].x + 4, Vertex[i].y + 4);
                SetROP2(rDC, oldROP);
                ReleaseDC(hwnd, rDC);
                LastPoint = Gridded;
                break;

              case T_LINE:
                MouseAction = MA_MOVELINEDEF;
                CURSOR_DRAGLS;
                // clear move arrays:
                for(tt = 0; tt < MAX_VERTEX; tt++)
                  eVertex[tt].Moving = FALSE;
                for(tt = 0; tt < MAX_LINEDEF; tt++)
                  eLineDef[tt].Moving = FALSE;
                // check each line
                for(i = 0; i < LineDefsNum; i++)
                  if(eLineDef[i].Used && eLineDef[i].Selected) {
                    // if both sides selected,
                    // add it to the movelist.
                    int   newv;
                      
                    newv = VertexNew();
                    Vertex[newv] = Vertex[LineDef[i].from];
                    LineDef[i].from = newv;
                    eVertex[newv].Moving = TRUE;

                    newv = VertexNew();
                    Vertex[newv] = Vertex[LineDef[i].to];
                    LineDef[i].to = newv;
                    eVertex[newv].Moving = TRUE;

                    eLineDef[i].Moving = TRUE;
                    }         // endif linedef Used & Selected
                // get device context
                rDC = GetDC(hwnd);
                // erase the old lines:
                SelectPen(rDC, hPenBackground);
                for(i = 0; i < LineDefsNum; i++)
                  if(eLineDef[i].Used && eLineDef[i].Moving) {
                    MoveTo(rDC, Vertex[LineDef[i].from].x,
                                Vertex[LineDef[i].from].y);
                    LineTo(rDC, Vertex[LineDef[i].to].x,
                                Vertex[LineDef[i].to].y);
                    n = (Vertex[LineDef[i].from].x + Vertex[LineDef[i].to].x) / 2;
                    m = (Vertex[LineDef[i].from].y + Vertex[LineDef[i].to].y) / 2;
                    MoveTo(rDC, n, m); // this is the center of the line just drawn
                    LineTo(rDC, n + (Vertex[LineDef[i].to].y - Vertex[LineDef[i].from].y) / 3,
                                m + (Vertex[LineDef[i].from].x - Vertex[LineDef[i].to].x) / 3);
                    }       // endif movelinedef
                for(i = 0; i < LineDefsNum; i++)
                  if(eLineDef[i].Used && !eLineDef[i].Moving) {
                    if((LineDef[i].sidedef1 != Nothing) &&
                       (LineDef[i].sidedef2 != Nothing))
                      SelectPen(rDC, hPenMapInnerLines);
                    else
                      SelectPen(rDC, hPenMapLines);
                    MoveTo(rDC, Vertex[LineDef[i].from].x,
                                Vertex[LineDef[i].from].y);
                    LineTo(rDC, Vertex[LineDef[i].to].x,
                                Vertex[LineDef[i].to].y);
                    n = (Vertex[LineDef[i].from].x + Vertex[LineDef[i].to].x) / 2;
                    m = (Vertex[LineDef[i].from].y + Vertex[LineDef[i].to].y) / 2;
                    MoveTo(rDC, n, m); // this is the center of the line just drawn
                    LineTo(rDC, n + (Vertex[LineDef[i].to].y - Vertex[LineDef[i].from].y) / 3,
                                m + (Vertex[LineDef[i].from].x - Vertex[LineDef[i].to].x) / 3);
                    }       // endif movelinedef
                // user probably dragged while waiting for timeout,
                // so update the vertex positions:
                dx = Gridded.x - StartPoint.x;
                dy = Gridded.y - StartPoint.y;
                for(i = 0; i < VertexNum; i++)
                  if(eVertex[i].Moving) {
                    Vertex[i].x += dx;
                    Vertex[i].y += dy;
                    }
                // now draw the new lines in XOR mode:
                oldROP = SetROP2(rDC, R2_XORPEN);
                SelectPen(rDC, hPenSelected);
                for(i = 0; i < LineDefsNum; i++)
                  if(eLineDef[i].Used && eLineDef[i].Moving) {
                    MoveTo(rDC, Vertex[LineDef[i].from].x,
                                Vertex[LineDef[i].from].y);
                    LineTo(rDC, Vertex[LineDef[i].to].x,
                                Vertex[LineDef[i].to].y);
                    n = (Vertex[LineDef[i].from].x + Vertex[LineDef[i].to].x) / 2;
                    m = (Vertex[LineDef[i].from].y + Vertex[LineDef[i].to].y) / 2;
                    MoveTo(rDC, n, m); // this is the center of the line just drawn
                    LineTo(rDC, n + (Vertex[LineDef[i].to].y - Vertex[LineDef[i].from].y) / 3,
                                m + (Vertex[LineDef[i].from].x - Vertex[LineDef[i].to].x) / 3);
                    }       // endif movelinedef
                SetROP2(rDC, oldROP);
                ReleaseDC(hwnd, rDC);
                LastPoint = Gridded;
                break;

              case T_SECTOR:
                // the user is dragging a sector or sectors
                MouseAction = MA_MOVESECTOR;
                CURSOR_DRAGLS;
                // break the sector away
                // if user holds SHIFT, turf connection lines
                if(wParam & MK_SHIFT) {
                  ExtractSectors();
                  }
                // otherwise, keep old connection lines
                else {
                  DetachSectors();
                  }             // end else

                // get device context
                rDC = GetDC(hwnd);
                // erase the old lines:
                SelectPen(rDC, hPenBackground);
                for(i = 0; i < LineDefsNum; i++)
                  if(eLineDef[i].Used && eLineDef[i].Moving) {
                    MoveTo(rDC, Vertex[LineDef[i].from].x,
                                Vertex[LineDef[i].from].y);
                    LineTo(rDC, Vertex[LineDef[i].to].x,
                                Vertex[LineDef[i].to].y);
                    }       // endif movelinedef
                // draw the newly added lines
                for(i = 0; i < LineDefsNum; i++)
                  if(eLineDef[i].Used && !eLineDef[i].Moving) {
                    if((LineDef[i].sidedef1 != Nothing) &&
                       (LineDef[i].sidedef2 != Nothing))
                      SelectPen(rDC, hPenMapInnerLines);
                    else
                      SelectPen(rDC, hPenMapLines);
                    MoveTo(rDC, Vertex[LineDef[i].from].x,
                                Vertex[LineDef[i].from].y);
                    LineTo(rDC, Vertex[LineDef[i].to].x,
                                Vertex[LineDef[i].to].y);
                    }       // endif movelinedef
                // user probably dragged while waiting for timeout,
                // so update the vertex positions:
                dx = Gridded.x - StartPoint.x;
                dy = Gridded.y - StartPoint.y;
                for(i = 0; i < VertexNum; i++)
                  if(eVertex[i].Used && eVertex[i].Moving) {
                    Vertex[i].x += dx;
                    Vertex[i].y += dy;
                    }
                // now draw the new lines in XOR mode:
                oldROP = SetROP2(rDC, R2_XORPEN);
                SelectPen(rDC, hPenSelected);
                for(i = 0; i < LineDefsNum; i++)
                  if(eLineDef[i].Used && eLineDef[i].Moving) {
                    MoveTo(rDC, Vertex[LineDef[i].from].x,
                                Vertex[LineDef[i].from].y);
                    LineTo(rDC, Vertex[LineDef[i].to].x,
                                Vertex[LineDef[i].to].y);
                    }       // endif movelinedef
                SetROP2(rDC, oldROP);
                ReleaseDC(hwnd, rDC);
                LastPoint = Gridded;
                break;
                }             // end switch Tool
            return;
            }                 // endif GetTickCount
          }                   // endif DoingLeft
      default:
        break;
      }     // end switch mouseaction
    LastPoint = Gridded;
    }       // endif HaveMouse
}

void DoomEdNCMouseMove(void)
{
  if(ShowingSplash)
    SplashDown();
}
