
 /* 
 
 Copyright(C) 2007 Simon Howard

 This program is free software; you can redistribute it and/or
 modify it under the terms of the GNU General Public License
 as published by the Free Software Foundation; either version 2
 of the License, or (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 02111-1307, USA.

 --

 Doom -statcopy external statistics driver.

 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <process.h>

#include "moushook.h"
#include "stats.h"
#include "statprnt.h"

/* Array of end-of-level statistics that have been captured. */

#define MAX_CAPTURES 32
static wbstartstruct_t captured_stats[MAX_CAPTURES];
static int num_captured_stats = 0;

/* Statistics buffer that Doom writes into. */

static wbstartstruct_t stats_buffer;

/* Command line to run Doom with. */

static int doom_argc;
static char **doom_argv;

/* Output file to write statistics to.  If NULL, print to stdout. */

static char *output_filename = NULL;

/*
 * This callback function is invoked whenever Doom calls into the mouse
 * driver.  We check the "maxfrags" variable, to see if the stats 
 * buffer has been written to.  If it has, save the contents of the
 * stats buffer into the captured_stats array for later processing.
 *
 * Why the mouse driver?  Well, ideally, we'd install a timer interrupt
 * to periodically check the statistics buffer.  However, Doom installs
 * its own timer interrupt and doesn't chain into whatever was there 
 * before.  The same is true of the keyboard interrupt, so we can't
 * hook into that either.  This was the best I could come up with.
 */

static void my_mouse_callback(void)
{
        if (stats_buffer.maxfrags == 0) {
                /* New data has been written to the statistics buffer.
                   Save it for later processing. */

                if (num_captured_stats < MAX_CAPTURES) {
                        memcpy(&captured_stats[num_captured_stats],
                               &stats_buffer,
                               sizeof(wbstartstruct_t));
                        ++num_captured_stats;
                }

                stats_buffer.maxfrags = 1;
        }
}

/* Help page. */

static void usage(char *program_name)
{
        printf("Usage: %s [options] program [program options]\n",
               program_name);
        printf("\n"
               "Options:\n"
               "\t-o <filename>   :  Write output to the given file.\n"
               "\n");

        printf("Examples:\n");
        printf("\t%s doom -warp 1\n"
               "\t   - Start doom.exe, warping to level 1.\n", program_name);
        printf("\t%s -o stats.txt ipxsetup -nodes 4 -warp 1 4\n"
               "\t   - Start a 4 player IPX multiplayer game, "
                       "warping to E1M4.\n"
               "\t     Output is saved to stats.txt.\n", program_name);

        exit(-1);
}

/* Parse the command line. */

static void parse_cmd_line(int argc, char *argv[])
{
        int i;

        doom_argv = NULL;

        for (i=1; i<argc; ++i) {

                /* If this is a command line option, parse it */

                if (argv[i][0] == '-') {
                        switch (argv[i][1]) {
                                case 'o':
                                        output_filename = argv[i + 1];
                                        ++i;
                                        break;
                                default:
                                        usage(argv[0]);
                                        break;
                        }
                } else {
                        /* Not a command line option, so this is the
                         * program name.  Save doom_argv and finish. */

                        doom_argc = argc - i;
                        doom_argv = &argv[i];
                        break;
                }
        }

        /* Check that a command line was provided. */

        if (doom_argv == NULL) {
                usage(argv[0]);
        }
}

/* Launch doom. */

static void launch_doom(void)
{
        int actual_argc;
        char **actual_argv;
        char addr_string[32];
        int i;

        /* Initialise maxfrags field, used to detect writes to the stats
           buffer. */

        stats_buffer.maxfrags = 1;

        /* Install the mouse hook. */

        init_mouse_hook(my_mouse_callback); 

        /* Build the command line arguments.  Add the -statcopy argument. */

        actual_argc = doom_argc + 2;
        actual_argv = malloc((actual_argc + 1) * sizeof(char *));
        memcpy(actual_argv, doom_argv, doom_argc * sizeof(char *));

        actual_argv[doom_argc] = "-statcopy";
        sprintf(addr_string, "%li", 
                (long) (_DS * 16) + (unsigned) (&stats_buffer));
        actual_argv[doom_argc + 1] = addr_string;
        actual_argv[doom_argc + 2] = NULL;

/*
        for (i=0; i<actual_argc; ++i) {
                printf("%i: %s\n", i, actual_argv[i]);
        }
*/

        /* Launch Doom! */
        
        spawnv(P_WAIT, actual_argv[0], actual_argv);

        free(actual_argv);

        /* Restore the mouse interrupt. */

        shutdown_mouse_hook(); 

}

/* Write the statistics to the output file. */

static void write_stats(void)
{
        FILE *outfile;
        int i;

        /* Open the output file for writing.  If none is specified,
         * write the data to stdout. */

        if (output_filename != NULL) {
                outfile = fopen(output_filename, "w");

                if (outfile == NULL) {
                        fprintf(stderr, "Failed to open '%s' for write.\n",
                                output_filename);
                        return;
                }
        } else {
                outfile = stdout;
        }

        /* Work out if this was Doom 1 or Doom 2. */

        discover_gamemode(captured_stats, num_captured_stats);

        /* Write the statistics for each level to the file. */

        for (i=0; i<num_captured_stats; ++i) {
                print_stats(outfile, &captured_stats[i]);
        }

        /* Close the output file */

        if (output_filename != NULL) {
                fclose(outfile);
        }
}

int main(int argc, char *argv[])
{
        /* Read the command line */

        parse_cmd_line(argc, argv);

        /* Launch Doom */

        launch_doom();

        printf("Statistics captured for %i level(s)\n", num_captured_stats);

        /* Write statistics to the output file. */

        write_stats();

        return 0;
}

