#include "wads.h"
#define MAXPATH 256

void Usage(void)
{
  printf ("Usage: extract in.wad pattern [-{A|a}ENTRY] [-{B|b}ENTRY] [-wout.wad|path]\n");
  printf ("       extract in.wad ENTRY   [-{A|a}ENTRY] [-{B|b}ENTRY] [-wout.wad|fname.ext]\n");
  printf ("\n");
  printf ("    in.wad is the source file (IWAD or PWAD);\n");
  printf ("    pattern specifies which entries to extract (using * and ?);\n");
  printf ("    ENTRY is an entry name (no wildcards)\n");
  printf ("          NOTE: both pattern and ENTRY are case sensitive;\n");
  printf ("    -aENTRY: only entries AFTER  the first occurrence of ENTRY are extracted;\n");
  printf ("    -bENTRY: only entries BEFORE the first occurrence of ENTRY are extracted;\n");
  printf ("          Use -A or -B to include ENTRY itself;\n");
  printf ("    -wout.wad: all spcified entries are extracted to a PWAD named out.wad;\n");
  printf ("    path specifies where the output files will be sent;\n");
  printf ("    fname.ext specifies the file to which ENTRY will be extracted.\n");
  printf ("\n");
  printf ("Examples: extract doom.wad E1M1\n");
  printf ("          extract doom.wad DS* \\tmp\\sounds\n");
  printf ("          extract doom.wad TITLEPIC doomlogo.pic\n");
  printf ("          extract doom.wad THINGS -aE1M6 -bE1M7 e1m6thng.raw\n");
  printf ("          extract doom.wad * -AF_START -BF_END -wflats.wad\n");
  exit (1);
}

/* counters */
unsigned long i=0;          /* incremented each time an entry is read in */
unsigned long item_count=0; /* incremented each time something is extracted */

/* flags set during command line parsing */
int wildcard=0; /* set if the ENTRY field contains wildcards */
int wad=0;      /* set if the '-w' option is used */
int outpath=0;  /* set if an output path/filename is specified */

/* flags set when an appropriate entry is read */
int sprites=0; /* set on S_START and reset to 0 on S_END */
int patches=0; /* set on P_START and reset to 0 on P_END */
int flats=0;   /* set on F_START and reset to 0 on F_END */
int extrmap=0; /* set while a map is being extracted */

/* misc */
FILE *infile;        /* pointer to the (open) input file */
WAD *outwad;         /* pointer to the output wad (if wad!=0 or extrmap!=0) */
char *pattern;       /* pointre to the pattern string (usually argv[2]) */
char *outpathstr=""; /* pointer to the output path/file string (if any) */

int exit_code=0;     /* 0 initially, set to 1 on any warning */

void process_entry (DirEntry entry)
/* This function and the loop it's called from form a finite state machine.
   The state is represented by extrmap. */
{
  char temp[9];

  if (extrmap) {
    if (map_data(name2str(entry.name,temp,8)))
      copy_entry_to_wad(outwad,infile,entry);
    else {
      if (!wad)
	close_wad(outwad,0);
      extrmap=0;
      process_entry(entry); /* This is not really recursion */
    }
  }
  else { /* not extrmap */
    if (!map_data(name2str(entry.name,temp,8)))
      extrmap=0;
    if (!strcmp("S_START",temp))
      sprites=1;
    else if (!strcmp("F_START",temp))
      flats=1;
    else if (!strcmp("P_START",temp))
      patches=1;
    else if (!strcmp("S_END",temp))
      sprites=0;
    else if (!strcmp("P_END",temp))
      patches=0;
    else if (!strcmp("F_END",temp))
      flats=0;
    if (wcmatch(pattern,temp)) {
      if (mapname(temp))
	extrmap=1;
      if (!wad) { /* open a file */
	char fname[MAXPATH];

	/* put the output file name together */
	if ((outpath)&&(!wildcard))
	  strcpy(fname,outpathstr);
	else {
	  strcpy (fname,temp);
	  if (sprites) strcat (fname,".spr");
	  else if (patches) strcat (fname,".ptc");
	  else if (flats) strcat (fname,".flt");
	  else if (extrmap) strcat(fname,".wad");
	  else if (wcmatch("DS*",temp)) strcat (fname,".snd");
	  else if (wcmatch("DP*",temp)) strcat (fname,".pcs");
	  else if (wcmatch("D_*",temp)) strcat (fname,".mus");
	  else strcat (fname,".raw");
	  if (outpath) {
	    char tmp[13];
	    strcpy(tmp,fname);
	    strcpy(fname,outpathstr);
	    strcat(fname,"\\");
	    strcat(fname,tmp);
	  }
	} /* ((!outpath)||(wildcard)) */
	/* done putting the filename together */

	if (extrmap) { /* this code is executed only once per map */
	  if ((outwad=new_wad(fname))==NULL) {
	    printf ("CAN'T OPEN %s;\n",fname);
	    extrmap=0;
            exit_code=1;
	  }
	  else {
	    printf ("extracting map %s to %s;\n",temp,fname);
	    item_count++;
	    copy_entry_to_wad (outwad,infile,entry);
	  }
	}
	else {
	  FILE *outfile;
          if ((outfile=fopen(fname,"wb"))==NULL) {
	    printf ("CAN'T OPEN %s;\n",fname);
            exit_code=1;
          }
	  else {
	    printf ("extracting %s to %s;\n",temp,fname);
	    item_count++;
	    copy_entry_to_file(outfile,infile,entry);
	    fclose(outfile);
	  }
	} /* if not extrmap */
      } /* if not wad */
      else { /* wad */
	printf ("extracting %s to %s;\n",temp,outpathstr);
	item_count++;
	copy_entry_to_wad(outwad,infile,entry);
      }
    } /* if wcmatch */
  } /* else (i. e., if not extrmap) */
}

char *assign_if_empty (char **dest, char *src)
{
  if (strcmp(*dest,""))
    UsageError ("Extra parameter: %s",src);
  else
    *dest=src;
  return (*dest);
}

main (int argc, char *argv[])
{
  Header header;
  DirEntry direntry;
  char temp[9];
  int after=0;
  int before=0;
  int After=0;
  int Before=0;
  char *afterstr="";
  char *beforestr="";

  printf ("WAD extract utility by Serge Smirnov -- compiled on %s at %s\n\n",__DATE__,__TIME__);

  if (argc<3)
    Usage();
  if (wc(pattern=argv[2]))
    wildcard=1;

  for (i=3;i<argc;i++) {
    if (argv[i][0]=='-')
      switch (argv[i][1]) {
	case 'a':after=1;  assign_if_empty(&afterstr,argv[i]+2);   break;
	case 'A':After=1;  assign_if_empty(&afterstr,argv[i]+2);   break;
	case 'b':before=1; assign_if_empty(&beforestr,argv[i]+2);  break;
	case 'B':Before=1; assign_if_empty(&beforestr,argv[i]+2);  break;
	case 'w':wad=1;    assign_if_empty(&outpathstr,argv[i]+2); break;
	default :UsageError ("Unrecognized option: %s",argv[i]);
      }
    else {
      outpath=1;
      assign_if_empty (&outpathstr,argv[i]);
    }
  }

  if ((infile=fopen(argv[1],"rb"))==NULL)
    UsageError ("Can't open %s",argv[1]);
  fread (&header, sizeof (header), 1, infile);
  if (strcmp (name2str(header.type+1,temp,3),"WAD"))
    UsageError ("%s is not a WAD file",argv[1]);
  if (wad)
    if ((outwad=new_wad(outpathstr))==NULL)
      UsageError ("Can't open output wad %s",outpathstr);

  fseek (infile, header.dir_start,SEEK_SET);
  item_count=0;
  i=0;
  if (after||After)
    while (i<header.num_entries) {
      fread (&direntry, sizeof (direntry), 1, infile);
      i++;
      if (!strcmp(afterstr,name2str(direntry.name,temp,8)))
	break;
    }
  if (After) /* include this entry */
    process_entry (direntry);

  for (;i<header.num_entries;i++) {
    fread (&direntry, sizeof (direntry), 1, infile);
    if ((before||Before)&&(!strcmp(beforestr,name2str(direntry.name,temp,8)))) {
      if (Before) /* include the last one */
	process_entry(direntry);
      break;
    }
    else
      process_entry(direntry);
  }

  if (extrmap) { /* if the input suddenly ends and a map is still not done */
    pattern=""; /* to make sure whatever it is it doesn't get processed */
    strcpy(direntry.name,"@#$%^&*"); /* i. e., not a map */
    process_entry(direntry); /* finish extracting the map */
  }

  if (wad)
    close_wad(outwad,0);

  fclose (infile);
  printf ("\ndone: extracted %lu items.\n",item_count);
  return(exit_code);
}
