/* mylib.c -- a collection
   of 'jackets' for some standard functions.  These jackets provide
   additional error checking and report errors gracefully.
   To use this mini-library, #include "mylib\mylib.h" into your source
   code.  If you wish to use my file IO routines, #define MYFILEIO_YES
   BEFORE the #include statement.
   The header #defines all standard functions for which this library
   provides jackets to the names of the corresponding jackets.  Instead of
   the standard FILE structure, my file IO routines use my own structure
   called MILE, which actually includes a pointer to a FILE structure.
   However, the header also redefines the meaning of the identifier FILE,
   so all you need to do is
     #define MYFILEIO_YES
     #include "mylib.h"

   NOTE: You will encounter problems if you define MYFILEIO_YES and try to
   use any function for which this library does not provide a jacket, and
   which requires you to define variables of type FILE*.  An example is
   getc().  If you wish to use such functions as well as my functions,
   you have two options:
     (1) Do not define MYFILEIO_YES and call my functions by their real
	 names (e. g., filewrite instead of fwrite);
     (2) Define MYFILIO_YES, but #undef FILE, so that you can still use it.
	 You will need to explicitly use MILE instead of FILE for those
	 files accessed through my functions.
   Similarly, confusion may arise if you try to use stdin and stdout
   with any of my functions.  This may be fixed in the future.  */

#define mylib_c

#define MYLIB_YES
#include "mylib.h"

void prog_error (char *file, long line)
{
  fprintf (stderr, "\7*** PROGRAM ERROR *** in file %s line %ld\n",file,line);
  exit (3);
}

void sys_error (char *format_string, ...)
{
  va_list ap;

  fprintf (stderr,"system error: %s -- ",sys_errlist[errno]);
  va_start (ap, format_string);
  vfprintf (stderr,format_string, ap);
  va_end (ap);
  fprintf (stderr, "\n");
  exit(2);
}


MILE *fps[MAX_FILES];
int first_time=1;

void file_init (void)
{ /* initialize the fps table */
  int i;
  first_time=0;
  for (i=0;i<MAX_FILES;i++)
	fps[i]=NULL;
}

void file_shutdown (void)
{
  int i;
  for (i=0;i<MAX_FILES;i++)
	if (fps[i]!=NULL)
	  fileclose (fps[i]);
}

char *strdp (const char *str)
{
  return (strcpy (memalloc(strlen(str)+1),str));
}

void *memalloc (size_t size)
{
  void *p;
  if ((p=malloc(size))==NULL) {
    #ifdef size_t_short
      sys_error ("error allocating %u bytes",size);
    #elif
      sys_error ("error allocating %lu bytes",size);
    #endif
    return (NULL);
  }
  else
    return (p);
}

void *memrealloc (void *block, size_t size)
{
  void *p;
  if ((p=realloc(block,size))==NULL) {
    #ifdef size_t_short
      sys_error ("error reallocating to %u bytes",size);
    #elif
      sys_error ("error reallocating to %lu bytes",size);
    #endif
    return(NULL);
  }
  else
    return (p);
}

void memfree (void *ptr)
{
  if (ptr!=NULL)
	free (ptr);
}

MILE *fileopen(const char *name, const char *mode)
{
  char *s;
  int i;
  if (first_time)
	file_init();
  for (i=0;i<MAX_FILES;i++) { /* find first available handle */
	if (fps[i]==NULL) { /* use it */
	  fps[i]=memalloc(sizeof(MILE));
	  if ((fps[i]->fp=fopen(name,mode))==NULL) {
		memfree (fps[i]);
		fps[i]=NULL;
		return(NULL);
	  }
	  else {
		fps[i]->name=strdup(name);
		return fps[i];
	  }
	}
  };
  /* no more handles */
  sys_error ("no available FILE slots to open %s",name);
  return (NULL);
}

void fileclose(MILE *handle)
{
  int i;
  int found=0;

  if (first_time)
	file_init();
  if (handle==NULL)
	return;
  for (i=0;i<MAX_FILES;i++)
	if (fps[i]==handle) {
	  found=1;
	  break;
	};
  if (!found)
	PROG_ERROR;
  if (fps[i]->fp==NULL)
	PROG_ERROR;
  if (fclose(fps[i]->fp)!=0)
	sys_error ("error closing file %s",fps[i]->name);
  memfree (fps[i]->name);
  fps[i]=NULL;
}

size_t fileread(void *ptr, size_t size, size_t n, MILE* handle)
{
  long init_pos;
  size_t tmp;

  if (first_time)
	file_init();
  if (handle==NULL)
	PROG_ERROR;
  if (handle->fp==NULL)
	PROG_ERROR;
  if ((size==0)||(n==0))
    return(n);
  init_pos=filetell (handle);
  if ((tmp=(fread(ptr, size, n, handle->fp))) != n) {
    #ifdef size_t_short
      sys_error ("error reading %u %u-byte entries at %ld from file %s",n,size,init_pos,handle->name);
    #elif
      sys_error ("error reading %lu %lu-byte entries at %ld from file %s",n,size,init_pos,handle->name);
    #endif
    return(tmp);
  }
  else return (n);
}

size_t filewrite(void *ptr, size_t size, size_t n, MILE* handle)
{
  long init_pos;
  size_t tmp;

  if (first_time)
	file_init();
  if (handle==NULL)
	PROG_ERROR;
  if (handle->fp==NULL)
	PROG_ERROR;
  if ((size==0)||(n==0))
    return(n);
  init_pos=filetell (handle);
  if ((tmp=(fwrite(ptr, size, n, handle->fp))) != n) {
    #ifdef size_t_short
      sys_error ("error writing %u %u-byte entries at %ld to file %s",n,size,init_pos,handle->name);
    #elif
      sys_error ("error writing %lu %lu-byte entries at %ld to file %s",n,size,init_pos,handle->name);
    #endif
    return(tmp);
  }
  else return n;
}

int fileseek (MILE* handle, long offset, int whence)
{
  if (first_time)
	file_init();
  if (handle==NULL)
	PROG_ERROR;
  if (handle->fp==NULL)
	PROG_ERROR;
  if (whence==SEEK_CUR) { /* remember the current location */
	long oldpos,newpos;
	if (offset==0)
	  return (0);
	if ((oldpos=ftell(handle->fp))<0)
	  PROG_ERROR; /* hey, if even ftell doesn't work... */
	if (fseek(handle->fp,offset,whence)!=0)
	  sys_error ("error incrementing current position %ld by %ld in file %s",oldpos,offset,handle->name);
	if ((newpos=ftell(handle->fp))<0)
	  PROG_ERROR;
	if (newpos-oldpos!=offset)
	  sys_error ("fseek failed to set the pointer to %ld in file %s",oldpos+offset,handle->name);
	return (0);
  };
  if (whence==SEEK_SET) {
	if (filetell(handle)==offset)
	  return(0);
	if (fseek(handle->fp,offset,whence)!=0)
	  sys_error ("error fseeking to %ld in file %s",offset,handle->name);
	if (filetell(handle)!=offset)
	  sys_error ("fseek failed to set the pointer to %ld in file %s",offset,handle->name);
	return(0);
  }
  if (whence==SEEK_END) { /* not as much error checking in this case */
	if (fseek(handle->fp,offset,whence)!=0)
	  sys_error ("error fseeking %ld from end of file %s",offset,handle->name);
	return(0);
  };
  PROG_ERROR;
  return(-1);
}

int fileeof (MILE* handle)
{
  if (first_time)
	file_init();
  if (handle==NULL)
	PROG_ERROR;
  if (handle->fp==NULL)
	PROG_ERROR;
  return (feof(handle->fp));
}

long int filetell (MILE* handle)
{
  long int temp;
  if (first_time)
	file_init();
  if (handle==NULL)
	PROG_ERROR;
  if (handle->fp==NULL)
	PROG_ERROR;
  if ((temp=ftell(handle->fp))<0) {
	PROG_ERROR;
	return(-1L);
  }
  else return(temp);
}

char *filegets(char *s, int n, MILE *stream)
{
  if ((n==0)||(s==NULL)||(stream==NULL))
    PROG_ERROR;
  return(fgets(s, n, stream->fp));
}

#undef mylib_c