#include <dir.h>
#include "wads.h"

void Usage(void)
{
  printf ("Usage: wadcomp [-I]outfname.wad { file[@[ENTRY]] | @ENTRY } ... \n");
  printf ("       wadcomp [-I]outfname.wad -rrespfile\n");
  printf ("\n");
  printf ("        outfname.wad is the destination PWAD file (-I makes it an IWAD);\n");
  printf ("        -rrespfile specifies the response file (arguments after it are ignored);\n");
  printf ("        each 'file' parameter is a filename or a wildcard;\n");
  printf ("        ENTRY specifies the entry name for the (single) file preceeding it;\n");
  printf ("        WAD files are copied entry by entry, unless followed by a '@';\n");
  printf ("\n");
  printf ("Examples: wadcomp mywad.wad e1m1.wad my.lmp@DEMO1 @S_START *.spr @S_END\n");
  printf ("          wadcomp mywad.wad e1m?.wad\n");
  printf ("          wadcomp -Inewmain.wad -rentries.lst\n");
  exit (1);
}

/* flags returned by parse */
#define TOOLONG 1
#define WILDCARD 2
#define ENTRYNAME 4
#define AT 8
#define FNAME 16

int parse (char *arg, char *entryname, char* filename)
/* breaks down arg into entryname and filename, setting flags as necessary */
{
  int i, j, code=0;
  for (i=0;(arg[i]!='\0')&&(arg[i]!='@');i++) {
    code|=FNAME;
    filename[i]=arg[i];
    if ((arg[i]=='?')||(arg[i]=='*'))
      code|=WILDCARD;
  }
  filename[i]='\0';
  if (arg[i]=='@') {
    i++;
    code|=AT;
    if (arg[i]!='\0')
      code|=ENTRYNAME;
    for (j=0;(arg[i+j]!='\0')&&(j<8);j++)
      entryname[j]=arg[i+j];
    if (arg[i+j]!='\0')
      code|=TOOLONG;
    for (i=j;i<8;i++)
      entryname[i]='\0';
  }
  return(code);
}

char *extrpath(char *path, char *pathname)
/* extracts everything from pathname up to and including the last '\' */
{
  int i,j;

  for (i=strlen(pathname);((i>=0)&&(pathname[i]!='\\'));i--);
  if (pathname[i]=='\\') {
    int j;
    for (j=0;j<=i;j++)
      path[j]=pathname[j];
    path[j]='\0';
  }
  else
    path[0]='\0';
  return(path);
}

FILE *respfile;
int resp=0;
int argcount=2;
char buffer[80];

char *getnextarg(int argc, char *argv[])
{
  if (resp) {
    if (fgets(buffer,80,respfile)==NULL) {
      fclose(respfile);
      respfile=NULL;
      return(NULL);
    }
    if (buffer[strlen(buffer)-1]=='\n')
      buffer[strlen(buffer)-1]='\0';
    return(buffer);
  }
  else {
    if (argcount>=argc)
      return (NULL);
    else
      return(argv[argcount++]);
  }
}

main (int argc, char *argv[])
{
  int n;
  char str[9];
  WAD *outwad;
  char *outfname;
  char *arg;
  int iwad=0;
  int exit_code=0; /* set to 1 if an input file can't be opened */

  printf ("WAD compiler by Serge Smirnov -- compiled on %s at %s\n\n",__DATE__,__TIME__);

  if (argc<3)
    Usage();
  if (argv[1][0]=='-') {
    if (argv[1][1]=='I') {
      iwad=1;
      outfname=argv[1]+2;
    }
    else
      UsageError("unrecognized option in parameter %s",argv[1]);
  }
  else
    outfname=argv[1];

  if (argv[2][0]=='-') {
    if ((argv[2][1]=='r')||(argv[2][1]=='R')) {
      resp=1;
      if ((respfile=fopen(argv[2]+2,"rt"))==NULL)
	UsageError("fatal: can't open response file %s",argv[2]+2);
    }
    else
      UsageError ("option not understood: %s",argv[2]);
  }

  if ((outwad=new_wad(outfname))==NULL) {
    fprintf(stderr,"fatal error: cannot open output wad file %s.\n",argv[1]);
    exit(1);
  }

  while ((arg=getnextarg(argc,argv))!=NULL) { /* process each group of files */
    DirEntry entry;
    char entryname[8];
    char pathname[MAXPATH];
    int result;
    int done;
    struct ffblk ffblk;

    result=parse(arg,entryname,pathname);
    if ((result&WILDCARD)&&(result&ENTRYNAME)) {
      printf ("WARNING: entry name for multiple files ingnored in %s;\n",argv[n]);
      result&=(~ENTRYNAME);
      exit_code=1;
    }
    if ((result&ENTRYNAME)&&(result&TOOLONG)) {
      printf ("WARNING: entry name truncated to 8 characters in %s;\n",argv[n]);
      exit_code=1;
    }

    if (!(result&FNAME)) { /* no filename, just an empty entry */
      DirEntry entry;
      memcpy(entry.name,entryname,8);
      printf("adding a null entry %s;\n",name2str(entryname,str,8));
      copy_entry_to_wad(outwad,NULL,entry);
    }
    else if ((done=findfirst(pathname,&ffblk,0))!=0) {
      printf ("WARNING: no files matching %s found\n",pathname);
      exit_code=1;
    }
    else {
      char path[MAXPATH];
      char fname[MAXPATH];

      extrpath(path,pathname);
      while (!done) {
	FILE *infile;
	strcpy(fname,path);
	strcat(fname,ffblk.ff_name);
	if ((infile=fopen(fname,"rb"))==NULL) {
	  printf ("WARNING: couldn't open %s;\n",fname);
	  exit_code=1;
	}
	else { /* process the file */
	  if ((!(result&AT))&&(wadfile(infile))) { /* treat as a wad */
	    Header hdr;
	    long i;
	    fread (&hdr,sizeof(hdr),1,infile);
	    printf ("adding entries from WAD file %s:\n",fname);
	    fseek(infile,hdr.dir_start,SEEK_SET);
	    for (i=0;i<hdr.num_entries;i++) {
	      fread (&entry, sizeof(DirEntry), 1, infile);
	      copy_entry_to_wad (outwad,infile,entry);
	      printf ("\t%s;\n",name2str(entry.name,str,8));
	    }
	    printf ("added %ld entries from %s;\n",hdr.num_entries,fname);
	  } /* treat as a wad */
	  else {
	    if (!(result&ENTRYNAME))
	      str2name(ffblk.ff_name,entryname,8);
	    printf ("adding file %s as entry %s;\n",fname,name2str(entryname,str,8));
	    copy_file_to_wad(outwad,infile,entryname);
	  }
	  fclose(infile);
	} /* process the file */
	done=findnext(&ffblk);
      } /* while not done with this group of files */
    } /* found matchingf files */
  } /* process each group of files */

  close_wad(outwad, iwad);
  printf("\ndone.\n");
  return (exit_code);
}
