/*//////////////////////////////////////////////////////////
Winwad
Copyright (C) 2000 John Gaughan

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

The full GNU General Public License can be found archived with these
source files as "gpl.txt"
*///////////////////////////////////////////////////////////

// Windows headers
#ifdef WIN32
#include <afxwin.h>
#endif

// Standard headers
#include <fstream>
#include <new>
#include <cstring>
#include <vector>
using namespace std;

// Winwad headers
#include "Global.h"
#include "CWad.h"



// Default constructor
CWad::CWad (void)
{
  // Initialize wad data
  memcpy (m_cWadType, "PWAD", 4);
  m_nDirectoryOffset   = 0;
  m_vEntries.clear ();
  memset (m_szFilename, 0, 1024);
  m_bNameChanged = false;

  // Done
  return;
}



// Copy constructor
CWad::CWad (const CWad &wOldWad)
{
  // Local variables
  register int   i;
  char           szNewType[5];
  char           szNewName[9];
  CWadEntry     *weTemp;
  INT32          nNumEntries;

  // First we need the source's type
  wOldWad.GetWadType (szNewType);
  memcpy (m_cWadType, szNewType, 4);

  // Next, get the number of entries
  nNumEntries = wOldWad.GetNumEntries ();

  // If there are no entries, we are done
  if (!nNumEntries) return;

  // If not, create enough entries to hold all
  // the new data
  for (i=0; i<nNumEntries; i++)
  {
    // Create a new entry
    weTemp = new CWadEntry;

    // Get the entry's name
    wOldWad.GetEntryName (szNewName, i);
    memcpy (weTemp->cName, szNewName, 8);

    // Get the entry's size
    weTemp->nSize = wOldWad.GetEntrySize (i);

    // Get the entry's data
    if (weTemp->nSize>0)
    {
      weTemp->cData = new char[weTemp->nSize];
    }
    else
    {
      weTemp->cData = (char *) NULL;
    }

    // Set file variables
    weTemp->bLoaded = true;
    weTemp->bZeroLength = (weTemp->nSize>0?false:true);
    weTemp->nOffset = -1;

    // Push it on the vector
    m_vEntries.push_back (weTemp);
  }

  // Set filename to blank
  memset (m_szFilename, 0, 1024);

  // All done!
  return;
}



// Destructor
CWad::~CWad (void)
{
  // Local variables
  register int i;

  // Delete all data associated with entries
  for (i=0; i<m_vEntries.size (); i++)
  {

    // If entry (is not zero length) and (is
    // loaded in memory)
    if (!m_vEntries[i]->bZeroLength &&
        m_vEntries[i]->bLoaded)
    {
      delete [] m_vEntries[i]->cData;
    }
  }

  // Delete each entry. Go in reverse order
  // through the vector to maximize speed
  for (i=m_vEntries.size () - 1; i>0; i--)
  {
    delete m_vEntries[i];
  }

  // Now clear the vector
  m_vEntries.clear ();

  // Done
  return;
}



// Addition operator
CWad CWad::operator+= (const CWad &wWad)
{
  // Local variables
  register int   i;
  char           szNewName[9];
  CWadEntry     *weTemp;
  INT16          nNewEntries;

  // Next, get the number of entries
  nNewEntries = wWad.GetNumEntries ();

  // If there are no new entries, we are done
  if (!nNewEntries) return *this;

  // If not, create enough entries to hold all
  // the new data
  for (i=0; i<nNewEntries; i++)
  {
    // Create a new entry
    weTemp = new CWadEntry;

    // Get the entry's name
    wWad.GetEntryName (szNewName, i);
    memcpy (weTemp->cName, szNewName, 8);

    // Get the entry's size
    weTemp->nSize = wWad.GetEntrySize (i);

    // Get the entry's data
    if (weTemp->nSize>0)
    {
      weTemp->cData = new char[weTemp->nSize];
    }
    else
    {
      weTemp->cData = (char *) NULL;
    }

    // Set file variables
    weTemp->bLoaded = true;
    weTemp->bZeroLength = (weTemp->nSize>0?false:true);
    weTemp->nOffset = -1;

    // Push it on the vector
    m_vEntries.push_back (weTemp);
  }

  // All done!
  return *this;
}



// Assignment operator
CWad CWad::operator= (CWad wWad)
{
  // Local variables
  register int   i;
  char           szNewType[5];
  char           szNewName[9];
  CWadEntry     *weTemp;
  INT32          nNumEntries;

  // Delete all data associated with entries
  for (i=0; i<m_vEntries.size (); i++)
  {

    // If entry (is not zero length) and (is
    // loaded in memory)
    if (!m_vEntries[i]->bZeroLength &&
        m_vEntries[i]->bLoaded)
    {
      delete [] m_vEntries[i]->cData;
    }
  }

  // Delete each entry. Go in reverse order
  // through the vector to maximize speed
  for (i=m_vEntries.size () - 1; i>0; i--)
  {
    delete m_vEntries[i];
  }

  // Now clear the vector
  m_vEntries.clear ();

  // Done clearing wad, now we need the
  // source's type
  wWad.GetWadType (szNewType);
  memcpy (m_cWadType, szNewType, 4);

  // Next, get the number of entries
  nNumEntries = wWad.GetNumEntries ();

  // If there are no entries, we are done
  if (!nNumEntries) return *this;

  // If not, create enough entries to hold all
  // the new data
  for (i=0; i<nNumEntries; i++)
  {
    // Create a new entry
    weTemp = new CWadEntry;

    // Get the entry's name
    wWad.GetEntryName (szNewName, i);
    memcpy (weTemp->cName, szNewName, 8);

    // Get the entry's size
    weTemp->nSize = wWad.GetEntrySize (i);

    // Get the entry's data
    if (weTemp->nSize>0)
    {
      weTemp->cData = new char[weTemp->nSize];
    }
    else
    {
      weTemp->cData = (char *) NULL;
    }

    // Set file variables
    weTemp->bLoaded = true;
    weTemp->bZeroLength = (weTemp->nSize>0?false:true);
    weTemp->nOffset = -1;

    // Push it on the vector
    m_vEntries.push_back (weTemp);
  }

  // All done!
  return *this;
}  



// Equality comparison
bool CWad::operator== (const CWad &r) const
{
  // Local variables
  char        szTemp[9];
  szTemp[8]   = ' ';
  INT32       nTemp;
  char       *cTemp = (char *) NULL;

  // First, check the type of wad
  r.GetWadType (szTemp);
  if (!memcmp (this->m_cWadType, szTemp, 4))
    return false;

  // Now check the number of entries
  if (m_vEntries.size () != r.GetNumEntries ())
    return false;

  // Wads are same type and have same number of entries.
  // Go through each entry, one at a time, and check to
  // see if they have the same names, sizes, and contents
  for (register int i=0; i<r.GetNumEntries (); i++)
  {
    // Check size of entry
    nTemp = r.GetEntrySize (i);
    if (nTemp != m_vEntries[i]->nSize) return false;

    // Check name of entry
    r.GetEntryName (szTemp, i);
    if (!memcmp (szTemp, m_vEntries[i]->cName, 8)) return false;

    // Now compare contents. Be careful of dynamic memory!
    if (nTemp && m_vEntries[i]->nSize)
    {
      cTemp = new char[nTemp];
      r.GetEntryData (cTemp, i);
      if (!memcmp (cTemp, m_vEntries[i]->cData, nTemp))
      {
        delete [] cTemp;
        return false;
      }
      delete [] cTemp;
    }
  }

  // Equal!
  return true;
}



// Inequality comparison
bool CWad::operator!= (const CWad &r) const
{
  // Simply return the inverse of "==" to save space
  return !(*this == r);
}



// Is the wad empty?
bool CWad::IsEmpty (void)
{
  return (m_vEntries.size ()>0?false:true);
}



// Add a new entry
bool CWad::AddEntry (void)
{
  // Append a new entry
  m_vEntries.push_back (new CWadEntry);

  // Set data in new entry
  m_vEntries[m_vEntries.size ()-1]->bLoaded = true;
  m_vEntries[m_vEntries.size ()-1]->bZeroLength = true;
  m_vEntries[m_vEntries.size ()-1]->cData = (char *) NULL;
  memcpy (m_vEntries[m_vEntries.size ()-1]->cName, "\0\0\0\0\0\0\0\0", 8);
  m_vEntries[m_vEntries.size ()-1]->nOffset = 0;
  m_vEntries[m_vEntries.size ()-1]->nSize = 0;

  // Done
  return true;
}



// Inserts an entry
bool CWad::InsertEntry (INT32 nIndex)
{
  // Check if index is valid
  if ((nIndex<0) || (nIndex>=m_vEntries.size ()))
    return false;

  // Create a new entry and insert it
  m_vEntries.insert (m_vEntries.begin () + nIndex, new CWadEntry);

  // Set data in new entry
  m_vEntries[m_vEntries.size ()-1]->bLoaded = true;
  m_vEntries[m_vEntries.size ()-1]->bZeroLength = true;
  m_vEntries[m_vEntries.size ()-1]->cData = (char *) NULL;
  memcpy (m_vEntries[m_vEntries.size ()-1]->cName, "\0\0\0\0\0\0\0\0", 8);
  m_vEntries[m_vEntries.size ()-1]->nOffset = 0;
  m_vEntries[m_vEntries.size ()-1]->nSize = 0;

  // Done
  return true;
}



// Delete an entry
bool CWad::DeleteEntry (INT32 nIndex)
{
  // Check if index is valid
  if ((nIndex<0) || (nIndex>=m_vEntries.size ()))
    return false;

  // If entry is not empty, free its memory
  if (!m_vEntries[nIndex]->bZeroLength)
    delete [] m_vEntries[nIndex]->cData;

  // Now delete entry...
  delete m_vEntries[nIndex];

  // ...and remove from vector
  m_vEntries.erase (m_vEntries.begin () + nIndex);

  // Done
  return true;
}



// Clear all entries
bool CWad::Clear (void)
{
  // Go through and DeleteEntry on each element
  while (!m_vEntries.empty ())
    this->DeleteEntry (m_vEntries.size ()-1);

  // Done
  return true;
}



// Get an entry's name
bool CWad::GetEntryName (char *szName, INT32 nIndex) const
{
  // Check pointer
  if (szName == (char *) NULL) return false;

  // Check if index is valid
  if ((nIndex<0) || (nIndex>=m_vEntries.size ()))
    return false;

  // Copy into buffer and hope variable has
  // enough memory! Add null just in case
  memcpy (szName, m_vEntries[nIndex]->cName, 8);
  szName[8] = 0;

  // Done
  return true;
}



// Get an entry's size
INT16 CWad::GetEntrySize (INT32 nIndex) const
{
  // Check if index is valid
  if ((nIndex<0) || (nIndex>=m_vEntries.size ()))
    return false;

  // Return the size
  return m_vEntries[nIndex]->nSize;
}



// Get an entry's data
bool CWad::GetEntryData (char *cData, INT32 nIndex) const
{
  // Check pointer
  if (cData == (char *) NULL) return false;

  // Check if index is valid
  if ((nIndex<0) || (nIndex>=m_vEntries.size ()))
    return false;

  // If not loaded, cache it in memory
  if (!m_vEntries[nIndex]->bLoaded) this->CacheEntry (nIndex);

  // If zero length
  if (m_vEntries[nIndex]->bZeroLength)
    return true;

  // Otherwise, stuff data into variable
  else
    memcpy (cData, m_vEntries[nIndex]->cData,
        m_vEntries[nIndex]->nSize);

  // Done
  return true;
}



// Get the type of wad
bool CWad::GetWadType (char *szWadType) const
{
  // Check pointer
  if (szWadType == (char *) NULL) return false;

  // Copy data, append null
  memcpy (szWadType, m_cWadType, 4);
  szWadType[4] = 0;

  // Done
  return true;
}



// Get number of entries in wad
INT32 CWad::GetNumEntries (void) const
{
  return m_vEntries.size ();
}



// Set an entry's name
bool CWad::SetEntryName (char *szName, INT32 nIndex)
{
  // Check pointer
  if (szName == (char *) NULL) return false;

  // Check if index is valid
  if ((nIndex<0) || (nIndex>=m_vEntries.size ()))
    return false;

  // Copy name
  memset (m_vEntries[nIndex]->cName, 0, 8);
  strncpy (m_vEntries[nIndex]->cName, szName, 8);

  // Done
  return true;
}



// Set an entry's data
bool CWad::SetEntryData (char *cData, INT16 nEntrySize, INT32 nIndex)
{
  // Check pointer
  if (cData == (char *) NULL) return false;

  // Check if index is valid
  if ((nIndex<0) || (nIndex>=m_vEntries.size ()))
    return false;

  // Delete existing memory if necessary
  if (!m_vEntries[nIndex]->bZeroLength)
    delete [] m_vEntries[nIndex]->cData;

  // Set new size
  m_vEntries[nIndex]->nSize = nEntrySize;
  m_vEntries[nIndex]->bZeroLength = (nEntrySize>0?false:true);

  // Allocate memory if necessary
  if (nEntrySize)
    m_vEntries[nIndex]->cData = new char[nEntrySize];

  // Copy memory
  if (nEntrySize)
    memcpy (m_vEntries[nIndex]->cData, cData, nEntrySize);

  // Done
  return true;
}



// Set the wad type
bool CWad::SetWadType (char *szType) const
{
  // Check pointer
  if (szType == (char *) NULL) return false;

  // Copy data. Explicit cast necessary because
  // this is a const function.
  memcpy ((void *) m_cWadType, szType, 4);

  // Done
  return true;
}



// Read a wad from disk
bool CWad::LoadWad (bool bPreCache)
{
  // Local variables
  INT32             nNumEntries;
  register int      i;
  CWadEntry        *weTemp = (CWadEntry *) NULL;

  // Check to make sure there is a filename
  if (m_szFilename[0] == 0)return false;

  // Open file
  ifstream in;
  in.open (m_szFilename, ios::in | ios::binary);

  // If file did not open, fail
  if (!in) return false;

  // File opened. Clear current wad
  this->Clear ();

  // Now read in header
  in.read (m_cWadType, 4);
  in.read ((char *) &nNumEntries, 4);
  in.read ((char *) &m_nDirectoryOffset, 4);

  // Seek to directory
  in.seekg (m_nDirectoryOffset, ios::beg);

  // Now create enough vector entries for wad
  for (i=0; i<nNumEntries; i++)
  {
    // Create new entry
    weTemp = new CWadEntry;

    // Read information into entry from directory
    in.read ((char *) &weTemp->nOffset, 4);
    in.read ((char *) &weTemp->nSize, 4);
    in.read (weTemp->cName, 8);

    // Set other data/flags
    weTemp->bZeroLength = (weTemp->nSize>0?false:true);
    weTemp->bLoaded = weTemp->bZeroLength;
    weTemp->cData = (char *) NULL;

    // Push it onto the back of the vector
    m_vEntries.push_back (weTemp);
  }

  // Load entries IF precache is set
  if (bPreCache)
  {
    // Loop through each entry
    for (i=0; i<m_vEntries.size (); i++)
    {
      // If entry is not zero length
      if (!(m_vEntries[i]->bZeroLength))
      {
        // Seek to entry
        in.seekg (m_vEntries[i]->nOffset, ios::beg);

        // Allocate memory
        try
        {
          m_vEntries[i]->cData = new char[m_vEntries[i]->nSize];
        }
        catch (bad_alloc e)
        {
#ifdef WIN32
          ::AfxMessageBox ("Memory Allocation Failure in CWad::LoadWad\nCannot allocate memory from heap for wad entry.", MB_ICONEXCLAMATION, NULL);
#endif
          return false;
        }

        // Read it in
        in.read (m_vEntries[i]->cData, m_vEntries[i]->nSize);

        // Set loaded flag
        m_vEntries[i]->bLoaded = true;
      }
    }
  }

  // Done
  in.close ();
  m_bNameChanged = false;
  return true;
}



// Save a wad to disk - aligns all data on 4 byte boundaries
bool CWad::SaveWad (void)
{
  // Local variables
  ofstream         out;
  register int     i;
  INT8             nTemp;
  INT32            nNumEntries;
  char             cNulls[16];
  memset (cNulls, 0, 16);

  // Check file
  if (m_szFilename[0] == 0) return false;

  // Make sure entire wad is loaded
  for (i=0; i<m_vEntries.size (); i++)
    if (!m_vEntries[i]->bLoaded) return false;

  // Wad is loaded, try to open file
  out.open (m_szFilename, ios::out | ios::trunc | ios::binary);

  // Get number of entries, ensure it is stored in a 4 byte
  // integer to avoid problems when writing to file stream
  nNumEntries = m_vEntries.size ();

  // Write directory
  out.write (m_cWadType, 4);
  out.write ((char *) &nNumEntries, 4);
  out.write ("----", 4);  // will come back for this one!
  out.write (cNulls, 4);  // Give some space before first entry

  // Write each entry
  for (i=0; i<m_vEntries.size (); i++)
  {
    // Get offset
    m_vEntries[i]->nOffset = out.tellp ();

    // If not zero length
    if (!(m_vEntries[i]->bZeroLength))
    {
      // Write data
      out.write (m_vEntries[i]->cData, m_vEntries[i]->nSize);

      // Align on 4 byte boundary
      nTemp = out.tellp () % 4;
      if (nTemp) out.write (cNulls, nTemp);
    }
  }

  // Get offset of directory
  m_nDirectoryOffset = out.tellp ();

  // Write directory offset
  out.seekp (8, ios::beg);
  out.write ((char *) &m_nDirectoryOffset, 4);

  // Seek back to directory
  out.seekp (m_nDirectoryOffset, ios::beg);

  // Loop through and write each directory entry
  for (i=0; i<m_vEntries.size (); i++)
  {
    out.write ((char *) &m_vEntries[i]->nOffset, 4);
    out.write ((char *) &m_vEntries[i]->nSize, 4);
    out.write (m_vEntries[i]->cName, 8);
  }

  // Done
  out.close ();
  m_bNameChanged = false;
  return true;
}



// Get the filename to save to/load from
bool CWad::GetFilename (char *szFilename)
{
  // Check pointer
  if (szFilename == (char *) NULL) return false;

  // Copy
  strncpy (szFilename, this->m_szFilename, 1024);

  // Done
  return true;
}



// Set the filename to use
bool CWad::SetFilename (char *szFilename)
{
  // Check pointer
  if (szFilename == (char *) NULL) return false;

  // Otherwise, copy into object's buffer
  strncpy (this->m_szFilename, szFilename, 1024);

  // Set filename flag
  m_bNameChanged = true;

  // Done
  return true;
}



// Precache an entry from disk
bool CWad::CacheEntry (INT32 nIndex) const
{
  // If index is bad, no file, or file has changed, fail
  if ((nIndex<0) || (nIndex>=m_vEntries.size ()) ||
      (m_szFilename[0] == 0) || m_bNameChanged) return false;

  // If entry is already loaded or is zero length, return
  if (m_vEntries[nIndex]->bLoaded || m_vEntries[nIndex]->bZeroLength)
    return true;

  // Otherwise, attempt to open file
  ifstream in;
  in.open (m_szFilename, ios::in | ios::binary);
  if (!in) return false;

  // Seek to proper location
  in.seekg (m_vEntries[nIndex]->nOffset, ios::beg);
  if ((!in) || (in.eof ()))
  {
    in.close ();
    return false;
  }

  // Allocate memory
  m_vEntries[nIndex]->cData = new char[m_vEntries[nIndex]->nSize];

  // Read data
  in.read (m_vEntries[nIndex]->cData,
      m_vEntries[nIndex]->nSize);

  // Set loaded flag
  m_vEntries[nIndex]->bLoaded = true;

  // Done
  in.close ();
  return true;
}



// Load all entries not already loaded
bool CWad::CacheAllEntries (void)
{
  // Local variables
  register int        i;
  ifstream            in;

  // If no file, or file has changed, fail
  if ((m_szFilename[0] == 0) || m_bNameChanged) return false;

  // Open file
  in.open (m_szFilename, ios::in);
  if (!in) return false;

  // Loop through each entry
  for (i=0; i<m_vEntries.size (); i++)
  {
    // If entry is zero length, just set the loaded flag
    if (m_vEntries[i]->bZeroLength) m_vEntries[i]->bLoaded = true;

    // Otherwise, load if not loaded
    else if (!m_vEntries[i]->bLoaded)
    {
      // Allocate memory - ok, since zero length entries
      // will never get to this branch
      try
      {
        m_vEntries[i]->cData = new char[m_vEntries[i]->nSize];
      }
      catch (bad_alloc e)
      {
#ifdef WIN32
        ::AfxMessageBox ("Memory Allocation Error in CWad::CacheAllEntries\nCannot allocate memory from heap for wad entry.", MB_ICONEXCLAMATION, NULL);
#endif
        return false;
      }

      // Read in data
      in.read (m_vEntries[i]->cData, m_vEntries[i]->nSize);
    }
  }

  // Done
  in.close ();
  return true;
}



// Import an entry from another file (raw)
bool CWad::ImportEntry (char *szFilename, INT32 nIndex)
{
  // Local variables
  ifstream in;

  // Check parameters
  if ((szFilename==(char *) NULL) || (szFilename[0]==0)) return false;
  if ((nIndex<0) || (nIndex>=m_vEntries.size ())) return false;

  // Deallocate entry
  if (!m_vEntries[nIndex]->bZeroLength)
    delete [] m_vEntries[nIndex]->cData;

  // Open file
  in.open (szFilename, ios::in | ios::binary);
  if (!in) return false;

  // Find size of file
  in.seekg (0, ios::end);
  m_vEntries[nIndex]->nSize = in.tellg ();
  in.seekg (0, ios::beg);

  // If size is zero
  if (!m_vEntries[nIndex]->nSize)
  {
    m_vEntries[nIndex]->bLoaded = true;
    m_vEntries[nIndex]->bZeroLength = true;
    m_vEntries[nIndex]->cData = (char *) NULL;
    m_vEntries[nIndex]->nOffset = 0;
    return true;
  }

  // Allocate memory
  m_vEntries[nIndex]->cData = new char[m_vEntries[nIndex]->nSize];

  // Read it in
  in.read (m_vEntries[nIndex]->cData, m_vEntries[nIndex]->nSize);

  // Set other entry information
  m_vEntries[nIndex]->bLoaded = true;
  m_vEntries[nIndex]->bZeroLength = false;
  m_vEntries[nIndex]->nOffset = 0;

  // Done
  in.close ();
  return true;
}



// Export an entry to another file (raw)
bool CWad::ExportEntry (char *szFilename, INT32 nIndex)
{
  // Local variables
  ofstream out;

  // Check parameters
  if ((szFilename==(char *) NULL) || (szFilename[0]==0)) return false;
  if ((nIndex<0) || (nIndex>=m_vEntries.size ())) return false;

  // Try to open file
  out.open (szFilename, ios::out | ios::binary | ios::trunc);
  if (!out) return false;

  // Write to file
  out.write (m_vEntries[nIndex]->cData, m_vEntries[nIndex]->nSize);

  // Done
  out.close ();
  return true;
}
