// Does two zbuffered cubes using display lists.
#include <windows.h>	
#include <string.h>
#include <gl\gl.h>		
#include <gl\glu.h>		
#include <gl\glaux.h>	

static HGLRC hRC;   
static HDC hDC;  

GLuint TextureID;
GLuint Cube; // ID for our cube display list
unsigned char *Texture;
int wWidth = 640, wHeight = 480;


void SetupTextures(), SetOpenGLOptions(), ResizeScene(GLsizei Width, GLsizei Height);
void DrawScene(), InitOpenGL(), BuildLists();
LRESULT CALLBACK WndProc(HWND hWnd, UINT Message, WPARAM wParam, LPARAM lParam);

int WINAPI WinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance, LPSTR lpCmdLine, int nCmdShow)
{
	if(MessageBox(NULL, "This program uses OpenGL Default rendering,\nit may be too slow to watch if you do not have a 3d graphics card", "OpenGL is used", MB_ICONINFORMATION | MB_YESNO) != IDYES)
		return 0;
	MSG			msg;		
	WNDCLASS	wc;			
	HWND		hWnd;		
	char ClassName[] = "Window class thingy for OpenGL App.";
	char AppName[] = "OpenGL 3D Hardware Rendering Window";		
	
	wc.style			= CS_HREDRAW | CS_VREDRAW | CS_OWNDC;
	wc.lpfnWndProc		= (WNDPROC) WndProc;
	wc.cbClsExtra		= 0;
	wc.cbWndExtra		= 0;
	wc.hInstance		= hInstance;
	wc.hIcon			= NULL;
	wc.hCursor			= LoadCursor(NULL, IDC_ARROW);
	wc.hbrBackground	= NULL;
	wc.lpszMenuName		= NULL;
	wc.lpszClassName	= ClassName;

	if(!RegisterClass(&wc))
	{
		MessageBox(0,"The window class could not be registered!","Window Creation Error", MB_ICONSTOP);
		return 0;
	}

	hWnd = CreateWindow(ClassName, AppName,	WS_OVERLAPPEDWINDOW |  WS_CLIPCHILDREN | WS_CLIPSIBLINGS, 0, 0, 
                		wWidth, wHeight, NULL, NULL, hInstance, NULL);

	if(!hWnd)
	{
		MessageBox(0,"Couldn't create window!", "Window Creation Error", MB_ICONSTOP);
		return 0;
	}

/*	DEVMODE dmScreenSettings;											

	memset(&dmScreenSettings, 0, sizeof(DEVMODE));
	dmScreenSettings.dmSize			= sizeof(DEVMODE);
	dmScreenSettings.dmPelsWidth	= wWidth;								
	dmScreenSettings.dmPelsHeight	= wHeight;
	dmScreenSettings.dmFields		= DM_PELSWIDTH | DM_PELSHEIGHT;
	ChangeDisplaySettings(&dmScreenSettings, CDS_FULLSCREEN);*/


	ShowWindow(hWnd, SW_SHOW);
	UpdateWindow(hWnd);
	wglMakeCurrent(hDC, hRC);

	while (msg.message != WM_QUIT)
	{
		if(PeekMessage(&msg, NULL, 0, 0, PM_NOREMOVE))
		{
			if(GetMessage(&msg, NULL, 0, 0))
			{
				TranslateMessage(&msg);
				DispatchMessage(&msg);
			}
		}
		DrawScene();
		SwapBuffers(hDC);
	}
	delete [] Texture;
	return 0;
}

void BuildLists()
{	
	Cube = glGenLists(1); // 1 object
	glNewList(Cube, GL_COMPILE);
	glBegin(GL_QUADS);
		// Front Face
		glNormal3f(0.0f, 0.0f, 1.0f);
		glTexCoord2f(0.0f, 0.0f); glVertex3f(-1.0f, -1.0f,  1.0f);
		glTexCoord2f(1.0f, 0.0f); glVertex3f( 1.0f, -1.0f,  1.0f);
		glTexCoord2f(1.0f, 1.0f); glVertex3f( 1.0f,  1.0f,  1.0f);
		glTexCoord2f(0.0f, 1.0f); glVertex3f(-1.0f,  1.0f,  1.0f);
		// Back Face
		glNormal3f(0.0f, 0.0f, -1.0f);
		glTexCoord2f(1.0f, 0.0f); glVertex3f(-1.0f, -1.0f, -1.0f);
		glTexCoord2f(1.0f, 1.0f); glVertex3f(-1.0f,  1.0f, -1.0f);
		glTexCoord2f(0.0f, 1.0f); glVertex3f( 1.0f,  1.0f, -1.0f);
		glTexCoord2f(0.0f, 0.0f); glVertex3f( 1.0f, -1.0f, -1.0f);
		// Top Face
		glNormal3f(0.0f, 1.0f, 0.0f);
		glTexCoord2f(0.0f, 1.0f); glVertex3f(-1.0f,  1.0f, -1.0f);
		glTexCoord2f(0.0f, 0.0f); glVertex3f(-1.0f,  1.0f,  1.0f);
		glTexCoord2f(1.0f, 0.0f); glVertex3f( 1.0f,  1.0f,  1.0f);
		glTexCoord2f(1.0f, 1.0f); glVertex3f( 1.0f,  1.0f, -1.0f);
		// Bottom Face
		glNormal3f(0.0f, -1.0f, 0.0f);
		glTexCoord2f(1.0f, 1.0f); glVertex3f(-1.0f, -1.0f, -1.0f);
		glTexCoord2f(0.0f, 1.0f); glVertex3f(1.0f, -1.0f, -1.0f);
		glTexCoord2f(0.0f, 0.0f); glVertex3f(1.0f, -1.0f,  1.0f);
		glTexCoord2f(1.0f, 0.0f); glVertex3f(-1.0f, -1.0f,  1.0f);
		// Right face
		glNormal3f(1.0f, 0.0f, 0.0f);		
		glTexCoord2f(1.0f, 0.0f); glVertex3f(1.0f, -1.0f, -1.0f);
		glTexCoord2f(1.0f, 1.0f); glVertex3f(1.0f,  1.0f, -1.0f);
		glTexCoord2f(0.0f, 1.0f); glVertex3f(1.0f,  1.0f,  1.0f);
		glTexCoord2f(0.0f, 0.0f); glVertex3f(1.0f, -1.0f,  1.0f);
		// Left Face
		glNormal3f(-1.0f, 0.0f, 0.0f);		
		glTexCoord2f(0.0f, 0.0f); glVertex3f(-1.0f, -1.0f, -1.0f);
		glTexCoord2f(1.0f, 0.0f); glVertex3f(-1.0f, -1.0f, 1.0f);
		glTexCoord2f(1.0f, 1.0f); glVertex3f(-1.0f, 1.0f, 1.0f);
		glTexCoord2f(0.0f, 1.0f); glVertex3f(-1.0f, 1.0f, -1.0f);
	glEnd();
	glEndList();
	return;
}


void SetupTextures()
{
	AUX_RGBImageRec *Texture;
   Texture = auxDIBImageLoad("../data/warp32.bmp");
	if(!Texture) exit(1);

	glGenTextures(1, &TextureID);
	glBindTexture(GL_TEXTURE_2D, TextureID);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
	glTexImage2D(GL_TEXTURE_2D, 0, 3, Texture->sizeX, Texture->sizeY, 0, GL_RGB, GL_UNSIGNED_BYTE, Texture->data);
	return;
}

void InitOpenGL()
{
	SetupTextures();					
	SetOpenGLOptions();
	BuildLists();
	glMatrixMode(GL_PROJECTION);
	glLoadIdentity();							
	gluPerspective(45.0f, (GLfloat) wWidth / (GLfloat) wHeight, 0.1f, 100.0f);	
	glMatrixMode(GL_MODELVIEW);
	glLoadIdentity();
	return;
}

void SetOpenGLOptions()	
{	
	GLfloat AmbientLight[] = { 0.6f, 0.6f, 0.6f, 1.0f };
	GLfloat DiffuseLight[] = { 0.8f, 0.8f, 0.8f, 1.0f };
	GLfloat LightPosition[] = { 0.0f, 0.0f, 1.0f, 1.0f };
	glLightfv(GL_LIGHT1, GL_AMBIENT, AmbientLight);
	glLightfv(GL_LIGHT1, GL_DIFFUSE, DiffuseLight);
	glLightfv(GL_LIGHT1, GL_POSITION, LightPosition);
	glEnable(GL_LIGHT1);
	glEnable(GL_LIGHTING);
	glEnable(GL_TEXTURE_2D);			
	glEnable(GL_CULL_FACE);
	glClearColor(0.0f, 0.0f, 0.0f, 0.0f);		
	glClearDepth(1.0);							
	glDepthFunc(GL_LESS);						
	glEnable(GL_DEPTH_TEST);					
	glShadeModel(GL_SMOOTH);					
	return;
}

void ResizeScene(GLsizei Width, GLsizei Height)
{
	if (!Height) return;
	glViewport(0, 0, Width, Height);
	glMatrixMode(GL_PROJECTION);
	glLoadIdentity();
	gluPerspective(45.0f, (GLfloat) Width / (GLfloat) Height, 0.1f, 100.0f);
	glMatrixMode(GL_MODELVIEW);
}

void DrawScene()
{
	static float XRot = 0, YRot = 0, ZRot = 0, CubesPos = -5.0;
	static float Dir = 0.03f;
	XRot += 1.5;
	YRot += 2;
	ZRot += 0.5;
	CubesPos += Dir;
	if(CubesPos >= -1.8f || CubesPos <= -8.0f) Dir *= -1; 
	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);		

	glPushMatrix();
	glTranslatef(0.0f, 0.0f, CubesPos);
	glPushMatrix();
	glRotatef(XRot, 1.0f, 0.0f, 0.0f);
	glRotatef(YRot, 0.0f, 1.0f, 0.0f);
	glRotatef(ZRot, 0.0f, 0.0f, 1.0f);
	glCallList(Cube);
	glPopMatrix();
	glRotatef(YRot, 0.0f, 1.0f, 0.0f);
	glRotatef(XRot, 1.0f, 0.0f, 0.0f);
	glRotatef(ZRot, 0.0f, 0.0f, 1.0f);
	glCallList(Cube);
	glPopMatrix();
}

void SetupScreenDetails(HWND hWnd)
{
	PIXELFORMATDESCRIPTOR PFD;
	memset(&PFD, 0, sizeof(PFD));
	PFD.nSize = sizeof(PFD);
	PFD.nVersion = 1;
	PFD.dwFlags = PFD_DRAW_TO_WINDOW | PFD_SUPPORT_OPENGL | PFD_DOUBLEBUFFER;
	PFD.iPixelType = PFD_TYPE_RGBA;
	PFD.cColorBits = 16;
	PFD.cDepthBits = 16;
	PFD.dwLayerMask = PFD_MAIN_PLANE;   
	
	GLuint PixelFormat;
	hDC = GetDC(hWnd); 
	PixelFormat = ChoosePixelFormat(hDC, &PFD);
	if(!PixelFormat)
	{   
		MessageBox(NULL, "Could not find suitable pixel format",
                          "Pixel Format Error", MB_OK);
        PostQuitMessage(0);
	}
    if(!SetPixelFormat(hDC, PixelFormat, &PFD)) 
    {
        MessageBox(NULL, "Can't set the pixel format",
                        "Pixel Format Error", MB_OK);
        PostQuitMessage(0);
    }
    hRC = wglCreateContext(hDC); 
    if(!hRC)
    {
        MessageBox(NULL, "Can't create render context.", "Render Context Error", MB_OK);
        PostQuitMessage(0);
    }
    if(!wglMakeCurrent(hDC, hRC))
    {
        MessageBox(NULL, "Can't set active render context.", "Render Context Error", MB_OK);
        PostQuitMessage(0);
    }
	return;
}


LRESULT CALLBACK WndProc(HWND hWnd, UINT Message, WPARAM wParam, LPARAM lParam)
{
	switch (Message)		
	{
		case WM_CREATE:
			SetupScreenDetails(hWnd);
			InitOpenGL();
		break;
		case WM_DESTROY:
		case WM_CLOSE:
		case WM_QUIT:			
			ChangeDisplaySettings(NULL, 0);
			wglMakeCurrent(hDC, NULL);
			wglDeleteContext(hRC);
			ReleaseDC(hWnd,hDC);
			PostQuitMessage(0);
		break;

		case WM_KEYDOWN:
			PostMessage(hWnd, WM_DESTROY, 0, 0);				
		break;

		case WM_SIZE:
			ResizeScene(LOWORD(lParam),HIWORD(lParam));
		break;
	}
	return DefWindowProc(hWnd, Message, wParam, lParam);
}

