// -----------------------------------------------------------------------------
//
//      Mimic Engine
//      Copyright (C) 1997-1999 by Maciej Sinilo
//
//      MODULE  : VL_MAT4.CPP - 4x4 matrices
//      CREATED : 24-03-99
//
// -----------------------------------------------------------------------------

#include <iomanip.h>
#include <string.h>

#include "vl_mat4.h"
#include "vl_quat.h"


// --- CLASS BODY --------------------------------------------------------------

vlMatrix4::vlMatrix4(vlMatrixInit init)
{
        if (init == VL_MATRIX_INIT_IDENTITY)
                MakeIdentity();
        else if (init == VL_MATRIX_INIT_ZERO)
                MakeZero();
}

vlMatrix4::vlMatrix4(const vl_real a, const vl_real b, const vl_real c, const vl_real d,
                  const vl_real e, const vl_real f, const vl_real g, const vl_real h,
                  const vl_real i, const vl_real j, const vl_real k, const vl_real l,
                  const vl_real m, const vl_real n, const vl_real o, const vl_real p)
{
        mRows[0] = vlVec4(a, b, c, d);
        mRows[1] = vlVec4(e, f, g, h);
        mRows[2] = vlVec4(i, j, k, l);
        mRows[3] = vlVec4(m, n, o, p);
}

vlMatrix4::vlMatrix4(const vlVec3& xAxis, const vlVec3& yAxis,
        const vlVec3& zAxis, const vlPnt3& origin)
{
#ifdef VL_ROW_ORIENT
        mRows[0] = vlVec4(xAxis);
        mRows[1] = vlVec4(yAxis);
        mRows[2] = vlVec4(zAxis);
        mRows[3] = vlVec4(origin[0], origin[1], origin[2], 1);
#else
        mRows[0] = vlVec4(xAxis[0], yAxis[0], zAxis[0], origin[0]);
        mRows[1] = vlVec4(xAxis[1], yAxis[1], zAxis[1], origin[1]);
        mRows[2] = vlVec4(xAxis[2], yAxis[2], zAxis[2], origin[2]);
        mRows[3] = vlVec4(VL_ZERO, VL_ZERO, VL_ZERO, VL_ONE);
#endif
}

vlMatrix4& vlMatrix4::operator = (const vlMatrix4& other)
{
        memcpy((void *)mRows, (void *)other.mRows, sizeof(other.mRows));
        return SELF;
}

vlMatrix4& vlMatrix4::operator += (const vlMatrix4& other)
{
        mRows[0] += other[0];
        mRows[1] += other[1];
        mRows[2] += other[2];
        mRows[3] += other[3];
        return SELF;
}

vlMatrix4& vlMatrix4::operator -= (const vlMatrix4& other)
{
        mRows[0] -= other[0];
        mRows[1] -= other[1];
        mRows[2] -= other[2];
        mRows[3] -= other[3];
        return SELF;
}

vlMatrix4& vlMatrix4::operator *= (const vlMatrix4& other)
{
        SELF = SELF * other;
        return SELF;
}

vlMatrix4& vlMatrix4::operator *= (const vl_real s)
{
        mRows[0] *= s;
        mRows[1] *= s;
        mRows[2] *= s;
        mRows[3] *= s;
        return SELF;
}

vlMatrix4& vlMatrix4::operator /= (const vl_real s)
{
        mRows[0] /= s;
        mRows[1] /= s;
        mRows[2] /= s;
        mRows[3] /= s;
        return SELF;
}

// -----------------------------------------------------------------------------

void vlMatrix4::MakeZero()
{
        memset((void *)mRows, 0, sizeof(mRows));
}

void vlMatrix4::MakeIdentity(const vl_real val)
{
        MakeZero();
        mRows[0][0] = val;
        mRows[1][1] = val;
        mRows[2][2] = val;
        mRows[3][3] = val;
}

void vlMatrix4::Transpose()
{
        vl_real tmp;

#define SWAP(x, y) tmp = y; y = x; x = tmp
        SWAP(mRows[0][1], mRows[1][0]);
        SWAP(mRows[0][2], mRows[2][0]);
        SWAP(mRows[0][3], mRows[3][0]);
        SWAP(mRows[1][2], mRows[2][1]);
        SWAP(mRows[1][3], mRows[3][1]);
        SWAP(mRows[2][3], mRows[3][2]);
#undef SWAP
}

void vlMatrix4::Transpose3x3()
{
        vl_real tmp;

#define SWAP(x, y) tmp = y; y = x; x = tmp
        SWAP(mRows[0][1], mRows[1][0]);
        SWAP(mRows[0][2], mRows[2][0]);
        SWAP(mRows[1][2], mRows[2][1]);
#undef SWAP
}


// -----------------------------------------------------------------------------
// vlMatrix4::InvertFull - complete inverse of 4x4 matrix [SLOW!]
// Consider using vlMatrix4::InvertAffine instead.
void vlMatrix4::InvertFull()
{
        // Create adjoint matrix
        vlMatrix4 adj;
        adj[0] =  Cross(mRows[1], mRows[2], mRows[3]);
        adj[1] = -Cross(mRows[0], mRows[2], mRows[3]);
        adj[2] =  Cross(mRows[0], mRows[1], mRows[3]);
        adj[3] = -Cross(mRows[0], mRows[1], mRows[2]);

        // Find determinant
        vl_real determinant = adj[0] * mRows[0];

        // Test singularity
        if (determinant == VL_ZERO)
                return;

        adj.Transpose();
        adj /= determinant;

        // Assign
        SELF = adj;
}

// vlMatrix4::InvertAffine - inverse transformation FAST. The price of
// speed is universality - it works only with affine matrices
// (no homogeneous stuff - last column (or row) assumed 0,0,0,1.
// upper 3x3 sub-matrix should be orthogonal)
//
void vlMatrix4::InvertAffine()
{
        // Transpose 3x3 sub-matrix (rotation)
        Transpose3x3();

        // Back-transform position
        // Set additional column/row to identity (0,0,0,1)
#ifdef VL_ROW_ORIENT
        mRows[3][0] = -(mRows[3][0]*mRows[0][0] + mRows[3][1]*mRows[1][0] +
                        mRows[3][2]*mRows[2][0]);
        mRows[3][1] = -(mRows[3][0]*mRows[0][1] + mRows[3][1]*mRows[1][1] +
                        mRows[3][2]*mRows[2][1]);
        mRows[3][2] = -(mRows[3][0]*mRows[0][2] + mRows[3][1]*mRows[1][2] +
                        mRows[3][2]*mRows[2][2]);
        mRows[0][3] = mRows[1][3] = mRows[2][3] = VL_ZERO;
        mRows[3][3] = VL_ONE;
#else
        mRows[0][3] = -(mRows[0][3]*mRows[0][0] + mRows[1][3]*mRows[0][1] +
                        mRows[2][3]*mRows[0][2]);
        mRows[1][3] = -(mRows[0][3]*mRows[1][0] + mRows[1][3]*mRows[1][1] +
                        mRows[2][3]*mRows[1][2]);
        mRows[2][3] = -(mRows[0][3]*mRows[2][0] + mRows[1][3]*mRows[2][1] +
                        mRows[2][3]*mRows[2][2]);
        mRows[3] = vlVec4(VL_ZERO, VL_ZERO, VL_ZERO, VL_ONE);
#endif
}


vlMatrix4 operator * (const vlMatrix4& a, const vlMatrix4& b)
{
        vlMatrix4 r;

#define MUL_ROW4(ro) \
        r(ro,0) = a(ro,0)*b(0,0) + a(ro,1)*b(1,0) + a(ro,2)*b(2,0) + a(ro,3)*b(3,0);\
        r(ro,1) = a(ro,0)*b(0,1) + a(ro,1)*b(1,1) + a(ro,2)*b(2,1) + a(ro,3)*b(3,1);\
        r(ro,2) = a(ro,0)*b(0,2) + a(ro,1)*b(1,2) + a(ro,2)*b(2,2) + a(ro,3)*b(3,2);\
        r(ro,3) = a(ro,0)*b(0,3) + a(ro,1)*b(1,3) + a(ro,2)*b(2,3) + a(ro,3)*b(3,3)

        MUL_ROW4(0);
        MUL_ROW4(1);
        MUL_ROW4(2);
        MUL_ROW4(3);

#undef MUL_ROW4

        return r;
}

bool operator == (const vlMatrix4& a, const vlMatrix4& b)
{
        for (int i = 0; i < 4; ++i)
                for (int j = 0; j < 4; ++j)
                        if (a(i, j) != b(i, j))
                                return false;
        return true;
}

bool operator != (const vlMatrix4& a, const vlMatrix4& b)
{
        for (int i = 0; i < 4; ++i)
                for (int j = 0; j < 4; ++j)
                        if (a(i, j) != b(i, j))
                                return true;
        return false;
}

bool AlmostEqual(const vlMatrix4& a, const vlMatrix4& b, vl_real tol)
{
        return ((AlmostEqual(a[0], b[0], tol)) && (AlmostEqual(a[1], b[1], tol)) &&
                (AlmostEqual(a[2], b[2], tol)) && (AlmostEqual(a[3], b[3], tol)));
}


// -----------------------------------------------------------------------------
// vlMatrix4::MakeRotation - prepare rotation matrix (angle/axis method)
// INPUT:  axis - axis of rotation
//         radians - angle of rotation in radians
// OUTPUT: reference to self
vlMatrix4& vlMatrix4::MakeRotation(const vlVec3& axis, vl_real radians)
{
        vl_real s;
        vlVec3 v = axis;

        // Normalize if neecessary()
        if (v.Mag2() != VL_ONE)
                v.Normalize();

        vl_real half = radians * VL_HALF;
        s = sin(half);

        // Convert quaternion to matrix
        SELF = (vlMatrix4)vlQuaternion(s * v[0], s * v[1],
                            s * v[2], cos(half)).ToMatrix();
        return SELF;
}

vlMatrix4& vlMatrix4::MakeScale(const vlVec3& v)
{
        MakeZero();
        mRows[0][0] = v[0];
        mRows[1][1] = v[1];
        mRows[2][2] = v[2];
        mRows[3][3] = VL_ONE;
        return SELF;
}

vlMatrix4& vlMatrix4::MakeTranslation(const vlVec3& v)
{
        MakeIdentity();
#ifdef VL_ROW_ORIENT
        mRows[3][0] = v[0];
        mRows[3][1] = v[1];
        mRows[3][2] = v[2];
#else
        mRows[0][3] = v[0];
        mRows[1][3] = v[1];
        mRows[2][3] = v[2];
#endif
        return SELF;
}


// -----------------------------------------------------------------------------
// Create perspective matrix
// NOTE: this should be moved on a higher level (preferrably to
// render class) because different APIs use different perspective
// matrices (for example OpenGL vs D3D).
// This is here only for really simple stuff, not using any of 3D
// interfaces.
vlMatrix4& vlMatrix4::MakePerspective(const vl_real D, const vl_real F,
        const vl_real radiansFOV, const vl_real aspect)
{
        MakeIdentity();
// These formulas comes from a document about texture-mapping :-).
// FATMAP by MRI/Doomsday, and they were provided to MRI by Kevin Baca.
//    1  0  0  0
//    0  a  0  0
//    0  0  b  c
//    0  0  f  0
//    where:
//    a = the aspect ratio (width / height of screen)
//    b = f * (yon / (yon - hither))
//    c = -f * (yon * hither / (yon - hither))
//    f = sin(aov / 2) / cos(aov / 2)
//    aov = angle of view
//    yon = distance to far clipping plane
//    hither = distance to near clipping plane
        vl_real halfFOV = radiansFOV * VL_HALF;
        vl_real sincos = (vl_real)sin(halfFOV) / (vl_real)cos(halfFOV);
        vl_real ooDMinF = VL_ONE / (D - F);

        mRows[0][0] = VL_ONE;
        mRows[1][1] = aspect;
        mRows[2][2] = sincos * D * ooDMinF;
        mRows[3][2] = -sincos * (D * F * ooDMinF);
        mRows[2][3] = sincos;
        mRows[3][3] = VL_ZERO;
        return SELF;
}

// -----------------------------------------------------------------------------
// vlMatrix4::PreTranslation - perform translation on matrix in
// preconcatenation fashion - translation will hapen AFTER all other
// transformations
// For row-oriented transforms it works like:
//      SELF = SELF * MakeTranslation(v)
// For column-oriented transforms it works like:
//      SELF = MakeTranslation(v) * SELF
// But it's faster then both mentioned methods, so use it if possible.
vlMatrix4& vlMatrix4::PreTranslation(const vlVec3& v)
{
#ifdef VL_ROW_ORIENT
        mRows[3][0] += v[0];
        mRows[3][1] += v[1];
        mRows[3][2] += v[2];
#else
        mRows[0][3] += v[1];
        mRows[1][3] += v[2];
        mRows[2][3] += v[2];
#endif
        return SELF;
}

// Normalize transformation.
// Warning! It will destroy all scaling stuff [bug? possible FIXME?]
void vlMatrix4::Normalize()
{
#ifdef VL_ROW_ORIENT
        vlVec3 right = vlVec3(mRows[0][0], mRows[0][1], mRows[0][2]);
        vlVec3 up = vlVec3(mRows[1][0], mRows[1][1], mRows[1][2]);
        vlVec3 fwd = vlVec3(mRows[2][0], mRows[2][1], mRows[2][2]);

        // Scale translation vector
        mRows[3][0] /= right.Mag();
        mRows[3][1] /= up.Mag();
        mRows[3][2] /= fwd.Mag();

        right.Normalize();
        up.Normalize();
        fwd = Cross(right, up);
        mRows[0] = vlVec4(right);
        mRows[1] = vlVec4(up);
        mRows[2] = vlVec4(fwd);
#else
        vlVec3 right = vlVec3(mRows[0][0], mRows[1][0], mRows[2][0]);
        vlVec3 up = vlVec3(mRows[0][1], mRows[1][1], mRows[2][1]);
        vlVec3 fwd = vlVec3(mRows[0][2], mRows[1][2], mRows[2][2]);

        // Scale translation vector
        mRows[0][3] /= right.Mag();
        mRows[1][3] /= up.Mag();
        mRows[2][3] /= fwd.Mag();

        right.Normalize();
        up.Normalize();
        fwd = Cross(right, up);

        mRows[0][0] = right[0]; mRows[1][0] = right[1]; mRows[2][0] = right[2];
        mRows[0][1] = up[0];    mRows[1][1] = up[1];    mRows[2][1] = up[2];
        mRows[0][2] = fwd[0];   mRows[1][2] = fwd[1];   mRows[2][2] = fwd[2];
#endif
}


// -----------------------------------------------------------------------------
// vlMatrix4::SetOrientation - set orientation matrix from position,
// target and roll (bank) angle
// NOTE: This wasn't tested very heavily so watch out for possible bugs.
const vl_real kSetOrientEpsilon = 0.01;
void vlMatrix4::SetOrientation(const vlPnt3&pos, const vlPnt3& at,
        const vl_real roll)
{
        // Create direction vector
        vlVec3 zAxis = at - pos;
        zAxis.Normalize();

        vlVec3 up(VL_AXIS_Y);
        vlVec3 xAxis = Cross(up, zAxis);
        if (xAxis.Mag() < kSetOrientEpsilon)
                xAxis = Cross(VL_AXIS_Z, zAxis);
        xAxis.Normalize();

        vlVec3 yAxis = Cross(xAxis, zAxis).Normalize();

        vlMatrix4 align;

        align[0] = vlVec4(xAxis);
        align[1] = vlVec4(yAxis);
        align[2] = vlVec4(zAxis);
        align[3] = VL_AXIS_W;

        //
        // ? should it be rotation against Z axis or maybe against
        //   (at - pos) axis ?
        //
        align *= vlRotation(VL_AXIS_Z, roll);
        
        SELF = align;
}

// -----------------------------------------------------------------------------

ostream& operator << (ostream& s, const vlMatrix4& m)
{
        int w = s.width();

        s << '[' << setw(w) << m[0] << endl
                 << setw(w) << m[1] << endl
                 << setw(w) << m[2] << endl
                 << setw(w) << m[3] << setw(w) << ']';
        return s;
}

// -----------------------------------------------------------------------------
//      VL_MAT4.CPP - MODULE END
// -----------------------------------------------------------------------------

