#ifndef _VL_MAT4_H_
#define _VL_MAT4_H_
// -----------------------------------------------------------------------------
//
//      Mimic Engine
//      Copyright (C) 1997-1999 by Maciej Sinilo
//
//      MODULE  : VL_MAT4.H - 4x4 matrices
//      CREATED : 24-03-99
//
// -----------------------------------------------------------------------------

#include "vl_main.h"
#include "vl_vec4.h"

// --- ENUMERATIONS ------------------------------------------------------------

enum vlMatrixInit
{
        VL_MATRIX_INIT_IDENTITY,
        VL_MATRIX_INIT_ZERO,
        VL_MATRIX_INIT_NONE
};




// --- CLASS DEFINITION --------------------------------------------------------
//
// 4x4 matrix class + basic 3D functions
// TODO: some speed-optimizations (matrix type detecting).
//
class vlMatrix4
{
public:
        vlMatrix4(vlMatrixInit eInit = VL_MATRIX_INIT_IDENTITY);
        vlMatrix4(const vlMatrix4& other);
        vlMatrix4(const vl_real a, const vl_real b, const vl_real c, const vl_real d,
                  const vl_real e, const vl_real f, const vl_real g, const vl_real h,
                  const vl_real i, const vl_real j, const vl_real k, const vl_real l,
                  const vl_real m, const vl_real n, const vl_real o, const vl_real p);
        vlMatrix4(const vlVec3& xAxis, const vlVec3& yAxis,
                  const vlVec3& zAxis, const vlPnt3& origin);
        vlMatrix4(const vl_real* pArray);

        vl_real&        operator () (const int r, const int c);
        const vl_real&  operator () (const int r, const int c) const;
        vlVec4&         operator [] (const int row);
        const vlVec4&   operator [] (const int row) const;
        vlVec4          GetColumn0() const;
        vlVec4          GetColumn1() const;
        vlVec4          GetColumn2() const;
        vlVec4          GetColumn3() const;

        vlMatrix4&      operator = (const vlMatrix4& other);
        vlMatrix4&      operator += (const vlMatrix4& other);
        vlMatrix4&      operator -= (const vlMatrix4& other);
        vlMatrix4&      operator *= (const vlMatrix4& other);
        vlMatrix4&      operator *= (const vl_real s);
        vlMatrix4&      operator /= (const vl_real s);

        void            MakeZero();
        void            MakeIdentity(const vl_real val = VL_ONE);
        void            Transpose();
        void            Transpose3x3();
        void            InvertFull();
        void            InvertAffine();

        vlMatrix4&      MakeRotation(const vlVec3& axis, vl_real radians);
        vlMatrix4&      MakeScale(const vlVec3& scale);
        vlMatrix4&      MakeTranslation(const vlVec3& trans);
        vlMatrix4&      MakePerspective(const vl_real D, const vl_real F,
                                const vl_real FOV, const vl_real aspect);
        vlMatrix4&      PreTranslation(const vlVec3& trans);

        void            Normalize();
        void            SetOrientation(const vlPnt3&pos, const vlPnt3& at,
                                const vl_real roll);

        vl_real* GetPtr()                  { return (vl_real*)mRows;       }
        const vl_real* GetReadPtr() const  { return (const vl_real*)mRows; }
        operator vl_real* ()               { return (vl_real *)mRows;      }
        operator const vl_real*() const    { return (const vl_real*)mRows; }

        void            FromArray(const vl_real* pArray);

protected:

        vlVec4  mRows[4];       // 4 rows, 4 elements each
};


vlMatrix4 operator * (const vlMatrix4& a, const vlMatrix4& b);
bool operator == (const vlMatrix4& a, const vlMatrix4& b);
bool operator != (const vlMatrix4& a, const vlMatrix4& b);
// Returns true if matrices are ALMOST equal. Each element of 'a' should
// be within 'tol' tolerance of the corresponding element 'b'.
bool AlmostEqual(const vlMatrix4& a, const vlMatrix4& b, vl_real tol);

VL_INL vlMatrix4 vlRotation(const vlVec3& axis, vl_real radians);
VL_INL vlMatrix4 vlScale(const vlVec3& s);
VL_INL vlMatrix4 vlTranslation(const vlVec3& v);
VL_INL vlMatrix4 vlPerspective(const vl_real D, const vl_real F,
        const vl_real FOV, const vl_real aspect);

// Streams
// TODO: istream operator
ostream& operator << (ostream& s, const vlMatrix4& m);


// --- CLASS IMPLEMENTATION (INLINES) ------------------------------------------

VL_INL vlMatrix4::vlMatrix4(const vlMatrix4& other) { SELF = other; }
VL_INL vlMatrix4::vlMatrix4(const vl_real* pArray)  { FromArray(pArray); }
VL_INL vl_real& vlMatrix4::operator () (const int r, const int c) { return mRows[r][c]; }
VL_INL const vl_real& vlMatrix4::operator () (const int r, const int c) const { return mRows[r][c]; }
VL_INL vlVec4& vlMatrix4::operator [] (const int row) { return mRows[row]; }
VL_INL const vlVec4& vlMatrix4::operator [] (const int row) const { return mRows[row]; }
VL_INL vlVec4 vlMatrix4::GetColumn0() const { return vlVec4(mRows[0][0], mRows[1][0], mRows[2][0], mRows[3][0]); }
VL_INL vlVec4 vlMatrix4::GetColumn1() const { return vlVec4(mRows[0][1], mRows[1][1], mRows[2][1], mRows[3][1]); }
VL_INL vlVec4 vlMatrix4::GetColumn2() const { return vlVec4(mRows[0][2], mRows[1][2], mRows[2][2], mRows[3][2]); }
VL_INL vlVec4 vlMatrix4::GetColumn3() const { return vlVec4(mRows[0][3], mRows[1][3], mRows[2][3], mRows[3][3]); }
VL_INL void vlMatrix4::FromArray(const vl_real* pArray)
{
        mRows[0].FromArray(pArray);
        mRows[1].FromArray(pArray + 4);
        mRows[2].FromArray(pArray + 8);
        mRows[3].FromArray(pArray + 12);
}

VL_INL vlMatrix4 vlRotation(const vlVec3& axis, vl_real radians)
{ vlMatrix4 res(VL_MATRIX_INIT_NONE); res.MakeRotation(axis, radians); return res; }
VL_INL vlMatrix4 vlScale(const vlVec3& s)
{ vlMatrix4 res(VL_MATRIX_INIT_NONE); res.MakeScale(s); return res; }
VL_INL vlMatrix4 vlTranslation(const vlVec3& v)
{ vlMatrix4 res(VL_MATRIX_INIT_NONE); res.MakeTranslation(v); return res; }
VL_INL vlMatrix4 vlPerspective(const vl_real D, const vl_real F, const vl_real FOV,
        const vl_real aspect)
{ vlMatrix4 res(VL_MATRIX_INIT_NONE); res.MakePerspective(D, F, FOV, aspect); return res; }



// -----------------------------------------------------------------------------
//      VL_MAT4.H - HEADER END
// -----------------------------------------------------------------------------
#endif  // !_VL_MAT4_H_
