#ifndef _VL_VEC2_H_
#define _VL_VEC2_H_
// -----------------------------------------------------------------------------
//
//      Mimic Engine
//      Copyright (C) 1997-1999 by Maciej Sinilo
//
//      MODULE  : VL_VEC2.H - 2D Vector/Point
//      CREATED : 24-03-99
//
// -----------------------------------------------------------------------------

#include "vl_main.h"




// --- vlBase2 DEFINITION ------------------------------------------------------
//
// Base class for 2D points/vectors. User don't have right to create
// instances of vlBase2 class, so all constructors are protected.
//
class vlBase2
{
public:
        // Set coordinates
        vlBase2& operator () (const vl_real x, const vl_real y);
        vlBase2& Set(const vl_real x, const vl_real y);

        // Get/set element (coordinate)
        vl_real& operator [] (const int index);
        const vl_real& operator [] (const int index) const;

        // Conversion
        operator vl_real* ()            { return (vl_real *)p;        }
        operator const vl_real*() const { return (const vl_real*)p;   }

        // Get/set element (coordinate)
        // Other fashion, more convenient when using pointers
        vl_real&            rX()             { return p[0]; }
        const vl_real&      rX() const       { return p[0]; }
        vl_real&            rY()             { return p[1]; }
        const vl_real&      rY() const       { return p[1]; }

        // Operations
        void MakeZero();
        vl_real* GetPtr()                 { return (vl_real*)p;       }
        const vl_real* GetReadPtr() const { return (const vl_real*)p; }

        void FromArray(const vl_real* pArray);

protected:

        vlBase2();
        vlBase2(const vl_real x, const vl_real y);
        vlBase2(const vlBase2& other);
        vlBase2(const vl_real* pArray);

        vlBase2& operator = (const vlBase2& other);

        vl_real   p[2];                   // Coordinates
};
// -------------------------------------------------
//
//  Base operators declared outside the class
//
// -------------------------------------------------
// Comparisons
VL_INL bool operator > (const vlBase2&, const vlBase2&);
VL_INL bool operator < (const vlBase2&, const vlBase2&);
VL_INL bool operator >= (const vlBase2&, const vlBase2&);
VL_INL bool operator <= (const vlBase2&, const vlBase2&);
VL_INL bool operator == (const vlBase2&, const vlBase2&);
VL_INL bool operator != (const vlBase2&, const vlBase2&);
// Return true if two vectors are ALMOST equal (== operator return true only
// when vectors are EXACTLY equal). The difference between each coordinate
// should be less than tol.
VL_INL bool AlmostEqual(const vlBase2&, const vlBase2&, vl_real tol);

// Streams
ostream& operator << (ostream &s, const vlBase2& v);



// --- vlPnt2 DEFINITION--------------------------------------------------------

class vlPnt2 : public vlBase2
{
public:
        vlPnt2();
        vlPnt2(const vl_real x, const vl_real y);
        vlPnt2(const vl_real* pArray);

        vlPnt2& operator += (const vlVec2& v);
        vlPnt2& operator -= (const vlVec2& v);

        operator vlVec2();
};
// -------------------------------------------------
//
//  Point operators declared outside the class
//
// -------------------------------------------------
VL_INL vlPnt2 operator + (const vlPnt2&, const vlVec2&);
VL_INL vlPnt2 operator + (const vlVec2&, const vlPnt2&);
VL_INL vlPnt2 operator - (const vlPnt2&, const vlVec2&);

// Distance between two points in 2D
VL_INL vl_real Dist(const vlPnt2& begin, const vlPnt2& end);
// Distance between two points in 2D
// (Use this one if possible because it's faster (no sqrt))
VL_INL vl_real Dist2(const vlPnt2& begin, const vlPnt2& end);




// --- vlVec2 DEFINITION -------------------------------------------------------

class vlVec2 : public vlBase2
{
public:

        vlVec2();
        vlVec2(const vl_real x, const vl_real y);
        vlVec2(vlAxis axis);
        vlVec2(const vl_real* pArray);

        vlVec2& operator = (vlAxis axis);
        vlVec2& operator += (const vlVec2& v);
        vlVec2& operator -= (const vlVec2& v);

        vlVec2& operator *= (const vl_real s);
        vlVec2& operator /= (const vl_real s);

        // Type casting (to point)
        operator vlPnt2 ();

        // Negation (unary operator).
        // Declared as class member to avoid automatic compiler conversions
        vlVec2 operator - () const;


// OPERATIONS

        vl_real Mag() const;              // Magnitude
        vl_real Mag2() const;             // Squared magnitude (FASTER!)
        int Dominant() const;             // Dominant axis (index)

        vlVec2& Normalize(const vl_real value = VL_ONE);
};
// -------------------------------------------------
//
//  Vector operators declared outside the class
//
// -------------------------------------------------
VL_INL vlVec2 operator - (const vlVec2&);      // Negation
VL_INL vlVec2 operator + (const vlVec2&, const vlVec2 &);
VL_INL vlVec2 operator - (const vlVec2&, const vlVec2 &);
// Make vector from two points
VL_INL vlVec2 operator - (const vlPnt2& end, const vlPnt2& begin);
// Scale vector by scalar
VL_INL vlVec2 operator * (const vlVec2&, const vl_real);
VL_INL vlVec2 operator * (const vl_real, const vlVec2&);
VL_INL vlVec2 operator / (const vlVec2&, const vl_real);
// Dot product of two vectors
VL_INL vl_real operator * (const vlVec2&, const vlVec2&);
// Cross product (one vector)
// I prefer clear function than some fancy operators... How am I supposed
// to know that ^ for example stands for cross product?
VL_INL vlVec2 Cross(const vlVec2&);
// Component mul/div Result vector:
//  result[0] = a[0] *(/) b[0]
//  result[1] = a[1] *(/) b[1]
// Convenient when using vectors as colors for example.
VL_INL vlVec2 BlockMul(const vlVec2& a, const vlVec2& b);
VL_INL vlVec2 BlockDiv(const vlVec2& a, const vlVec2& b);




// --- vlBase2 IMPLEMENTATION --------------------------------------------------

VL_INL vlBase2::vlBase2() { }
VL_INL vlBase2::vlBase2(const vl_real x, const vl_real y)
{
        p[0] = x;
        p[1] = y;
}
VL_INL vlBase2::vlBase2(const vlBase2& other)
{
        p[0] = other[0];
        p[1] = other[1];
}
VL_INL vlBase2::vlBase2(const vl_real* pArray) { FromArray(pArray); }
VL_INL vlBase2& vlBase2::operator = (const vlBase2& other) { p[0] = other[0]; p[1] = other[1]; return SELF; }
VL_INL vlBase2& vlBase2::operator () (const vl_real x, const vl_real y) { p[0] = x; p[1] = y; return SELF; }
VL_INL vlBase2& vlBase2::Set(const vl_real x, const vl_real y) { p[0] = x; p[1] = y; return SELF; }
VL_INL vl_real& vlBase2::operator [] (const int i) { return p[i]; }
VL_INL const vl_real& vlBase2::operator [] (const int i) const { return p[i]; }
VL_INL void vlBase2::MakeZero() { p[0] = VL_ZERO; p[1] = VL_ZERO; }
VL_INL void vlBase2::FromArray(const vl_real* pArray) { p[0] = pArray[0]; p[1] = pArray[1]; }

VL_INL bool operator > (const vlBase2& a, const vlBase2& b) { return ((a[0]>b[0]) && (a[1]>b[1])); }
VL_INL bool operator >= (const vlBase2& a, const vlBase2& b) { return ((a[0]>=b[0]) && (a[1]>=b[1])); }
VL_INL bool operator < (const vlBase2& a, const vlBase2& b) { return ((a[0]<b[0]) && (a[1]<b[1])); }
VL_INL bool operator <= (const vlBase2& a, const vlBase2& b) { return ((a[0]<=b[0]) && (a[1]<=b[1])); }
VL_INL bool operator == (const vlBase2& a, const vlBase2& b) { return ((a[0]==b[0]) && (a[1]==b[1])); }
VL_INL bool operator != (const vlBase2& a, const vlBase2& b) { return ((a[0]!=b[0]) || (a[1]!=b[1])); }
VL_INL bool AlmostEqual(const vlBase2& a, const vlBase2& b, vl_real tol) { return ((vlTolComp(a[0],b[0],tol)==0) && (vlTolComp(a[1],b[1],tol)==0)); }


// --- vlPnt2 IMPLEMENTATION ---------------------------------------------------

VL_INL vlPnt2::vlPnt2() : vlBase2() { }
VL_INL vlPnt2::vlPnt2(const vl_real x, const vl_real y) : vlBase2(x, y) { }
VL_INL vlPnt2::vlPnt2(const vl_real* pArray) : vlBase2(pArray) { }
VL_INL vlPnt2::operator vlVec2 () { return vlVec2(p[0], p[1]); }
VL_INL vlPnt2& vlPnt2::operator += (const vlVec2& v) { p[0] += v[0]; p[1] += v[1]; return SELF; }
VL_INL vlPnt2& vlPnt2::operator -= (const vlVec2& v) { p[0] -= v[0]; p[1] -= v[1]; return SELF; }

VL_INL vlPnt2 operator + (const vlPnt2& a, const vlVec2& b) { return vlPnt2(a[0]+b[0], a[1]+b[1]); }
VL_INL vlPnt2 operator + (const vlVec2& a, const vlPnt2& b) { return vlPnt2(a[0]+b[0], a[1]+b[1]); }
VL_INL vlPnt2 operator - (const vlPnt2& a, const vlVec2& b) { return vlPnt2(a[0]-b[0], a[1]-b[1]); }
VL_INL vl_real Dist(const vlPnt2& a, const vlPnt2& b) { return vlVec2(b-a).Mag(); }
VL_INL vl_real Dist2(const vlPnt2& a, const vlPnt2& b) { return vlVec2(b-a).Mag2(); }




// --- vlVec2 IMPLEMENTATION ---------------------------------------------------


VL_INL vlVec2::vlVec2() : vlBase2() { }
VL_INL vlVec2::vlVec2(const vl_real x, const vl_real y) : vlBase2(x, y) { }
VL_INL vlVec2::vlVec2(vlAxis axis) { SELF = axis; }
VL_INL vlVec2::vlVec2(const vl_real* pArray) : vlBase2(pArray) { }

VL_INL vlVec2& vlVec2::operator = (vlAxis axis) { axis == VL_AXIS_X ? Set(VL_ONE, VL_ZERO) : Set(VL_ZERO, VL_ONE); return SELF; }
VL_INL vlVec2& vlVec2::operator += (const vlVec2& v) { p[0] += v[0]; p[1] += v[1]; return SELF; }
VL_INL vlVec2& vlVec2::operator -= (const vlVec2& v) { p[0] -= v[0]; p[1] -= v[1]; return SELF; }
VL_INL vlVec2& vlVec2::operator *= (const vl_real s) { p[0] *= s; p[1] *= s; return SELF; }
// WARNING: divides are slow, use multiplying by reciprocal if possible!
VL_INL vlVec2& vlVec2::operator /= (const vl_real s) { p[0] /= s; p[1] /= s; return SELF; }
VL_INL vlVec2::operator vlPnt2 () { return vlPnt2(p[0], p[1]); }
VL_INL vlVec2 vlVec2::operator - () const { return vlVec2(-p[0], -p[1]); }

VL_INL vl_real vlVec2::Mag() const { return vlSqrt(vlSquare(p[0]) + vlSquare(p[1])); }
VL_INL vl_real vlVec2::Mag2() const { return (vlSquare(p[0]) + vlSquare(p[1])); }
VL_INL int vlVec2::Dominant() const { return (vlAbs(p[0]) < vlAbs(p[1])); }
VL_INL vlVec2& vlVec2::Normalize(const vl_real val)
{
        vl_real mag = Mag();

        // How to handle this case? No idea... so return
        if (mag == VL_ZERO)
                return SELF;

        vl_real recip = val / mag;
        p[0] *= recip;
        p[1] *= recip;
        return SELF;
}

VL_INL vlVec2 operator + (const vlVec2& a, const vlVec2& b) { return vlVec2(a[0]+b[0], a[1]+b[1]); }
VL_INL vlVec2 operator - (const vlVec2& a, const vlVec2& b) { return vlVec2(a[0]-b[0], a[1]-b[1]); }
VL_INL vlVec2 operator - (const vlPnt2& a, const vlPnt2& b) { return vlVec2(a[0]-b[0], a[1]-b[1]); }
VL_INL vlVec2 operator * (const vlVec2& v, const vl_real s) { return vlVec2(v[0]*s, v[1]*s); }
VL_INL vlVec2 operator * (const vl_real s, const vlVec2& v) { return vlVec2(v[0]*s, v[1]*s); }
// WARNING: divides are slow, use multiplying by reciprocal if possible!
VL_INL vlVec2 operator / (const vlVec2& v, const vl_real s) { return vlVec2(v[0]/s, v[1]/s); }
VL_INL vl_real operator * (const vlVec2& a, const vlVec2& b) { return (a[0]*b[0] + a[1]*b[1]); }
VL_INL vlVec2 Cross(const vlVec2& a) { return vlVec2(a[1], -a[0]); }
VL_INL vlVec2 BlockMul(const vlVec2& a, const vlVec2& b) { return vlVec2(a[0]*b[0], a[1]*b[1]); }
VL_INL vlVec2 BlockDiv(const vlVec2& a, const vlVec2& b) { return vlVec2(a[0]/b[0], a[1]/b[1]); }


// -----------------------------------------------------------------------------
//      VL_VEC2.H - HEADER END
// -----------------------------------------------------------------------------
#endif  // !_VL_VEC2_H_
