#ifndef _VL_VEC3_H_
#define _VL_VEC3_H_
// -----------------------------------------------------------------------------
//
//      Mimic Engine
//      Copyright (C) 1997-1999 by Maciej Sinilo
//
//      MODULE  : VL_VEC3.H - 3D Vector/Point
//      CREATED : 24-03-99
//
// -----------------------------------------------------------------------------

#include "vl_main.h"



// --- vlBase3 DEFINITION ------------------------------------------------------
//
// Base class for 3D points/vectors. User don't have right to create
// instances of vlBase3 class, so all constructors are protected.
//
class vlBase3
{
public:

        // Set coordinates
        vlBase3& operator () (const vl_real x, const vl_real y, const vl_real z);
        vlBase3& Set(const vl_real x, const vl_real y, const vl_real z);

        // Get/set element (coordinate)
        vl_real& operator [] (const int index);
        const vl_real& operator [] (const int index) const;

        // Conversion
        operator vl_real* ()            { return (vl_real *)p;        }
        operator const vl_real*() const { return (const vl_real*)p;   }

        // Get/set element (coordinate)
        // Other fashion, more convenient when using pointers
        vl_real&            rX()             { return p[0]; }
        const vl_real&      rX() const       { return p[0]; }
        vl_real&            rY()             { return p[1]; }
        const vl_real&      rY() const       { return p[1]; }
        vl_real&            rZ()             { return p[2]; }
        const vl_real&      rZ() const       { return p[2]; }


        // Operations
        void MakeZero();
        vl_real* GetPtr()                 { return (vl_real*)p;       }
        const vl_real* GetReadPtr() const { return (const vl_real*)p; }

        void FromArray(const vl_real* pArray);


protected:

        vlBase3();
        vlBase3(const vl_real x, const vl_real y, const vl_real z);
        vlBase3(const vlBase3& other);
        vlBase3(const vl_real* pArray);

        vlBase3& operator = (const vlBase3& other);

        vl_real   p[3];                   // Coordinates
};
// -------------------------------------------------
//
//  Base operators declared outside the class
//
// -------------------------------------------------
// Comparisons
VL_INL bool operator > (const vlBase3&, const vlBase3&);
VL_INL bool operator < (const vlBase3&, const vlBase3&);
VL_INL bool operator >= (const vlBase3&, const vlBase3&);
VL_INL bool operator <= (const vlBase3&, const vlBase3&);
VL_INL bool operator == (const vlBase3&, const vlBase3&);
VL_INL bool operator != (const vlBase3&, const vlBase3&);
// Return true if two vectors are ALMOST equal (== operator return true only
// when vectors are EXACTLY equal). The difference between each coordinate
// should be less than tol.
VL_INL bool AlmostEqual(const vlBase3&, const vlBase3&, vl_real tol);

// Streams
ostream& operator << (ostream &s, const vlBase3& a);

// --- vlPnt3 DEFINITION -------------------------------------------------------

class vlPnt3 : public vlBase3
{
public:
        vlPnt3();
        vlPnt3(const vl_real x, const vl_real y, const vl_real z);
        vlPnt3(const vl_real* pArray);

        vlPnt3& operator += (const vlVec3& v);
        vlPnt3& operator -= (const vlVec3& v);

        operator vlVec3();
};
// -------------------------------------------------
//
//  Point operators declared outside the class
//
// -------------------------------------------------
VL_INL vlPnt3 operator + (const vlPnt3&, const vlVec3&);
VL_INL vlPnt3 operator + (const vlVec3&, const vlPnt3&);
VL_INL vlPnt3 operator - (const vlPnt3&, const vlVec3&);

// Distance between two points in 3D
VL_INL vl_real Dist(const vlPnt3& begin, const vlPnt3& end);
// Distance between two points in 3D
// (Use this one if possible because it's faster (no vlSqrt))
VL_INL vl_real Dist2(const vlPnt3& begin, const vlPnt3& end);




// --- vlVec3 DEFINITION -------------------------------------------------------

class vlVec3 : public vlBase3
{
public:

        vlVec3();
        vlVec3(const vl_real x, const vl_real y, const vl_real z);
        vlVec3(vlAxis axis);
        vlVec3(const vl_real* pArray);

        vlVec3& operator = (vlAxis axis);
        vlVec3& operator += (const vlVec3& v);
        vlVec3& operator -= (const vlVec3& v);

        vlVec3& operator *= (const vl_real s);
        vlVec3& operator /= (const vl_real s);

        // Type casting (to point)
        operator vlPnt3 ();

        // Negation (unary operator).
        // Declared as class member to avoid automatic compiler conversions
        vlVec3 operator - () const;

// OPERATIONS

        vl_real Mag() const;              // Magnitude
        vl_real Mag2() const;             // Squared magnitude (FASTER!)
        int Dominant() const;           // Dominant axis (index)

        vlVec3& Normalize(const vl_real value = VL_ONE);
};
// -------------------------------------------------
//
//  Vector operators declared outside the class
//
// -------------------------------------------------
//VL_INL vlVec3 operator - (const vlVec3&);      // Negation
VL_INL vlVec3 operator + (const vlVec3&, const vlVec3 &);
VL_INL vlVec3 operator - (const vlVec3&, const vlVec3 &);
// Make vector from two points
VL_INL vlVec3 operator - (const vlPnt3& end, const vlPnt3& begin);
// Scale vector by scalar
VL_INL vlVec3 operator * (const vlVec3&, const vl_real);
VL_INL vlVec3 operator * (const vl_real, const vlVec3&);
VL_INL vlVec3 operator / (const vlVec3&, const vl_real);
// Dot product of two vectors
VL_INL vl_real operator * (const vlVec3&, const vlVec3&);
// Cross product (two vectors)
// I prefer clear function than some fancy operators... How am I supposed
// to know that ^ for example stands for cross product?
VL_INL vlVec3 Cross(const vlVec3&, const vlVec3&);
// Component mul/div Result vector:
//  result[0] = a[0] *(/) b[0]
//  result[1] = a[1] *(/) b[1]
//  result[2] = a[2] *(/) b[2]
// Convenient when using vectors as colors for example.
VL_INL vlVec3 BlockMul(const vlVec3& a, const vlVec3& b);
VL_INL vlVec3 BlockDiv(const vlVec3& a, const vlVec3& b);
// Create normal vector to plane defined by three points
VL_INL vlVec3 CreateNormal(const vlPnt3& a, const vlPnt3& b, const vlPnt3& c);





// --- vlBase3 IMPLEMENTATION --------------------------------------------------

VL_INL vlBase3::vlBase3() { }
VL_INL vlBase3::vlBase3(const vl_real x, const vl_real y, const vl_real z) { p[0]=x; p[1]=y; p[2]=z; }
VL_INL vlBase3::vlBase3(const vlBase3& other) { p[0]=other[0]; p[1]=other[1]; p[2]=other[2]; }
VL_INL vlBase3::vlBase3(const vl_real* pArray) { FromArray(pArray); }
VL_INL vlBase3& vlBase3::operator = (const vlBase3& other) { p[0]=other[0]; p[1]=other[1]; p[2]=other[2]; return SELF; }
VL_INL vlBase3& vlBase3::operator () (const vl_real x, const vl_real y, const vl_real z) { p[0] = x; p[1] = y; p[2] = z; return SELF; }
VL_INL vlBase3& vlBase3::Set(const vl_real x, const vl_real y, const vl_real z) { p[0] = x; p[1] = y; p[2] = z; return SELF; }
VL_INL vl_real& vlBase3::operator [] (const int i) { return p[i]; }
VL_INL const vl_real& vlBase3::operator [] (const int i) const { return p[i]; }
VL_INL void vlBase3::MakeZero() { p[0] = VL_ZERO; p[1] = VL_ZERO; p[2] = VL_ZERO; }
VL_INL void vlBase3::FromArray(const vl_real* pArray) { p[0] = pArray[0]; p[1] = pArray[1]; p[2] = pArray[2]; }

VL_INL bool operator > (const vlBase3& a, const vlBase3& b) { return ((a[0]>b[0]) && (a[1]>b[1]) && (a[2]>b[2])); }
VL_INL bool operator >= (const vlBase3& a, const vlBase3& b) { return ((a[0]>=b[0]) && (a[1]>=b[1]) && (a[2]>=b[2])); }
VL_INL bool operator < (const vlBase3& a, const vlBase3& b) { return ((a[0]<b[0]) && (a[1]<b[1]) && (a[2]<b[2])); }
VL_INL bool operator <= (const vlBase3& a, const vlBase3& b) { return ((a[0]<=b[0]) && (a[1]<=b[1])) && (a[2]<=b[2]); }
VL_INL bool operator == (const vlBase3& a, const vlBase3& b) { return ((a[0]==b[0]) && (a[1]==b[1]) && (a[2]==b[2])); }
VL_INL bool operator != (const vlBase3& a, const vlBase3& b) { return ((a[0]!=b[0]) || (a[1]!=b[1]) || (a[2]!=b[2])); }
VL_INL bool AlmostEqual(const vlBase3& a, const vlBase3& b, vl_real tol)
{
        return ((vlTolComp(a[0], b[0], tol) == 0) &&
                (vlTolComp(a[1], b[1], tol) == 0) &&
                (vlTolComp(a[2], b[2], tol) == 0));
}



// --- vlPnt3 IMPLEMENTATION ---------------------------------------------------

VL_INL vlPnt3::vlPnt3() : vlBase3() { }
VL_INL vlPnt3::vlPnt3(const vl_real x, const vl_real y, const vl_real z) : vlBase3(x, y, z) { }
VL_INL vlPnt3::vlPnt3(const vl_real* pArray) : vlBase3(pArray) { }
VL_INL vlPnt3::operator vlVec3 () { return vlVec3(p[0], p[1], p[2]); }
VL_INL vlPnt3& vlPnt3::operator += (const vlVec3& v) { p[0] += v[0]; p[1] += v[1]; p[2] += v[2]; return SELF; }
VL_INL vlPnt3& vlPnt3::operator -= (const vlVec3& v) { p[0] -= v[0]; p[1] -= v[1]; p[2] -= v[2]; return SELF; }
VL_INL vlPnt3 operator + (const vlPnt3& a, const vlVec3& b) { return vlPnt3(a[0]+b[0], a[1]+b[1], a[2]+b[2]); }
VL_INL vlPnt3 operator + (const vlVec3& a, const vlPnt3& b) { return vlPnt3(a[0]+b[0], a[1]+b[1], a[2]+b[2]); }
VL_INL vlPnt3 operator - (const vlPnt3& a, const vlVec3& b) { return vlPnt3(a[0]-b[0], a[1]-b[1], a[2]-b[2]); }

VL_INL vl_real Dist(const vlPnt3& a, const vlPnt3& b) { return vlVec3(b-a).Mag(); }
VL_INL vl_real Dist2(const vlPnt3& a, const vlPnt3& b) { return vlVec3(b-a).Mag2(); }





// --- vlVec3 IMPLEMENTATION ---------------------------------------------------

VL_INL vlVec3::vlVec3() : vlBase3() { }
VL_INL vlVec3::vlVec3(const vl_real x, const vl_real y, const vl_real z) : vlBase3(x, y, z) { }
VL_INL vlVec3::vlVec3(vlAxis axis) { SELF = axis; }
VL_INL vlVec3::vlVec3(const vl_real* pArray) : vlBase3(pArray) { }

VL_INL vlVec3& vlVec3::operator = (vlAxis axis)
{
        if (axis == VL_AXIS_X)         Set(VL_ONE,  VL_ZERO, VL_ZERO);
        else if (axis == VL_AXIS_Y)    Set(VL_ZERO, VL_ONE,  VL_ZERO);
        else                           Set(VL_ZERO, VL_ZERO, VL_ONE );

        return SELF;
}
VL_INL vlVec3& vlVec3::operator += (const vlVec3& v) { p[0] += v[0]; p[1] += v[1]; p[2] += v[2]; return SELF; }
VL_INL vlVec3& vlVec3::operator -= (const vlVec3& v) { p[0] -= v[0]; p[1] -= v[1]; p[2] -= v[2]; return SELF; }
VL_INL vlVec3& vlVec3::operator *= (const vl_real s) { p[0] *= s; p[1] *= s; p[2] *= s; return SELF; }
// WARNING: divides are slow, use multiplying by reciprocal if possible!
VL_INL vlVec3& vlVec3::operator /= (const vl_real s) { p[0] /= s; p[1] /= s; p[2] /= s; return SELF; }
VL_INL vlVec3::operator vlPnt3 () { return vlPnt3(p[0], p[1], p[2]); }
VL_INL vlVec3 vlVec3::operator - () const { return vlVec3(-p[0], -p[1], -p[2]); }

VL_INL vl_real vlVec3::Mag() const { return vlSqrt(vlSquare(p[0])+vlSquare(p[1])+vlSquare(p[2])); }
VL_INL vl_real vlVec3::Mag2() const { return (vlSquare(p[0])+vlSquare(p[1])+vlSquare(p[2])); }
VL_INL int vlVec3::Dominant() const
{
        return (vlAbs(p[0]) > vlAbs(p[1]) ? 0 : (vlAbs(p[1]) > vlAbs(p[2]) ? 1 : 2));
}
VL_INL vlVec3& vlVec3::Normalize(const vl_real val)
{
        vl_real mag = Mag();

        // How to handle this case? No idea... so return
        if (mag == VL_ZERO)
                return SELF;

        vl_real recip = val / mag;
        p[0] *= recip;
        p[1] *= recip;
        p[2] *= recip;
        return SELF;
}

VL_INL vlVec3 operator + (const vlVec3& a, const vlVec3& b) { return vlVec3(a[0]+b[0], a[1]+b[1], a[2]+b[2]); }
VL_INL vlVec3 operator - (const vlVec3& a, const vlVec3& b) { return vlVec3(a[0]-b[0], a[1]-b[1], a[2]-b[2]); }
VL_INL vlVec3 operator - (const vlPnt3& a, const vlPnt3& b) { return vlVec3(a[0]-b[0], a[1]-b[1], a[2]-b[2]); }
VL_INL vlVec3 operator * (const vlVec3& v, const vl_real s) { return vlVec3(v[0]*s, v[1]*s, v[2]*s); }
VL_INL vlVec3 operator * (const vl_real s, const vlVec3& v) { return vlVec3(v[0]*s, v[1]*s, v[2]*s); }
// WARNING: divides are slow, use multiplying by reciprocal if possible!
VL_INL vlVec3 operator / (const vlVec3& v, const vl_real s) { return vlVec3(v[0]/s, v[1]/s, v[2]/s); }
VL_INL vl_real operator * (const vlVec3& a, const vlVec3& b) { return (a[0]*b[0] + a[1]*b[1] + a[2]*b[2]); }
VL_INL vlVec3 Cross(const vlVec3& a, const vlVec3& b)
{
        return vlVec3(a[1]*b[2]-a[2]*b[1], a[2]*b[0]-a[0]*b[2], a[0]*b[1]-a[1]*b[0]);
}
VL_INL vlVec3 BlockMul(const vlVec3& a, const vlVec3& b) { return vlVec3(a[0]*b[0], a[1]*b[1], a[2]*b[2]); }
VL_INL vlVec3 BlockDiv(const vlVec3& a, const vlVec3& b) { return vlVec3(a[0]/b[0], a[1]/b[1], a[2]/b[2]); }
VL_INL vlVec3 CreateNormal(const vlPnt3& a, const vlPnt3& b, const vlPnt3& c)
{
        vlVec3 v1 = (b - a).Normalize();
        vlVec3 v2 = (c - a).Normalize();
        return (Cross(v1, v2).Normalize());
}

// -----------------------------------------------------------------------------
//      VL_VEC3.H - HEADER END
// -----------------------------------------------------------------------------
#endif  // !_VL_VEC3_H_
