#include <err.h>
#include <stdlib.h>

#include <EGL/egl.h>

#include "egl.h"

EGLDisplay display;
EGLSurface surface;
EGLContext context;

void
egl_cleanup()
{
   eglMakeCurrent(display, EGL_NO_SURFACE, EGL_NO_SURFACE, EGL_NO_CONTEXT);
   eglDestroySurface(display, surface );
   eglDestroyContext(display, context );
   eglTerminate(display );
}

void
egl_init(NativeWindowType native_window)
{
    static const EGLint config_attrib_list[] = {
        EGL_RED_SIZE, 8,
        EGL_GREEN_SIZE, 8,
        EGL_BLUE_SIZE, 8,
        EGL_ALPHA_SIZE, 8,
        EGL_NONE
    };
    static const EGLint context_attrib_list[] = {
        EGL_CONTEXT_CLIENT_VERSION, 2,
        EGL_NONE
    };
    EGLConfig config;
    EGLint num_config;

    if(EGL_DEFAULT_DISPLAY == (display = eglGetDisplay(EGL_DEFAULT_DISPLAY)))
        errx(EXIT_FAILURE, "eglGetDisplay: EGL_NO_DISPLAY, can't obtain display");

    if(EGL_FALSE == eglInitialize(display, NULL, NULL))
        errx(EXIT_FAILURE, "eglInitialize: %s", egl_strerror(eglGetError()));

    if(EGL_FALSE == eglChooseConfig(display, config_attrib_list, &config, 1, &num_config))
        errx(EXIT_FAILURE, "eglChooseConfig: %s", egl_strerror(eglGetError()));

    if(EGL_FALSE == eglBindAPI(EGL_OPENGL_ES_API))
        errx(EXIT_FAILURE, "eglBindAPI: %s", egl_strerror(eglGetError()));

    if(EGL_NO_CONTEXT == (context = eglCreateContext(display, config, EGL_NO_CONTEXT, context_attrib_list)))
        errx(EXIT_FAILURE, "eglCreateContext: %s", egl_strerror(eglGetError()));

    if(EGL_NO_SURFACE == (surface = eglCreateWindowSurface(display, config, native_window, NULL)))
        errx(EXIT_FAILURE, "eglCreateWindowSurface: %s", egl_strerror(eglGetError()));

    if(EGL_FALSE == eglMakeCurrent(display, surface, surface, context))
        errx(EXIT_FAILURE, "eglMakeCurrent: %s", egl_strerror(eglGetError()));
}

void
egl_swapbuffers()
{
   eglSwapBuffers(display, surface);
}

const char *
egl_strerror(EGLint error)
{
    switch(error)
    {
    case EGL_SUCCESS:
        return "EGL_SUCCESS, The last function succeeded without error.";
        break;
    case EGL_NOT_INITIALIZED:
        return "EGL_NOT_INITIALIZED, EGL is not initialized, or could not be initialized, for the specified EGL display connection.";
        break;
    case EGL_BAD_ACCESS:
        return "EGL_BAD_ACCESS, EGL cannot access a requested resource (for example a context is bound in another thread).";
        break;
    case EGL_BAD_ALLOC:
        return "EGL_BAD_ALLOC, EGL failed to allocate resources for the requested operation.";
        break;
    case EGL_BAD_ATTRIBUTE:
        return "EGL_BAD_ATTRIBUTE, An unrecognized attribute or attribute value was passed in the attribute list.";
        break;
    case EGL_BAD_CONTEXT:
        return "EGL_BAD_CONTEXT, An EGLContext argument does not name a valid EGL rendering context.";
        break;
    case EGL_BAD_CONFIG:
        return "EGL_BAD_CONFIG, An EGLConfig argument does not name a valid EGL frame buffer configuration.";
        break;
    case EGL_BAD_CURRENT_SURFACE:
        return "EGL_BAD_CURRENT_SURFACE, The current surface of the calling thread is a window, pixel buffer or pixmap that is no longer valid.";
        break;
    case EGL_BAD_DISPLAY:
        return "EGL_BAD_DISPLAY, An EGLDisplay argument does not name a valid EGL display connection.";
        break;
    case EGL_BAD_SURFACE:
        return "EGL_BAD_SURFACE, An EGLSurface argument does not name a valid surface (window, pixel buffer or pixmap) configured for GL rendering.";
        break;
    case EGL_BAD_MATCH:
        return "EGL_BAD_MATCH, Arguments are inconsistent (for example, a valid context requires buffers not supplied by a valid surface).";
        break;
    case EGL_BAD_PARAMETER:
        return "EGL_BAD_PARAMETER, One or more argument values are invalid.";
        break;
    case EGL_BAD_NATIVE_PIXMAP:
        return "EGL_BAD_NATIVE_PIXMAP, A NativePixmapType argument does not refer to a valid native pixmap.";
        break;
    case EGL_BAD_NATIVE_WINDOW:
        return "EGL_BAD_NATIVE_WINDOW, A NativeWindowType argument does not refer to a valid native window.";
        break;
    case EGL_CONTEXT_LOST:
        return "EGL_CONTEXT_LOST, A power management event has occurred. The application must destroy all contexts and reinitialise OpenGL ES state and objects to continue rendering.";
        break;
    default:
        return "Unknown EGL error.";
    }
}
