#include <err.h>
#include <math.h>
#include <signal.h>
#include <stdint.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>

#include <EGL/egl.h>
#include <GLES2/gl2.h>

#include <egl.h>
#include <tty.h>
#include <vc.h>

const GLfloat kubb_vertices[] = {
    -1, 1, -1, 1, 0, 1,
    -1, 1, 1, 1, 0, 0,
    1, 1, 1, 1, 1, 0,
    1, 1, -1, 1, 1, 1,

    -1, -1, -1, 1, 0, 0,
    -1, -1, 1, 1, 0, 1,
    1, -1, 1, 1, 1, 1,
    1, -1, -1, 1, 1, 0,

    -1, -1, 1, 1, 0, 0,
    1, -1, 1, 1, 1, 0,
    1, 1, 1, 1, 1, 1,
    -1, 1, 1, 1, 0, 1,

    -1, -1, -1, 1, 0, 1,
    1, -1, -1, 1, 1, 1,
    1, 1, -1, 1, 1, 0,
    -1, 1, -1, 1, 0, 0,

    1, -1, -1, 1, 0, 0,
    1, 1, -1, 1, 0, 1,
    1, 1, 1, 1, 1, 1,
    1, -1, 1, 1, 1, 0,

    -1, -1, -1, 1, 0, 0,
    -1, 1, -1, 1, 0, 1,
    -1, 1, 1, 1, 1, 1,
    -1, -1, 1, 1, 1, 0,
};
const GLubyte kubb_indices[] = {
    2, 1, 0,
    0, 3, 2,

    6, 7, 4,
    4, 5, 6,

    10, 9, 8,
    8, 11, 10,

    12, 13, 14,
    14, 15, 12,

    16, 18, 17,
    16, 19, 18,

    21, 22, 20,
    22, 23, 20,
};

const GLfloat stop = 16.28;

EGLDisplay display;
EGLSurface surface;
EGLContext context;

GLuint program;

GLint position_attrib;
GLint texcoord_attrib;
GLint beats_uniform;
GLint cycle_uniform;
GLint texture_uniform;

uint32_t sh;
uint32_t sw;

GLfloat rotate_x = 0;
GLfloat rotate_y = 0;
GLfloat rotate_z = 0;

static const char *
gl_strerror(GLint error)
{
    switch(error)
    {
    case GL_NO_ERROR:
        return "GL_NO_ERROR, No error has been recorded.";
        break;
    case GL_INVALID_ENUM:
        return "GL_INVALID_ENUM An unacceptable value is specified for an enumerated argument. The offending command is ignored and has no other side effect than to set the error flag.";
        break;

    case GL_INVALID_VALUE:
        return "GL_INVALID_VALUE A numeric argument is out of range. The offending command is ignored and has no other side effect than to set the error flag.";
        break;

    case GL_INVALID_OPERATION:
        return "GL_INVALID_OPERATION The specified operation is not allowed in the current state. The offending command is ignored and has no other side effect than to set the error flag.";
        break;

    case GL_INVALID_FRAMEBUFFER_OPERATION:
        return "GL_INVALID_FRAMEBUFFER_OPERATION The command is trying to render to or read from the framebuffer while the currently bound framebuffer is not framebuffer complete (i.e. the return value from glCheckFramebufferStatus is not GL_FRAMEBUFFER_COMPLETE). The offending command is ignored and has no other side effect than to set the error flag.";
        break;

    case GL_OUT_OF_MEMORY:
        return "GL_OUT_OF_MEMORY There is not enough memory left to execute the command. The state of the GL is undefined, except for the state of the error flags, after this error is recorded.";
        break;
    default:
        return "Unknown EGL error.";
    }
}

static void
cleanup(void)
{
   glClear(GL_COLOR_BUFFER_BIT);

   egl_cleanup();
   tty_cleanup();
}

static void
kill_handler(int signal)
{
    exit(EXIT_FAILURE);
}

GLuint
load_shader(GLenum type, const char *path)
{
#define SHADER_MAXLEN 4096

    FILE *f;
    size_t numbytes;
    char *source;
    GLuint shader;
    GLint compiled;

    if(NULL == (f = fopen(path, "r")))
        err(EXIT_FAILURE, "fopen");

    source = calloc(SHADER_MAXLEN, sizeof(char));

    numbytes = fread(source, sizeof(char), SHADER_MAXLEN - 1, f);
    source[numbytes] = '\0';

    if(0 == (shader = glCreateShader(type)))
        errx(EXIT_FAILURE, "glCreateShader: %s", gl_strerror(glGetError()));

    glShaderSource(shader, 1, (const GLchar **)&source, NULL);

    glCompileShader(shader);

    glGetShaderiv(shader, GL_COMPILE_STATUS, &compiled);

    if(!compiled)
    {
        GLint infolen = 0;

        glGetShaderiv(shader, GL_INFO_LOG_LENGTH, &infolen);

        if(infolen > 1)
        {
            char *infolog = calloc(infolen, sizeof(char));

            glGetShaderInfoLog(shader, infolen, NULL, infolog);

            fprintf(stderr, "shader compilation:\n%s\n", infolog);

            free(infolog);
        }

        glDeleteShader(shader);

        exit(EXIT_FAILURE);
    }

    return shader;
}

static void
init()
{
    GLuint vertex_shader;
    GLuint fragment_shader;
    GLint linked;

    vertex_shader = load_shader(GL_VERTEX_SHADER, "vertex.glsl");
    fragment_shader = load_shader(GL_FRAGMENT_SHADER, "fragment.glsl");

    if(0 == (program = glCreateProgram()))
        errx(EXIT_FAILURE, "glCreateProgram: %s", gl_strerror(glGetError()));

    glAttachShader(program, vertex_shader);
    glAttachShader(program, fragment_shader);

    glLinkProgram(program);

    glGetProgramiv(program, GL_LINK_STATUS, &linked);

    if(!linked)
    {
        GLint infolen = 0;

        glGetProgramiv(program, GL_INFO_LOG_LENGTH, &infolen);

        if(infolen > 1)
        {
            char *infolog = calloc(infolen, sizeof(char));

            glGetProgramInfoLog(program, infolen, NULL, infolog);

            fprintf(stderr, "shader linking:\n%s\n", infolog);

            free(infolog);
        }

        glDeleteProgram(program);

        exit(EXIT_FAILURE);
    }

    glEnable(GL_CULL_FACE);
}

uint8_t
mandelbrot(int ca, int cb) {
    int za = ca;
    int zb = cb;

    int i;
    for(i = 0; i < 253; ++i) {
        int zaa = za * za >> 10;
        int zbb = zb * zb >> 10;

        if(zaa + zaa > 4 << 10)
            return i;

        zb = ((za * zb) >> 9) + cb;
        za = zaa - zbb + ca;
    }

    return 254;
}

static int jiffies = 0;

static void
draw(GLfloat beats)
{
    static int mastensg = 0;

    if(beats > stop - 1)
        glClearColor(0, 0, 0, 2.0 * (stop - 0.5 - beats));
    else
        glClearColor(0, 0, 0, beats);

    glClear(GL_COLOR_BUFFER_BIT);

    glUniform1f(beats_uniform, beats);
    glUniform1f(cycle_uniform, fmod(beats, 256));

    glDrawElements(GL_TRIANGLES, sizeof(kubb_indices) / sizeof(kubb_indices[0]), GL_UNSIGNED_BYTE, 0);

    if(beats > 15.7)
    {
        if(!mastensg)
        {
            fprintf(stderr, "\r                                mastensg");
            mastensg = 1;
        }

        if(jiffies++ < 16)
            fprintf(stderr, "\n\r");
    }
}

static void *
escape_thread(void *arg)
{
    for(;;)
    {
        if(getchar() != EOF)
            exit(EXIT_SUCCESS);

        usleep(16000);
    }

    return arg;
}

int
main(int argc, char *argv[]) 
{
    pthread_attr_t attr;
    pthread_t escape;

    NativeWindowType window;

    GLuint vertex_buffer;
    GLuint index_buffer;
    GLuint texture;

    GLubyte *pixels;
    int y, x;

    atexit(cleanup);
    signal(SIGINT, kill_handler);

    pthread_attr_init(&attr);

    if(pthread_create(&escape, &attr, escape_thread, NULL))
        err(EXIT_FAILURE, "pthread_create");

    tty_init();

    if(argc == 3)
    {
        sw = atoi(argv[1]);
        sh = atoi(argv[2]);
    }

    window = vc_init(&sw, &sh);
    egl_init(window);

    init();

    glUseProgram(program);

    position_attrib = glGetAttribLocation(program, "position");
    texcoord_attrib = glGetAttribLocation(program, "texcoord");
    beats_uniform = glGetUniformLocation(program, "beats");
    cycle_uniform = glGetUniformLocation(program, "cycle");
    texture_uniform = glGetUniformLocation(program, "texture");

    glEnableVertexAttribArray(position_attrib);
    glEnableVertexAttribArray(texcoord_attrib);

    glGenBuffers(1, &vertex_buffer);
    glBindBuffer(GL_ARRAY_BUFFER, vertex_buffer);
    glBufferData(GL_ARRAY_BUFFER, sizeof(kubb_vertices), kubb_vertices, GL_STATIC_DRAW);

    glGenBuffers(1, &index_buffer);
    glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, index_buffer);
    glBufferData(GL_ELEMENT_ARRAY_BUFFER, sizeof(kubb_indices), kubb_indices, GL_STATIC_DRAW);

    glVertexAttribPointer(position_attrib, 4, GL_FLOAT, GL_FALSE, 6 * sizeof(GLfloat), 0);
    glVertexAttribPointer(texcoord_attrib, 2, GL_FLOAT, GL_FALSE, 6 * sizeof(GLfloat), (GLvoid *)(4 * sizeof(GLfloat)));

    int sz = 1024;

    pixels = calloc(sz * sz, 4);

    for(y = 0; y < sz; ++y)
    {
        int b = (2 << 10) - (y << 2);
        fprintf(stderr, "*");

        for(x = 0; x < sz; ++x)
        {
            int a = -(2 << 10) + (x << 2);
            int i = 254 - mandelbrot(a, b);

            if(i > 2 && i < 253)
                fprintf(stderr, " ");

            pixels[4 * (sz * y + x) + 0] = i;
            pixels[4 * (sz * y + x) + 1] = i;
            pixels[4 * (sz * y + x) + 2] = i;
            pixels[4 * (sz * y + x) + 3] = 255;
        }
    }

    glGenTextures(1, &texture);
    glBindTexture(GL_TEXTURE_2D, texture);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);
    glTexImage2D(GL_TEXTURE_2D, 0, GL_RGBA, sz, sz, 0, GL_RGBA, GL_UNSIGNED_BYTE, pixels);

    free(pixels);

    glActiveTexture(GL_TEXTURE0);
    glUniform1i(texture_uniform, 0);

    glViewport(0, 0, (GLsizei)sw, (GLsizei)sh);

    GLfloat beats = 0;

    for(;;)
    {
        draw(beats);

        egl_swapbuffers();

        if(beats < 3)
            fprintf(stderr, "\n");

        beats += 0.01;

        if(beats > stop)
            break;
    }

    sleep(2);

    for(y = 0; y < 1024; ++y)
        fprintf(stderr, "\n\r");

    fprintf(stderr, "                                Solskogen 2012\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n\r");

    sleep(3);

    for(y = 0; y < 1024; ++y)
        fprintf(stderr, "\n\r");

    fprintf(stderr, "$ ");

    return EXIT_SUCCESS;
}
