/*
 *	This program is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU General Public License
 *	as published by the Free Software Foundation; either version
 *	2 of the License, or (at your option) any later version.
 *
 *  Copyright 1999 Michael Klein <michael.klein@puffin.lb.shuttle.de>
*/

#include "cbm4linux.h"
#include "d64copy.h"

#include <stdlib.h>

#ifndef USE_CBM_IEC_WAIT
#include <unistd.h>
#endif

static unsigned char pp1541_drive_prog[] = {
#include "pp1541.inc"
};

static unsigned char pp1571_drive_prog[] = {
#include "pp1571.inc"
};

static int pp_write(int fd, char c1, char c2)
{
    cbm_pp_write(fd, c1);
    cbm_iec_release(fd, IEC_CLOCK);
#ifndef USE_CBM_IEC_WAIT
    while(cbm_iec_get(fd, IEC_DATA));
#else
    cbm_iec_wait(fd, IEC_DATA, 0);
#endif

    cbm_pp_write(fd, c2);
    cbm_iec_set(fd, IEC_CLOCK);
#ifndef USE_CBM_IEC_WAIT
    while(!cbm_iec_get(fd, IEC_DATA));
#else
    cbm_iec_wait(fd, IEC_DATA, 1);
#endif

    return 0;
}

static int pp_read(int fd, unsigned char *c1, unsigned char *c2)
{
#ifndef USE_CBM_IEC_WAIT
    while(cbm_iec_get(fd, IEC_DATA));
#else
    cbm_iec_wait(fd, IEC_DATA, 0);
#endif
    *c1 = cbm_pp_read(fd);
    cbm_iec_release(fd, IEC_CLOCK);

#ifndef USE_CBM_IEC_WAIT
    while(!cbm_iec_get(fd, IEC_DATA));
#else
    cbm_iec_wait(fd, IEC_DATA, 1);
#endif
    *c2 = cbm_pp_read(fd);
    cbm_iec_set(fd, IEC_CLOCK);

    return 0;
}

static int read_block(int tr, int se, char *block)
{
    int  i;
    unsigned char status;

    pp_write(fd_cbm, tr, se);
#ifndef USE_CBM_IEC_WAIT    
    usleep(20000);
#endif
    pp_read(fd_cbm, &status, &status);

    for(i=0;i<BLOCKSIZE;i+=2) pp_read(fd_cbm, &block[i], &block[i+1]);

    return status;
}

static int write_block(int tr, int se, char *block, int size)
{
    int i = 0;
    unsigned char status;

    pp_write(fd_cbm, tr, se);

    if(size % 2) {
        pp_write(fd_cbm, *block, *block);
        i = 1;
    }

    for(;i<size;i+=2) pp_write(fd_cbm, block[i], block[i+1]);

#ifndef USE_CBM_IEC_WAIT    
    if(size == BLOCKSIZE) {
        usleep(20000);
    }
#endif

    pp_read(fd_cbm, &status, &status);

    return status;
}

static int open_disk(char *name, int for_writing, int ext, turbo_start start)
{
    int d = atoi(name);
    unsigned char *drive_prog;
    int prog_size;

    if(drive_type) {
        drive_prog = pp1571_drive_prog;
        prog_size  = sizeof(pp1571_drive_prog);
    } else {
        drive_prog = pp1541_drive_prog;
        prog_size  = sizeof(pp1541_drive_prog);
    }

    cbm_upload(fd_cbm, d, 0x700, drive_prog, prog_size);
    start(d);
    while(!cbm_iec_get(fd_cbm, IEC_DATA));
    return 0;
}

static void close_disk(void)
{
    pp_write(fd_cbm, 0, 0);
    while(cbm_iec_get(fd_cbm, IEC_DATA));
}

static int send_track_map(int tr, char *trackmap, int count)
{
    int i;
    unsigned char c;
    pp_write(fd_cbm, tr, count);
    for(i = 0; i < sector_map[tr]; i++) {
        c = trackmap[i] != BS_MUST_COPY;
        pp_write(fd_cbm, c, c);
    }
    return 0;
}

static int read_gcr_block(int *se, unsigned char *gcrbuf)
{
    int i;
    unsigned char s;

    pp_read(fd_cbm, &s, &s);
    *se = s;
    pp_read(fd_cbm, &s, &s);

    if(s) {
        return s;
    }

    for(i = 0; i < GCRBUFSIZE; i += 2) {
        pp_read(fd_cbm, &gcrbuf[i], &gcrbuf[i+1]);
    }

    return 0;
}

DECLARE_TRANSFER_FUNCS_EX(pp_transfer, 1, 1);
