/*
 *	This program is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU General Public License
 *	as published by the Free Software Foundation; either version
 *	2 of the License, or (at your option) any later version.
 *
 *  Copyright 1999-2001 Michael Klein <michael.klein@puffin.lb.shuttle.de>
*/

#include "cbm4linux.h"

#include <ctype.h>
#include <errno.h>
#include <error.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <unistd.h>

typedef int (*mainfunc)(int fd, char *argv[]);

static char petscii2ascii_c(char p)
{
    switch(p & 0xff) {
        case 0x0a:
        case 0x0d:
            return '\n';
        case 0x40:
        case 0x60:
            break;
        case 0xa0:
        case 0xe0:
            return ' ';
        default:
            switch(p & 0xe0) {
                case 0x40:
                case 0x60:
                    return p ^ 0x20;
                case 0xc0:
                    return p ^ 0x80;
            }
    }
    return isprint(p) ? p : '.';
}

static char *petscii2ascii(char *p)
{
    char *c;

    for(c = p; *c; c++) *c = petscii2ascii_c(*c);

    return p;
}

static int do_reset(int fd, char *argv[])
{
    return cbm_reset(fd);
}

static int do_listen(int fd, char *argv[])
{
    return cbm_listen(fd, atoi(argv[0]), atoi(argv[1]));
}

static int do_talk(int fd, char *argv[])
{
    return cbm_talk(fd, atoi(argv[0]), atoi(argv[1]));
}

static int do_unlisten(int fd, char *argv[])
{
    return cbm_unlisten(fd);
}

static int do_untalk(int fd, char *argv[])
{
    return cbm_untalk(fd);
}

static int do_open(int fd, char *argv[])
{
    int rv;
    
    rv = cbm_open(fd, atoi(argv[0]), atoi(argv[1]));
    write(fd, argv[2], strlen(argv[2]));
    cbm_unlisten(fd);
    
    return rv;
}

static int do_close(int fd, char *argv[])
{
    return cbm_close(fd, atoi(argv[0]), atoi(argv[1]));
}

static int do_status(int fd, char *argv[])
{
    char c;
    int  rv;

    rv = cbm_talk(fd, atoi(argv[0]), 15);
    if(rv == 0) {
        while(read(fd, &c, 1) == 1) {
            putchar(c == 0x0d ? 0x0a : tolower(c));
        }
        rv = cbm_untalk(fd);
    }
    return rv;
}

static int do_command(int fd, char *argv[])
{
    int  rv;

    rv = cbm_listen(fd, atoi(argv[0]), 15);
    if(rv == 0) {
        write(fd, argv[1], strlen(argv[1]));
        rv = cbm_unlisten(fd);
    }
    return rv;
}

static int do_dir(int fd, char *argv[])
{
    unsigned char c, buf[40];
    int rv;
    int unit;

    unit = atoi(argv[0]);
    rv = cbm_open(fd, unit, 0);
    if(rv == 0) {
        write(fd, "$", 1);
        cbm_unlisten(fd);
        if(cbm_device_status(fd, unit, buf, sizeof(buf)) == 0) {
            cbm_talk(fd, unit, 0);
            if(read(fd, buf, 2) == 2) {
                while(read(fd, buf, 2) == 2) {
                    if(read(fd, buf, 2) == 2) {
                        printf("%u ", buf[0] | (buf[1] << 8));
                        while((read(fd, &c, 1) == 1) && c) {
                            putchar(petscii2ascii_c(c));
                        }
                        putchar('\n');
                    }
                }
                cbm_device_status(fd, unit, buf, sizeof(buf));
                fprintf(stderr, petscii2ascii(buf));
            }
            cbm_untalk(fd);
        } else fprintf(stderr, petscii2ascii(buf));
        cbm_close(fd, unit, 0);
    }
    return rv;
}

static int do_download(int fd, char *argv[])
{
    int unit, c, addr, count, i, rv = 0;
    char *tail, buf[32], cmd[7];

    unit = atoi(argv[0]);

    addr = strtol(argv[1], &tail, 0);
    if(addr < 0 || addr > 0xffff || *tail) {
        error(0, 0, "invalid address: %s", argv[1]);
        return 1;
    }
    
    count = strtol(argv[2], &tail, 0);
    if((count + addr) > 0x10000 || *tail) {
        error(0, errno, "invalid byte count %s", argv[2]);
        return 1;
    }
    
    for(i = 0; (rv == 0) && (i < count); i+=32) {
        c = count - i;
        if(c > 32) c = 32;
        sprintf(cmd, "M-R%c%c%c", addr%256, addr/256, c);
        cbm_listen(fd, unit, 15);
        rv = write(fd, cmd, 6) == 6 ? 0 : 1;
        cbm_unlisten(fd);
        if(rv == 0) {
            addr += c;
            cbm_talk(fd, unit, 15);
            rv = read(fd, buf, c) == c ? 0 : 1;
            cbm_untalk(fd);
            if(rv == 0) {
                write(1, buf, c);
            }
        }
    }
    return rv;
}

static int do_upload(int fd, char *argv[])
{
    int unit, addr, size, rv = 0;
    char *tail, *buf;
    unsigned char addr_buf[2];
    struct stat statrec;
    FILE *f;

    unit = atoi(argv[0]);
    
    addr = strtoul(argv[1], &tail, 0);
    if(addr < -1 || addr > 0xffff || *tail) {
        error(0, 0, "invalid address: %s", argv[1]);
        return 1;
    }
    
    if(stat(argv[2], &statrec)) {
        error(0, errno, "could not stat %s", argv[2]);
        return 1;
    }
    
    size = statrec.st_size;
    if(size <= ((addr >= 0) ? 0 : 2)) {
        error(0, 0, "empty program: %s", argv[2]);
        return 1;
    }

    f = fopen(argv[2], "rb");
    if(f == NULL) {
        error(0, errno, "could not open %s", argv[2]);
        return 1;
    }
    
    if(addr == -1) {
        /* read address from file */
        if(fread(addr_buf, 2, 1, f) != 1) {
            error(0, errno, "could not read %s", argv[2]);
            fclose(f);
            return 1;
        }
        size -= 2;
        /* don't assume special endianess, although the cbm4linux package is
           i386 currently.  */
        addr = addr_buf[0] | (addr_buf[1] << 8);
    }

    if(addr + size > 0x10000) {
        error(0, 0, "program too big: %s", argv[2]);
        fclose(f);
        return 1;
    }
    
    buf = malloc(size);
    if(buf == NULL) abort();
    
    if(fread(buf, size, 1, f) != 1) {
        error(0, errno, "could not read %s", argv[2]);
        free(buf);
        fclose(f);
        return 1;
    }
    
    fclose(f);
    
    rv = (cbm_upload(fd, unit, addr, buf, size) == size) ? 0 : 1;

    free(buf);
    return rv;
}

struct prog {
    char    *name;
    mainfunc prog;
    int      req_args;
    char    *arglist;
};

static struct prog prog_table[] = {
    {"listen"  , do_listen  , 2, "<device> <secadr>"               },
    {"talk"    , do_talk    , 2, "<device> <secadr>"               },
    {"unlisten", do_unlisten, 0, ""                                },
    {"untalk"  , do_untalk  , 0, ""                                },
    {"open"    , do_open    , 3, "<device> <secadr> <filename>"    },
    {"close"   , do_close   , 2, "<device> <secadr>"               },
    {"status"  , do_status  , 1, "<device>"                        },
    {"command" , do_command , 2, "<device> <cmdstr>"               },
    {"dir"     , do_dir     , 1, "<device>"                        },
    {"download", do_download, 3, "<device> <adr> <count>"          },
    {"upload"  , do_upload  , 3, "<device> <adr> <file>"           },
    {"reset"   , do_reset   , 0, ""                                },
    {NULL,NULL}
};

static struct prog *find_main(char *name)
{
    int i;

    for(i=0; prog_table[i].name; i++) {
        if(strcmp(name, prog_table[i].name) == 0) {
            return &prog_table[i];
        }
    }
    return NULL;
}

int main(int argc, char *argv[])
{
    struct prog *p;
    int i;

    p = argc < 2 ? NULL : find_main(argv[1]);
    if(p) {
        if(p->req_args == argc-2) {

            int fd = open(cbm_dev, O_RDWR);
            int rv = 1;

            if(fd != -1) {
                rv = p->prog(fd, &argv[2]) != 0;
                close(fd);
            } else {
                if(errno) {
                    error(0, errno, "%s", cbm_dev);
                }
            }
            return rv;
        } else {
            error(0, errno, "wrong number of arguments:\n\n  %s %s %s\n",
                        argv[0], argv[1], p->arglist);
        }
    } else {
        printf("invalid command, available ones are:\n\n");
        for(i=0; prog_table[i].prog; i++) {
            printf("  %s %s\n", prog_table[i].name, prog_table[i].arglist);
        }
    }
    return 2;
}
